/*
    libzint - the open source barcode library
    Copyright (C) 2008-2019 Robin Stuart <rstuart114@gmail.com>

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions
    are met:

    1. Redistributions of source code must retain the above copyright
       notice, this list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright
       notice, this list of conditions and the following disclaimer in the
       documentation and/or other materials provided with the distribution.
    3. Neither the name of the project nor the names of its contributors
       may be used to endorse or promote products derived from this software
       without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
    ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
    IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
    ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
    FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
    DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
    OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
    HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
    LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
    OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
    SUCH DAMAGE.
 */
/* vim: set ts=4 sw=4 et : */

#include "testcommon.h"

//#define TEST_EXAMPLES_GENERATE_EXPECTED 1
//#define TEST_ODD_NUMBERED_NUMERIC_GENERATE_EXPECTED 1
//#define TEST_EAN128_CC_SHIFT_GENERATE_EXPECTED 1
//#define TEST_EAN128_CC_WIDTH_GENERATE_EXPECTED 1
//#define TEST_ENCODATION_0_GENERATE_EXPECTED 1
//#define TEST_ENCODATION_10_GENERATE_EXPECTED 1
//#define TEST_ENCODATION_11_GENERATE_EXPECTED 1

static void test_eanx_leading_zeroes(void)
{
    testStart("");

    int ret;
    struct item {
        int symbology;
        unsigned char* data;
        unsigned char* composite;
        int ret;

        int expected_rows;
        int expected_width;
    };
    // s/\/\*[ 0-9]*\*\//\=printf("\/*%2d*\/", line(".") - line("'<"))
    struct item data[] = {
        /* 0*/ { BARCODE_EANX_CC, "1", "[21]A12345678", 0, 8, 72 }, // EAN-8
        /* 1*/ { BARCODE_EANX_CC, "12", "[21]A12345678", 0, 8, 72 },
        /* 2*/ { BARCODE_EANX_CC, "123", "[21]A12345678", 0, 8, 72 },
        /* 3*/ { BARCODE_EANX_CC, "1234", "[21]A12345678", 0, 8, 72 },
        /* 4*/ { BARCODE_EANX_CC, "12345", "[21]A12345678", 0, 8, 72 },
        /* 5*/ { BARCODE_EANX_CC, "123456", "[21]A12345678", 0, 8, 72 },
        /* 6*/ { BARCODE_EANX_CC, "1234567", "[21]A12345678", 0, 8, 72 },
        /* 7*/ { BARCODE_EANX_CC, "12345678", "[21]A12345678", 0, 7, 99 }, // EAN-13
        /* 8*/ { BARCODE_EANX_CC, "1+12", "[21]A12345678", 0, 8, 101 }, // EAN-8 + EAN-2
        /* 9*/ { BARCODE_EANX_CC, "12+12", "[21]A12345678", 0, 8, 101 },
        /*10*/ { BARCODE_EANX_CC, "123+12", "[21]A12345678", 0, 8, 101 },
        /*11*/ { BARCODE_EANX_CC, "1234+12", "[21]A12345678", 0, 8, 101 },
        /*12*/ { BARCODE_EANX_CC, "12345+12", "[21]A12345678", 0, 8, 101 },
        /*13*/ { BARCODE_EANX_CC, "123456+12", "[21]A12345678", 0, 8, 101 },
        /*14*/ { BARCODE_EANX_CC, "1234567+12", "[21]A12345678", 0, 8, 101 },
        /*15*/ { BARCODE_EANX_CC, "12345678+12", "[21]A12345678", 0, 7, 128 }, // EAN-13 + EAN-2
        /*16*/ { BARCODE_EANX_CC, "1+123", "[21]A12345678", 0, 8, 128 }, // EAN-8 + EAN-5
        /*17*/ { BARCODE_EANX_CC, "12+123", "[21]A12345678", 0, 8, 128 },
        /*18*/ { BARCODE_EANX_CC, "123+123", "[21]A12345678", 0, 8, 128 },
        /*19*/ { BARCODE_EANX_CC, "1234+123", "[21]A12345678", 0, 8, 128 },
        /*20*/ { BARCODE_EANX_CC, "12345+123", "[21]A12345678", 0, 8, 128 },
        /*21*/ { BARCODE_EANX_CC, "123456+123", "[21]A12345678", 0, 8, 128 },
        /*22*/ { BARCODE_EANX_CC, "1234567+123", "[21]A12345678", 0, 8, 128 },
        /*23*/ { BARCODE_EANX_CC, "12345678+123", "[21]A12345678", 0, 7, 155 }, // EAN-13 + EAN-5
    };
    int data_size = sizeof(data) / sizeof(struct item);

    for (int i = 0; i < data_size; i++) {

        struct zint_symbol* symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        symbol->symbology = data[i].symbology;
        int length = strlen(data[i].data);
        assert_zero(length >= 128, "i:%d length %d >= 128\n", i, length);
        strcpy(symbol->primary, data[i].data);

        int composite_length = strlen(data[i].composite);

        ret = ZBarcode_Encode(symbol, data[i].composite, composite_length);
        assert_equal(ret, data[i].ret, "i:%d ret %d != %d\n", i, ret, data[i].ret);
        assert_equal(symbol->rows, data[i].expected_rows, "i:%d symbol->rows %d != %d\n", i, symbol->rows, data[i].expected_rows);
        assert_equal(symbol->width, data[i].expected_width, "i:%d symbol->width %d != %d\n", i, symbol->width, data[i].expected_width);

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

static void test_helper_generate(const struct zint_symbol* symbol, int ret, int i, unsigned char* data, unsigned char* composite, int option_1, int option_2, int option_3, char* comment)
{
    if (ret == 0) {
        printf("        /*%2d*/ { %s, \"%s\", \"%s\", %d, %d, %d, %d, %d, %d, \"%s\",\n",
                i, testUtilBarcodeName(symbol->symbology), data, composite, option_1, option_2, option_3, ret, symbol->rows, symbol->width, comment);
        testUtilModulesDump(symbol, "                    ", "\n");
        printf("               },\n");
    } else {
        printf("        /*%2d*/ { %s, \"%s\", \"%s\", %d, %d, %d, %s, %d, %d, \"%s\", \"\" },\n",
                i, testUtilBarcodeName(symbol->symbology), data, composite, option_1, option_2, option_3, testUtilErrorName(ret), symbol->rows, symbol->width, comment);
    }
}

// Replicate examples from GS1 General Specifications 19.1 and ISO/IEC 24723:2010
static void test_examples(void)
{
    testStart("");

    int ret;
    struct item {
        int symbology;
        unsigned char* data;
        unsigned char* composite;
        int option_1;
        int option_2;
        int option_3;
        int ret;

        int expected_rows;
        int expected_width;
        char* comment;
        unsigned char* expected;
    };
    // Verified manually against GS1 General Specifications 19.1 and ISO/IEC 24723:2010, with noted exceptions
    struct item data[] = {
        /* 0*/ { BARCODE_RSS14_OMNI_CC, "0401234567890", "[17]050101[10]ABC123", 1, 0, 0, 0, 11, 56, "Figure 5.1-5. GS1 DataBar Stacked Omnidirectional barcode with a Composite Component",
                    "01101100110101110001001100001000000110100111011110101001"
                    "01101101110110001100010100001100001000010110011100101001"
                    "01101101100111000101110001101001100011111010011101101001"
                    "01101101000110111100010011001111110100111011011101001001"
                    "01101001000111110111110100101010100000010000011101001101"
                    "00000000100110111010100000101010101100110001100000000000"
                    "01001111011001000100011111000001010011001110011010000000"
                    "00000000100110111010100000101010101100110001100000000000"
                    "00000101010101010101010101010101010101010101010000000000"
                    "00001000110000101010000000101010111011001111000000000000"
                    "10100111001111010101111111000001000100110000110101000000"
               },
        /* 1*/ { BARCODE_RSS_LTD_CC, "1311234567890", "[17]010615[10]A123456", 1, 0, 0, 0, 6, 74, "Figure 5.9.2-1. GS1 DataBar Limited Composite symbol with CC-A",
                    "01111000101101100010100110001111101001100111011101001111001110111010011010"
                    "01001111100011010010101110001111011110101111010011110111001110111010111010"
                    "01001100110100000010101100001110010001101111011110111100101000111010110010"
                    "01111000101010000010100100001011101101111110011100110011100100111010100010"
                    "00000011000001010100110011101010110101001100101011110001011001101001110000"
                    "01011100111110101011001100010101001010110011010100001110100110010110000101"
               },
        /* 2*/ { BARCODE_EAN128_CC, "[01]03812345678908", "[10]ABCD123456[410]3898765432108", 3, 0, 0, 0, 7, 154, "Figure 5.9.2-2. GS1-128 Composite symbol with CC-C **NOT SAME** (uses encodation '10')",
                    "1111111101010100011110101011110000111101011001111101110111110111010010000010000100010110010000101100001111011110110011011110101001111000111111101000101001"
                    "1111111101010100011111101010001110100001000111101001100101110010000011100001011000100100100111110110001011100001011111011111101010111000111111101000101001"
                    "1111111101010100011101010011111100110001111010001101000101011110000010001111101100010111101101111101001001011000111110011101010001111110111111101000101001"
                    "1111111101010100010101111001111000110010011001110001111010100111100010011110111101000110000110001000101100001011101111011111010111111010111111101000101001"
                    "1111111101010100011101011100001100101000111111011101011110001001111011111011001000100111100111011101001101101111001000011101011100110000111111101000101001"
                    "0000000001011000110000101000100110010011011011001110110100001100010010001010001001110111101001100100100001011100110110100001000100100001001001110001010000"
                    "0000000110100111001111010111011001101100100100110001001011110011101101110101110110001000010110011011011110100011001001011110111011011110110110001110101100"
               },
        /* 3*/ { BARCODE_EANX_CC, "331234567890", "[21]1234-abcd", 1, 0, 0, 0, 7, 99, "Figure 5.9.8-1. EAN-13 symbol with a four-column CC-A component (note [21] not [99])",
                    "110110111011010000100000110100110011101100001001110100100001011001100001100111000110001011011000101"
                    "110110110011000110111100010111011001110000101001100100100000010111101001101011100010000011001000101"
                    "110110100010011010001110000111111010001100101001100110111111010001101001010000011011111011101000101"
                    "000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010"
                    "001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001"
                    "000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010"
                    "000101011110100110010011011010000100111010110001010101010000100010010010001110100111001010000101010"
               },
        /* 4*/ { BARCODE_UPCA_CC, "61414101234", "[91]abcdefghijklmnopqrstuvwxyz", 2, 0, 0, 0, 14, 99, "Figure 5.9.8-2. UPC-A symbol with a four-column CC-B component **NOT SAME** (using [91] not [10] as length > 20 max for [10])",
                    "110001001010000001110010110110011111101100101001111010100100101111000001110101001111110011000100101"
                    "111001001011101110101000000111101101000111001011111010100011000110000101110011010000110011100100101"
                    "111101001011110110001101000111101000100000101011110010101001111001000001011111010001110011110100101"
                    "111101011011110100111100010111011111001011001011110110111110001001110101011111001110011011110101101"
                    "111101010010001100011100110111001000010011101001110110100011011010000001001100000110001011110101001"
                    "111001010010000001011110100111100110110100001001110100111001011000010001111010010100000011100101001"
                    "111011010010011111101110100101110001000011001001100100100011110010000101011110000110011011101101001"
                    "111010010011100011101000100100001011011000001001100110110001110110100001000110011101100011101001001"
                    "111010011010110111111011000101111101100111101001000110110100111100001101111100001101010011101001101"
                    "111010111010111000001000110110111110010001001001000010111100101111101101000110001000111011101011101"
                    "000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010"
                    "001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001"
                    "000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010"
                    "000101010111100110010100011001100101000110011001010101110010110011011011001000010101110010000101010"
               },
        /* 5*/ { BARCODE_EANX_CC, "1234567", "[21]A12345678", -1, 0, 0, 0, 8, 72, "Figure 5.9.8-3. EAN-8 symbol with a three-column CC-A",
                    "101001111000001001010011000111110101110111101001101001111110011101001101"
                    "111110010011100101010111000101110011011100001111110100011001011101011101"
                    "110011001000010001010110000101000001000010001001000110110000011101011001"
                    "101011110010000001010010000111011100111101001111001001000010011101010001"
                    "000010000000000000000000000000000000000000000000000000000000000000000010"
                    "000100000000000000000000000000000000000000000000000000000000000000000001"
                    "000010000000000000000000000000000000000000000000000000000000000000000010"
                    "000010100110010010011011110101000110101010011101010000100010011100101010"
               },
        /* 6*/ { BARCODE_UPCE_CC, "121230", "[15]021231", 1, 0, 0, 0, 9, 55, "Figure 5.9.8-4. UPC-E symbol with a two-column CC-A",
                    "1101100110111010011111010001100111100010110011110101001"
                    "1101101110110010010000110001101000011011100011100101001"
                    "1101101100111101001000000101000101111000010011101101001"
                    "1101101000111011110100011101001110011011100011101001001"
                    "1101001000111000010110111101001110001110010011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010110011001001100110010011011011110101001110101010"
               },
        /* 7*/ { BARCODE_RSS14_CC, "0361234567890", "[11]990102", 1, 0, 0, 0, 5, 100, "Figure 5.9.8-5. GS1 DataBar Omnidirectional symbol with a four-column CC-A",
                    "1101101110110000101000110001111001010111100010011101001110011101100110011001001100111000110110001010"
                    "1101101100110111011111001001000011010111111010011001001101000000111010010010111111001110110010001010"
                    "1101101000110010010111110001011001101111000010011001101111010011110010010000011001011100111010001010"
                    "0000000000010110001110100000000101001011010111111011001101010000011010000000010100101000110011110000"
                    "0000010011101001110001001111111000010100101000000100110010101111100101111111100011010111001100001101"
               },
        /* 8*/ { BARCODE_RSS14STACK_CC, "0341234567890", "[17]010200", 1, 0, 0, 0, 9, 56, "Figure 5.9.8-6. GS1 DataBar Stacked symbol with a two-column CC-A",
                    "01101100110101110011100111101010000100001111011110101001"
                    "01101101110110110001000010001110111101100100011100101001"
                    "01101101100110100001111011001111110011010110011101101001"
                    "01101101000100110000101110001011100000110111011101001001"
                    "01101001000101110111110111001001100110100000011101001101"
                    "00000001011011100010000010101010100000101101010000000000"
                    "01001110100100011101111100000001011111010010100010000000"
                    "00000011010111101010000010101010101001001101010000000000"
                    "10101100111000010101111111110111000110110011100101000000"
               },
        /* 9*/ { BARCODE_RSS_LTD_CC, "0351234567890", "[91]abcdefghijklmnopqrstuv", 2, 0, 0, 0, 17, 83, "Figure 5.9.8-7. GS1 DataBar Limited symbol with a three-column CC-B **NOT SAME** (using [91] not [21] as length > 20 max for [21])",
                    "11011101001110111110111010010110001001000001000010001011111011010011110110111010010"
                    "11011001001111110101001110010110001101111011000011001010100001111000100110110010010"
                    "11011001101001111000010010010100001101110111001011110011011100100011100110110011010"
                    "11011011101110100000100111010100011101110100001001110010000110011001000110110111010"
                    "11011011001111010010000001010100011001000110111110010011111011001000100110110110010"
                    "11011010001011111001110011010100111001011111000111011010001010000111100110110100010"
                    "11010010001000110110100000010100110001000100000010001011010111000111100110100100010"
                    "11010110001010011111100111010101110001111010010100000011101011001000000110101100010"
                    "11010111001011111000011001010101100001100011110001011011111101001101000110101110010"
                    "11010111101100010000100011010100100001100010010110000010001000010100000110101111010"
                    "11010011101000001001111001010110100001111001010000010011110100001000100110100111010"
                    "11010011001100111001001110010010100001011000111011111010000110001001110110100110010"
                    "11010001001011110001111010010010110001100001000010110010100001100011000110100010010"
                    "11010001101110110111000001010010111001101000011100001011110110111001100110100011010"
                    "11010000101111101001110100010110111001001011000111110010111110111000110110100001010"
                    "00000000000001111011100011010001110101010110101001100101110100100111000110101110000"
                    "00000000001010000100011100101110001010101001010110011010001011011000111001010000101"
               },
        /*10*/ { BARCODE_RSS_EXP_CC, "[01]93712345678904[3103]001234", "[91]1A2B3C4D5E", 1, 0, 0, 0, 5, 151, "Figure 5.9.8-8. GS1 DataBar Expanded symbol with a four-column CC-A",
                    "0011011011101110011010011000011100011100110110100111010011010001000011000101101110011000001101100010100000000000000000000000000000000000000000000000000"
                    "0011011011001101110111110100011010001111001100100110010010111111001001100100101111110011101100100010100000000000000000000000000000000000000000000000000"
                    "0011011010001010111011111100011111011011110010100110011011000011010011110100001011001111101110100010100000000000000000000000000000000000000000000000000"
                    "0000011011111011000100000000101001010000011101001110100110001100111001000010101000011010001110001000100001010000111001010000001010010111000110010110000"
                    "0101100100000100111011111111000010101111100010110001011001110011000010111100000011100101110001110111011110101111000110001111110000101000111001101000010"
               },
        /*11*/ { BARCODE_EAN128_CC, "[01]03212345678906", "[21]A1B2C3D4E5F6G7H8", 1, 0, 0, 0, 6, 145, "Figure 5.9.8-9. GS1-128 symbol with a four-column CC-A",
                    "0000000000000000000001101001000110100001000001101101011110111110010010001101010000010010000011101110100010000111011001010000000000000000000000000"
                    "0000000000000000000001101011000110101111001100001111010001101100010010000101111000011001101011100101100001000110011001010000000000000000000000000"
                    "0000000000000000000001101011100100011001100111101011000101110000010110000101001100110011110011011110011001110110111001010000000000000000000000000"
                    "0000000000000000000001101011110111000111011011001110010001011100010111000101011000011100110010000100000100010110111101010000000000000000000000000"
                    "0010110001100001010001001100100110110110011100100011011000100100010100010011101111010011001001000010110011011100010100001000100010010011100010100"
                    "1101001110011110101110110011011001001001100011011100100111011011101011101100010000101100110110111101001100100011101011110111011101101100011101011"
               },
        /*12*/ { BARCODE_RSS_EXPSTACK_CC, "[01]00012345678905[10]ABCDEF", "[21]12345678", 1, 0, 0, 0, 13, 102, "24723:2010 Figure 10 — A GS1 DataBar Expanded Stacked Composite symbol (with CC-A)",
                    "001101101110110100001000001101001100111011000010011101001000110011100110010100111011100000110110001010"
                    "001101101100101111110100011001111101101000001010011001001011111011011110011010111000100000110010001010"
                    "001101101000100101001111000001000111011101111010011001101011110110110000011010001011111000111010001010"
                    "000001000111011100010000000010100100001101100000101010000110010000010100000111111001110100111100100000"
                    "010110111000100011101111111100001011110010011111010101111001101111101011111000000110001011000011011101"
                    "000001000111011100010000000010100100001101100000101010000110010000010100000101010001110100111100100000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000000110001000010100001000000001010010000001000110101111011001110001001010000001010000101001000010000"
                    "101111001110111101011100111111110101101111110111001010000100110001110100001111110001111010110111100010"
                    "000000110001000010100001000000001010010000001000110101111011001110001001010000001010000101001000010000"
                    "000001010101010101010101010101010101010101010101010101010101010101010101010101010101010101010101010000"
                    "000000111001111101010100001010100101011111000010100000000000000000000000000000000000000000000000000000"
                    "010111000110000010100011110000001010100000111101000100000000000000000000000000000000000000000000000000"
               },
        /*13*/ { BARCODE_EAN128_CC, "[00]030123456789012340", "[02]13012345678909[37]24[10]1234567ABCDEFG", 3, 0, 0, 0, 7, 174, "24723:2010 Figure 12 — A GS1-128 Composite symbol (with CC-C)",
                    "111111110101010001111010101111000011010111011110000111011111011101001000001000010001011110101100111110111010010001110001000100011000011011111010100111110111111101000101001000"
                    "111111110101010001111110101000111010000100111101000110011110101111101111010001010000011111000110010100111001011100011001001001111101100011111101010111000111111101000101001000"
                    "111111110101010001010100011110000011001111100001010110100010111110001110111101011100011000001101011110101111001000000101100001011111101011101010001111110111111101000101001000"
                    "111111110101010001010111100111100011110100001011110101011111011111001101010000110000011010011100011110101111001111010001100011101101000011101001011100000111111101000101001000"
                    "111111110101010001110101110000110011111010100011000100100001111000101110000001011001011110010110001100111100101101100001111011000110100011101011100110000111111101000101001000"
                    "000000000101100011000010100010010011001101101100111001100100110001001000101000100111011110100110010010000100110010011000100100010011101011101000010001000100001010011100010100"
                    "000000011010011100111101011101101100110010010011000110011011001110110111010111011000100001011001101101111011001101100111011011101100010100010111101110111011110101100011101011"
               },
    };
    int data_size = sizeof(data) / sizeof(struct item);

    for (int i = 0; i < data_size; i++) {

        struct zint_symbol* symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        symbol->symbology = data[i].symbology;
        symbol->option_1 = data[i].option_1;
        symbol->option_2 = data[i].option_2;
        symbol->option_3 = data[i].option_3;
        int length = strlen(data[i].data);
        assert_zero(length >= 128, "i:%d length %d >= 128\n", i, length);
        strcpy(symbol->primary, data[i].data);

        int composite_length = strlen(data[i].composite);

        ret = ZBarcode_Encode(symbol, data[i].composite, composite_length);
        assert_equal(ret, data[i].ret, "i:%d ret %d != %d %s\n", i, ret, data[i].ret, symbol->errtxt);

        #ifdef TEST_EXAMPLES_GENERATE_EXPECTED
        test_helper_generate(symbol, ret, i, data[i].data, data[i].composite, data[i].option_1, data[i].option_2, data[i].option_3, data[i].comment);
        #else

        assert_equal(symbol->rows, data[i].expected_rows, "i:%d %s symbol->rows %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->rows, data[i].expected_rows, data[i].data);
        assert_equal(symbol->width, data[i].expected_width, "i:%d %s symbol->width %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->width, data[i].expected_width, data[i].data);

        if (ret == 0) {
            int width, row;
            ret = testUtilModulesCmp(symbol, data[i].expected, &width, &row);
            assert_zero(ret, "i:%d %s testUtilModulesCmp ret %d != 0 width %d row %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), ret, width, row, data[i].data);
        }
        #endif

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

static void test_odd_numbered_numeric(void)
{
    testStart("");

    int ret;
    struct item {
        int symbology;
        unsigned char* data;
        unsigned char* composite;
        int option_1;
        int option_2;
        int option_3;
        int ret;

        int expected_rows;
        int expected_width;
        char* comment;
        unsigned char* expected;
    };
    // Verified manually against tec-it.com and bwipp
    struct item data[] = {
        /* 0*/ { BARCODE_UPCE_CC, "1234567", "[91]1234567890123", 1, 0, 0, 0, 9, 55, "Test odd-numbered numeric, 1st fit, 9-bit remainder, 7-bit final, 2-bit alphanumeric latch, no padding",
                    "1101100110111011101011110001001111100110010011110101001"
                    "1101101110110001100001000101100010000100110011100101001"
                    "1101101100111110001001001101100100111110111011101101001"
                    "1101101000110101100011111101100111110100100011101001001"
                    "1101001000110111000110001001010001100001100011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 1*/ { BARCODE_UPCE_CC, "1234567", "[91]12345678901234", 1, 0, 0, 0, 9, 55, "Test even-numbered numeric, 1st fit, 2-bit remainder, 2-bit alphanumeric latch, no padding",
                    "1101100110111011101011110001001111100110010011110101001"
                    "1101101110110001100001000101100010000100110011100101001"
                    "1101101100111110001001001101100100111001000011101101001"
                    "1101101000101000010011110001101101001111000011101001001"
                    "1101001000100011101110010001011100011101000011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 2*/ { BARCODE_UPCE_CC, "1234567", "[91]123456789012345", 1, 0, 0, 0, 10, 55, "Test odd-numbered numeric, 2nd fit, alphanumeric latch, padding",
                    "1100100010111100110100111001011101110001000011100101101"
                    "1110100010110001011101000001000111010111110011000101101"
                    "1110110010110101100111111001000111100001001011000101001"
                    "1100110010100100000010010001010000010000100011001101001"
                    "1101110010111111010101110001110101110000110011011101001"
                    "1101111010110000111101001101100011110110111011011001001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 3*/ { BARCODE_UPCE_CC, "1234567", "[91]ABCD12345678901", 1, 0, 0, 0, 10, 55, "Test odd-numbered numeric, 1st fit, 4-bit final, no alphanumeric latch or padding",
                    "1100100010111100110100111001110101110111110011100101101"
                    "1110100010101111100110111101111000001000101011000101101"
                    "1110110010110111100001001101101000111111001011000101001"
                    "1100110010101000110011000001111101101011110011001101001"
                    "1101110010110101111000011001111000001001010011011101001"
                    "1101111010100010000010111101011100000100011011011001001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 4*/ { BARCODE_UPCE_CC, "1234567", "[91]ABCDE123456789", 1, 0, 0, 0, 10, 55, "Test odd-numbered numeric, 1st fit, 5-bit final, no alphanumeric latch or padding",
                    "1100100010111100110100111001110101110111110011100101101"
                    "1110100010101111100110111101100000100011101011000101101"
                    "1110110010100110000010111001111101011110011011000101001"
                    "1100110010100011000110000101111101110101110011001101001"
                    "1101110010110100011111011101000111100101111011011101001"
                    "1101111010110111101000110001100011001011110011011001001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 5*/ { BARCODE_UPCE_CC, "1234567", "[91]1234567890123456789", 1, 0, 0, 0, 10, 55, "Test odd-numbered numeric, 1st fit, 7-bit final, no alphanumeric latch or padding",
                    "1100100010111100110100111001011101110001000011100101101"
                    "1110100010110001011101000001000111010111110011000101101"
                    "1110110010110101100111111001000111100001001011000101001"
                    "1100110010111100100010111101100100010110000011001101001"
                    "1101110010111110010110011101110000100110100011011101001"
                    "1101111010111111011001110101011100001001100011011001001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 6*/ { BARCODE_UPCE_CC, "1234567", "[91]1234A", 1, 0, 0, 0, 9, 55, "Test even-numbered numeric, ending in non-digit",
                    "1101100110111011101011110001001111100110010011110101001"
                    "1101101110100001000100001001110110011001110011100101001"
                    "1101101100100101111100110001000001011111011011101101001"
                    "1101101000110001001001111101011111011001000011101001001"
                    "1101001000100110001100000101110110000010110011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 7*/ { BARCODE_UPCE_CC, "1234567", "[91]12345A", 1, 0, 0, 0, 9, 55, "Test odd-numbered numeric, ending in non-digit",
                    "1101100110111011101011110001001111100110010011110101001"
                    "1101101110100111011100100001100010110001110011100101001"
                    "1101101100100100011111001101111101001101110011101101001"
                    "1101101000101111100110010001011111000110001011101001001"
                    "1101001000100111000111010001011000011000100011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
    };
    int data_size = sizeof(data) / sizeof(struct item);

    for (int i = 0; i < data_size; i++) {

        struct zint_symbol* symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        symbol->symbology = data[i].symbology;
        symbol->option_1 = data[i].option_1;
        symbol->option_2 = data[i].option_2;
        symbol->option_3 = data[i].option_3;
        int length = strlen(data[i].data);
        assert_zero(length >= 128, "i:%d length %d >= 128\n", i, length);
        strcpy(symbol->primary, data[i].data);

        int composite_length = strlen(data[i].composite);

        ret = ZBarcode_Encode(symbol, data[i].composite, composite_length);
        assert_equal(ret, data[i].ret, "i:%d ret %d != %d %s\n", i, ret, data[i].ret, symbol->errtxt);

        #ifdef TEST_ODD_NUMBERED_NUMERIC_GENERATE_EXPECTED
        test_helper_generate(symbol, ret, i, data[i].data, data[i].composite, data[i].option_1, data[i].option_2, data[i].option_3, data[i].comment);
        #else

        assert_equal(symbol->rows, data[i].expected_rows, "i:%d %s symbol->rows %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->rows, data[i].expected_rows, data[i].data);
        assert_equal(symbol->width, data[i].expected_width, "i:%d %s symbol->width %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->width, data[i].expected_width, data[i].data);

        if (ret == 0) {
            int width, row;
            ret = testUtilModulesCmp(symbol, data[i].expected, &width, &row);
            assert_zero(ret, "i:%d %s testUtilModulesCmp ret %d != 0 width %d row %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), ret, width, row, data[i].data);
        }
        #endif

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

static void test_ean128_cc_shift(void)
{
    testStart("");

    int ret;
    struct item {
        int symbology;
        unsigned char* data;
        unsigned char* composite;
        int option_1;
        int option_2;
        int option_3;
        int ret;

        int expected_rows;
        int expected_width;
        char* comment;
        unsigned char* expected;
    };
    // Verified manually with bwipp (tec-it.com seems to be off by 2 for top shifts > 1)
    struct item data[] = {
        /* 0*/ { BARCODE_EAN128_CC, "[91]A", "[21]A1B2C3D4E5F6G7H8", -1, 0, 0, 0, 6, 100, "CC-A alignment, bottom shift 10",
                    "1101001000110100001000001101101011110111110010010001101010000010010000011101110100010000111011001010"
                    "1101011000110101111001100001111010001101100010010000101111000011001101011100101100001000110011001010"
                    "1101011100100011001100111101011000101110000010110000101001100110011110011011110011001110110111001010"
                    "1101011110111000111011011001110010001011100010111000101011000011100110010000100000100010110111101010"
                    "0000000000001011011110000101000100011010011011000110010101110011101000100001011001110110011100010100"
                    "0000000000110100100001111010111011100101100100111001101010001100010111011110100110001001100011101011"
               },
        /* 1*/ { BARCODE_EAN128_CC, "[91]AB", "[21]A1B2C3D4E5F6G7H8", -1, 0, 0, 0, 6, 101, "CC-A alignment, top shift 1",
                    "01101001000110100001000001101101011110111110010010001101010000010010000011101110100010000111011001010"
                    "01101011000110101111001100001111010001101100010010000101111000011001101011100101100001000110011001010"
                    "01101011100100011001100111101011000101110000010110000101001100110011110011011110011001110110111001010"
                    "01101011110111000111011011001110010001011100010111000101011000011100110010000100000100010110111101010"
                    "00101101111000010100010001101001101100011001010111001110111010011101000100001010011110110011100010100"
                    "11010010000111101011101110010110010011100110101000110001000101100010111011110101100001001100011101011"
               },
        /* 2*/ { BARCODE_EAN128_CC, "[91]ABC", "[21]A1B2C3D4E5F6G7H8", -1, 0, 0, 0, 6, 112, "CC-A alignment, top shift 12",
                    "0000000000001101001000110100001000001101101011110111110010010001101010000010010000011101110100010000111011001010"
                    "0000000000001101011000110101111001100001111010001101100010010000101111000011001101011100101100001000110011001010"
                    "0000000000001101011100100011001100111101011000101110000010110000101001100110011110011011110011001110110111001010"
                    "0000000000001101011110111000111011011001110010001011100010111000101011000011100110010000100000100010110111101010"
                    "0010110111100001010001000110100110110001100101011100111011101001110111011100101000100001010011110110011100010100"
                    "1101001000011110101110111001011001001110011010100011000100010110001000100011010111011110101100001001100011101011"
               },
        /* 3*/ { BARCODE_EAN128_CC, "[91]ABCD", "[21]A1B2C3D4E5F6G7H8", -1, 0, 0, 0, 6, 123, "CC-A alignment, top shift 10",
                    "000000000011010010001101000010000011011010111101111100100100011010100000100100000111011101000100001110110010100000000000000"
                    "000000000011010110001101011110011000011110100011011000100100001011110000110011010111001011000010001100110010100000000000000"
                    "000000000011010111001000110011001111010110001011100000101100001010011001100111100110111100110011101101110010100000000000000"
                    "000000000011010111101110001110110110011100100010111000101110001010110000111001100100001000001000101101111010100000000000000"
                    "001011011110000101000100011010011011000110010101110011101110100111011101110010100111011101000100001001101101110011100010100"
                    "110100100001111010111011100101100100111001101010001100010001011000100010001101011000100010111011110110010010001100011101011"
               },
        /* 4*/ { BARCODE_EAN128_CC, "[91]ABCDEF", "[21]A1B2C3D4E5F6G7H8", -1, 0, 0, 0, 6, 145, "CC-A alignment, top shift 21",
                    "0000000000000000000001101001000110100001000001101101011110111110010010001101010000010010000011101110100010000111011001010000000000000000000000000"
                    "0000000000000000000001101011000110101111001100001111010001101100010010000101111000011001101011100101100001000110011001010000000000000000000000000"
                    "0000000000000000000001101011100100011001100111101011000101110000010110000101001100110011110011011110011001110110111001010000000000000000000000000"
                    "0000000000000000000001101011110111000111011011001110010001011100010111000101011000011100110010000100000100010110111101010000000000000000000000000"
                    "0010110111100001010001000110100110110001100101011100111011101001110111011100101001110111011100101110111001110101000100001001000110110011100010100"
                    "1101001000011110101110111001011001001110011010100011000100010110001000100011010110001000100011010001000110001010111011110110111001001100011101011"
               },
    };
    int data_size = sizeof(data) / sizeof(struct item);

    for (int i = 0; i < data_size; i++) {

        struct zint_symbol* symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        symbol->symbology = data[i].symbology;
        symbol->option_1 = data[i].option_1;
        symbol->option_2 = data[i].option_2;
        symbol->option_3 = data[i].option_3;
        int length = strlen(data[i].data);
        assert_zero(length >= 128, "i:%d length %d >= 128\n", i, length);
        strcpy(symbol->primary, data[i].data);

        int composite_length = strlen(data[i].composite);

        ret = ZBarcode_Encode(symbol, data[i].composite, composite_length);
        assert_equal(ret, data[i].ret, "i:%d ret %d != %d %s\n", i, ret, data[i].ret, symbol->errtxt);

        #ifdef TEST_EAN128_CC_SHIFT_GENERATE_EXPECTED
        test_helper_generate(symbol, ret, i, data[i].data, data[i].composite, data[i].option_1, data[i].option_2, data[i].option_3, data[i].comment);
        #else

        assert_equal(symbol->rows, data[i].expected_rows, "i:%d %s symbol->rows %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->rows, data[i].expected_rows, data[i].data);
        assert_equal(symbol->width, data[i].expected_width, "i:%d %s symbol->width %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->width, data[i].expected_width, data[i].data);

        if (ret == 0) {
            int width, row;
            ret = testUtilModulesCmp(symbol, data[i].expected, &width, &row);
            assert_zero(ret, "i:%d %s testUtilModulesCmp ret %d != 0 width %d row %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), ret, width, row, data[i].data);
        }
        #endif

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

static void test_ean128_cc_width(void)
{
    testStart("");

    int ret;
    struct item {
        unsigned char* data;
        unsigned char* composite;
        int ret;

        int expected_rows;
        int expected_width;
        char* comment;
    };
    // Verified manually with bwipp (except very large tests)
    struct item data[] = {
        /* 0*/ { "[91]1", "[02]13012345678909", 0, 11, 103, "" },
        /* 1*/ { "[91]12", "[02]13012345678909", 0, 20, 86, "" },
        /* 2*/ { "[91]123", "[02]13012345678909", 0, 11, 108, "" },
        /* 3*/ { "[91]123A", "[02]13012345678909", 0, 8, 120, "" },
        /* 4*/ { "[91]123A1", "[02]13012345678909", 0, 7, 137, "" },
        /* 5*/ { "[91]123A12", "[02]13012345678909", 0, 7, 141, "" },
        /* 6*/ { "[91]123A123", "[02]13012345678909", 0, 6, 154, "" },
        /* 7*/ { "[91]123A1234", "[02]13012345678909", 0, 6, 154, "" },
        /* 8*/ { "[91]123A1234A", "[02]13012345678909", 0, 5, 174, "" },
        /* 9*/ { "[91]123A1234A1", "[02]13012345678909", 0, 5, 188, "" },
        /*10*/ { "[91]123A1234A12", "[02]13012345678909", 0, 5, 205, "" },
        /*11*/ { "[00]123456789012345678", "[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[91]1234567890", 0, 32, 579, "With composite 2372 digits == max" },
        /*12*/ { "[00]123456789012345678", "[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[00]123456789012345678[91]12345678901", ZINT_ERROR_TOO_LONG, 0, 0, "With composite 2373 digits > max" },
    };
    int data_size = sizeof(data) / sizeof(struct item);

    for (int i = 0; i < data_size; i++) {

        struct zint_symbol* symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        symbol->symbology = BARCODE_EAN128_CC;
        symbol->option_1 = 3;
        int length = strlen(data[i].data);
        assert_zero(length >= 128, "i:%d length %d >= 128\n", i, length);
        strcpy(symbol->primary, data[i].data);

        int composite_length = strlen(data[i].composite);

        ret = ZBarcode_Encode(symbol, data[i].composite, composite_length);
        assert_equal(ret, data[i].ret, "i:%d ret %d != %d %s\n", i, ret, data[i].ret, symbol->errtxt);

        #ifdef TEST_EAN128_CC_WIDTH_GENERATE_EXPECTED
        printf("        /*%2d*/ { \"%s\", \"%s\", %s, %d, %d, \"%s\" },\n",
                i, data[i].data, data[i].composite, testUtilErrorName(ret), symbol->rows, symbol->width, data[i].comment);
        #else

        assert_equal(symbol->rows, data[i].expected_rows, "i:%d symbol->rows %d != %d (%s)\n", i, symbol->rows, data[i].expected_rows, data[i].data);
        assert_equal(symbol->width, data[i].expected_width, "i:%d symbol->width %d != %d (%s)\n", i, symbol->width, data[i].expected_width, data[i].data);
        #endif

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

// Test general-purpose data compaction
static void test_encodation_0(void)
{
    testStart("");

    int ret;
    struct item {
        int symbology;
        unsigned char* data;
        unsigned char* composite;
        int option_1;
        int option_2;
        int option_3;
        int ret;

        int expected_rows;
        int expected_width;
        char* comment;
        unsigned char* expected;
    };
    // Verified manually against tec-it.com (with noted exception) and/or bwipp
    struct item data[] = {
        /* 0*/ { BARCODE_UPCE_CC, "1234567", "[91]1", 1, 0, 0, 0, 9, 55, "Single numeric",
                    "1101100110111101110101111101010001000111100011110101001"
                    "1101101110111011000010001101110010101110000011100101001"
                    "1101101100110001011111011101111010000100100011101101001"
                    "1101101000100111011100111101110011110101110011101001001"
                    "1101001000110011001000000101110100011011110011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 1*/ { BARCODE_UPCE_CC, "1234567", "[91]12", 1, 0, 0, 0, 9, 55, "2 numerics",
                    "1101100110111011101011110001101111110110010011110101001"
                    "1101101110101110011110011001001100110000100011100101001"
                    "1101101100100101111010000001101000001110100011101101001"
                    "1101101000110010010001111101011000011001111011101001001"
                    "1101001000110010000010011001101001000110000011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 2*/ { BARCODE_UPCE_CC, "1234567", "[91]123", 1, 0, 0, 0, 9, 55, "Odd-numbered numeric",
                    "1101100110111011101011110001000111100010001011110101001"
                    "1101101110100111000111001001101000010001100011100101001"
                    "1101101100110101110001000001111010001000001011101101001"
                    "1101101000101000010000111101110011111100101011101001001"
                    "1101001000101100011100001101101110110010000011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 3*/ { BARCODE_UPCE_CC, "1234567", "[91]1234", 1, 0, 0, 0, 9, 55, "Even-numbered numeric",
                    "1101100110111011101011110001001111100110010011110101001"
                    "1101101110110001100110011001101100100000010011100101001"
                    "1101101100101111101001110001001110010111110011101101001"
                    "1101101000111000110101111101000011110101000011101001001"
                    "1101001000110101101110000001001101110001100011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 4*/ { BARCODE_UPCE_CC, "1234567", "[20]12[91]12", 1, 0, 0, 0, 9, 55, "Fixed len + even-numbered numeric",
                    "1101100110111111010001100101001111010001000011110101001"
                    "1101101110100011001110011001110010000010111011100101001"
                    "1101101100101111000101111001111010010001000011101101001"
                    "1101101000110011110100110001110100011111001011101001001"
                    "1101001000111100100101111001101001101110000011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 5*/ { BARCODE_UPCE_CC, "1234567", "[20]12[91]123", 1, 0, 0, 0, 9, 55, "Fixed len + odd-numbered numeric",
                    "1101100110111111010001100101001111010001000011110101001"
                    "1101101110100011001110011001100001101000100011100101001"
                    "1101101100101110010001111101111000010001010011101101001"
                    "1101101000110011110010110001100111101000110011101001001"
                    "1101001000100001101100100001110010001000111011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 6*/ { BARCODE_UPCE_CC, "1234567", "[91]A1234567C", 1, 0, 0, 0, 9, 55, "Alphanumeric followed by 7 digits and alphanumeric",
                    "1101100110111011101011110001011100010001100011110101001"
                    "1101101110111001001011100001110010001011100011100101001"
                    "1101101100111000001000110101001011111100111011101101001"
                    "1101101000111101011110001001001111000010001011101001001"
                    "1101001000101000001100011001010001000010000011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 7*/ { BARCODE_UPCE_CC, "1234567", "[91]A123456C", 1, 0, 0, 0, 9, 55, "Alphanumeric followed by 6 digits and alphanumeric",
                    "1101100110111011101011110001011100010001100011110101001"
                    "1101101110111001001011100001110010001011100011100101001"
                    "1101101100111000011001110101111110011010110011101101001"
                    "1101101000100110100011100001000100110111110011101001001"
                    "1101001000100000100010010001000001011011000011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 8*/ { BARCODE_UPCE_CC, "1234567", "[91]A12345B", 1, 0, 0, 0, 9, 55, "Alphanumeric followed by 5 digits and alphanumeric",
                    "1101100110111011101011110001011100010001100011110101001"
                    "1101101110110011101100100001110110001000110011100101001"
                    "1101101100101111100110111101111110100101110011101101001"
                    "1101101000111111010011000101100101100111111011101001001"
                    "1101001000110100010110000001000100010000001011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 9*/ { BARCODE_UPCE_CC, "1234567", "[91]A1234567", 1, 0, 0, 0, 9, 55, "Alphanumeric followed by 7 digits, terminating",
                    "1101100110111011101011110001011100010001100011110101001"
                    "1101101110111001001011100001111001000101111011100101001"
                    "1101101100111101011001100001111100110001010011101101001"
                    "1101101000111110010011100101110100011111001011101001001"
                    "1101001000110101100011100001101110110000100011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*10*/ { BARCODE_UPCE_CC, "1234567", "[91]A123456", 1, 0, 0, 0, 9, 55, "Alphanumeric followed by 6 digits, terminating",
                    "1101100110111011101011110001011100010001100011110101001"
                    "1101101110111001001011100001110010001011100011100101001"
                    "1101101100111100000100010101001111101000111011101101001"
                    "1101101000100110001001110001011100110111000011101001001"
                    "1101001000110110000010000101100110010010000011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*11*/ { BARCODE_UPCE_CC, "1234567", "[91]A12345", 1, 0, 0, 0, 9, 55, "Alphanumeric followed by 5 digits, terminating",
                    "1101100110111011101011110001011100010001100011110101001"
                    "1101101110111001001011100001111001000101111011100101001"
                    "1101101100111000101110001101001111001011110011101101001"
                    "1101101000111010011111101101100111110010001011101001001"
                    "1101001000111000100010011101000101110011100011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*12*/ { BARCODE_UPCE_CC, "1234567", "[91]A1234", 1, 0, 0, 0, 9, 55, "Alphanumeric followed by 4 digits, terminating",
                    "1101100110111011101011110001011100010001100011110101001"
                    "1101101110111001001011100001110110010011000011100101001"
                    "1101101100111100110111001101011111101011000011101101001"
                    "1101101000101111001100011001000001010001111011101001001"
                    "1101001000110111011000100001110000100100111011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*13*/ { BARCODE_UPCE_CC, "1234567", "[91]A123", 1, 0, 0, 0, 9, 55, "Alphanumeric followed by 3 digits, terminating",
                    "1101100110111011101011110001011100010001100011110101001"
                    "1101101110110011101100100001001000111001110011100101001"
                    "1101101100100101111100110001011101001111100011101101001"
                    "1101101000110111110000100101001111000001010011101001001"
                    "1101001000110010100001100001110010001101111011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*14*/ { BARCODE_UPCE_CC, "1234567", "[91]A123[10]C", 1, 0, 0, 0, 9, 55, "Alphanumeric followed by 3 digits, 2-digit AI (3 numerics including FNC1) and alphanumeric",
                    "1101100110111011101011110001011100010001100011110101001"
                    "1101101110111001001011100001100010101100000011100101001"
                    "1101101100110100001110010001110111101111101011101101001"
                    "1101101000101110001100111001001110010011000011101001001"
                    "1101001000101100111001100001100100111001111011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*15*/ { BARCODE_UPCE_CC, "1234567", "[91]A12[10]C", 1, 0, 0, 0, 9, 55, "Alphanumeric followed by 2 digits, 2-digit AI (3 numerics including FNC1) and alphanumeric",
                    "1101100110111011101011110001011100010001100011110101001"
                    "1101101110110011101100100001000001001010000011100101001"
                    "1101101100111110110100001001111110111101101011101101001"
                    "1101101000100110010000011101100111000010111011101001001"
                    "1101001000110110001000100001101100100000010011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*16*/ { BARCODE_UPCE_CC, "1234567", "[91]A12[10]1", 1, 0, 0, 0, 9, 55, "Alphanumeric followed by 2 digits, 2-digit AI (3 numerics including FNC1) and 1 digit, terminating",
                    "1101100110111011101011110001011100010001100011110101001"
                    "1101101110111100100101111001000010000110110011100101001"
                    "1101101100100100011110010001001110010111110011101101001"
                    "1101101000111101111010111101000100100111100011101001001"
                    "1101001000111101110010011001001000001101100011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*17*/ { BARCODE_UPCE_CC, "1234567", "[91]A1[10]1", 1, 0, 0, 0, 9, 55, "Alphanumeric followed by 1 digit, 2-digit AI (3 numerics including FNC1) and 1 digit, terminating; **NOT SAME** as tec-it.com, which does not switch to numeric mode after 'A'; same as bwipp",
                    "1101100110111011101011110001011100010001100011110101001"
                    "1101101110110010011001110001110010010011100011100101001"
                    "1101101100111110110001001001010111111000111011101101001"
                    "1101101000111011110010011101011101100111000011101001001"
                    "1101001000110100001101110001000001010100000011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*18*/ { BARCODE_UPCE_CC, "1234567", "[91]A[10]1", 1, 0, 0, 0, 9, 55, "Alphanumeric followed by 2-digit AI (3 numerics including FNC1) and 1 digit, terminating",
                    "1101100110111011101011110001101110001000111011110101001"
                    "1101101110111001100010110001100010000010110011100101001"
                    "1101101100101111110010011001001101111110110011101101001"
                    "1101101000100100100111100001110010111110100011101001001"
                    "1101001000101110001111011001111011000100111011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*19*/ { BARCODE_UPCE_CC, "1234567", "[91]a1234ABCDEFGb", 1, 0, 0, 0, 12, 55, "ISO-646 followed by 11 non-ISO-646 non-terminating, starting 4 digits",
                    "1110111010101100111111011001011111000100111011011011101"
                    "1110011010101100001100111101011101000001100011011011001"
                    "1111011010100100001001000001001110011110110011011010001"
                    "1111001010100111100100111101111110110001011011010010001"
                    "1110001010101100110011110001000111001101110011010110001"
                    "1100001010101000001000100001000010111001110011010111001"
                    "1100011010111011101111000101111100110010001011010111101"
                    "1100010010100111010000001101100001010111110011010011101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*20*/ { BARCODE_UPCE_CC, "1234567", "[91]a1234ABCDEFb", 1, 0, 0, 0, 12, 55, "ISO-646 followed by 10 non-ISO-646 non-terminating, starting 4 digits",
                    "1110111010101100111111011001011111000100111011011011101"
                    "1110011010101100001100111101011101000001100011011011001"
                    "1111011010100100001001000001001110011110110011011010001"
                    "1111001010100111100100111101111110110001011011010010001"
                    "1110001010111101101011111101100111100010110011010110001"
                    "1100001010110110001000001001111001011011111011010111001"
                    "1100011010110111110010111101111100011101101011010111101"
                    "1100010010101110000010001101101110000100111011010011101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*21*/ { BARCODE_UPCE_CC, "1234567", "[91]a1234ABCDEF", 1, 0, 0, 0, 11, 55, "ISO-646 followed by 10 non-ISO-646 terminating, starting 4 digits",
                    "1110110110101100111111011001011111000100111011100010101"
                    "1110010110101100001100111101011101000001100011000010101"
                    "1100010110100100001001000001001110011110110011000110101"
                    "1100010100100111100100111101001001111001000011000100101"
                    "1100110100110011111000101001011110000011011011100100101"
                    "1101110100111011101001000001110001000100111011110100101"
                    "1101100100110001011100100001110001011000100011110101101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*22*/ { BARCODE_UPCE_CC, "1234567", "[91]a1234ABCDEb", 1, 0, 0, 0, 11, 55, "ISO-646 followed by 9 non-ISO-646 non-terminating, starting 4 digits",
                    "1110110110101100111111011001011111000100111011100010101"
                    "1110010110101100000010011101110111100010111011000010101"
                    "1100010110110010011100111101011000111100111011000110101"
                    "1100010100111101000110001101001011110010000011000100101"
                    "1100110100100001100001011101011110011011000011100100101"
                    "1101110100111010101110000001100101001100000011110100101"
                    "1101100100111100100100100001110001100111010011110101101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*23*/ { BARCODE_UPCE_CC, "1234567", "[91]a1234ABCDE", 1, 0, 0, 0, 10, 55, "ISO-646 followed by 9 non-ISO-646 terminating, starting 4 digits",
                    "1100100010111001101001100001111001111010100011100101101"
                    "1110100010111110010010001101110110011100100011000101101"
                    "1110110010100010100011110001001111000011011011000101001"
                    "1100110010100000100010000101110110001000110011001101001"
                    "1101110010100010001111000101000011001111101011011101001"
                    "1101111010111110101111011001010001000001111011011001001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*24*/ { BARCODE_UPCE_CC, "1234567", "[91]a123ABCDEFGb", 1, 0, 0, 0, 12, 55, "ISO-646 followed by 10 non-ISO-646 non-terminating, starting 3 digits (5 alphanumerics rule applies)",
                    "1110111010101100111111011001011111000100111011011011101"
                    "1110011010100100000010111101101111001100111011011011001"
                    "1111011010111101100100011101001011110111100011011010001"
                    "1111001010111110001010110001101001110000100011010010001"
                    "1110001010110011100010111001101110100001110011010110001"
                    "1100001010110100111101111101100110100010000011010111001"
                    "1100011010101000111111011101110110111001000011010111101"
                    "1100010010100011110110110001000111100100001011010011101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*25*/ { BARCODE_UPCE_CC, "1234567", "[91]aABCDEF12345b", 1, 0, 0, 0, 12, 55, "ISO-646 followed by 11 non-ISO-646 non-terminating, starting 6 letters",
                    "1110111010101100111111011001011111000100111011011011101"
                    "1110011010101100000110111101111010011111011011011011001"
                    "1111011010100001001100001101010010000010000011011010001"
                    "1111001010101111110011011101110100110100000011010010001"
                    "1110001010111110101110100001001101011100000011010110001"
                    "1100001010100010011000011001100001000010011011010111001"
                    "1100011010111000101110110001110100111000011011010111101"
                    "1100010010101100010011100001001110000110111011010011101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*26*/ { BARCODE_UPCE_CC, "1234567", "[91]aABCDEF1234b", 1, 0, 0, 0, 12, 55, "ISO-646 followed by 10 non-ISO-646 non-terminating, starting 6 letters",
                    "1110111010101100111111011001011111000100111011011011101"
                    "1110011010101100000110111101111010011111011011011011001"
                    "1111011010100001001100001101010010000010000011011010001"
                    "1111001010101111110011011101110100110100000011010010001"
                    "1110001010111111010110000101110001110101111011010110001"
                    "1100001010111010001011100001100001001000011011010111001"
                    "1100011010101001111010000001111100100010011011010111101"
                    "1100010010110100111111010001011001000001110011010011101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*27*/ { BARCODE_UPCE_CC, "1234567", "[91]aABCDE12345b", 1, 0, 0, 0, 12, 55, "ISO-646 followed by 10 non-ISO-646 non-terminating, starting 5 letters",
                    "1110111010101100111111011001011111000100111011011011101"
                    "1110011010101100000110111101111010011111011011011011001"
                    "1111011010100001001100001101010010000010000011011010001"
                    "1111001010111001101110010001101000011100010011010010001"
                    "1110001010111011011111101001011101000000110011010110001"
                    "1100001010100011000111011001110000100101110011010111001"
                    "1100011010110100001110100001110110011100010011010111101"
                    "1100010010111111010111001101110111100101110011010011101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*28*/ { BARCODE_UPCE_CC, "1234567", "[91]aABCDE12345", 1, 0, 0, 0, 11, 55, "ISO-646 followed by 10 non-ISO-646 terminating, starting 5 letters",
                    "1110110110101100111111011001011111000100111011100010101"
                    "1110010110101100000110111101111010011111011011000010101"
                    "1100010110100001001100001101010010000010000011000110101"
                    "1100010100101100101111110001101000001111011011000100101"
                    "1100110100101000000100111101010000011011111011100100101"
                    "1101110100110011101100010001110001100101100011110100101"
                    "1101100100111100110110000101111110101001110011110101101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*29*/ { BARCODE_UPCE_CC, "1234567", "[91]aABCD012345b", 1, 0, 0, 0, 12, 55, "ISO-646 followed by 10 non-ISO-646 non-terminating, starting 4 letters + numeric",
                    "1110111010101100111111011001011111000100111011011011101"
                    "1110011010101100000110111101111010011111011011011011001"
                    "1111011010100001001100001101110101000111000011011010001"
                    "1111001010110010111000100001001101011111100011010010001"
                    "1110001010110010101111100001000010010111100011010110001"
                    "1100001010111101001011110001001011110011110011010111001"
                    "1100011010111000101110110001110001011000100011010111101"
                    "1100010010111101101011111101000000110101110011010011101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*30*/ { BARCODE_UPCE_CC, "1234567", "[91]aABCD012345", 1, 0, 0, 0, 10, 55, "ISO-646 followed by 10 non-ISO-646 terminating, starting 4 letters + numeric",
                    "1100100010111001101001100001111001111010100011100101101"
                    "1110100010111001001101000001111010011011000011000101101"
                    "1110110010100001100100011101110101000111111011000101001"
                    "1100110010111101100100011101110001011011110011001101001"
                    "1101110010111000100011010001110010110001000011011101001"
                    "1101111010111000110101111101000110010001110011011001001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*31*/ { BARCODE_UPCE_CC, "1234567", "[91]aABCDE1234b", 1, 0, 0, 0, 11, 55, "ISO-646 followed by 9 non-ISO-646 non-terminating, starting 5 letters",
                    "1110110110101100111111011001011111000100111011100010101"
                    "1110010110100010110001111101011000001101111011000010101"
                    "1100010110111101101100111101000110000110100011000110101"
                    "1100010100111001101110010001101000111000010011000100101"
                    "1100110100100111001110111101001111101100100011100100101"
                    "1101110100110011000100000101100101001100000011110100101"
                    "1101100100111100001010000101100111011111010011110101101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*32*/ { BARCODE_UPCE_CC, "1234567", "[91]aABCDE1234", 1, 0, 0, 0, 10, 55, "ISO-646 followed by 9 non-ISO-646 terminating, starting 5 letters",
                    "1100100010111001101001100001111001111010100011100101101"
                    "1110100010111001001101000001111010011011000011000101101"
                    "1110110010100001100100011101110101111110011011000101001"
                    "1100110010100100000100001001111011010011100011001101001"
                    "1101110010111101011001100001111010011000011011011101001"
                    "1101111010111010111110100001111100101110010011011001001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*33*/ { BARCODE_UPCE_CC, "1234567", "[91]aABCDE123", 1, 0, 0, 0, 10, 55, "ISO-646 followed by 8 non-ISO-646 terminating, starting 5 letters",
                    "1100100010111001101001100001111001111010100011100101101"
                    "1110100010111001001101000001111010011011000011000101101"
                    "1110110010100001100100011101110101111110011011000101001"
                    "1100110010110011000100000101010000010001000011001101001"
                    "1101110010111000111011110101100010011101000011011101001"
                    "1101111010101101100001111001010000011011111011011001001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
    };
    int data_size = sizeof(data) / sizeof(struct item);

    for (int i = 0; i < data_size; i++) {

        struct zint_symbol* symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        symbol->symbology = data[i].symbology;
        symbol->option_1 = data[i].option_1;
        symbol->option_2 = data[i].option_2;
        symbol->option_3 = data[i].option_3;
        int length = strlen(data[i].data);
        assert_zero(length >= 128, "i:%d length %d >= 128\n", i, length);
        strcpy(symbol->primary, data[i].data);

        int composite_length = strlen(data[i].composite);

        ret = ZBarcode_Encode(symbol, data[i].composite, composite_length);
        assert_equal(ret, data[i].ret, "i:%d ret %d != %d %s\n", i, ret, data[i].ret, symbol->errtxt);

        #ifdef TEST_ENCODATION_0_GENERATE_EXPECTED
        test_helper_generate(symbol, ret, i, data[i].data, data[i].composite, data[i].option_1, data[i].option_2, data[i].option_3, data[i].comment);
        #else

        assert_equal(symbol->rows, data[i].expected_rows, "i:%d %s symbol->rows %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->rows, data[i].expected_rows, data[i].data);
        assert_equal(symbol->width, data[i].expected_width, "i:%d %s symbol->width %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->width, data[i].expected_width, data[i].data);

        if (ret == 0) {
            int width, row;
            ret = testUtilModulesCmp(symbol, data[i].expected, &width, &row);
            assert_zero(ret, "i:%d %s testUtilModulesCmp ret %d != 0 width %d row %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), ret, width, row, data[i].data);
        }
        #endif

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

static void test_encodation_10(void)
{
    testStart("");

    int ret;
    struct item {
        int symbology;
        unsigned char* data;
        unsigned char* composite;
        int option_1;
        int option_2;
        int option_3;
        int ret;

        int expected_rows;
        int expected_width;
        char* comment;
        unsigned char* expected;
    };
    // Verified manually against bwipp (and with noted exceptions, tec-it.com)
    struct item data[] = {
        /* 0*/ { BARCODE_UPCE_CC, "1234567", "[11]201001[10]AB1234", 1, 0, 0, 0, 9, 55, "Mode '10' date + even-numbered numeric lot, 1st fit, alphanumeric latch and padding",
                    "1101100110101110000100011001001000100111100011110101001"
                    "1101101110110010100011000001100110010000001011100101001"
                    "1101101100111011011101000001110100111000110011101101001"
                    "1101101000111110101110000101000110000010111011101001001"
                    "1101001000101100011010000001100110001010000011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 1*/ { BARCODE_UPCE_CC, "1234567", "[11]201001[10]AB12345", 1, 0, 0, 0, 9, 55, "Mode '10' date + odd-numbered numeric lot, 1st fit, 7-bit final, no alphanumeric latch or padding",
                    "1101100110101110000100011001001000100111100011110101001"
                    "1101101110110010100011000001100110010000001011100101001"
                    "1101101100111011011101000001111010000110110011101101001"
                    "1101101000100111101110111001000001001001111011101001001"
                    "1101001000101100011100011001111101010111110011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 2*/ { BARCODE_UPCE_CC, "1234567", "[11]201001", 1, 0, 0, 0, 9, 55, "Mode '10' date, no lot or other data",
                    "1101100110101110000100011001001000100111100011110101001"
                    "1101101110110010100110000001010000010010000011100101001"
                    "1101101100111011000001110101000001011111011011101101001"
                    "1101101000101000100111100001011110000101000011101001001"
                    "1101001000111100111001001101000110001100001011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 3*/ { BARCODE_UPCE_CC, "1234567", "[11]201001[20]12", 1, 0, 0, 0, 9, 55, "Mode '10' date, no lot, other data",
                    "1101100110101110000100011001100100010011111011110101001"
                    "1101101110101110111010000001000101100001100011100101001"
                    "1101101100111010011000100001111110101001110011101101001"
                    "1101101000111111010110010001010100001111000011101001001"
                    "1101001000111001011011110001100001101010000011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 4*/ { BARCODE_UPCE_CC, "1234567", "[10]1234", 1, 0, 0, 0, 9, 55, "Mode '10' no date, numeric lot, no other data; **NOT SAME** as tec-it.com; same as bwipp",
                    "1101100110100111000110011101110011100101111011110101001"
                    "1101101110110100100110000001111011100101100011100101001"
                    "1101101100100111101011110001111110011010110011101101001"
                    "1101101000100110110000111101101010111110000011101001001"
                    "1101001000110100111000111101000001000010001011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 5*/ { BARCODE_UPCE_CC, "1234567", "[10]ABCD", 1, 0, 0, 0, 9, 55, "Mode '10' no date, alphanumeric lot, no other data; **NOT SAME** as tec-it.com; same as bwipp",
                    "1101100110101111001100011001111000010111101011110101001"
                    "1101101110100010111100111101100110010010000011100101001"
                    "1101101100100111101001111001000010111110110011101101001"
                    "1101101000110001100010111101011101000001100011101001001"
                    "1101001000111010000101110001101000011101111011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
    };
    int data_size = sizeof(data) / sizeof(struct item);

    for (int i = 0; i < data_size; i++) {

        struct zint_symbol* symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        symbol->symbology = data[i].symbology;
        symbol->option_1 = data[i].option_1;
        symbol->option_2 = data[i].option_2;
        symbol->option_3 = data[i].option_3;
        int length = strlen(data[i].data);
        assert_zero(length >= 128, "i:%d length %d >= 128\n", i, length);
        strcpy(symbol->primary, data[i].data);

        int composite_length = strlen(data[i].composite);

        ret = ZBarcode_Encode(symbol, data[i].composite, composite_length);
        assert_equal(ret, data[i].ret, "i:%d ret %d != %d %s\n", i, ret, data[i].ret, symbol->errtxt);

        #ifdef TEST_ENCODATION_10_GENERATE_EXPECTED
        test_helper_generate(symbol, ret, i, data[i].data, data[i].composite, data[i].option_1, data[i].option_2, data[i].option_3, data[i].comment);
        #else

        assert_equal(symbol->rows, data[i].expected_rows, "i:%d %s symbol->rows %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->rows, data[i].expected_rows, data[i].data);
        assert_equal(symbol->width, data[i].expected_width, "i:%d %s symbol->width %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->width, data[i].expected_width, data[i].data);

        if (ret == 0) {
            int width, row;
            ret = testUtilModulesCmp(symbol, data[i].expected, &width, &row);
            assert_zero(ret, "i:%d %s testUtilModulesCmp ret %d != 0 width %d row %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), ret, width, row, data[i].data);
        }
        #endif

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

static void test_encodation_11(void)
{
    testStart("");

    int ret;
    struct item {
        int symbology;
        unsigned char* data;
        unsigned char* composite;
        int option_1;
        int option_2;
        int option_3;
        int ret;

        int expected_rows;
        int expected_width;
        char* comment;
        unsigned char* expected;
    };
    // Verified manually against tec-it.com (with noted exception) (bwipp 2019-10-13 has some issues with encodation 11 so not used)
    struct item data[] = {
        /* 0*/ { BARCODE_UPCE_CC, "1234567", "[90]A", 1, 0, 0, 0, 9, 55, "Mode '11', letter prefix only",
                    "1101100110100111100000101001110110010011111011110101001"
                    "1101101110110001000101100001100011010010000011100101001"
                    "1101101100110001000011100101001110010011111011101101001"
                    "1101101000110100111111010001000001100010111011101001001"
                    "1101001000110001010110000001001100011100011011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 1*/ { BARCODE_UPCE_CC, "1234567", "[90]1A", 1, 0, 0, 0, 9, 55, "Mode '11', 1 digit letter prefix only",
                    "1101100110100111100000101001011001000111000011110101001"
                    "1101101110101101111011100001011100111010000011100101001"
                    "1101101100111000010011010001101011100001000011101101001"
                    "1101101000101100011001111001000111110110001011101001001"
                    "1101001000100010001110111001001100111001100011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 2*/ { BARCODE_UPCE_CC, "1234567", "[90]12A", 1, 0, 0, 0, 9, 55, "Mode '11', 2 digit letter prefix only",
                    "1101100110100111100000101001011001101111000011110101001"
                    "1101101110110001010000110001110111010000010011100101001"
                    "1101101100111110011010000101111100100110111011101101001"
                    "1101101000110111100101100001000110010001110011101001001"
                    "1101001000100100111101111001000010011100111011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 3*/ { BARCODE_UPCE_CC, "1234567", "[90]123A", 1, 0, 0, 0, 9, 55, "Mode '11', 3 digit letter prefix only",
                    "1101100110100111100000101001100110101111000011110101001"
                    "1101101110100001001010000001101110110100000011100101001"
                    "1101101100111110110000101001100101111100111011101101001"
                    "1101101000111001110010111101110101111110011011101001001"
                    "1101001000110011000001001001101110111001100011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 4*/ { BARCODE_UPCE_CC, "1234567", "[90]A1234", 1, 0, 0, 0, 9, 55, "Mode '11', even-numbered numeric [90]",
                    "1101100110100111100000101001110110010011111011110101001"
                    "1101101110110001000100110001101000100110000011100101001"
                    "1101101100111100010111011101100011011110100011101101001"
                    "1101101000101001100001111101001001110111111011101001001"
                    "1101001000111011100100100001101000010000011011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 5*/ { BARCODE_UPCE_CC, "1234567", "[90]A12345", 1, 0, 0, 0, 9, 55, "Mode '11', odd-numbered numeric [90]",
                    "1101100110100111100000101001110110010011111011110101001"
                    "1101101110110001000100110001110100010011100011100101001"
                    "1101101100101000011110010001110100011010000011101101001"
                    "1101101000101110000010110001001111000100001011101001001"
                    "1101001000111000100010111001011000111100111011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 6*/ { BARCODE_UPCE_CC, "1234567", "[90]1ABC4", 1, 0, 0, 0, 9, 55, "Mode '11', alpha [90]",
                    "1101100110100111110110010001001110011000111011110101001"
                    "1101101110110110011000011001010011000011000011100101001"
                    "1101101100101000111100000101110000100110010011101101001"
                    "1101101000100111101100001101000001001011110011101001001"
                    "1101001000110111001101000001000010000110011011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 7*/ { BARCODE_UPCE_CC, "1234567", "[90]1AB34", 1, 0, 0, 0, 9, 55, "Mode '11', alphanumeric [90] (letters <= numbers)",
                    "1101100110100011100110111001011111101110010011110101001"
                    "1101101110110000110100010001100101100011100011100101001"
                    "1101101100110010000011100101101100011110001011101101001"
                    "1101101000100111011101111001110011111101001011101001001"
                    "1101001000111100010101111001110011000100011011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 8*/ { BARCODE_UPCE_CC, "1234567", "[90]1AB.D", 1, 0, 0, 0, 9, 55, "Mode '11', alphanumeric [90]",
                    "1101100110100011100110111001011111101110010011110101001"
                    "1101101110110000110100010001000010100000100011100101001"
                    "1101101100101110100001111101110111000111101011101101001"
                    "1101101000100011001000011101011011001111000011101001001"
                    "1101001000110010000010001101101000111011110011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /* 9*/ { BARCODE_UPCE_CC, "1234567", "[90]1AB+D", 1, 0, 0, 0, 9, 55, "Mode '11', ISO-646 [90]",
                    "1101100110100011100110111001011111101110010011110101001"
                    "1101101110110000110100010001101000111001111011100101001"
                    "1101101100111110110100001001100101111101110011101101001"
                    "1101101000100001011001111101000111100001001011101001001"
                    "1101001000100010000001000101101100100010000011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*10*/ { BARCODE_UPCE_CC, "1234567", "[90]1A+BD", 1, 0, 0, 0, 9, 55, "Mode '11', immediate ISO-646 [90]",
                    "1101100110100011100110111001011111101110010011110101001"
                    "1101101110101111000111100101111011101011000011100101001"
                    "1101101100101000011110010001001111001011110011101101001"
                    "1101101000100111110011001001101111101001000011101001001"
                    "1101001000110110001000001001101110001100001011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*11*/ { BARCODE_UPCE_CC, "1234567", "[90]1AB#D", 1, 0, 0, ZINT_ERROR_INVALID_DATA, 0, 0, "Mode '11', invalid char [90]", "" },
        /*12*/ { BARCODE_UPCE_CC, "1234567", "[90]A12345[21]AB", 1, 0, 0, 0, 10, 55, "Mode '11', numeric [90], with [21]",
                    "1100100010111000111011011001001100011000010011100101101"
                    "1110100010101000001111001001111100110100010011000101101"
                    "1110110010101110010000110001100111000100111011000101001"
                    "1100110010100001100011000101010000010000100011001101001"
                    "1101110010111101101110011001101000001110010011011101001"
                    "1101111010100001000100111101001010001111000011011001001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*13*/ { BARCODE_UPCE_CC, "1234567", "[90]1ABC4[21]AB12", 1, 0, 0, 0, 10, 55, "Mode '11', alpha [90], with [21]",
                    "1100100010100011100111101101011000111000110011100101101"
                    "1110100010111010000110001001110100011000001011000101101"
                    "1110110010100001100100011101110111111010100011000101001"
                    "1100110010110100100001100001001001111011110011001101001"
                    "1101110010111100101100011001111101001000110011011101001"
                    "1101111010101110100001100001011101100111000011011001001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*14*/ { BARCODE_UPCE_CC, "1234567", "[90]1AB.D[21]AB", 1, 0, 0, 0, 10, 55, "Mode '11', alphanumeric [90], with [21]",
                    "1100100010110011100011000101110001110110011011100101101"
                    "1110100010111101101100100001110010011100011011000101101"
                    "1110110010110000110101111001011100110001110011000101001"
                    "1100110010110011101100000101101001110001111011001101001"
                    "1101110010110110011110001001100110001111001011011101001"
                    "1101111010100110111011111001111001000111101011011001001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*15*/ { BARCODE_UPCE_CC, "1234567", "[90]1AB+D[21]A.", 1, 0, 0, 0, 10, 55, "Mode '11', ISO-646 [90], with [21]; **NOT SAME** as tec-it.com, which probably includes '21' in 5 alphanumeric count",
                    "1100100010110011100011000101110001110110011011100101101"
                    "1110100010111101101100100001110101110001100011000101101"
                    "1110110010100000111000101101110010111110001011000101001"
                    "1100110010110111011000100001110110000100011011001101001"
                    "1101110010111000010001100101111010001110111011011101001"
                    "1101111010110001100100111101111101001110010011011001001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*16*/ { BARCODE_UPCE_CC, "1234567", "[90]1A+BD[21]A12", 1, 0, 0, 0, 11, 55, "Mode '11', immediate ISO-646 [90], with [21]; tec-it.com same, probably since have 5 alphanumerics with or without '21'",
                    "1110110110111100011000110101100111101011111011100010101"
                    "1110010110100100001101111101000100010111100011000010101"
                    "1100010110100010100000100001110011101000001011000110101"
                    "1100010100100001011111011001010111110110000011000100101"
                    "1100110100100000011010011101110001111101011011100100101"
                    "1101110100100000100011011001100001101001000011110100101"
                    "1101100100111010111100011101011110100011110011110101101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*17*/ { BARCODE_UPCE_CC, "1234567", "[90]A12[8004]12", 1, 0, 0, 0, 9, 55, "Mode '11', numeric [90], with [8004]",
                    "1101100110110001111010011001000001101101111011110101001"
                    "1101101110110100100000110001011100111101100011100101001"
                    "1101101100110100000111010001100100001110001011101101001"
                    "1101101000101110001001100001101111001011000011101001001"
                    "1101001000100100110000110001101000010000110011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*18*/ { BARCODE_UPCE_CC, "1234567", "[90]AB[8004]12", 1, 0, 0, 0, 9, 55, "Mode '11', alpha [90], with [8004]",
                    "1101100110100111110011001001100101111110010011110101001"
                    "1101101110101000100000100001011000111001100011100101001"
                    "1101101100100111110100011101111000100001001011101101001"
                    "1101101000101000110011111001010001110111111011101001001"
                    "1101001000110011101100001001111011100001011011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*19*/ { BARCODE_UPCE_CC, "1234567", "[90]A.[8004]12", 1, 0, 0, 0, 9, 55, "Mode '11', alphanumeric [90], with [8004]",
                    "1101100110110111110011001101111110100110001011110101001"
                    "1101101110100010001001000001100101100011100011100101001"
                    "1101101100100001011110001001100001100111101011101101001"
                    "1101101000100111100100001001011110001110111011101001001"
                    "1101001000101101111100111101100011010100000011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*20*/ { BARCODE_UPCE_CC, "1234567", "[90]A+[8004]12", 1, 0, 0, 0, 9, 55, "Mode '11', ISO-646 [90], with [8004]",
                    "1101100110110111110011001101111110100110001011110101001"
                    "1101101110111100110010111001001110001110100011100101001"
                    "1101101100111001011100011001101001110000010011101101001"
                    "1101101000101100000110111101000110001001110011101001001"
                    "1101001000100011000110000101111100110101111011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*21*/ { BARCODE_UPCE_CC, "1234567", "[90]12A1[10]12", 1, 0, 0, 0, 9, 55, "Mode '11', numeric [90], other data",
                    "1101100110100111100000101001011001101111000011110101001"
                    "1101101110111000101101111001110110100001100011100101001"
                    "1101101100111100001010001001110000100110010011101101001"
                    "1101101000100000111010001101001000011011111011101001001"
                    "1101001000100010111011100001100001101100011011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*22*/ { BARCODE_UPCE_CC, "1234567", "[90]123AB[10]12", 1, 0, 0, 0, 9, 55, "Mode '11', alpha [90], other data",
                    "1101100110100111110110010001100011101000111011110101001"
                    "1101101110111001110100000101100110001100110011100101001"
                    "1101101100110110011111001101111010100001000011101101001"
                    "1101101000100110111011111001000001100010111011101001001"
                    "1101001000110010010000110001011110011110010011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*23*/ { BARCODE_UPCE_CC, "1234567", "[90]123AB.D[10]12", 1, 0, 0, 0, 10, 55, "Mode '11', alphanumeric [90], other data",
                    "1100100010100011000110000101110110000100110011100101101"
                    "1110100010101101111100100001011111001001110011000101101"
                    "1110110010101111100110001001101111001000110011000101001"
                    "1100110010110110111100111101010111110111110011001101001"
                    "1101110010101000011111001101111100110100100011011101001"
                    "1101111010110011111100110101110101111101000011011001001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*24*/ { BARCODE_UPCE_CC, "1234567", "[90]123AB+D[10]12", 1, 0, 0, 0, 10, 55, "Mode '11', ISO-646 [90], other data",
                    "1100100010100011000110000101110110000100110011100101101"
                    "1110100010101101111100100001100100111000001011000101101"
                    "1110110010100010110011111001000110100111000011000101001"
                    "1100110010111011100100001001110010110011110011001101001"
                    "1101110010111101000000100101111001101101000011011101001"
                    "1101111010101101000001110001101111100001001011011001001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*25*/ { BARCODE_UPCE_CC, "1234567", "[90]A123[21]AB[91]A123", 1, 0, 0, 0, 12, 55, "Mode '11', numeric [90], with [21], other data",
                    "1110111010100001100111110101100010011110110011011011101"
                    "1110011010111111011011100101001110110011100011011011001"
                    "1111011010101000111011100001001011110011110011011010001"
                    "1111001010111100100100100001111010111101111011010010001"
                    "1110001010101100110111100001100111110100010011010110001"
                    "1100001010100110111001100001111101111001011011010111001"
                    "1100011010111110001011011101101001110010000011010111101"
                    "1100010010111001110010111101101111101100110011010011101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
        /*26*/ { BARCODE_UPCE_CC, "1234567", "[90]AB[8004]1[10]12", 1, 0, 0, 0, 9, 55, "Mode '11', alpha [90], with [8004], other data",
                    "1101100110100111110011001001100101111110010011110101001"
                    "1101101110101000100000100001101100011100111011100101001"
                    "1101101100100010011111011001100110011110001011101101001"
                    "1101101000110101100111111001000011001000111011101001001"
                    "1101001000100010000010100001101001111011111011101001101"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0010000000000000000000000000000000000000000000000000001"
                    "0001000000000000000000000000000000000000000000000000010"
                    "0001010010011011110101000110111001000010100100010101010"
               },
    };
    int data_size = sizeof(data) / sizeof(struct item);

    for (int i = 0; i < data_size; i++) {

        struct zint_symbol* symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        symbol->symbology = data[i].symbology;
        symbol->option_1 = data[i].option_1;
        symbol->option_2 = data[i].option_2;
        symbol->option_3 = data[i].option_3;
        int length = strlen(data[i].data);
        assert_zero(length >= 128, "i:%d length %d >= 128\n", i, length);
        strcpy(symbol->primary, data[i].data);

        int composite_length = strlen(data[i].composite);

        ret = ZBarcode_Encode(symbol, data[i].composite, composite_length);
        assert_equal(ret, data[i].ret, "i:%d ret %d != %d %s\n", i, ret, data[i].ret, symbol->errtxt);

        #ifdef TEST_ENCODATION_11_GENERATE_EXPECTED
        test_helper_generate(symbol, ret, i, data[i].data, data[i].composite, data[i].option_1, data[i].option_2, data[i].option_3, data[i].comment);
        #else

        assert_equal(symbol->rows, data[i].expected_rows, "i:%d %s symbol->rows %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->rows, data[i].expected_rows, data[i].data);
        assert_equal(symbol->width, data[i].expected_width, "i:%d %s symbol->width %d != %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), symbol->width, data[i].expected_width, data[i].data);

        if (ret == 0) {
            int width, row;
            ret = testUtilModulesCmp(symbol, data[i].expected, &width, &row);
            assert_zero(ret, "i:%d %s testUtilModulesCmp ret %d != 0 width %d row %d (%s)\n", i, testUtilBarcodeName(data[i].symbology), ret, width, row, data[i].data);
        }
        #endif

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

// #181 Christian Hartlage OSS-Fuzz
static void test_fuzz(void)
{
    testStart("");

    int ret;
    struct item {
        int symbology;
        unsigned char* data;
        int length;
        unsigned char* composite;
        int ret;
    };
    // s/\/\*[ 0-9]*\*\//\=printf("\/*%2d*\/", line(".") - line("'<"))
    struct item data[] = {
        /* 0*/ { BARCODE_EANX_CC, "+123456789012345678", -1, "[21]A12345678", ZINT_ERROR_TOO_LONG },
        /* 1*/ { BARCODE_UPCA_CC, "+123456789012345678", -1, "[21]A12345678", ZINT_ERROR_TOO_LONG },
        /* 2*/ { BARCODE_UPCE_CC, "+123456789012345678", -1, "[21]A12345678", ZINT_ERROR_TOO_LONG },
        /* 3*/ { BARCODE_EANX_CC, "+12345", -1, "[21]A12345678", 0 },
        /* 4*/ { BARCODE_EANX_CC, "+123456", -1, "[21]A12345678", ZINT_ERROR_TOO_LONG },
    };
    int data_size = sizeof(data) / sizeof(struct item);

    for (int i = 0; i < data_size; i++) {

        struct zint_symbol* symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        symbol->symbology = data[i].symbology;
        int length = data[i].length;
        if (length == -1) {
            length = strlen(data[i].data);
        }
        assert_zero(length >= 128, "i:%d length %d >= 128\n", i, length);
        strcpy(symbol->primary, data[i].data);

        int composite_length = strlen(data[i].composite);

        ret = ZBarcode_Encode(symbol, data[i].composite, composite_length);
        assert_equal(ret, data[i].ret, "i:%d ret %d != %d (%s)\n", i, ret, data[i].ret, symbol->errtxt);

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

int main()
{
    test_eanx_leading_zeroes();
    test_examples();
    test_odd_numbered_numeric();
    test_ean128_cc_shift();
    test_ean128_cc_width();
    test_encodation_0();
    test_encodation_10();
    test_encodation_11();
    test_fuzz();

    testReport();

    return 0;
}
