package Zim::GUI::Component;

use strict;
use vars qw/$AUTOLOAD/;
use Carp;

our $VERSION = '0.17';

=head1 NAME

Zim::GUI::Component - GUI base class

=head1 SYNOPSIS

FIXME example of a component init using actions etc.

=head1 DESCRIPTION

This class provides a base class for GUI components in zim.
Modules can inherit a number of convenience methods from it.

=head1 METHODS

=over 4

=item C<new(%ATTRIBUTES)>

Simple constructor, calls C<init()>.

=cut

our $_popup_args; # used to pass args to UI popups

sub new {
	my $class = shift;
	my $self = bless {@_}, $class;
	$self->{_block_actions} = 0;
	$self->init();
	return $self;
}

=item C<init()>

Called by the constructor, to be overloaded.

=cut

sub init {};

=item C<add_actions($ACTIONS, $TYPE, $NAME)>

Add plain text action descriptions to the interface.

TYPE can be C<undef>, "menu", "toggle" or "radio".

NAME is an optonal group name. This is used to set a single
callback for all actions. If NAME is not defined the callback
for each action will be the name of the action.

=cut

sub add_actions {
	my ($self, $actions, $type, $name) = @_;
	$type = lc $type;
	my $class = ref $self;
	
	unless ($self->{actions}) {
		my $a = Gtk2::ActionGroup->new($class);
		$self->{app}{ui}->insert_action_group($a, 0);
		$self->{actions} = $a;
	}
	return $self->_add_radio_group($actions, $name) if $type eq 'radio';
	
	my @actions;
	for (grep /\S/, split /\n/, $actions) {
		my @a = map {($_ eq '.') ? undef : $_} split /\t+/, $_;
		my $n = $name || $a[0] ;
		push @a, \&{$class.'::on_'.$n} unless $type eq 'menu';
		push @a, '0' if $type eq 'toggle';
		push @actions, \@a;
	};
	#use Data::Dumper; warn "Actions ($type): ", Dumper \@actions;
	
	($type eq 'toggle')
		? $self->{actions}->add_toggle_actions(\@actions, $self)
		: $self->{actions}->add_actions(\@actions, $self)  ;
}

sub _add_radio_group {
	my ($self, $actions, $name) = @_;
	my $class = ref $self;
	
	my @l = grep /\S/, split /\n/, $actions;
	my $val = 0;
	my @actions;
	for (@l) {
		my @a = map {($_ eq '.') ? undef : $_} split /\t+/, $_;
		push @a, $val++;
		push @actions, \@a;
	}
	#use Data::Dumper; warn "Actions (radio): ", Dumper \@actions;

	$self->{actions}->add_radio_actions(
		\@actions, -1, \&{$class.'::on_'.$name}, $self );
}

=item C<get_action($NAME)>

=cut

sub get_action { $_[0]->{actions}->get_action($_[1]) }


=item C<< actions_set_sensitive($NAME => $VAL, ...) >>

Set the sensitivity for one or more actions by name.

=cut

sub actions_set_sensitive {
	my ($self, %actions) = @_;
	for my $name (keys %actions) {
		my $action = $self->{actions}->get_action($name);
		_gtk_action_set_sensitive($action, $actions{$name});
	}
}

sub _gtk_action_set_sensitive { # **sigh**
	my ($action, $bit) = @_;
	if (Gtk2->CHECK_VERSION(2, 6, 0)) { $action->set_sensitive($bit) }
	else { $_->set_sensitive($bit) for $action->get_proxies }
}

=item C<< actions_set_active($NAME => $VAL, ...) >>

Set the one or more actions active by name.

Used to make the state of the actions match the settings.
When it results in a change of state the handler is called,
which in turn makes the state of the application match the settings.

=cut

sub actions_set_active {
	my ($self, %actions) = @_;
	for my $name (keys %actions) {
		my $action = $self->{actions}->get_action($name);
		$action->set_active($actions{$name} ? 1 : 0);
	}
}

=item C<< actions_show_active($NAME => $VAL, ..) >>

Like C<actions_set_active()> but prevents the action callback
to be called. This method is used to make the appearance of 
the action match the state of the application.

=cut

sub actions_show_active {
	my $self = shift;
	$self->{_block_actions} = 1;
	$self->actions_set_active(@_);
	$self->{_block_actions} = 0;
}

=item C<add_ui($UI)>

Add a xml style ui description to the interface.

=cut

sub add_ui { $_[0]->{app}{ui}->add_ui_from_string($_[1]) }

=item C<popup($NAME, $BUTTON, $TIME, @ARGS)>

Popup the menu called NAME from the ui spec.
BUTTON and TIME are passed to C<< Gtk2::Menu->popup() >>.
Any ARGS are forwarded to the actions.

=cut

sub popup {
	my ($self, $name, $button, $time, @args) = @_;
	my $menu = $self->{app}{ui}->get_widget('/'.$name) or return 0;
	$_popup_args = scalar(@args) ? \@args : undef;
	$menu->popup(undef, undef, undef, undef, $button, $time);
	return 1;
}

=item C<AUTOLOAD()>

Autoloader for object methods.

If you have a C<{widget}> attribute in your object this will be
the target for unknown methods.

=cut

sub AUTOLOAD {
	$AUTOLOAD =~ s/^.*:://;
	return if $AUTOLOAD eq 'DESTROY';
	#warn "AUTOLOAD: $AUTOLOAD(@_)\n";
	#warn "AUTOLOAD caller: ", caller(), "\n";
	if ($AUTOLOAD =~ s/^on_(\w+)$/$1/) { # could be an action handler
		my $self = pop;
		return if $self->{_block_actions};
		if ($AUTOLOAD =~ s/^popup_//) { # popup menu
			return $self->$AUTOLOAD(
				defined($_popup_args) ? (@$_popup_args) : () );
		}
		else { return $self->$AUTOLOAD() }
	}
	else {
		my $self = shift;
		my $class = ref $self;
		croak "No such method: $class::$AUTOLOAD"
			unless $self->{widget}->can($AUTOLOAD);
		return $self->{widget}->$AUTOLOAD(@_);
	}
}

1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2006 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

=cut

