/* xstp.h
 * Definitions for Spanning Tree Protocol
 *
 * $Id: xstp.h,v 1.5 2005/07/29 20:37:21 t0mac Exp $ 
 *
 * Yersinia
 * By David Barroso <tomac@wasahero.org> and Alfredo Andres <slay@wasahero.org>
 * Copyright 2005 Alfredo Andres and David Barroso
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifndef __XSTP_H__
#define __XSTP_H__

#include <libnet.h>

#include "terminal-defs.h"
#include "interfaces.h"

#define BPDU_CONF_STP  0x00
#define BPDU_CONF_RSTP 0x02
#define BPDU_TCN       0x80


/* STP stuff */
#define STP_VERSION  0x00

#define STP_TOPOLOGY_CHANGE     0x01
#define STP_TOPOLOGY_CHANGE_ACK 0x80


/* RSTP stuff */
#define RSTP_VERSION 0x02

#define RSTP_NOFLAGS           0
#define RSTP_TOPOLOGY_CHANGE   0x01
#define RSTP_PROPOSAL          0x02
#define RSTP_LEARNING          0x10
#define RSTP_FORWARDING        0x20
#define RSTP_AGREEMENT         0x40

#define RSTP_PORT_ROLE_MASK    0x0C
#define RSTP_PORT_ROLE_SHIFT   0x02

#define RSTP_PORT_ROLE_UNKNOWN    0
#define RSTP_PORT_ROLE_BACKUP     0x04
#define RSTP_PORT_ROLE_ROOT       0x08
#define RSTP_PORT_ROLE_DESIGNATED 0x0C

#define RSTP_TOPOLOGY_CHANGE_ACK 0x80


/* MSTP stuff */
#define MSTP_VERSION     0x03


struct xstp_mitm_args {
       struct attacks *attacks;
       struct stp_data *stp_data2;
};


static const struct tuple_type_desc xstp_version[] = {
    { STP_VERSION,  "(STP)" },
    { RSTP_VERSION, "(RSTP)" },
    { MSTP_VERSION, "(MSTP)" },
    { 0, NULL }
};

static const struct tuple_type_desc xstp_type[] = {
    { BPDU_CONF_STP,  "(Conf STP)" },
    { BPDU_CONF_RSTP, "(Conf (M|R)STP)" },
    { BPDU_TCN,       "(TCN)" },
    { 0,              NULL }
};

static const struct tuple_type_desc xstp_flags[] = {
    { 0,                       "NO FLAGS" },
    { STP_TOPOLOGY_CHANGE,     "(TC)" },
    { STP_TOPOLOGY_CHANGE_ACK, "(TC ACK)" },
    { RSTP_PROPOSAL,           "(Proposal)" },
    { RSTP_LEARNING,           "(Learning)" },
    { RSTP_FORWARDING,         "(Forwarding)" },
    { RSTP_AGREEMENT,          "(Agreement)" },
    { 0, NULL }
};

static struct proto_features xstp_features[] = {
    { F_LLC_DSAP, LIBNET_SAP_STP },
    { -1, 0 }
};

/* Default values */
#define XSTP_DFL_PROTOCOL_ID  0x0000
#define XSTP_DFL_VERSION      STP_VERSION
#define XSTP_DFL_BPDU_TYPE    BPDU_CONF_STP
#define XSTP_DFL_PORT_ID      0x8002
#define XSTP_DFL_MSG_AGE      0
#define XSTP_DFL_MAX_AGE      20
#define XSTP_DFL_HELLO_TIME   2
#define XSTP_DFL_FORW_DELAY   15
#define XSTP_DFL_PORT_ROLE    RSTP_PORT_ROLE_UNKNOWN
#define XSTP_DFL_PORT_STATE   (RSTP_FORWARDING | RSTP_AGREEMENT)

#define XSTP_TOTAL_FIELDS 15

#define XSTP_SMAC       0
#define XSTP_DMAC       1
#define XSTP_ID         2
#define XSTP_VER        3
#define XSTP_TYPE       4
#define XSTP_FLAGS      5 
#define XSTP_ROOTID     6
#define XSTP_PATHCOST   7
#define XSTP_BRIDGEID   8
#define XSTP_PORTID     9
#define XSTP_AGE        10
#define XSTP_MAX        11
#define XSTP_HELLO      12
#define XSTP_FWD        13


static struct position_fields xstp_fields[] = {
   { XSTP_SMAC,     "Source MAC",      1, 1,  12, 17, 0,  0, FIELD_MAC,  FIELD_FIRST,  NULL },
   { XSTP_DMAC,     "Destination MAC", 1, 46, 62, 17, 0,  0, FIELD_MAC,  FIELD_NORMAL, NULL },
   { XSTP_ID,       "Id",              2,  1,  4,  4, 0,  0, FIELD_HEX,  FIELD_NORMAL, NULL },
   { XSTP_VER,      "Ver",             2, 10, 14,  2, 0,  0, FIELD_HEX,  FIELD_NORMAL,xstp_version },
   { XSTP_TYPE,     "Type",            2, 18, 23,  2, 0,  0, FIELD_HEX,  FIELD_NORMAL, xstp_type   },
   { XSTP_FLAGS,    "Flags",           2, 27, 33,  2, 0,  0, FIELD_HEX,  FIELD_NORMAL, xstp_flags  },
   { XSTP_ROOTID,   "RootId",          2, 37, 44, 17, 1, 21, FIELD_BRIDGEID, FIELD_NORMAL, NULL    },
   { XSTP_PATHCOST, "Pathcost",        2, 62, 71,  8, 0,  0, FIELD_HEX,  FIELD_NORMAL, NULL },
   { XSTP_BRIDGEID, "BridgeId",        3,  1, 10, 17, 1,  2, FIELD_BRIDGEID, FIELD_NORMAL, NULL },
   { XSTP_PORTID,   "PortId",          3, 28, 35,  4, 1, 40, FIELD_HEX,  FIELD_NORMAL, NULL },
   { XSTP_AGE,      "Age",             3, 41, 45,  4, 0,  0, FIELD_HEX,  FIELD_NORMAL, NULL },
   { XSTP_MAX,      "Max",             3, 51, 55,  4, 0,  0, FIELD_HEX,  FIELD_NORMAL, NULL },
   { XSTP_HELLO,    "Hello",           3, 60, 66,  4, 0,  0, FIELD_HEX,  FIELD_NORMAL, NULL },
   { XSTP_FWD,      "Fwd",             3, 71, 75,  4, 0,  0, FIELD_HEX,  FIELD_LAST,   NULL },
   { 0,             NULL,              0,  0,  0,  0, 0,  0, FIELD_NONE, FIELD_NORMAL, NULL }
};

/* XSTP mode stuff */
struct stp_data { /* STP and Ethernet fields*/
                 u_int8_t  mac_source[ETHER_ADDR_LEN];
                 u_int8_t  mac_dest[ETHER_ADDR_LEN];
                 u_int16_t id;
                 u_int8_t  version;
                 u_int8_t  bpdu_type;
                 u_int8_t  flags;
                 u_int8_t  root_id[8];
                 u_int32_t root_pc;
                 u_int8_t  bridge_id[8];
                 u_int16_t port_id;
                 u_int16_t message_age;
                 u_int16_t max_age;
                 u_int16_t hello_time;
                 u_int16_t forward_delay;
                 u_int8_t  *rstp_data;
                 u_int8_t  rstp_len;
                 int8_t    do_ack;        /* Do TOP_CHANGE_ACK */
};


void   xstp_th_send_bpdu_conf(void *);
void   xstp_th_send_bpdu_conf_exit(struct attacks *);
int8_t xstp_send_all_bpdu_conf(struct attacks *);
int8_t xstp_send_bpdu_conf(u_int8_t, struct stp_data *, u_int8_t);
void   xstp_th_send_bpdu_tcn(void *);
void   xstp_th_send_bpdu_tcn_exit(struct attacks *);
int8_t xstp_send_all_bpdu_tcn(struct attacks *);
int8_t xstp_send_bpdu_tcn(u_int8_t, struct stp_data *, u_int8_t);
void   xstp_th_dos_conf(void *);
void   xstp_th_dos_conf_exit(struct attacks *);
void   xstp_th_dos_tcn(void *);
void   xstp_th_dos_tcn_exit(struct attacks *);
void   xstp_th_nondos_role(void *);
void   xstp_th_nondos_role_exit(struct attacks *);
void   xstp_th_nondos_other_role(void *);
void   xstp_th_nondos_other_role_exit(struct attacks *);
void   xstp_th_dos_mitm(void *);
void   xstp_th_dos_mitm_exit(struct attacks *);

/*void xstp_dos_elect(void);
void xstp_nondos_main_read_pcap(void);
void xstp_pcap_callback(struct pcap_pkthdr *, const u_char *, int);*/

#define XSTP_MITM_IFACE1 0
#define XSTP_MITM_IFACE2 1

static struct attack_param xstp_mitm_params[] = {
    { NULL, "Interface 1", 1, 0, 0, FIELD_IFACE, IFNAMSIZ, NULL },
    { NULL, "Interface 2", 1, 0, 0, FIELD_IFACE, IFNAMSIZ, NULL }
};

#define STP_ATTACK_SEND_CONF  0
#define STP_ATTACK_SEND_TCN   1
#define STP_ATTACK_DOS_CONF   2 
#define STP_ATTACK_DOS_TCN    3
#define STP_ATTACK_NONDOS_RR  4
#define STP_ATTACK_NONDOS_OR  5
#define STP_ATTACK_RR_MITM    6
/*#define STP_ATTACK_DOS_ELECT    6
#define STP_ATTACK_DOS_DISSAP   7
*/

static struct attack stp_attack[] = {
  { STP_ATTACK_SEND_CONF, "sending conf BPDU",   NONDOS,  SINGLE,    xstp_th_send_bpdu_conf, NULL, 0    },
  { STP_ATTACK_SEND_TCN,  "sending tcn BPDU",    NONDOS,  SINGLE,    xstp_th_send_bpdu_tcn, NULL, 0     },
  { STP_ATTACK_DOS_CONF,  "sending conf BPDUs",  DOS,     CONTINOUS, xstp_th_dos_conf, NULL, 0          },
  { STP_ATTACK_DOS_TCN,   "sending tcn BPDUs",   DOS,     CONTINOUS, xstp_th_dos_tcn, NULL, 0           }, 
  { STP_ATTACK_NONDOS_RR, "Claiming Root Role",  NONDOS,  CONTINOUS, xstp_th_nondos_role, NULL, 0       },
  { STP_ATTACK_NONDOS_OR, "Claiming Other Role", NONDOS,  CONTINOUS, xstp_th_nondos_other_role, NULL, 0 },
  { STP_ATTACK_RR_MITM,   "Claiming Root Role with MiTM", DOS, CONTINOUS, xstp_th_dos_mitm, xstp_mitm_params,
               SIZE_ARRAY(xstp_mitm_params)                    },
/*       { STP_ATTACK_DOS_ELECT,   "Causing Eternal Root Elections", NONDOS, xstp_th_nondos_role, NULL, 0 },
       { STP_ATTACK_DOS_DISSAP,  "Causing Root Dissappearance", NONDOS, xstp_th_nondos_role, NULL, 0 },*/
  { 0,                    NULL,                  0,       0,         NULL, NULL, 0                   }
};


void    xstp_register(void);
void    xstp_send_hellos(void *);
void    xstp_send_hellos_mitm(void *);
int8_t  xstp_learn_packet(int8_t, u_int8_t *, void *, struct pcap_pkthdr *);
int8_t  xstp_decrement_bridgeid(struct stp_data *);
int8_t  xstp_increment_bridgeid(struct stp_data *); 
char    **xstp_get_printable_packet(struct pcap_data *);
char    **xstp_get_printable_store(struct term_node *);
char    *xstp_get_info(u_int8_t, char *);
int8_t  xstp_update_field(int8_t, struct term_node *, void *);
int8_t  xstp_init_attribs(struct term_node *);
int8_t  xstp_load_values(struct pcap_data *, void *);
int8_t  xstp_update_data(int8_t, int8_t, int8_t, struct term_node *);
void    xstp_help(void);
int8_t  xstp_parser( struct term_node *, int8_t, char **, void *);

#ifndef HAVE_USLEEP
extern int usleep(unsigned long);
#endif

extern void   thread_libnet_error( char *, libnet_t *);
extern int8_t thread_create(pthread_t *, void *, void *);
extern void   write_log( u_int16_t mode, char *msg, ... );
extern int8_t attack_th_exit(struct attacks *);
extern void   attack_gen_mac(u_int8_t *);
extern int8_t interfaces_get_packet(u_int8_t *, struct pcap_pkthdr *, u_int8_t *, u_int16_t, time_t);
extern int8_t parser_command2index(register const struct attack *, register int8_t);
extern int8_t parser_vrfy_mac(char *, u_int8_t *);
extern int8_t parser_vrfy_bridge_id(char *, u_int8_t * );
extern void   parser_str_tolower( char *);

extern struct interface_data interfaces[];
extern struct terminals *terms;
extern int8_t bin_data[];

#endif

/* vim:set tabstop=4:set expandtab:set shiftwidth=4:set textwidth=78: */
