/* protocols.c
 * Protocols stuff
 *
 * Yersinia
 * By David Barroso <tomac@wasahero.org> and Alfredo Andres <slay@wasahero.org>
 * Copyright 2005 Alfredo Andres and David Barroso
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifndef lint
static const char rcsid[] =
"$Id: protocols.c,v 1.5 2005/07/30 15:17:13 t0mac Exp $";
#endif

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#define _REENTRANT

#include <stdio.h>
#include <errno.h>

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif

#ifdef HAVE_BSTRING_H
#include <bstring.h>
#endif

#ifdef STDC_HEADERS
#include <stdlib.h>
#endif

#include "protocols.h"

void
protocol_init(void)
{
    protocol_register_all();
}


int8_t
protocol_register(u_int8_t proto, const char *name, const char *desc, u_int16_t size, init_attribs_t init,
                  learn_packet_t learn, get_printable_packet_t packet,
                  get_printable_store_t store, load_values_t load,
                  update_data_t update, struct attack *attacks, display_help_t help,
		  parser_t parser, struct position_fields *fields, get_info_t get_info,
		  update_field_t update_field, struct proto_features *features)
{
    int8_t i;

    if (proto > MAX_PROTOCOLS)
	return -1;
    
    protocols[proto].proto = proto;
    strncpy(protocols[proto].name, name, MAX_PROTO_NAME);
    strncpy(protocols[proto].description, desc, MAX_PROTO_DESCRIPTION);
    protocols[proto].size = size;
    protocols[proto].active = 1;      /* default is active */
    protocols[proto].init_attribs = init;
    protocols[proto].learn_packet = learn;
    protocols[proto].get_printable_packet = packet;
    protocols[proto].get_printable_store = store;
    protocols[proto].load_values = load;
    protocols[proto].update_data = update;
    protocols[proto].attacks = attacks;
    protocols[proto].display_help = help;
    protocols[proto].parser = parser;
    protocols[proto].fields = fields;
    protocols[proto].get_info = get_info;
    protocols[proto].update_field = update_field;
    protocols[proto].features = features;
    
    for (i = 0; i < MAX_PACKET_STATS; i++) {
	if ((protocols[proto].stats[i].header = (struct pcap_pkthdr *) calloc(1, sizeof(struct pcap_pkthdr))) == NULL)
	    return -1;

	if ((protocols[proto].stats[i].packet = (u_char *) calloc(1, SNAPLEN)) == NULL)
	    return -1;
    }

    protocols[proto].packets = 0;
    protocols[proto].packets_out = 0;

    /* Default values */
    if ((protocols[proto].default_values = calloc(1, size)) == NULL)
       return -1;

    return 0;
}


int8_t
protocol_register_tlv(u_int8_t proto, edit_tlv_t edit_tlv, const struct tuple_type_desc *ttd, 
	struct attack_param *tlv, u_int16_t params)
{
    if (proto > MAX_PROTOCOLS)
	return -1;
    
    protocols[proto].edit_tlv = edit_tlv;
    protocols[proto].ttd = ttd;
    protocols[proto].tlv = tlv;
    protocols[proto].tlv_params = params;

    return 0;
}


void
protocol_register_all(void)
{
    { extern void xstp_register(void);  xstp_register();  }
    { extern void cdp_register(void);   cdp_register();   }
    { extern void dtp_register(void);   dtp_register();   }
    { extern void dhcp_register(void);  dhcp_register();  }
    { extern void hsrp_register(void);  hsrp_register();  }
    { extern void dot1q_register(void); dot1q_register(); }
    { extern void isl_register(void);   isl_register();   }
    { extern void vtp_register(void);   vtp_register();   }
    { extern void arp_register(void);   arp_register();   }
}


void
protocol_destroy(void)
{
    int8_t i, j;

    for (i = 0; i < MAX_PROTOCOLS; i++)
    {
        for (j = 0; j < MAX_PACKET_STATS; j++) 
        {
            if (protocols[i].stats[j].header)
               free(protocols[i].stats[j].header);
            if (protocols[i].stats[j].packet)
               free(protocols[i].stats[j].packet);
        }
        if (protocols[i].default_values)
           free(protocols[i].default_values);
    }
}

char **
protocol_create_printable(u_int8_t size, struct position_fields *fields)
{
    u_int8_t i;
    char **field_values;

    field_values = NULL;

    if ((field_values = (char **) calloc(1, size* sizeof(u_int8_t *))) == NULL) {
        printf("Error in calloc\n");
        return NULL;
    }

    i = 0;

    while(fields[i].name) {
	if (fields[i].l > 0)
	    if ((field_values[i] = (char *) calloc(1, (fields[i].l + 1) * sizeof(char))) == NULL) {
		printf("Error in calloc\n");
		return NULL;
	    }
	i++;
    }

    return field_values;
}


u_int8_t
protocol_return_last_field(struct position_fields *fields)
{
    u_int8_t i;

    i = 0;
    while(fields[i].name && fields[i].y > 0)
	    i++;

    return (i - 1);
}


int8_t
protocol_proto2index(char *name)
{
    int i=0;

    while (i < MAX_PROTOCOLS) 
    {
        if (!strcasecmp(protocols[i].name, name))
            return (protocols[i].proto);
        ++i;
    }

    return -1;
}
