/* dot1q.h
 * Definitions for IEEE 802.1Q
 *
 * $Id: dot1q.h,v 1.4 2005/07/29 20:37:21 t0mac Exp $ 
 *
 * Yersinia
 * By David Barroso <tomac@wasahero.org> and Alfredo Andres <slay@wasahero.org>
 * Copyright 2005 Alfredo Andres and David Barroso
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifndef __DOT_1Q_H
#define __DOT_1Q_H__

#include <libnet.h>

#include "terminal-defs.h"
#include "interfaces.h"

#define MAX_ICMP_PAYLOAD 16
 
#define DOT1Q_DFL_MAC_DST "FF:FF:FF:FF:FF:FF"
#define DOT1Q_DFL_TPI1    ETHERTYPE_VLAN
#define DOT1Q_DFL_PRIO1   0x007
#define DOT1Q_DFL_CFI1    0x000
#define DOT1Q_DFL_VLAN1   0x001
#define DOT1Q_DFL_TPI2    ETHERTYPE_IP
#define DOT1Q_DFL_PRIO2   0x007
#define DOT1Q_DFL_CFI2    0x000
#define DOT1Q_DFL_VLAN2   0x002
#define DOT1Q_DFL_TPI3    ETHERTYPE_IP
#define DOT1Q_DFL_DST_IP  0xffffffff
#define DOT1Q_DFL_PAYLOAD "YERSINIA"
#define DOT1Q_DFL_PAY_LEN 8

struct dot1q_data { 
     u_int8_t  mac_source[ETHER_ADDR_LEN];
     u_int8_t  mac_dest[ETHER_ADDR_LEN];
     u_int16_t tpi1;
     u_int8_t  priority1;
     u_int8_t  cfi1;
     u_int16_t vlan1;
     u_int16_t tpi2;
     u_int8_t  priority2;
     u_int8_t  cfi2;
     u_int16_t vlan2;
     u_int16_t tpi3;
     u_int32_t src_ip;
     u_int32_t dst_ip;
     u_int8_t  ip_proto;
     u_int8_t  icmp_payload[MAX_ICMP_PAYLOAD+1];
     u_int8_t  icmp_pay_len;
};

#define DOT1Q_TOTAL_FIELDS 14

#define DOT1Q_SMAC       0
#define DOT1Q_DMAC       1
#define DOT1Q_VLAN1      2
#define DOT1Q_PRIORITY1  3
#define DOT1Q_CFI1       4
#define DOT1Q_TPI2       5 
#define DOT1Q_VLAN2      6
#define DOT1Q_PRIORITY2  7
#define DOT1Q_CFI2       8
#define DOT1Q_TPI3       9
#define DOT1Q_SRC_IP    10
#define DOT1Q_DST_IP    11
#define DOT1Q_IP_PROTO  12

static const struct tuple_type_desc dot1q_tpi[] = {
     { ETHERTYPE_IP,     "IP"   },
     { ETHERTYPE_VLAN,   ".1Q"  },
     { ETHERTYPE_ARP,    "ARP"  },
     { ETHERTYPE_REVARP, "RARP" },
     { 0x2000,           "CDP"  },
     { 0x2003,           "VTP"  },
     { 0x2004,           "DTP"  },
     { 0x9000,           "LOOP" },
     { 0x010b,           "PVST" },
     { 0, NULL }
};

static const struct tuple_type_desc dot1q_ip_proto[] = {
     { 0x01,   "icmp" },
     { 0x06,   "tcp"  },
     { 0x11,   "udp"  },
     { 0,      NULL   }
};

static struct proto_features dot1q_features[] = {
     { F_ETHERTYPE, ETHERTYPE_VLAN },
     { -1, 0 }
};

static struct position_fields dot1q_fields[] = {
     { DOT1Q_SMAC,       "Source MAC",       1,  1, 12, 17, 0,  0, FIELD_MAC, FIELD_FIRST, NULL },
     { DOT1Q_DMAC,       "Destination MAC",  1, 39, 55, 17, 0,  0, FIELD_MAC, FIELD_NORMAL, NULL },
     { DOT1Q_VLAN1,      "VLAN",             2,  1, 6,   4, 1,  2, FIELD_DEC, FIELD_NORMAL, NULL }, 
     { DOT1Q_PRIORITY1,  "Priority",         2, 11, 20,  2, 0,  0, FIELD_HEX, FIELD_NORMAL, NULL },
     { DOT1Q_CFI1,       "CFI",              2, 23, 27,  2, 0,  0, FIELD_HEX, FIELD_NORMAL, NULL },
     { DOT1Q_TPI2,       "L2Proto1",         2, 30, 39,  4, 1,  7, FIELD_HEX, FIELD_NORMAL, dot1q_tpi },
     { DOT1Q_VLAN2,      "VLAN2",            2, 44, 50,  4, 0,  8, FIELD_DEC, FIELD_NORMAL, NULL },   
     { DOT1Q_PRIORITY2,  "Priority",         2, 55, 64,  2, 0,  0, FIELD_HEX, FIELD_NORMAL, NULL },
     { DOT1Q_CFI2,       "CFI",              2, 67, 71,  2, 0,  0, FIELD_HEX, FIELD_NORMAL, NULL },
     { DOT1Q_TPI3,       "L2Proto2",         3,  1, 10,  4, 0,  0, FIELD_HEX, FIELD_NORMAL, dot1q_tpi },   
     { DOT1Q_SRC_IP,     "Src IP",           3, 15, 22, 15, 1, 17, FIELD_IP,  FIELD_NORMAL, NULL },
     { DOT1Q_DST_IP,     "Dst IP",           3, 38, 45, 15, 1, 33, FIELD_IP,  FIELD_NORMAL, NULL },   
     { DOT1Q_IP_PROTO,   "IP Prot",          3, 61, 69,  2, 1, 50, FIELD_DEC, FIELD_LAST,   dot1q_ip_proto },
     { 0,                NULL,               0,  0,  0,  0, 0,  0, FIELD_NONE,FIELD_NORMAL, NULL }
};


void dot1q_th_send(void *);
void dot1q_th_send_exit(struct attacks *);
void dot1q_double_th_send(void *);
void dot1q_double_th_send_exit(struct attacks *);
void dot1q_th_poison(void *);
void dot1q_th_poison_exit(struct attacks *);


#define DOT1Q_ARP_IP     0
#define DOT1Q_ARP_VLAN   1
#define DOT1Q_ARP_IP_SRC 2

static struct attack_param dot1q_arp_params[] = {
    { NULL, "IP to poison",  4, 0,    0, FIELD_IP,  15, NULL },
    { NULL, "IP VLAN",       2, 0, 4095, FIELD_DEC,  4, NULL },
    { NULL, "ARP IP Source", 4, 0,    0, FIELD_IP,  15, NULL }
};


#define DOT1Q_ATTACK_SEND   0
#define DOT1Q_ATTACK_DOUBLE 1
#define DOT1Q_ATTACK_POISON 2

static struct attack dot1q_attack[] = {
    { DOT1Q_ATTACK_SEND,   "sending 802.1Q packet",             NONDOS, SINGLE,    dot1q_th_send,        NULL, 0           },
    { DOT1Q_ATTACK_DOUBLE, "sending 802.1Q double enc. packet", NONDOS, SINGLE,    dot1q_double_th_send, NULL, 0           },
    { DOT1Q_ATTACK_POISON, "sending 802.1Q arp poisoning",      DOS,    CONTINOUS, dot1q_th_poison,      dot1q_arp_params,
              SIZE_ARRAY(dot1q_arp_params) },
    { 0,                   NULL,                                0,      0,         NULL,                 NULL, 0           }
};


void dot1q_register(void);
int8_t dot1q_send_icmp(struct attacks *, u_int8_t);
void   dot1q_send_arp_poison(void *);
int8_t dot1q_send_arp(u_int32_t, u_int16_t, u_int8_t *, u_int8_t *, u_int8_t *, u_int8_t *, u_int8_t *, u_int16_t, u_int8_t);
int8_t dot1q_learn_mac(struct attacks *, struct pcap_pkthdr *, u_int8_t *);
int8_t dot1q_return_mac(struct attacks *, u_int8_t *);
int8_t dot1q_init_attribs(struct term_node *);
int8_t dot1q_learn_packet(int8_t, u_int8_t *,void *, struct pcap_pkthdr *);
char **dot1q_get_printable_packet(struct pcap_data *);
char **dot1q_get_printable_store(struct term_node *);
char *dot1q_get_info(u_int8_t, char *);
int8_t dot1q_load_values(struct pcap_data *, void *);
int8_t dot1q_update_data(int8_t, int8_t, int8_t, struct term_node *);
void   dot1q_help(void);
int8_t dot1q_parser( struct term_node *, int8_t, char **, void *);
int8_t dot1q_update_field(int8_t, struct term_node *, void *);

extern void   thread_libnet_error( char *, libnet_t *);
extern int8_t vrfy_bridge_id( char *, u_int8_t * );
extern int8_t thread_create(pthread_t *, void *, void *);
extern void   write_log( u_int16_t mode, char *msg, ... );
extern int8_t attack_th_exit(struct attacks *);
extern void   attack_gen_mac(u_int8_t *);
extern int8_t interfaces_get_packet(u_int8_t *, struct pcap_pkthdr *, u_int8_t *, u_int16_t, time_t);
extern int8_t parser_vrfy_mac(char *, u_int8_t *);
extern int8_t parser_get_inet_aton(char *, struct in_addr *);
extern int8_t parser_get_formated_inet_address(u_int32_t, char *, u_int16_t);
extern int8_t parser_command2index(register const struct attack *, register int8_t);

extern struct interface_data interfaces[];
extern struct terminals *terms;

extern int8_t bin_data[];

#endif
/* vim:set tabstop=4:set expandtab:set shiftwidth=4:set textwidth=78: */
