#!/bin/sh
#
# make_xprint_binary_tarball.sh - packager script for Xprint stuff
#
# $Xorg: make_xprint_binary_tarball.sh,v 1.2 2003/02/09 04:41:52 gisburn Exp $
#
# This script creates a binary tarball for Xprint stuff (Xprt, Xprt config
# files and Xprint helper applications.
#
# Written by Roland Mainz <roland.mainz@nrubsig.org>
#

# These should be provided by Imakefile
#XTOP=..
#XPRINTDIR=/usr/openwin/lib/X11/xserver
#XBINDIR=/usr/openwin/bin
#
# XPDESTTARFILE may be optionally set as a command line argument to give the 
# destination filepath for the tarball, otherwise it will be found in /tmp.
# Note that the tarball is automatically compressed (to .gz, if gzip is
# available, otherwise to .Z); your name for XPDESTTARFILE should reflect this.

PATH=/usr/bin:/bin:/usr/sbin:/sbin
if [ "`uname -s`" = "SunOS" ] ; then
    PATH=/usr/xpg4/bin:/usr/ccs/bin:${PATH}
fi
export PATH

fatal_error()
{
    echo "${this_script}: ## FATAL ERROR: ${1}" 1>&2
    exit 1
}

# Check if mandatory env vars have been set
[ "${XTOP}"                       = "" ] && fatal_error "XTOP env var not set."
[ "${XPRINTDIR}"                  = "" ] && fatal_error "XPRINTDIR env var not set."
[ "${XBINDIR}"                    = "" ] && fatal_error "XBINDIR env var not set."
[ "${XMANPATH}"                   = "" ] && fatal_error "XMANPATH env var not set."
[ "${XFONTENCPATH}"               = "" ] && fatal_error "XFONTENCPATH env var not set."
[ "${XINSTALLXF86ENCODINGFILES}"  = "" ] && fatal_error "XINSTALLXF86ENCODINGFILES env var not set."


# Do not edit
# REL_* vars are paths relative to '/'
# Thanks to the "wonders" of some /bin/sh I cannot use ...
# REL_XPRINTDIR=${XPRINTDIR#*/}
# REL_XBINDIR=${XBINDIR#*/}
# ... ;-(
REL_XPRINTDIR="`echo ${XPRINTDIR} | sed 's/\///'`"
REL_XBINDIR="`echo ${XBINDIR} | sed 's/\///'`"
REL_XMANPATH="`echo ${XMANPATH} | sed 's/\///'`"
REL_XFONTENCPATH="`echo ${XFONTENCPATH} | sed 's/\///'`"

XPINSTALL_SESSION="`date +%y%m%d%H%M%S`"
XPINSTALLDIR_BASE=/tmp/xpinstall_${XPINSTALL_SESSION}
XPINSTALLDIR=${XPINSTALLDIR_BASE}/xprint
XPINSTALLDIR_X11=${XPINSTALLDIR}/install
XPLOGFILE=/tmp/xpinstall_${XPINSTALL_SESSION}.log
XPTARFILE=/tmp/xprint_server_${XPINSTALL_SESSION}.tar

CURR_DIR="`pwd`"

DO_BUILD=false

# List of files/dirs which should be put into the tarball
if [ "`uname -s`" = "FreeBSD" ] ; then
  REL_XETCDIR="`echo ${XBINDIR} | sed 's/\///' | sed 's/bin/etc/'`"
else
  REL_XETCDIR="etc"
fi
BINFILELIST=" \
  xprint/install/${REL_XBINDIR}/Xprt \
  xprint/install/${REL_XBINDIR}/xplsprinters \
  xprint/install/${REL_XBINDIR}/xprehashprinterlist \
  xprint/install/${REL_XBINDIR}/xphelloworld \
  xprint/install/${REL_XBINDIR}/xpxthelloworld \
  xprint/install/${REL_XBINDIR}/xpsimplehelloworld \
  xprint/install/${REL_XBINDIR}/pclcomp \
  xprint/install/${REL_XBINDIR}/xdpr \
  xprint/install/${REL_XBINDIR}/xpr \
  xprint/install/${REL_XETCDIR} \
  xprint/install/${REL_XPRINTDIR} \
  xprint/install/${REL_XPRINTDIR}/C/print/models/PSdefault/fonts/fonts.dir \
  xprint/install/${REL_XPRINTDIR}/C/print/models/PSdefault/fonts/Courier.pmf \
  xprint/install/${REL_XPRINTDIR}/C/print/attributes/document \
  xprint/install/${REL_XPRINTDIR}/en_US/print/attributes/document \
  xprint/install/${REL_XMANPATH} \
  xprint/run_xprint_from_tarball.sh \
  xprint/README"

# Solaris/CDE1.x || AIX/CDE extra files
if [ "`uname -s`" = "SunOS" -o "`uname -s`" = "AIX" ] ; then
  BINFILELIST="${BINFILELIST} \
    xprint/install/usr/dt/config/Xsession.d"
fi

# Should we install our own copies of the XF86 encoding files ?
if [ ${XINSTALLXF86ENCODINGFILES} -eq 1 ] ; then
  BINFILELIST="${BINFILELIST} \
    xprint/install/${REL_XFONTENCPATH}"
fi

# List of manual page names (suffix is omitted here since it differs
# from platform to platform)
MANPAGELIST=" \
  Xprint. \
  Xprt. \
  xplsprinters. \
  xprehashprinterlist. \
  xphelloworld \
  xpxthelloworld. \
  xpawhelloworld. \
  xpxmhelloworld. \
  xpsimplehelloworld. \
  pclcomp. \
  xdpr. \
  xpr."

step=0

umask 022

inc_step_counter()
{
  step=`expr ${step} + 1`
}

has_gzip()
{
  for i in `echo ${PATH} | tr ":" " "` 
  do
    if [ -x "${i}/gzip" ] ; then
      return 0
    fi
  done

  return 1
}

# Step 1:
# Build xc/ tree...
if ${DO_BUILD} ; then
  inc_step_counter
  echo "## Step ${step}: Building..."
  (
    (
      cd ${XTOP}
      if make World 2>&1 | tee -a buildlog.log >&${XPLOGFILE} ; then
        echo "Build OK."
      else
        echo "Build failed:"
        echo "-- snip --"
        tail ${XPLOGFILE}
        echo "-- snip --"
        exit 1
      fi
    )
  )
  echo "##         Build completed DONE."
fi # ${DO_BUILD}

# Step 2:
# Verification...
inc_step_counter
echo "## Step ${step}: Verifying..."
(
  status=true

  if [ ! -x ${XTOP}/programs/Xserver/Xprt ] ; then
    echo "error: X print server (${XTOP}/programs/Xserver/Xprt) binary missing."
    status=false
  fi

  if [ ! -x ${XTOP}/programs/xplsprinters/xplsprinters ] ; then
    echo "error: ${XTOP}/programs/xplsprinters/xplsprinters binary missing."
    status=false
  fi

  if [ ! -x ${XTOP}/programs/xprehashprinterlist/xprehashprinterlist ] ; then
    echo "error: ${XTOP}/programs/xprehashprinterlist/xprehashprinterlist binary missing."
    status=false
  fi

  if [ ! -x ${XTOP}/programs/xphelloworld/xpxthelloworld/xpxthelloworld ] ; then
    echo "error: ${XTOP}/programs/xphelloworld/xpxthelloworld/xpxthelloworld binary missing."
    status=false
  fi

  if [ ! -x ${XTOP}/programs/xphelloworld/xphelloworld/xphelloworld ] ; then
    echo "error: ${XTOP}/programs/xphelloworld/xphelloworld/xphelloworld binary missing."
    status=false
  fi

  if [ ! -x ${XTOP}/programs/xphelloworld/xpsimplehelloworld/xpsimplehelloworld  ] ; then
    echo "error: ${XTOP}/programs/xphelloworld/xpsimplehelloworld/xpsimplehelloworld binary missing."
    status=false
  fi

  if ${status} ; then
    echo "No errors found."
  else
    fatal_error "Verification failed."
  fi
)
[ $? != 0 ] && exit 1;
echo "##         Verification DONE."

# Step 3:
# Install into temp. location
inc_step_counter
echo "## Step ${step}: Install into temporary location..."
(
  [ -d ${XPINSTALLDIR} ] && fatal_error "Temp. installation dir ${XPINSTALLDIR} already exists."
  mkdir -p ${XPINSTALLDIR}
  [ ! -d ${XPINSTALLDIR} ] && fatal_error "Could not create temp. installation dir ${XPINSTALLDIR}."

  mkdir ${XPINSTALLDIR_X11}
  [ ! -d ${XPINSTALLDIR_X11} ] && fatal_error "Could not create temp. installation dir ${XPINSTALLDIR_X11}."
  
  (
    cd ${XTOP}
    
    # Install binaries, config files etc.
    if make install DESTDIR=${XPINSTALLDIR_X11} >>${XPLOGFILE} 2>&1 ; then
      echo "Installation (1/3) OK."
    else
      echo "Installation (1/3) failed:"
      echo "-- snip --"
      tail ${XPLOGFILE}
      echo "-- snip --"
      exit 1
    fi
    
    # Install manual pages...
    if make install.man DESTDIR=${XPINSTALLDIR_X11} >>${XPLOGFILE} 2>&1 ; then
      echo "Installation (2/3) OK."
    else
      echo "Installation (2/3) failed:"
      echo "-- snip --"
      tail ${XPLOGFILE}
      echo "-- snip --"
      exit 1
    fi
    
    # Copy tarball startxprint.sh and README_tarball
    cp "${CURR_DIR}/README_tarball" "${XPINSTALLDIR}/README"
    cp "${CURR_DIR}/startxprint.sh" "${XPINSTALLDIR}/run_xprint_from_tarball.sh"
    chmod a+rx "${XPINSTALLDIR}/run_xprint_from_tarball.sh"
    echo "Installation (3/3) OK."
    
    # Search the ${XPINSTALLDIR_X11}/${REL_XMANPATH} dir for manual pages which
    # do not occur in the ${MANPAGELIST} list and remove those files
    cd ${XPINSTALLDIR_X11}/${REL_XMANPATH}
    for i in */*.* ; do
      NO_MATCH="true"
      
      for j in ${MANPAGELIST} ; do
        [ "`echo "${i}" | fgrep "${j}"`" != "" ] && NO_MATCH="false"
      done
      
      if ${NO_MATCH} ; then 
        rm -f "${i}"
      fi
    done
  )
)
[ $? != 0 ] && exit 1;
echo "##         Temp. installation DONE."

# Step 4:
# Verify installation
inc_step_counter
echo "## Step ${step}: Verifying installation..."
(
  # Part 1: Test for missing files
  cd ${XPINSTALLDIR_BASE}
  status=true
  for i in ${BINFILELIST} ; do
    if [ ! -r ${i} ] ; then
      echo "File ${i} missing."
      status=false
    fi
  done
  
  # Part 2: Test "PSdefault" printer model
  cd ${XPINSTALLDIR_BASE}
  x=`cat "xprint/install/${REL_XPRINTDIR}/C/print/models/PSdefault/fonts/fonts.dir" | wc -l`
  if [ "$x" -ne 31 ] ; then
    echo "PSdefault model fonts.dir misses font entries."
    status=false
  fi
  
  if ${status} ; then
    echo "No errors found."
  else
    fatal_error "Verification failed."
  fi
)
[ $? != 0 ] && exit 1;
echo "##        Verification of installation DONE."

# Step 5.a:
# Create binary tarball
inc_step_counter
echo "## Step ${step}: Create binary tarball..."
(
  cd ${XPINSTALLDIR_BASE}
  if has_gzip ; then
    tar -cf - ${BINFILELIST} | gzip --best --stdout >${XPTARFILE}.gz
  else
    tar -cf - ${BINFILELIST} | compress >${XPTARFILE}.Z
  fi
)
[ $? != 0 ] && exit 1;
echo "##        Binary tarball creation DONE."

# Step 5.b:
# Copy binary tarball to destination, if required.
if [ "${XPDESTTARFILE}" != "" ] ; then
    inc_step_counter
    echo "## Step ${step}: Copying binary tarball to destination..."
    (
    if has_gzip ; then
	mv ${XPTARFILE}.gz ${XPDESTTARFILE}
    else
	mv ${XPTARFILE}.Z ${XPDESTTARFILE}
    fi
    )
    echo "##        Copying tarball to destination DONE."
fi

# Step 6:
# Cleaning up
inc_step_counter
echo "## Step ${step}: Cleaning up..."
(
  # Be carefull... =:-)
  if [ "${XPINSTALLDIR_BASE}" != "/" ] ; then
    rm -Rf ${XPINSTALLDIR_BASE}
  fi
)
echo "##        Cleanup done."


# Done.
echo "#### Binary tarball is now available at "
if [ "${XPDESTTARFILE}" != "" ] ; then
    ls -l ${XPDESTTARFILE}*
else
    ls -l ${XPTARFILE}*
fi
echo "#### All done."

exit 0
# EOF.
