/*
 *  This file is part of X-File Manager XFM
 *  ----------------------------------------------------------------------
  mime.c

  (c) 2005,2006 Bernhard R. Link

  mime-type related code
 *  ----------------------------------------------------------------------
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <xfmconfig.h>

#define _GNU_SOURCE 1 // for strndup
#include <errno.h>
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <string.h>
#include <fcntl.h>
#include <stdbool.h>
#include <stdint.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "mime.h"
#include "suffix.h"

static bool mime_savememory;

static struct mime_major_type {
	struct mime_major_type *smaller,*larger;
	char *typename;
	struct mime_filetype *catchall;
	size_t childcount;
	struct mime_filetype **children;
} *mime_root = NULL;
struct mime_filetype *catchallall = NULL;

static int mailcap_parse(const char *filename);

static struct mime_filetype *newfiletype(char *identifier,const char *subtypename,struct mime_filetype *fallback) {
	struct mime_filetype *f;
	
	f = calloc(1,sizeof(struct mime_filetype));
	if( f == NULL )
		return f;
	f->fulltypename = identifier;
	f->subtypename = subtypename;
	f->fallback = fallback;
	return f;
}

static inline struct mime_major_type *mime_newtype(const char *typename,size_t len) {
	struct mime_major_type *n = calloc(1,sizeof(*n));
	char *ca;

	if( n == NULL )
		return NULL;
	n->typename = strndup(typename,len);
	if( n->typename == NULL ) {
		free(n);
		return NULL;
	}
	ca = malloc(len+9);
	if( ca == NULL ) {
		free(n->typename);
		free(n);
		return NULL;
	}
	memcpy(ca,typename,len);
	memcpy(ca+len,"/default",8);
	ca[len+8] = '\0';
	n->catchall = newfiletype(ca,"default",catchallall);
	if( n->catchall == NULL ) {
		free(n->typename);
		free(n);
		return NULL;
	}
	return n;
}

static inline int lcmp(const char *s1,const char *s2,size_t len2) {
	int cmp;
	cmp = strncmp(s1,s2,len2);
	if( cmp != 0 )
		return cmp;
	if( s1[len2] == '\0' )
		return 0;
	return 1;
}

static struct mime_major_type *mime_get_type(const char *typename, size_t len, bool generate) {
	struct mime_major_type *p;
	int cmp;

	if( mime_root == NULL ) {
		if( generate ) {
			mime_root = mime_newtype(typename,len);
			return mime_root;
		} else
			return NULL;
	}
	p = mime_root;
	while( (cmp = lcmp(p->typename,typename,len)) != 0 ) {
		if( cmp < 0 ) {
			if( p->larger == NULL ) {
				if( generate ) {
					p->larger = mime_newtype(typename,len);
					return p->larger;
				}
				return NULL;
			}
			p = p->larger;
		} else {
			if( p->smaller == NULL ) {
				if( generate ) {
					p->smaller = mime_newtype(typename,len);
					return p->smaller;
				}
				return NULL;
			}
			p = p->smaller;
		}
	}
	return p;
}

struct mime_filetype *mime_get_filetype(const char *mimetype) {
	char *separator = strchr(mimetype,'/');
	char *subtypename;
	size_t l;
	struct mime_major_type *type;

	assert(catchallall != NULL);


	if( mimetype[0] == '*' && mimetype[1] == '\0' )
		return catchallall;

	if( separator != NULL)
		l = separator-mimetype;
	else
		l = strlen(mimetype);

	type = mime_get_type(mimetype,l,true);
	if( type == NULL ) { /* OOM*/
		return NULL;
	}
	if( separator == NULL || separator[1] == '\0' || separator[1] == '*' ) {
		return type->catchall;
	}
	subtypename = separator + 1;
	if( type->children == NULL ) {
		char *newmimetype;
		type->children = calloc(8,sizeof(struct mime_filetype*));
		if( type->children == NULL )
			return NULL;
		newmimetype = strdup(mimetype);
		if( newmimetype == NULL ) {
			free(type->children);
			type->children = NULL;
			type->childcount = 0;
			return NULL;
		}
		subtypename = newmimetype + (subtypename - mimetype);
		type->children[0] = newfiletype(newmimetype,subtypename,type->catchall);
		if( type->children[0] == NULL ) {
			free(type->children);
			type->children = NULL;
			type->childcount = 0;
			return NULL;
		}
		type->childcount = 1;
		return type->children[0];
	} else {
		char *newmimetype;
		struct mime_filetype *nft;
		size_t i = 0;
		size_t j;
		int c;

		c = strcmp(type->children[type->childcount-1]->subtypename,subtypename);
		if( c == 0 )
			return type->children[type->childcount - 1];
		if( c < 0 ) {
			i = type->childcount;
		} else if( type->childcount > 1 ) {
			int cc;
			j = type->childcount - 2;
			while( i <= j ) {
				size_t m = (i+j)/2;
				cc = strcmp(type->children[m]->subtypename,subtypename);
				if( cc == 0 )
					return type->children[m];
				if( cc > 0 ) {
					if( m == 0 )
						break;
					j = m-1;
				} else
					i = m+1;
			}
		}
		/* now i is the new place for subtypename */
		if( ((type->childcount+1) & 7) == 0 ) {
			struct mime_filetype **n;
			n = realloc(type->children,
				(type->childcount+8)*sizeof(struct mime_filetype*));
			if( n == NULL )
				return NULL;
			type->children = n;
		}
		newmimetype = strdup(mimetype);
		if( newmimetype == NULL )
			return NULL;
		subtypename = newmimetype + (subtypename - mimetype);
		nft = newfiletype(newmimetype,subtypename,type->catchall);
		if( nft == NULL )
			return NULL;
		for( j = type->childcount; j > i ; j-- ) {
			type->children[j] = type->children[j-1];
		}
		type->children[i] = nft;
		type->childcount++;
		return nft;
	}
}

static inline void mime_free(struct mime_action *action) {
	assert(action != NULL);
	free(action->view);
	free(action->compose);
	free(action->edit);
	free(action->print);
	free(action->drop);
	free(action->nameprefix);
	free(action->test);
	free(action->notes);
	free(action->description);
	free(action->test);
	free(action->x11bitmap);
	free(action);
}

static int mime_read_charskipspace(int *linenr,int *columnnr, FILE *f);

static inline int mime_read_char(int *linenr,int *columnnr, FILE *f) {
	int c;

	c = fgetc(f);
	*columnnr += 1;
	if( c != '\\' ) {
		return c;
	} else {
		c = fgetc(f);
		if( c == EOF)
			return c;
		else if( c != '\n') {
			ungetc(c,f);
			return '\\';
		} else {
			*linenr += 1;
			*columnnr=0;
			return mime_read_charskipspace(linenr,columnnr,f);
		}
	}
}

static int mime_read_charskipspace(int *linenr,int *columnnr, FILE *f) {
	int c;

	do {
		c = mime_read_char(linenr,columnnr,f);
	} while( c > 0 && c != '\n' && isspace(c) );
	return c;
}

static inline void mime_read_skipline(int *linenr,int *columnnr, FILE *f) {
 
	while( 1 ) {
		int c = mime_read_char(linenr,columnnr,f);
		if( c == EOF || c == '\n' )
			return;
	}
}

static inline int mime_read_statement(int *linenr,int *colnr,
		char *buffer, size_t buflen, FILE *f) {
	int c;
	size_t i = 0;
	bool escaped = false;

	c = mime_read_charskipspace(linenr,colnr,f);

	while(1) {
		if( c == EOF )
			c = '\n'; /* ignore files not properly ending */
		if( i+1 >= buflen )
			return -2;
		if( c == '\\' && !escaped )
			escaped = true;
		else {
			if( c == ';' || c == '=' || c == '\n' ) {
				while( i > 0 && isspace(buffer[i-1]))
					i--;
				buffer[i] = '\0';
				return c;
			}
			escaped = false;
			buffer[i++] = c;
		}
		c = mime_read_char(linenr,colnr,f);
	}
}

static inline int mime_read_filename(int *linenr,int *colnr,
		char **bufptr, size_t *buflen, FILE *f) {
	int c;
	size_t i = 0;
	bool escaped = false;
	char *buffer = *bufptr;

	c = mime_read_charskipspace(linenr,colnr,f);

	while(1) {
		if( c == EOF )
			c = '\n'; /* ignore files not properly ending */
		if( i+1 >= *buflen ) {
			char *nb; 

			if( i >= SIZE_MAX - 1029 )
				return -2;
			nb = realloc(buffer,i+1024);
			if( nb == NULL )
				return -2;
			*bufptr = nb;
			buffer = nb;
			*buflen = i+1023;
		}
		if( c == '\\' && !escaped )
			escaped = true;
		else {
			char *homedir;

			if( c == '\n' ) {
				while( i > 0 && isspace(buffer[i-1]))
					i--;
				buffer[i] = '\0';
				return c;
			}
			if( !escaped && c == '~' && 
			    (homedir = getenv("HOME")) != NULL ) {
				size_t l = strlen(homedir);

				if( i+l >= *buflen ) {
					char *nb; 

					if( i >= SIZE_MAX -1 - l )
						return -2;
					nb = realloc(buffer,i+l+1);
					if( nb == NULL )
						return -2;
					*bufptr = nb;
					buffer = nb;
					*buflen = i+l+1;
				}
				memcpy(buffer+i,homedir,l);
				i+=l;
			} else {
				buffer[i++] = c;
			}
			escaped = false;
		}
		c = mime_read_char(linenr,colnr,f);
	}
}

static inline int mime_read_data(int *linenr,int *colnr,
		char **buffer_p, FILE *f) {
	int c;
	size_t i = 0;
	bool escaped = false;
	size_t buflen = 1000;
	char *buffer = malloc(128);

	if( buffer == NULL )
		return -2;

	c = mime_read_charskipspace(linenr,colnr,f);

	while(1) {
		if( c == EOF )
			c = '\n'; /* ignore files not properly ending */
		if( i+1 >= buflen ) {
			char *h;
			buflen += 128;
			h = realloc(buffer,buflen);
			if( h == NULL )
				return -2;
			buffer = h;
		}
		if( c == '\\' && !escaped ) {
			escaped = true;
			buffer[i++] = c;
		} else {
			if( c == ';' || c == '\n' ) {
				char *h;
				while( i > 0 && isspace(buffer[i-1]))
					i--;
				buffer[i++] = '\0';
				h = realloc(buffer,i);
				*buffer_p = (h==NULL)?buffer:h;
				return c;
			}
			escaped = false;
			buffer[i++] = c;
		}
		c = mime_read_char(linenr,colnr,f);
	}
}

static bool runtest(const char *command) {
	/* we are an X program, we have a display */
	if( strcmp(command,"test -n \"$DISPLAY\"") == 0 ||
	    strcmp(command,"test \"$DISPLAY\"") == 0 ||
	    strcmp(command,"test \"$DISPLAY\" != \"\"") == 0 ) {
		return true;
	} else {
		int r;
		r = system(command);
		if( r != -1 && WIFEXITED(r) ) {
			return WEXITSTATUS(r)==0;
		} else {
			fprintf(stderr,"Error starting '%s'\n",command);
			return false;
		}
	}
}

/* unescape the given string, overwriting the existing content */
static void mime_unescape_simple(char *v) {
	char *n = v;
	bool escaped = false;
	char c;

	while( (c = *(v++)) != '\0' ) {
		if( c == '\\' && !escaped ) 
			escaped = true;
		else {
			escaped = false;
			*(n++) = c;
		}
	}
}

/* unsecape the given string, splitting it in two parts seperated by
 * a '%s'. */
static bool mime_split_by_placeholder(char *v, const char **snd) {
	char *n = v;
	bool escaped = false;
	char c;

	*snd = NULL;
	while( (c = *(v++)) != '\0' ) {
		if( escaped ) {
			escaped = false;
			*(n++) = c;
		} else if( c == '\\' )
			escaped = true;
		else if( c != '%' ) {
			escaped = false;
			*(n++) = c;
		} else if( *v == '%' ) {
			escaped = false;
			*(n++) = c;
			v++;
		} else if( *v != 's' ) {
			escaped = false;
			*(n++) = c;
		} else {
			if( *snd != NULL ) {
				return false;
			} else {
				*(n++) = '\0';
				*snd = n;
				v++;
			}
		}
	}
	*n = '\0';
	return *snd != NULL;
}

static bool checkCallable(const char *v) {
	bool escaped = false;
	char c;

	if( strcmp(v,"false") == 0 )
		return false;
	if( strcmp(v,"LOAD") == 0 )
		return true;
	if( strcmp(v,"OPEN") == 0 )
		return true;
	while( (c = *(v++)) != '\0' ) {
		if( c == '\\' && !escaped ) 
			escaped = true;
		else {
			escaped = false;
			if( c == '%' && *v == 's' )
				return true;
		}
	}
	return false;
}

static int mime_parse_line(const char *filename,int *linenr,FILE *f) {
	int colnr=0;
	char mime_subtype[100];
	int e;
	struct mime_action *type;
	struct mime_filetype *typep;
	char *p;

	*linenr += 1;
	e = mime_read_statement(linenr,&colnr,mime_subtype,sizeof(mime_subtype),f);
	if( mime_subtype[0] == '#' || mime_subtype[0] == '\0' ) {
		/* ignore comments and empty lines */
		if( e != '\n' )
			mime_read_skipline(linenr,&colnr,f);
		return 0;
	}
	if( e != ';' ) {
		fprintf(stderr,"%s:%d:%d: Malformed mailcap: expecting ';' after type\n",
				filename,*linenr,colnr);
		if( e != '\n' )
			mime_read_skipline(linenr,&colnr,f);
		return 0;
	}
	if( strcmp(mime_subtype,"include") == 0 || strcmp(mime_subtype,"!include") == 0) {
		char *fn;
		size_t buflen = 
#ifdef PATH_MAX
			PATH_MAX;
#else
#ifdef MAXPATHLEN
			MAXPATHLEN;
#else
			4094;
#endif
#endif
		fn = malloc(buflen);
		if( filename == NULL )
			return -2;

		e = mime_read_filename(linenr,&colnr,&fn,&buflen,f);
		if( e != '\n' ) {
			fprintf(stderr,"%s:%d:%d: Malformed include line (too long?)\n",
				filename,*linenr,colnr);
			mime_read_skipline(linenr,&colnr,f);
		} else {
			mailcap_parse(fn);
		}
		free(fn);
		return 0;
	}
	p = strchr(mime_subtype,'/');
	if( p == NULL ) {
		fprintf(stderr,"%s:%d:%d: Malformed mailcap: expecting '/' within type\n",
				filename,*linenr,colnr);

	}

	type = calloc(1,sizeof(struct mime_action));
	if( type == NULL )
		return -2;
	/* no test means all tests worked */
	type->testOK = true;
	
	e = mime_read_data(linenr,&colnr,&type->view,f);
	if( e < 0 ) {
		mime_free(type);
		return e;
	}
	type->viewOK = checkCallable(type->view);
	while( e != '\n' ) {
		char var[100];
		int oldlinenr = *linenr;

		e = mime_read_statement(linenr,&colnr,var,sizeof(var),f);
		if( e < 0 ) {
			fprintf(stderr,"%s:%d:%d: overlong mailcap attribute-name (starting in line %d)\n", filename,*linenr,colnr,oldlinenr);
			mime_read_skipline(linenr,&colnr,f);
			mime_free(type);
			return 0;
		}
		if( e == '=' ) {
			char *v;
			e = mime_read_data(linenr,&colnr,&v,f);

			if( strcmp(var,"notes") == 0 ) {
				if( mime_savememory )
					free(v);
				else {
					mime_unescape_simple(v);
					type->notes = v;
				}
			} else if( strcmp(var,"test") == 0 ) {
				mime_unescape_simple(v);
				type->testOK = runtest(v);
				if( mime_savememory )
					free(v);
				else
					type->test = v;
			} else if( strcmp(var,"print") == 0 ) {
				type->print = v;
			} else if( strcmp(var,"compose") == 0 ) {
				type->compose = v;
			} else if( strcmp(var,"description") == 0 ) {
				if( mime_savememory )
					free(v);
				else {
					mime_unescape_simple(v);
					type->description = v;
				}
			} else if( strcmp(var,"nametemplate") == 0 ) {
				const char *snd;
				if( mime_split_by_placeholder(v, &snd)) {
					type->nameprefix = v;
					type->namesuffix = snd;
				} else {
					fprintf(stderr,
"%s:%d:%d: too few or too many %%s in nametamplate! Ignoring it.\n", filename,*linenr,colnr);
					free(v);
				}
			} else if( strcmp(var,"x11-bitmap") == 0 ) {
				mime_unescape_simple(v);
				type->x11bitmap = v;
			} else if( strcmp(var,"edit") == 0 ) {
				type->edit = v;
				type->editOK |= checkCallable(type->edit);
			} else if( strcmp(var,"textualnewlines") == 0 ) {
				/* not yet supported */
				free(v);
			} else if( strcmp(var,"composetyped") == 0 || 
				   strcmp(var,"composedtype") == 0) {
				free(v);
			} else if( strcmp(var,"priority") == 0 ) {
				mime_unescape_simple(v);
				type->priority = atoi(v);
				free(v);
			} else if( strcmp(var,"x-xfm-drop") == 0 ) {
				type->drop = v;
			} else {
				if( var[0] != 'x' || var[1] != '-' || strncmp(var,"x-xfm-",6) == 0 ) 
					fprintf(stderr,"%s:%d:%d: unknown mailcap entry %s\n",filename,*linenr,colnr,var);
				free(v);
			}
		} else {
			if( var[0] == '\0' ) { 
				; /* ignore trailing ';' */
			} else if( strcmp(var,"needsterminal") == 0 ) {
				type->needsterminal = true;
			} else if( strcmp(var,"copiousoutput") == 0 ) {
				type->copiosoutput = true;
			} else
				fprintf(stderr,"%s:%d:%d: unknown mailcap flag %s\n",filename,*linenr,colnr,var);
		}
	}
	if( !type->testOK && mime_savememory )
		free(type);
	else {
		typep = mime_get_filetype(mime_subtype);
		if( typep == NULL ) {
			free(type);
			return -2;
		}
		type->type = typep;
		if( typep->actions == NULL ) {
			typep->actions = type;
		} else if( mime_savememory ) {
			free(type);
			return 1;
		} else {
			struct mime_action *a;

			for( a = typep->actions ; a->next != NULL ; a = a->next)
				;
			a->next = type;
		}
	}

	return 1;
}

static int mailcap_parse(const char *filename) {
	int linenr = 0;
	FILE *f;
	int e;

	f = fopen(filename,"r");
	if( f == NULL )
		return 0;
	while( !feof(f) ) {
		e = mime_parse_line(filename,&linenr,f);
		if( e < 0 ) {
			fclose(f);
			return e;
		}
	}
	if( ferror(f) ) {
		fprintf(stderr,"Error reading from '%s'\n",filename);
	}
	if( fclose(f) ) {
		e = errno;
		fprintf(stderr,"Error reading from '%s': %d=%s\n",
				filename,e,strerror(e));
	}
	return 1;
}

int mime_init(void) {
	assert(catchallall == NULL);
	mime_root = NULL;
	catchallall = newfiletype((char*)"default","default",NULL);
	if( catchallall == NULL )
		return -2;
	return 1;
}

int mime_parse_mailcap(const char *filename) {
	assert(catchallall != NULL);
	return mailcap_parse(filename);
}

static int types_parse(const char *filename, initfunc func) {
	FILE *f;
	char buffer[2000];

	f = fopen(filename,"r");
	if( f == NULL )
		return 0;
	while( fgets(buffer,sizeof(buffer),f) != NULL ) {
		char *s,*e;
		size_t l;
		struct mime_filetype *type;
		bool added;

		l = strlen(buffer);
		if( l == 0 )
			continue;
		if( l>= sizeof(buffer)-1 && buffer[l-1] != '\n' ) { 
			/* overlong line  */
			do {
				if( fgets(buffer,sizeof(buffer),f) == NULL )
					break;
			} while( buffer[strlen(buffer)-1] != '\n');
			continue;
		}
		/* Only overwrite newline when it is a proper text
		 * file and has one. (corrupt ones miss one in the last line) */
		if( buffer[l-1] == '\n' )
			buffer[--l] = '\0';
		s = buffer;
		while( *s == ' ' || *s == '\t' )
			s++;
		if( *s == '#' )
			continue;
		e = s;
		while( *e != '\0' && *e != ' ' && *e != '\t' )
			e++;
		if( *e != '\0' ) {
			*(e++) = '\0';
		}
		if( strcmp(s,"!include") == 0 || strcmp(s,"include") == 0 ) {
			while( *e == ' ' || *e == '\t' )
				e++;
			if( *e == '\0' ) {
				fprintf(stderr,"%s: unexpected end of line!\n",filename);
				continue;
			}
			if( e[0] == '~' && e[1] == '/' ) {
				const char *homedir = getenv("HOME");
				size_t homelen = strlen(homedir);
				size_t elen = strlen(e);
				char *n = malloc(homelen+elen);

				if( n == NULL ) {
					fclose(f);
					return -2;
				}
				memcpy(n,homedir,homelen);
				memcpy(n+homelen,e+1,elen);
				types_parse(n,func);
				free(n);
			} else
				types_parse(e,func);
			continue;
		}
		type = mime_get_filetype(s);
		if( type == NULL ) {
			fclose(f);
			return -2;
		}
		/* prior ones overwrite later ones */
		if( type->official ) 
			continue;
		added = false;
		while( *e != '\0' ) {
			s = e;
			while( *s == ' ' || *s == '\t' )
				s++;
			while( *e != '\0' && *e != ' ' && *e != '\t' )
				e++;
			if( *e != '\0' ) {
				*(e++) = '\0';
			}
			if( *s != '\0' ) {
				added = added || suffix_add(s,e-s,type);
			}
		}
		type->official = true;
		if( added )
			func(type);

	}
	if( ferror(f) ) {
		fprintf(stderr,"Error reading from '%s'\n",filename);
	}
	if( fclose(f) ) {
		int e = errno;
		fprintf(stderr,"Error reading from '%s': %d=%s\n",
				filename,e,strerror(e));
	}
	return 1;
}

int mime_parse_types(const char *filename, initfunc func) {
	assert(catchallall != NULL);
	return types_parse(filename,func);
}

bool mime_best_action(struct mime_filetype *mime_type,const char *directory,const char *filename,enum mime_start_type how,
		mime_action_func *start, bool echo) {
	bool r;
	static const char * const typestring[3] = {"edit or view","edit","view"};
	assert(mime_type != NULL);

	if( echo ) {
		fprintf(stderr,"Searching how to %s the file '%s' in '%s' of type '%s'\n",typestring[how],filename,directory,mime_type->fulltypename);
	}
	if( how != MIME_START_VIEW ) {
		struct mime_filetype *type = mime_type;

		for( type = mime_type ; type!=NULL ; type = type->fallback ) {
			struct mime_action *action = type->actions;

			for( action = type->actions ; action != NULL ; 
					action = action->next) {
				if( action->edit == NULL )
					continue;
				if( echo ) {
					fprintf(stderr,"possible action %s (edit for type %s):",action->edit,action->type->fulltypename);
				}
				if( !action->testOK ) {
					if( echo )
						fprintf(stderr," test negative\n");
					continue;
				}
				if( !action->editOK ) {
					if( echo )
						fprintf(stderr," not suitable (missing %%s)\n");
					continue;
				}
				if( echo )
					fprintf(stderr," trying...\n");
				r = start(action,action->edit,directory,filename);
				if( r )
					return r;
				
			}
		}
	}
	if( how != MIME_START_EDIT ) {
		struct mime_filetype *type = mime_type;

		for( type = mime_type ; type!=NULL ; type = type->fallback ) {
			struct mime_action *action = type->actions;

			for( action = type->actions ; action != NULL ; 
					action = action->next) {
				if( action->view == NULL )
					continue;
				if( echo ) {
					fprintf(stderr,"possible action %s (view for type %s):",action->view,action->type->fulltypename);
				}
				if( !action->testOK ) {
					if( echo )
						fprintf(stderr," test negative\n");
					continue;
				}
				if( !action->viewOK ) {
					if( echo )
						fprintf(stderr," not suitable (missing %%s)\n");
					continue;
				}
				if( echo )
					fprintf(stderr," trying...\n");
				r = start(action,action->view,directory,filename);
				if( r )
					return r;
				
			}
		}
	}
	if( echo ) {
		fprintf(stderr,"nothing found\n");
	}
	return false;
}
