/*
 *  This file is part of X-File Manager XFM
 *  ----------------------------------------------------------------------
  FmPopup.c

  (c) Simon Marlow 1990-92
  (c) Albert Graef 1994

  modified 1-29-95 by rodgers@lvs-emh.lvs.loral.com (Kevin M. Rodgers)
  to add filtering of icon/text directory displays by a filename filter.

  modified 7-1997 by strauman@sun6hft.ee.tu-berlin.de to add
  different enhancements (see README-1.4).

  modified 2004,2005,2006 by Bernhard R. Link (see Changelog)

  Routines for creating and managing popup forms and dialog boxes
 *  ----------------------------------------------------------------------
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
#include <xfmconfig.h>

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <assert.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw3d/Dialog.h>
#include <X11/Xaw3d/Toggle.h>

#include "global.h"
#include "FileList.h"
#include "Fm.h"
#ifdef ENHANCE_HISTORY
#include "Am.h" 	/* need to know aw.shell */
#include "FmHistory.h"
#endif

#ifdef ENHANCE_SELECTION
#include "FmSelection.h"
#endif

/*---------------------------------------------------------------------------
  STATIC DATA
---------------------------------------------------------------------------*/

typedef struct {
  struct PopupDialog *mkdir, *createFile, *goTo, *rename, *move, *copy, *link,
		     *select, *filter;
} PopupsRec;

static PopupsRec popups;

/*---------------------------------------------------------------------------
  PRIVATE FUNCTIONS
---------------------------------------------------------------------------*/

static char *dir_prefix(char *dir, char *path)
{
  char *p;
  if ((p = strrchr(path, '/'))) {
    strcpy(dir, path);
    if (p == path)
      dir[1] = '\0';
    else
      dir[p-path] = '\0';
  } else
    dir[0] = '\0';
  return dir;
}

static PDCallbackProc 
  mkdirOkCb, mkdirCancelCb, createFileOkCb, createFileCancelCb, goToOkCb,
  goToCancelCb, moveOkCb, moveCancelCb, copyOkCb, copyCancelCb, linkOkCb,
  linkCancelCb, selectAddCb, selectRemoveCb, selectCancelCb, selectReplaceCb,
  filterOkCb, filterClearCb, filterCancelCb;

static void mkdirOkCb(UNUSED(Widget w), struct PopupDialog *p,
                      UNUSED(void *call_data))
{
  char *dir = getPopupAnswer(p,0);
  FileWindowRec *fw = p->data;

  popdownPopup(p);
  fnexpand(dir);
  if (chdir(fw->directory))
    sysError("System error:");
  else if (mkdir(dir, user.umask & 0777)) {
    char s[0xff];
    sprintf(s, "Error creating folder %s:", dir);
    sysError(s);
  } else
    intUpdate();
  freeze = False;
}

/*---------------------------------------------------------------------------*/

static void mkdirCancelCb(UNUSED(Widget w), struct PopupDialog *p,
                          UNUSED(void *call_data))
{
  popdownPopup(p);
  freeze = False;
}

/*---------------------------------------------------------------------------*/

static void createFileOkCb(UNUSED(Widget w), struct PopupDialog *p,
		           UNUSED(void *call_data))
{
  char *fn = getPopupAnswer(p,0);
  FileWindowRec *fw = p->data;

  popdownPopup(p);
  fnexpand(fn);
  if (chdir(fw->directory))
    sysError("System error:");
  else if (create(fn, user.umask & 0666)) {
    char s[0xff];
    sprintf(s, "Error creating file %s:", fn);
    sysError(s);
  } else
    intUpdate();
  freeze = False;
}

/*---------------------------------------------------------------------------*/

static void createFileCancelCb(UNUSED(Widget w), struct PopupDialog *p,
                               UNUSED(void *call_data))
{
  popdownPopup(p);
  freeze = False;
}

/*---------------------------------------------------------------------------*/

static void goToOkCb(UNUSED(Widget w), struct PopupDialog *p,
		     UNUSED(void *call_data))
{
  char *dir = getPopupAnswer(p,0);
  FileWindowRec *fw = p->data;
  char path[MAXPATHLEN];
#ifdef ENHANCE_SCROLL
  Boolean keep_position=True;
#endif

  popdownPopup(p);
  fnexpand(dir);
#ifdef ENHANCE_BUGFIX
  /* if it's impossible to chdir(popups.fw->directory)
   * it still might be possible to chdir(popups.goTo_s) !
   * eg. if the actual directory was deleted, an we want to
   * change to another one.
   */
  // TODO: this breaks chdir to relative directories, doesn't it?
  if (chdir(dir)) {
    char s[0xff];
    sprintf(s, "Can't open folder %s:", dir);
    sysError(s);
	if (chdir(fw->directory)) {
    	  sprintf(s, "Can't open folder %s:", fw->directory);
    	  sysError(s);
	}
#else
  if (chdir(popups.fw->directory) || chdir(dir)) {
    char s[0xff];
    sprintf(s, "Can't open folder %s:", dir);
    sysError(s);
#endif
  } else if (!getwd(path))
    sysError("System error:");
  else {
#ifdef ENHANCE_SCROLL
    if (!(keep_position= (!strcmp(fw->directory,path))))
#endif
    strcpy(fw->directory, path);
#ifdef ENHANCE_SCROLL
    updateFileDisplay(fw,keep_position);
#else
    updateFileDisplay(fw);
#endif
  }
  freeze = False;
}

/*---------------------------------------------------------------------------*/

static void goToCancelCb(UNUSED(Widget w), struct PopupDialog *p,
		         UNUSED(void *call_data))
{
  popdownPopup(p);
  freeze = False;
}

/*---------------------------------------------------------------------------*/

static void renameOkCb(UNUSED(Widget w), struct PopupDialog *p,
		       UNUSED(void *call_data))
{
  FileWindowRec *fw = p->data;
  struct stat stats;
  int i;
  char *from = NULL, to[MAXPATHLEN], todir[MAXPATHLEN];

  popdownPopup(p);
  strcpy(to, getPopupAnswer(p,0));
  fnexpand(to);

  if (chdir(fw->directory))

    sysError("System error:");

  else {

    struct stat stats1;

    for (i = 0; i < fw->n_files; i++)
      if (fw->files[i]->selected) {
	from = fw->files[i]->name;
	break;
      }

    if (!strcmp(from, ".") || !strcmp(from, "..")) {
      error("Cannot rename . or ..", "");
      goto out;
    } else if (!lstat(to, &stats) && !lstat(from, &stats1) &&
	       stats.st_ino == stats1.st_ino) {
      error("Rename:", "Source and destination are identical");
      goto out;
    }

    if (exists(to) && resources.confirm_overwrite) {
      char s[0xff];
      sprintf(s, "Rename: file %s already exists", to);
      if (!confirm(s, "Overwrite?", ""))
	goto out;
    }

    if (rmove(from, to)) {
      char s[0xff];
      sprintf(s, "Error renaming %s:", from);
      sysError(s);
    } else {
      dir_prefix(todir, to);
      if ((*todir?chdir(todir):0) || !getwd(todir))
	sysError("System error:");
      markForUpdate(fw->directory); markForUpdate(todir);
      intUpdate();
    }

  }

 out:
  freeze = False;
}

/*---------------------------------------------------------------------------*/

static void renameCancelCb(UNUSED(Widget w), struct PopupDialog *p,
			   UNUSED(void *call_data))
{
  popdownPopup(p);
  freeze = False;
}

/*---------------------------------------------------------------------------*/

static void moveOkCb(UNUSED(Widget w), struct PopupDialog *p,
		     UNUSED(void *call_data))
{
  char *move = getPopupAnswer(p,0);
  FileWindowRec *fw = p->data;
  struct stat stats;
  int i, toi, n_moved = 0;
  char *from = NULL, to[MAXPATHLEN], todir[MAXPATHLEN];

  popdownPopup(p);
  strcpy(to, move);
  fnexpand(to);

  if (chdir(fw->directory))

    sysError("System error:");

  else {

    /* if target exists and is a directory, move the source into that
       directory */

    if (!stat(to, &stats) && S_ISDIR(stats.st_mode)) {

      if (chdir(to) || !getwd(to) || chdir(fw->directory)) {
	sysError("System error:");
	goto out;
      } else if (!strcmp(fw->directory, to)) {
	error("Move:", "Source and destination are identical");
	goto out;
      }

      strcpy(todir, to);

      toi = strlen(to);
      if (to[toi-1] != '/') {
	to[toi++] = '/';
	to[toi] = '\0';
      }

      for (i=0; i < fw->n_files; i++)
	if (fw->files[i]->selected) {
	  if (!strcmp(fw->files[i]->name, ".") ||
	      !strcmp(fw->files[i]->name, "..")) {
	    error("Cannot move . or ..", "");
	    continue;
	  }
	  from = fw->files[i]->name;
	  strcpy(to+toi, from);
	  if (exists(to) && resources.confirm_overwrite) {
	    char s[0xff];
	    sprintf(s, "Move: file %s already exists at destination", from);
	    if (!confirm(s, "Overwrite?", "")) {
	      if (aborted)
		break;
	      else
		continue;
	    }
	  }
	  if (rmove(from,to)) {
	    char s[0xff];
	    sprintf(s, "Error moving %s:", from);
	    sysError(s);
	  } else
	    n_moved++;
	}
    }

    /* otherwise only a single file may be selected; move it to the target
       file */

    else if (fw->n_selections > 1) {

      error("Move: target for multiple files", "must be a folder");
      goto out;

    } else {

      struct stat stats1;

      for (i = 0; i < fw->n_files; i++)
	if (fw->files[i]->selected) {
	  from = fw->files[i]->name;
	  break;
	}

      if (!strcmp(from, ".") || !strcmp(from, "..")) {
	error("Cannot move . or ..", "");
	goto out;
      } else if (!lstat(to, &stats) && !lstat(from, &stats1) &&
		 stats.st_ino == stats1.st_ino) {
	error("Move:", "Source and destination are identical");
	goto out;
      }

      if (exists(to) && resources.confirm_overwrite) {
	char s[0xff];
	sprintf(s, "Move: file %s already exists", to);
	if (!confirm(s, "Overwrite?", ""))
	  goto out;
      }

      if (rmove(from, to)) {
	char s[0xff];
	sprintf(s, "Error moving %s:", from);
	sysError(s);
      } else {
	n_moved = 1;
	dir_prefix(todir, to);
	if ((*todir?chdir(todir):0) || !getwd(todir))
	  sysError("System error:");
      }
    }

    if (n_moved) {
      markForUpdate(fw->directory); markForUpdate(todir);
      intUpdate();
    }

  }

 out:
  freeze = False;
}

/*---------------------------------------------------------------------------*/

static void moveCancelCb(UNUSED(Widget w), struct PopupDialog *p,
			 UNUSED(void *call_data))
{
  popdownPopup(p);
  freeze = False;
}

/*---------------------------------------------------------------------------*/

static void copyOkCb(UNUSED(Widget w), struct PopupDialog *p,
		     UNUSED(void *call_data))
{
  char *copy = getPopupAnswer(p,0);
  FileWindowRec *fw = p->data;
  struct stat stats;
  int i, toi, n_copied = 0;
  char *from = NULL, to[MAXPATHLEN], todir[MAXPATHLEN];

  popdownPopup(p);
  strcpy(to, copy);
  fnexpand(to);

  if (chdir(fw->directory))

    sysError("System error:");

  else {

    /* if target exists and is a directory, copy the source into that
       directory */

    if (!stat(to, &stats) && S_ISDIR(stats.st_mode)) {

      if (chdir(to) || !getwd(to) || chdir(fw->directory)) {
	sysError("System error:");
	goto out;
      } else if (!strcmp(fw->directory, to)) {
	error("Copy:", "Source and destination are identical");
	goto out;
      }

      strcpy(todir, to);

      toi = strlen(to);
      if (to[toi-1] != '/') {
	to[toi++] = '/';
	to[toi] = '\0';
      }

      for (i=0; i < fw->n_files; i++)
	if (fw->files[i]->selected) {
	  if (!strcmp(fw->files[i]->name, ".") ||
	      !strcmp(fw->files[i]->name, "..")) {
	    error("Cannot copy . or ..", "");
	    continue;
	  }
	  from = fw->files[i]->name;
	  strcpy(to+toi, from);
	  if (exists(to) && resources.confirm_overwrite) {
	    char s[0xff];
	    sprintf(s, "Copy: file %s already exists at destination", from);
	    if (!confirm(s, "Overwrite?", "")) {
	      if (aborted)
		break;
	      else
		continue;
	    }
	  }
	  if (rcopy(from,to)) {
	    char s[0xff];
	    sprintf(s, "Error copying %s:", from);
	    sysError(s);
	  } else
	    n_copied++;
	}
    }

    /* otherwise only a single file may be selected; copy it to the target
       file */

    else if (fw->n_selections > 1) {

      error("Copy: target for multiple files", "must be a folder");
      goto out;

    } else {

      struct stat stats1;

      for (i = 0; i < fw->n_files; i++)
	if (fw->files[i]->selected) {
	  from = fw->files[i]->name;
	  break;
	}

      if (!strcmp(from, ".") || !strcmp(from, "..")) {
	error("Cannot copy . or ..", "");
	goto out;
      } else if (!lstat(to, &stats) && !lstat(from, &stats1) &&
		 stats.st_ino == stats1.st_ino) {
	error("Copy:", "Source and destination are identical");
	goto out;
      }

      if (exists(to) && resources.confirm_overwrite) {
	char s[0xff];
	sprintf(s, "Copy: file %s already exists", to);
	if (!confirm(s, "Overwrite?", ""))
	  goto out;
      }

      if (rcopy(from, to)) {
	char s[0xff];
	sprintf(s, "Error copying %s:", from);
	sysError(s);
      } else {
	n_copied = 1;
	dir_prefix(todir, to);
	if ((*todir?chdir(todir):0) || !getwd(todir))
	  sysError("System error:");
      }
    }

    if (n_copied) {
      markForUpdate(todir);
      intUpdate();
    }

  }

 out:
  freeze = False;
}

/*---------------------------------------------------------------------------*/

static void copyCancelCb(UNUSED(Widget w), struct PopupDialog *p,
			 UNUSED(void *call_data))
{
  popdownPopup(p);
  freeze = False;
}

/*---------------------------------------------------------------------------*/

static void linkOkCb(UNUSED(Widget w), struct PopupDialog *p,
		     UNUSED(void *call_data))
{
  FileWindowRec *fw = p->data;
  struct stat stats;
  int i, namei, toi, n_linked = 0;
  char *from = NULL, name[MAXPATHLEN], to[MAXPATHLEN], todir[MAXPATHLEN];

  popdownPopup(p);
  strcpy(to, getPopupAnswer(p,0));
  fnexpand(to);

  strcpy(name, fw->directory);

  namei = strlen(name);
  if (name[namei-1] != '/') {
    name[namei++] = '/';
    name[namei] = '\0';
  }

  if (chdir(fw->directory))

    sysError("System error:");

  else {

    /* if target exists and is a directory, link the source into that
       directory */

    if (!stat(to, &stats) && S_ISDIR(stats.st_mode)) {

      if (chdir(to) || !getwd(to) || chdir(fw->directory)) {
	sysError("System error:");
	goto out;
      } else if (!strcmp(fw->directory, to)) {
	error("Link:", "Source and destination are identical");
	goto out;
      }

      strcpy(todir, to);

      toi = strlen(to);
      if (to[toi-1] != '/') {
	to[toi++] = '/';
	to[toi] = '\0';
      }

      for (i=0; i < fw->n_files; i++)
	if (fw->files[i]->selected) {
	  from = fw->files[i]->name;
	  strcpy(name+namei, from);
	  strcpy(to+toi, from);
	  if (exists(to) && resources.confirm_overwrite) {
	    char s[0xff];
	    sprintf(s, "Link: file %s already exists at destination", from);
	    if (!confirm(s, "Overwrite?", "")) {
	      if (aborted)
		break;
	      else
		continue;
	    }
	  }
	  if (symlink(name,to)) {
	    char s[0xff];
	    sprintf(s, "Error linking %s:", from);
	    sysError(s);
	  } else
	    n_linked++;
	}
    }

    /* otherwise only a single file may be selected; link it to the target
       file */

    else if (fw->n_selections > 1) {

      error("Link: target for multiple files", "must be a folder");
      goto out;

    } else {

      struct stat stats1;

      for (i = 0; i < fw->n_files; i++)
	if (fw->files[i]->selected) {
	  from = fw->files[i]->name;
	  break;
	}

      strcpy(name+namei, from);

      if (!lstat(to, &stats) && !lstat(from, &stats1) &&
		 stats.st_ino == stats1.st_ino) {
	error("Link:", "Source and destination are identical");
	goto out;
      }

      if (exists(to) && resources.confirm_overwrite) {
	char s[0xff];
	sprintf(s, "Link: file %s already exists", to);
	if (!confirm(s, "Overwrite?", ""))
	  goto out;
      }

      if (symlink(name, to)) {
	char s[0xff];
	sprintf(s, "Error linking %s:", from);
	sysError(s);
      } else {
	n_linked = 1;
	dir_prefix(todir, to);
	if ((*todir?chdir(todir):0) || !getwd(todir))
	  sysError("System error:");
      }
    }

    if (n_linked) {
      markForUpdate(todir);
      intUpdate();
    }

  }

 out:
  freeze = False;
}

/*---------------------------------------------------------------------------*/

static void linkCancelCb(UNUSED(Widget w), struct PopupDialog *p,
			 UNUSED(void *call_data))
{
  popdownPopup(p);
  freeze = False;
}

/*---------------------------------------------------------------------------*/

/* The following variant of fnmatch matches the . and .. dirs only if
   specified explicitly. */

#define fnmatchnodot(pattern,fn) (strcmp(fn,".")&&strcmp(fn,"..")? \
				  fnmatch(pattern,fn):!strcmp(pattern,fn))

static void selectReplaceCb(UNUSED(Widget w), struct PopupDialog *p,
			    UNUSED(void *call_data))
{
  FileWindowRec *fw = p->data;
  char *select_s = getPopupAnswer(p,0);
  int i;
  Pixel pix;
  Boolean new_val;
  int onewid=XtIsSubclass(fw->icon_box,fileListWidgetClass);

  popdownPopup(p);
  fw->n_selections = 0;
  fw->n_bytes_selected = 0;
  for (i=0; i<fw->n_files; i++) {
    if (onewid || fw->files[i]->icon.toggle) {
      if (fnmatchnodot(select_s, fw->files[i]->name)) {
	new_val = True;
	fw->n_selections++;
	fw->n_bytes_selected += fw->files[i]->stats.st_size;
      }
      else
	new_val = False;
      if (new_val!= fw->files[i]->selected) {
	fw->files[i]->selected = new_val;
        if (!onewid) {
          XtVaGetValues(fw->files[i]->icon.toggle,
	  	      fw->files[i]->selected?XtNforeground:XtNbackground,
		      &pix, NULL);
          XtVaSetValues(fw->files[i]->icon.toggle, XtNborder,
		      (XtArgVal) pix, NULL);
        } else 
	  FileListRefreshItem((FileListWidget)fw->icon_box,i);
     }
    }
  }
  updateStatus(fw);
  freeze = False;
#ifdef ENHANCE_SELECTION
  FmOwnSelection(fw,CurrentTime);
#endif
}

/*---------------------------------------------------------------------------*/

static void selectAddCb(UNUSED(Widget w), struct PopupDialog *p,
		        UNUSED(void *call_data))
{
  char *select_s = getPopupAnswer(p,0);
  FileWindowRec *fw = p->data;
  int i;
  Pixel pix;
  int onewid=XtIsSubclass(fw->icon_box,fileListWidgetClass);
  
  popdownPopup(p);
  for(i=0; i<fw->n_files; i++)
    if (onewid || fw->files[i]->icon.toggle) {
      if (!fw->files[i]->selected && 
	  (fnmatchnodot(select_s, fw->files[i]->name))) {
	fw->files[i]->selected = True;
	fw->n_selections++;
	fw->n_bytes_selected += fw->files[i]->stats.st_size;
	if (!onewid) {
	  XtVaGetValues(fw->files[i]->icon.toggle, XtNforeground, &pix,
		      NULL);
	  XtVaSetValues(fw->files[i]->icon.toggle, XtNborder,
		      (XtArgVal) pix, NULL);
	} else FileListRefreshItem((FileListWidget)fw->icon_box,i);
      }
    }
  updateStatus(fw);
  freeze = False;
#ifdef ENHANCE_SELECTION
  FmOwnSelection(fw,CurrentTime);
#endif
}

/*---------------------------------------------------------------------------*/

static void selectRemoveCb(UNUSED(Widget w), struct PopupDialog *p,
		           UNUSED(void *call_data))
{
  char *select_s = getPopupAnswer(p,0);
  FileWindowRec *fw = p->data;
  int i;
  Pixel pix;
  int onewid=XtIsSubclass(fw->icon_box,fileListWidgetClass);
  
  popdownPopup(p);
  for(i=0; i<fw->n_files; i++)
    if (onewid || fw->files[i]->icon.toggle) {
      if (fw->files[i]->selected && 
	  (fnmatch(select_s, fw->files[i]->name))) {
	fw->files[i]->selected = False;
	fw->n_selections--;
	fw->n_bytes_selected -= fw->files[i]->stats.st_size;
	if (!onewid) {
	  XtVaGetValues(fw->files[i]->icon.toggle, XtNbackground, &pix,
		      NULL);
	  XtVaSetValues(fw->files[i]->icon.toggle, XtNborder,
		      (XtArgVal) pix, NULL);
	} else FileListRefreshItem((FileListWidget)fw->icon_box,i);
      }
    }
  updateStatus(fw);
  freeze = False;
#ifdef ENHANCE_SELECTION
  FmOwnSelection(fw,CurrentTime);
#endif
}

/*---------------------------------------------------------------------------*/

static void selectCancelCb(UNUSED(Widget w), struct PopupDialog *p,
			   UNUSED(void *call_data))
{
  popdownPopup(p);
  freeze = False;
}

/*---------------------------------------------------------------------------*/

/* KMR */
static void filterCancelCb(UNUSED(Widget w), struct PopupDialog *p,
			   UNUSED(void *call_data))
{
  popdownPopup(p);
  freeze = False;
}

/*---------------------------------------------------------------------------*/

/* KMR */
static void filterOkCb(UNUSED(Widget w), struct PopupDialog *p,
		       UNUSED(void *call_data))
{
  char *filter_s = getPopupAnswer(p,0);
  FileWindowRec *fw = p->data;

  popdownPopup(p);
  fw->do_filter = True;
  strcpy(fw->dirFilter,filter_s);
#ifdef ENHANCE_SCROLL
  updateFileDisplay(fw,True);
#else
  updateFileDisplay(fw);
#endif
  freeze = False;
}

/*---------------------------------------------------------------------------*/

/* KMR */
static void filterClearCb(UNUSED(Widget w), struct PopupDialog *p,
			  UNUSED(void *call_data))
{
  FileWindowRec *fw = p->data;

  popdownPopup(p);
  fw->do_filter = False;
  fw->dirFilter[0] = '\0';
#ifdef ENHANCE_SCROLL
  updateFileDisplay(fw,True);
#else
  updateFileDisplay(fw);
#endif
  freeze = False;
}

/*---------------------------------------------------------------------------
  Button information for popups
---------------------------------------------------------------------------*/

static const PDButtonRec mkdir_buttons[] = {
  { "ok", "Ok", mkdirOkCb },
  { "cancel", "Cancel", mkdirCancelCb }
};

static const PDButtonRec createFile_buttons[] = {
  { "ok", "Ok", createFileOkCb },
  { "cancel", "Cancel", createFileCancelCb }
};

static const PDButtonRec goTo_buttons[] = {
  { "ok", "Ok", goToOkCb },
  { "cancel", "Cancel", goToCancelCb }
};

static const PDButtonRec rename_buttons[] = {
  { "ok", "Ok", renameOkCb },
  { "cancel", "Cancel", renameCancelCb }
};

static const PDButtonRec move_buttons[] = {
  { "ok", "Ok", moveOkCb },
  { "cancel", "Cancel", moveCancelCb }
};

static const PDButtonRec copy_buttons[] = {
  { "ok", "Ok", copyOkCb },
  { "cancel", "Cancel", copyCancelCb }
};

static const PDButtonRec link_buttons[] = {
  { "ok", "Ok", linkOkCb },
  { "cancel", "Cancel", linkCancelCb }
};

static const PDButtonRec select_buttons[] = {
  { "replace", "Replace", selectReplaceCb },
  { "add", "Add", selectAddCb },
  { "remove", "Remove", selectRemoveCb },
  { "cancel", "Cancel", selectCancelCb }
};

static const PDButtonRec filter_buttons[] = {       /* KMR */
  { "ok", "Ok", filterOkCb },
  { "clear", "Clear", filterClearCb },
  { "cancel", "Cancel", filterCancelCb }
}; 

/*---------------------------------------------------------------------------
  Question information for popups
---------------------------------------------------------------------------*/

static const QuestionRec mkdir_questions[] = {
  { "Create folder:", MAXPATHLEN }
};

static const QuestionRec createFile_questions[] = {
  { "Create file:", MAXPATHLEN }
};

static const QuestionRec goTo_questions[] = {
  { "Go to folder:", MAXPATHLEN }
};

static const QuestionRec rename_questions[] = {
  { "Rename to:", MAXPATHLEN }
};

static const QuestionRec move_questions[] = {
  { "Move to:", MAXPATHLEN }
};

static const QuestionRec copy_questions[] = {
  { "Copy to:", MAXPATHLEN }
};

static const QuestionRec link_questions[] = {
  { "Link to:", MAXPATHLEN }
};

static const QuestionRec select_questions[] = {
  { "Filename pattern:", MAXPATHLEN }
};

static const QuestionRec filter_questions[] = {             /* KMR */
  { "Filename pattern:", MAXPATHLEN }
};

#ifdef ENHANCE_HISTORY
/*--------------------------------------------------------------------------
  PUBLIC VARIABLES
--------------------------------------------------------------------------*/

HistoryList path_history;

#endif

/*---------------------------------------------------------------------------
  PUBLIC FUNCTIONS
---------------------------------------------------------------------------*/

void createMainPopups()
{
#ifdef ENHANCE_HISTORY
  char *fixed[5];
  int	nfixed;

  /* Path History menu is initialized
   * (need to do this first because the history feature
   * provides an additional action that must be added to the action table
   * so the different popups can use it :-)
   */

  nfixed=0;
  fixed[nfixed++]=user.home;
#ifdef ENHANCE_USERINFO
  fixed[nfixed++]=user.cwd;
#endif
  fixed[nfixed++]=0;
  path_history=FmCreateHistoryList(
	0, /* let the menu's name be "fm_history" */
	aw.shell,
	fixed);

  /* Note: important resources of the history menu are _not_ configured
   *       statically but by the resource files.
   */
#endif

  /* New Folder */
  popups.mkdir = createPopupQuestions("mkdir", "New Folder", bm[DIR_BM], 
			       mkdir_questions, XtNumber(mkdir_questions),
			       mkdir_buttons, XtNumber(mkdir_buttons),
			       XtNumber(mkdir_buttons)-1);

  /* New File */
  popups.createFile = createPopupQuestions("create", "New File", bm[FILE_BM],
			       createFile_questions,
			       XtNumber(createFile_questions),
			       createFile_buttons,
			       XtNumber(createFile_buttons),
			       XtNumber(createFile_buttons)-1);

  /* Change current folder */
  popups.goTo = createPopupQuestions("goto", "Go To", bm[DIR_BM],
			       goTo_questions, XtNumber(goTo_questions),
			       goTo_buttons, XtNumber(goTo_buttons),
			       XtNumber(goTo_buttons)-1);

  /* Rename file */
  popups.rename = createPopupQuestions("rename", "Rename", bm[FILE_BM], 
			       rename_questions, XtNumber(rename_questions),
			       rename_buttons, XtNumber(rename_buttons),
			       XtNumber(rename_buttons)-1);

  /* Move file */
  popups.move = createPopupQuestions("move", "Move", bm[FILES_BM], 
			       move_questions, XtNumber(move_questions),
			       move_buttons, XtNumber(move_buttons),
			       XtNumber(move_buttons)-1);

  /* Copy file */
  popups.copy = createPopupQuestions("copy", "Copy", bm[FILES_BM],
			       copy_questions, XtNumber(copy_questions),
			       copy_buttons, XtNumber(copy_buttons),
			       XtNumber(copy_buttons)-1);

  /* Create link */
  popups.link = createPopupQuestions("link", "Link", bm[SYMLNK_BM],
			       link_questions, XtNumber(link_questions),
			       link_buttons, XtNumber(link_buttons),
			       XtNumber(link_buttons)-1);

  /* Select */
  popups.select = createPopupQuestions("select", "Select", bm[FILES_BM],
			       select_questions, XtNumber(select_questions),
			       select_buttons, XtNumber(select_buttons),
			       XtNumber(select_buttons)-1);

  /* Filter */  /* KMR */
  popups.filter = createPopupQuestions("filter", "Filter", bm[FILES_BM],
			       filter_questions, XtNumber(filter_questions),
			       filter_buttons, XtNumber(filter_buttons),
			       XtNumber(filter_buttons)-1);

  /* Change Access Mode */
  createChmodPopup();

  /* Info */
  createInfoPopup();

  /* Errors */
  createErrorPopup();

  /* Deletions */
  createConfirmPopup();

}

/*---------------------------------------------------------------------------*/ 

void selectPopup(UNUSED(Widget w), FileWindowRec *fw, UNUSED(void *call_data))
{
  reloadPopupData(popups.select);
  freeze = True;
  popupPopupByCursor(popups.select, fw);
}

/*---------------------------------------------------------------------------*/ 

/* KMR */
void filterPopup(UNUSED(Widget w), FileWindowRec *fw, UNUSED(void *call_data))
{
  setPopupData(popups.filter,0,fw->dirFilter);
  freeze = True;
  popupPopupByCursor(popups.filter, fw);
}

/*---------------------------------------------------------------------------*/

void mkdirPopup(UNUSED(Widget w), FileWindowRec *fw, UNUSED(void *call_data))
{
  reloadPopupData(popups.mkdir);
  freeze = True;
  popupPopupByCursor(popups.mkdir, fw);
}

/*---------------------------------------------------------------------------*/

void createFilePopup(UNUSED(Widget w), FileWindowRec *fw, UNUSED(void *call_data))
{
  reloadPopupData(popups.createFile);
  freeze = True;
  popupPopupByCursor(popups.createFile, fw);
}

/*---------------------------------------------------------------------------*/

void goToPopup(UNUSED(Widget w), FileWindowRec *fw, UNUSED(void *call_data))
{
  reloadPopupData(popups.goTo);
  freeze = True;
  popupPopupByCursor(popups.goTo, fw);
}

/*---------------------------------------------------------------------------*/

void renamePopup(UNUSED(Widget w), FileWindowRec *fw, UNUSED(void *call_data))
{
  int i;

  assert( fw != NULL );

  if (!fw->n_selections) return;

  for (i = 0; i < fw->n_files; i++)
    if (fw->files[i]->selected) {
      setPopupData(popups.rename,0,fw->files[i]->name);
      break;
    }
  freeze = True;
  popupPopupByCursor(popups.rename, fw);
}

/*---------------------------------------------------------------------------*/

void movePopup(UNUSED(Widget w), FileWindowRec *fw, UNUSED(void *call_data))
{
  assert( fw != NULL );

  if (!fw->n_selections) return;

  reloadPopupData(popups.move);
  freeze = True;
  popupPopupByCursor(popups.move, fw);
}

/*---------------------------------------------------------------------------*/

void copyPopup(UNUSED(Widget w), FileWindowRec *fw, UNUSED(void *call_data))
{
  assert( fw != NULL );

  if (!fw->n_selections) return;

  reloadPopupData(popups.copy);
  freeze = True;
  popupPopupByCursor(popups.copy, fw);
}

/*---------------------------------------------------------------------------*/

void linkPopup(UNUSED(Widget w), FileWindowRec *fw, UNUSED(void *call_data))
{
  assert( fw != NULL );

  if (!fw->n_selections) return;

  reloadPopupData(popups.link);
  freeze = True;
  popupPopupByCursor(popups.link, fw);
}
