// File list. Taken from the FOX library and slightly modified.
#include "config.h"
#include "i18n.h"

#include <ctype.h>
#include <time.h>


#include <fox-1.6/fx.h>
#include <fox-1.6/fxkeys.h>
#include <fox-1.6/FXPNGIcon.h>
#include <fox-1.6/FXJPGIcon.h>
#include <fox-1.6/FXTIFIcon.h>
#if defined(linux)
#include <mntent.h>
#endif

#include "xfedefs.h"
#include "icons.h"
#include "FileList.h"
#include "InputDialog.h"
#include "File.h"
#include "FileDict.h"
#include "MessageBox.h"
#include "StringList.h"


// Minimum file name size in detailed view
#ifndef MIN_NAME_SIZE
#define MIN_NAME_SIZE 75
#endif

// For thumbnail image preview
#define MAX_BIG_ICON_SIZE 64
#define MAX_MINI_ICON_SIZE 20

// Time interval before refreshing the view
#define REFRESH_INTERVAL     1000
#define REFRESH_FREQUENCY    30

// Time interval before opening a folder
#define OPEN_INTERVAL		1000

#define HASH1(x,n) (((unsigned int)(x)*13)%(n))           // Probe Position [0..n-1]
#define HASH2(x,n) (1|(((unsigned int)(x)*17)%((n)-1)))   // Probe Distance [1..n-1]


#if defined(linux)
#define FSTAB_PATH "/etc/fstab"
#define MTAB_PATH "/proc/mounts"
FXStringDict* fsdevices=NULL; // Devices from fstab
FXStringDict* mtdevices=NULL; // Mounted devices
FXStringDict* updevices=NULL; // Responding devices
#endif


#define SELECT_MASK   (ICONLIST_EXTENDEDSELECT|ICONLIST_SINGLESELECT|ICONLIST_BROWSESELECT|ICONLIST_MULTIPLESELECT)



// Object implementation
FXIMPLEMENT(FileItem,FXIconItem,NULL,0)


// Map
FXDEFMAP(FileList) FileListMap[]={
                                     FXMAPFUNC(SEL_DRAGGED,0,FileList::onDragged),
                                     FXMAPFUNC(SEL_TIMEOUT,FileList::ID_REFRESH_TIMER,FileList::onRefreshTimer),
                                     FXMAPFUNC(SEL_TIMEOUT,FileList::ID_OPEN_TIMER,FileList::onOpenTimer),
                                     FXMAPFUNC(SEL_DND_ENTER,0,FileList::onDNDEnter),
                                     FXMAPFUNC(SEL_DND_LEAVE,0,FileList::onDNDLeave),
                                     FXMAPFUNC(SEL_DND_DROP,0,FileList::onDNDDrop),
                                     FXMAPFUNC(SEL_DND_MOTION,0,FileList::onDNDMotion),
                                     FXMAPFUNC(SEL_DND_REQUEST,0,FileList::onDNDRequest),
                                     FXMAPFUNC(SEL_BEGINDRAG,0,FileList::onBeginDrag),
									 FXMAPFUNC(SEL_ENDDRAG,0,FileList::onEndDrag),
									 FXMAPFUNC(SEL_COMMAND,FileList::ID_DIRECTORY_UP,FileList::onCmdDirectoryUp),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SORT_BY_NAME,FileList::onCmdSortByName),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SORT_BY_TYPE,FileList::onCmdSortByType),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SORT_BY_SIZE,FileList::onCmdSortBySize),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SORT_BY_EXT,FileList::onCmdSortByExt),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SORT_BY_TIME,FileList::onCmdSortByTime),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SORT_BY_USER,FileList::onCmdSortByUser),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SORT_BY_GROUP,FileList::onCmdSortByGroup),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SORT_BY_PERM,FileList::onCmdSortByPerm),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SORT_BY_DELTIME,FileList::onCmdSortByDeltime),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SORT_REVERSE,FileList::onCmdSortReverse),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SORT_CASE,FileList::onCmdSortCase),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_DIRS_FIRST,FileList::onCmdDirsFirst),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SET_PATTERN,FileList::onCmdSetPattern),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SHOW_HIDDEN,FileList::onCmdShowHidden),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_HIDE_HIDDEN,FileList::onCmdHideHidden),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_TOGGLE_HIDDEN,FileList::onCmdToggleHidden),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_TOGGLE_THUMBNAILS,FileList::onCmdToggleThumbnails),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_HEADER_CHANGE,FileList::onCmdHeader),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_REFRESH,FileList::onCmdRefresh),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_HEADER_CHANGE,FileList::onUpdHeader),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_DIRECTORY_UP,FileList::onUpdDirectoryUp),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SORT_BY_NAME,FileList::onUpdSortByName),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SORT_BY_TYPE,FileList::onUpdSortByType),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SORT_BY_SIZE,FileList::onUpdSortBySize),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SORT_BY_EXT,FileList::onUpdSortByExt),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SORT_BY_TIME,FileList::onUpdSortByTime),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SORT_BY_USER,FileList::onUpdSortByUser),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SORT_BY_GROUP,FileList::onUpdSortByGroup),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SORT_BY_PERM,FileList::onUpdSortByPerm),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SORT_BY_DELTIME,FileList::onUpdSortByDeltime),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SORT_REVERSE,FileList::onUpdSortReverse),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SORT_CASE,FileList::onUpdSortCase),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_DIRS_FIRST,FileList::onUpdDirsFirst),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SET_PATTERN,FileList::onUpdSetPattern),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SHOW_HIDDEN,FileList::onUpdShowHidden),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_HIDE_HIDDEN,FileList::onUpdHideHidden),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_TOGGLE_HIDDEN,FileList::onUpdToggleHidden),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_TOGGLE_THUMBNAILS,FileList::onUpdToggleThumbnails),
                                 };


// Object implementation
FXIMPLEMENT(FileList,FXIconList,FileListMap,ARRAYNUMBER(FileListMap))



// File List
FileList::FileList(FXComposite *p,FXObject* tgt,FXSelector sel,FXuint opts,FXint x,FXint y,FXint w,FXint h):
        FXIconList(p,tgt,sel,opts,x,y,w,h),directory(ROOTDIR),orgdirectory(ROOTDIR),pattern("*")
{
    flags|=FLAG_ENABLED|FLAG_DROPTARGET;
    associations=NULL;
    appendHeader(_("Name"),NULL,200);
    appendHeader(_("Size"),NULL,60);
    appendHeader(_("Type"),NULL,100);
    appendHeader(_("Extension"),NULL,100);
    appendHeader(_("Modified Date"),NULL,150);
    appendHeader(_("User"),NULL,50);
    appendHeader(_("Group"),NULL,50);
    appendHeader(_("Permissions"),NULL,100);

    // Initializations
	matchmode=FILEMATCH_FILE_NAME|FILEMATCH_NOESCAPE;
	associations=new FileDict(getApp());
    dropaction=DRAG_MOVE;
    sortfunc=ascendingCase;
    ignorecase=TRUE;
    dirsfirst=TRUE;
	allowrefresh=TRUE;
    timestamp=0;
    counter=0;
    prevIndex=-1;
    list=NULL;
	displaythumbnails=FALSE;
	backhist=NULL;
	forwardhist=NULL;
	
#if defined(linux)

	// Initialize the fsdevices, mtdevices and updevices lists
	// if it was not done in DirList (useful for XFileView, XFileQuery and XFileImage) 
    struct mntent *mnt;
    if (fsdevices==NULL)
	{
		// To list file system devices
    	fsdevices=new FXStringDict();
		FILE *fstab=setmntent(FSTAB_PATH,"r");
    	if(fstab)
    	{
        	while((mnt=getmntent(fstab)))
        	{
            	if(!::streq(mnt->mnt_type,MNTTYPE_IGNORE) && !::streq(mnt->mnt_type,MNTTYPE_SWAP)
					&& !::streq(mnt->mnt_type,"proc") && !::streq(mnt->mnt_dir,"/"))
            	{
                	if(!strncmp(mnt->mnt_fsname,"/dev/fd",7))
                    	fsdevices->insert(mnt->mnt_dir,"floppy");
                	else if (!strncmp(mnt->mnt_type,"iso",3))
                    	fsdevices->insert(mnt->mnt_dir,"cdrom");
                	else if (!strncmp(mnt->mnt_fsname,"/dev/zip",8))
                    	fsdevices->insert(mnt->mnt_dir,"zip");
                	else if (::streq(mnt->mnt_type,"nfs"))
                    	fsdevices->insert(mnt->mnt_dir,"nfsdisk");
                	else if (::streq(mnt->mnt_type,"smbfs"))
                    	fsdevices->insert(mnt->mnt_dir,"smbdisk");
                	else
                    	fsdevices->insert(mnt->mnt_dir,"harddisk");
            	}
        	}
        	endmntent(fstab);
    	}
	}
	if (mtdevices==NULL)
	{
		// To list mounted devices
    	mtdevices=new FXStringDict();
		FILE *mtab=setmntent(MTAB_PATH,"r");
		if(mtab)
		{
			while((mnt=getmntent(mtab)))
			{
				// To fix an issue with some Linux distributions
				if (!::streq(mnt->mnt_dir,"/dev/.static/dev"))
					mtdevices->insert(mnt->mnt_dir,mnt->mnt_type);
			}
			endmntent(mtab);
		}
	}
	if (updevices==NULL)
	{
		// To mark mount points that are up or down
    	updevices=new FXStringDict();
 		struct stat statbuf;
		FXString mtstate;
		FILE *mtab=setmntent(MTAB_PATH,"r");
		if(mtab)
    	{
        	while((mnt=getmntent(mtab)))
			{
				// To fix an issue with some Linux distributions
 				if (!::streq(mnt->mnt_dir,"/dev/.static/dev"))
				{
					if (lstatmt(mnt->mnt_dir,&statbuf)==-1)
						mtstate="down";
					else
						mtstate="up";	
					updevices->insert(mnt->mnt_dir,mtstate.text());
				}
			}
        	endmntent(mtab);
    	}
	}
#endif
	
	// Trahscan location (with trailing slash)
	FXString homelocation=getenv("HOME");
    if(homelocation=="")
        homelocation=ROOTDIR;
	trashlocation=homelocation+PATHSEPSTRING+TRASHPATH+PATHSEPSTRING;	
}


// Create the file list
void FileList::create()
{
    FXIconList::create();
    if(!deleteType)
        deleteType=getApp()->registerDragType(deleteTypeName);
    if(!urilistType)
        urilistType=getApp()->registerDragType(urilistTypeName);
	getApp()->addTimeout(this,ID_REFRESH_TIMER,REFRESH_INTERVAL);
}


// Open up folder when hovering long over a folder
long FileList::onOpenTimer(FXObject*,FXSelector,void*)
{
    FXint xx,yy,index;
    FXuint state;
    getCursorPosition(xx,yy,state);
    index=getItemAt(xx,yy);
    if(0<=index && isItemDirectory(index))
    {
        dropdirectory=getItemPathname(index);
        setDirectory(dropdirectory);
        getApp()->addTimeout(this,ID_OPEN_TIMER,OPEN_INTERVAL);
        prevIndex=-1;
    }
    return 1;
}

// Handle drag-and-drop enter
long FileList::onDNDEnter(FXObject* sender,FXSelector sel,void* ptr)
{
   FXIconList::onDNDEnter(sender,sel,ptr);

    // Keep original directory
    orgdirectory=getDirectory();

    return 1;
}


// Handle drag-and-drop leave
long FileList::onDNDLeave(FXObject* sender,FXSelector sel,void* ptr)
{
   FXIconList::onDNDLeave(sender,sel,ptr);

    if(prevIndex!=-1)
    {
        setItemMiniIcon(prevIndex,minifoldericon);
        setItemBigIcon(prevIndex,bigfoldericon);
        prevIndex=-1;
    }

    // Cancel open up timer
	getApp()->removeTimeout(this,ID_OPEN_TIMER);

    // Stop scrolling
    stopAutoScroll();

    // Restore original directory
    setDirectory(orgdirectory);
    return 1;
}


// Handle drag-and-drop motion
long FileList::onDNDMotion(FXObject* sender,FXSelector sel,void* ptr)
{
    FXEvent *event=(FXEvent*)ptr;
    FXint index=-1;

    // Cancel open up timer
	getApp()->removeTimeout(this,ID_OPEN_TIMER);

    // Start autoscrolling
    if(startAutoScroll(event,FALSE))
        return 1;

    // Give base class a shot
    if(FXIconList::onDNDMotion(sender,sel,ptr))
        return 1;

    // Dropping list of filenames
    if(offeredDNDType(FROM_DRAGNDROP,urilistType))
    {
        index=getItemAt(event->win_x,event->win_y);
        if(prevIndex!=-1 && prevIndex != index)
        {
            setItemMiniIcon(prevIndex,minifoldericon);
			setItemBigIcon(prevIndex,bigfoldericon);
            prevIndex=-1;
        }

        // Drop in the background
        dropdirectory=getDirectory();

        // What is being done (move,copy,link)
        dropaction=inquireDNDAction();

        // We will open up a folder if we can hover over it for a while
        if(0<=index && isItemDirectory(index))
        {
            // Set open up timer
            getApp()->addTimeout(this,ID_OPEN_TIMER,OPEN_INTERVAL);

            prevIndex=index;
            setItemMiniIcon(index,minifolderopenicon);
            setItemBigIcon(index,bigfolderopenicon);

            // Directory where to drop, or directory to open up
            dropdirectory=getItemPathname(index);
        }

        // See if dropdirectory is writable
        if(::isWritable(dropdirectory))
            acceptDrop(DRAG_ACCEPT);
        return 1;
    }
    return 0;
}


// Handle drag-and-drop drop
long FileList::onDNDDrop(FXObject* sender,FXSelector sel,void* ptr)
{
    FXuchar *data;
    FXuint len;
    FXbool showdialog=TRUE;
    File *f=NULL;
	FXbool ret;

    FXbool ask_before_copy=getApp()->reg().readUnsignedEntry("OPTIONS","ask_before_copy",TRUE);
	
	if(prevIndex!=-1)
    {
        setItemMiniIcon(prevIndex,minifoldericon);
        setItemBigIcon(prevIndex,bigfoldericon);
        prevIndex=-1;
    }

    // Cancel open up timer
	getApp()->removeTimeout(this,ID_OPEN_TIMER);

    // Stop scrolling
    stopAutoScroll();

    // Restore original directory
    setDirectory(orgdirectory);

    // Perhaps target wants to deal with it
    if(FXIconList::onDNDDrop(sender,sel,ptr))
        return 1;

    // Get uri-list of files being dropped
    if(getDNDData(FROM_DRAGNDROP,urilistType,data,len))
    {
        FXRESIZE(&data,FXuchar,len+1);
        data[len]='\0';
        FXchar *p,*q;
        p=q=(FXchar*)data;
		        
		// Number of selected items
		FXString buf=p;
        int num=buf.contains('\n')+1;
		
		// Eventually correct the number of selected items
		// because sometimes there is another '\n' at the end of the string
		FXint pos=buf.rfind('\n');
		if (pos==buf.length()-1)
			num=num-1;

        // File object
		if (dropaction==DRAG_COPY)
            f=new File(this,_("File copy"),COPY);
        else if (dropaction==DRAG_MOVE)
				f=new File(this,_("File move"),MOVE);
		else if ((dropaction==DRAG_LINK) && (num==1))
            f=new File(this,_("File symlink"),SYMLINK);
        else if ((dropaction==DRAG_LINK) && (num>1))
		{
			MessageBox::error(this,BOX_OK,_("Error"),_("Can't symlink multiple selection!"));
			return 0;
		}
        else
            return 0;

        while(*p)
        {
            while(*q && *q!='\r')
                q++;
            FXString url(p,q-p);
            FXString source(FXURL::decode(FXURL::fileFromURL(url)));
            FXString target(dropdirectory+PATHSEPSTRING+FXPath::name(source));
            FXString sourcedir=FXPath::directory(source);
            FXString targetdir=FXPath::directory(target);
			
            // File operation dialog, if needed
            if(ask_before_copy & showdialog)
            {
                FXIcon *icon=NULL;
                FXString title,message;
                if (dropaction==DRAG_COPY)
                {
                    title=_("Copy ");
                    icon=copy_bigicon;
                    if (num==1)
                        message=title+source;
                    else
                        message=_("Copy ")+FXStringVal(num)+_(" files/folders.\nFrom: ")+sourcedir;
                }
                else if (dropaction==DRAG_MOVE)
                {
                    title=_("Move ");
                    icon=move_bigicon;
                    if (num==1)
                        message=title+source;
                    else
                        message=_("Move ")+FXStringVal(num)+_(" files/folders.\nFrom: ")+sourcedir;
                }
                else if ((dropaction==DRAG_LINK) && (num==1))
                {
                    title=_("Symlink ");
                    icon=link_bigicon;
					message=title+source;
                }

                InputDialog* dialog=new InputDialog(this,targetdir,message,title,_("To:"),icon);
                dialog->CursorEnd();
                int rc=1;
                rc=dialog->execute();
                target=dialog->getText();
                target=::filePath(target);
                if (num>1)
                    showdialog=FALSE;
                delete dialog;
                if (!rc)
                    return 0;
            }

            // Move the source file
            if(dropaction==DRAG_MOVE)
            {
                // Move file
                f->create();
                ret=f->move(source,target);

				// An error has occurred
				if (ret==0 && !f->isCancelled)
				{
					f->hide();
					MessageBox::error(this,BOX_OK,_("Error"),_("An error has occurred during the move file operation!"));
				}
					
                // If action is cancelled in progress dialog
                if (f->isCancelled)
                {
                    f->hide();
                    MessageBox::error(this,BOX_OK,_("Error"),_("Move file operation cancelled!"));
                }
            }
            // Copy the source file
            else if(dropaction==DRAG_COPY)
            {
                // Copy file
                f->create();
                ret=f->copy(source,target);

				// An error has occurred
				if (ret==0 && !f->isCancelled)
				{
					f->hide();
					MessageBox::error(this,BOX_OK,_("Error"),_("An error has occurred during the copy file operation!"));
				}
					
                // If action is cancelled in progress dialog
                if (f->isCancelled)
                {
                    f->hide();
                    MessageBox::error(this,BOX_OK,_("Error"),_("Copy file operation cancelled!"));
                }
            }
            // Link the source file (no progress dialog in this case)
            else if(dropaction==DRAG_LINK)
            {
                // Link file
                f->create();
                f->symlink(source,target);
            }
            if(*q=='\r')
                q+=2;
            p=q;
        }
        delete f;
        FXFREE(&data);
        return 1;
    }

    return 0;
}


// Somebody wants our dragged data
long FileList::onDNDRequest(FXObject* sender,FXSelector sel,void* ptr)
{
    FXEvent *event=(FXEvent*)ptr;
    FXuchar *data;
    FXuint len;

    // Perhaps the target wants to supply its own data
    if(FXIconList::onDNDRequest(sender,sel,ptr))
        return 1;

    // Return list of filenames as a uri-list
    if(event->target==urilistType)
    {
        if(!dragfiles.empty())
        {
            len=dragfiles.length();
            FXMEMDUP(&data,dragfiles.text(),FXuchar,len);
            setDNDData(FROM_DRAGNDROP,event->target,data,len);
        }
        return 1;
    }

    // Delete selected files
    if(event->target==deleteType)
        return 1;

    return 0;
}



// Start a drag operation
long FileList::onBeginDrag(FXObject* sender,FXSelector sel,void* ptr)
{
	register FXint i;
	if(FXIconList::onBeginDrag(sender,sel,ptr))
		return 1;
  	if(beginDrag(&urilistType,1))
	{
    	dragfiles=FXString::null;
    	for(i=0; i<getNumItems(); i++)
		{
      		if(isItemSelected(i))
			{
        		if (getItemFilename(i)=="..")
					deselectItem(i);
				else
				{
					if(!dragfiles.empty())
						dragfiles+="\r\n";
        			dragfiles+=FXURL::encode(FXURL::fileToURL(getItemPathname(i)));
				}
        	}
      	}
    	return 1;
	}
	return 0;
}


// End drag operation
long FileList::onEndDrag(FXObject* sender,FXSelector sel,void* ptr)
{
    if(FXIconList::onEndDrag(sender,sel,ptr))
        return 1;
    endDrag((didAccept()!=DRAG_REJECT));
    setDragCursor(getDefaultCursor());
    dragfiles=FXString::null;
    return 1;
}


// Dragged stuff around
long FileList::onDragged(FXObject* sender,FXSelector sel,void* ptr)
{
    FXEvent* event=(FXEvent*)ptr;
    FXDragAction action;
    if(FXIconList::onDragged(sender,sel,ptr))
        return 1;
    action=DRAG_MOVE;

    if(event->state&CONTROLMASK)
        action=DRAG_COPY;
    if(event->state&SHIFTMASK)
        action=DRAG_MOVE;
	if((event->state&CONTROLMASK) && (event->state&SHIFTMASK))
		action=DRAG_LINK;
	
	// If source directory is read-only, convert move action to copy
	if (!::isWritable(orgdirectory) && (action==DRAG_MOVE))
		action=DRAG_COPY;
	
    handleDrag(event->root_x,event->root_y,action);
    if(didAccept()!=DRAG_REJECT)
    {
        if(action==DRAG_MOVE)
            setDragCursor(getApp()->getDefaultCursor(DEF_DNDMOVE_CURSOR));
    	else if(action==DRAG_LINK)
      		setDragCursor(getApp()->getDefaultCursor(DEF_DNDLINK_CURSOR));
        else
            setDragCursor(getApp()->getDefaultCursor(DEF_DNDCOPY_CURSOR));
    }
    else
        setDragCursor(getApp()->getDefaultCursor(DEF_DNDSTOP_CURSOR));
    return 1;
}



// Toggle hidden files display
long FileList::onCmdToggleHidden(FXObject*,FXSelector,void*)
{
    showHiddenFiles(!showHiddenFiles());
    return 1;
}


// Toggle thumbnails display
long FileList::onCmdToggleThumbnails(FXObject*,FXSelector,void*)
{
    showThumbnails(!displaythumbnails);
    return 1;
}


// Update toggle thumbnails button
long FileList::onUpdToggleThumbnails(FXObject* sender,FXSelector,void*)
{
    if(showThumbnails())
        sender->handle(this,FXSEL(SEL_COMMAND,ID_CHECK),NULL);
    else
        sender->handle(this,FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


// Update toggle hidden files button
long FileList::onUpdToggleHidden(FXObject* sender,FXSelector,void*)
{
    if(showHiddenFiles())
        sender->handle(this,FXSEL(SEL_COMMAND,ID_CHECK),NULL);
    else
        sender->handle(this,FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


// Show hidden files
long FileList::onCmdShowHidden(FXObject*,FXSelector,void*)
{
    showHiddenFiles(TRUE);
    return 1;
}


// Update show hidden files widget
long FileList::onUpdShowHidden(FXObject* sender,FXSelector,void*)
{
    if(showHiddenFiles())
        sender->handle(this,FXSEL(SEL_COMMAND,ID_CHECK),NULL);
    else
        sender->handle(this,FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


// Hide hidden files
long FileList::onCmdHideHidden(FXObject*,FXSelector,void*)
{
    showHiddenFiles(FALSE);
    return 1;
}


// Update hide hidden files widget
long FileList::onUpdHideHidden(FXObject* sender,FXSelector,void*)
{
    if(!showHiddenFiles())
        sender->handle(this,FXSEL(SEL_COMMAND,ID_CHECK),NULL);
    else
        sender->handle(this,FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


// Move up one level
long FileList::onCmdDirectoryUp(FXObject*,FXSelector,void*)
{
	setDirectory(FXPath::upLevel(directory));
    return 1;
}


// Determine if we can still go up more
long FileList::onUpdDirectoryUp(FXObject* sender,FXSelector,void* ptr)
{
    FXuint msg=FXPath::isTopDirectory(directory) ? ID_DISABLE : ID_ENABLE;
    sender->handle(this,FXSEL(SEL_COMMAND,msg),ptr);
    return 1;
}


// Change pattern
long FileList::onCmdSetPattern(FXObject*,FXSelector,void* ptr)
{
    if(!ptr)
        return 0;
    setPattern((const char*)ptr);
    return 1;
}


// Update pattern
long FileList::onUpdSetPattern(FXObject* sender,FXSelector,void*)
{
    sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_SETVALUE),(void*)pattern.text());
    return 1;
}


// Sort by name
long FileList::onCmdSortByName(FXObject*,FXSelector,void*)
{
	if (dirsfirst==FALSE)
	{
		if (ignorecase==TRUE)
			sortfunc=(sortfunc==ascendingCaseMix) ? descendingCaseMix : ascendingCaseMix;
		else
			sortfunc=(sortfunc==ascendingMix) ? descendingMix : ascendingMix;
	}
	else
	{
		if (ignorecase==TRUE)
			sortfunc=(sortfunc==ascendingCase) ? descendingCase : ascendingCase;
		else
			sortfunc=(sortfunc==ascending) ? descending : ascending;
	}
    scan(TRUE);
    return 1;
}


// Update sender
long FileList::onUpdSortByName(FXObject* sender,FXSelector,void*)
{
    sender->handle(this,(sortfunc==ascending || sortfunc==descending || sortfunc==ascendingCase || sortfunc==descendingCase
	                     || sortfunc==ascendingMix || sortfunc==descendingMix || sortfunc==ascendingCaseMix || sortfunc==descendingCaseMix) ? FXSEL(SEL_COMMAND,ID_CHECK) : FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


// Sort by type
long FileList::onCmdSortByType(FXObject*,FXSelector,void*)
{
    if (dirsfirst==FALSE)
		sortfunc=(sortfunc==ascendingTypeMix) ? descendingTypeMix : ascendingTypeMix;
	else
		sortfunc=(sortfunc==ascendingType) ? descendingType : ascendingType;
    scan(TRUE);
    return 1;
}


// Update sender
long FileList::onUpdSortByType(FXObject* sender,FXSelector,void*)
{
    sender->handle(this,(sortfunc==ascendingType || sortfunc==descendingType || sortfunc==ascendingTypeMix || sortfunc==descendingTypeMix) ? FXSEL(SEL_COMMAND,ID_CHECK) : FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


// Sort by size
long FileList::onCmdSortBySize(FXObject*,FXSelector,void*)
{
    if (dirsfirst==FALSE)
		sortfunc=(sortfunc==ascendingSizeMix) ? descendingSizeMix : ascendingSizeMix;
	else
		sortfunc=(sortfunc==ascendingSize) ? descendingSize : ascendingSize;
    scan(TRUE);
    return 1;
}


// Update sender
long FileList::onUpdSortBySize(FXObject* sender,FXSelector,void*)
{
    sender->handle(this,(sortfunc==ascendingSize || sortfunc==descendingSize || sortfunc==ascendingSizeMix || sortfunc==descendingSizeMix) ? FXSEL(SEL_COMMAND,ID_CHECK) : FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


// Sort by ext
long FileList::onCmdSortByExt(FXObject*,FXSelector,void*)
{
    if (dirsfirst==FALSE)
		sortfunc=(sortfunc==ascendingExtMix) ? descendingExtMix : ascendingExtMix;
	else		
		sortfunc=(sortfunc==ascendingExt) ? descendingExt : ascendingExt;
    scan(TRUE);
    return 1;
}


// Sort by deletion time
long FileList::onCmdSortByDeltime(FXObject*,FXSelector,void*)
{
    if (dirsfirst==FALSE)
		sortfunc=(sortfunc==ascendingDeltimeMix) ? descendingDeltimeMix : ascendingDeltimeMix;
	else
		sortfunc=(sortfunc==ascendingDeltime) ? descendingDeltime : ascendingDeltime;
    scan(TRUE);
    return 1;
}



// Update sender
long FileList::onUpdSortByExt(FXObject* sender,FXSelector,void*)
{
    sender->handle(this,(sortfunc==ascendingExt || sortfunc==descendingExt || sortfunc==ascendingExtMix || sortfunc==descendingExtMix) ? FXSEL(SEL_COMMAND,ID_CHECK) : FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


// Sort by time
long FileList::onCmdSortByTime(FXObject*,FXSelector,void*)
{
    if (dirsfirst==FALSE)
		sortfunc=(sortfunc==ascendingTimeMix) ? descendingTimeMix : ascendingTimeMix;
	else
		sortfunc=(sortfunc==ascendingTime) ? descendingTime : ascendingTime;
    scan(TRUE);
    return 1;
}


// Update sender
long FileList::onUpdSortByTime(FXObject* sender,FXSelector,void*)
{
    sender->handle(this,(sortfunc==ascendingTime || sortfunc==descendingTime || sortfunc==ascendingTimeMix || sortfunc==descendingTimeMix) ? FXSEL(SEL_COMMAND,ID_CHECK) : FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


// Sort by user
long FileList::onCmdSortByUser(FXObject*,FXSelector,void*)
{
	if (dirsfirst==FALSE)
		sortfunc=(sortfunc==ascendingUserMix) ? descendingUserMix : ascendingUserMix;
	else
		sortfunc=(sortfunc==ascendingUser) ? descendingUser : ascendingUser;
	scan(TRUE);
    return 1;
}


// Update sender
long FileList::onUpdSortByUser(FXObject* sender,FXSelector,void*)
{
    sender->handle(this,(sortfunc==ascendingUser || sortfunc==descendingUser || sortfunc==ascendingUserMix || sortfunc==descendingUserMix) ? FXSEL(SEL_COMMAND,ID_CHECK) : FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


// Sort by group
long FileList::onCmdSortByGroup(FXObject*,FXSelector,void*)
{
	if (dirsfirst==FALSE)
		sortfunc=(sortfunc==ascendingGroupMix) ? descendingGroupMix : ascendingGroupMix;
	else
		sortfunc=(sortfunc==ascendingGroup) ? descendingGroup : ascendingGroup;
    scan(TRUE);
    return 1;
}


// Update sender
long FileList::onUpdSortByGroup(FXObject* sender,FXSelector,void*)
{
    sender->handle(this,(sortfunc==ascendingGroup || sortfunc==descendingGroup || sortfunc==ascendingGroupMix || sortfunc==descendingGroupMix) ? FXSEL(SEL_COMMAND,ID_CHECK) : FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


// Sort by permissions
long FileList::onCmdSortByPerm(FXObject*,FXSelector,void*)
{
	if (dirsfirst==FALSE)
		sortfunc=(sortfunc==ascendingPermMix) ? descendingPermMix : ascendingPermMix;
	else
		sortfunc=(sortfunc==ascendingPerm) ? descendingPerm : ascendingPerm;		
    scan(TRUE);
    return 1;
}


// Update sender
long FileList::onUpdSortByPerm(FXObject* sender,FXSelector,void*)
{
    sender->handle(this,(sortfunc==ascendingPerm || sortfunc==descendingPerm || sortfunc==ascendingPermMix || sortfunc==descendingPermMix) ? FXSEL(SEL_COMMAND,ID_CHECK) : FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


// Update sender
long FileList::onUpdSortByDeltime(FXObject* sender,FXSelector,void*)
{
    sender->handle(this,(sortfunc==ascendingDeltime || sortfunc==descendingDeltime || sortfunc==ascendingDeltimeMix || sortfunc==descendingDeltimeMix) ? FXSEL(SEL_COMMAND,ID_CHECK) : FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}

// Reverse sort order
long FileList::onCmdSortReverse(FXObject*,FXSelector,void*)
{
	if(sortfunc==ascending)
        sortfunc=descending;
    
	else if(sortfunc==ascendingMix)
        sortfunc=descendingMix;
	
	else if(sortfunc==descending)
        sortfunc=ascending;
    
	else if(sortfunc==descendingMix)
        sortfunc=ascendingMix;
    
	else if(sortfunc==ascendingCase)
        sortfunc=descendingCase;
    
	else if(sortfunc==ascendingCaseMix)
        sortfunc=descendingCaseMix;
    
	else if(sortfunc==descendingCase)
        sortfunc=ascendingCase;
    
	else if(sortfunc==descendingCaseMix)
        sortfunc=ascendingCaseMix;
    
	else if(sortfunc==ascendingType)
        sortfunc=descendingType;
    
	else if(sortfunc==ascendingTypeMix)
        sortfunc=descendingTypeMix;
	
	else if(sortfunc==descendingType)
        sortfunc=ascendingType;
	
	else if(sortfunc==descendingTypeMix)
        sortfunc=ascendingTypeMix;
    
	else if(sortfunc==ascendingExt)
        sortfunc=descendingExt;
    
	else if(sortfunc==ascendingExtMix)
        sortfunc=descendingExtMix;
    
	else if(sortfunc==descendingExt)
        sortfunc=ascendingExt;
    
	else if(sortfunc==descendingExtMix)
        sortfunc=ascendingExtMix;
    
	else if(sortfunc==ascendingSize)
        sortfunc=descendingSize;
    
	else if(sortfunc==ascendingSizeMix)
        sortfunc=descendingSizeMix;
    
	else if(sortfunc==descendingSize)
        sortfunc=ascendingSize;
    
	else if(sortfunc==descendingSizeMix)
        sortfunc=ascendingSizeMix;
    
	else if(sortfunc==ascendingTime)
        sortfunc=descendingTime;
    
	else if(sortfunc==ascendingTimeMix)
        sortfunc=descendingTimeMix;
    
	else if(sortfunc==descendingTime)
        sortfunc=ascendingTime;
    
	else if(sortfunc==descendingTimeMix)
        sortfunc=ascendingTimeMix;
    
	else if(sortfunc==ascendingUserMix)
        sortfunc=descendingUser;
    
	else if(sortfunc==ascendingUserMix)
        sortfunc=descendingUser;
    
	else if(sortfunc==descendingUser)
        sortfunc=ascendingUser;
    
	else if(sortfunc==descendingUserMix)
        sortfunc=ascendingUserMix;
    
	else if(sortfunc==ascendingGroup)
        sortfunc=descendingGroup;
    
	else if(sortfunc==ascendingGroupMix)
        sortfunc=descendingGroupMix;
    
	else if(sortfunc==descendingGroup)
        sortfunc=ascendingGroup;
    
	else if(sortfunc==descendingGroupMix)
        sortfunc=ascendingGroupMix;
    
	else if(sortfunc==ascendingPerm)
        sortfunc=descendingPerm;
    
	else if(sortfunc==ascendingPermMix)
        sortfunc=descendingPermMix;
    
	else if(sortfunc==descendingPerm)
        sortfunc=ascendingPerm;
    
	else if(sortfunc==descendingPermMix)
        sortfunc=ascendingPermMix;
    
	else if(sortfunc==ascendingDeltime)
        sortfunc=descendingDeltime;
    
	else if(sortfunc==ascendingDeltimeMix)
        sortfunc=descendingDeltimeMix;
    
	else if(sortfunc==descendingDeltime)
        sortfunc=ascendingDeltime;
    
	else if(sortfunc==descendingDeltimeMix)
        sortfunc=ascendingDeltimeMix;
	
    scan(TRUE);
    return 1;
}


// Update sender
long FileList::onUpdSortReverse(FXObject* sender,FXSelector,void*)
{
    FXSelector selector=FXSEL(SEL_COMMAND,ID_UNCHECK);
    
	if(sortfunc==descending)
        selector=FXSEL(SEL_COMMAND,ID_CHECK);
    
	else if(sortfunc==descendingMix)
        selector=FXSEL(SEL_COMMAND,ID_CHECK);
    
	else if(sortfunc==descendingCase)
        selector=FXSEL(SEL_COMMAND,ID_CHECK);
    
	else if(sortfunc==descendingCaseMix)
        selector=FXSEL(SEL_COMMAND,ID_CHECK);
    
	else if(sortfunc==descendingType)
        selector=FXSEL(SEL_COMMAND,ID_CHECK);
    
	else if(sortfunc==descendingTypeMix)
        selector=FXSEL(SEL_COMMAND,ID_CHECK);
    
	else if(sortfunc==descendingExt)
        selector=FXSEL(SEL_COMMAND,ID_CHECK);
    
	else if(sortfunc==descendingExtMix)
        selector=FXSEL(SEL_COMMAND,ID_CHECK);
    
	else if(sortfunc==descendingSize)
        selector=FXSEL(SEL_COMMAND,ID_CHECK);
    
	else if(sortfunc==descendingSizeMix)
        selector=FXSEL(SEL_COMMAND,ID_CHECK);
    
	else if(sortfunc==descendingTime)
        selector=FXSEL(SEL_COMMAND,ID_CHECK);
    
	else if(sortfunc==descendingTimeMix)
        selector=FXSEL(SEL_COMMAND,ID_CHECK);
    
	else if(sortfunc==descendingUser)
        selector=FXSEL(SEL_COMMAND,ID_CHECK);
    
	else if(sortfunc==descendingUserMix)
        selector=FXSEL(SEL_COMMAND,ID_CHECK);
    
	else if(sortfunc==descendingGroup)
        selector=FXSEL(SEL_COMMAND,ID_CHECK);
	
	else if(sortfunc==descendingGroupMix)
        selector=FXSEL(SEL_COMMAND,ID_CHECK);
    
	else if(sortfunc==descendingPerm)
        selector=FXSEL(SEL_COMMAND,ID_CHECK);
    
	else if(sortfunc==descendingPermMix)
        selector=FXSEL(SEL_COMMAND,ID_CHECK);
    
	else if(sortfunc==descendingDeltime)
        selector=FXSEL(SEL_COMMAND,ID_CHECK);
	
	else if(sortfunc==descendingDeltimeMix)
        selector=FXSEL(SEL_COMMAND,ID_CHECK);

	sender->handle(this,selector,NULL);
    
	return 1;
}


// Toggle case sensitivity
long FileList::onCmdSortCase(FXObject*,FXSelector,void*)
{
	ignorecase=!ignorecase;
	if (dirsfirst==FALSE)
	{
		if (ignorecase==TRUE)
		{
			if(sortfunc==ascending || sortfunc==ascendingMix || sortfunc==ascendingCase)
				sortfunc=ascendingCaseMix;
			
			else if(sortfunc==descending || sortfunc==descendingMix || sortfunc==descendingCase)
				sortfunc=descendingCaseMix;
		}
		else
		{
			if(sortfunc==ascending || sortfunc==ascendingCase || sortfunc==ascendingCaseMix)
				sortfunc=ascendingMix;
			
			else if(sortfunc==descending || sortfunc==descendingCase || sortfunc==descendingCaseMix)
				sortfunc=descendingMix;
		}
	}
	else
	{
		if (ignorecase==TRUE)
		{
			if(sortfunc==ascending || sortfunc==ascendingMix || sortfunc==ascendingCaseMix)
				sortfunc=ascendingCase;
			
			else if(sortfunc==descending || sortfunc==descendingMix || sortfunc==descendingCaseMix)
				sortfunc=descendingCase;
		}
		else
		{
			if(sortfunc==ascendingMix || sortfunc==ascendingCase || sortfunc==ascendingCaseMix)
				sortfunc=ascending;
			
			else if(sortfunc==descendingMix || sortfunc==descendingCase || sortfunc==descendingCaseMix)
				sortfunc=descending;
		}
	}
    scan(TRUE);
    return 1;
}


// Update case sensitivity
long FileList::onUpdSortCase(FXObject* sender,FXSelector,void* ptr)
{

    if (sortfunc==ascending || sortfunc==descending || sortfunc==ascendingMix || sortfunc==descendingMix
		|| sortfunc==ascendingCase || sortfunc==descendingCase || sortfunc==ascendingCaseMix || sortfunc==descendingCaseMix)
	{
        sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),ptr);
		
		if (ignorecase==TRUE)
			sender->handle(this,FXSEL(SEL_COMMAND,ID_CHECK),ptr);
		else
			sender->handle(this,FXSEL(SEL_COMMAND,ID_UNCHECK),ptr);
	}
	else
        sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),ptr);

    return 1;
}


// Toggle directories first
long FileList::onCmdDirsFirst(FXObject*,FXSelector,void*)
{
	dirsfirst=!dirsfirst;
	if (dirsfirst==FALSE)
	{
		if (sortfunc==ascending)
			sortfunc=ascendingMix;
		
		else if (sortfunc==descending)
			sortfunc=descendingMix;
		
		else if (sortfunc==ascendingCase)
			sortfunc=ascendingCaseMix;
		
		else if (sortfunc==descendingCase)
			sortfunc=descendingCaseMix;
		
		else if (sortfunc==ascendingType)
			sortfunc=ascendingTypeMix;
		
		else if (sortfunc==descendingType)
			sortfunc=descendingTypeMix;
		
		else if (sortfunc==ascendingExt)
			sortfunc=ascendingExtMix;
		
		else if (sortfunc==descendingExt)
			sortfunc=descendingExtMix;
		
		else if (sortfunc==ascendingSize)
			sortfunc=ascendingSizeMix;
		
		else if (sortfunc==descendingSize)
			sortfunc=descendingSizeMix;
		
		else if (sortfunc==ascendingTime)
			sortfunc=ascendingTimeMix;
		
		else if (sortfunc==descendingTime)
			sortfunc=descendingTimeMix;
		
		else if (sortfunc==ascendingUser)
			sortfunc=ascendingUserMix;
		
		else if (sortfunc==descendingUser)
			sortfunc=descendingUserMix;
		
		else if (sortfunc==ascendingGroup)
			sortfunc=ascendingGroupMix;
		
		else if (sortfunc==descendingGroup)
			sortfunc=descendingGroupMix;
		
		else if (sortfunc==ascendingPerm)
			sortfunc=ascendingPermMix;
		
		else if (sortfunc==descendingPerm)
			sortfunc=descendingPermMix;
		
		else if (sortfunc==ascendingDeltime)
			sortfunc=ascendingDeltimeMix;
		
		else if (sortfunc==descendingDeltime)
			sortfunc=descendingDeltimeMix;
	}
	else
	{
		if (sortfunc==ascendingMix)
			sortfunc=ascending;
		
		else if (sortfunc==descendingMix)
			sortfunc=descending;
		
		else if (sortfunc==ascendingCaseMix)
			sortfunc=ascendingCase;
		
		else if (sortfunc==descendingCaseMix)
			sortfunc=descendingCase;
		
		else if (sortfunc==ascendingTypeMix)
			sortfunc=ascendingType;
		
		else if (sortfunc==descendingTypeMix)
			sortfunc=descendingType;
		
		else if (sortfunc==ascendingExtMix)
			sortfunc=ascendingExt;
		
		else if (sortfunc==descendingExtMix)
			sortfunc=descendingExt;
		
		else if (sortfunc==ascendingSizeMix)
			sortfunc=ascendingSize;
		
		else if (sortfunc==descendingSizeMix)
			sortfunc=descendingSize;
		
		else if (sortfunc==ascendingTimeMix)
			sortfunc=ascendingTime;
		
		else if (sortfunc==descendingTimeMix)
			sortfunc=descendingTime;
		
		else if (sortfunc==ascendingUserMix)
			sortfunc=ascendingUser;
		
		else if (sortfunc==descendingUserMix)
			sortfunc=descendingUser;
		
		else if (sortfunc==ascendingGroupMix)
			sortfunc=ascendingGroup;
		
		else if (sortfunc==descendingGroupMix)
			sortfunc=descendingGroup;
		
		else if (sortfunc==ascendingPermMix)
			sortfunc=ascendingPerm;
		
		else if (sortfunc==descendingPermMix)
			sortfunc=descendingPerm;
		
		else if (sortfunc==ascendingDeltimeMix)
			sortfunc=ascendingDeltime;
		
		else if (sortfunc==descendingDeltimeMix)
			sortfunc=descendingDeltime;
	}
    scan(TRUE);
    return 1;
}


// Update directories first
long FileList::onUpdDirsFirst(FXObject* sender,FXSelector,void* ptr)
{	
	if (dirsfirst==TRUE)
        sender->handle(this,FXSEL(SEL_COMMAND,ID_CHECK),ptr);
    else
        sender->handle(this,FXSEL(SEL_COMMAND,ID_UNCHECK),ptr);
 
    return 1;
}

// Clicked header button
long FileList::onCmdHeader(FXObject*,FXSelector,void* ptr)
{
    if (getNumHeaders()==9)
	{
		if(((FXuint)(FXuval)ptr)<9)			
			handle(this,FXSEL(SEL_COMMAND,(ID_SORT_BY_NAME+(FXuint)(FXuval)ptr)),NULL);
	}
	else
	{
		if(((FXuint)(FXuval)ptr)<8)
			handle(this,FXSEL(SEL_COMMAND,(ID_SORT_BY_NAME+(FXuint)(FXuval)ptr)),NULL);
	}
    if (getHeaderSize(0)<MIN_NAME_SIZE)
        setHeaderSize(0,MIN_NAME_SIZE);
    return 1;
}


// Clicked header button
long FileList::onUpdHeader(FXObject*,FXSelector,void*)
{	
	header->setArrowDir(0,(sortfunc==ascending      || sortfunc==ascendingCase
	                    || sortfunc==ascendingMix   || sortfunc==ascendingCaseMix)  ? FALSE : (sortfunc==descending      || sortfunc==descendingCase
                    	|| sortfunc==descendingMix  || sortfunc==descendingCaseMix) ? TRUE : MAYBE);                                                                        // Name
    header->setArrowDir(1,(sortfunc==ascendingSize  || sortfunc==ascendingSizeMix)  ? FALSE : (sortfunc==descendingSize  || sortfunc==descendingSizeMix) ? TRUE : MAYBE);   // Size
    header->setArrowDir(2,(sortfunc==ascendingType  || sortfunc==ascendingTypeMix)  ? FALSE : (sortfunc==descendingType  || sortfunc==descendingTypeMix) ? TRUE : MAYBE);   // Type
    header->setArrowDir(3,(sortfunc==ascendingExt   || sortfunc==ascendingExtMix)   ? FALSE : (sortfunc==descendingExt   || sortfunc==descendingExtMix)  ? TRUE : MAYBE);   // Extension
    header->setArrowDir(4,(sortfunc==ascendingTime  || sortfunc==ascendingTimeMix)  ? FALSE : (sortfunc==descendingTime  || sortfunc==descendingTimeMix) ? TRUE : MAYBE);   // Date
    header->setArrowDir(5,(sortfunc==ascendingUser  || sortfunc==ascendingUserMix)  ? FALSE : (sortfunc==descendingUser  || sortfunc==descendingUserMix) ? TRUE : MAYBE);   // User
    header->setArrowDir(6,(sortfunc==ascendingGroup || sortfunc==ascendingGroupMix) ? FALSE : (sortfunc==descendingGroup || sortfunc==descendingGroupMix)? TRUE : MAYBE);   // Group
    header->setArrowDir(7,(sortfunc==ascendingPerm  || sortfunc==ascendingPermMix)  ? FALSE : (sortfunc==descendingPerm  || sortfunc==descendingPermMix) ? TRUE : MAYBE);   // Permissions
    if (getNumHeaders()==9)
		header->setArrowDir(8,(sortfunc==ascendingDeltime || sortfunc==ascendingDeltimeMix) ? FALSE : (sortfunc==descendingDeltime || sortfunc==descendingDeltimeMix) ? TRUE : MAYBE);   // Deletion date
	if (getHeaderSize(0)<MIN_NAME_SIZE)
        setHeaderSize(0,MIN_NAME_SIZE);
    return 1;
}


// Compare file names
FXint FileList::ascending(const FXIconItem* pa,const FXIconItem* pb)
{

    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

	register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return diff;
	
	while(1)
    {
        if(*p > *q)
            return 1;
        if(*p < *q)
            return -1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return 0;
}


// Compare file names, mixing files and directories
FXint FileList::ascendingMix(const FXIconItem* pa,const FXIconItem* pb)
{

    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;
	
	while(1)
    {
        if(*p > *q)
            return 1;
        if(*p < *q)
            return -1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return 0;
}

// Compare file names, case insensitive
FXint FileList::ascendingCase(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

	register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return diff;
    while(1)
    {
        if(tolower(*p) > tolower(*q))
            return 1;
        if(tolower(*p) < tolower(*q))
            return -1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return 0;
}


// Compare file names, case insensitive, mixing files and directories
FXint FileList::ascendingCaseMix(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    while(1)
    {
        if(tolower(*p) > tolower(*q))
            return 1;
        if(tolower(*p) < tolower(*q))
            return -1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return 0;
}


// Compare file types
FXint FileList::ascendingType(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;
    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

	register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return diff;
	
    register int i;
    for(i=2; *p && i; i-=(*p++=='\t'))
        ;
    for(i=2; *q && i; i-=(*q++=='\t'))
        ;
    while(1)
    {
        if(*p > *q)
            return 1;
        if(*p < *q)
            return -1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return ascendingCase(pa,pb);
}


// Compare file types, mixing files and directories
FXint FileList::ascendingTypeMix(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;
    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register int i;
    for(i=2; *p && i; i-=(*p++=='\t'))
        ;
    for(i=2; *q && i; i-=(*q++=='\t'))
        ;
    while(1)
    {
        if(*p > *q)
            return 1;
        if(*p < *q)
            return -1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return ascendingCaseMix(pa,pb);
}


// Compare file extension
FXint FileList::ascendingExt(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;
    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

	register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return diff;

     register int i;
   for(i=3; *p && i; i-=(*p++=='\t'))
        ;
    for(i=3; *q && i; i-=(*q++=='\t'))
        ;

    while(1)
    {
        if(*p > *q)
            return 1;
        if(*p < *q)
            return -1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return ascendingCase(pa,pb);
}


// Compare file extension, mixing files and directories
FXint FileList::ascendingExtMix(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;
    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register int i;
    for(i=3; *p && i; i-=(*p++=='\t'))
        ;
    for(i=3; *q && i; i-=(*q++=='\t'))
        ;

    while(1)
    {
        if(*p > *q)
            return 1;
        if(*p < *q)
            return -1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return ascendingCaseMix(pa,pb);
}


// Compare file size - Warning : only returns the sign of the comparison!!!
FXint FileList::ascendingSize(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

	register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return diff;
    register long long l=a->size - b->size;
    if(l)
    {
        if (l>=0)
            return 1;
        else
            return -1;
    }
    return ascendingCase(pa,pb);
}


// Compare file size - Warning : only returns the sign of the comparison!!!
// Mixing files and directories
FXint FileList::ascendingSizeMix(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register long long l=a->size - b->size;
    if(l)
    {
        if (l>=0)
            return 1;
        else
            return -1;
    }
    return ascendingCaseMix(pa,pb);
}


// Compare file time
FXint FileList::ascendingTime(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;
		
	register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return diff;

    register long l=(long)a->date - (long)b->date;
    if(l)
        return l;
    return ascendingCase(pa, pb);
}


// Compare file time, mixing files and directories
FXint FileList::ascendingTimeMix(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;
	
    register long l=(long)a->date - (long)b->date;
    if(l)
        return l;
    return ascendingCaseMix(pa, pb);
}


// Compare file user
FXint FileList::ascendingUser(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register int i;
	register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return diff;
    for(i=5; *p && i; i-=(*p++=='\t'))
        ;
    for(i=5; *q && i; i-=(*q++=='\t'))
        ;
    while(1)
    {
        if(*p > *q)
            return 1;
        if(*p < *q)
            return -1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return ascendingCase(pa,pb);
}



// Compare file user, mixing files and directories
FXint FileList::ascendingUserMix(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register int i;
    for(i=5; *p && i; i-=(*p++=='\t'))
        ;
    for(i=5; *q && i; i-=(*q++=='\t'))
        ;
    while(1)
    {
        if(*p > *q)
            return 1;
        if(*p < *q)
            return -1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return ascendingCaseMix(pa,pb);
}


// Compare file group
FXint FileList::ascendingGroup(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register int i;
    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
	if(diff)
        return diff;
    for(i=6; *p && i; i-=(*p++=='\t'))
        ;
    for(i=6; *q && i; i-=(*q++=='\t'))
        ;
    while(1)
    {
        if(*p > *q)
            return 1;
        if(*p < *q)
            return -1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return ascendingCase(pa,pb);
}


// Compare file group, mixing files and directories
FXint FileList::ascendingGroupMix(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register int i;
    for(i=6; *p && i; i-=(*p++=='\t'))
        ;
    for(i=6; *q && i; i-=(*q++=='\t'))
        ;
    while(1)
    {
        if(*p > *q)
            return 1;
        if(*p < *q)
            return -1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return ascendingCaseMix(pa,pb);
}


// Compare file permissions
FXint FileList::ascendingPerm(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;
    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

	register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return diff;

     register int i;
   for(i=7; *p && i; i-=(*p++=='\t'))
        ;
    for(i=7; *q && i; i-=(*q++=='\t'))
        ;

    while(1)
    {
        if(*p > *q)
            return 1;
        if(*p < *q)
            return -1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return ascendingCase(pa,pb);
}


// Compare file permissions, mixing files and directories
FXint FileList::ascendingPermMix(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;
    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

     register int i;
    for(i=7; *p && i; i-=(*p++=='\t'))
        ;
    for(i=7; *q && i; i-=(*q++=='\t'))
        ;

    while(1)
    {
        if(*p > *q)
            return 1;
        if(*p < *q)
            return -1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return ascendingCaseMix(pa,pb);
}


// Compare file deletion time
FXint FileList::ascendingDeltime(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

	register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return diff;
    register long l=(long)a->deldate - (long)b->deldate;
    if(l)
        return l;
    return ascendingCase(pa, pb);
}


// Compare file deletion time, mixing files and directories
FXint FileList::ascendingDeltimeMix(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register long l=(long)a->deldate - (long)b->deldate;
    if(l)
        return l;
    return ascendingCaseMix(pa, pb);
}


// Reversed compare file name, case insensitive
FXint FileList::descendingCase(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

	register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return -diff;
    while(1)
    {
        if(tolower(*p) > tolower(*q))
            return -1;
        if(tolower(*p) < tolower(*q))
            return 1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return 0;
}


// Reversed compare file name, case insensitive, mixing files and directories
FXint FileList::descendingCaseMix(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    while(1)
    {
        if(tolower(*p) > tolower(*q))
            return -1;
        if(tolower(*p) < tolower(*q))
            return 1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return 0;
}


// Reversed compare file name
FXint FileList::descending(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return -diff;
    while(1)
    {
        if(*p > *q)
            return -1;
        if(*p < *q)
            return 1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return 0;
}


// Reversed compare file name, mixing files and directories
FXint FileList::descendingMix(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    while(1)
    {
        if(*p > *q)
            return -1;
        if(*p < *q)
            return 1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return 0;
}


// Reversed compare file type
FXint FileList::descendingType(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return -diff;

    register int i;
    for(i=2; *p && i; i-=(*p++=='\t'))
        ;
    for(i=2; *q && i; i-=(*q++=='\t'))
        ;
    while(1)
    {
        if(*p > *q)
            return -1;
        if(*p < *q)
            return 1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return -ascendingCase(pa,pb);
}


// Reversed compare file type, mixing files and directories
FXint FileList::descendingTypeMix(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register int i;
    for(i=2; *p && i; i-=(*p++=='\t'))
        ;
    for(i=2; *q && i; i-=(*q++=='\t'))
        ;
    while(1)
    {
        if(*p > *q)
            return -1;
        if(*p < *q)
            return 1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return -ascendingCaseMix(pa,pb);
}


// Reversed compare file extension
FXint FileList::descendingExt(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return -diff;

    register int i;
    for(i=3; *p && i; i-=(*p++=='\t'))
        ;
    for(i=3; *q && i; i-=(*q++=='\t'))
        ;
    while(1)
    {
        if(*p > *q)
            return -1;
        if(*p < *q)
            return 1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return -ascendingCase(pa,pb);
}


// Reversed compare file extension, mixing files and directories
FXint FileList::descendingExtMix(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register int i;
    for(i=3; *p && i; i-=(*p++=='\t'))
        ;
    for(i=3; *q && i; i-=(*q++=='\t'))
        ;
    while(1)
    {
        if(*p > *q)
            return -1;
        if(*p < *q)
            return 1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return -ascendingCaseMix(pa,pb);
}


// Reversed compare file size
FXint FileList::descendingSize(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return -diff;
    register long long l=a->size - b->size;
    if(l)
    {
        if (l>=0)
            return -1;
        else
            return 1;
    }
    return -ascendingCase(pa,pb);
}


// Reversed compare file size, mixing files and directories
FXint FileList::descendingSizeMix(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register long long l=a->size - b->size;
    if(l)
    {
        if (l>=0)
            return -1;
        else
            return 1;
    }
    return -ascendingCaseMix(pa,pb);
}


// Reversed compare file time
FXint FileList::descendingTime(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return -diff;
    register long l=(long)a->date - (long)b->date;
    if(l)
        return -l;
    return -ascendingCase(pa, pb);
}


// Reversed compare file time, mixing files and directories
FXint FileList::descendingTimeMix(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register long l=(long)a->date - (long)b->date;
    if(l)
        return -l;
    return -ascendingCaseMix(pa, pb);
}


// Reversed compare file user
FXint FileList::descendingUser(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register int i;
    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return -diff;
    for(i=5; *p && i; i-=(*p++=='\t'))
        ;
    for(i=5; *q && i; i-=(*q++=='\t'))
        ;
    while(1)
    {
        if(*p > *q)
            return -1;
        if(*p < *q)
            return 1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return -ascendingCase(pa,pb);
}


// Reversed compare file user, mixing files and directories
FXint FileList::descendingUserMix(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register int i;
    for(i=5; *p && i; i-=(*p++=='\t'))
        ;
    for(i=5; *q && i; i-=(*q++=='\t'))
        ;
    while(1)
    {
        if(*p > *q)
            return -1;
        if(*p < *q)
            return 1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return -ascendingCaseMix(pa,pb);
}


// Reversed compare file group
FXint FileList::descendingGroup(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register int i;
    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return -diff;
    for(i=6; *p && i; i-=(*p++=='\t'))
        ;
    for(i=6; *q && i; i-=(*q++=='\t'))
        ;
    while(1)
    {
        if(*p > *q)
            return -1;
        if(*p < *q)
            return 1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return -ascendingCase(pa,pb);
}


// Reversed compare file group, mixing files and directories
FXint FileList::descendingGroupMix(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register int i;
    for(i=6; *p && i; i-=(*p++=='\t'))
        ;
    for(i=6; *q && i; i-=(*q++=='\t'))
        ;
    while(1)
    {
        if(*p > *q)
            return -1;
        if(*p < *q)
            return 1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return -ascendingCaseMix(pa,pb);
}


// Reversed compare file permission
FXint FileList::descendingPerm(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return -diff;

    register int i;
    for(i=7; *p && i; i-=(*p++=='\t'))
        ;
    for(i=7; *q && i; i-=(*q++=='\t'))
        ;
    while(1)
    {
        if(*p > *q)
            return -1;
        if(*p < *q)
            return 1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return -ascendingCase(pa,pb);
}



// Reversed compare file permission, mixing files and directories
FXint FileList::descendingPermMix(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register int i;
    for(i=7; *p && i; i-=(*p++=='\t'))
        ;
    for(i=7; *q && i; i-=(*q++=='\t'))
        ;
    while(1)
    {
        if(*p > *q)
            return -1;
        if(*p < *q)
            return 1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return -ascendingCaseMix(pa,pb);
}


// Reversed compare file deletion time
FXint FileList::descendingDeltime(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return -diff;
    register long l=(long)a->deldate - (long)b->deldate;
    if(l)
        return -l;
    return -ascendingCase(pa, pb);
}


// Reversed compare file deletion time, mixing files and directories
FXint FileList::descendingDeltimeMix(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register long l=(long)a->deldate - (long)b->deldate;
    if(l)
        return -l;
    return -ascendingCaseMix(pa, pb);
}


// Scan items to see if listing is necessary
void FileList::scan(FXbool force)
{
    struct stat info;

    // Stat the current directory
    if(::info(directory,info))
    {
        // New date of directory
        FXTime newdate=(FXTime)FXMAX(info.st_mtime,info.st_ctime);

        // Forced, date was changed, or failed to get proper date or counter expired
        if(force || (timestamp!=newdate) || (counter==0))
        {
            // And do the refresh
            listItems();
            sortItems();

            // Remember when we did this
            timestamp=newdate;
        }
    }

    // Move to higher directory
    else
        setDirectory(FXPath::upLevel(directory));
}


// Force an immediate update of the list
long FileList::onCmdRefresh(FXObject*,FXSelector,void*)
{
    allowrefresh=TRUE;
	scan(TRUE);
    return 1;
}

// Allow or forbid file list refresh
void FileList::setAllowRefresh(const FXbool allow)
{
	if (allow==FALSE)
		allowrefresh=FALSE;
	else
		allowrefresh=TRUE;
}


// Refresh; don't update if user is interacting with the list
long FileList::onRefreshTimer(FXObject*,FXSelector,void*)
{
	if(flags&FLAG_UPDATE && allowrefresh)
    {
        scan(FALSE);
        counter=(counter+1)%REFRESH_FREQUENCY;
    }
	
    // Reset timer again
    getApp()->addTimeout(this,ID_REFRESH_TIMER,REFRESH_INTERVAL);
    return 0;
}


// Set current filename
void FileList::setCurrentFile(const FXString& pathname)
{    // FIXME notify argument?
    if(!pathname.empty())
    {
        setDirectory(FXPath::directory(pathname));
        setCurrentItem(findItem(FXPath::name(pathname)));
    }
}


// Get pathname to current file, if any
FXString FileList::getCurrentFile() const
{
    if(current<0)
        return FXString::null;
    return getItemPathname(current);
}


// Set directory being displayed
// and update the history if necessary
void FileList::setDirectory(const FXString& pathname, FXbool histupdate)
{
	// Only update the history if it was requested
	if (histupdate)
	{		
		// At first call, allocate the history
		if (forwardhist==NULL)
			forwardhist=new StringList();
		if (backhist==NULL)
			backhist=new StringList();
		
		// Update the history
		else
		{
			backhist->insertFirstItem(getDirectory());
			forwardhist->removeAllItems();
		}
	}

	// FIXME notify argument?
    if(!pathname.empty())
    {
        FXString path=FXPath::absolute(directory,pathname);
        while(!FXPath::isTopDirectory(path) && !::isDirectory(path))
            path=FXPath::upLevel(path);
        if(directory!=path)
        {
            directory=path;
            clearItems();
            list=NULL;
            scan(TRUE);
        }
    }	
}


// Set the pattern to filter
void FileList::setPattern(const FXString& ptrn)
{
    if(ptrn.empty())
        return;
    if(pattern!=ptrn)
    {
        pattern=ptrn;
		scan(TRUE);
    }
}


// Change file match mode
void FileList::setMatchMode(FXuint mode)
{
    if(matchmode!=mode)
    {
        matchmode=mode;
		scan(TRUE);
    }
}


// Return TRUE if showing hidden files
FXbool FileList::showHiddenFiles() const
{
    return (options&FILELIST_SHOWHIDDEN)!=0;
}


// Change show hidden files mode
void FileList::showHiddenFiles(FXbool shown)
{
    FXuint opts=shown ? (options|FILELIST_SHOWHIDDEN) : (options&~FILELIST_SHOWHIDDEN);
    if(opts!=options)
    {
        options=opts;
		scan(TRUE);
    }
}


// Return show thumbnails
FXbool FileList::showThumbnails() const
{
	return displaythumbnails;
}

// Change show thumbnails mode
void FileList::showThumbnails(FXbool display)
{
	displaythumbnails=display;
		
	// Refresh to display or hide thumbnails 
	scan(TRUE);
}


// Return TRUE if showing directories only
FXbool FileList::showOnlyDirectories() const
{
    return (options&FILELIST_SHOWDIRS)!=0;
}


// Change show directories only mode
void FileList::showOnlyDirectories(FXbool shown)
{
    FXuint opts=shown ? (options|FILELIST_SHOWDIRS) : (options&~FILELIST_SHOWDIRS);
    if(opts!=options)
    {
        options=opts;
		scan(TRUE);
    }
}


// Compare till '\t' or '\0'
static FXbool fileequal(const FXString& a,const FXString& b)
{
  register const FXuchar *p1=(const FXuchar *)a.text();
  register const FXuchar *p2=(const FXuchar *)b.text();
  register FXint c1,c2;
  do
  {
    c1=*p1++;
    c2=*p2++;
	}
  while(c1!='\0' && c1!='\t' && c1==c2);
  return (c1=='\0' || c1=='\t') && (c2=='\0' || c2=='\t');
}


// Create custom item
FXIconItem *FileList::createItem(const FXString& text,FXIcon *big,FXIcon* mini,void* ptr)
{
    return new FileItem(text,big,mini,ptr);
}


// List directory (from Fox library)
void FileList::listItems()
{
    FileItem *oldlist, *newlist;
    FileItem **po, **pn, **pp;
    FXString grpid, usrid, atts, mod, ext, del;
    FXString name, dirname, pathname;
    FileItem *curitem=NULL;
    FileItem *item, *link;
    FileAssoc *fileassoc;
    FXString filetype, lowext;
    FXIcon *big, *mini;
	FXbool image;
    time_t filetime;
    struct stat info, linfo;
    struct dirent *dp;
    DIR *dirp;
	FXbool isInTrash, isLink, isBrokenLink;
	long deldate;

	// Start wait cursor
	getApp()->beginWaitCursor();

  	// Build old and new insert-order lists
  	oldlist=list;
  	newlist=NULL;

  	// Head of old and new lists
  	po=&oldlist;
  	pn=&newlist;
  
    // Remember current item
    if(0<=current)
        curitem=(FileItem*)items[current];

    // Start inserting
    items.clear();

    // Get info about directory
    if(statrep(directory.text(),&info)==0)
    {
        // Need latest change no matter what actually changed!
        timestamp=FXMAX(info.st_mtime,info.st_ctime);

        // Set path to stat with
		dirname=directory.text();

    	if(dirname!=ROOTDIR) 
			dirname+=PATHSEPSTRING;

		// Add the deletion header if we are in trash can
		if (dirname==trashlocation)
		{
			if (getNumHeaders()==8)
				appendHeader(_("Deletion date"),NULL,150);
			isInTrash=TRUE;
		}
		else
		{
			if (getNumHeaders()==9)
				removeHeader(8);
			isInTrash=FALSE;
		}

        // Get directory stream pointer
        dirp=opendir(directory.text());

        // Managed to open directory
        if(dirp)
        {
       		// Loop over directory entries
#ifdef FOX_THREAD_SAFE
        	struct fxdirent dirresult;
        	while(!readdir_r(dirp,&dirresult,&dp) && dp)
        	{
#else
        	while((dp=readdir(dirp))!=NULL)
			{
#endif
				// Directory name
				name=dp->d_name;

                // Hidden file (.xxx) or directory (. or .yyy) normally not shown,
                // but directory .. is always shown so we can navigate up or down
                if(name[0]=='.' && (name[1]==0 || (!(name[1]=='.'&& name[2]==0) && !(options&FILELIST_SHOWHIDDEN))))
                    continue;

                // Build full pathname
				pathname=dirname+name;

				// Get file/link info and indicate if it is a link
                if(lstatrep(pathname.text(),&linfo)!=0)
                    continue;
                isLink=S_ISLNK(linfo.st_mode);
				isBrokenLink=FALSE;

 				// Find if it is a broken link
                if(isLink && (statrep(pathname.text(),&info)!=0))
					isBrokenLink=TRUE;

                // If not a directory and we want only directories, skip it
                if(!S_ISDIR(linfo.st_mode) && (options&FILELIST_SHOWDIRS))
                    continue;

                // Is it a directory or does it match the pattern?
                if(!S_ISDIR(linfo.st_mode) && !FXPath::match(pattern,name,matchmode))
                    continue;

                // Anything about file has changed
                filetime=linfo.st_mtime;

            	// Find it, and take it out from the old list if found
            	for(pp=po; (item=*pp)!=NULL; pp=&item->link)
            	{
                	if(fileequal(item->label,name))
                	{
                    	*pp=item->link;
                    	item->link=NULL;
                    	po=pp;
                    	goto fnd;
                	}
            	}

                // Make new item if we have to
                item=(FileItem*)createItem(FXString::null,NULL,NULL,NULL);

                // Append item in list
fnd:          	*pn=item;
            	pn=&item->link;

            	// Append
      			if(item==curitem)
					current=items.no();
      			items.append(item);

				// Obtain user name
                usrid=FXSystem::userName(linfo.st_uid);

                // Obtain group name
                grpid=FXSystem::groupName(linfo.st_gid);

                // Permissions (caution : we don't use the FXSystem::modeString() function because
				// it seems to be incompatible with the info.st_mode format)
                atts=::permissions(linfo.st_mode);
								
				// Mod time
                mod=FXSystem::time("%x %X",filetime);
				
				// If we are in trash can, obtain the deletion time
				deldate=0;
				if (isInTrash)
				{
					char *endptr;
					FXString str;
					str=pathname.rafter('_');
					str=str.rbefore('-');
					deldate=strtol(str.text(),&endptr,10);
					if (deldate!=0)
						del=FXSystem::time("%x %X",deldate);
					else
						del="";

					// Obtain the extension for files only
					if (!S_ISDIR(linfo.st_mode))
						ext=FXPath::extension(pathname.rbefore('_',4));
					else
						ext="";
				}
				else
				{
					// Obtain the extension for files only
					if (!S_ISDIR(linfo.st_mode))
						ext=FXPath::extension(pathname);
					else
						ext="";
				}

				// Obtain the stat info on the file itself
				if (statrep(pathname.text(),&info)!=0)
				{						
					// Except in the case of a broken link
					if (isBrokenLink)
						lstatrep(pathname.text(),&info);
					else						
						continue;
				}
						
				// Set item flags from the obtained info
                if(S_ISDIR(info.st_mode))
                    item->state|=FileItem::FOLDER;
                else
                    item->state&=~FileItem::FOLDER;
                if(S_ISLNK(info.st_mode))
                    item->state|=FileItem::SYMLINK;
                else
                    item->state&=~FileItem::SYMLINK;
                if(S_ISCHR(info.st_mode))
                    item->state|=FileItem::CHARDEV;
                else
                    item->state&=~FileItem::CHARDEV;
                if(S_ISBLK(info.st_mode))
                    item->state|=FileItem::BLOCKDEV;
                else
                    item->state&=~FileItem::BLOCKDEV;
                if(S_ISFIFO(info.st_mode))
                    item->state|=FileItem::FIFO;
                else
                    item->state&=~FileItem::FIFO;
                if(S_ISSOCK(info.st_mode))
                    item->state|=FileItem::SOCK;
                else
                    item->state&=~FileItem::SOCK;
                if((info.st_mode&(S_IXUSR|S_IXGRP|S_IXOTH)) && !(S_ISDIR(info.st_mode)||S_ISCHR(info.st_mode)||S_ISBLK(info.st_mode)||S_ISFIFO(info.st_mode)||S_ISSOCK(info.st_mode)))
                    item->state|=FileItem::EXECUTABLE;
                else
                    item->state&=~FileItem::EXECUTABLE;

                // We can drag items
                item->state|=FileItem::DRAGGABLE;

                // Assume no associations
                fileassoc=NULL;

                // Determine icons and type
				if(item->state&FileItem::FOLDER)
                {
				    if(!::isReadExecutable(pathname))
                    {
                        big=bigfolderlockedicon;
                        mini=minifolderlockedicon;
                        filetype=_("Folder");
                    }
                    else
                    {
                        big=bigfoldericon;
                        mini=minifoldericon;
                        filetype=_("Folder");
                    }
                    if(associations)
                        fileassoc=associations->findDirBinding(pathname.text());
                }
                else if(item->state&FileItem::CHARDEV)
    			{
    				big=bigchardevicon;
    				mini=minichardevicon;
    				filetype=_("Character Device");
    			}
    			else if(item->state&FileItem::BLOCKDEV)
    			{
    				big=bigblockdevicon;
    				mini=miniblockdevicon;
    				filetype=_("Block Device");
    			}
    			else if(item->state&FileItem::FIFO)
    			{
    				big=bigpipeicon;
    				mini=minipipeicon;
    				filetype=_("Named Pipe");
    			}
    			else if(item->state&FileItem::SOCK)
                {
                	big=bigsocketicon;
                	mini=minisocketicon;
                	filetype=_("Socket");
                }
                else if(item->state&FileItem::EXECUTABLE)
                {
                	big=bigexecicon;
                	mini=miniexecicon;
                	filetype=_("Executable");
                	if(associations)
					{
						fileassoc=associations->findFileBinding(pathname.text());	

						// In some special cases where the file is executable and its file name
						// is equal to the short extension (ex: zip, cc, ...) we don't want to use the association
						if ( fileassoc && (name==fileassoc->key) )
							fileassoc=NULL;
					}					
                }
                else
                {
                	big=bigdocicon;
               		mini=minidocicon;
                	filetype=_("Document");
                	if(associations)
					{
						// If we are in trash can, first strip the deletion date
						if (isInTrash)
						{                		
							FXString stripname=pathname.rbefore('_',4);
							if (stripname=="")
								fileassoc=associations->findFileBinding(pathname.text());
							else
								fileassoc=associations->findFileBinding(stripname.text());
						}
						else
							fileassoc=associations->findFileBinding(pathname.text());
					}
                }

                // If association is found, use it
				if (fileassoc)
				{
					filetype=fileassoc->extension.text();
					
					if(fileassoc->bigicon)
						big=fileassoc->bigicon;
					if(fileassoc->miniicon)
						mini=fileassoc->miniicon;
													
					// Eventually display thumbnails, based on file extension
					if (displaythumbnails)
					{
						image=FALSE;
						lowext=ext.lower();
						if (lowext=="jpg" || lowext=="jpeg")
						{
							big=new FXJPGIcon(getApp());
							mini=new FXJPGIcon(getApp());
							image=TRUE;
						}
						else if (lowext=="png")
						{
							big=new FXPNGIcon(getApp());
							mini=new FXPNGIcon(getApp());
							image=TRUE;
						}
						else if(lowext=="gif")
						{
							big=new FXGIFIcon(getApp());
							mini=new FXGIFIcon(getApp());
							image=TRUE;
						}
						else if (lowext=="xpm")
						{
							big=new FXXPMIcon(getApp());
							mini=new FXXPMIcon(getApp());
							image=TRUE;
						}
						else if (lowext=="tif" || lowext=="tiff")
						{
							big=new FXTIFIcon(getApp());
							mini=new FXTIFIcon(getApp());
							image=TRUE;
						}
						else if (lowext=="bmp")
						{
							big=new FXBMPIcon(getApp());
							mini=new FXBMPIcon(getApp());
							image=TRUE;
						}
						if (image)
						{
							FXFileStream str;

							if(str.open(pathname,FXStreamLoad))
							{
								big->loadPixels(str);
								str.close();
								
								// Eventually scale the big icon (best quality)
								if((big->getWidth()>MAX_BIG_ICON_SIZE) || (big->getHeight()>MAX_BIG_ICON_SIZE))
								{
									if(big->getWidth()>big->getHeight())
										big->scale(MAX_BIG_ICON_SIZE,(MAX_BIG_ICON_SIZE*big->getHeight())/big->getWidth(),1);
									else
										big->scale((MAX_BIG_ICON_SIZE*big->getWidth())/big->getHeight(),MAX_BIG_ICON_SIZE,1);
								}
								
								// Copy the mini icon to the big icon (faster!)
								if (!FXMEMDUP(&tmpdata,big->getData(),FXColor,big->getWidth()*big->getHeight()))
									throw FXMemoryException(_("Unable to load image"));
								mini->setData(tmpdata,IMAGE_OWNED,big->getWidth(),big->getHeight());	

								// Eventually scale the mini icon (best quality)
								if((mini->getWidth()>MAX_MINI_ICON_SIZE) || (mini->getHeight()>MAX_MINI_ICON_SIZE))
								{
									if(mini->getWidth()>mini->getHeight())
										mini->scale(MAX_MINI_ICON_SIZE,(MAX_MINI_ICON_SIZE*mini->getHeight())/mini->getWidth(),1);
									else
										mini->scale((MAX_MINI_ICON_SIZE*mini->getWidth())/mini->getHeight(),MAX_MINI_ICON_SIZE,1);
								}
							}
						}
					}
				}

				// Symbolic links have a specific type
				if (isBrokenLink)
				{
					filetype=_("Broken link");
				}
				else if (isLink)
				{
					if (associations)
					{
						// Don't forget to remove trailing '/' here!
						fileassoc=associations->findFileBinding(::cleanPath(readlink(pathname)).text());
						if (fileassoc)
							filetype=_("Link to ")+fileassoc->extension;
						else
							filetype=_("Link to ")+filetype;
					}
				}
				
				// Update item information
				FXchar size[64];
				snprintf(size,sizeof(size)-1,"%llu",(unsigned long long)linfo.st_size);
				FXString hsize=::hSize(size);

				// If we are in trash can, display the deletion date
				if (isInTrash)
					item->label.format("%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s",name.text(),hsize.text(),filetype.text(),ext.text(),mod.text(),usrid.text(),grpid.text(),atts.text(),del.text());
				else	
					item->label.format("%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s",name.text(),hsize.text(),filetype.text(),ext.text(),mod.text(),usrid.text(),grpid.text(),atts.text());
				
				item->bigIcon=big;
				item->miniIcon=mini;
				item->size=(unsigned long long)linfo.st_size;
				item->assoc=fileassoc;
				item->date=filetime;
				item->deldate=deldate;

#if defined(linux)
				 // Devices have a specific icon
				if(fsdevices->find(pathname.text()))
				{
					if(::streq(fsdevices->find(pathname.text()),"harddisk"))
					{
						item->bigIcon=bigharddiskicon;
						item->miniIcon=harddiskicon;
					}
					else if(::streq(fsdevices->find(pathname.text()),"nfsdisk"))
					{
						item->bigIcon=bignfsdriveicon;
						item->miniIcon=nfsdriveicon;
					}
					else if(::streq(fsdevices->find(pathname.text()),"smbdisk"))
					{
						item->bigIcon=bignfsdriveicon;
						item->miniIcon=nfsdriveicon;
					}
					else if(::streq(fsdevices->find(pathname.text()),"floppy"))
					{
						item->bigIcon=bigfloppyicon;
						item->miniIcon=floppyicon;
					}
					else if(::streq(fsdevices->find(pathname.text()),"cdrom"))
					{
						item->bigIcon=bigcdromicon;
						item->miniIcon=cdromicon;
					}
					else if(::streq(fsdevices->find(pathname.text()),"zip"))
					{
						item->bigIcon=bigzipicon;
						item->miniIcon=zipicon;
					}
				}
#endif
				// Dotdot folders have a specific icon
				if(name[0]=='.' && name[1]=='.' && name[2]==0)
				{
					item->bigIcon=bigfolderupicon;
					item->miniIcon=minifolderupicon;
				}

				// Symbolic links have a specific icon
				if (isLink)
				{
					// Broken link
					if (isBrokenLink)
					{
						item->bigIcon=bigbrokenlinkicon;
						item->miniIcon=minibrokenlinkicon;
					}
					else
					{
						item->bigIcon=biglinkicon;
						item->miniIcon=minilinkicon;
					}
				}

				// Create item
				if(id())
					item->create();
			}
			closedir(dirp);
		}
	}

    // Wipe items remaining in list:- they have disappeared!!
    for(item=oldlist; item; item=link)
    {
    	link=item->link;
    	delete item;
    }

    // Validate
    if(current>=items.no())
    	current=-1;
    if(anchor>=items.no())
    	anchor=-1;
    if(extent>=items.no())
    	extent=-1;

    // Remember new list
    list=newlist;

    // Gotta recalc size of content
    recalc();

	// Stop wait cursor
	getApp()->endWaitCursor();
}


// Is directory
FXbool FileList::isItemDirectory(FXint index) const
{
    if(index<0 || items.no()<=index)
        fxerror("%s::isItemDirectory: index out of range.\n",getClassName());
    return (((FileItem*)items[index])->state&FileItem::FOLDER)!=0;
}


// Is file
FXbool FileList::isItemFile(FXint index) const
{
    if(index<0 || items.no()<=index)
        fxerror("%s::isItemFile: index out of range.\n",getClassName());
    return (((FileItem*)items[index])->state&(FileItem::FOLDER|FileItem::CHARDEV|FileItem::BLOCKDEV|FileItem::FIFO|FileItem::SOCK))==0;
}


// Is executable
FXbool FileList::isItemExecutable(FXint index) const
{
    if(index<0 || items.no()<=index)
        fxerror("%s::isItemExecutable: index out of range.\n",getClassName());
    return (((FileItem*)items[index])->state&FileItem::EXECUTABLE)!=0;
}

// Get number of selected items
FXint FileList::getNumSelectedItems(void) const
{
    FXint num=0;
    for (int u=0; u<getNumItems(); u++)
        if (isItemSelected(u))
            num++;
    return num;
}


// Get number of selected items and index of first selected item
FXint FileList::getNumSelectedItems(FXint *index) const
{
    FXint num=0, itm=-1;
    for (int u=0; u<getNumItems(); u++)
        if (isItemSelected(u))
        {
            if (itm== -1)
                itm= u;
            num++;
        }
    (*index)=itm;
    return num;
}


// Get file name from item
FXString FileList::getItemFilename(FXint index) const
{
    if(index<0 || items.no()<=index)
        fxerror("%s::getItemFilename: index out of range.\n",getClassName());
    FXString label=items[index]->getText();
    return label.section('\t',0);
}


// Get full pathname to item
FXString FileList::getItemPathname(FXint index) const
{
    if(index<0 || items.no()<=index)
        fxerror("%s::getItemPathname: index out of range.\n",getClassName());
    FXString label=items[index]->getText();
    return FXPath::absolute(directory,label.section('\t',0));
}


// Get associations (if any) from the file
FileAssoc* FileList::getItemAssoc(FXint index) const
{
    if(index<0 || items.no()<=index)
        fxerror("%s::getItemAssoc: index out of range.\n",getClassName());
    return ((FileItem*)items[index])->assoc;
}

// Return file size of the item
unsigned long long FileList::getItemFileSize(FXint index) const
{
    if(index<0 || items.no()<=index)
        fxerror("%s::getItemFileSize: index out of range.\n",getClassName());
    return ((FileItem*)items[index])->size;
}


// Change associations table; force a rescan so as to
// update the bindings in each item to the new associations
void FileList::setAssociations(FileDict* assocs)
{
    if(associations!=assocs)
    {
        associations=assocs;
		scan(TRUE);
    }
}


// Cleanup
FileList::~FileList()
{
	getApp()->removeTimeout(this,ID_REFRESH_TIMER);
	getApp()->removeTimeout(this,ID_OPEN_TIMER);
	delete associations;
	delete forwardhist;
	delete backhist;

    associations=(FileDict*)-1;
	list=(FileItem*)-1L;
}
