// This is adapted from 'QueryRPM', a simple RPM viewer/installer found in X Win Commander

#include "config.h"
#include "i18n.h"

#include <fox-1.4/fx.h>
#include <fox-1.4/fxkeys.h>
#include <fox-1.4/FXPNGIcon.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>

#include "icons.h"
#include "File.h"
#include "FileDialog.h"
#include "MessageBox.h"
#include "XFileQuery.h"
#include "CommandWindow.h"

// Maximum length of a file name
#ifndef MAXPATHLEN
#define MAXPATHLEN 1024
#endif

FXMainWindow *mainWindow;
char **args;

// Map
FXDEFMAP(XFileQuery) XFileQueryMap[]={
                                         FXMAPFUNC(SEL_COMMAND,XFileQuery::ID_QUIT,XFileQuery::onCmdQuit),
                                         FXMAPFUNC(SEL_COMMAND,XFileQuery::ID_UNINSTALL,XFileQuery::onCmdUninstall),
                                       	 FXMAPFUNC(SEL_COMMAND,XFileQuery::ID_ABOUT,XFileQuery::onCmdAbout),
                                       	 FXMAPFUNC(SEL_COMMAND,XFileQuery::ID_OPEN,XFileQuery::onCmdOpen),
                                         FXMAPFUNC(SEL_COMMAND,XFileQuery::ID_UPGRADE,XFileQuery::onCmdUpgrade),
                                         FXMAPFUNC(SEL_UPDATE,XFileQuery::ID_DESCRIPTION,XFileQuery::onUpdDescription),
                                         FXMAPFUNC(SEL_UPDATE,XFileQuery::ID_FILELIST,XFileQuery::onUpdFileList),
                                     };


// Object implementation
FXIMPLEMENT(XFileQuery,FXMainWindow,XFileQueryMap,ARRAYNUMBER(XFileQueryMap))

// Constructor
XFileQuery::XFileQuery(FXApp *a):FXMainWindow(a,"Xfq",NULL,NULL,DECOR_ALL,0,0,600,500,0,0)
{
    FXIcon *bigicon=new FXPNGIcon(getApp(),xfq);
    setIcon(bigicon);

    // Create icons
	FXPNGIcon *fileopenicon=new FXPNGIcon(getApp(),fileopen);
	FXPNGIcon *quiticon=new FXPNGIcon(getApp(),quit);

	// Site where to dock
	FXDockSite *docksite=new FXDockSite(this,LAYOUT_SIDE_TOP|LAYOUT_FILL_X);

    // Make menu bar
  	FXToolBarShell* dragshell=new FXToolBarShell(this,FRAME_RAISED);
    menubar=new FXMenuBar(docksite,dragshell,LAYOUT_DOCK_NEXT|LAYOUT_SIDE_TOP|LAYOUT_FILL_X|FRAME_RAISED);
	new FXToolBarGrip(menubar,menubar,FXMenuBar::ID_TOOLBARGRIP,TOOLBARGRIP_DOUBLE);

    // Tool bar
  	FXToolBarShell* dragshell1=new FXToolBarShell(this,FRAME_RAISED);
  	toolbar=new FXToolBar(docksite,dragshell1,LAYOUT_DOCK_NEXT|LAYOUT_SIDE_TOP|LAYOUT_FILL_X|FRAME_RAISED);
  	new FXToolBarGrip(toolbar,toolbar,FXToolBar::ID_TOOLBARGRIP,TOOLBARGRIP_DOUBLE);
    int style=BUTTON_TOOLBAR;

    // File menu
    filemenu=new FXMenuPane(this);
    new FXMenuTitle(menubar,_("&File"),NULL,filemenu);

    // Preferences Menu
    FXMenuPane *prefsmenu=new FXMenuPane(this);
    new FXMenuTitle(menubar,_("&Preferences"),NULL,prefsmenu);

    // Help menu
    helpmenu=new FXMenuPane(this);
    new FXMenuTitle(menubar,_("&Help"),NULL,helpmenu);

    // Toolbar button: File open
    new FXButton(toolbar,_("\tOpen package file (Ctrl-O)"),fileopenicon,this,ID_OPEN,style|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);

    // Toolbar button: Quit
    new FXFrame(toolbar,LAYOUT_TOP|LAYOUT_LEFT|LAYOUT_FIX_WIDTH|LAYOUT_FIX_HEIGHT,0,0,10,20);
    new FXButton(toolbar,_("\tQuit Xfq (Ctrl-Q)"),quiticon,this,ID_QUIT,style|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);

    // File Menu entries
    new FXMenuCommand(filemenu,_("&Open file...\tCtrl-O"),fileopenicon,this,ID_OPEN);
    new FXMenuCommand(filemenu,_("&Quit\tCtrl-Q"),quiticon,this,ID_QUIT);

    // Preferences menu
    new FXMenuCheck(prefsmenu,_("&Toolbar\t\tDisplay toolbar"),toolbar,FXWindow::ID_TOGGLESHOWN);

    // Help Menu entries
    new FXMenuCommand(helpmenu,_("&About X File Query\tF1"),NULL,this,ID_ABOUT,0);

    // Make a tool tip
    new FXToolTip(getApp(),0);

    // Buttons
    FXHorizontalFrame *buttons=new FXHorizontalFrame(this,LAYOUT_SIDE_BOTTOM|LAYOUT_FILL_X,0,0,0,0,10,10,5,5);

    // Make contents
    contents=new FXHorizontalFrame(this,LAYOUT_FILL_X|LAYOUT_FILL_Y,0,0,0,0, 0,0,0,0, 0,0);

    // Close
    new FXButton(buttons,_("&Close"),NULL,this,XFileQuery::ID_QUIT,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,20,20);

    // Uninstall
    new FXButton(buttons,_("&Uninstall"),NULL,this,XFileQuery::ID_UNINSTALL,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,20,20);

    // Install/Upgrade
    new FXButton(buttons,_("&Install/Upgrade"),NULL,this,XFileQuery::ID_UPGRADE,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,20,20);

    // Switcher
    FXTabBook *tabbook = new FXTabBook(contents,NULL,0,LAYOUT_FILL_X|LAYOUT_FILL_Y|LAYOUT_RIGHT);

    // First item is Description
    new FXTabItem(tabbook,_("&Description"),NULL);
    FXVerticalFrame *descr=new FXVerticalFrame(tabbook,FRAME_RAISED);
    FXPacker *p=new FXPacker(descr,FRAME_GROOVE|LAYOUT_FILL_X|LAYOUT_FILL_Y);
    description=new FXText(p,this,XFileQuery::ID_DESCRIPTION,FRAME_SUNKEN|LAYOUT_FILL_X|LAYOUT_FILL_Y);

    // Second item is File List
    new FXTabItem(tabbook,_("&File List"),NULL);
    FXVerticalFrame *flistfr=new FXVerticalFrame(tabbook,FRAME_RAISED);
    p=new FXPacker(flistfr,FRAME_GROOVE|LAYOUT_FILL_X|LAYOUT_FILL_Y);
	list=new FXTreeList(p,this,XFileQuery::ID_FILELIST,LAYOUT_FILL_X|LAYOUT_FILL_Y|TREELIST_SHOWS_LINES);

	// Folder icons for file list
    folder_open=new FXPNGIcon(getApp(),package);
    folder_closed=new FXPNGIcon(getApp(),package);

    // Document icon for file list
    doc=new FXPNGIcon(getApp(),minidoc);

	// Initialize file name
	filename="";
}


// About box
long XFileQuery::onCmdAbout(FXObject*,FXSelector,void*)
{
    FXPNGIcon icon(getApp(),xfq);
    MessageBox about(this,_("About X File Query"),_("X File Query is a simple RPM manager.\n\nCopyright (C) 2002-2005 Roland Baudin (roland65@free.fr)"),&icon,MBOX_OK|DECOR_TITLE|DECOR_BORDER);
    about.execute();
    return 1;
}

// Open package
long XFileQuery::onCmdOpen(FXObject*,FXSelector,void*)
{
    const FXchar *patterns[]=
        {
            _("All Files"),			"*",
            _("RPM source packages"),	"*.src.rpm",
            _("RPM packages"),			"*.rpm",
			NULL
        };
    FileDialog open(this,_("Open Document"));
	open.setSelectMode(SELECTFILE_EXISTING);
    open.setPatternList(patterns);
  	open.setDirectory(FXFile::directory(filename));
    if(open.execute())
    {
        filename=open.getFilename();
		list_clean=TRUE;
		desc_clean=TRUE;
    }
    return 1;
}

// Install/upgrade RPM package
long XFileQuery::onCmdUpgrade(FXObject*,FXSelector,void*)
{
	getApp()->flush();

	if (strlen(filename.text())==0)
	{
        MessageBox::error(this,MBOX_OK,_("Error"),_("No RPM package loaded"));
		return 0;
	}
		
	// Package name
    FXString package;
	package=FXFile::name(filename);

    // Command to perform
	FXString cmd="rpm -Uvh " +filename;

    // Make and show command window
	CommandWindow *cmdwin=new CommandWindow(mainWindow,_("Install/Upgrade RPM"),cmd,10,80);			
	cmdwin->create();

	// The CommandWindow object will delete itself when closed!

    return 1;
}


// Uninstall RPM package based on the package name (version is ignored)
long XFileQuery::onCmdUninstall(FXObject*,FXSelector,void*)
{
	getApp()->flush();
	
	if (strlen(filename.text())==0)
	{
        MessageBox::error(this,MBOX_OK,_("Error"),_("No RPM package loaded"));
		return 0;
	}

	// Get package name
    FXString package;
	package=FXFile::name(filename);
	package=package.section('-',0);
    
	// Uninstall command
	FXString cmd="rpm -e " + package;

    // Make and show command window
	CommandWindow *cmdwin=new CommandWindow(mainWindow,_("Uninstall RPM"),cmd,10,80);			
	cmdwin->create();

	// The CommandWindow object will delete itself when closed!

    return 1;
}


// Save configuration when quitting
void XFileQuery::saveConfig()
{
    // Write new window size and position back to registry
    getApp()->reg().writeUnsignedEntry("SETTINGS","width",(FXuint)getWidth());
    getApp()->reg().writeUnsignedEntry("SETTINGS","height",(FXuint)getHeight());
	getApp()->reg().writeIntEntry("SETTINGS","xpos",getX());
	getApp()->reg().writeIntEntry("SETTINGS","ypos",getY());

    // Last opened filename
    getApp()->reg().writeStringEntry("SETTINGS","filename",filename.text());

    // Toolbar status
	if(toolbar->shown())
        getApp()->reg().writeUnsignedEntry("SETTINGS","showtoolbar",TRUE);
    else
        getApp()->reg().writeUnsignedEntry("SETTINGS","showtoolbar",FALSE);

	// Write registry settings
	getApp()->reg().write();
}


// Quit application
long XFileQuery::onCmdQuit(FXObject*,FXSelector,void*)
{
	// Save settings 
    saveConfig();

    // Exit
	getApp()->exit(0);
    return 1;
}


// Update file list
long XFileQuery::onUpdFileList(FXObject*,FXSelector,void*)
{
	if(!list_clean)
        return 0;
	
    list_clean=FALSE;
    FXString cmd="rpm -qlp '"+filename+"'";
    FILE *rpmcmd=popen(cmd.text(),"r");
    if(!rpmcmd)
    {
        perror("popen");
        exit(1);
    }
    char text[10000]={0};
    FXString str;
    str=FXFile::name(filename);
    strlcpy(text,str.text(),str.length()+1);

	// Clear list
    list->clearItems();

    // First item
	FXTreeItem *topmost;
	topmost=list->prependItem(NULL,text,folder_open,folder_closed);
    
	// Next items
	while(fgets(text,sizeof(text),rpmcmd))
    {
        text[strlen(text)-1]='\0';// kill trailing lf
        list->appendItem(topmost,text,doc,doc);
    }
    list->expandTree(topmost);
    pclose(rpmcmd);
    return 1;
}


// Update description
long XFileQuery::onUpdDescription(FXObject*,FXSelector,void*)
{
    if(!desc_clean)
        return 0;
    desc_clean=FALSE;
    FXString cmd="rpm -qip '"+filename+"'";
    FILE *rpmcmd=popen(cmd.text(),"r");
    if(!rpmcmd)
    {
        perror("popen");
        exit(1);
    }
    char text[10000]={0};
    FXString buf;
    while(fgets(text,sizeof(text),rpmcmd))
        buf+=text;
    if(pclose(rpmcmd))
    {
        MessageBox::error(this,MBOX_OK,_("Error"),_("Query of %s failed!"),filename.text());
		list->clearItems();
        description->setText("");
		filename="";
		desc_clean=FALSE;
    	list_clean=FALSE;
    }
    else
        description->setText(buf.text());

    return 1;
}

// Start the ball rolling
void XFileQuery::start(FXString startpkg)
{
    filename=startpkg;

    if(filename != "")
    {
    	struct stat info;
    	FILE *fp;
    	if(stat(filename.text(),&info)!=0)
    	{
        	MessageBox::error(this,MBOX_OK,_("Error Loading File"),_("Non-existing file: %s"),filename.text());
			filename="";
			desc_clean=FALSE;
    		list_clean=FALSE;
    	}
    	else
		{
			fp=fopen(filename.text(),"r");
    		if(!fp)
    		{
        		MessageBox::error(this,MBOX_OK,_("Error Loading File"),_("Unable to open file: %s"),filename.text());
				filename="";
				desc_clean=FALSE;
    			list_clean=FALSE;
    		}
			else
			{
				fclose(fp);
    			desc_clean=TRUE;
    			list_clean=TRUE;
			}
		}
	}
	else
	{
		filename="";
		desc_clean=FALSE;
    	list_clean=FALSE;
	}
}

void XFileQuery::create()
{
    // Read the Xfe registry
	FXRegistry* reg_xfe=new FXRegistry("Xfe",NULL);
	reg_xfe->read();
	
	// Set Xfq text font according to the Xfe registry
	FXString fontspec;
	fontspec=reg_xfe->readStringEntry("SETTINGS","textfont","Helvetica,100,normal,regular");
	if(!fontspec.empty())
	{
    	FXFont* font=new FXFont(getApp(),fontspec);
        description->setFont(font);
        list->setFont(font);
	}
	delete reg_xfe;
	
	// Create icons
	folder_open->create();
	folder_closed->create();
	doc->create();

	// Wheel scrolling
    FXint wheellines=getApp()->reg().readUnsignedEntry("SETTINGS","wheellines",5);
	getApp()->setWheelLines(wheellines);
	
    // Get size
    FXuint width=getApp()->reg().readUnsignedEntry("SETTINGS","width",600);
    FXuint height=getApp()->reg().readUnsignedEntry("SETTINGS","height",400);
    FXint xpos=getApp()->reg().readIntEntry("SETTINGS","xpos",0);
    FXint ypos=getApp()->reg().readIntEntry("SETTINGS","ypos",0);

	// Get toolbar status
    if(getApp()->reg().readUnsignedEntry("SETTINGS","showtoolbar",TRUE)==FALSE)
        toolbar->hide();

    // Reposition window
    position(xpos,ypos,width,height);
    FXMainWindow::create();
    show();
    description->handle(this,FXSEL(SEL_COMMAND,FXText::ID_TOGGLE_EDITABLE),NULL);
}



// Usage message
#define USAGE_MSG	_("\
Usage: xfq [options] [package] \n\
\n\
    [options] can be any of the following:\n\
\n\
        -h, --help         Print (this) help screen and exit.\n\
        -v, --version      Print version information and exit.\n\
\n\
    [package] is the path to the rpm package you want to open on start up.\n\
\n")



// Main function to start the program
int main(int argc,char* argv[])
{
	int i;
	FXString startpkg="";
	char *appname = "Xfq";
	char *xfename = "Xfe";
	char *vdrname = "XFileExplorer";

#ifdef HAVE_SETLOCALE
  // Set locale via LC_ALL.
  setlocale (LC_ALL, "");
#endif

#if ENABLE_NLS
  // Set the text message domain.
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);
#endif

	XFileQuery *window;
	
	// Parse basic arguments
	for(i = 1; i < argc; ++i)
	{
    	if(compare(argv[i],"-v")==0 || compare(argv[i],"--version")==0)
		{
			fprintf(stdout,"%s version %s\n",PACKAGE,VERSION);
			exit(0);
		}
    	else if(compare(argv[i],"-h")==0 || compare(argv[i],"--help")==0)
		{
			fprintf(stdout,USAGE_MSG);
			exit(0);
		}
		else
	    {
			// Start file, if any
			startpkg=argv[i];
	    }
	}

	// Global variable
	args=argv;

    // Make application
    FXApp* application=new FXApp(appname,vdrname);

    // Open display
    application->init(argc,argv);

    // Read the Xfe registry
	FXRegistry *reg;
  	reg=new FXRegistry(xfename,vdrname);
	reg->read();
	
	// Set Xfq normal font according to the Xfe registry
	FXString fontspec;
	fontspec=reg->readStringEntry("SETTINGS","font","Helvetica,100,normal,regular");
	if(!fontspec.empty())
	{
    	FXFont* normalFont=new FXFont(application,fontspec);
        application->setNormalFont(normalFont);
    }
	
	// Set Xfq colors according to the Xfe registry
	FXColor basecolor,forecolor,backcolor,selforecolor,selbackcolor,bordercolor;
    basecolor=reg->readColorEntry("SETTINGS","basecolor",FXRGB(214,207,198));
    forecolor=reg->readColorEntry("SETTINGS","forecolor",FXRGB(0,0,0));
    backcolor=reg->readColorEntry("SETTINGS","backcolor",FXRGB(255,255,255));
    selforecolor=reg->readColorEntry("SETTINGS","selforecolor",FXRGB(255,255,255));
    selbackcolor=reg->readColorEntry("SETTINGS","selbackcolor",FXRGB(10,36,106));
    bordercolor=reg->readColorEntry("SETTINGS","bordercolor",FXRGB(0,0,0));
	application->setBaseColor(basecolor);
	application->setForeColor(forecolor);
	application->setBackColor(backcolor);
	application->setSelforeColor(selforecolor);
	application->setSelbackColor(selbackcolor);
	application->setBorderColor(bordercolor);

	// Delete the Xfe registry
	delete reg;

    // Make window
	window=new XFileQuery(application);
	mainWindow=window;

    // Create it
    application->create();

    // Test the existence of the RedHat package manager (RPM)
	// This is done by checking the existence of the string 'RPM'
	// in the response to the command 'rpm' (with no arguments)
	FXString cmd="rpm 2>&1";
    FILE *rpmcmd=popen(cmd.text(),"r");
    if(!rpmcmd)
    {
        perror("popen");
        exit(1);
    }
    char text[10000]={0};
    FXString buf;
    while(fgets(text,sizeof(text),rpmcmd))
        buf+=text;
    snprintf(text,sizeof(text)-1,"%s",buf.text());
	if (strstr(text,"RPM")==NULL)
	{
        MessageBox::error(window,MBOX_OK,_("Error"),_("RedHat package manager (RPM) not found!"));
		exit(1);
	}

    // Start
    window->start(startpkg);

    // Run the application
    application->run();

    return 0;
}
