// Properties box

#include "config.h"
#include "i18n.h"

#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <signal.h>
#include <pwd.h>
#include <grp.h>
#if defined(linux)
#include <mntent.h>
#endif

#include <fox-1.2/fx.h>
#include <fox-1.2/fxkeys.h>
#include <fox-1.2/FXPNGIcon.h>

#include "icons.h"
#include "FileDialog.h"
#include "OverwriteBox.h"
#include "File.h"
#include "Properties.h"
#include "FilePanel.h"
#include "XFileExplorer.h"
#include "MessageBox.h"



// Maximum file name length
#ifndef MAXPATHLEN
#define MAXPATHLEN 1024
#endif

// Root directory string
#ifndef ROOTDIR
#define ROOTDIR "/"
#endif


// Global variables
extern FXMainWindow *mainWindow;
extern FXStringDict *devices;


// Map
FXDEFMAP(PermFrame) PermFrameMap[]={
                                   };

// Object implementation
FXIMPLEMENT(PermFrame,FXVerticalFrame,PermFrameMap,ARRAYNUMBER(PermFrameMap))

PermFrame::PermFrame(FXComposite* parent,FXObject *target):
        FXVerticalFrame(parent,FRAME_THICK|FRAME_RAISED)
{
    FXHorizontalFrame *accessframe=new FXHorizontalFrame(this,LAYOUT_FILL_X|LAYOUT_FILL_Y);
    FXHorizontalFrame *chmodframe=new FXHorizontalFrame(this,PACK_UNIFORM_WIDTH|PACK_UNIFORM_HEIGHT|LAYOUT_FILL_X|LAYOUT_FILL_Y);
    FXGroupBox *group1=new FXGroupBox(accessframe,_("User"),GROUPBOX_TITLE_LEFT|FRAME_GROOVE|LAYOUT_FILL_X);
    ur=new FXCheckButton(group1,_("Read"),target,PropertiesBox::ID_RUSR);
    uw=new FXCheckButton(group1,_("Write"),target,PropertiesBox::ID_WUSR);
    ux=new FXCheckButton(group1,_("Execute"),target,PropertiesBox::ID_XUSR);
	FXGroupBox *group2=new FXGroupBox(accessframe,_("Group"),GROUPBOX_TITLE_LEFT|FRAME_GROOVE|LAYOUT_FILL_X);
    gr=new FXCheckButton(group2,_("Read"),target,PropertiesBox::ID_RGRP);
    gw=new FXCheckButton(group2,_("Write"),target,PropertiesBox::ID_WGRP);
    gx=new FXCheckButton(group2,_("Execute"),target,PropertiesBox::ID_XGRP);
    FXGroupBox *group3=new FXGroupBox(accessframe,_("Others"),GROUPBOX_TITLE_LEFT|FRAME_GROOVE|LAYOUT_FILL_X);
    or_=new FXCheckButton(group3,_("Read"),target,PropertiesBox::ID_ROTH);
    ow=new FXCheckButton(group3,_("Write"),target,PropertiesBox::ID_WOTH);
    ox=new FXCheckButton(group3,_("Execute"),target,PropertiesBox::ID_XOTH);

    FXGroupBox *group4=new FXGroupBox(chmodframe,_("Command"),GROUPBOX_TITLE_LEFT|FRAME_GROOVE|LAYOUT_FILL_X|LAYOUT_FILL_Y);
	radiotarget.connect(cmd);
	set=new FXRadioButton(group4,_("Set marked"),&radiotarget,FXDataTarget::ID_OPTION+PropertiesBox::ID_SET);
    clear=new FXRadioButton(group4,_("Clear marked"),&radiotarget,FXDataTarget::ID_OPTION+PropertiesBox::ID_CLEAR);
    add=new FXRadioButton(group4,_("Add marked"),&radiotarget,FXDataTarget::ID_OPTION+PropertiesBox::ID_ADD);
    rec=new FXCheckButton(group4,_("Recursive folders"),NULL,0);
    dironly=new FXCheckButton(group4,_("Folders only"),NULL,0);
    dironly->disable();

    FXGroupBox *group5=new FXGroupBox(chmodframe,_("Owner"),GROUPBOX_TITLE_LEFT|FRAME_GROOVE|LAYOUT_FILL_X|LAYOUT_FILL_Y);
    new FXLabel(group5,_("User"));
    user=new FXComboBox(group5,10,NULL,0,COMBOBOX_NO_REPLACE|FRAME_SUNKEN|FRAME_THICK);
  	user->setNumVisible(5);
    new FXLabel(group5,_("Group"));
    grp=new FXComboBox(group5,10,NULL,0,COMBOBOX_NO_REPLACE|FRAME_SUNKEN|FRAME_THICK);
  	grp->setNumVisible(5);

	// User names (sorted in ascending order)
    struct passwd* pwde;
    while( (pwde = getpwent()) )
    {
        user->appendItem(pwde->pw_name);
    }
    endpwent();
	user->setSortFunc(FXList::ascending);
	user->sortItems();	

	// Group names (sorted in ascending order)
    struct group* grpe;
    while( (grpe = getgrent()) )
    {
        grp->appendItem(grpe->gr_name);
    }
    endgrent();
	grp->setSortFunc(FXList::ascending);
	grp->sortItems();	
}

// Map
FXDEFMAP(PropertiesBox) PropertiesBoxMap[]={
            FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_ACCEPT_SINGLE,PropertiesBox::onCmdAcceptSingle),
            FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_ACCEPT_MULT,PropertiesBox::onCmdAcceptMult),
            FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_RUSR,PropertiesBox::onCmdCheck),
            FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_WUSR,PropertiesBox::onCmdCheck),
            FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_XUSR,PropertiesBox::onCmdCheck),
            FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_RGRP,PropertiesBox::onCmdCheck),
            FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_WGRP,PropertiesBox::onCmdCheck),
            FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_XGRP,PropertiesBox::onCmdCheck),
            FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_ROTH,PropertiesBox::onCmdCheck),
            FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_WOTH,PropertiesBox::onCmdCheck),
            FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_XOTH,PropertiesBox::onCmdCheck),
  			FXMAPFUNC(SEL_UPDATE,0,PropertiesBox::onUpdDironly),
            FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_SET,PropertiesBox::onCmdCommand),
            FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_CLEAR,PropertiesBox::onCmdCommand),
            FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_ADD,PropertiesBox::onCmdCommand),
            FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_BIG_ICON,PropertiesBox::onCmdBrowseIcon),
            FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_MINI_ICON,PropertiesBox::onCmdBrowseIcon),
            FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_BROWSE_OPEN,PropertiesBox::onCmdBrowse),
            FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_BROWSE_VIEW,PropertiesBox::onCmdBrowse),
            FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_BROWSE_EDIT,PropertiesBox::onCmdBrowse),
			FXMAPFUNC(SEL_KEYPRESS,0,PropertiesBox::onCmdKeyPress),
        };

// Object implementation
FXIMPLEMENT(PropertiesBox,FXDialogBox,PropertiesBoxMap,ARRAYNUMBER(PropertiesBoxMap))

// Construct window for one file
PropertiesBox::PropertiesBox(FXWindow *win,FXString file,FXString path): FXDialogBox(win,_("Properties"), DECOR_TITLE|DECOR_BORDER/*,0,0,350,200*/)
{
    unsigned long long filesize, dsize;
    FXString mod,changed,accessed;
	FXString grpid, usrid;
    struct stat info;
    FXString type, extension;
    int islink = 0;
	int isdirectory = 0;
    char filename[MAXPATHLEN+1],linkto[MAXPATHLEN+1];
    char mnttype[64], used[64], avail[64], size[64];
    char buf[MAXPATHLEN+1];
	FXbool ismountpoint=FALSE;
	FILE *p;

    // Buttons
    FXHorizontalFrame *buttons=new FXHorizontalFrame(this,LAYOUT_SIDE_BOTTOM|LAYOUT_FILL_X,0,0,0,0,10,10,5,5);

    // Contents
    FXVerticalFrame *contents=new FXVerticalFrame(this,LAYOUT_SIDE_TOP|FRAME_NONE|LAYOUT_FILL_X|LAYOUT_FILL_Y|PACK_UNIFORM_WIDTH);

    // Accept
	if (file!="..")
    {
		FXButton *ok = new FXButton(buttons,_("&Accept"),NULL,this,PropertiesBox::ID_ACCEPT_SINGLE,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,20,20);
    	ok->addHotKey(KEY_Return);
	}

    // Cancel
    new FXButton(buttons,_("&Cancel"),NULL,this,PropertiesBox::ID_CANCEL,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,20,20);

    // Switcher
    FXTabBook *tabbook = new FXTabBook(contents,NULL,0,LAYOUT_FILL_X|LAYOUT_FILL_Y|LAYOUT_RIGHT);

    // First item is General
    new FXTabItem(tabbook,_("&General"),NULL);
    FXPacker *genpack = new FXPacker(tabbook,FRAME_THICK|FRAME_RAISED);
    FXGroupBox *generalgroup=new FXGroupBox(genpack,NULL,FRAME_GROOVE|LAYOUT_FILL_X|LAYOUT_FILL_Y);
    FXVerticalFrame *generalframe=new FXVerticalFrame(generalgroup,LAYOUT_FILL_X|LAYOUT_FILL_Y);

    // Second item is Access Permissions
	FXTabItem *permtab = new FXTabItem(tabbook,_("&Permissions"),NULL);
    perm = new PermFrame(tabbook,this);
	
	// Permission tab is disabled for parent directory
	if (file=="..")
		permtab->disable();

    // Third tab - file associations
    FXTabItem *fassoctab = new FXTabItem(tabbook,_("&File Associations"),NULL);
    FXPacker *fassocpack = new FXPacker(tabbook,FRAME_THICK|FRAME_RAISED);
    FXGroupBox *fassocgroup=new FXGroupBox(fassocpack,NULL,FRAME_GROOVE|LAYOUT_FILL_X|LAYOUT_FILL_Y);
    FXVerticalFrame *contassoc=new FXVerticalFrame(fassocgroup,LAYOUT_FILL_X|LAYOUT_FILL_Y);//,FRAME_THICK|FRAME_RAISED);
    FXMatrix *matrix = new FXMatrix(contassoc,3,MATRIX_BY_COLUMNS|LAYOUT_SIDE_TOP|LAYOUT_FILL_X|LAYOUT_FILL_Y);
    fassoctab->disable();

    new FXLabel(matrix,_("Extension:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    ext = new FXTextField(matrix,10,NULL,0,FRAME_THICK|FRAME_SUNKEN|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    new FXLabel(matrix,"",NULL,0);

    new FXLabel(matrix,_("Description:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    descr = new FXTextField(matrix,10,NULL,0,FRAME_THICK|FRAME_SUNKEN|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    new FXLabel(matrix,"",NULL,0);

    new FXLabel(matrix,_("Open:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    open = new FXTextField(matrix,10,NULL,0,FRAME_THICK|FRAME_SUNKEN|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    new FXButton(matrix,_(" Select..."),NULL,this,ID_BROWSE_OPEN,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y);//,0,0,0,0,0,0);

    int is_ar = FALSE;

    FXString viewlbl=_("View:");
    FXString editlbl=_("Edit:");

	extension=file.rafter('.',1);
	if(extension=="gz" || extension=="GZ"  || extension=="tgz" || extension=="TGZ" ||
		extension=="tar"|| extension=="TAR"|| extension=="bz2" || extension=="BZ2" ||
		extension=="tbz2" || extension=="TBZ2" || extension=="zip" || extension=="ZIP" ||
		extension=="Z")
	{
		is_ar = TRUE; // archive
		viewlbl = _("Extract:");
	}
#if defined(linux)
	else if(extension=="rpm")
	{
		editlbl = _("Install/Upgrade:");
	}
#endif

    new FXLabel(matrix,viewlbl,NULL,JUSTIFY_LEFT|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    view = new FXTextField(matrix,10,NULL,0,FRAME_THICK|FRAME_SUNKEN|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    new FXButton(matrix,_(" Select..."),NULL,this,ID_BROWSE_VIEW,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y);//,0,0,0,0,0,0);

    if(!is_ar)
    {
        new FXLabel(matrix,editlbl,NULL,JUSTIFY_LEFT|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
        edit = new FXTextField(matrix,10,NULL,0,FRAME_THICK|FRAME_SUNKEN|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
        new FXButton(matrix,_(" Select..."),NULL,this,ID_BROWSE_EDIT,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y);//,0,0,0,0,0,0);
    }
    else
        edit = NULL;

    new FXLabel(matrix,_("Big Icon:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    bigic = new FXTextField(matrix,10,NULL,0,FRAME_THICK|FRAME_SUNKEN|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    new FXButton(matrix,_(" Select..."),NULL,this,ID_BIG_ICON,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y);//,0,0,0,0,0,0);

    new FXLabel(matrix,_("Mini Icon:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    miniic = new FXTextField(matrix,10,NULL,0,FRAME_THICK|FRAME_SUNKEN|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    new FXButton(matrix,_(" Select..."),NULL,this,ID_MINI_ICON,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y);//,0,0,0,0,0,0);

    // Label
    new FXLabel(generalframe,_("Name"),NULL,JUSTIFY_LEFT);

    input = new FXTextField(generalframe,20,NULL,0,LAYOUT_SIDE_TOP|/*LAYOUT_CENTER_Y|*/LAYOUT_CENTER_X|FRAME_SUNKEN|FRAME_THICK|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW|LAYOUT_FILL_X);

    // Complete filename (with path)
	strcpy(filename,(path+PATHSEPSTRING+file).text());
	parentdir=path;
	
    // Get file/link info
    if(lstatout(filename,&info)!=0)
        return ;

	// Obtain user name
	usrid=FXFile::owner(info.st_uid);

	// Obtain group name
	grpid=FXFile::group(info.st_gid);
    perm->user->setText(usrid);
    perm->grp->setText(grpid);
    oldgrp = grpid;
    oldusr = usrid;

	// If it is a link
    islink=S_ISLNK(info.st_mode);
	if(islink)
	{
		// If it is a broken link
		if (statout(filename,&info)!=0)
			type="Broken link to ";
		else
			type="Link to ";

		// Get the name of the linked file
		int nread;
		nread=readlink(filename,linkto,sizeof(linkto)-1);
		if(nread>0)
		{
			linkto[nread]='\0';
			strcpy(filename,linkto);
		}
	}
	else
		type="";

    orig_mode=(info.st_mode & (S_IRWXU|S_IRWXG|S_IRWXO));

    // Mod time
	mod=FXFile::time("%c",info.st_mtime);

    // Change time
	changed=FXFile::time("%c",info.st_ctime);

    // Accessed time
    accessed=FXFile::time("%c",info.st_atime);;

    // Size
    filesize=(unsigned long long)info.st_size;
    FXString fileassoc;

    // Is it a directory?
	isdirectory=S_ISDIR(info.st_mode);
    if(isdirectory)
    {
		
		// Directory path
		FXString dupath=FXFile::absolute(path,file);
		dupath=quote(dupath,FALSE);

#if defined(linux)
		FILE *mtab=setmntent("/etc/mtab","r");
        struct mntent *mnt;
        if(mtab)
        {
            while((mnt=getmntent(mtab)))
            {
                if(strcmp(mnt->mnt_type,MNTTYPE_IGNORE) && strcmp(mnt->mnt_type,MNTTYPE_SWAP) && strcmp(mnt->mnt_type,"proc"))
                {
                    if(!strcmp(mnt->mnt_dir,dupath.text()))
                    {
                        ismountpoint=TRUE;
                        sprintf(buf,_("Filesystem (%s)"),mnt->mnt_fsname);
                        type+=buf;
						strcpy(mnttype,mnt->mnt_type);
                    }
                }
            }
            endmntent(mtab);
        }
#endif
			// If it is a mount point
		if(ismountpoint)
		{
			sprintf(buf,"df --block-size=1 %s",filename);
			p=popen(buf,"r");
			fgets(buf,sizeof(buf),p);
			fgets(buf,sizeof(buf),p);
			strtok(buf," "); // should check if str!=NULL
			strtok(NULL," ");// get size
			strcpy(used,strtok(NULL," "));// get used
			strcpy(avail,strtok(NULL," "));// get available
			pclose(p);
		}
			
		// If it is a folder
		else
		{
            type+=_("Folder");
    		getApp()->beginWaitCursor();
			strcpy(buf,dupath.text());
			dsize=dirpath(buf);
			sprintf(size,"%llu",dsize);
    		getApp()->endWaitCursor();
		}
    }
	
    else if(S_ISCHR(info.st_mode))
        type+="Character Device";
    else if(S_ISBLK(info.st_mode))
        type+="Block Device";
    else if(S_ISFIFO(info.st_mode))
        type+="Named Pipe";
    else if(S_ISSOCK(info.st_mode))
        type+="Socket";
    
	// Regular file or link
	else
    {
		// Try to use association table	
		extension=FXFile::name(filename).rafter('.',1);
        if (extension != "")
			fileassoc=getApp()->reg().readStringEntry("FILETYPES",extension.text(),"");

		// If we have an association
        if(!fileassoc.empty())
        {
            FXString c;
            type+=fileassoc.section(';',1);
            ext->setText(extension);
            ext->disable();
            c=fileassoc.section(';',0);
            descr->setText(fileassoc.section(';',1));
            open->setText(c.section(',',0));
            view->setText(c.section(',',1));
            if(edit)
                edit->setText(c.section(',',2));
            bigic->setText(fileassoc.section(';',2));
            miniic->setText(fileassoc.section(';',3));

            fassoctab->enable();
        }
        else
        {
            ext->setText(extension);
            ext->disable();
            if(info.st_mode&(S_IXUSR|S_IXGRP|S_IXOTH))
                type+="Application";
            else
            {
                type+="Document";
                fassoctab->enable();
            }
        }
    }

	// Parent directory name not editable
	if (file=="..")
		input->setEditable(FALSE);

	// Root directory name not editable
    if (file=="" & path==ROOTDIR)
	{
		input->setText(ROOTDIR);
		input->setEditable(FALSE);
	}
	else		
		input->setText(file);
    
	input->setFocus();
    input->onCmdSelectAll(0,0,0);

    // Set permissions

    perm->ur->setCheck((orig_mode & S_IRUSR) ? TRUE : FALSE);
    perm->uw->setCheck((info.st_mode & S_IWUSR) ? TRUE : FALSE);
    perm->ux->setCheck((info.st_mode & S_IXUSR) ? TRUE : FALSE);

    perm->gr->setCheck((info.st_mode & S_IRGRP) ? TRUE : FALSE);
    perm->gw->setCheck((info.st_mode & S_IWGRP) ? TRUE : FALSE);
    perm->gx->setCheck((info.st_mode & S_IXGRP) ? TRUE : FALSE);

    perm->or_->setCheck((info.st_mode & S_IROTH) ? TRUE : FALSE);
    perm->ow->setCheck((info.st_mode & S_IWOTH) ? TRUE : FALSE);
    perm->ox->setCheck((info.st_mode & S_IXOTH) ? TRUE : FALSE);
    perm->set->setCheck();

	FXLabel *mtType=NULL, *mtUsed=NULL, *mtFree=NULL, *fileType=NULL, *fileSize=NULL, *fileChanged=NULL, *fileAccessed=NULL, *fileModified=NULL;

	// Properties are different for mount points
	if (ismountpoint)
	{
  		FXGroupBox *mtgroup=new FXGroupBox(generalframe,_("Mount point"),GROUPBOX_TITLE_LEFT|FRAME_GROOVE|LAYOUT_FILL_X);
  		FXMatrix *mtmatrix=new FXMatrix(mtgroup,2,MATRIX_BY_COLUMNS|LAYOUT_FILL_X|LAYOUT_FILL_Y);
  		new FXLabel(mtmatrix,_("Mount type:"),NULL,LAYOUT_LEFT);
  		fileType=new FXLabel(mtmatrix,NULL,NULL,LAYOUT_LEFT|LAYOUT_FILL_COLUMN);
  		new FXLabel(mtmatrix,_("Used:"),NULL,LAYOUT_LEFT);
  		mtUsed=new FXLabel(mtmatrix,NULL,NULL,LAYOUT_LEFT|LAYOUT_FILL_COLUMN);
  		new FXLabel(mtmatrix,_("Free:"),NULL,LAYOUT_LEFT);
  		mtFree=new FXLabel(mtmatrix,NULL,NULL,LAYOUT_LEFT|LAYOUT_FILL_COLUMN);
  		new FXLabel(mtmatrix,_("File system:"),NULL,LAYOUT_LEFT);
  		mtType=new FXLabel(mtmatrix,NULL,NULL,LAYOUT_LEFT|LAYOUT_FILL_COLUMN);
	}
	else
	{
  		FXGroupBox *attrgroup=new FXGroupBox(generalframe,_("Properties"),GROUPBOX_TITLE_LEFT|FRAME_GROOVE|LAYOUT_FILL_X);
  		FXMatrix *attrmatrix=new FXMatrix(attrgroup,2,MATRIX_BY_COLUMNS|LAYOUT_FILL_X|LAYOUT_FILL_Y);
  		new FXLabel(attrmatrix,_("Type:"),NULL,LAYOUT_LEFT);
  		fileType=new FXLabel(attrmatrix,NULL,NULL,LAYOUT_LEFT|LAYOUT_FILL_COLUMN);
  		new FXLabel(attrmatrix,_("Size:"),NULL,LAYOUT_LEFT);
  		fileSize=new FXLabel(attrmatrix,NULL,NULL,LAYOUT_LEFT|LAYOUT_FILL_COLUMN);

  		FXGroupBox *timegroup=new FXGroupBox(generalframe,_("File Time"),GROUPBOX_TITLE_LEFT|FRAME_GROOVE|LAYOUT_FILL_X);
  		FXMatrix *timematrix=new FXMatrix(timegroup,2,MATRIX_BY_COLUMNS|LAYOUT_FILL_X|LAYOUT_FILL_Y);
  		new FXLabel(timematrix,_("Last Modified:"),NULL,LAYOUT_LEFT);
  		fileModified=new FXLabel(timematrix,NULL,NULL,LAYOUT_LEFT|LAYOUT_FILL_COLUMN);
  		new FXLabel(timematrix,_("Last Changed:"),NULL,LAYOUT_LEFT);
  		fileChanged=new FXLabel(timematrix,NULL,NULL,LAYOUT_LEFT|LAYOUT_FILL_COLUMN);
  		new FXLabel(timematrix,_("Last Accessed:"),NULL,LAYOUT_LEFT);
  		fileAccessed=new FXLabel(timematrix,NULL,NULL,LAYOUT_LEFT|LAYOUT_FILL_COLUMN);
	}

    // File or mount type
	fileType->setText(type.text());

	// If directory
	if (isdirectory)
	{
		// if mount point
		if(ismountpoint)
		{
			::hSize(used,used);
			::hSize(avail,avail);
			mtType->setText(mnttype);
			mtUsed->setText(used);
			mtFree->setText(avail);
		}
		// if folder
		else
		{
        	fileModified->setText(mod);
        	fileChanged->setText(changed);
        	fileAccessed->setText(accessed);
			::hSize(size,size);
			fileSize->setText(size);
		}
	}
	// Regular file
    else
    {
        FXchar size[64], hsize[64];
        sprintf(size,"%llu",filesize);
        ::hSize(size,hsize);
        sprintf(size,"%s (%llu bytes)",hsize,filesize);        
		fileSize->setText(size);
        fileModified->setText(mod);
        fileChanged->setText(changed);
        fileAccessed->setText(accessed);
    }
    mode = orig_mode;
    perm->cmd=PropertiesBox::ID_SET;
    files = &file;
    source = file;
    num = 1;	
}

// Construct window for multiple files
PropertiesBox::PropertiesBox(FXWindow *win,FXString *file,int n): FXDialogBox(win,_("Properties"), DECOR_TITLE|DECOR_BORDER/*,0,0,350,200*/)
{
	struct stat info;
	FXString grpid, usrid;

    // Buttons
    FXHorizontalFrame *buttons=new FXHorizontalFrame(this,LAYOUT_SIDE_BOTTOM|LAYOUT_FILL_X,0,0,0,0,10,10,5,5);

    // Contents
    FXVerticalFrame *contents=new FXVerticalFrame(this,LAYOUT_SIDE_TOP|FRAME_NONE|LAYOUT_FILL_X|LAYOUT_FILL_Y|PACK_UNIFORM_WIDTH);

    // Accept
    FXButton *ok = new FXButton(buttons,_("&Accept"),NULL,this,PropertiesBox::ID_ACCEPT_MULT,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,20,20);
    ok->addHotKey(KEY_Return);

    // Cancel
    new FXButton(buttons,_("&Cancel"),NULL,this,FXDialogBox::ID_CANCEL,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,20,20);

    // Switcher
    FXTabBook *tabbook = new FXTabBook(contents,NULL,0,LAYOUT_FILL_X|LAYOUT_FILL_Y|LAYOUT_RIGHT);


    // Here, access Permissions is the only item shown
    new FXTabItem(tabbook,_("&Permissions"),NULL);
    perm = new PermFrame(tabbook,this);

    // Get file/link info of the first file of the list
	// This is used as a guess for the username, group and permissions of the whole list
    if(lstatout(file[0].text(),&info)!=0)
        return ;

	// Obtain user name
	usrid=FXFile::owner(info.st_uid);

	// Obtain group name
	grpid=FXFile::group(info.st_gid);

    orig_mode=(info.st_mode & (S_IRWXU|S_IRWXG|S_IRWXO));

    perm->ur->setCheck((orig_mode & S_IRUSR) ? TRUE : FALSE);
    perm->uw->setCheck((info.st_mode & S_IWUSR) ? TRUE : FALSE);
    perm->ux->setCheck((info.st_mode & S_IXUSR) ? TRUE : FALSE);

    perm->gr->setCheck((info.st_mode & S_IRGRP) ? TRUE : FALSE);
    perm->gw->setCheck((info.st_mode & S_IWGRP) ? TRUE : FALSE);
    perm->gx->setCheck((info.st_mode & S_IXGRP) ? TRUE : FALSE);

    perm->or_->setCheck((info.st_mode & S_IROTH) ? TRUE : FALSE);
    perm->ow->setCheck((info.st_mode & S_IWOTH) ? TRUE : FALSE);
    perm->ox->setCheck((info.st_mode & S_IXOTH) ? TRUE : FALSE);
    perm->add->setCheck();

    perm->user->setText(usrid);
    perm->grp->setText(grpid);

    mode = orig_mode;
    perm->cmd = PropertiesBox::ID_ADD;
    files = file;
    source = "";
    num = n;
}


long PropertiesBox::onCmdAcceptMult(FXObject* o,FXSelector s,void* p)
{
    int rc=0,i;
	File *f=NULL;
	char file[MAXPATHLEN];

    FXDialogBox::onCmdAccept(o,s,p);
    if(mode)
    {
		f = new File(getApp(),_("File permissions"),CHMOD);
		f->create();
        for(i=0;i<num;i++)
        {
            struct stat info;
            mode_t m;
            if(lstatout(files[i].text(),&info)!=0)
                continue ;

            switch(perm->cmd)
            {
            	case PropertiesBox::ID_ADD:
                	m=info.st_mode|mode;
                	break;
            	case PropertiesBox::ID_CLEAR:
                	m=info.st_mode & ~mode;
                	break;
            	case PropertiesBox::ID_SET:
                	m=mode;
                	break;
            	default :
                	return 1;
            }
            
			if (files[i]!="..")
			{
				rc=f->chmod(strdup(files[i].text()),file,m,perm->rec->getCheck(),perm->dironly->getCheck());
				
				// If action is cancelled in progress dialog
				if (f->isCancelled)
				{
					f->hide();
					MessageBox::error(this,MBOX_OK,_("Error"),_("Change file(s) permissions cancelled!"));
					delete f;
					return 1;
				}
        		
				// Handle chmod errors
				if(rc)
				{
            		MessageBox::error(this,MBOX_OK,_("Error"),_("Chmod failed in %s : %s"),file,strerror(errno));
					break;
				}
			}
        }
		delete f;
    }

    rc = 0;
    // chown if needed
    if(!perm->grp->getText().empty() || !perm->user->getText().empty())
    {
        f = new File(getApp(),_("File owner"),CHOWN);
        f->create();
        for(i=0;i<num;i++)
        {
            struct stat info;
            if(lstatout(files[i].text(),&info)!=0)
                continue ;

            uid_t uid = 32768;
            gid_t gid = 32768;
            struct passwd* pwde;
            while( (pwde = getpwent()) )
                if(perm->user->getText() == pwde->pw_name)
                    uid = pwde->pw_uid;
            endpwent();

            struct group* grpe;
            while( (grpe = getgrent()) )
                if(perm->grp->getText() == grpe->gr_name)
                    gid = grpe->gr_gid;
            endgrent();

			if (files[i]!="..")
            	rc |= f->chown(strdup(files[i].text()),file,uid,gid,perm->rec->getCheck(),perm->dironly->getCheck());

			// If action is cancelled in progress dialog
			if (f->isCancelled)
			{
				f->hide();
				MessageBox::error(this,MBOX_OK,_("Error"),_("Change owner cancelled!"));
				delete f;
				return 1;
			}
			
			// Handle chmod errors
            if(rc)
            {
                MessageBox::error(this,MBOX_OK,_("Error"),_("Chown failed in %s : %s"),file,strerror(errno));
                break;
            }
        }
		delete f;
    }
    return 1;
}


long PropertiesBox::onCmdAcceptSingle(FXObject* o,FXSelector s,void* p)
{
	char **str=NULL;
    int rc=0;
	File *f=NULL;
	char file[MAXPATHLEN];
    FXString oldfileassoc,fileassoc,op,v,e;

	// Don't proceed any task on the parent directory
	if (!strcmp(source.text(),".."))
		return 1;

	op = open->getText();
    v = view->getText();
    if(!v.empty())
        v = "," + v;
    if(edit)
    {
        e = edit->getText();
        if(!e.empty())
            e = "," + e;
    }

    fileassoc = ext->getText();
    fileassoc += "=";
    fileassoc += op + v + e + ";";
    fileassoc += descr->getText() + ";";
    fileassoc += bigic->getText() + ";" + miniic->getText() + ";;";

	if (ext->getText()!="")
	{
		oldfileassoc=getApp()->reg().readStringEntry("FILETYPES",ext->getText().text(),"");
    	if(oldfileassoc == "" || fileassoc.section('=',1) != oldfileassoc)
    	{
			FXString command=fileassoc.section('=',1);
        	getApp()->reg().writeStringEntry("FILETYPES",ext->getText().text(),command.text());

			// Handle file association
			str=new char*[2];
			str[0]=new char[strlen(ext->getText().text())+1];
			str[1]=new char[strlen(command.text())+1];
			strcpy(str[0],ext->getText().text());
			strcpy(str[1],command.text());
			mainWindow->handle(this,FXSEL(SEL_COMMAND,XFileExplorer::ID_FILE_ASSOC),str);
    	}
	}
    // Rename file
    FXString target=input->getText();
	if(source != target)
	{
		// Source is not writable
		if (!FXFile::isWritable(source))
        	MessageBox::error(this,MBOX_OK,_("Error"),_("Can't write to %s : Permission denied"),source.text());
		
		// Target parent directory doesn't exist or is not writable
		FXString targetparentdir=FXFile::directory(target);
		if (targetparentdir=="")
			targetparentdir=parentdir;
		if (!exists(dequote(targetparentdir)))
        {
			MessageBox::error(this,MBOX_OK,_("Error"),_("Folder %s doesn't exist"),targetparentdir.text());
        	return 0;
		}
		if (!FXFile::isWritable(targetparentdir))
		{
         	MessageBox::error(this,MBOX_OK,_("Error"),_("Can't write to %s : Permission denied"),targetparentdir.text());
        	return 0;
		}

		// Rename file or directory
		else
		{
			File *f;
			f=new File(getApp(),_("File rename"),RENAME);
       		f->create();
			f->rename(source,input->getText());
			delete f;
		}
	}

    // Change perm
    FXDialogBox::onCmdAccept(o,s,p);
	if (input->getText()==".." || input->getText()==".")
	{
		input->setText(source);
		return 1;
	}
    if(!(orig_mode == mode && perm->cmd == PropertiesBox::ID_SET && !perm->rec->getCheck()))//no change
    {
        f = new File(getApp(),_("File permissions"),CHMOD);
        f->create();

        struct stat info;
        mode_t m;
        if(lstatout(input->getText().text(),&info)!=0)
            return 1;

        switch(perm->cmd)
        {
        	case PropertiesBox::ID_ADD:
            	m=info.st_mode|mode;
            	break;
        	case PropertiesBox::ID_CLEAR:
            	m=info.st_mode & ~mode;
            	break;
        	case PropertiesBox::ID_SET:
            	m=mode;
            	break;
        	default :
            	return 1;
        }
		// Perform chmod on the selected file or directory
		rc=f->chmod(strdup(input->getText().text()),file,m,perm->rec->getCheck(),perm->dironly->getCheck());

		// If action is cancelled in progress dialog
		if (f->isCancelled)
		{
			f->hide();
			MessageBox::error(this,MBOX_OK,_("Error"),_("Change file permissions cancelled!"));
			delete f;
			return 1;
		}
		delete f;
        
		// Handle chmod errors
		if(rc)
            MessageBox::error(this,MBOX_OK,_("Error"),_("Chmod in %s failed: %s"), file, strerror(errno));
    }

    // chown if needed
    rc = 0;
    if(oldgrp != perm->grp->getText() || oldusr != perm->user->getText())
    {
        f = new File(getApp(),_("File owner"),CHOWN);
        f->create();

        uid_t uid = 32768;
        gid_t gid = 32768;
        struct passwd* pwde;
        while( (pwde = getpwent()) )
            if(perm->user->getText() == pwde->pw_name)
                uid = pwde->pw_uid;
        endpwent();

        struct group* grpe;
        while( (grpe = getgrent()) )
            if(perm->grp->getText() == grpe->gr_name)
                gid = grpe->gr_gid;
        endgrent();

        // Perform chown on the selected file or directory
		rc = f->chown(strdup(input->getText().text()),file,uid,gid,perm->rec->getCheck(),perm->dironly->getCheck());

		// If action is cancelled in progress dialog
		if (f->isCancelled)
		{
			f->hide();
			MessageBox::error(this,MBOX_OK,_("Error"),_("Change owner cancelled!"));
			delete f;
			return 1;
		}
		delete f;
		
		// Handle chown errors
        if(rc)
            MessageBox::error(this,MBOX_OK,_("Error"),_("Chown failed in %s: %s"),file,strerror(errno));
    }
    return 1;
}

long PropertiesBox::onCmdCommand(FXObject* o,FXSelector s,void* p)
{
    perm->cmd = FXSELID(s);
    return 1;
}


long PropertiesBox::onCmdCheck(FXObject* o,FXSelector s,void* p)
{
    FXint xmode=0;
	switch (FXSELID(s))
	{
		case PropertiesBox::ID_RUSR:
			xmode=S_IRUSR;
			break;
		case PropertiesBox::ID_WUSR:
			xmode=S_IWUSR;
			break;
		case PropertiesBox::ID_XUSR:
			xmode=S_IXUSR;
			break;
		case PropertiesBox::ID_RGRP:
			xmode=S_IRGRP;
			break;
		case PropertiesBox::ID_WGRP:
			xmode=S_IWGRP;
			break;
		case PropertiesBox::ID_XGRP:
			xmode=S_IXGRP;
			break;
		case PropertiesBox::ID_ROTH:
			xmode=S_IROTH;
			break;
		case PropertiesBox::ID_WOTH:
			xmode=S_IWOTH;
			break;
		case PropertiesBox::ID_XOTH:
			xmode=S_IXOTH;
			break;
	}
	FXCheckButton* ch = (FXCheckButton*)o;    
	if(!ch->getCheck())
       mode &= ~xmode;
    else
        mode |= xmode;
    return 1;

}


long PropertiesBox::onCmdBrowse(FXObject* o,FXSelector s,void* p)
{
    FileDialog browseProgram(this,_("Select Program"));
    const FXchar *patterns[]=
        {
            _("All files"),     "*",NULL
        };
	browseProgram.setPatternList(patterns);
    if(browseProgram.execute())
    {
        char *path=strdup(browseProgram.getFilename().text());
        switch(FXSELID(s))
        {
        	case ID_BROWSE_OPEN:
            	open->setText(FXFile::name(path).text());
            	break;
        	case ID_BROWSE_VIEW:
            	view->setText(FXFile::name(path).text());
            	break;
        	case ID_BROWSE_EDIT:
            	if(edit)
                	edit->setText(FXFile::name(path).text());
            	break;
        }
    }
    return 1;
}


long PropertiesBox::onCmdBrowseIcon(FXObject* o,FXSelector s,void* p)
{
    FXString icon;
    if(FXSELID(s)==ID_BIG_ICON)
        icon = bigic->getText();
    else
        icon= miniic->getText();

	FXString iconpath=getApp()->reg().readStringEntry("SETTINGS","iconpath","/usr/lib/foxicons");
    const FXchar *patterns[]=
        {
            _("PNG Images"),     "*.png",
			_("GIF Images"),     "*.gif",
            _("BMP Images"),     "*.bmp",NULL
        };
    FileDialog browseIcon(this,_("Select Icon"));
    browseIcon.setFilename(iconpath+PATHSEPSTRING+icon);
    browseIcon.setPatternList(patterns);
    if(browseIcon.execute())
    {
        char *path=strdup(browseIcon.getFilename().text());
        if(!exists(dequote(path)))
            return 0;
        if(FXSELID(s)==ID_BIG_ICON)
            bigic->setText(path);
        else
            miniic->setText(path);
    }
    return 1;
}

long PropertiesBox::onUpdDironly(FXObject* o,FXSelector s,void* p)
{
    if(perm->rec->getCheck())
        perm->dironly->enable();
    else
        perm->dironly->disable();
    return 1;

}

long PropertiesBox::onCmdKeyPress(FXObject* sender,FXSelector sel,void* ptr)
{
    FXEvent* event=(FXEvent*)ptr;
    switch(event->code)
    {
    case KEY_Escape:
        handle(this,FXSEL(SEL_COMMAND,ID_CANCEL),NULL);
		return 1;
    case KEY_KP_Enter:
    case KEY_Return:
        handle(this,FXSEL(SEL_COMMAND,ID_ACCEPT_SINGLE),NULL);
		return 1;
    default:
        FXTopWindow::onKeyPress(sender,sel,ptr);
		return 1;
    }
	return 0;
}
