#!/usr/bin/env python
#
#   XenMan   -  Copyright (c) 2007 Jd, Hap Hazard & Yves Perrenoud
#   ======
#
# XenMan is a Xen management tool with a GTK based graphical interface
# that allows for performing the standard set of domain operations
# (start, stop, pause, kill, shutdown, reboot, snapshot, etc...). It
# also attempts to simplify certain aspects such as the creation of
# domains, as well as making the consoles available directly within the
# tool's user interface.
#
#
# This software is subject to the GNU General Public License (GPL)
# and for details, please consult it at:
#
#    http://www.fsf.org/licensing/licenses/gpl.txt
#
import types
import os, sys
import gtk, gtk.glade, gobject

import constants
from ManagedNode import ManagedNode, NodeException
from XenNode import XenNode, DomConfig
from utils import XMConfig, is_host_remote, guess_value, PyConfig
import urllib, urlparse
import xml.parsers.expat
import time
import traceback

from dialogs import main_context, wtree, showmsg, confirmation,file_selection, ProvStatusDlg, show_wait_cursor, hide_wait_cursor


class DomSettings:
    """Class that handles everything related to the Dom Settings dialog"""

    class DiskEdit:
        """Class that handles everything related to the disk edit area of
        the settings dialog"""

        diskmapping = ["file", "phy"]
        modemapping = ["w", "r"]
        colmap = ["Type", "Device Path", "Guest Device", "Mode"]
        guestdevs = ["hda","hdb","xvda", "xvdb", "xvdc"]

        def __init__(self, wtree, filesel, parentwin):
            self.dialog = wtree.get_widget("DiskEditDialog")
            self.addbutton = wtree.get_widget("DiskAddButton")
            self.propbutton = wtree.get_widget("DiskPropButton")
            self.rembutton = wtree.get_widget("DiskRemoveButton")
            self.disktype = wtree.get_widget("DiskTypeCombo")
            self.diskdev = wtree.get_widget("DiskDeviceEntry")
            self.diskdevfsel = wtree.get_widget("DiskDeviceButton")
            self.guestdev = wtree.get_widget("GuestDevCombo")
            self.mode = wtree.get_widget("ModeCombo")
            self.inst_guestdevs = [] + self.guestdevs

            self.diskview = wtree.get_widget("DiskTreeView")
            textrenderer = gtk.CellRendererText()

            self.dialog.set_transient_for(parentwin)

            for dev in self.inst_guestdevs:
                self.guestdev.append_text(dev)

            pos = 0
            for colname in self.colmap:
                column = gtk.TreeViewColumn(colname, textrenderer, text=pos)
                self.diskview.append_column(column)
                pos += 1

            wtree.signal_autoconnect(
                {"on_DiskAddButton_clicked": self.__showSettings,
                 "on_DiskPropButton_clicked": (self.__showSettings, True),
                 "on_DiskRemoveButton_clicked": self.__removeClicked,
                 "on_DiskDeviceButton_clicked": (filesel, self.diskdev,
                                                 "open", self.dialog),
                 })

        def __removeClicked(self, widget):
            """Handles the clicking of the remove button in the disk section
            of the settings dialog"""

            model, iter = self.diskview.get_selection().get_selected()

            if iter:
                model.remove(iter)

        def showtree(self, dom_config=None):
            """Populates the disk tree view with all disks for this dom"""

            self.diskmodel = gtk.ListStore(*[gobject.TYPE_STRING]*4)
            self.diskview.set_model(self.diskmodel)

            if dom_config:
                for disk in dom_config.getDisks():
                    if disk.type == "file":
                        dtype = disk.type
                    else:
                        dtype = "phy"

                    self.diskmodel.append([dtype, disk.filename,
                                           disk.device, disk.mode])

        def __showSettings(self, widget, edit=False):
            """Show the dialog and optionally populate with the selected
            disk entry"""

            model, iter = self.diskview.get_selection().get_selected()

            if edit and iter:
                for combo, mapping, entry in (
                    (self.disktype, self.diskmapping, "Type"),
                    (self.mode, self.modemapping, "Mode")):
                    combo.set_active(mapping.index(
                        model.get_value(iter, self.colmap.index(entry))))

                self.diskdev.set_text(
                    model.get_value(iter, self.colmap.index("Device Path")))

                guestdev =  model.get_value(iter,
                                            self.colmap.index("Guest Device"))
                if guestdev not in self.inst_guestdevs:
                    self.inst_guestdevs.insert(0,guestdev)
                    self.guestdev.prepend_text(guestdev)
                    self.guestdev.set_active(0)
                else:
                    self.guestdev.set_active(self.inst_guestdevs.index(guestdev))
            else:
                self.diskdev.set_text('')

                # The following three lines are a hack to compensate for the
                # fact that the following call doesn't work as documented
                ## self.guestdev.set_active(-1)
                self.guestdev.prepend_text('')
                self.guestdev.set_active(0)
                self.guestdev.remove_text(0)

                for combo in (self.disktype, self.mode):
                    combo.set_active(0)

            res = self.dialog.run() == gtk.RESPONSE_OK

            if not res:
                self.dialog.hide()
                return

            if edit and iter:
                model.remove(iter)

            model.append(
                [self.diskmapping[self.disktype.get_active()],
                 self.diskdev.get_text(),
                 self.guestdev.get_active_text(),
                 self.modemapping[self.mode.get_active()],
                 ])

            self.dialog.hide()

        def getDiskEntries(self):
            """Returns a list of DiskEntry instances for each row of the
            ListStore"""

            delist = []
            iter = self.diskmodel.get_iter_first()
            while iter:
                diskparms = []
                for row in enumerate(self.colmap):
                    diskparms.append(self.diskmodel.get_value(iter, row[0]))

                #_RESTRUCT
                delist.append(DomConfig.DiskEntry(diskparms))
                iter = self.diskmodel.iter_next(iter)

            return delist

    combomapping = ["destroy", "restart", "preserve", "rename-restart"]

    textfields = ("name", "kernel", "ramdisk", "root", "extra", "bootloader")
    valuefields = ("memory", "vcpus")
    combofields = ("on_shutdown", "on_reboot", "on_crash")

    minmem = 16
    defaultmem = 256
    ignoreTypeChange = False

    ############# DomSettings constructor ###############################
    def __init__(self, wtree, image_store):
        self.dialog = wtree.get_widget("SettingsDialog")

        self.main_notebook = wtree.get_widget("SettingsMain")
        self.vm_notebook = wtree.get_widget("SettingsVM")
        self.prov_page_label = wtree.get_widget("prov_page_label")
        self.prov_page = self.main_notebook.get_nth_page(1)
        
        self.kernel = wtree.get_widget("KernelEntry")
        self.ramdisk = wtree.get_widget("RamDiskEntry")
        self.use_bootloader = wtree.get_widget("UseBootloaderCheckButton")
        self.bootloader = wtree.get_widget("BootloaderEntry")
        self.rootdev = wtree.get_widget("RootDevEntry")
        self.kernargs = wtree.get_widget("KernArgsEntry")
        self.diskframe = wtree.get_widget("DiskFrame")
        self.usablecpu = wtree.get_widget("UsableCPUEntry")
        self.vcpu = wtree.get_widget("VCPUSpin")
        self.power = wtree.get_widget("PowerCombo")
        self.reboot = wtree.get_widget("RebootCombo")
        self.crash = wtree.get_widget("CrashCombo")
        self.networkframe = wtree.get_widget("NetworkFrame")
        self.name = wtree.get_widget("NameEntry")
        self.kernelsel = wtree.get_widget("KernelButton")
        self.ramdisksel = wtree.get_widget("RamDiskButton")
        self.memory = wtree.get_widget("MemoryScale")
        self.chg_settings_label = wtree.get_widget("chg_settings_label")
        self.inmemory = wtree.get_widget("InMemory")
        self.ondisk = wtree.get_widget("OnDisk")
        self.filename = wtree.get_widget("FileNameEntry")
        self.filenamesel = wtree.get_widget("FileNameButton")
        self.okbutton = wtree.get_widget("settings_okbutton")
        self.image_name_combo = wtree.get_widget("ImageName")
        self.update_image = wtree.get_widget("update_image_check_button")
        self.wtree = wtree
        self.dom_name_label = wtree.get_widget("dom_name_label")
        self.dom_config_name_label = wtree.get_widget("dom_config_name_label")
        
        self.conf_dir = ''
        self.managed_node = None
        self.blank = True


        # add two tree views for misc properties and provisioning props
        self.misc_props_view = wtree.get_widget("MiscXenPropsTable")
        self.image_props_view = wtree.get_widget("ProvisioningPropsTable")

        self.misc_property_add = wtree.get_widget("misc_props_add")
        self.misc_property_remove = wtree.get_widget("misc_props_remove")

        self.img_property_add = wtree.get_widget("image_props_add")
        self.img_property_remove = wtree.get_widget("image_props_remove")
        self.last_dom_name = ''

        self.misc_props_model = gtk.ListStore(gobject.TYPE_STRING,
                                              gobject.TYPE_STRING,
                                              gobject.TYPE_BOOLEAN,
                                              gobject.TYPE_BOOLEAN,
                                              gobject.TYPE_BOOLEAN)
        self.image_props_model = gtk.ListStore(gobject.TYPE_STRING,
                                              gobject.TYPE_STRING,
                                              gobject.TYPE_BOOLEAN,
                                              gobject.TYPE_BOOLEAN,
                                              gobject.TYPE_BOOLEAN)

        self.misc_props_view.set_model(self.misc_props_model)
        self.image_props_view.set_model(self.image_props_model)


        #_RESTRUCT

        self.image_store = image_store

        self.diskedit = self.DiskEdit(wtree, self.__filesel, self.dialog)

        self.diskmapping = {"name": self.name, "memory": self.memory,
                            "kernel": self.kernel, "ramdisk": self.ramdisk,
                            "root": self.rootdev, "cpus": self.usablecpu,
                            "extra": self.kernargs, "vcpus": self.vcpu,
                            "on_shutdown": self.power,
                            "on_reboot": self.reboot,
                            "on_crash": self.crash,
                            "bootloader": self.bootloader,}
        
        # stuff already displayed in UI hence should be
        # excluded in NV pair format
        self.ui_displayed = self.diskmapping.keys()
        self.ui_displayed.append("disk")
        
        wtree.signal_autoconnect(
            {"on_InMemory_clicked": self.__typeChange,
             "on_KernelButton_clicked": (self.__filesel, self.kernel),
             "on_RamDiskButton_clicked": (self.__filesel, self.ramdisk),
             "on_FileNameButton_clicked": (self.__filesel, self.filename,
                                           "save"),
             "on_UseBootloaderCheckButton_toggled": self.__boot_toggled,
             "on_ImageName_changed": self.__image_changed,
             "on_misc_props_add_clicked": (self.__on_props_add,
                                           self.misc_props_model,
                                           self.misc_props_view),
             "on_misc_props_remove_clicked": (self.__on_props_remove,
                                              self.misc_props_view),
             "on_image_props_add_clicked": (self.__on_props_add,
                                           self.image_props_model,
                                            self.image_props_view),
             "on_image_props_remove_clicked": (self.__on_props_remove,
                                              self.image_props_view),
             "on_NameEntry_changed" : self.__on_domname_changed,
             "on_NameEntry_focus_out_event" : self.__on_domname_focus_out
             
             })

        # populate the image list
        self.availableImages = self.image_store.list()
        #self.image_name_combo.append_text("")
        for image in self.availableImages:
            self.image_name_combo.append_text(image)


        self.__misc_props_clear_model()
        self.__misc_props_prepare_view()

        self.__image_props_clear_model()
        self.__image_props_prepare_view()


    def __on_domname_changed(self, widget):
        if self.blank:
            t = os.path.join(self.conf_dir, widget.get_text())
            self.filename.set_text(t)

    def __on_domname_focus_out(self, widget, event):
        if self.blank:
            dom_name = self.name.get_text()
            if self.last_dom_name != dom_name: 
                self.__image_changed(self.image_name_combo)
            self.last_dom_name = dom_name

    def __misc_props_prepare_view(self):
        """ Prepare the view """
        treeview = self.misc_props_view
        treeview.set_rules_hint(True)
        #treeview.set_grid_lines(gtk.TREE_VIEW_GRID_LINES_BOTH)
        #treeview.set_enable_grid_lines(true)
        model = treeview.get_model()

        # Name
        renderer = gtk.CellRendererText()
        renderer.connect("edited",
                         self.__on_props_cell_edited,
                         model)
        renderer.set_data("column", 0)
        renderer.set_property("weight",700)
        renderer.set_property("weight-set",True)
        column = gtk.TreeViewColumn("Attribute",
                                    renderer,
                                    text=0, editable = 2, foreground_set=4)
        treeview.append_column(column)



        # Value
        renderer = gtk.CellRendererText()
        renderer.connect("edited",
                         self.__on_props_cell_edited,
                         model)
        renderer.set_data("column", 1)
        renderer.set_property("foreground","gray")
        column = gtk.TreeViewColumn("Value",
                                    renderer,
                                    text=1, editable=3, foreground_set=4)
                                    
        treeview.append_column(column)

    def __misc_props_clear_model(self):
        model = self.misc_props_model
        model.clear()


    def __misc_props_prepare_model(self):
        model = self.misc_props_model
        self.__misc_props_clear_model()
        display_list = []

        if self.domconfig is not None:
            if not self.image_edit_mode:
                customizable = self.domconfig.get_customizable_options()
            else:
                customizable = self.domconfig.keys()
            for key in self.domconfig:
                if key not in self.ui_displayed and \
                   key not in self.domconfig.get_computed_options() and \
                   (self.image_edit_mode or key != self.domconfig.CUSTOMIZABLE_OPTIONS):
                    display_list.append(key)
            display_list.sort()
            editable = self.ondisk.get_active() and key in customizable
            for key in display_list:
                iter = model.append()
                model.set(iter,
                          0, key,
                          1, self.domconfig[key],
                          2, False,
                          3, editable,
                          4, not editable)

    def __on_props_cell_edited(self,cell,path_string,new_text, model):
        iter = model.get_iter_from_string(path_string)
        path = model.get_path(iter)[0]
        column = cell.get_data("column")
        model.set(iter, column, new_text)
        

    def __on_props_add(self, widget, model, treeview):
        iter = model.append()
        model.set(iter,
                  0, "",
                  1, "",
                  2, True,
                  3, True,
                  4, False)
        # set the higlight to newly added row.
        path = model.get_path(iter)
        treeview.set_cursor(path,treeview.get_column(0),True)

    def __on_props_remove(self, widget, view):
        selection = view.get_selection()
        if selection:
            model, iter = selection.get_selected()
            if iter:
                path = model.get_path(iter)[0]
                model.remove(iter)

    def __update_config_props(self, config, model, excluded = []):
        # take misc model remove deleted props
        delete_props = []
        if config is None:
            return
        for prop in config:
            if prop not in excluded:
                # check if it exists in the misc model
                found = False
                iter = model.get_iter_first()
                while iter:
                    key = model.get_value(iter, 0)
                    if key == prop:
                        found = True
                        break
                    iter = model.iter_next(iter)
                    
                if not found:
                    delete_props.append(prop)

        for prop in delete_props:
            del config.options[prop]
                    
        # now copy update values from model to the config.
        iter = model.get_iter_first()
        while iter:
            key = model.get_value(iter, 0)
            value = model.get_value(iter,1)
            # We got the string representation, guess and make it proper value
            key = key.strip()
            value = value.strip()
            value = guess_value(value)
            if key is not None and key is not '':
                config[key] = value
            iter = model.iter_next(iter)

        #config.dump()

    def __image_props_prepare_view(self):
        """ Prepare the view """
        treeview = self.image_props_view
        treeview.set_rules_hint(True)
        #treeview.set_grid_lines(gtk.TREE_VIEW_GRID_LINES_BOTH)
        #treeview.set_enable_grid_lines(true)
        model = treeview.get_model()

        # Name
        renderer = gtk.CellRendererText()
        renderer.connect("edited",
                         self.__on_props_cell_edited,
                         model)
        renderer.set_data("column", 0)
        renderer.set_property("foreground","gray")
        renderer.set_property("weight",700)
        renderer.set_property("weight-set",True)
        column = gtk.TreeViewColumn("Attribute",
                                    renderer,
                                    text=0, editable = 2, foreground_set=4)
        treeview.append_column(column)

        # Value
        renderer = gtk.CellRendererText()
        renderer.connect("edited",
                         self.__on_props_cell_edited,
                         model)
        renderer.set_data("column", 1)
        renderer.set_property("foreground","gray")
        column = gtk.TreeViewColumn("Value",
                                    renderer,
                                    text=1, editable=3, foreground_set=4)
                                    
        treeview.append_column(column)



    def __image_props_clear_model(self):
        model = self.image_props_model
        model.clear()

    def __image_props_prepare_model(self):
        model = self.image_props_model
        self.__image_props_clear_model()
        display_list = []
        if self.image_config is not None:
            if not self.image_edit_mode:
                customizable = self.image_config.get_customizable_options()
            else:
                customizable = self.image_config.keys()
                
            for key in self.image_config:
                if  key not in self.image_config.get_computed_options() and \
                       (self.image_edit_mode or key != self.image_config.CUSTOMIZABLE_OPTIONS):
                    display_list.append(key)
                    
            display_list.sort()
            for key in display_list:
                iter = model.append()
                editable = self.ondisk.get_active() and key in customizable
                model.set(iter,
                          0, key,
                          1, self.image_config[key],
                          2, False,
                          3, editable,
                          4, not editable)
                    

    def __image_changed(self, widget):
        image_name = self.image_name_combo.get_active_text()
        if self.blank:
            if self.managed_node is None:
                return
            if image_name is not None and image_name.strip() != '':
                # get xen conf for the selected image
                template_file = self.image_store.get_vm_template(image_name)
                local_node = main_context["local_node"]
                self.domconfig = DomConfig(local_node, template_file)
                self.domconfig.set_filename(self.filename.get_text())
                self.domconfig["name"] = self.name.get_text()
                # get prepare template map
                template_map = {}
                template_map["IMAGE_NAME"] = image_name
                if self.name.get_text() is not None and \
                   self.name.get_text() != '':
                    template_map["VM_NAME"] = self.name.get_text()
                    
                # get the image conf related properties
                image_conf_file = self.image_store.get_image_conf(image_name)
                self.image_config = PyConfig(local_node, image_conf_file)
                if not self.image_edit_mode:
                    self.image_config.instantiate_config(template_map)

                # instantiate domconfig with image config
                if not self.image_edit_mode:
                    self.domconfig.instantiate_config(template_map)
                    if self.image_config is not None:
                        self.domconfig.instantiate_config(self.image_config)
                
                self.__showDisk()
            else:
                self.__showBlank()
            

    def __boot_toggled(self, widget):
        use_boot = widget.get_active()
        self.kernel.set_sensitive(not use_boot)
        self.ramdisk.set_sensitive(not use_boot)
        self.bootloader.set_sensitive(use_boot)
        
    

    def __typeChange(self, widget):
        """Handles the radio button that allows one to select what kind
        of settings (either memory resident or on disk) one wishes to edit"""

        if self.ignoreTypeChange:
            return

        if self.inmemory.get_active():
            self.type = "memory"
            self.__showMem()
        elif self.ondisk.get_active():
            self.type = "disk"
            self.__showDisk()
        else:
            raise TypeError

    def __noOpActive(self, widget, state):
        """Change the state of the settings type widgets without performing
        any actions as a result"""

        self.ignoreTypeChange = True
        widget.set_active(state)
        self.ignoreTypeChange = False

    def __filesel(self, widget, field,
                  fstype="open", parentwin=None):
        """Handles the pressing of the file selection buttons"""

        if not parentwin:
            parentwin = self.dialog

        filename = field.get_text()

        init_folder = os.path.dirname(filename)
        if fstype == "open":
            init_file = None
        else:
            init_file = filename

        result, filename = file_selection(self.managed_node,
                                          "Select file", fstype, init_folder,
                                          init_file, parentwin)

        if result:
            field.set_text(filename)

    def __dom0WidgetsOn(self, state):
        """sets the state of the widgets that are immutable only for dom0"""

        for widget in (self.diskframe, self.usablecpu, self.vcpu, #self.memory,
                       self.networkframe):
            widget.set_sensitive(state)
        

    def __diskWidgetsOn(self, state):
        """sets the state of the widgets that only apply to on disk settings"""

        for widget in (self.kernel, self.ramdisk, self.rootdev, self.kernargs,
                       self.kernelsel, self.ramdisksel,
                       self.name,
                       self.usablecpu,
                       self.power, self.reboot, self.crash,
                       self.filename, self.filenamesel, self.diskframe,
                       self.bootloader, self.use_bootloader,
                       self.misc_property_add, self.misc_property_remove,
                       self.img_property_add, self.img_property_remove):
            widget.set_sensitive(state)

            if self.blank:
                self.image_name_combo.set_sensitive(self.image_name is None)
            else:
                self.image_name_combo.set_sensitive(False)

    def __showMem(self):
        """Displays the current selection's memory resident settings"""

        self.__dom0WidgetsOn(self.dom.id)
        self.__diskWidgetsOn(False)

        self.memory.set_sensitive(self.dom.is_resident)
        if self.dom.get_config() and self.dom.get_config().filename:
            self.filename.set_text(self.dom.get_config().filename)
        else:
            self.filename.set_text("")

        for widget, parm in ((self.name, "name"),
                             (self.kernel, "kernel"),
                             (self.ramdisk, "ramdisk"),
                             (self.usablecpu, "vcpu_avail"),
                             (self.bootloader, "bootloader"),
                             ):
            value = self.dom[parm]
            if value:
                widget.set_text(value)
            else:
                widget.set_text('')

        if self.dom["image_name"] is not None:
            self.image_name_combo.set_active_text(dom["image_name"])
       

        curmem = int(self.dom["memory"])
        freemem = int(self.managed_node["free_memory"])
        self.memory.set_range(self.minmem, curmem+freemem)

        for widget, parm in ((self.memory, "memory"), (self.vcpu, "vcpus")):
            widget.set_value(int(self.dom[parm]))

        for widget, parm in ((self.power, "on_poweroff"),
                             (self.reboot, "on_reboot"),
                             (self.crash, "on_crash")):
            widget.set_active(
                self.combomapping.index(self.dom[parm]))

        for widget in (self.rootdev, self.kernargs):
            widget.set_text('')

        # For now show it from the conf file.
        # TODO : show it from the dominfo.
        if self.dom and self.dom.get_config():
            self.diskedit.showtree(self.dom.get_config())
        else:
            self.diskedit.showtree()
        self.__misc_props_prepare_model()
        self.__image_props_prepare_model()

    def __showDisk(self):
        """Displays the current selection's on disk settings"""

        self.__dom0WidgetsOn(self.ondisk.flags() & gtk.SENSITIVE)
        self.__diskWidgetsOn(self.ondisk.flags() & gtk.SENSITIVE)

        # Special handling for memory. (Some weired legacy)
        self.memory.set_sensitive(self.ondisk.flags() & gtk.SENSITIVE)

        #_RESTRUCT
        #if self.dom.isResident:
        #    self.domconfig = self.dom.get_config()
        #else:
        #    self.domconfig = self.dom

        #self.domconfig = self.dom.get_config()
        
        self.filename.set_text(self.domconfig.filename)

        bl = self.domconfig["bootloader"]
        if bl and bl.strip() != '':
            self.use_bootloader.set_active(True)
        else:
            self.use_bootloader.set_active(False)

        if self.ondisk.flags() & gtk.SENSITIVE:
            self.__boot_toggled(self.use_bootloader)
       
        for entry, widget in self.diskmapping.iteritems():
            if entry in self.textfields:
                if entry in self.domconfig:
                    value = self.domconfig[entry]
                else:
                    value = ''

                if not value:
                    value = ''

                if entry == 'bootloader':
                    widget.set_text('/usr/bin/pygrub')
                else:
                    widget.set_text(value)

            elif entry == "memory":
                self.memory.set_range(self.minmem,
                                      int(self.managed_node["total_memory"]))

                if entry in self.domconfig:
                    value = int(self.domconfig[entry])
                else:
                    value = self.defaultmem

                widget.set_value(value)

            elif entry == "vcpus":
                if entry in self.domconfig:
                    value = int(self.domconfig[entry])
                else:
                    value = 1

                widget.set_value(value)

            elif entry in self.combofields:
                if entry in self.domconfig:
                    value = self.combomapping.index(self.domconfig[entry])
                else:
                    value = -1

                widget.set_active(value)

        self.diskedit.showtree(self.domconfig)
        self.__misc_props_prepare_model()
        self.__image_props_prepare_model()
        
    def __showBlank(self):
        """Clears all the fields"""

        self.__dom0WidgetsOn(True)
        self.__diskWidgetsOn(True)


        self.filename.set_text(self.conf_dir)

        for entry in ("name", "kernel", "ramdisk", "root", "extra"):
            self.diskmapping[entry].set_text('')
            
        self.memory.set_value(self.defaultmem)
        self.vcpu.set_value(1)

        for entry in ("on_shutdown", "on_crash"):
            self.diskmapping[entry].set_active(0)

        self.diskmapping["on_reboot"].set_active(1)

        self.diskedit.showtree()
        self.use_bootloader.set_active(False)
        self.bootloader.set_sensitive(False)

    
    def show(self, widget, managed_node, dom = None,
             image_name = None, image_edit_mode = False,
             blank=False, rerun=False):
        """Displays the settings dialog"""

        # Requires
        #  - managed_node to get information on available cpu and mem
        #  - dom : for which the settings are being changed.

        ## Kludge, for now only one dialgbox allowed
        self.managed_node = managed_node
        self.dom = dom
        self.image_name = image_name
        self.image_edit_mode = image_edit_mode
        self.blank = blank
        self.enable_write = False
        self.vcpu.set_range(1.0, float(self.managed_node["nr_cpus"]))

        if not rerun:
            if not self.image_edit_mode:
                if self.blank:
                    self.dialog.set_title("Provision VM")
                else:
                    self.dialog.set_title("VM Configuration Settings")
            else:
                self.dialog.set_title("Edit Image : " + self.image_name)


            show_name = not self.image_edit_mode
            self.filename.set_property("visible", show_name)
            self.name.set_property("visible", show_name)
            self.filenamesel.set_property("visible",show_name)
            self.dom_name_label.set_property("visible", show_name)
            self.dom_config_name_label.set_property("visible",show_name)

            self.last_dom_name = ''
            self.update_image.set_active(False)
            # set initial tab correctly.
            self.main_notebook.set_current_page(0)
            self.vm_notebook.set_current_page(0)

            if not self.blank:
                self.chg_settings_label.set_property("visible",True)
                self.inmemory.set_property("visible", True)
                self.ondisk.set_property("visible",True)
                if self.main_notebook.get_n_pages() > 1:
                    self.main_notebook.remove_page(1)
                self.update_image.set_property("visible", False)
            else:
                self.update_image.set_property("visible", False) #unused
                if self.main_notebook.get_n_pages() < 2:

                    self.main_notebook.append_page(self.prov_page,
                                                   self.prov_page_label)
                    
            if dom is not None and dom.get_config() is not None:
                self.domconfig = dom.get_config()
            else:
                if self.image_edit_mode:
                    template_file = self.image_store.get_vm_template(image_name)
                    local_node = main_context["local_node"]
                    self.domconfig = DomConfig(local_node, template_file)
                else:
                    self.domconfig = None

            self.image_config = None

            if not blank and dom and self.managed_node.isResident(dom.name):
                self.type = "memory"
                self.inmemory.set_sensitive(True)
                self.__noOpActive(self.inmemory, True)

                if self.dom.get_config() is not None and \
                   self.dom.get_config().is_xenman_generated() :
                    self.ondisk.set_sensitive(True)
                else:
                    self.ondisk.set_sensitive(False)

                self.__showMem()
            else:
                self.type = "disk"
                self.inmemory.set_sensitive(False)
                self.__noOpActive(self.ondisk, True)
                if blank or (dom and dom.get_config().is_xenman_generated()):
                    self.ondisk.set_sensitive(True)
                else:
                    self.ondisk.set_sensitive(False)

                if blank and dom is None:
                    # set default conf dir for xen domu config files
                    if self.managed_node is not None:
                        self.conf_dir = self.managed_node.config.get(XMConfig.PATHS,
                                                                constants.prop_xenconf_dir)
                    else:
                        self.conf_dir = ''
            
                    if self.conf_dir is None:
                        self.conf_dir = ''
                    
                    self.__showBlank()
                    self.__misc_props_prepare_model()
                    self.__image_props_prepare_model()
                    # if image name is specified, set it.
                    if self.image_edit_mode :
                        # make special adjustments
                        self.filename.set_text(self.image_store.get_vm_template(image_name))
                    if len(self.availableImages) > 0:
                        default = self.image_name
                        if not default and \
                               self.image_store.get_default_image() is not None:
                                default = self.image_store.get_default_image()
                        if default in self.availableImages:
                            pos = self.availableImages.index(default)
                            if pos >= 0:
                                self.image_name_combo.set_active(pos)
                                self.__image_changed(self.image_name_combo)
                        else:
                            self.image_name_combo.set_active(0)

                else:
                    self.__showDisk()

                    
                
        res = self.dialog.run() == gtk.RESPONSE_OK

        if not res:
            self.dialog.hide()
            return

        try: # prevent exception in call back. Causes segfault.
            if self.type == "memory":
                self.dom.setMem(int(self.memory.get_value()))
                self.dom.setVCPUs(int(self.vcpu.get_value()))
            else:
                ## if disk was disabled return
                if self.blank or self.dom.get_config().is_xenman_generated():
                    pass
                else:
                    self.dialog.hide()
                    return

                errmsgs = []

                filename = self.filename.get_text()

                if not filename:
                    errmsgs.append("You must specify a file name.")

                if not self.image_edit_mode:
                    if not self.name.get_text():
                        errmsgs.append("You must specify a name for the domain.")
                    elif self.managed_node.isResident(self.name.get_text()):
                        errmsgs.append("Running VM with the same name exists.")
                        
                #TO_DO: centralize is_hvm. move it to XenDomain ?
                if not self.use_bootloader.get_active():
                    kernel_value = self.kernel.get_text()
                    if not kernel_value:
                        errmsgs.append("You must specify a valid kernel file name.")
                    is_hvm = (kernel_value is not None) and \
                             kernel_value.endswith("hvmloader")
                    ramdisk_value = self.ramdisk.get_text()
                    if not is_hvm and  not ramdisk_value : 
                        errmsgs.append("You must specify a valid ram disk file name." )
                else:
                    value = self.bootloader.get_text()
                    if not value:
                        errmsgs.append("You must specify a valid bootloader.")

                if errmsgs:
                    errmsg = "The following errors were found:\n"
                    for msg in errmsgs:
                        errmsg += " - " + msg + "\n"

                    showmsg(errmsg)
                    self.show(widget, self.managed_node, self.dom,
                              self.image_name, self.image_edit_mode,
                              blank, True)
                    return

                if not self.image_edit_mode and \
                       managed_node.node_proxy.file_exists(filename) and \
                       ((not blank and filename != self.domconfig.filename) \
                        or blank) and not \
                        confirmation("Do you wish to overwrite: %s" % filename):
                    self.show(widget, self.managed_node,self.dom,
                              self.image_name,self.image_edit_mode,
                              blank, True)
                    return


                if self.blank:
                    if self.domconfig is None:
                        self.domconfig = DomConfig(self.managed_node)
                else:
                    self.domconfig = DomConfig(self.managed_node, filename)

                # Consider enhancing the for loop below to test whether config
                # elements have actually changed and only update the DomFile
                # instance entry for those that have changed

                for entry, widget in self.diskmapping.iteritems():
                    if entry in self.textfields:
                        self.domconfig[entry] = widget.get_text()

                    elif entry in self.valuefields:
                        self.domconfig[entry] = int(widget.get_value())

                    elif entry in self.combofields:
                        self.domconfig[entry] = self.combomapping[widget.get_active()]

                self.domconfig["disk"] = [
                    repr(i) for i in self.diskedit.getDiskEntries()
                    ]

                # reset depending on bootloader being used or not.
                if self.use_bootloader.get_active() and \
                       self.bootloader.get_text() and \
                       self.bootloader.get_text().strip() != '':
                    self.domconfig['ramdisk'] = ''
                    self.domconfig['kernel'] = ''
                else:
                    self.domconfig['bootloader']=''

                # process the misc property savings
                excluded = self.ui_displayed  #\
                           #+ self.domconfig.get_computed_options()
                self.__update_config_props(self.domconfig,
                                           self.misc_props_model,
                                           excluded)

                if self.blank:
                    self.domconfig['image_name'] = self.image_name_combo.get_active_text()

                # in blank mode, domconfig represents the template
                if self.blank and not self.image_edit_mode: 
                    self.domconfig.set_managed_node(self.managed_node)

                self.domconfig.save(filename)


                self.__update_config_props(self.image_config,
                                           self.image_props_model,
                                           [])
                

                # update the image.conf image, if in image edit mode
                if self.blank and self.image_edit_mode:
                    self.image_config.write(image_edit_mode)

                # NOTE : the image_conf would get written to the managed_node
                # by the execute_provisioning script as a part of preparing
                # environment
                    
                # execute the provisioning script
                if self.blank and not self.image_edit_mode:
                    store = self.image_store
                    img_name = self.image_name_combo.get_active_text()
                    try:
                        show_wait_cursor(self.dialog)
                        (out, exit_code, log) = \
                              store.execute_provisioning_script(self.managed_node,
                                                                img_name,
                                                                self.domconfig,
                                                                self.image_config)
                    finally:
                        hide_wait_cursor(self.dialog)
                        
                    if exit_code != 0:
                        #showmsg("Provisioning script failed: " + out)
                        msg = "Provisioning Failed"
                        dlg = ProvStatusDlg(self.wtree).show(widget,
                                                             msg,
                                                             self.managed_node,
                                                             log_filename=log,
                                                             output=out,
                                                             parentwin = self.dialog)
    

                        self.show(widget, self.managed_node,self.dom,
                                  self.image_name,self.image_edit_mode,
                                  blank, True)
                        return
                    self.managed_node.add_dom_config(filename)
                    
        except xml.parsers.expat.ExpatError, e:
            showmsg("Internal Error : Try again.")
        except Exception ,ex:
            traceback.print_exc()
            showmsg("Exception : " + str(ex))
            
        self.dialog.hide()

