/* -*-Mode: C++;-*-
 * $Id: extracttest.cc 1.17 Wed, 16 May 2001 03:33:56 +0400 jmacd $
 *
 * Copyright (C) 2000, Joshua P. MacDonald <jmacd@CS.Berkeley.EDU>
 * and The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 *    Neither name of The University of California nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "rcswalk.h"
#include "xdfs_cpp.h"
#include "xdfs_comp.h"
#include <xdelta.h>
#include <edsiostdio.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <dirent.h>

DBFS        *dbfs;

char        *delta_file;
char        *orig_file;
char        *cons_file;

XdfsPolicy   extract_policy;

gint32       etest_min_versions    = 0;
gint32       etest_max_versions    = 40;
const char*  etest_test            = NULL;
const char*  etest_base_dir        = NULL;
const char*  etest_log_dir         = NULL;

static ConfigOption options [] = {
    { "min_versions", "min",    CS_Use,    CO_Optional, CD_Int32,   & etest_min_versions },
    { "max_versions", "max",    CS_Use,    CO_Optional, CD_Int32,   & etest_max_versions },
    { "method",       "mth",    CS_Use,    CO_Required, CD_String,  & etest_test },
    { "archive_base", "ab",     CS_Ignore, CO_Required, CD_String,  & etest_base_dir },
    { "archive_log",  "al",     CS_Ignore, CO_Optional, CD_String,  & etest_log_dir }
};

int
write_orig_file (const MAJORC &ino)
{
    FileHandle *w, *r;
    int ret;

    if ((ret = ino.read_segment (& r, 0))) {
	PROP_ERROR (ret) ("read_segment");
	return ret;
    }

    if (! (w = handle_write_file (orig_file))) {
	return -1;
    }

    if (! (handle_drain (r, w))) {
	return -1;
    }

    if (! handle_close (r)) {
	return -1;
    }

    if (! handle_close (w)) {
	return -1;
    }

    handle_free (r);
    handle_free (w);

    return 0;
}

int
test_compare (const char *recons_file, const MAJORC& orig_node)
{
    FileHandle *orig, *recons;
    guint8 obuf[1024], rbuf[1024];
    int offset = 0;
    int ret;
    int oc, rc;

    if ((ret = orig_node.read_segment (& orig, 0))) {
	PROP_ERROR (ret) ("read_segment");
	return ret;
    }

    if (! (recons = handle_read_file (recons_file))) {
	PROP_ERROR ("handle_read_file");
	return -1;
    }

    for (;;) {

	oc = handle_read (orig, obuf, 1024);
	rc = handle_read (recons, rbuf, 1024);

	if (oc < 0 || rc < 0) {
	    PROP_ERROR ("read failed");
	    return -1;
	}

	if (oc != rc) {
	    XDFS_ERROR ("verify failed: different lengths: %d != %d", oc, rc);
	    return -1;
	}

	if (oc == 0) {
	    break;
	}

	for (int i = 0; i < oc; i += 1) {
	    if (obuf[i] != rbuf[i]) {
		XDFS_ERROR ("verify failed: different content at offset: %d", offset+i);
		return -1;
	    }
	}

	offset += oc;
    }

    handle_close (orig);
    handle_close (recons);

    handle_free (orig);
    handle_free (recons);

    return 0;
}

int
extract_test_pair (XdfsLocation &loc, const MAJORC &from_ino, const MAJORC &to_ino, const guint8* orig_digest)
{
    // FROM has been copied to ORIG_FILE, for verification, should
    // produce ORIG_DIGEST.
    static int times = 0;

    FileHandle *delta_out;
    FileHandle *orig_in;
    FileHandle *delta_in;
    FileHandle *cons_out;
    const guint8* cons_digest;
    const MessageDigest *md = edsio_message_digest_md5 ();
    int ret;

    times += 1;

    if (! (delta_out = handle_write_file (delta_file))) {
	return -1;
    }

    if ((ret = loc.extract_delta (from_ino, to_ino, delta_out))) {
	PROP_ERROR (ret) ("extract_delta");
	return ret;
    }

    if (! handle_close (delta_out)) {
	return -1;
    }

    handle_free (delta_out);

    if (! (orig_in = handle_read_file (orig_file))) {
	return -1;
    }

    if (! (delta_in = handle_read_file (delta_file))) {
	return -1;
    }

    if (! (cons_out = handle_write_file (cons_file))) {
	return -1;
    }

    if (! handle_digest_compute (cons_out, md)) {
	return -1;
    }

    if ((ret = xdfs_apply_delta (orig_in, delta_in, cons_out))) {
	PROP_ERROR (ret) ("apply_delta");
	return ret;
    }

    g_assert ((guint) handle_length (cons_out) == to_ino.length ().key ());

    if (! handle_close (orig_in)) {
	return -1;
    }

    if (! handle_close (delta_in)) {
	return -1;
    }

    if (! handle_close (cons_out)) {
	return -1;
    }

    if (! (cons_digest = handle_digest (cons_out, md))) {
	return -1;
    }

    if ((ret = test_compare (cons_file, to_ino))) {
	XDFS_ERROR ("test_compare");
	return -1;
    }

    if (memcmp (cons_digest, orig_digest, md->cksum_len) != 0) {
	XDFS_ERROR ("reconstruction failed");
	return -1;
    }

    XDFS_ERROR ("reconstruction succeeded");

    handle_free (orig_in);
    handle_free (delta_in);
    handle_free (cons_out);

    return 0;
}

int
extract_test (XdfsLocation &loc)
{
    DIRC  c1;
    NODEC idx;
    int   ret;

    idx = loc.index_dir ();

    if ((ret = idx.dir_open (c1))) {
	PROP_ERROR (ret) ("idx_dir_open");
	return ret;
    }

    while (c1.next ()) {

	DIRC   c2;
	MAJORC ino1;

	if ((ret = c1.get_node (ino1))) {
	    PROP_ERROR (ret) ("c1_get_node");
	    return ret;
	}

	if ((ret = write_orig_file (ino1))) {
	    PROP_ERROR ("write_orig_file");
	    return ret;
	}

	if ((ret = idx.dir_open (c2))) {
	    PROP_ERROR (ret) ("idx_dir_open");
	    return ret;
	}

	while (c2.next ()) {

	    DKEY   digest;
	    MAJORC ino2;

	    if ((ret = c2.get_key (digest))) {
		PROP_ERROR (ret) ("c2_get_key");
		return ret;
	    }

	    if ((ret = c2.get_node (ino2))) {
		PROP_ERROR (ret) ("c2_get_node");
		return ret;
	    }

	    if ((ret = extract_test_pair (loc, ino1, ino2, digest.data ()))) {
		PROP_ERROR (ret) ("extract_test_pair");
		//return ret;
	    }
	}

	if ((ret = c2.close ())) {
	    PROP_ERROR (ret) ("c2_close");
	    return ret;
	}
    }

    if ((ret = c1.close ())) {
	PROP_ERROR (ret) ("c1_close");
	return ret;
    }

    return 0;
}

int
extract_dateorder (RcsFile* rcs, RcsVersion *v, void* data)
{
    FileHandle  *fh;
    TXN          txn;
    MAJORC       root;
    XdfsLocation loc;
    int          ret;

    if ((ret = txn.begin_root (*dbfs, DBFS_TXN_SYNC, root))) {
	PROP_ERROR (ret) ("txn_begin");
	return ret;
    }

    MAJORC loc_node;
    DKEY   key (rcs->filename);

    if ((ret = root.dir_link_lookup (key, loc_node, DBFS_NOFLAG))) {

	if (ret != DBFS_NOTFOUND) {
	    PROP_ERROR (ret) ("root_dir_link_lookup");
	    return ret;
	}

	SAREA      area;
	XdfsParams params;

	if ((ret = txn.create_area (area))) {
	    PROP_ERROR (ret) ("txn_create_area");
	    return ret;
	}

	params.flags  = XF_MD5Equality;
	params.policy = extract_policy;

	if ((ret = loc.create (area, & params, loc_node))) {
	    PROP_ERROR (ret) ("xdfs_loc_create");
	    return ret;
	}

	if ((ret = root.dir_link_insert (key, loc_node, DBFS_NOOVERWRITE))) {
	    PROP_ERROR (ret) ("root_dir_link_insert");
	    return ret;
	}
    } else {

	if ((ret = loc.open (loc_node))) {
	    PROP_ERROR (ret) ("xdfs_loc_open");
	    return ret;
	}
    }

    if (! (fh = handle_read_file (v->filename))) {
	PROP_ERROR (errno) ("handle_read_file");
	return -1;
    }

    MAJORC insert_node;

    if ((ret = loc.insert_version (fh, insert_node))) {
	PROP_ERROR (ret) ("xdfs_insert_version");
	return ret;
    }

    handle_free (fh);

    if ((ret = txn.commit ())) {
	PROP_ERROR (ret) ("txn_commit");
	return ret;
    }

    return 0;
}

int
extract_onefile (RcsFile *rcs, RcsStats* set, void* data)
{
    TXN          txn;
    MAJORC       root;
    XdfsLocation loc;
    int          ret;

    if ((ret = txn.begin_root (*dbfs, DBFS_TXN_SYNC, root))) {
	PROP_ERROR (ret) ("txn_begin");
	return ret;
    }

    MAJORC loc_node;
    DKEY   key (rcs->filename);

    if ((ret = root.dir_link_lookup (key, loc_node, DBFS_NOFLAG))) {
	PROP_ERROR (ret) ("root_dir_link_lookup");
	return ret;
    }

    if ((ret = loc.open (loc_node))) {
	PROP_ERROR (ret) ("xdfs_loc_open");
	return ret;
    }

    if ((ret = extract_test (loc))) {
	PROP_ERROR (ret) ("extract_test");
	return ret;
    }

    if ((ret = txn.commit ())) {
	PROP_ERROR (ret) ("txn_commit");
	return ret;
    }

    return 0;
}

RcsWalker extract_walker = {
    NULL,
    NULL,
    & extract_onefile,
    & extract_dateorder,
    NULL,
    NULL,
    0,
    G_MAXINT,
    TRUE
};

int
main (int argc, char** argv)
{
    int i, ret;
    DBFS dbfs_x (argc, argv);

    dbfs = & dbfs_x;

    rcswalk_init ();

    orig_file  = g_strdup_printf ("%s/orig.%d", g_get_tmp_dir (), (int) getpid ());
    cons_file  = g_strdup_printf ("%s/cons.%d", g_get_tmp_dir (), (int) getpid ());
    delta_file = g_strdup_printf ("%s/delta.%d", g_get_tmp_dir (), (int) getpid ());

    config_register (options, ARRAY_SIZE (options));

    config_set_string ("rcswalk_experiment", "et");

    if (argc < 2) {
	config_help ();
	goto bail;
    }

    for (i = 1; i < argc; i += 1) {
	if ((ret = config_parse (argv[i]))) {
	    goto bail;
	}
    }

    if ((ret = config_done ())) {
	goto bail;
    }

    if (strcmp (etest_test, "XDFS-r") == 0) {
	extract_policy = XP_Reverse;
    } else if (strcmp (etest_test, "XDFS-f") == 0) {
	extract_policy = XP_Forward;
    } else if (strcmp (etest_test, "XDFS-fs") == 0) {
	extract_policy = XP_ForwardSingle;
    } else {
	g_error ("unknown TEST: %s\n", argv[1]);
    }

    extract_walker.min_versions = etest_min_versions;
    extract_walker.max_versions = etest_max_versions;

    if ((ret = config_clear_dir (etest_base_dir))) {
	goto bail;
    }

    if ((ret = config_clear_dir (etest_log_dir))) {
	goto bail;
    }

    if ((ret = xdfs_library_init ())) {
	goto bail;
    }

    if ((ret = dbfs->open (etest_base_dir, etest_log_dir, DBFS_CREATE | DBFS_CLEAR))) {
	goto bail;
    }

    if ((ret = rcswalk (& extract_walker, NULL))) {
	goto bail;
    }

    unlink (orig_file);
    unlink (cons_file);
    unlink (delta_file);

    dbfs->close ();

    return 0;

  bail:

    dbfs->close ();

    unlink (orig_file);
    unlink (cons_file);
    unlink (delta_file);

    return 2;
}
