/* -*-Mode: C++;-*-
 * $Id: dxn.cc 1.23 Wed, 16 May 2001 03:33:56 +0400 jmacd $
 *
 * Copyright (C) 1999, 2000, Joshua P. MacDonald <jmacd@CS.Berkeley.EDU>
 * and The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 *    Neither name of The University of California nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "xdfs_cpp.h"

#include <unistd.h>
#include <errno.h>

//////////////////////////////////////////////////////////////////////
//			     EXTERNAL
//////////////////////////////////////////////////////////////////////

int
SAREA::allocate (MAJORC &major, int flags) const
{
    int ret;
    DXN dxn (txn (), ret);

    if ((ret = dxn.create_major (*this, major, flags))) {
	PROP_ERROR (ret) ("create_major");
	return ret;
    }

    return 0;
}

//////////////////////////////////////////////////////////////////////
//		          DXN STORAGE AREA
//////////////////////////////////////////////////////////////////////

int
DXN::create_area (SAREA &area)
{
    int         ret;
    SAREA_DESC *desc;

    // @@ How to cleanup unreferenced areas?

    // New descriptor
    if ((ret = _txn._areas.alloc (desc))) {
	PROP_ERROR (ret) ("AREA descriptor allocate");
	return ret;
    }

    if ((ret = allocate_fid (desc->rec.container_fid))) {
	PROP_ERROR (ret) ("allocate_fid");
	return ret;
    }

    // Create the container
    if ((ret = desc->open_cont (_txn, DBFS_CREATE))) {
	PROP_ERROR (ret) ("area_create_dbs");
	return ret;
    }

    // Append area record: get new area ID
    if ((ret = desc->create (_txn))) {
	PROP_ERROR (ret) ("area_desc_create");
	return ret;
    }

    _txn._area_map[desc->area_id.key ()] = desc;

    area = SAREA (_txn, *desc);

    return 0;
}

int
DXN::get_area (const XAREA   &area_id,
	       SAREA         &area_out)
{
    SAREA_MAP::iterator exist = _txn._area_map.find (area_id.key ());

    if (exist != _txn._area_map.end ()) {

	SAREA_DESC *area = (*exist).second;

	area_out = SAREA (_txn, *area);

    } else {

	SAREA_DESC *area;
	int         ret;

	if ((ret = _txn._areas.alloc (area))) {
	    PROP_ERROR (ret) ("AREA descriptor allocate");
	    return ret;
	}

	area->area_id = area_id;

	if ((ret = area->get (_txn))) {
	    PROP_ERROR (ret) ("area_desc_get");
	    return ret;
	}

	if ((ret = area->open_cont (_txn, 0))) {
	    PROP_ERROR (ret) ("area_open_cont");
	    return ret;
	}

	_txn._area_map[area->area_id.key ()] = area;

	area_out = SAREA (_txn, *area);
    }

    return 0;
}

//////////////////////////////////////////////////////////////////////
//			        MAJOR
//////////////////////////////////////////////////////////////////////

int
DXN::create_major (const SAREA   &area,
		   MAJORC        &node,
		   int            flags)
{
    int         ret;
    MAJOR_DESC *major;
    MINOR_DESC *minor;

    if ((ret = _txn._majors.alloc (major))) {
	PROP_ERROR (ret) ("MAJOR descriptor allocate");
	return ret;
    }

    if ((ret = _txn._minors.alloc (minor))) {
	PROP_ERROR (ret) ("MINOR descriptor allocate");
	return ret;
    }

    if ((ret = allocate_major (area, major->num, flags))) {
	PROP_ERROR (ret) ("allocate_major");
	return ret;
    }

    major->def_minor  = minor;
    major->area       = & const_cast<SAREA_DESC&> (area.ref ());
    major->rec.refs   = DBFS_ZERO_REFS;
    major->all_minors = minor;

    minor->mkey          = DBFS_DEF_MKEY;
    minor->major         = major;
    minor->rec.type      = XTYPE_NOT_PRESENT;
    minor->next_in_major = NULL;

    major->set_clean_notpresent ();
    minor->set_clean_notpresent ();

    clean_refs (* major);

    node = MAJORC (_txn, *major);

    return 0;
}

int
DXN::get_major (const XNUM  &majnum,
		MAJORC      &major_out)
{
    MAJOR_MAP::iterator exist = _txn._major_map.find (majnum.key ());

    if (exist != _txn._major_map.end ()) {

	MAJOR_DESC *major = (*exist).second;

	major_out = MAJORC (_txn, *major);

    } else {

	int         ret;
	MAJOR_DESC *major;
	MINOR_DESC *minor;
	SAREA       area;

	if ((ret = _txn._majors.alloc (major))) {
	    PROP_ERROR (ret) ("MAJOR descriptor allocate");
	    return ret;
	}

	if ((ret = _txn._minors.alloc (minor))) {
	    PROP_ERROR (ret) ("MINOR descriptor allocate");
	    return ret;
	}

	if ((ret = get_area (majnum.area_id (), area))) {
	    PROP_ERROR (ret) ("get_area");
	    return ret;
	}

	major->num        = majnum;
	major->def_minor  = minor;
	major->area       = & const_cast<SAREA_DESC&> (area.ref ());
	major->all_minors = minor;

	minor->mkey          = DBFS_DEF_MKEY;
	minor->major         = major;
	minor->next_in_major = NULL;

	if ((ret = major->get ())) {
	    PROP_ERROR (ret) ("major_desc_get");
	    return ret;
	}

	if ((ret = minor->get ())) {
	    PROP_ERROR (ret) ("minor_desc_get");
	    return ret;
	}

	clean_refs (* major);

	major_out = MAJORC (_txn, *major);
    }

    return 0;
}

//////////////////////////////////////////////////////////////////////
//			        MINOR
//////////////////////////////////////////////////////////////////////

int
DXN::create_view (const NODEC   &node,
		  const XSIZ    &length,
		  const VIEWDEF &vdef,
		  int            flags)
{
    int ret;
    MINOR_DESC &desc = const_cast<MINOR_DESC&> (node.ref ());

    if ((ret = node.pre_overwrite_now (flags))) {
	return ret;
    }

    desc.rec.type    = XTYPE_VIEWSEG;
    desc.rec.length  = length;
    desc.rec.cont    = XCONT (XAREA (0), vdef.id ());

    desc.set_dirty ();

    return 0;
}

int
DXN::create_reflink (const NODEC   &node,
		     const MAJORC  &linkto,
		     int            flags)
{
    int ret;
    MINOR_DESC &desc = const_cast<MINOR_DESC&> (node.ref ());

    if ((ret = node.pre_overwrite_now (flags))) {
	return ret;
    }

    desc.rec.type    = XTYPE_REFLINK;
    desc.rec.length  = XSIZ  (0);
    desc.rec.cont    = XCONT (linkto.number ());

    desc.set_dirty ();

    return 0;
}

int
DXN::create_arealink (const NODEC   &node,
		      const SAREA   &area,
		      int            flags)
{
    int ret;
    MINOR_DESC &desc  = const_cast<MINOR_DESC&> (node.ref ());
    SAREA_DESC &sarea = const_cast<SAREA_DESC&> (area.ref ());

    if ((ret = node.pre_overwrite_now (flags))) {
	return ret;
    }

    desc.rec.type    = XTYPE_AREALINK;
    desc.rec.length  = XSIZ  (0);
    desc.rec.cont    = XCONT (sarea.area_id, XSEQNO (0));

    desc.set_dirty ();

    return 0;
}

int
DXN::get_minor (const MAJORC &major,
		const MKEY   &mkey,
		NODEC        &node)
{
    int         ret;
    MINOR_DESC *minor;
    MAJOR_DESC &major_desc = const_cast<MAJOR_DESC&> (major.ref ());

    if (mkey.get_stck () == DBFS_INV_STACK) {
	DBFS_ERROR ("The DBFS_INV_STCK is reserved for inverse links");
	return DBFS_INVAL;
    }

    ret = major_desc.find_minor (_txn, mkey, node);

    if (ret == 0) {
	return 0;
    }

    if ((ret = _txn._minors.alloc (minor))) {
	PROP_ERROR (ret) ("MINOR descriptor allocate");
	return ret;
    }

    minor->mkey          = mkey;
    minor->major         = & major_desc;
    minor->next_in_major = major_desc.all_minors;

    major_desc.all_minors = minor;

    if ((ret = minor->get ())) {
	PROP_ERROR (ret) ("minor_desc_get");
	return ret;
    }

    node = NODEC (_txn, *minor);

    return 0;
}

//////////////////////////////////////////////////////////////////////
//		      DXN UNDERLYING FILE CREATION
//////////////////////////////////////////////////////////////////////

int
DXN::create_long (const NODEC   &node,
		  const XFID    &alloc_fid,
		  const XSIZ    &length,
		  int            flags)
{
    int ret;

    if ((ret = node.pre_overwrite_now (flags))) {
	return ret;
    }

    MINOR_DESC &desc = const_cast<MINOR_DESC&> (node.ref ());

    // Note: Handled in seg.cc

    desc.rec.type    = XTYPE_LONGSEG;
    desc.rec.length  = length;
    desc.rec.cont    = XCONT (XAREA (0), alloc_fid);

    desc.set_dirty ();

    return 0;
}

int
DXN::create_index (const NODEC   &node,
		   const XTYPE   &type,
		   int            flags)
{
    int    ret;
    XFID   alloc_fid;
    DBTYPE dbtype;
    int    rec_size = -1;

    if ((ret = node.pre_overwrite_now (flags))) {
	return ret;
    }

    if ((ret = allocate_fid (alloc_fid))) {
	PROP_ERROR (ret) ("allocate_fid");
	return ret;
    }

    switch (type.key ()) {
    case FT_DirSeq:
	dbtype   = DB_QUEUE;
	rec_size = sizeof (XNUM);
	break;
    case FT_DirBtree:
	dbtype = DB_BTREE;
	break;
    case FT_DirHash:
	dbtype = DB_HASH;
	break;
    default:
	g_assert_not_reached ();
    }

    DYNAMIC_DB dir_db;

    if ((ret = dir_db.open (_txn.dbfs (), alloc_fid, dbtype, rec_size, DBFS_CREATE))) {
	PROP_ERROR (ret) ("directory_create");
	return ret;
    }

    MINOR_DESC &desc = const_cast<MINOR_DESC&> (node.ref ());

    desc.rec.type    = type;
    desc.rec.length  = XSIZ  (0);
    desc.rec.cont    = XCONT (XAREA (0), alloc_fid);

    desc.set_dirty ();

    return 0;
}

int
DXN::get_index (const NODEC   &node,
		DBREF         &dbr)
{
    int        ret;
    DBTYPE     dbtype;

    switch (node.type ()) {
    case FT_DirSeq:
	dbtype = DB_QUEUE;
	break;
    case FT_DirBtree:
	dbtype = DB_BTREE;
	break;
    case FT_DirHash:
	dbtype = DB_HASH;
	break;
    default:
	g_assert_not_reached ();
    }

    DYNAMIC_DB dir_db;

    if ((ret = dir_db.open (_txn.dbfs (), node.cont_id ().file_id (), dbtype, -1, 0))) {
	PROP_ERROR (ret) ("directory_open");
	return ret;
    }

    dbr = dir_db;

    return 0;
}

//////////////////////////////////////////////////////////////////////
//			     DXN SHORTS
//////////////////////////////////////////////////////////////////////

// CREATE is in alloc.cc

int
DXN::get_short (const NODEC   &node,
		guint8        *buf)
{
    int ret;

    if (node.length ().key () > 0) {

	BYTEDBT short_dbt (buf, node.length ().key ());

	if ((ret = node.locc ().get_must_exist (RECDBT<XSEQNO> (node.cont_id ().cont_seq ()),
						short_dbt,
						DBFS_NORMW))) { // @@ Should pass RMW
	    PROP_ERROR (ret) ("get_must_exist");
	    return ret;
	}
    }

    return 0;
}

int
DXN::del_short (const NODEC& node)
{
    int ret;

    if (node.length ().key () > 0) {
	if ((ret = node.locc ().template delete_must_exist_T<XSEQNO> (node.cont_id ().cont_seq ()))) {
	    PROP_ERROR (ret) ("delete_must_exist");
	    return ret;
	}
    }

    return 0;
}

//////////////////////////////////////////////////////////////////////
//			     DXN ACCESSORS
//////////////////////////////////////////////////////////////////////

int
MAJOR_DESC::get ()
{
    int ret;

    if ((ret = area->locc.template get_must_exist_T<XSEQNO,MAJOR_REC> (num.cont_seq (), rec, DBFS_NORMW))) {
	return ret;
    }

    return 0;
}

int
MAJOR_DESC::find_minor (TXN        &txn, // Note: Unless MAJOR_DESC has _txn
			const MKEY &mkey,
			NODEC      &node)
{
    for (MINOR_DESC* minp = all_minors;
	 minp;
	 minp = minp->next_in_major) {

	if (mkey == minp->mkey) {

	    node = NODEC (txn, *minp);

	    return 0;
	}
    }

    return DBFS_NOTFOUND;
}

int
MAJOR_DESC::minor_final (TXN &txn)
{
    int ret;

    for (MINOR_DESC *mdesc = all_minors;
	 mdesc;
	 mdesc = mdesc->next_in_major) {

	if ((ret = mdesc->final (txn))) {
	    PROP_ERROR (ret) ("mdesc_final");
	    return ret;
	}
    }

    return 0;
}

int
MAJOR_DESC::minor_delete_notpresent (TXN &txn)
{
    int ret;

    for (MINOR_DESC *mdesc = all_minors;
	 mdesc;
	 mdesc = mdesc->next_in_major) {

	NODEC node (txn, *mdesc);

	// this checks its descriptor status, this routine only touches those
	// which are not yet reflected in the database
	if (! mdesc->get_notpresent ()) {
	    continue;
	}

	// this checks current type
	if (node.type () == FT_NotPresent) {
	    DEBUG_PRECOMMIT (mdesc) ("transient not present");
	    continue;
	}

	DEBUG_PRECOMMIT (mdesc) ("transient delete %s", dbfs_xtype_to_string (mdesc->rec.type));

	if ((ret = node.del_minor ())) {
	    PROP_ERROR (ret) ("final_minor_delete");
	    return ret;
	}
    }

    return 0;
}

int
MAJOR_DESC::minor_delete_present (TXN &txn)
{
    int ret;

    CKEY cont_key;
    CKEY cont_rec (sizeof (MINOR_REC));

    DBCREF scan;

    if ((ret = area->container_dbr.cursor (txn, scan))) {
	PROP_ERROR (ret) ("scan_cont_cursor");
	return ret;
    }

    cont_key.template set_rec<XSEQNO> (num.cont_seq ());

    if ((ret = scan.set_range (cont_key, cont_rec))) {

	if (ret == DBFS_NOTFOUND) {

	    if (get_notpresent ()) {
		return 0;
	    } else {
		ret = DBFS_LINK_INCONSISTENT;
	    }
	}

	PROP_ERROR (ret) ("final_set_range");
	return ret;
    }

    while (ret != DBFS_NOTFOUND) {

	XSEQNO cont_seq;

	cont_key.template get_rec_prefix<XSEQNO> (cont_seq);

	if (cont_seq != num.cont_seq ()) {
	    // Past the prefix
	    break;
	}

	const size_t seqno_prefix = (sizeof (XSEQNO));
	const size_t stack_prefix = (sizeof (XSEQNO) + sizeof (XSTCK));

	if (cont_key.size () >= stack_prefix) {

	    XSTCK stck;

	    cont_key.template get_rec_at<XSTCK> (seqno_prefix, stck);

	    if (stck == DBFS_INV_STACK) {
		// This is an inverse link: not reached because the major's
 		// refcount shouldn't be zero unless all dir links are gone
		DBFS_ERROR (this) ("unexpected inverse links in delete");
		return DBFS_LINK_INCONSISTENT;

	    } else {
		// This is a minor node

		NODEC mnode;
		MKEY  mkey;
		MINOR_DESC tmpdesc;

		g_assert (cont_rec.size () == sizeof (MINOR_REC));

		mkey.append_buf (cont_key.data () + seqno_prefix, cont_key.size () - seqno_prefix);

		ret = find_minor (txn, mkey, mnode);

		if (ret != 0) {

		    cont_rec.template get_rec<MINOR_REC> (tmpdesc.rec);

		    tmpdesc.mkey          = mkey;
		    tmpdesc.major         = NULL;
		    tmpdesc.next_in_major = NULL;

		    mnode = NODEC (txn, tmpdesc);

		    g_assert (mnode.type () != FT_NotPresent);
		} else {
		    g_assert (! mnode.ref ().get_notpresent ());
		}

		if (mnode.type () == FT_NotPresent) {
		    DEBUG_PRECOMMIT (mnode) ("stable not present");
		} else {

		    DEBUG_PRECOMMIT (mnode) ("stable delete %s", dbfs_type_to_string (mnode.type ()));

		    if ((ret = mnode.del_minor ())) {
			PROP_ERROR (ret) ("final_delete_minor");
			return ret;
		    }
		}
	    }

	} else {
	    // Then this is the major record
	    g_assert (cont_rec.size () == sizeof (MAJOR_REC));
	    g_assert (! get_notpresent ());
	}

	if ((ret = scan.delete_current ())) {
	    PROP_ERROR (ret) ("final_delete_current");
	    return ret;
	}

	if ((ret = scan.move_pos (cont_key, cont_rec, DB_NEXT, DBFS_RMW)) && (ret != DBFS_NOTFOUND)) {
	    PROP_ERROR (ret) ("final_move_pos");
	    return ret;
	}
    }

    if ((ret = scan.close ())) {
	PROP_ERROR (ret) ("scan_cont_close");
	return ret;
    }

    return 0;
}

int
MAJOR_DESC::final (TXN &txn)
{
    int ret;

    if (rec.refs == DBFS_ZERO_REFS) {

	// Garbage collection

	DEBUG_PRECOMMIT (this) ("%s garbage",
				get_notpresent () ? "transient" : "stable");

	// First scan all_minors for notpresent minors, delete.  They
	// all become clean.

	if ((ret = minor_delete_notpresent (txn))) {
	    PROP_ERROR (ret) ("minor_delete_notpresent");
	    return ret;
	}

	// Now, there may be minor nodes in the DB which are not in
	// the all_minors list, even when this node is notpresent
	// there may be inverse links (which are not cached).  Do a
	// range search in the DB, deleting all records with the
	// major-cont-seq prefix and calling del_minor on all minor
	// nodes found.

	if ((ret = minor_delete_present (txn))) {
	    PROP_ERROR (ret) ("minor_delete_present");
		return ret;
	}

	set_clean_notpresent ();

    } else {

	// Write-back updates, minors first

	if ((ret = minor_final (txn))) {
	    PROP_ERROR (ret) ("minor_final");
	    return ret;
	}

	g_assert (rec.refs > DBFS_ZERO_REFS);

	DEBUG_PRECOMMIT (this) ("final refcount: ") () << rec.refs << (get_dirty () ? " (dirty)" : " (clean)");

	if (get_dirty () && (ret = area->locc.template put_overwrite_T<XSEQNO,MAJOR_REC> (num.cont_seq (), rec))) {
	    PROP_ERROR (ret) ("final_put_major");
	    return ret;
	}

	set_clean ();
    }

    return 0;
}

void
DXN::clean_refs (MAJOR_DESC &major)
{
    _txn._major_map[major.num.key ()] = & major;

    _txn._major_order.slp_insert (major.num.key (), & major);
}

void
MAJORC::incr_refcount (int change) const
{
    MAJOR_DESC &desc = const_cast<MAJOR_DESC&> (ref ());

    DEBUG_PRECOMMIT (desc) ("incr refcount:") () << " old " << desc.rec.refs
						 << " new " << (desc.rec.refs + change);

    desc.rec.refs += change;

    desc.set_dirty ();

    txn ()._major_order.slp_insert (desc.num.key (), & desc);
}

int
MINOR_DESC::get ()
{
    int               ret;
    RECDBT<MINOR_REC> rec_dbt (rec);

    if ((ret = major->area->locc.get_or_notfound (MINOR_DB_KEY (major->num, mkey),
						  rec_dbt,
						  DBFS_NORMW)) &&
	(ret == DBFS_NOTFOUND)) {

	set_clean_notpresent ();

	rec.type = XTYPE_NOT_PRESENT;
	return 0;
    }

    if (ret != 0) {
	PROP_ERROR (ret) ("minor_get_or_notfound");
    }

    return ret;
}

int
MINOR_DESC::final (TXN &txn)
{
    int ret;

    // @@@ SYNC: This is where the FD's minor node is committed, wait on fsync
    // Dirty means: (modified) || (present && deleted)
    // Clean means: (unmodified) || (new && deleted)
    if (! get_dirty ()) {
	return 0;
    }

    if (get_deleted ()) {

	g_assert (! get_notpresent ());
	g_assert (rec.type == XTYPE_NOT_PRESENT);

	DEBUG_PRECOMMIT (this) ("deleted");

	if ((ret = major->area->locc.delete_must_exist (MINOR_DB_KEY (major->num, mkey)))) {
	    PROP_ERROR (ret) ("final_delete_must_exist");
	    return ret;
	}

	set_clean_notpresent ();

    } else {

	g_assert (rec.type != XTYPE_NOT_PRESENT);

	DEBUG_PRECOMMIT (this) ("%s: %s",
				get_notpresent () ? "created" : "updated",
				dbfs_xtype_to_string (rec.type));

	if ((ret = major->area->locc.put_overwrite (MINOR_DB_KEY (major->num, mkey),
						    RECDBT<MINOR_REC> (rec)))) {
	    PROP_ERROR (ret) ("final_put_overwrite");
	    return ret;
	}

	set_clean ();
    }

    return 0;
}

int
SAREA_DESC::get (DBCREF &curs)
{
    int ret;

    if ((ret = curs.template get_must_exist_T<XAREA,SAREA_REC> (area_id, rec, DBFS_NORMW))) {
	return ret;
    }

    return 0;
}

int
SAREA_DESC::put (DBCREF &curs)
{
    int ret;

    if ((ret = curs.template put_overwrite_T<XAREA,SAREA_REC> (area_id, rec))) {
	return ret;
    }

    return 0;
}

int
SAREA_DESC::get (STXN &txn)
{
    int ret;

    if ((ret = txn.dbfs ()._area_db.template get_must_exist_T<XAREA,SAREA_REC> (txn, area_id, rec, DBFS_NORMW))) {
	return ret;
    }

    return 0;
}

int
SAREA_DESC::create (STXN &txn)
{
    int    ret;
    XSEQNO seqno;

    if ((ret = txn.dbfs ()._area_db.template append_T<SAREA_REC> (txn, seqno, rec))) {
	return ret;
    }

    area_id = XAREA (seqno.key ());

    return 0;
}

int
SAREA_DESC::open_cont (TXN &txn, int flag)
{
    int ret;

    if ((ret = container_dbr.open (txn.dbfs (), rec.container_fid, DB_BTREE, -1, flag))) {
	PROP_ERROR (ret) ("area_cont_open");
	return ret;
    }

    if ((ret = container_dbr.cursor (txn, locc))) {
	PROP_ERROR (ret) ("area_cont_cursor");
	return ret;
    }

    return 0;
}

int
SAREA_DESC::close ()
{
    int ret;

    if ((ret = locc.close ())) {
	PROP_ERROR (ret) ("cursor_close");
	return ret;
    }

    container_dbr.close ();

    return 0;
}
