/* -*-Mode: C;-*-
 * $Id: fh.c,v 1.11 2001/12/08 03:17:21 jmacd Exp $
 *
 * Copyright (C) 1998, 1999, 2000, Joshua P. MacDonald <jmacd@CS.Berkeley.EDU>
 * and The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 *    Neither name of The University of California nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <unistd.h>

#include "edsio.h"
#include "edsiopriv.h"

#ifdef HAVE_FDATASYNC
#define FSYNC fdatasync
#else
#define FSYNC fsync
#endif

static gboolean handle_unmap_current_page (FileHandle* fh);
static gssize   handle_map_current_page   (FileHandle* fh);


/* Handle source/sink impls
 */
typedef struct _HandleSerialSource HandleSerialSource;
typedef struct _HandleSerialSink HandleSerialSink;

struct _HandleSerialSource {
  SerialSource source;

  FileHandle* fh;
};

struct _HandleSerialSink {
  SerialSink sink;

  FileHandle* fh;
  gpointer data1;
  gpointer data2;
  gpointer data3;
  gboolean (* cont_onclose) (gpointer data1, gpointer data2, gpointer data3);
};

static SerialType handle_source_type           (SerialSource* source, gboolean set_allocation);
static gboolean   handle_source_close          (SerialSource* source);
static gboolean   handle_source_read           (SerialSource* source, guint8 *ptr, guint32 len);
static void       handle_source_free           (SerialSource* source);

static gboolean     handle_sink_type             (SerialSink* sink, SerialType type, guint len, gboolean set_allocation);
static gboolean     handle_sink_close            (SerialSink* sink);
static gboolean     handle_sink_write            (SerialSink* sink, const guint8 *ptr, guint32 len);
static void         handle_sink_free             (SerialSink* sink);

SerialSource*
handle_source (FileHandle* fh)
{
  HandleSerialSource* it = g_new0 (HandleSerialSource, 1);

  serializeio_source_init (&it->source,
			   handle_source_type,
			   handle_source_close,
			   handle_source_read,
			   handle_source_free,
			   NULL,
			   NULL);

  it->fh = fh;

  return &it->source;
}

SerialSink*
handle_sink (FileHandle* fh, gpointer data1, gpointer data2, gpointer data3, gboolean (* cont_onclose) (gpointer data1, gpointer data2, gpointer data3))
{
  HandleSerialSink* it = g_new0 (HandleSerialSink, 1);

  serializeio_sink_init (&it->sink,
			 handle_sink_type,
			 handle_sink_close,
			 handle_sink_write,
			 handle_sink_free,
			 NULL);

  it->fh = fh;
  it->data1 = data1;
  it->data2 = data2;
  it->data3 = data3;
  it->cont_onclose = cont_onclose;

  return &it->sink;
}

SerialType
handle_source_type (SerialSource* source, gboolean set_allocation)
{
  HandleSerialSource* ssource = (HandleSerialSource*) source;
  guint32 x;

  if (! handle_getui (ssource->fh, &x))
    return ST_Error;

  if (set_allocation)
    {
      if (! handle_getui (ssource->fh, &source->alloc_total))
	return ST_Error;
    }

  return x;
}

gboolean
handle_source_close (SerialSource* source)
{
  HandleSerialSource* ssource = (HandleSerialSource*) source;

  return handle_close (ssource->fh);
}

gboolean
handle_source_read (SerialSource* source, guint8 *ptr, guint32 len)
{
  HandleSerialSource* ssource = (HandleSerialSource*) source;

  return handle_read (ssource->fh, ptr, len) == len;
}

void
handle_source_free (SerialSource* source)
{
  g_free (source);
}

gboolean
handle_sink_type (SerialSink* sink, SerialType type, guint len, gboolean set_allocation)
{
  HandleSerialSink* ssink = (HandleSerialSink*) sink;

  if (! handle_putui (ssink->fh, type))
    return FALSE;

  if (set_allocation && ! handle_putui (ssink->fh, len))
    return FALSE;

  return TRUE;
}

gboolean
handle_sink_close (SerialSink* sink)
{
  HandleSerialSink* ssink = (HandleSerialSink*) sink;

  if (handle_close (ssink->fh))
    {
      if (ssink->cont_onclose)
	return ssink->cont_onclose (ssink->data1, ssink->data2, ssink->data3);

      return TRUE;
    }

  return FALSE;
}

gboolean
handle_sink_write (SerialSink* sink, const guint8 *ptr, guint32 len)
{
  HandleSerialSink* ssink = (HandleSerialSink*) sink;

  return handle_write (ssink->fh, ptr, len);
}

void
handle_sink_free (SerialSink* sink)
{
  g_free (sink);
}

/* Table-dependent handle functions
 */

void
handle_free (FileHandle *fh)
{
  MessageDigestCtx* mdctx, *tmp;
  HookList *hl, *tmp2;

  g_assert (! fh->fh_cur_page);

  if (fh->fh_read_page)
    g_free (fh->fh_read_page);

  for (mdctx = fh->fh_digests; mdctx; )
    {
      tmp = mdctx;
      mdctx = mdctx->next;
      g_free (tmp->md_ctx);
      g_free (tmp->md_val);
      g_free (tmp);
    }

  for (hl = fh->fh_close_hooks; hl; )
    {
      tmp2 = hl;
      hl = hl->next;
      g_free (tmp2);
    }

  return (* fh->table->fh_free) (fh);
}

void
handle_close_hook (FileHandle *fh, void* data, gboolean (* func) (FileHandle* fh, void* data))
{
  HookList *hl = g_new (HookList, 1);

  hl->data = data;
  hl->func = func;
  hl->next = fh->fh_close_hooks;
  fh->fh_close_hooks = hl;
}

gboolean
handle_close (FileHandle *fh)
{
  HookList *hl;

  g_return_val_if_fail (fh, FALSE);

  if (fh->fh_closed)
    {
      edsio_generate_handle_event (EC_EdsioHandleClosed, fh);
      return FALSE;
    }

  if (! handle_unmap_current_page (fh))
    return FALSE;

  if (! handle_digest_close (fh))
    return FALSE;

  fh->fh_closed = TRUE;

  if (! (* fh->table->fh_close) (fh))
    return FALSE;

  for (hl = fh->fh_close_hooks; hl; hl = hl->next)
    {
      if (! hl->func (fh, hl->data))
	return FALSE;
    }

  return TRUE;
}

void
digest (FileHandle* fh, const guint8* buf, guint offset, guint len)
{
  MessageDigestCtx* mdctx;

  if (len == 0)
    return;

  for (mdctx = fh->fh_digests; mdctx; mdctx = mdctx->next)
    {
      if (offset == mdctx->update_pos)
	{
	  mdctx->update_pos += len;
	  mdctx->md->update (mdctx->md_ctx, buf, len);
	}
    }
}

void
handle_abort (FileHandle *fh)
{
  g_return_if_fail (fh);

  if (fh->fh_closed)
    {
      edsio_generate_handle_event (EC_EdsioHandleClosed, fh);
      return;
    }

  fh->fh_closed = TRUE;

  handle_unmap_current_page (fh);

  (* fh->table->fh_abort) (fh);
}

gboolean
handle_unmap_current_page (FileHandle* fh)
{
  gssize rem;

  if (! fh->fh_cur_page)
    return TRUE;

  g_assert (! fh->table->fh_read);

  rem = file_position_rem_on_page_no (&fh->fh_file_len, fh->fh_cur_pos.page);

  g_assert (rem >= 0);

  if (fh->fh_open_mode == HV_Replace)
    digest (fh, fh->fh_cur_page, fh->fh_cur_pos.page * fh->fh_cur_pos.page_size, rem);

  return (* fh->table->fh_unmap) (fh, fh->fh_cur_pos.page, (const guint8**) & fh->fh_cur_page);
}

gssize
handle_map_current_page (FileHandle* fh)
{
  gssize rem, rem_on_page;

  g_assert (! fh->table->fh_read);

  rem_on_page = file_position_rem_on_page_no (&fh->fh_file_len, fh->fh_cur_pos.page);
  rem = rem_on_page - fh->fh_cur_pos.offset;

  g_assert (rem >= 0);

  if (rem > 0 && fh->fh_cur_page != NULL)
    return rem;

  if (rem == 0 && fh->fh_cur_pos.page == fh->fh_file_len.page)
    return 0;

  if (! handle_unmap_current_page (fh))
    return -1;

  g_assert (! fh->fh_cur_page);

  if (rem == 0)
    {
      fh->fh_cur_pos.offset = 0;
      fh->fh_cur_pos.page += 1;

      rem_on_page = file_position_rem_on_page_no (&fh->fh_file_len, fh->fh_cur_pos.page);
      rem = rem_on_page - fh->fh_cur_pos.offset;
    }

  if (rem > 0)
    {
      if ((* fh->table->fh_map) (fh, fh->fh_cur_pos.page, (const guint8**) & fh->fh_cur_page) != rem_on_page)
	return -1;

      if (fh->fh_open_mode == HV_Read)
	digest (fh, fh->fh_cur_page, fh->fh_cur_pos.page * fh->fh_cur_pos.page_size, rem_on_page);
    }

  return rem;
}

gssize
handle_read (FileHandle *fh,
	     guint8     *buf,
	     gsize       nbyte0)
{
  gsize nbyte = nbyte0;
  gssize nread;

  g_return_val_if_fail (fh && buf, -1);

  if (fh->fh_closed)
    {
      edsio_generate_handle_event (EC_EdsioHandleClosed, fh);
      return -1;
    }

  if (fh->fh_open_mode != HV_Read)
    {
      edsio_generate_handlestring_event (EC_EdsioInvalidHandleMode, fh, "read");
      return -1;
    }

  if (nbyte == 0)
    return 0;

  if (fh->table->fh_read)
    {
      nread = (* fh->table->fh_read) (fh, buf, nbyte);

      if (nread > 0)
	{
	  digest (fh, buf, file_position_to_abs (& fh->fh_cur_pos), nread);

	  fh->fh_cur_pos.offset += nread;

	  file_position_from_abs (fh->fh_cur_pos.page_size,
				  file_position_to_abs (& fh->fh_cur_pos),
				  &fh->fh_cur_pos);
	}

      return nread;
    }

  while (nbyte > 0)
    {
      gssize rem = handle_map_current_page (fh);
      gsize cp;

      if (rem < 0)
	return -1;

      if (rem > 0)
	{
	  cp = MIN (rem, nbyte);

	  memcpy (buf, fh->fh_cur_page + fh->fh_cur_pos.offset, cp);

	  fh->fh_cur_pos.offset += cp;

	  nbyte -= cp;
	  buf += cp;
	}

      if (rem == 0)
	break;
    }

  return nbyte0 - nbyte;
}

gboolean
handle_write (FileHandle *fh,
	      const guint8 *buf,
	      gsize nbyte)
{
  gssize len;
  gssize pos;

  g_return_val_if_fail (fh, FALSE);

  if (fh->fh_closed)
    {
      edsio_generate_handle_event (EC_EdsioHandleClosed, fh);
      return FALSE;
    }

  if (fh->fh_open_mode != HV_Replace)
    {
      edsio_generate_handlestring_event (EC_EdsioInvalidHandleMode, fh, "replace");
      return FALSE;
    }

  if (nbyte == 0)
    return TRUE;
  else
    g_assert (buf);

  g_assert (fh->fh_has_len);

  len = file_position_to_abs (&fh->fh_file_len);
  pos = file_position_to_abs (&fh->fh_cur_pos);

  len = MAX (len, pos + nbyte);

  file_position_from_abs (fh->fh_file_len.page_size, len, &fh->fh_file_len);

  if (fh->table->fh_write)
    {
      gboolean res = (* fh->table->fh_write) (fh, buf, nbyte);

      if (res)
	{
	  digest (fh, buf, file_position_to_abs (& fh->fh_cur_pos), nbyte);

	  fh->fh_cur_pos.offset += nbyte;

	  file_position_from_abs (fh->fh_cur_pos.page_size,
				  file_position_to_abs (& fh->fh_cur_pos),
				  &fh->fh_cur_pos);
	}

      return res;
    }

  while (nbyte > 0)
    {
      gssize rem = handle_map_current_page (fh);
      gsize cp;

      if (rem < 0)
	return FALSE;

      g_assert (rem > 0);

      cp = MIN (rem, nbyte);

      /* same as above except switch buffers. */
      g_assert (fh->fh_cur_page);
      memcpy (fh->fh_cur_page + fh->fh_cur_pos.offset, buf, cp);

      fh->fh_cur_pos.offset += cp;

      nbyte -= cp;
      buf += cp;
    }

  return TRUE;
}

gssize
handle_map_page (FileHandle *fh, guint pgno, const guint8** mem)
{
  gssize res;

  if (fh->fh_closed)
    {
      edsio_generate_handle_event (EC_EdsioHandleClosed, fh);
      return -1;
    }

  if (fh->fh_open_mode != HV_Read)
    {
      edsio_generate_handlestring_event (EC_EdsioInvalidHandleMode, fh, "map");
      return -1;
    }

  if (fh->table->fh_read)
    {
      if (fh->fh_cur_pos.page != pgno || fh->fh_cur_pos.offset != 0)
	{
	  edsio_generate_handle_event (EC_EdsioIllegalMapSeek, fh);
	  return -1;
	}

      if (fh->fh_read_mapped)
	{
	  edsio_generate_handle_event (EC_EdsioIllegalMapUnmapFirst, fh);
	  return -1;
	}

      if (! fh->fh_read_page)
	fh->fh_read_page = g_malloc (fh->fh_cur_pos.page_size);

      if ((res = (* fh->table->fh_read) (fh, fh->fh_read_page, fh->fh_cur_pos.page_size)) < 0)
	return -1;

      fh->fh_read_mapped = TRUE;

      digest (fh, fh->fh_read_page, file_position_to_abs (& fh->fh_cur_pos), res);

      fh->fh_cur_pos.offset += res;

      file_position_from_abs (fh->fh_cur_pos.page_size,
			      file_position_to_abs (& fh->fh_cur_pos),
			      &fh->fh_cur_pos);

      (* mem) = fh->fh_read_page;

      return res;
    }

  if (pgno > fh->fh_file_len.page)
    {
      edsio_generate_handleint_event (EC_EdsioInvalidPage, fh, pgno);
      return -1;
    }

  res = (* fh->table->fh_map) (fh, pgno, mem);

  if (res > 0)
    digest (fh, *mem, pgno * fh->fh_cur_pos.page_size, res);

  return res;
}

gboolean
handle_unmap_page (FileHandle *fh, guint pgno, const guint8** mem0_ptr)
{
  if (fh->fh_closed)
    {
      edsio_generate_handle_event (EC_EdsioHandleClosed, fh);
      return FALSE;
    }

  if (fh->fh_open_mode != HV_Read)
    {
      edsio_generate_handlestring_event (EC_EdsioInvalidHandleMode, fh, "unmap");
      return FALSE;
    }

  if (fh->table->fh_read)
    {
      if (! fh->fh_read_mapped)
	{
	  edsio_generate_handle_event (EC_EdsioIllegalUnmap, fh);
	  return -1;
	}

      fh->fh_read_mapped = FALSE;

      if (mem0_ptr)
	(* mem0_ptr) = NULL;

      return TRUE;
    }

  if (pgno > fh->fh_file_len.page)
    {
      edsio_generate_handleint_event (EC_EdsioInvalidPage, fh, pgno);
      return FALSE;
    }

  return (* fh->table->fh_unmap) (fh, pgno, mem0_ptr);
}

const char*
eventdelivery_handle_to_string (FileHandle* fh)
{
  g_return_val_if_fail (fh, g_strdup ("*error*"));

  return fh->table->fh_name (fh);
}

gssize
handle_seek (FileHandle *fh,
	     gssize offset,
	     gint whence)
{
  gssize cur;

  g_return_val_if_fail (fh, FALSE);
  g_return_val_if_fail (whence == HANDLE_SEEK_SET ||
			whence == HANDLE_SEEK_CUR ||
			whence == HANDLE_SEEK_END, FALSE);

  if (fh->fh_closed)
    {
      edsio_generate_handle_event (EC_EdsioHandleClosed, fh);
      return -1;
    }

  switch (whence)
    {
    case HANDLE_SEEK_SET:
      {
	cur = offset;
      }
      break;
    case HANDLE_SEEK_CUR:
      {
	cur = file_position_to_abs (&fh->fh_cur_pos);
	cur += offset;
      }
      break;
    case HANDLE_SEEK_END:
      {
	if (! fh->fh_has_len)
	  {
	    edsio_generate_handle_event (EC_EdsioHandleHasNoLength, fh);
	    return -1;
	  }

	cur = file_position_to_abs (&fh->fh_file_len);
	cur += offset;
      }
    break;
    default:
      abort ();
      return -1;
    }

  if (file_position_to_abs (&fh->fh_cur_pos) != cur)
    {
      FilePosition new_cur;

      if (fh->fh_open_flags & (HV_NoSeek | HV_Replace) || fh->table->fh_read)
	{
	  edsio_generate_handle_event (EC_EdsioIllegalSeek, fh);
	  return -1;
	}

      if (file_position_to_abs (&fh->fh_cur_pos) > file_position_to_abs (& fh->fh_file_len))
	{
	  edsio_generate_handle_event (EC_EdsioIllegalSeek, fh);
	  return -1;
	}

      file_position_from_abs (fh->fh_cur_pos.page_size, cur, &new_cur);

      if (fh->fh_cur_pos.page != new_cur.page && ! handle_unmap_current_page (fh))
	return -1;

      file_position_from_abs (fh->fh_cur_pos.page_size, cur, &fh->fh_cur_pos);
    }

  return cur;
}

gssize
handle_pages (FileHandle *fh)
{
  if (! fh->fh_has_len)
    {
      edsio_generate_handle_event (EC_EdsioHandleHasNoLength, fh);
      return -1;
    }

  return fh->fh_file_len.page + (fh->fh_file_len.offset > 0);
}

gboolean
handle_eof (FileHandle* fh)
{
  if (fh->fh_open_flags & HV_Read)
    {
      if (fh->fh_has_len)
	return file_position_to_abs (& fh->fh_cur_pos) == file_position_to_abs (& fh->fh_file_len);

      g_assert (fh->table->fh_eof);

      return fh->table->fh_eof (fh);
    }
  else
    {
      return TRUE;
    }
}

gssize
handle_length (FileHandle* fh)
{
  if (! fh->fh_has_len)
    {
      edsio_generate_handle_event (EC_EdsioHandleHasNoLength, fh);
      return -1;
    }

  return file_position_to_abs (& fh->fh_file_len);
}

gsize
handle_position (FileHandle* fh)
{
  return file_position_to_abs (& fh->fh_cur_pos);
}

/* Type independent functions
 */

gssize
handle_pagesize (FileHandle *fh)
{
  return fh->fh_cur_pos.page_size;
}

gboolean
handle_putc (FileHandle *fh, gchar c)
{
  return handle_write (fh, (guint8*) &c, 1);
}

gint
handle_getc (FileHandle *fh)
{
  guint8 x;

  if (handle_read (fh, &x, 1) != 1)
    return -1;

  return x;
}

gboolean
handle_getui (FileHandle *fh, guint32* i)
{
  if (handle_read (fh, (guint8*)i, 4) != 4)
    return FALSE;

  *i = g_ntohl (*i);

  return TRUE;
}

gboolean
handle_putui (FileHandle *fh, guint32 i)
{
  guint32 hi = g_htonl (i);

  return handle_write (fh, (guint8*)&hi, 4);
}

gboolean
handle_printf (FileHandle* handle,
	       const char* fmt,
	       ...
	       )
{
  va_list args;
  gboolean result;
  guint8* buf;

  va_start (args, fmt);

  buf = g_strdup_vprintf (fmt, args);

  result = handle_write (handle, buf, strlen (buf));

  g_free (buf);

  va_end (args);

  return result;
}

gpointer
handle_unique (FileHandle* fh)
{
  /* @@@ This can be improved to share buffers between multiple file
   * handles of the same open file, LATER. */
  return fh;
}

gboolean
handle_puts (FileHandle *fh, const char* linebuf)
{
  for (; *linebuf; linebuf += 1)
    {
      if (! handle_putc (fh, *linebuf))
	return FALSE;
    }

  return TRUE;
}

gssize
handle_gets (FileHandle *fh, char* buf, gsize nbyte)
{
  int i = 0;

  for (;;)
    {
      int c;

      if ((c = handle_getc (fh)) < 0)
	{
	  if (i == 0)
	    return -1;

	  break;
	}

      buf[i++] = c;

      if (c == '\n')
	break;

      if (i == nbyte)
	break;
    }

  if (i < nbyte)
    buf[i] = 0;

  return i;
}

gboolean
handle_copy_len (FileHandle *from,
		 FileHandle *to,
		 guint       len)
{
  /* Double buffering here... blah.
   */
  guint ncopy = 0;
  guint8 copybuf[1<<12];

  while (ncopy < len)
    {
      guint thiscopy = MIN (len-ncopy, 1<<12);
      guint actual;

      actual = handle_read (from, copybuf, thiscopy);

      if (actual != thiscopy)
	{
	  edsio_generate_handle_event (EC_EdsioHandleEof, from);
	  return FALSE;
	}

      if (! handle_write (to, copybuf, thiscopy))
	return FALSE;

      ncopy += thiscopy;
    }

  return TRUE;
}

gboolean
handle_copy (FileHandle *from,
	     FileHandle *to,
	     guint       off,
	     guint       len)
{
  if (handle_seek (from, off, HANDLE_SEEK_SET) != off)
    return FALSE;

  return handle_copy_len (from, to, len);
}

/* File position
 */

gssize
file_position_rem_on_page_no (const FilePosition *len, guint page)
{
  if (page > len->page)
    {
      abort ();
      return -1;
    }

  if (page == len->page)
    return len->offset;

  return len->page_size;
}

gssize
file_position_to_abs (const FilePosition *pos)
{
  return pos->page_size * pos->page + pos->offset;
}

void
file_position_from_abs (guint page_size, guint abs, FilePosition *pos)
{
  pos->page_size = page_size;
  pos->page = abs / page_size;
  pos->offset = abs % page_size;
}

void
file_position_incr (FilePosition *pos, gint incr_by)
{
  gint nabs = file_position_to_abs (pos) + incr_by;

  g_assert (nabs >= 0);

  file_position_from_abs (pos->page_size, nabs, pos);
}

gboolean handle_sync (FileHandle *fh)
{
  return (* fh->table->fh_sync) (fh);
}

/* Stdio
 */

#include "edsiostdio.h"
#include <errno.h>

FileHandle* _stdout_handle;
FileHandle* _stdin_handle;
FileHandle* _stderr_handle;

typedef struct _StdioHandle StdioHandle;

struct _StdioHandle {
  FileHandle handle;

  FILE* file;
  const char* name;
};

static gboolean     stdio_handle_close (FileHandle* fh);
static void         stdio_handle_abort (FileHandle* fh);
static const gchar* stdio_handle_name  (FileHandle *fh);
static void         stdio_handle_free  (FileHandle* fh);
static gssize       stdio_handle_read  (FileHandle *fh, guint8* buf, gsize nbyet);
static gboolean     stdio_handle_write (FileHandle *fh, const guint8 *buf, gsize nbyte);
static gboolean     stdio_handle_eof   (FileHandle *fh);
static gssize       stdio_handle_map   (FileHandle *fh, guint pgno, const guint8** mem);
static gboolean     stdio_handle_unmap (FileHandle *fh, guint pgno, const guint8** mem);
static gboolean     stdio_handle_sync  (FileHandle *fh);

static HandleFuncTable stdio_noseek_table = {
  stdio_handle_close,
  stdio_handle_abort,
  stdio_handle_name,
  stdio_handle_free,
  NULL,
  NULL,
  stdio_handle_read,
  stdio_handle_write,
  stdio_handle_eof,
  stdio_handle_sync
};

static HandleFuncTable stdio_seek_table = {
  stdio_handle_close,
  stdio_handle_abort,
  stdio_handle_name,
  stdio_handle_free,
  stdio_handle_map,
  stdio_handle_unmap,
  NULL,
  NULL,
  NULL,
  stdio_handle_sync
};

#define STDIO_PAGE_SIZE (1<<13)

FileHandle*
handle_read_stdio (FILE* f, const char* name)
{
  StdioHandle *h = g_new0 (StdioHandle, 1);

  struct stat buf;

  h->file = f;
  h->name = name;

  h->handle.fh_open_flags = HV_Read;
  h->handle.fh_open_mode = HV_Read;

  if (fstat (fileno (f), & buf) < 0 || ! S_ISREG (buf.st_mode))
    {
      h->handle.table = & stdio_noseek_table;
      h->handle.fh_open_flags |= HV_NoSeek;
      h->handle.fh_has_len = FALSE;
    }
  else
    {
      h->handle.table = & stdio_seek_table;
      h->handle.fh_has_len = TRUE;

      file_position_from_abs (STDIO_PAGE_SIZE, buf.st_size, &h->handle.fh_file_len);
    }

  file_position_from_abs (STDIO_PAGE_SIZE, 0, &h->handle.fh_cur_pos);

  return & h->handle;
}

FileHandle*
handle_write_stdio (FILE* f, const char* name)
{
  StdioHandle *h = g_new0 (StdioHandle, 1);

  h->file = f;
  h->name = name;

  h->handle.table = & stdio_noseek_table;
  h->handle.fh_open_flags = HV_Replace;
  h->handle.fh_open_mode = HV_Replace;

  h->handle.fh_has_len = TRUE;

  file_position_from_abs (STDIO_PAGE_SIZE, 0, &h->handle.fh_file_len);
  file_position_from_abs (STDIO_PAGE_SIZE, 0, &h->handle.fh_cur_pos);

  return & h->handle;
}

gboolean
stdio_handle_sync (FileHandle *fh)
{
  StdioHandle *s = (StdioHandle*) fh;

  return FSYNC (fileno(s->file)) == 0;
}

gboolean
stdio_handle_close (FileHandle* fh)
{
  StdioHandle *s = (StdioHandle*) fh;

  return fclose (s->file) == 0;
}

void
stdio_handle_abort (FileHandle* fh)
{
  g_warning ("cannot abort stdio handles\n");
  abort ();
}

const gchar*
stdio_handle_name  (FileHandle *fh)
{
  StdioHandle *s = (StdioHandle*) fh;

  return g_strdup (s->name);
}

void
stdio_handle_free  (FileHandle* fh)
{
  g_free (fh);
}

gssize
stdio_handle_read  (FileHandle *fh, guint8* buf, gsize nbyte)
{
  StdioHandle *s = (StdioHandle*) fh;
  gssize r;

  if (fh->fh_at_eof)
    return 0;

  clearerr (s->file);

  r = fread (buf, 1, nbyte, s->file);

  if (ferror (s->file))
    {
      edsio_generate_stringerrno_event (EC_EdsioFreadFailed, s->name);
      return -1;
    }

  if (r == nbyte)
    return r;

  if (feof (s->file))
    {
      fh->fh_at_eof = TRUE;
      return r;
    }

  edsio_generate_stringerrno_event (EC_EdsioFreadFailed, s->name);
  return -1;
}

gboolean
stdio_handle_write (FileHandle *fh, const guint8 *buf, gsize nbyte)
{
  StdioHandle *s = (StdioHandle*) fh;
  gssize r;

  clearerr (s->file);

  r = fwrite (buf, 1, nbyte, s->file);

  if (ferror (s->file))
    {
      edsio_generate_stringerrno_event (EC_EdsioFwriteFailed, s->name);
      return FALSE;
    }

  if (r == nbyte)
    return TRUE;

  edsio_generate_stringerrno_event (EC_EdsioFwriteFailed, s->name);
  return FALSE;
}

gboolean
stdio_handle_eof   (FileHandle *fh)
{
  StdioHandle *s = (StdioHandle*) fh;

  if (fh->fh_at_eof)
    return TRUE;

  if (feof (s->file))
    {
      fh->fh_at_eof = TRUE;
      return TRUE;
    }

  return FALSE;
}

gssize
stdio_handle_map (FileHandle *fh, guint pgno, const guint8** mem)
{
  StdioHandle *s = (StdioHandle*) fh;
  gssize res = file_position_rem_on_page_no (& fh->fh_file_len, pgno);

  if (res == 0)
    return res;

  if (((*mem) = mmap (NULL, res, PROT_READ, MAP_PRIVATE, fileno (s->file), pgno * STDIO_PAGE_SIZE)) == MAP_FAILED)
    {
      edsio_generate_handleint_event (EC_EdsioMmapFailed, fh, errno);
      return -1;
    }

  return res;
}

gboolean
stdio_handle_unmap (FileHandle *fh, guint pgno, const guint8** mem)
{
  gssize res = file_position_rem_on_page_no (& fh->fh_file_len, pgno);

  if (res > 0 && munmap ((void*) *mem, res) < 0)
    {
      edsio_generate_handleint_event (EC_EdsioMunmapFailed, fh, errno);
      return FALSE;
    }

  (*mem) = NULL;

  return TRUE;
}

gboolean
edsio_stdio_init (void)
{
  if (! (_stdout_handle = handle_write_stdio (stdout, "<standard output>")))
    return FALSE;

  if (! (_stderr_handle = handle_write_stdio (stderr, "<standard error>")))
    return FALSE;

  if (! (_stdin_handle  = handle_read_stdio (stdin, "<standard input>")))
    return FALSE;

  return TRUE;
}

gboolean
handle_drain (FileHandle *from,
	      FileHandle *to)
{
  guint8 copybuf[1<<12];
  gint onpage;

  do
    {
      onpage = handle_read (from, copybuf, 1<<12);

      if (onpage < 0)
	return FALSE;

      if (onpage > 0 && ! handle_write (to, copybuf, onpage))
	return FALSE;
    }
  while (onpage > 0);

  return TRUE;
}

FileHandle*
handle_read_file (const char* name)
{
  FILE* f;

  if (! (f = fopen (name, "r")))
    {
      edsio_generate_stringint_event (EC_EdsioFopenReadFailed, name, errno);
      return NULL;
    }

  return handle_read_stdio (f, name);
}

FileHandle*
handle_write_file (const char* name)
{
  FILE* f;

  if (! (f = fopen (name, "w")))
    {
      edsio_generate_stringint_event (EC_EdsioFopenWriteFailed, name, errno);
      return NULL;
    }

  return handle_write_stdio (f, name);
}

/* Cmp
 */

gboolean
handle_cmp (FileHandle *f1, FileHandle *f2, guint *pos, gboolean *diff)
{
  guint8 buf1[1024], buf2[1024];
  int offset = 0;
  int oc, rc;
  int i;

  if (diff)
    (*diff) = TRUE;

  for (;;)
    {
      oc = handle_read (f1, buf1, 1024);
      rc = handle_read (f2, buf2, 1024);

      if (oc < 0 || rc < 0)
	return FALSE;

      if (oc != rc)
	{
	  if (pos)
	    (*pos) = offset + MIN (oc, rc);

	  return TRUE;
	}

      if (oc == 0)
	break;

      for (i = 0; i < oc; i += 1)
	{
	  if (buf1[i] != buf2[i])
	    {
	      if (pos)
		(*pos) = offset+i;

	      return TRUE;
	    }
	}

      offset += oc;
    }

  if (diff)
    (*diff) = FALSE;

  return TRUE;
}

/* Memory handles
 */

typedef struct _MemHandle MemHandle;

struct _MemHandle {
  FileHandle handle;

  const guint8* rbuf;
  GByteArray* warray;
};

static gboolean     mem_handle_close (FileHandle* fh);
static void         mem_handle_abort (FileHandle* fh);
static const gchar* mem_handle_name  (FileHandle *fh);
static void         mem_handle_free  (FileHandle* fh);
static gssize       mem_handle_map   (FileHandle *fh, guint pgno, const guint8** mem);
static gboolean     mem_handle_unmap (FileHandle *fh, guint pgno, const guint8** mem);

#define MEM_PAGE_SIZE (1<<14)

static HandleFuncTable mem_handle_table = {
  mem_handle_close,
  mem_handle_abort,
  mem_handle_name,
  mem_handle_free,
  mem_handle_map,
  mem_handle_unmap,
  NULL,
  NULL,
  NULL,
  NULL
};

FileHandle*
handle_read_mem (const guint8* buf, guint len)
{
  MemHandle *h = g_new0 (MemHandle, 1);

  h->rbuf = buf;

  h->handle.fh_open_flags = HV_Read;
  h->handle.fh_open_mode = HV_Read;

  h->handle.table = & mem_handle_table;
  h->handle.fh_has_len = TRUE;

  file_position_from_abs (MEM_PAGE_SIZE, len, &h->handle.fh_file_len);
  file_position_from_abs (MEM_PAGE_SIZE, 0,   &h->handle.fh_cur_pos);

  return & h->handle;
}

FileHandle*
handle_write_mem ()
{
  MemHandle *h = g_new0 (MemHandle, 1);

  h->warray = g_byte_array_new ();

  h->handle.table = & mem_handle_table;
  h->handle.fh_open_flags = HV_Replace;
  h->handle.fh_open_mode  = HV_Replace;

  h->handle.fh_has_len = TRUE;

  file_position_from_abs (MEM_PAGE_SIZE, 0, &h->handle.fh_file_len);
  file_position_from_abs (MEM_PAGE_SIZE, 0, &h->handle.fh_cur_pos);

  return & h->handle;
}

void
handle_write_mem_result (FileHandle* fh, const guint8** buf, guint* len)
{
  MemHandle *h = (MemHandle*) fh;

  if (buf)
    (*buf) = h->warray->data;

  if (len)
    (*len) = file_position_to_abs (&h->handle.fh_file_len);

}

gboolean
mem_handle_close (FileHandle* fh)
{
  return TRUE;
}

void
mem_handle_abort (FileHandle* fh)
{
}

const gchar* mem_handle_name  (FileHandle *fh)
{
  return g_strdup ("<memory handle>");
}

void
mem_handle_free  (FileHandle* fh)
{
  MemHandle *h = (MemHandle*) fh;

  if (h->warray)
    g_byte_array_free (h->warray, TRUE);

  g_free (h);
}

gssize
mem_handle_map (FileHandle *fh, guint pgno, const guint8** mem)
{
  MemHandle *h = (MemHandle*) fh;
  gssize res = file_position_rem_on_page_no (& fh->fh_file_len, pgno);

  if (h->warray)
    {
      /* Write */
      g_byte_array_set_size (h->warray, (1+pgno)*MEM_PAGE_SIZE);

      (*mem) = h->warray->data + (pgno*MEM_PAGE_SIZE);
    }
  else
    {
      /* Read */
      (*mem) = h->rbuf + (pgno*MEM_PAGE_SIZE);
    }

  return res;
}

gboolean
mem_handle_unmap (FileHandle *fh, guint pgno, const guint8** mem)
{
  (*mem) = NULL;

  return TRUE;
}
