/* -*-Mode: C;-*-
 * $Id: edsio.c,v 1.10 2001/12/08 03:17:21 jmacd Exp $
 *
 * Copyright (C) 1998, 1999, 2000, Joshua P. MacDonald <jmacd@CS.Berkeley.EDU>
 * and The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 *    Neither name of The University of California nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "edsio.h"
#include "edsiopriv.h"
#include "edsiostdio.h"

#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

/* Missing glib stuff
 */
typedef struct _GQueue GQueue;

struct _GQueue
{
  GList *list;
  GList *list_end;
  guint list_size;
};

GQueue *	g_queue_new		(void);
void		g_queue_free		(GQueue *q);
guint		g_queue_get_size	(GQueue *q);
void		g_queue_push_front	(GQueue *q, gpointer data);
void		g_queue_push_back	(GQueue *q, gpointer data);
gpointer	g_queue_pop_front	(GQueue *q);
gpointer	g_queue_pop_back	(GQueue *q);

#define g_queue_empty(queue) \
	((((GQueue *)(queue)) && ((GQueue *)(queue))->list) ? FALSE : TRUE)

#define g_queue_peek_front(queue) \
	((((GQueue *)(queue)) && ((GQueue *)(queue))->list) ? \
		((GQueue *)(queue))->list->data : NULL)

#define g_queue_peek_back(queue) \
	((((GQueue *)(queue)) && ((GQueue *)(queue))->list_end) ? \
		((GQueue *)(queue))->list_end->data : NULL)

#define g_queue_index(queue,ptr) \
	((((GQueue *)(queue)) && ((GQueue *)(queue))->list) ? \
		g_list_index (((GQueue *)(queue))->list, (ptr)) : -1)

#define		g_queue_push		g_queue_push_back
#define		g_queue_pop		g_queue_pop_front
#define		g_queue_peek		g_queue_peek_front

GQueue *
g_queue_new (void)
{
  GQueue *q = g_new (GQueue, 1);

  q->list = q->list_end = NULL;
  q->list_size = 0;

  return q;
}


void
g_queue_free (GQueue *q)
{
  if (q)
    {
      if (q->list)
        g_list_free (q->list);
      g_free (q);
    }
}


guint
g_queue_get_size (GQueue *q)
{
  return (q == NULL) ? 0 : q->list_size;
}


void
g_queue_push_front (GQueue *q, gpointer data)
{
  if (q)
    {
      q->list = g_list_prepend (q->list, data);

      if (q->list_end == NULL)
        q->list_end = q->list;

      q->list_size++;
    }
}


void
g_queue_push_back (GQueue *q, gpointer data)
{
  if (q)
    {
      q->list_end = g_list_append (q->list_end, data);

      if (! q->list)
        q->list = q->list_end;
      else
        q->list_end = q->list_end->next;

      q->list_size++;
    }
}


gpointer
g_queue_pop_front (GQueue *q)
{
  gpointer data = NULL;

  if ((q) && (q->list))
    {
      GList *node;

      node = q->list;
      data = node->data;

      if (! node->next)
        {
          q->list = q->list_end = NULL;
          q->list_size = 0;
        }
      else
        {
          q->list = node->next;
          q->list->prev = NULL;
          q->list_size--;
        }

      g_list_free_1 (node);
    }

  return data;
}


gpointer
g_queue_pop_back (GQueue *q)
{
  gpointer data = NULL;

  if ((q) && (q->list))
    {
      GList *node;

      node = q->list_end;
      data = node->data;

      if (! node->prev)
	{
          q->list = q->list_end = NULL;
          q->list_size = 0;
        }
      else
	{
          q->list_end = node->prev;
          q->list_end->next = NULL;
          q->list_size--;
        }

      g_list_free_1 (node);
    }

  return data;
}

/* Event delivery
 */

static GHashTable* all_event_defs = NULL;

static GPtrArray* all_event_watchers = NULL;

typedef struct _ErrorDeliveryWatcher ErrorDeliveryWatcher;
typedef struct _DelayedEvent DelayedEvent;

struct _ErrorDeliveryWatcher {
  ErrorDeliveryFunc deliver;
};

struct _DelayedEvent {
  GenericEvent     ev;
  GenericEventDef *def;
  const char      *msg;
};

void
eventdelivery_initialize_event_def (gint        code,
				    gint        level,
				    gint        flags,
				    const char* name,
				    const char* oneline,
				    const char * (* field_to_string) (GenericEvent* ev, gint field))
{
  GenericEventDef* def = g_new0 (GenericEventDef, 1);

  if (! all_event_defs)
    all_event_defs = g_hash_table_new (g_int_hash, g_int_equal);

  def->code = code;
  def->level = level;
  def->flags = flags;
  def->name = name;
  def->oneline = oneline;
  def->field_to_string = field_to_string;

  g_hash_table_insert (all_event_defs, & def->code, def);
}

void
eventdelivery_event_watch_all (ErrorDeliveryFunc func)
{
  ErrorDeliveryWatcher* w = g_new0 (ErrorDeliveryWatcher, 1);

  w->deliver = func;

  if (! all_event_watchers)
    all_event_watchers = g_ptr_array_new ();

  g_ptr_array_add (all_event_watchers, w);
}

GenericEventDef*
eventdelivery_event_lookup (gint code)
{
  return g_hash_table_lookup (all_event_defs, & code);
}

void
eventdelivery_event_deliver (GenericEvent* e)
{
  static gint in_call = FALSE;
  static GQueue* queued = NULL;
  static GPtrArray* free_strings = NULL;

  if (! queued)
    {
      queued = g_queue_new ();
      free_strings = g_ptr_array_new ();
    }

  in_call += 1;

  g_assert (e);

  edsio_edsio_init ();

  if (all_event_defs)
    {
      GenericEventDef* def = g_hash_table_lookup (all_event_defs, & e->code);

      if (def)
	{
	  GString* out;
	  const char* oneline = def->oneline;
	  char c;

	  out = g_string_new (NULL);

	  while ((c = *oneline++))
	    {
	      switch (c)
		{
		case '$':
		  {
		    const char* field;
		    char *end;
		    int f;

		    if ((*oneline++) != '{')
		      goto badevent;

		    f = strtol (oneline, &end, 10);

		    if (f < 0 || !end || end[0] != '}')
		      goto badevent;

		    oneline = end+1;

		    g_assert (def->field_to_string);

		    field = def->field_to_string (e, f);

		    if (field)
		      {
			g_string_append (out, field);

			g_free ((void*) field);
		      }
		    else
		      goto badevent;
		  }
		  break;
		default:
		  g_string_append_c (out, c);
		}
	    }

  	  if (! all_event_watchers)
	    {
	      handle_printf (_stderr_handle, "%s:%d: %s\n", e->srcfile, e->srcline, out->str);

	      g_string_free (out, TRUE);
	    }
	  else if (in_call == 1)
	    {
	      gint i;

	      for (i = 0; i < all_event_watchers->len; i += 1)
		{
		  ErrorDeliveryWatcher* w = all_event_watchers->pdata[i];

		  if (! w->deliver (e, def, out->str))
		    {
		      g_warning ("%s:%d: An error delivery routine failed: %s\n", e->srcfile, e->srcline, out->str);
		      in_call = 0;
		      return;
		    }
		}

	      while (g_queue_get_size (queued) > 0)
		{
		  DelayedEvent* de = g_queue_pop (queued);

		  for (i = 0; i < all_event_watchers->len; i += 1)
		    {
		      ErrorDeliveryWatcher* w = all_event_watchers->pdata[i];

		      if (! w->deliver (& de->ev, de->def, de->msg))
			{
			  g_warning ("%s:%d: An error delivery routine failed: %s\n", e->srcfile, e->srcline, out->str);
			  in_call = 0;
			  return;
			}
		    }
		}

	      for (i = 0; i < free_strings->len; i += 1)
		g_string_free (free_strings->pdata[i], TRUE);

	      g_ptr_array_set_size (free_strings, 0);

	      g_string_free (out, TRUE);
	    }
	  else
	    {
	      DelayedEvent* de = g_new (DelayedEvent, 1);

	      de->ev = *e;
	      de->def = def;
	      de->msg = out->str;

	      g_queue_push (queued, de);

	      g_ptr_array_add (free_strings, out);
	    }

	      in_call -= 1;

	  return;
	}
    }

  g_warning ("%s:%d: Unrecognized event delivered (code=%d)\n", e->srcfile, e->srcline, e->code);

  in_call -= 1;

  return;

 badevent:

  g_warning ("%s:%d: An malformed error could not print here (code=%d)\n", e->srcfile, e->srcline, e->code);

  in_call -= 1;

  return;
}

const char*
eventdelivery_int_to_string (int x)
{
  return g_strdup_printf ("%d", x);
}

const char*
eventdelivery_string_to_string  (const char* x)
{
  return g_strdup (x);
}

const char*
eventdelivery_source_to_string  (SerialSource* x)
{
  return g_strdup ("@@@SerialSource");
}

const char*
eventdelivery_sink_to_string  (SerialSink* x)
{
  return g_strdup ("@@@SerialSink");
}

/* Misc crap.
 */

gboolean
edsio_time_of_day (SerialGenericTime* setme)
{
#if HAVE_GETTIMEOFDAY

  struct timeval tv;

  if (gettimeofday (& tv, NULL))
    {
      edsio_generate_errno_event (EC_EdsioGetTimeOfDayFailure);
      goto bail;
    }

  if (setme)
    {
      setme->nanos = tv.tv_usec * 1000;
      setme->seconds = tv.tv_sec;
    }

#else

  time_t t = time (NULL);

  if (t < 0)
    {
      edsio_generate_errno_event (EC_EdsioTimeFailure);
      goto bail;
    }

  if (setme)
    {
      setme->nanos = 0;
      setme->seconds = t;
    }

#endif

  return TRUE;

 bail:

  setme->nanos = 0;
  setme->seconds = 10;

  return FALSE;
}

#if 0
gchar*
edsio_time_to_iso8601 (SerialGenericTime *tp)
{
  return edsio_time_t_to_iso8601 (tp->seconds);
}

gchar*
edsio_time_t_to_iso8601 (GTime t0)
{
  static char timebuf[64];
  time_t t = t0;

  struct tm lt = *localtime(&t);
  int utc_offset = difftm(&lt, gmtime(&t));
  char sign = utc_offset < 0 ? '-' : '+';
  int minutes = abs (utc_offset) / 60;
  int hours = minutes / 60;

  sprintf(timebuf,
	  "%d-%02d-%02d %02d:%02d:%02d%c%02d%02d",
	  lt.tm_year + 1900,
	  lt.tm_mon + 1,
	  lt.tm_mday,
	  lt.tm_hour,
	  lt.tm_min,
	  lt.tm_sec,
	  sign,
	  hours,
	  minutes % 60);

  return timebuf;
}
#endif

static gboolean
strtosl_checked (const char* str, long* l, const char* errmsg)
{
  char* end;

  (*l) = strtol (str, &end, 10);

  if (!end || end[0])
    {
      if (errmsg)
	edsio_generate_stringstring_event (EC_EdsioInvalidIntegerString, errmsg, str);

      (*l) = 0;
      return FALSE;
    }

  return TRUE;
}

gboolean
strtosi_checked (const char* str, gint32* i,  const char* errmsg)
{
  long l;

  if (! strtosl_checked (str, &l, errmsg))
    {
      (*i) = 0;
      return FALSE;
    }

  if (l > G_MAXINT || l < G_MININT)
    {
      if (errmsg)
	edsio_generate_stringstring_event (EC_EdsioIntegerOutOfRange, errmsg, str);

      (*i) = 0;
      return FALSE;
    }

  (*i) = l;

  return TRUE;
}

gboolean
strtoss_checked (const char* str, gint16* i,  const char* errmsg)
{
  long l;

  if (! strtosl_checked (str, &l, errmsg))
    {
      (*i) = 0;
      return FALSE;
    }

  if (l > G_MAXSHORT || l < G_MINSHORT)
    {
      if (errmsg)
	edsio_generate_stringstring_event (EC_EdsioIntegerOutOfRange, errmsg, str);

      (*i) = 0;
      return FALSE;
    }

  (*i) = l;

  return TRUE;
}

gboolean
strtoui_checked (const char* str, guint32* i, const char* errmsg)
{
  long l;

  if (! strtosl_checked (str, &l, errmsg))
    {
      (*i) = 0;
      return FALSE;
    }

  if (l < 0)
    {
      if (errmsg)
	edsio_generate_stringstring_event (EC_EdsioInvalidIntegerSign, errmsg, str);

      (*i) = 0;
      return FALSE;
    }

  (*i) = l;

  if (l != (*i))
    {
      if (errmsg)
	edsio_generate_stringstring_event (EC_EdsioIntegerOutOfRange, errmsg, str);

      (*i) = 0;
      return FALSE;
    }

  return TRUE;
}

gboolean
strtous_checked (const char* str, guint16* i, const char* errmsg)
{
  long l;

  if (! strtosl_checked (str, &l, errmsg))
    {
      (*i) = 0;
      return FALSE;
    }

  if (l < 0)
    {
      if (errmsg)
	edsio_generate_stringstring_event (EC_EdsioInvalidIntegerSign, errmsg, str);

      (*i) = 0;
      return FALSE;
    }

  (*i) = l;

  if (l != (*i))
    {
      if (errmsg)
	edsio_generate_stringstring_event (EC_EdsioIntegerOutOfRange, errmsg, str);

      (*i) = 0;
      return FALSE;
    }

  return TRUE;
}

void
serializeio_print_bytes (const guint8* bytes, guint len0)
{
  char buf[100];
  int i;
  guint len;

  buf[0] = 0;

  len = MIN (len0, 32);

  for (i = 0; i < len; i += 1)
    sprintf (buf + 2*i, "%02x", bytes[i]);

  if (len0 > len)
    strcat (buf, "...");

  g_print ("%s\n", buf);
}

/* Strings
 */

const char* edsio_intern_string (const char* str)
{
  static GStringChunk* chunk = NULL;

  if (! chunk)
    chunk = g_string_chunk_new (256);

  return g_string_chunk_insert_const (chunk, str);
}

/* Misc source/sink stuff
 */

SerialSink*
serializeio_gzip_sink (SerialSink* sink)
{
  /* @@@ not implemented */
  return sink;
}

SerialSource*
serializeio_gzip_source (SerialSource* source)
{
  /* @@@ not implemented */
  return source;
}

/* Checksum sink
 */
typedef struct _ChecksumSink ChecksumSink;

static gboolean checksum_sink_close (SerialSink* sink);
static gboolean checksum_sink_write (SerialSink* sink, const guint8 *ptr, guint32 len);
static void     checksum_sink_free (SerialSink* sink);
static gboolean checksum_sink_quantum (SerialSink* sink);

struct _ChecksumSink
{
  SerialSink sink;

  SerialSink* out;

  EdsioMD5Ctx ctx;
  guint8      md5[16];
  gboolean    md5_done;
  gboolean    md5_written;
};

SerialSink*
serializeio_checksum_sink   (SerialSink* out)
{
  ChecksumSink* it = g_new0 (ChecksumSink, 1);
  SerialSink* sink = (SerialSink*) it;

  serializeio_sink_init (sink,
			 NULL,
			 checksum_sink_close,
			 checksum_sink_write,
			 checksum_sink_free,
			 checksum_sink_quantum);

  it->out = out;

  edsio_md5_init (& it->ctx);

  return sink;
}

gboolean
checksum_sink_write (SerialSink* fsink, const guint8 *ptr, guint32 len)
{
  ChecksumSink* sink = (ChecksumSink*) fsink;

  if (! sink->out->sink_write (sink->out, ptr, len))
    return FALSE;

  edsio_md5_update (& sink->ctx, ptr, len);

  return TRUE;
}

gboolean
checksum_sink_close (SerialSink* fsink)
{
  ChecksumSink* sink = (ChecksumSink*) fsink;

  if (! sink->md5_done)
    {
      edsio_md5_final (sink->md5, & sink->ctx);
      sink->md5_done = TRUE;
    }

  if (! sink->out->sink_write (sink->out, sink->md5, 16))
    return FALSE;

  if (! sink->out->sink_close (sink->out))
    return FALSE;

  return TRUE;
}

void
checksum_sink_free (SerialSink* fsink)
{
  ChecksumSink* sink = (ChecksumSink*) fsink;

  sink->out->sink_free (sink->out);

  g_free (sink);
}

gboolean
checksum_sink_quantum (SerialSink* fsink)
{
  ChecksumSink* sink = (ChecksumSink*) fsink;

  if (sink->out->sink_quantum)
    return sink->out->sink_quantum (sink->out);

  return TRUE;
}

/* Checksum source
 */

typedef struct _ChecksumSource ChecksumSource;

struct _ChecksumSource {
  SerialSource source;

  SerialSource *in;

  EdsioMD5Ctx ctx;
};

static gboolean     checksum_source_close        (SerialSource* source);
static gboolean     checksum_source_read         (SerialSource* source, guint8 *ptr, guint32 len);
static void         checksum_source_free         (SerialSource* source);

SerialSource*
serializeio_checksum_source (SerialSource* in0)
{
  ChecksumSource* it = g_new0 (ChecksumSource, 1);
  SerialSource* source = (SerialSource*) it;

  serializeio_source_init (source,
			   NULL,
			   checksum_source_close,
			   checksum_source_read,
			   checksum_source_free,
			   NULL,
			   NULL);

  it->in = in0;

  edsio_md5_init (& it->ctx);

  return source;
}

gboolean
checksum_source_close (SerialSource* fsource)
{
  ChecksumSource* source = (ChecksumSource*) fsource;
  guint8 buf1[16];
  guint8 buf2[16];

  if (! source->in->source_read (source->in, buf1, 16))
    return FALSE;

  edsio_md5_final (buf2, & source->ctx);

  if (memcmp (buf1, buf2, 16) != 0)
    {
      edsio_generate_void_event (EC_EdsioInvalidStreamChecksum);
      return FALSE;
    }

  if (! source->in->source_close (source->in))
    return FALSE;

  return TRUE;
}

gboolean
checksum_source_read (SerialSource* fsource, guint8 *ptr, guint32 len)
{
  ChecksumSource* source = (ChecksumSource*) fsource;

  if (! source->in->source_read (source->in, ptr, len))
    return FALSE;

  edsio_md5_update (& source->ctx, ptr, len);

  return TRUE;
}

void
checksum_source_free (SerialSource* fsource)
{
  ChecksumSource* source = (ChecksumSource*) fsource;

  source->in->source_free (source->in);

  g_free (source);
}
