/* -*-Mode: C++;-*-
 * $Id: xdelta2.cc 1.28 Fri, 29 Jun 2001 16:59:43 +0400 jmacd $
 *
 * Copyright (C) 1999, 2000, Joshua P. MacDonald <jmacd@CS.Berkeley.EDU>
 * and The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 *    Neither name of The University of California nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "xdfs_cpp.h"
#include "xdfs_comp.h"
#include "xdfs_list.h"
#include "getopt.h"
#include <errno.h>
#include <edsiostdio.h>

/* $Format: "const char* _xdfsapp_version = \"$ReleaseVersion$\"; " $ */
const char* _xdfsapp_version = "2.0-beta10"; 

typedef struct _Command Command;

struct _Command {
    char* name;
    int (* func) (TXN& txn, int argc, char** argv);
    int nargs;
    int flags;
};


#define COMMAND_ADMIN       (1 << 1)
#define COMMAND_KILLPROC    ((1 << 2) | COMMAND_ADMIN)
#define COMMAND_LISTPROC    ((1 << 3) | COMMAND_ADMIN)

#define COMMAND_CREATE      (1 << 4)
#define COMMAND_RECOVER     (1 << 5)
#define COMMAND_CHECKPOINT  (1 << 6)

#define COMMAND_PATHNAME (1 << 7)

static int register_command (TXN& txn, int argc, char** argv);
static int insert_command   (TXN& txn, int argc, char** argv);
static int retrieve_command (TXN& txn, int argc, char** argv);

static int xdfs_stat_command      (TXN& txn, int argc, char** argv);
static int xdfs_versions_command  (TXN& txn, int argc, char** argv);

static int listidx_command  (TXN& txn, int argc, char** argv);
static int printfs_command  (TXN& txn, int argc, char** argv);

static int listproc_command (DBFS &dbfs);
static int killproc_command (DBFS &dbfs);

static const Command commands[] =
{
    // The basic Xdelta compression ops... need to generalize
    { "register",   register_command,       1, COMMAND_PATHNAME }, /* register PATH */
    { "insert",     insert_command,         2, COMMAND_PATHNAME }, /* insert   PATH (infile|-) */
    { "retrieve",   retrieve_command,       3, COMMAND_PATHNAME }, /* retrieve PATH (NUMBER|DIGEST) (outfile|-) */

    // XDFS-specific operations
    { "xversions",  xdfs_versions_command,  1, COMMAND_PATHNAME },
    { "xstat",      xdfs_stat_command,      1, COMMAND_PATHNAME },

    // General DBFS operations
    { "listidx",    listidx_command,        0, 0 },
    { "printfs",    printfs_command,        0, 0 },

    // ADMIN commands (NULL proc means no transaction)
    { "create",     NULL,                   0, COMMAND_CREATE },
    { "recover",    NULL,                   0, COMMAND_RECOVER },
    { "checkpoint", NULL,                   0, COMMAND_CHECKPOINT },
    { "listproc",   NULL,                   0, COMMAND_LISTPROC },
    { "killproc",   NULL,                   0, COMMAND_KILLPROC },

    { NULL, NULL, 0 }
};

static struct option const long_options[] =
{
    {"sha",                 no_argument, 0, 'S' },
    {"md5",                 no_argument, 0, 'M' },
    {"help",                no_argument, 0, 'h' },
    {"version",             no_argument, 0, 'v' },
    {"logdir",              no_argument, 0, 'L' },
    {"recover",             no_argument, 0, 'r' },
    {"admin-recover",       no_argument, 0, 'R' },
    {"nuke",                no_argument, 0, 'n' },
    { 0,0,0,0 }
};

static const char* program_name   = "prog";
static const char* archiveloc     = NULL;
static int         register_flags = 0;

static void
usage ()
{
    NOPREFIX_OUT ("usage: %s COMMAND [OPTIONS] [ARG1 ...]\n", program_name);
    NOPREFIX_OUT ("use --help for more help\n");
    exit (2);
}

static void
version ()
{
    NOPREFIX_OUT ("version %s\n", _xdfsapp_version);
    exit (2);
}

static void
help ()
{
    NOPREFIX_OUT ("usage: %s COMMAND [OPTIONS] [ARG1 ...]", program_name);
    NOPREFIX_OUT ("COMMAND is one of:");
    NOPREFIX_OUT ("  create     FSHOME");
    NOPREFIX_OUT ("  register   FSHOME ARCHIVE_NAME");
    NOPREFIX_OUT ("  insert     FSHOME ARCHIVE_NAME VERSION_FILE");
    NOPREFIX_OUT ("  retrieve   FSHOME ARCHIVE_NAME VERSION_NUMBER OUTPUT_FILE");
    NOPREFIX_OUT ("  xversions  FSHOME ARCHIVE_NAME");
    NOPREFIX_OUT ("  xstat      FSHOME ARCHIVE_NAME");
    NOPREFIX_OUT ("  listidx    FSHOME");
    NOPREFIX_OUT ("  printfs    FSHOME");
    NOPREFIX_OUT ("  listproc   FSHOME");
    NOPREFIX_OUT ("  killproc   FSHOME");
    NOPREFIX_OUT ("  recover    FSHOME");
    NOPREFIX_OUT ("  checkpoint FSHOME");
    NOPREFIX_OUT ("OPTIONS are:");
    NOPREFIX_OUT ("  -v, --version");
    NOPREFIX_OUT ("  -h, --help");
    NOPREFIX_OUT ("  -M, --md5");
    NOPREFIX_OUT ("  -S, --sha");
    NOPREFIX_OUT ("  -r, --recover");
    NOPREFIX_OUT ("  -R, --admin-recover");
    NOPREFIX_OUT ("  -n, --nuke");
    NOPREFIX_OUT ("  -L, --logdir");
    NOPREFIX_OUT ("All commands operate on an FSHOME, a directory containing a file");
    NOPREFIX_OUT ("system volume.  A volume contains any number of archives, given");
    NOPREFIX_OUT ("by their ARCHIVE_NAME.");
    NOPREFIX_OUT ("The CREATE command creates a volume, and the REGISTER command creates");
    NOPREFIX_OUT ("an archive within that volume.  Optionally, the REGISTER command may");
    NOPREFIX_OUT ("be accompanied by --md5 or --sha, causing XDFS to index versions by the");
    NOPREFIX_OUT ("MD5 or SHA-1 message digest function.");
    NOPREFIX_OUT ("The INSERT and RETRIEVE commands correspond to RCS ci and co");
    NOPREFIX_OUT ("The XVERSIONS command lists versions in an archive (optionally, the message");
    NOPREFIX_OUT ("digest value).");
    NOPREFIX_OUT ("The LISTIDX command lists all the archives in a volume.");
    NOPREFIX_OUT ("The PRINTFS command prints the internal representation of the volume.");

    exit (2);
}

static char*
strip_leading_path (char* p)
{
    char* ls = strrchr (p, '/');

    if (ls)
	return ls + 1;

    return p;
}

static int
dotxn (DBFS &dbfs, const Command *cmd, int argc, char** argv)
{
    TXN txn;
    int ret;

    if (cmd->func) {
	if ((ret = txn.begin (dbfs, DBFS_TXN_SYNC))) {
	    return ret;
	}

	if ((ret = (* cmd->func) (txn, argc, argv))) {
	    return ret;
	}

	if ((ret = txn.commit ())) {
	    return ret;
	}
    }

    return 0;
}

int
main(int argc, char** argv)
{
    const Command *cmd = NULL;
    const char *fsdir, *logdir = NULL;
    int   c, ret = 0;
    int   longind;
    int   fatal_mask;
    int   oflags = 0;
    DBFS  dbfs (argc, argv);

    g_assert (argc > 0 && argv[0]);

    program_name = strip_leading_path (argv[0]);

    fatal_mask  = g_log_set_always_fatal ((GLogLevelFlags) G_LOG_FATAL_MASK);
    fatal_mask |= G_LOG_LEVEL_CRITICAL;

    g_log_set_always_fatal ((GLogLevelFlags) fatal_mask);

    if (argc < 2) {
	usage ();
    }

    for (cmd = commands; cmd->name; cmd += 1) {
	if (strcmp (cmd->name, argv[1]) == 0) {
	    break;
	}
    }

    if (strcmp (argv[1], "-h") == 0 ||
	strcmp (argv[1], "--help") == 0) {
	help ();
    }

    if (strcmp (argv[1], "-v") == 0 ||
	strcmp (argv[1], "--version") == 0) {
	version ();
    }

    if (cmd->name == NULL) {
	XDFS_ERROR ("unrecognized command: %s", argv[1]);
	help ();
    }

    argc -= 1;
    argv += 1;

    while ((c = getopt_long (argc,
			     argv,
			     "+hvSnrRML:",
			     long_options,
			     &longind)) != EOF) {
	switch (c) {
	case 'n': oflags        |= DBFS_CLEAR; break;
	case 'r': oflags        |= DBFS_RECOVER; break;
	case 'R': oflags        |= DBFS_RECOVER_ADMIN; break;
	case 'L': logdir         = g_strdup (optarg); break;
	case 'M': register_flags = XF_MD5Equality | XF_IdxReverse; break;
	case 'S': register_flags = XF_SHAEquality | XF_IdxReverse; break;
	case 'h': help (); break;
	case 'v': version (); break;
	default:
	    XDFS_ERROR ("illegal argument, use --help for help");
	    goto bail;
	}
    }

    argc -= optind;
    argv += optind;

    if (argc != cmd->nargs + 1) {
	XDFS_ERROR ("Wrong number of arguments");
	goto bail;
    }

    if ((ret = xdfs_library_init ())) {
	goto bail;
    }

    fsdir = argv[0];

    argc -= 1;
    argv += 1;

    if (cmd->flags & COMMAND_PATHNAME) {
	archiveloc = argv[0];
	argc -= 1;
	argv += 1;
    }

    if (cmd->flags & COMMAND_ADMIN) {
	oflags |= DBFS_ADMIN_TOOL;
    }

    if (cmd->flags & COMMAND_CREATE) {
	oflags |= DBFS_CREATE;
    }

    if (cmd->flags & COMMAND_RECOVER) {
	oflags |= DBFS_RECOVER;
    }

    if ((ret = dbfs.open (fsdir, logdir, oflags))) {
	goto bail;
    }

    if ((ret = dotxn (dbfs, cmd, argc, argv))) {
	goto bail;
    }

    if ((cmd->flags == COMMAND_LISTPROC) && (ret = listproc_command (dbfs))) {
	goto bail;
    }

    if ((cmd->flags == COMMAND_KILLPROC) && (ret = killproc_command (dbfs))) {
	goto bail;
    }

    if ((ret = dbfs.close ())) {
	goto bail;
    }

    return 0;

  bail:

    if (ret != 0) {
	XDFS_ERROR (ret) ("command failed");
    }

    dbfs.close ();

    return 2;
}

int
register_command (TXN& txn, int argc, char** argv)
{
    MAJORC        loc_node;
    SAREA         area;
    XdfsParams    params;
    XdfsLocation  loc;
    CIRCULAR_LIST cll;
    int           ret;

    memset (& params, 0, sizeof (params));

    // If you wanted to change any of the XDFS compression settings,
    // here is where you would do it.
    params.flags = register_flags;

    DEBUG_XDFS ("create XDFS location: %s", xdfs_flags_to_string (params.flags));

    if ((ret = txn.create_area (area))) {
	PROP_ERROR (ret) ("txn_create_area");
	return ret;
    }

    if ((ret = loc.create (area, & params, loc_node))) {
	PROP_ERROR (ret) ("xdfs_loc_create");
	return ret;
    }

    if ((ret = cll.create (loc_node))) {
	PROP_ERROR (ret) ("cll_create");
	return ret;
    }

    if ((ret = txn.root ().dir_link_insert (archiveloc, loc_node, DBFS_NOOVERWRITE))) {
	if (ret != DBFS_EXISTS) {
	    PROP_ERROR (ret) ("root_dir_link_insert");
	} else {
	    XDFS_ERROR (ret) ("register: %s", archiveloc);
	}

	return ret;
    }

    return 0;
}

static int
find_loc (TXN &txn, MAJORC &loc_node, XdfsLocation &loc)
{
    int ret;

    if ((ret = txn.root ().dir_link_lookup (archiveloc, loc_node, DBFS_NOFLAG))) {
	PROP_ERROR (ret) ("dir_link_lookup");
	return ret;
    }

    if ((ret = loc.open (loc_node))) {
	PROP_ERROR (ret) ("xdfs_loc_open");
	return ret;
    }

    return 0;
}

int
insert_command (TXN& txn, int argc, char** argv)
{
    MAJORC        loc_node;
    MAJORC        ins_node;
    FileHandle   *insert_handle;
    XdfsLocation  loc;
    CIRCULAR_LIST cll;
    int           ret;

    if (strcmp (argv[0], "-") == 0) {
	insert_handle = _stdin_handle;
    } else {

	if (! (insert_handle = handle_read_file (argv[0]))) {
	    PROP_ERROR ("handle_read_file: %s", argv[0]);
	    return -1;
	}
    }

    if ((ret = find_loc (txn, loc_node, loc))) {
	PROP_ERROR (ret) ("find_loc");
	return ret;
    }

    if ((ret = cll.open (loc_node))) {
	PROP_ERROR (ret) ("cll_open");
	return ret;
    }

    if ((ret = loc.insert_version (insert_handle, ins_node))) {
	PROP_ERROR (ret) ("xdfs_loc_insert_version");
	return ret;
    }

    if ((ret = cll.push_back (ins_node)) && (ret != DBFS_EXISTS)) {
	PROP_ERROR (ret) ("cll_push_back");
	return ret;
    }

    handle_free (insert_handle);

    return 0;
}

int
retrieve_command (TXN& txn, int argc, char** argv)
{
    const char   *vstr = argv[0];
    guint         version;
    MAJORC        retr;
    MAJORC        loc_node;
    FileHandle   *src_handle;
    FileHandle   *out_handle;
    const MessageDigest *md;
    XdfsLocation  loc;
    int           ret;

    if ((ret = find_loc (txn, loc_node, loc))) {
	return ret;
    }

    md = loc.message_digest ();

    if (! strtoui_checked (vstr, & version, NULL)) {

	CIRCULAR_LIST cll;

	if ((ret = cll.open (loc_node))) {
	    PROP_ERROR (ret) ("cll_open");
	    return ret;
	}

	if ((ret = cll.find_nth (version, retr))) {
	    XDFS_ERROR (ret) ("Version number %d", version);
	    return ret;
	}

    } else if (loc.has_idx () && strlen (vstr) == (md->cksum_len*2)) {
	// Its a digest value
	NODEC  idx = loc.index_dir ();
	guint8 dehex[128];

	g_assert (md->cksum_len < 128);

	if (! edsio_digest_from_string (md, dehex, vstr)) {
	    return EINVAL;
	}

	if ((ret = idx.dir_link_lookup (DKEY (dehex, md->cksum_len), retr, 0))) {
	    XDFS_ERROR (ret) ("Digest value %s", vstr);
	    return ret;
	}

    } else {
	XDFS_ERROR ("Invalid version label: should be number%s: %s",
		    (loc.has_idx () ? " or digest value" : ""),
		    vstr);
	return EINVAL;
    }

    if ((ret = retr.read_segment (& src_handle, DBFS_NOFLAG))) {
	return ret;
    }

    if (strcmp (argv[1], "-") == 0) {
	out_handle = _stdout_handle;
    } else {
	if (! (out_handle = handle_write_file (argv[1]))) {
	    return -1;
	}
    }

    if (! handle_drain (src_handle, out_handle)) {
	return -1;
    }

    if (! handle_close (src_handle)) {
	return -1;
    }

    handle_free (src_handle);

    if (! handle_close (out_handle)) {
	ret = -1;
    }

    handle_free (out_handle);

    return ret;
}

int
xdfs_stat_command (TXN& txn, int argc, char** argv)
{
    MAJORC        loc_node;
    XdfsLocation  loc;
    int           ret;

    if ((ret = find_loc (txn, loc_node, loc))) {
	return ret;
    }

    if ((ret = loc.print (_stdout_handle))) {
	return ret;
    }

    return 0;
}

int
xdfs_versions_command (TXN& txn, int argc, char** argv)
{
    XdfsLocation  loc;
    MAJORC        loc_node;
    CIRCULAR_LIST cll;
    int           version = 0;
    int           ret;
    MAJORC        node;

    const MessageDigest *md;

    if ((ret = find_loc (txn, loc_node, loc))) {
	return ret;
    }

    if ((ret = cll.open (loc_node))) {
	return ret;
    }

    md = loc.message_digest ();

    INFO_XDFS ("Version  Length  %s", md ? md->name : "");

    while (cll.next (node)) {
	char    sbuf[64];

	if (md) {
	    DKEY  digval;
	    NODEC idx;

	    idx = loc.index_dir ();

	    if ((ret = idx.dir_link_invert (node, digval))) {
		return ret;
	    }

	    g_assert (digval.size () == md->cksum_len);

	    edsio_digest_to_string (md, digval.data (), sbuf);
	} else {
	    sbuf[0] = 0;
	}

	INFO_XDFS ("%7d %7d  %s", version++, node.length ().key (), sbuf);
    }

    return 0;
}

int
listidx_command (TXN& txn, int argc, char** argv)
{
    DIRC   curs;
    int    ret;

    // @@@ This should be generalized for arbitrary paths
    if ((ret = txn.root ().dir_open (curs))) {
	return ret;
    }

    while (curs.next ()) {
	DKEY key;

	if ((ret = curs.get_key (key))) {
	    return ret;
	}

	NOPREFIX_OUT (key);
    }

    if ((ret = curs.close ())) {
	PROP_ERROR (ret) ("curs_close");
	return ret;
    }

    return 0;
}

int
printfs_command (TXN& txn, int argc, char** argv)
{
    int ret;

    if ((ret = txn.printfs (0))) {
	PROP_ERROR (ret) ("printfs");
	return ret;
    }

    return 0;
}

int
listproc_command (DBFS &dbfs)
{
    int ret;
    LISTPROC_DATA data;

    if ((ret = dbfs.admin_listproc (data))) {
	PROP_ERROR (ret) ("admin_listproc");
	return ret;
    }

    NOPREFIX_OUT ("PID\tSTATUS\t\tCOMMAND");

    for (int i = 0; i < data.size (); i += 1) {

	NOPREFIX_OUT ("%d\t%s\t\t", data.pid (i), (data.exists (i) ? "alive" : "unknown")) (data.desc (i));
    }

    return 0;
}

int
killproc_command (DBFS &dbfs)
{
    int ret;
    KILLPROC_DATA data;

    if ((ret = dbfs.admin_killproc (data))) {
	PROP_ERROR (ret) ("admin_killproc");
	return ret;
    }

    NOPREFIX_OUT ("PID\tSTATUS\t\tCOMMAND");

    for (int i = 0; i < data.size (); i += 1) {

	const char *stat = "unknown";

	switch (data.status (i)) {
	case KILLPROC_SIGTERM: stat = "sigterm"; break;
	case KILLPROC_SIGKILL: stat = "sigkill"; break;
	case KILLPROC_NOTFOUND: stat = "notfound"; break;
	case KILLPROC_PERMISSION: stat = "permission"; break;
	case KILLPROC_NODEATH: stat = "nodeath"; break;
	}

	NOPREFIX_OUT ("%d\t%s\t\t", data.pid (i), stat) (data.desc (i));
    }

    return 0;
}
