// -*- mode: c++; c-file-style: "linux"; c-basic-offset: 2; indent-tabs-mode: nil -*-
//
//  Copyright (C) 2006-2015 Andrej Vodopivec <andrej.vodopivec@gmail.com>
//            (C) 2012 Doug Ilijev <doug.ilijev@gmail.com>
//            (C) 2014-2015 Gunter Königsmann <wxMaxima@physikbuch.de>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//

#ifndef EDITORCELL_H
#define EDITORCELL_H

#include "MathCell.h"

#include <vector>
#include <list>
#include <vector>
#include <wx/tokenzr.h>

/*! \file

  This file contains the definition of the class EditorCell
 */

/*! This class defines what the user sees as input cell
  
  This class handles input cells including:
    - The per-cell undo buffer
    - The handling of key presses when this cell is active
  
  Since Unicode doesn't provide us with a "soft linebreak" letter we 
  use <code>\r</code> as a marker that this line has to be broken here if we
  don't want it to extend beyond the right margin of the screen.

  In a few places we use wxString::iterator instead of accessing individual 
  letters within the string using the [] operator. This might look overly
  complicated. But in UTF-8 all non-standard-ASCII-characters use more than one
  byte making iterating over every single char of the string the only way of
  determining which address char n is at. An iterator is the only way of not
  having to determine the address of every single char indepently. 
 */
class EditorCell : public MathCell
{
private:
  int ChangeNumpadToChar(int c);
  //! A list of all potential autoComplete targets within this cell
  wxArrayString m_wordList;
 //! Draw a box that marks the current selection
  void MarkSelection(long start, long end,double scale, wxDC& dc, TextStyle style,int fontsize);
  /*! The start of the current selection.

     - >0: the position of the cursors in characters from start
     - -1: Currently no selection is active

     If the selection has been done from right to left m_selectionStart>m_selectionEnd.
   */
  long m_selectionStart;
  /*! The end of the current selection.

     - >0: the position of the cursors in characters from start
     - -1: Currently no selection is active

     If the selection has been done from right to left m_selectionStart>m_selectionEnd.
   */
  long m_selectionEnd;
  /*! The currently selected string. 

    Since this string is defined statically it is available in every editor cell
    for highlighting other instances of the selected string.
  */
  static wxString m_selectionString;
  long m_oldSelectionStart;
  long m_oldSelectionEnd;
  /*! Replace a "*" by a centered dot?
    
    Normally we ask the parser for this piece of information. But during recalculation
    of widths while selecting text we don't know our parser.
   */
  bool m_changeAsterisk;
public:
  //! Set the string that is to be highlighted as "identical to the curent selection"
  static void SetSelectionString(wxString string)
    {m_selectionString = string;}
  //! A list of words that might be applicable to the autocomplete function.
  wxArrayString GetWordList(){return m_wordList;}
  //! Has the selection changed since the last draw event?
  bool m_selectionChanged;
  //! The constructor
  EditorCell(wxString text = wxEmptyString);
  //! The destructor
  ~EditorCell();

  /*! Expand all tabulators.

    \param input The string the tabulators should be expanded in
    \param posInLine The number of characters that come before the input in the same line
    \todo Implement the actual TAB expansion
  */
  wxString TabExpand(wxString input, long posInLine);
  //! Escape all chars that cannot be used in HTML otherwise
  static wxString EscapeHTMLChars(wxString input);
  //! Convert all but the first of a row of multiple spaces to non-breakable
  static wxString PrependNBSP(wxString input);
  void Destroy();
  MathCell* Copy();
  //! Recalculate the widths of the current cell.
  void RecalculateWidths(int fontsize);
  void Draw(wxPoint point, int fontsize);
  wxString ToString();
  /*! Convert the current cell to a string
  
    \param dontLimitToSelection
    - false: If text is selected return only the selected text
    - true:  Always return all text in this text cell
  */
  wxString ToString(bool dontLimitToSelection);
  //! Convert the current cell to LaTeX code
  wxString ToTeX();
  //! Convert the current cell to XML code for inclusion in a .wxmx file.
  wxString ToXML();
  //! Convert the current cell to HTML code.
  wxString ToHTML();
  //! Convert the current cell to RTF code
  wxString ToRTF();
  //! Set the currently used font to the one that matches this cell's formatting
  void SetFont(int fontsize);
  //! Sets the current color to this cell's foreground color
  void SetForeground();

  /*! Sets the text that is to be displayed.
    
    Automatically calls StyleText().
   */
  void SetValue(const wxString &text);
  /*! Returns the text contained in this cell

    Naturally all soft line breaks are converted back to spaces beforehand.
   */
  wxString GetValue()
  {
    return m_text;
  }
  /*! Converts m_text to a list of styled text snippets that will later be displayed by draw().

    This function also generates a wordlist for this EditorCell so Autocompletion can learn
    about variable names contained in lists or cells that still haven't been evaluated.

    For cells containing text instead of code this function adds a <code>\r</code> as a marker
    that this line is to be broken here until the window's width changes.
   */
  void StyleText();
  void Reset();
  //! Decide what to do if the user pressed a key when this cell was selected
  void ProcessEvent(wxKeyEvent& event);
  //! If the cell is activated it contains a blinking cursor.
  bool ActivateCell(bool active);
  //! Return the index of the 1st char of the line containing the letter #pos.
  size_t BeginningOfLine(long pos);
  //! Return the index of the last char of the line containing the letter #pos,
  size_t EndOfLine(long pos);
  //! Adds a ";" to the end of the last command in this cell in case that it doesn't end in $ or ;
  bool AddEnding();
  //! Determines which line and column the pos'th char is at.
  void PositionToXY(int pos, unsigned int* line, unsigned int* col);
  //! Determines which index the char at the position "x chars left, y chars down" is at.
  int XYToPosition(int x, int y);
  //! The screen coordinates of the cursor
  wxPoint PositionToPoint(int fontsize, int pos = -1);
  //! Sets the cursor to the screen coordinate point
  void SelectPointText(wxDC &dc, wxPoint& point);
  //! Selects the text beween the screen coordinates one and two
  void SelectRectText(wxDC &dc, wxPoint& one, wxPoint& two);
  //! Selects the word the cursor is currently at.
  wxString SelectWordUnderCaret(bool selectParens = true, bool toRight = true);
  //! Is the point point inside the currently selected text?
  bool IsPointInSelection(wxDC& dc, wxPoint point);
  bool CopyToClipboard();
  bool CutToClipboard();
  void PasteFromClipboard(bool primary = false);
  //! Get the character position the selection has been started with
  int GetSelectionStart(){return m_selectionStart;}
  //! Get the character position the selection has been ended with
  int GetSelectionEnd(){return m_selectionEnd;}
  //! Select the whole text contained in this Cell
  void SelectAll()
  {
    m_selectionStart = 0;
    m_selectionEnd = m_positionOfCaret = m_text.Length();
  }
  //! Does the selection currently span the whole cell?
  bool AllSelected()
    {
      return (m_selectionStart==0)&&(m_selectionEnd == (long) m_text.Length());
    }
  //! Unselect everything.
  void SelectNone()
  {
    m_selectionStart = m_selectionEnd = 0;
  }
  //! Is there any text selected right now?
  bool SelectionActive()
  {
    return (m_selectionStart >= 0)&&(m_selectionEnd >= 0);
  }

  bool CanCopy()
  {
    return m_selectionStart != -1;
  }
  bool FindMatchingQuotes();
  void FindMatchingParens();
  int GetLineWidth(wxDC& dc,unsigned int line, int end);
  //! true, if this cell's width has to be recalculated.
  bool IsDirty()
  {
    return m_isDirty;
  }

  //! Toggles the visibility of the cursor which is used to make it blink.
  void SwitchCaretDisplay()
  {
    m_displayCaret = !m_displayCaret;
  }
  void SetFocus(bool focus)
  {
    m_hasFocus = focus;
  }
  void SetFirstLineOnly(bool show = true) {
    if (m_firstLineOnly != show) { m_width = m_height = -1; m_firstLineOnly = show; }
    // Style the text anew.
    StyleText();
  }
  bool IsActive() { return m_isActive; }
  //! Is the cursor at the start of this cell?
  bool CaretAtStart() { return m_positionOfCaret == 0; }
  //! Move the cursor to the start of this cell
  void CaretToStart();
  //! Is the cursor at the end of this cell?
  bool CaretAtEnd() { return m_positionOfCaret == (long)m_text.Length(); }
  //! Move the cursor to the end of this cell
  void CaretToEnd();
  //! Move the cursor to a certain position in the cell
  void CaretToPosition(int pos);
  //! True, if there is undo information for this cell
  bool CanUndo();
  //! Issue an undo command
  void Undo();
  //! True, if a redo can be done for this cell.
  bool CanRedo();
  //! Issu a redo command
  void Redo();
  //! Save the current contents of this cell in the undo buffer.
  void SaveValue();
  /*! DivideAtCaret
    Returns the string from caret to end and
    modifies the m_text so it contains only the string
    from beginning to caret
    Used for 'Divide Cell', called from MathCtrl
  */
  wxString DivideAtCaret();
  void CommentSelection();
  void ClearUndo();
  //! Query if this cell needs to be re-evaluated by maxima
  bool ContainsChanges() { return m_containsChanges; }
  //! Set the information if this cell needs to be re-evaluated by maxima
  void ContainsChanges(bool changes) { m_containsChanges = m_containsChangesCheck = changes; }
  bool CheckChanges();
  /*! Replaces all occurrences of a given string

    TODO: Implement the IgnoreCase case.
   */
  int ReplaceAll(wxString oldString, wxString newString,bool IgnoreCase);
  /*! Finds the next occurrences of a string

    \param str The string to search for
    \param down 
     - true: search downwards
     - false: search upwards
    \param ignoreCase
     - true: Case-insensitive search
     - false: Case-sensitive search
   */
  bool FindNext(wxString str, bool down, bool ignoreCase);
  void SetSelection(int start, int end);
  void GetSelection(int *start, int *end)
  {
    *start = m_selectionStart; *end = m_selectionEnd;
  }
  /*! Replace the current selection with a string

    \param oldStr The old string in the selection. If this string doesn't match
                  the selection this function doesn't replace it.
    \param newString The new string oldStr has to be replaced by
    \param keepSelected 
      - true = we want the new string to be selected afterwards
      - false = the selection is cleared after replacing the string
        and moving the cursor to its end.
    \param ignoreCase
   */
  bool ReplaceSelection(wxString oldStr, wxString newString,bool keepSelected = false,bool ignoreCase = false);
  //! Convert the current selection to a string
  wxString GetSelectionString();
  //! Unselect everything
  void ClearSelection();
  //! Get the cursor's current position inside the cell.
  int GetCaretPosition() { return m_positionOfCaret; }
  bool FindNextTemplate(bool left = false);
  void InsertText(wxString text);
  wxString TextInFrontOfSelection()
    {
      return GetValue().Mid(1,m_selectionStart);
    }
  //! Return to the selection after the cell has been left upwards
  void ReturnToSelectionFromTop()
    {
      SetSelection(m_lastSelectionStart,0);
    }
  //! Return to the selection after the cell has been left downwards
  void ReturnToSelectionFromBot()
    {
      SetSelection(m_lastSelectionStart,m_text.Length());
    }
private:
  /*! Divide a string into tokens

    Used when styling text.
   */
  wxArrayString StringToTokens(wxString string);

#if defined __WXMAC__
  bool HandleCtrlCommand(wxKeyEvent& ev);
#endif
  bool HandleSpecialKey(wxKeyEvent& ev);
  bool HandleOrdinaryKey(wxKeyEvent& ev);
  
  bool IsAlpha(wxChar c);
  bool IsNum(wxChar c);
  bool IsAlphaNum(wxChar c);
  
  /*! A piece of styled text for syntax highlighting

   */
  class StyledText
  {
  private:
    //! The color of this text portion
    TextStyle  m_style;
    //! The text of this text portion
    wxString m_text;
    //! Do we really want to style this text portion different than the default?
    bool m_styleThisText;
    //! By How many pixels we want to indent this line?
    int m_indentPixels;
    //! Chars that mark continued indentation
    wxString m_indentChar;
  public:    
    //! Defines a piece of styled text
    StyledText(TextStyle style,wxString text)
      {
        m_text = text;
        m_style = style;
        m_styleThisText = true;
      }

    /*! Defines a piece of text with the default style that possibly is indented

      \todo Currently we access every char of the string by telling wxWidgets that we want
      char #n. The wxWidgets docs tell that using an iterator is much more efficient.
     */
    StyledText(wxString text,int indentPixels = 0,wxString indentChar=wxEmptyString)
      {
        m_text = text;
        m_style = TS_DEFAULT;
        m_styleThisText = false;
        m_indentPixels = indentPixels;
        m_indentChar = indentChar;
      }
    //! Returns the piece of text
    wxString GetText()
      {
        return m_text;
      }
    //! By how many pixels do we need to indent this line due to a bullet list or similar?
    int GetIndentPixels()
      {
        return m_indentPixels;
      }

    wxString GetIndentChar()
      {
        return m_indentChar;
      }
//! If StyleSet() is true this function returns the color of this text portion
    TextStyle GetStyle()
      {
        return m_style;
      }
    // Has a individual text style been set for this text portion?
    bool StyleSet()
      {
        return m_styleThisText;
      }
  };
  
  std::vector<StyledText> m_styledText;

#if wxUSE_UNICODE
  /*! Handle ESC shortcuts for special characters

    These characters can be tought to LaTeX and the html browser if necessary in
    TextCell::ToTeX and EditorCell::ToTeX. They can also be
    converted to maxima strings in wxMaxima::SendMaxima.
   */
  wxString InterpretEscapeString(wxString txt);
#endif
  wxString m_text;
  wxArrayString m_textHistory;
  std::vector<int> m_positionHistory;
  std::vector<int> m_startHistory;
  std::vector<int> m_endHistory;
  ptrdiff_t m_historyPosition;
  //! Where inside this cell is the cursor?
  int m_positionOfCaret;
  int m_caretColumn;
  long m_lastSelectionStart;
//  long m_oldStart, m_oldEnd;
  unsigned int m_numberOfLines;
  bool m_isActive;
  int m_fontSize;
  /*! The font size we were called with  the last time

    We need to know this in order to be able to detect we need a full recalculation.
   */
  int m_fontSize_Last;
  int m_charHeight;
  int m_paren1, m_paren2;
  //! Does this cell's size have to be recalculated?
  bool m_isDirty;
  bool m_displayCaret;
  bool m_hasFocus;
  wxFontStyle m_fontStyle;
  wxFontWeight m_fontWeight;
  bool m_underlined;
  wxString m_fontName;
  wxFontEncoding m_fontEncoding;
  bool m_saveValue;
  //! true, if this function has changed since the last evaluation by maxima
  bool m_containsChanges;
  bool m_containsChangesCheck;
  bool m_firstLineOnly;
};

#endif // EDITORCELL_H
