/* deeppathstore.cc
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2012-2015 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "deeppathstore.hh"

DeepPathStore::DeepPathStore()
{
}

void DeepPathStore::storePath( const std::string &path, time_t ts, DeepPathNode::node_change_t *changed )
{
    storePath( m_root, path, ts, 0, changed );
}

static void push_leafs( const DeepPathNode &node,
                        std::list< std::pair< std::string, time_t > > &l,
                        const std::string &prefix )
{
    std::string new_prefix( prefix );

    new_prefix += node.get_name();

    if ( node.get_children().empty() ) {
        if ( ! new_prefix.empty() ) {
            l.push_back( std::make_pair( new_prefix,
                                         node.get_ts() ) );
        }
    } else {
        for ( std::list< DeepPathNode >::const_iterator it1 = node.get_children().begin();
              it1 != node.get_children().end();
              it1++ ) {
            push_leafs( *it1, l, new_prefix );
        }
    }
}

std::list< std::pair< std::string, time_t > > DeepPathStore::getPaths()
{
    std::list< std::pair< std::string, time_t > > l;

    push_leafs( m_root, l, "" );

    return l;
}

void DeepPathStore::storePath( DeepPathNode &node,
                               const std::string &path,
                               time_t ts,
                               std::string::size_type start_pos,
                               DeepPathNode::node_change_t *changed )
{
    if ( start_pos == std::string::npos ) {
        if ( node.get_ts() != ts ) {
            node.set_ts( ts );
            if ( changed ) {
                if ( *changed == DeepPathNode::NODE_UNCHANGED ) {
                    *changed = DeepPathNode::NODE_UPDATED;
                }
            }
        }
        return;
    }

    std::string::size_type slash = path.find( '/', start_pos );
        
    if ( slash == start_pos ) {
        slash = path.find( '/', start_pos + 1 );
    }

    int basename_length;

    if ( slash == std::string::npos ) {
        basename_length = path.length() - start_pos;
    } else {
        basename_length = slash - start_pos;
    }

    if ( basename_length < 1 ) return;

    std::string basename( path, start_pos, basename_length );

    if ( ! ( path[start_pos] == '/' && basename_length == 1 ) ) {
        storePath( node.lookup_or_insert( basename, ts, changed ),
                   path, ts, slash, changed );
    } else {
        if ( slash != std::string::npos ) {
            storePath( node, path, ts, slash, changed );
        }
    }
}

void DeepPathStore::clear()
{
    m_root = DeepPathNode();
}
