/* normalmode.cc
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2001-2012 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "normalmode.h"
#include "worker_locale.h"
#include "wconfig.h"
#include "worker.h"
#include "configheader.h"
#include "configparser.tab.h"
#include "nmrowdata.h"
#include "basic_actions.h"
#include "startprogop.h"
#include <aguix/acontainer.h>
#include <aguix/acontainerbb.h>
#include "wcfiletype.hh"
#include "copyopwin.hh"
#include "nwc_fsentry.hh"
#include "run_custom_action.hh"
#include "filenameshrinker.hh"
#include "bookmarkdbentry.hh"
#include "nmbookmarkcb.hh"
#include "bookmarkdbproxy.hh"
#include "datei.h"
#include "deleteop.h"
#include "nmcacheentry.hh"
#include "nmspecialsourceext.hh"
#include <algorithm>
#include <aguix/karteibutton.h>
#include "nwc_path.hh"
#include "method_callback.hh"
#include <argclass.hh>
#include "string_completion.hh"
#include <aguix/utf8.hh>
#include "nmfieldlistviewdnd.hh"
#include "simplelist.hh"
#include <aguix/stringgadget.h>
#include <aguix/button.h>
#include <aguix/choosebutton.h>
#include "dnd.h"
#include "fileentry.hh"
#include "stringcomparator.hh"
#include <set>
#include "persdeeppathstore.hh"

const char *NormalMode::type="NormalMode";

int NormalMode::maxfilestrlen=1;
int NormalMode::maxdirstrlen=1;
int NormalMode::maxbytesstrlen=1;
int NormalMode::maxwbytesstrlen=1;
int NormalMode::maxnochanges=0;

NormalMode::NormalMode(Lister *tparent):ListerMode(tparent)
{
    ce=NULL;
    cache=new List();
    req=new Requester(aguix);
    searchmodeon=false;
    lastsearch=NULL;
    sortmode=SORT_NAME;
    lv=NULL;
    lastrow=-1;
    showfreespace=true;
    lastfsupdate=0;
    updatetime=10;
    parentlister->getWorker()->PS_setLifetime( (double)updatetime );
    ssh_allow = SSH_ASK;

    lasteagain = false;
  
    // to preventing slave access
    reclist = new ft_rec_list();

    m_ft_thread.setWorker( parentlister->getWorker() );
    ft_list_update();
    m_ft_thread.start();
    m_ft_thread.switchOrder( NM_Filetype_Thread::THREAD_RUN );
  
    chownUserList = NULL;
    chownGroupList = NULL;

    visChanged = true;
    oldlvy = -1;
  
    busyFlag = 0;
    lastBusyFlagUpdate = time( NULL );

    quicksearch_enabled = true;
    _filtered_search_enabled = true;
    m_filtered_search_active = false;

    _lv_ids_serial = -1;
  
    m_last_statebar_update = (time_t)0;

    m_bookmarks_has_been_changed = false;
    m_bookmark_cb = new NMBookmarkCB( this );
    Worker::getBookmarkDBInstance().registerChangeCallback( m_bookmark_cb );

    m_cont = NULL;
    m_cont2 = NULL;
    m_tab_b = NULL;
    m_tab_new = NULL;
    m_tab_close = NULL;

    m_off_store.active_tab = -1;

    m_shrinker = RefCount<TextShrinker>( new HalfShrinker() );
    m_filtered_search_infix_search = false;

    m_current_space_update_ms = 0;

    registerCommand( "new_tab", new MethodCallback<NormalMode, &NormalMode::newTab>( this ) );
    registerCommand( "enter_dir", new MethodCallbackArg<NormalMode, const std::list< RefCount< ArgClass > > &,
                     &NormalMode::enter_dir>( this ) );

    m_flexible_matching_active = false;
    m_flexible_matching_enabled = true;
}

NormalMode::~NormalMode()
{
    m_ft_thread.switchOrder( NM_Filetype_Thread::THREAD_EXIT );
    m_ft_thread.join();
    ft_list_clear();
    ft_rec_list_clear();
    ft_recres_list_clear();
    delete reclist;

    if(searchmodeon==true) finishsearchmode();
    deleteCache();
    delete cache;
    delete req;

    Worker::getBookmarkDBInstance().unregisterChangeCallback( m_bookmark_cb );
    delete m_bookmark_cb;
}

void NormalMode::messageHandler(AGMessage *msg)
{
    bool ma=false;
    switch(msg->type) {
        case AG_STRINGGADGET_OK:
            if(searchmodeon==false) {
                if(msg->stringgadget.sg==sg) {
                    //TODO: better not replacing env vars at all here?
                    std::string tstr1 = sg->getText();
                    if ( tstr1[0] == '$' ) {
                        char *tstr2 = HandlePathExt( tstr1.c_str() );
                        tstr1 = tstr2;
                        _freesafe( tstr2 );
                    }
                    enterPath( tstr1.c_str() );
                }
                break;
            }
        case AG_STRINGGADGET_CANCEL:
            if(searchmodeon==false) {
                if(msg->stringgadget.sg==sg) {
                    if ( getCurrentDir() != NULL ) sg->setText( getCurrentDir() );
                }
                break;
            }
        case AG_STRINGGADGET_DEACTIVATE:
            if(searchmodeon==true) {
                finishsearchmode();
            }
            break;
        case AG_STRINGGADGET_CONTENTCHANGE:
            if ( searchmodeon == true ) {
                std::string newtext = getSearchModeString();
                if ( shownextentry( newtext.c_str(), false, true ) == true ) {
                    if ( lastsearch != NULL )
                        _freesafe( lastsearch );
                    else
                        debugmsg("oops1\n");
                    lastsearch = dupstring( newtext.c_str() );
                } else {
                    resetToLastSearchString();
                }
            }
            break;
        case AG_SIZECHANGED:
            break;
        case AG_STRINGGADGET_ACTIVATE:
            if(msg->stringgadget.sg==sg) ma=true;
            break;
        case AG_BUTTONCLICKED:
            if(msg->button.button==hb[0]) {
                showcache(-1);
                ma=true;
            } else if(msg->button.button==hb[1]) { 
                showcache(1);
                ma=true;
            } else if ( msg->button.button == parentb ) { 
                parent();
                ma = true;
            } else if ( msg->button.button == m_tab_new ) {
                newTab();
            } else if ( msg->button.button == m_tab_close ) {
                closeCurrentTab();
            }
            break;
        case AG_FIELDLV_PRESSED:
            if(msg->fieldlv.lv==lv) ma=true;
            break;
        case AG_FIELDLV_ONESELECT:
            if(msg->fieldlv.lv==lv) {
                if(lastrow==-1) {
                    lastrow=msg->fieldlv.row;
                    gettimeofday(&lastclick,NULL);
                } else {
                    if(msg->fieldlv.row==lastrow) {
                        struct timeval acttime;
                        gettimeofday(&acttime,NULL);
                        if ( aguix->isDoubleClick( &acttime, &lastclick ) == true ) {
                            //TODO: Wir haben einen Doppelclick
                            // Aus lvc->data==fe->nr bekommen wir das fe (nr beschreibt Position)
                            if(ce!=NULL) {
                                FileEntry *tfe = getFEForRow( lastrow );
                                startAction(tfe);
                            }
                            // when had a doubleclick the next click on the same should not be a
                            // doubleclick again
                            lastrow=-1;
                        } else {
                            lastrow=msg->fieldlv.row;
                            gettimeofday(&lastclick,NULL);
                        }
                    } else {
                        lastrow=msg->fieldlv.row;
                        gettimeofday(&lastclick,NULL);
                    }
                }
            }
            break;
        case AG_FIELDLV_HEADERCLICKED:
            if ( ( msg->fieldlv.lv == lv ) &&
                 ( msg->fieldlv.button == Button1 ) ) {
                changeSortModeForField( msg->fieldlv.row );
            }
            break;
        case AG_FIELDLV_ENTRY_PRESSED:
            if ( msg->fieldlv.lv == lv ) {
                lv_pressed( msg->fieldlv.row );
            }
            break;
        case AG_POPUPMENU_CLICKED:
        case AG_POPUPMENU_ENTRYEDITED:
            if ( msg->popupmenu.menu == m_current_popup_settings.lv_popup_menu ) {
                startLVPopUpAction( msg );
            } else if ( msg->popupmenu.menu == m_current_popup_settings.label_popup_menu.get() ) {
                handleLabelPopUp( msg );
            } else if ( msg->popupmenu.menu == m_current_popup_settings.tab_popup_menu.get() ) {
                handleTabPopUp( msg );
            }
            break;
        case AG_KEYPRESSED:
            if ( parentlister->isActive() == true ) {
                if ( msg->key.key == XK_Prior && KEYSTATEMASK( msg->key.keystate ) == ControlMask ) {
                    switchToTabDirection( -1 );
                } else if ( msg->key.key == XK_Next && KEYSTATEMASK( msg->key.keystate ) == ControlMask ) {
                    switchToTabDirection( 1 );
                } else if ( msg->key.key == XK_t && KEYSTATEMASK( msg->key.keystate ) == ControlMask ) {
                    newTab();
                } else if ( msg->key.key == XK_w && KEYSTATEMASK( msg->key.keystate ) == ControlMask ) {
                    closeCurrentTab();
                } else if ( searchmodeon == true &&
                            msg->key.key == XK_Tab &&
                            KEYSTATEMASK( msg->key.keystate ) == 0 ) {
                    if ( ! m_last_completion.empty() ) {
                        std::string s = getSearchModeString();
                        s += m_last_completion;

                        sg->setText( s.c_str() );
                        sg->setCursor( s.length() );

                        if ( lastsearch != NULL )
                            _freesafe( lastsearch );
                        lastsearch = dupstring( s.c_str() );

                        shownextentry( s.c_str(), false, true );
                    }
                } else if ( searchmodeon == false &&
                            msg->key.keybuf != NULL &&
                            quicksearch_enabled == true ) {
                    // key press msg only arrives if a string gadget doesn't have the focus
                    // but it won't hurt to check for active search mode
                    if ( strlen( msg->key.keybuf ) > 0 ) {
                        //TODO mh, this is not the best way to set case insensitive
                        //     but currently it's the easiest way
                        //     an argument would be better for sure
                        setEntrySearchCaseSensitive( false );
                        shownextentry( msg->key.keybuf );
                    }
                }
            }
            break;
        case AG_KARTEIBUTTONCLICKED:
            if ( msg->karteibutton.karteibutton == m_tab_b ) {
                ma = true;

                if ( msg->karteibutton.mousebutton == Button3 ) {
                    openTabPopUp( msg->karteibutton.option );
                } else {
                    switchToTab( msg->karteibutton.option );
                }
            }
            break;
        default:
            break;
    }
    if(ma==true) parentlister->makeActive();

    //TODO: Brauche ich das hier noch? So richtig Sinn macht es nicht
    //showFreeSpace( false );
}

void NormalMode::on()
{
    int side=parentlister->getSide();
    AGUIXFont *afont = aguix->getFont( wconfig->getFont( 2 + side ).c_str() );
    int sgh = ( ( afont != NULL ) ? afont->getCharHeight() : aguix->getCharHeight() ) +8;
    int hbw = aguix->getTextWidth( "<", afont ) + 10;
    int m;

    m_cont = new AContainer( parentawindow, 1, 3 );
    m_cont->setBorderWidth( 0 );
    m_cont->setMinSpace( 0 );
    m_cont->setMaxSpace( 0 );
    parentlister->setContainer( m_cont );

    m_cont2 = m_cont->add( new AContainer( parentawindow, 4, 1 ), 0, 2 );
    m_cont2->setBorderWidth( 0 );
    m_cont2->setMinSpace( 0 );
    m_cont2->setMaxSpace( 0 );

    parentb = new Button( aguix, 0, 0, aguix->getTextWidth( "..", afont ) + 10, "..", 1, 0, 5 );
    hb[0] = new Button( aguix, 0, 0, hbw, "<", 1, 0, 2 );
    hb[1] = new Button( aguix, 0, 0, hbw, ">", 1, 0, 3 );

    m_tab_cont = m_cont->add( new AContainer( parentawindow, 3, 1 ), 0, 0 );
    m_tab_cont->setBorderWidth( 0 );
    m_tab_cont->setMinSpace( 0 );
    m_tab_cont->setMaxSpace( 0 );

    m_tab_entries.clearEntries();
    m_tab_b = (KarteiButton*)m_tab_cont->add( new KarteiButton( aguix,
                                                                0, 0,
                                                                100, 1, 0, 0 ), 1, 0, AContainer::ACONT_MINH + AContainer::ACONT_MAXH );
    m_tab_b->setAcceptFocus( false );
    addTabOption( "" );

    m_tab_b->setTextShrinker( m_shrinker );
    m_tab_b->setFont( wconfig->getFont( 2 + side ).c_str() );

    m_tab_new = (Button*)m_tab_cont->add( new Button( aguix, 0, 0,
                                                      "N", 1, 0, 0 ), 0, 0, AContainer::ACONT_MINW + AContainer::ACONT_MAXW );
    m_tab_new->setAcceptFocus( false );
    m_tab_new->setFont( wconfig->getFont( 2 + side ).c_str() );
    m_tab_new->resize( aguix->getTextWidth( "N", afont ) + 10, m_tab_new->getHeight() );

    m_tab_close = (Button*)m_tab_cont->add( new Button( aguix, 0, 0,
                                                        "X", 1, 0, 0 ), 2, 0, AContainer::ACONT_MINW + AContainer::ACONT_MAXW );
    m_tab_close->setAcceptFocus( false );
    m_tab_close->setFont( wconfig->getFont( 2 + side ).c_str() );
    m_tab_close->resize( aguix->getTextWidth( "X", afont ) + 10, m_tab_new->getHeight() );

    m_tab_cont->readLimits();

    lv = (NMFieldListViewDND*)m_cont->add( new NMFieldListViewDND( aguix,
                                                                   0, 0,
                                                                   50, 50,
                                                                   1 ), 0, 1 );
    lv->setAcceptFocus( false );
    lv->setMBG(wconfig->getListerBG());
    m=(wconfig->getHBarTop(side)==true)?1:2;
    lv->setHBarState(m);
    m=(wconfig->getVBarLeft(side)==true)?1:2;
    lv->setVBarState(m);
    m=wconfig->getHBarHeight(side);
    lv->setHBarHeight(m);
    m=wconfig->getVBarWidth(side);
    lv->setVBarWidth(m);
    lv->setShowHeader( wconfig->getShowHeader( side ) );
    lv->setFont( wconfig->getFont( 2 + side ).c_str() );
    lv->setHeaderFG( wconfig->getLVHeader( 0 ) );
    lv->setHeaderBG( wconfig->getLVHeader( 1 ) );
  
    parentb->resize( parentb->getWidth(), sgh );
    m_cont2->add( parentb, ( side == 0 ) ? 0 : 3, 0, AContainer::CO_FIX );
    parentb->setAcceptFocus( false );
    parentb->setFont( wconfig->getFont( 2 + side ).c_str() );
  
    hb[0]->resize( hb[0]->getWidth(), sgh );
    m_cont2->add( hb[0], ( side == 0 ) ? 1 : 0, 0, AContainer::CO_FIX );
    hb[0]->setAcceptFocus( false );
    hb[0]->setFont( wconfig->getFont( 2 + side ).c_str() );
  
    hb[1]->resize( hb[1]->getWidth(), sgh );
    m_cont2->add( hb[1], ( side == 0 ) ? 3 : 2, 0, AContainer::CO_FIX );
    hb[1]->setAcceptFocus( false );
    hb[1]->setFont( wconfig->getFont( 2 + side ).c_str() );

    sg = (StringGadget*)m_cont2->add( new StringGadget( aguix,
                                                        0,
                                                        0,
                                                        50, "", 4 ),
                                      ( side == 0 ) ? 2 : 1, 0 );
    sg->setAcceptFocus( false );
    sg->resize( sg->getWidth(), sgh );
    sg->setFont( wconfig->getFont( 2 + side ).c_str() );

    parentlister->setActiveMode(this);
    setName();
    lv->setSelectHandler(this);

    setupLVFields();
  
    if ( m_off_store.list_of_used_tabs.size() > 0 ) {
        std::list<std::string>::iterator it1 = m_off_store.list_of_used_tabs.begin();
        if ( it1 != m_off_store.list_of_used_tabs.end() ) {
            setTabOption( 0, *it1 );
            it1++;
        }
        for ( ; it1 != m_off_store.list_of_used_tabs.end(); it1++ ) {
            addTabOption( *it1 );
        }

        if ( m_off_store.active_tab >= 0 ) {
            switchToTab( m_off_store.active_tab );
        }
    }

    parentawindow->updateCont();
}

void NormalMode::off()
{
    finishsearchmode();

    // backup field width for later reuse or save
    writeFieldWidthsToLoadMap();

    // store the current pos for reactivating
    if ( ( ce != NULL ) && ( lv != NULL ) ) {
        ce->setPos( lv->getYOffset() );
        ce->setXPos( lv->getXOffset() );
        ce->setActiveIsVisible( lv->isRowVisible( lv->getActiveRow() ) );

        m_off_store.active_tab = m_tab_b->getSelectedOption();
        m_off_store.list_of_used_tabs.clear();
        for ( int i = 0; i < m_tab_b->getNrOfOptions(); i++ ) {
            m_off_store.list_of_used_tabs.push_back( m_tab_entries.getEntry( i ) );
        }
    }
    ce = NULL;

    freePopUpMenus();
  
    delete lv;
    lv = NULL;
    delete hb[0];
    hb[0] = NULL;
    delete hb[1];
    hb[1] = NULL;
    delete sg;
    sg = NULL;
    delete parentb;
    parentb = NULL;

    delete m_tab_b;
    m_tab_b = NULL;
    delete m_tab_new;
    m_tab_new = NULL;
    delete m_tab_close;
    m_tab_close = NULL;

    parentlister->setContainer( NULL );
    delete m_cont;

    m_cont = m_cont2 = NULL;
    m_tab_cont = NULL;

    parentlister->setActiveMode(NULL);
    parentlister->setName("");
    if ( parentlister->getFocus() == true )
        parentlister->getWorker()->setTitle(NULL);
}

void NormalMode::activate()
{
    int row;

    if ( ce != NULL ) {
        if ( ce->getActiveFE() != NULL ) {
            row = 0;
            while ( row < lv->getElements() ) {
                if ( lv->getData( row ) == ce->getActiveFE()->getID() ) break;
                row++;
            }
            if( row < lv->getElements() ) lv->setActiveRow( row );
        }
    }
    showCacheState();
    parentlister->getWorker()->setTitle( sg->getText() );
}

void NormalMode::deactivate()
{
    int row=lv->getActiveRow();
    if ( row >= 0 ) {
        lv->setActiveRow( -1);
    }
    finishsearchmode();
    parentlister->getWorker()->setTitle(NULL);
}

void NormalMode::buildListView()
{
    if ( ce == NULL ) return;
    if ( lv == NULL ) return;

    int side = parentlister->getSide();
    const std::vector<WorkerTypes::listcol_t> *dis;
    int nr_of_entries = ce->getNrOfFiles( 0 ) + ce->getNrOfDirs( 0 ) + 1;

    lv->setSize( nr_of_entries );
    lv->setSizeDNDText( nr_of_entries );
    lv->setActiveRowQ( -1 );

    dis = wconfig->getVisCols( side );

    int tpos = 0;

    for ( Verzeichnis::verz_it it1 = ce->begin();
          it1 != ce->end();
          it1++ ) {
        FileEntry *fe = *it1;
        if ( fe->use == true ) {
            fe->setLVC_DND( lv, tpos, dis );
            lv->setData( tpos, fe->getID());
            if ( fe->select == true ) lv->setSelectQ( tpos, true );
            else lv->setSelectQ( tpos, false );
            if ( fe == ce->getActiveFE() ) {
                // only make the lvc active if we are active
                if ( parentlister->isActive() == true ) lv->setActiveRowQ( tpos );
            }
            tpos++;
        }
    }

    _lv_ids_serial = ce->getSerialOfIDs();

    if ( tpos != nr_of_entries ) {
        fprintf( stderr, "Worker error: Nr of entries does not match real value!\n" );
        //TODO remove potential remaining entries?
    }
  
    lv->redraw();
  
    visChanged = true;
}

void NormalMode::deleteCache()
{
    NMCacheEntry *tce=(NMCacheEntry*)cache->getFirstElement();
    while(tce!=NULL) {
        delete tce;
        tce=(NMCacheEntry*)cache->getNextElement();
    }
}

bool NormalMode::checkEnterDir( const char *dir )
{
    std::string str1;
    bool allow_entering = true;
    
#ifdef HAVE_AVFS
    if ( strncmp( dir, "/#ssh:", 6 ) == 0 && ssh_allow != SSH_ALWAYS ) {
        if ( ssh_allow == SSH_NEVER ) {
            allow_entering = false;
        } else {
            bool var1;
            int erg;
            
            // show requester
            str1 = catalog.getLocale( 629 );
            str1 += "|";
            str1 += catalog.getLocale( 8 );
            
            var1 = false;
            
            erg = request_choose( catalog.getLocale( 123 ),
                                  catalog.getLocale( 630 ),
                                  catalog.getLocale( 631 ), var1,
                                  str1.c_str() );
            if ( var1 == true ) {
                if ( erg != 0 ) {
                    ssh_allow = SSH_NEVER;
                } else {
                    ssh_allow = SSH_ALWAYS;
                }
            }
            
            if ( erg != 0 ) {
                allow_entering = false;
            }
        }
    }
#endif

    if ( allow_entering == true ) {
        if ( worker_access( dir, R_OK | X_OK ) != 0 ) {
            char *tstr2;
            const char *myerror;
            
            // show requester
            myerror = strerror( errno );
            tstr2 = (char*)_allocsafe( strlen( catalog.getLocale( 532 ) ) +
                                       strlen( dir ) +
                                       strlen( myerror ) + 1 );
            sprintf( tstr2, catalog.getLocale( 532 ), dir, myerror );
            request( catalog.getLocale( 347 ), tstr2, catalog.getLocale( 11 ) );
            _freesafe( tstr2 );
            
            allow_entering = false;
        }
    }

    return allow_entering;
}

int NormalMode::enterDir(const char *dirstr)
{
    char *tstr;
    std::auto_ptr<Verzeichnis> tverz;
    NMCacheEntry *tce = NULL;
    int comperg;
    bool reset;
    int returnvalue = 0;  /* 0 = OK
                           * 1 = Not found
                           * 2 = not found, but lv cleared (was actually)
                           */
    std::string str1;
    bool find_new_active_row = false;
  
    if ( lv == NULL ) return 1;

    if ( dirstr == NULL ) return 1;

    parentlister->getWorker()->setWaitCursor();
    aguix->Flush();

    // only reset string filter if not entering same directory
    if ( ce == NULL || strcmp( ce->getDir(), dirstr ) != 0 ) {
        setEntrySearchString( "", false );
    }
    
    // normalize given dir
    if ( dirstr[0] != '/' ) {
        // resolve current dir
        char *cwd = HandlePathExt( "." );
        str1 = cwd;
        str1 += "/";
        str1 += dirstr;
        
        _freesafe( cwd );
        tstr = HandlePath( str1.c_str() );
    } else {
        tstr = HandlePath( dirstr );
    }
    
    // check whether it is okay to enter the dir
    if ( checkEnterDir( tstr ) == false ) {
        // restore stringgadget
        if ( ce != NULL ) {
            _freesafe( tstr );
            tstr = HandlePath( ce->getDir() );
            sg->setText( tstr );
        } else sg->setText( "" );
      
        _freesafe( tstr );
        parentlister->getWorker()->unsetWaitCursor();
        return 1;
    }
    
    finishsearchmode();
    
    // clear the reclist so no wrong pointers are in the list
    ft_rec_list_clear();
    
    // search for existing cache entry
    if ( cache->size() > 0 ) {
        tce = (NMCacheEntry*)cache->getFirstElement();
        while ( tce != NULL ) {
            if ( strcmp( tce->getDir(), tstr ) == 0 ) {
                break;
            }
            tce = (NMCacheEntry*)cache->getNextElement();
        }
    }
    
    // read directory
    tverz = std::auto_ptr<Verzeichnis>( new Verzeichnis() );
    if ( tverz->readDir( tstr ) != 0 ) {
        tverz.release();
    }
    
    if ( tverz.get() != NULL ) {
        if ( tce == NULL ) {
            // no cache entry found
            while ( cache->size() >= (int)wconfig->getCacheSize() ) {
                //TODO maybe use max cache size plus number of tabs as limit
                tce = findOldestCacheEntryNotInTab();
                if ( tce == NULL ) break;

                cache->removeElement( tce );
                delete tce;
                if ( ce == tce ) ce = NULL;
            }
            tce = new NMCacheEntry( tverz,
                                    &_dir_filter_sets,
                                    &_dir_sort_sets,
                                    &m_dir_bookmarks_sets );
            cache->addElement( tce );
            tce->restartcheck();

            if ( tce->dirOpened() == true ) {
                if ( tce->begin() != tce->end() ) {
                    tce->setActiveFE( *( tce->begin() ) );
                }
            }
            tce->checkfordcd();
        } else {
            // put tce at the end of the cache to realize Not-Recently-Used
            cache->removeElement(tce);
            cache->addElement(tce);
            
            // insert old values into to new tverz
            // first sort both for names so we can easily (and fast) get old values
            // set new sort mode so getFiles will init resort
            int old_sort_mode = _dir_sort_sets.getSortMode();
            _dir_sort_sets.setSortMode( SORT_NAME | SORT_DIRMIXED );

            tverz->sort( SORT_NAME | SORT_DIRMIXED );
            
            Verzeichnis::verz_it old_it1 = tce->begin();
            Verzeichnis::verz_it new_it1 = tverz->begin();

            FileEntry *old_activefe = tce->getActiveFE();
            FileEntry *new_activefe = NULL;
            
            while ( new_it1 != tverz->end() ) {
                FileEntry *nfe = *new_it1;
                if ( old_activefe != NULL )
                    if ( strcmp( nfe->fullname, old_activefe->fullname ) == 0 )
                        new_activefe = nfe;
                while ( old_it1 != tce->end() ) {
                    FileEntry *ofe = *old_it1;
                    comperg = StringComparator::compare( ofe->fullname, nfe->fullname );
                    if ( comperg > 0 ) break;
                    else if ( comperg == 0 ) {
                        // found an entry matching the current
                        // now update values

                        if ( ofe->select == true )
                            nfe->select = true;
                        else
                            nfe->select = false;

                        if ( ofe->isDir() == true )
                            if ( ofe->dirsize >= 0 )
                                nfe->dirsize = ofe->dirsize;

                        if ( ofe->filetype != NULL )
                            nfe->filetype = ofe->filetype;

                        nfe->setColor( ofe->getColor() );
                    }
                    old_it1++;
                    if ( strcmp( nfe->name, ".." ) == 0 )
                        break; /* the ".." is always first in new and old so stop here
                                * because following elements can be "smaller" then ".."
                                * f.i. "#343" < ".."
                                */
                }

                if ( old_it1 == tce->end() )
                    break; // there are no further old files => no need to continue

                new_it1++;
            }
            
            if ( new_activefe == NULL && old_activefe != NULL ) {
                // corresponding active entry wasn't found
                find_new_active_row = true;
            }
            
            // update done
            // reset original sort mode
            _dir_sort_sets.setSortMode( old_sort_mode );

            // now set new new Verzechnis
            tce->setVerz( tverz );
            // and active FE
            tce->setActiveFE( new_activefe );
        }

        if ( tce->getActiveFE() != NULL ) {
            if ( tce->getActiveFE()->use == false ) {
                // this will never happen because the fallback is
                // the first entry in the filelist which is always ".."
                // but this doesn't hurt
                tce->setActiveFE( NULL );
            }
        }

        if ( ce != NULL ) {
            ce->setPos( lv->getYOffset() );
            ce->setXPos( lv->getXOffset() );
            ce->setActiveIsVisible( lv->isRowVisible( lv->getActiveRow() ) );
            ce->setLastActiveRow( lv->getActiveRow() );
        }

        ce = tce;
        rebuildView();

        if ( find_new_active_row == true ) {
            // find used entry around old_act_row

            int new_row = ce->getLastActiveRow() - 1;
            while ( lv->isValidRow( new_row ) == false && new_row > 0 ) {
                new_row--;
            }
            
            makeRowActive( new_row );
        }

        lv->setYOffset( ce->getPos() );
        lv->setXOffset( ce->getXPos() );
        if ( ce->getActiveIsVisible() == true &&
             lv->isRowVisible( lv->getActiveRow() ) == false ) {
            lv->centerActive();
        }
        sg->setText( tstr );

        bool new_entry = false;

        parentlister->getWorker()->getPathStore().storePath( tstr, &new_entry );

        if ( new_entry ) {
            parentlister->getWorker()->storePathPers( tstr );
        }
    } else { /* tverz == NULL */
        // can't read the dir
        returnvalue = 1;
        reset = true;
        if ( tce != NULL ) {
            //TODO: Verzeichnis gibt es nicht mehr, also aus Cache entfernen
            // klappt jetzt so
            if ( tce == ce ) {
                // it's the current cacheentry
                reset = false;
                returnvalue = 2;
                cache->removeElement( ce );
                delete ce;
                ce = NULL;
                lv->setSize( 0 );
                lv->redraw();
                sg->setText( "" );
                updateTabs();
            } else {
                cache->removeElement( tce );
                delete tce;
            }
        }
        if ( reset == true ) {
            // wiederherstellen
            if ( ce != NULL ) {
                _freesafe( tstr );
                tstr = HandlePath( ce->getDir() );
                sg->setText( tstr );
            } else sg->setText( "" );
        }
    }
    _freesafe( tstr );
    
    showFreeSpace( true );
    
    if ( parentlister->getFocus() == true )
        parentlister->getWorker()->setTitle( sg->getText() );
    parentlister->getWorker()->unsetWaitCursor();

    return returnvalue;
}

void NormalMode::showcache(int direction)
{
    NMCacheEntry *tce;
    int cs=cache->size();
    int i=0,id;
    if(cs<1) return;

    finishsearchmode();
  
    if(ce!=NULL) {
        ce->setPos( lv->getYOffset() );
        ce->setXPos( lv->getXOffset() );
        ce->setActiveIsVisible( lv->isRowVisible( lv->getActiveRow() ) );
    }
    int t = cache->getIndex( ce );
    if ( t < 0 ) t = 0;
  
    id=cache->initEnum();
    do {
        i++;
        t+=direction;
        if(t>=cache->size()) t=0;
        else if(t<0) t=cache->size()-1;
        tce=(NMCacheEntry*)cache->getElementAt(id,t);
        if(tce!=NULL) {
            if(enterDir(tce->getDir())==0) {
                break;
            }
        }
    } while(i<cs);
    cache->closeEnum(id);
}

void NormalMode::selhandler(FieldListView *tlv, int row)
{
    if(ce==NULL) return;
    if ( ce->dirOpened() == false ) return;
    FileEntry *fe;
    fe = getFEForRow( row );

    if(fe==NULL) return;

    bool state=lv->getSelect( row );
    bool oldstate=fe->select;
  
    if(state!=oldstate) {
        // Aenderung
    
        if( row == 0 ) {
            lv->setSelect( row, false );
        } else {
            if(state==true) {
                // Aus -> An
                ce->updateStatEntrySelected( fe );
            } else {
                // An -> Aus
                ce->updateStatEntryDeselected( fe );
            }
            fe->select=state;
        }
    }
    showCacheState();
    if ( lv->getActiveRow() == row ) {
        ce->setActiveFE( fe );
    }
}

void NormalMode::showCacheState()
{
    if(parentlister->isActive()==false) return;  // do nothing when we are not active
    if ( ce == NULL ) {
        // no directory but at least clear statebar
        parentlister->setStatebarText( "" );
        return;
    }
    const char *tstr1=catalog.getLocale(107),*tstr2=catalog.getLocale(108);
    const char *tstr3=catalog.getLocale(13);
    char *str;
    std::string buf1, buf2, buf3, buf4;
    bool human_strings = false;
    loff_t l1,l2;
    int l,tl;
    char *formatstr;
  
    l1 = ce->getSizeOfFiles( 1 ) + ce->getSizeOfDirs( 1 );
    l2 = ce->getSizeOfFiles( 0 ) + ce->getSizeOfDirs( 0 );

    buf3 = AGUIXUtils::bytes_to_human_readable_f( l1, 10 );
    buf4 = AGUIXUtils::bytes_to_human_readable_f( l2, 10 );

    if ( l2 > 10 * 1024 ) {
        human_strings = true;
    }

    MakeLong2NiceStr( l1, buf1 );
    MakeLong2NiceStr( l2, buf2 );
  
    // wenn mehr als 100 mal nichts an diesen Variablen
    // geaendert wurde, jeweils um 1 verringern, so dass
    // nicht ewig zu viel Platz verwendet wird, nur weil
    // einmal eine besonders grosse Datei da war
    if(maxnochanges>100) {
        maxnochanges=0;
        // auf >0 muss nicht geprueft werden, da danach sowieso nochmal geprueft
        // wird
        maxfilestrlen--;
        maxdirstrlen--;
        maxbytesstrlen--;
        maxwbytesstrlen--;
    }

    tl=0;
    l = (int)( log10( (double)ce->getNrOfFiles( 0 ) + 1 ) + 1.0 );
    if(l>maxfilestrlen) {
        maxfilestrlen=l;
        tl++;
    }
    l = (int)( log10( (double)ce->getNrOfDirs( 0 ) + 1 ) +1.0 );
    if(l>maxdirstrlen) {
        maxdirstrlen=l;
        tl++;
    }
    l = buf2.length();
    if(l>maxbytesstrlen) {
        maxbytesstrlen=l;
        tl++;
    }
  
    // we have to check for buf3 and buf4 because this is possibly:
    //  1200 KB / 100 MB
    // the left part takes more space
    l = buf3.length();
    if(l>maxwbytesstrlen) {
        maxwbytesstrlen=l;
        tl++;
    }
    l = buf4.length();
    if(l>maxwbytesstrlen) {
        maxwbytesstrlen=l;
        tl++;
    }

    // this are the "double" format strings for the statebar
    // the first is when we will also print some nice byte-values (in B, KB or MB)
    // the second otherwise
    // they have to be same length and modifiers
    // from this strings the real format string is build according to the needed string/number lengths
    // NOTE: the two last %s in _SLIM will be replaced by spaces
    //       this is needed to keep the string at the same place in the statebar when switching between listers
#define STATEBAR_FORMAT      "[%%s %%%dld / %%%dld]   [%%s %%%dld / %%%dld]   [%%s %%%ds / %%%ds   (%%%ds / %%%ds) ]"
#define STATEBAR_FORMAT_SLIM "[%%s %%%dld / %%%dld]   [%%s %%%dld / %%%dld]   [%%s %%%ds / %%%ds]   %%%ds   %%%ds   "

    formatstr = (char*)_allocsafe( strlen( STATEBAR_FORMAT ) +
                                   8 * ( A_BYTESFORNUMBER( maxfilestrlen ) ) + 1 );
    // for a 4 byte value I need 11 decimals
    // there will be 8 replacements

    str = (char*)_allocsafe( sizeof( char ) *
                             ( strlen( STATEBAR_FORMAT ) +     // space of the format string
                               strlen( tstr1 ) +               // length of string for "files" (first %s)
                               strlen( tstr2 ) +               // length of string for "dirs" (second %s)
                               strlen( tstr3 ) +               // length of string for "bytes" (third %s)
                               2 * maxfilestrlen +             // space needed for "sel files / files"
                               2 * maxdirstrlen +              // space needed for "sel dirs / dirs"
                               2 * maxbytesstrlen +            // space needed for "sel bytes / bytes"
                               2 * maxwbytesstrlen +           // space needed for "sel bytes / bytes" (in B/KB/MB)
                               1 ) );                          // and not to forget the final zero
    //NOTE: this is more than we will need because the modifiers in the formatstring will be replaced
    //      but it's okay to alloc some more bytes than needed
    if ( human_strings == true ) {
        // this only when the field is visible

        sprintf( formatstr, STATEBAR_FORMAT,
                 maxfilestrlen,
                 maxfilestrlen,
                 maxdirstrlen,
                 maxdirstrlen,
                 maxbytesstrlen,
                 maxbytesstrlen,
                 maxwbytesstrlen,
                 maxwbytesstrlen );
        sprintf( str, formatstr,
                 tstr1,
                 ce->getNrOfFiles( 1 ),
                 ce->getNrOfFiles( 0 ),
                 tstr2,
                 ce->getNrOfDirs( 1 ),
                 ce->getNrOfDirs( 0 ),
                 tstr3,
                 buf1.c_str(),
                 buf2.c_str(),
                 buf3.c_str(),
                 buf4.c_str() );
    } else {
        sprintf( formatstr, STATEBAR_FORMAT_SLIM,
                 maxfilestrlen,
                 maxfilestrlen,
                 maxdirstrlen,
                 maxdirstrlen,
                 maxbytesstrlen,
                 maxbytesstrlen,
                 maxwbytesstrlen,
                 maxwbytesstrlen );
        sprintf( str, formatstr,
                 tstr1,
                 ce->getNrOfFiles( 1 ),
                 ce->getNrOfFiles( 0 ),
                 tstr2,
                 ce->getNrOfDirs( 1 ),
                 ce->getNrOfDirs( 0 ),
                 tstr3,
                 buf1.c_str(),
                 buf2.c_str(),
                 "",    // print just spaces
                 "" );  // print just spaces
    }
    _freesafe(formatstr);
  
    if(tl==0) // no changes
        maxnochanges++;
  
    parentlister->setStatebarText(str);
    _freesafe(str);
#undef STATEBAR_FORMAT
#undef STATEBAR_FORMAT_SLIM
}

bool NormalMode::isType(const char *str)
{
    if(strcmp(str,type)==0) return true; else return false;
}

const char *NormalMode::getType()
{
    return type;
}

const char *NormalMode::getStaticType()
{
    return type;
}

int NormalMode::configure()
{
    Button *fb;
    AWindow *win;
    CycleButton *cyb[2];
    ChooseButton *chb, *ucb, *no_qs_cb;
    AGMessage *msg;
    int endmode=-1;
    char *tstr;
    bool tshowhidden;
    StringGadget *tsg;
    int tut;
    const int cincw = AContainer::ACONT_MINH +
        AContainer::ACONT_MINW +
        AContainer::ACONT_MAXH;
    const int cincwnr = cincw +
        AContainer::ACONT_NORESIZE;
    const int cfix = AContainer::ACONT_MINH +
        AContainer::ACONT_MINW +
        AContainer::ACONT_MAXH +
        AContainer::ACONT_MAXW;
    std::list<NM_Filter> tfilters;
  
    tstr = (char*)_allocsafe( strlen( catalog.getLocale( 293 ) ) + strlen( getLocaleName() ) + 1 );
    sprintf( tstr, catalog.getLocale( 293 ), getLocaleName() );
    win = new AWindow( aguix, 10, 10, 10, 10, 0, tstr );
    win->create();
    _freesafe(tstr);

    AContainer *ac1 = win->setContainer( new AContainer( win, 1, 5 ), true );
    ac1->setMinSpace( 5 );
    ac1->setMaxSpace( 5 );

    AContainerBB *ac1_1 = static_cast<AContainerBB*>( ac1->add( new AContainerBB( win, 1, 3 ), 0, 0 ) );
    ac1_1->setMinSpace( 5 );
    ac1_1->setMaxSpace( 5 );

    AContainer *ac1_1_1 = ac1_1->add( new AContainer( win, 2, 1 ), 0, 0 );
    ac1_1_1->setMinSpace( 5 );
    ac1_1_1->setMaxSpace( 5 );
    ac1_1_1->add( new Text( aguix, 0, 0, catalog.getLocale( 158 ), 1 ), 0, 0, cfix );
    cyb[0]=(CycleButton*)ac1_1_1->add( new CycleButton( aguix, 0, 0, 100, 1, 0, 0 ), 1, 0, cincw );
    cyb[0]->addOption(catalog.getLocale(160));
    cyb[0]->addOption(catalog.getLocale(161));
    cyb[0]->addOption(catalog.getLocale(162));
    cyb[0]->addOption(catalog.getLocale(163));
    cyb[0]->addOption(catalog.getLocale(164));
    cyb[0]->addOption(catalog.getLocale( 432 ));
    cyb[0]->addOption(catalog.getLocale( 433 ));
    cyb[0]->addOption(catalog.getLocale( 552 ));
    cyb[0]->addOption(catalog.getLocale( 553 ));
    cyb[0]->addOption(catalog.getLocale( 907 ));
    cyb[0]->resize(cyb[0]->getMaxSize(),cyb[0]->getHeight());
    if((sortmode&0xff)==SORT_SIZE) cyb[0]->setOption(1);
    else if((sortmode&0xff)==SORT_ACCTIME) cyb[0]->setOption(2);
    else if((sortmode&0xff)==SORT_MODTIME) cyb[0]->setOption(3);
    else if((sortmode&0xff)==SORT_CHGTIME) cyb[0]->setOption(4);
    else if((sortmode&0xff)==SORT_TYPE) cyb[0]->setOption(5);
    else if((sortmode&0xff)==SORT_OWNER) cyb[0]->setOption(6);
    else if ( ( sortmode & 0xff ) == SORT_INODE ) cyb[0]->setOption( 7 );
    else if ( ( sortmode & 0xff ) == SORT_NLINK ) cyb[0]->setOption( 8 );
    else if ( ( sortmode & 0xff ) == SORT_PERMISSION ) cyb[0]->setOption( 9 );
    else cyb[0]->setOption(0);

    chb=(ChooseButton*)ac1_1->add( new ChooseButton( aguix, 0, 0,
                                                     ((sortmode&SORT_REVERSE)==SORT_REVERSE)?1:0,
                                                     catalog.getLocale(165),LABEL_RIGHT,1,0), 0, 1, cincwnr );

    AContainer *ac1_1_2 = ac1_1->add( new AContainer( win, 2, 1 ), 0, 2 );
    ac1_1_2->setMinSpace( 5 );
    ac1_1_2->setMaxSpace( 5 );
    ac1_1_2->add( new Text( aguix, 0, 0, catalog.getLocale( 353 ), 1 ), 0, 0, cfix );
    cyb[1]=(CycleButton*)ac1_1_2->add( new CycleButton( aguix, 0, 0, 100, 1, 0, 0 ), 1, 0, cincw );
    cyb[1]->addOption(catalog.getLocale(354));
    cyb[1]->addOption(catalog.getLocale(355));
    cyb[1]->addOption(catalog.getLocale(356));
    cyb[1]->resize(cyb[1]->getMaxSize(),cyb[1]->getHeight());
    if((sortmode&SORT_DIRLAST)==SORT_DIRLAST) cyb[1]->setOption(1);
    else if((sortmode&SORT_DIRMIXED)==SORT_DIRMIXED) cyb[1]->setOption(2);
    else cyb[1]->setOption(0);

    ac1_1_1->readLimits();
    ac1_1_2->readLimits();

    fb = (Button*)ac1->add( new Button( aguix, 0, 0, catalog.getLocale( 159 ), 1, 0, 0 ), 0, 1, cincw );

    AContainerBB *ac1_2 = static_cast<AContainerBB*>( ac1->add( new AContainerBB( win, 1, 2 ), 0, 2 ) );
    ac1_2->setMinSpace( 5 );
    ac1_2->setMaxSpace( 5 );

    ucb=(ChooseButton*)ac1_2->add( new ChooseButton( aguix, 0, 0,
                                                     (showfreespace==true)?1:0,
                                                     catalog.getLocale(128),LABEL_RIGHT,1,0), 0, 0, cincwnr );

    AContainer *ac1_2_1 = ac1_2->add( new AContainer( win, 2, 1 ), 0, 1 );
    ac1_2_1->setMinSpace( 5 );
    ac1_2_1->setMaxSpace( 5 );
    ac1_2_1->add( new Text( aguix, 0, 0, catalog.getLocale( 129 ), 1 ), 0, 0, cfix );
    tstr = (char*)_allocsafe( A_BYTESFORNUMBER( int ) );
    sprintf(tstr,"%d",updatetime);
    tsg=(StringGadget*)ac1_2_1->add( new StringGadget( aguix, 0, 0, 100, tstr, 0 ), 1, 0, cincw );
    _freesafe(tstr);

    AContainer *ac1_4 = ac1->add( new AContainer( win, 2, 1 ), 0, 3 );
    ac1_4->setMinSpace( 5 );
    no_qs_cb = (ChooseButton*)ac1_4->add( new ChooseButton( aguix, 0, 0,
                                                            ( quicksearch_enabled == true) ? 1 : 0,
                                                            catalog.getLocale( 649 ), LABEL_RIGHT, 1, 0 ), 0, 0, AContainer::CO_FIXNR );
    ChooseButton *fs_cb = (ChooseButton*)ac1_4->add( new ChooseButton( aguix, 0, 0,
                                                                       ( _filtered_search_enabled == true) ? 1 : 0,
                                                                       catalog.getLocale( 754 ), LABEL_RIGHT, 1, 0 ), 1, 0, AContainer::CO_FIXNR );

    AContainer *ac1_3 = ac1->add( new AContainer( win, 2, 1 ), 0, 4 );
    ac1_3->setMinSpace( 5 );
    ac1_3->setMaxSpace( -1 );
    ac1_3->setBorderWidth( 0 );
    Button *okb =(Button*)ac1_3->add( new Button( aguix,
                                                  0,
                                                  0,
                                                  catalog.getLocale( 11 ),
                                                  1,
                                                  0,
                                                  0 ), 0, 0, cfix );
    Button *cb = (Button*)ac1_3->add( new Button( aguix,
                                                  0,
                                                  0,
                                                  catalog.getLocale( 8 ),
                                                  1,
                                                  0,
                                                  0 ), 1, 0, cfix );
    win->setDoTabCycling( true );
    win->contMaximize( true );
    win->show();

    tshowhidden = _dir_filter_sets.getShowHidden();
    tfilters = _dir_filter_sets.getFilters();

    DirFilterSettings t_dir_settings( _dir_filter_sets );

    win->show();
    for(;endmode==-1;) {
        msg=aguix->WaitMessage(win);
        if(msg!=NULL) {
            switch(msg->type) {
                case AG_CLOSEWINDOW:
                    if(msg->closewindow.window==win->getWindow()) endmode=1;
                    break;
                case AG_BUTTONCLICKED:
                    if(msg->button.button==okb) endmode=0;
                    else if(msg->button.button==cb) endmode=1;
                    else if(msg->button.button==fb) {
                        configureFilters( &tshowhidden, tfilters,
                                          t_dir_settings );
                    }
                    break;
            }
            aguix->ReplyMessage(msg);
        }
    }
  
    if(endmode==0) {
        // ok
        int tsortmode=0;
        switch(cyb[0]->getSelectedOption()) {
            case 1:
                tsortmode=SORT_SIZE;
                break;
            case 2:
                tsortmode=SORT_ACCTIME;
                break;
            case 3:
                tsortmode=SORT_MODTIME;
                break;
            case 4:
                tsortmode=SORT_CHGTIME;
                break;
            case 5:
                tsortmode=SORT_TYPE;
                break;
            case 6:
                tsortmode=SORT_OWNER;
                break;
            case 7:
                tsortmode = SORT_INODE;
                break;
            case 8:
                tsortmode = SORT_NLINK;
                break;
            case 9:
                tsortmode = SORT_PERMISSION;
                break;
            default:
                tsortmode=SORT_NAME;
                break;
        }
        if ( chb->getState() == true ) tsortmode |= SORT_REVERSE;
        switch(cyb[1]->getSelectedOption()) {
            case 1:
                tsortmode|=SORT_DIRLAST;
                break;
            case 2:
                tsortmode|=SORT_DIRMIXED;
                break;
            default:
                break;
        }
        setSortmode(tsortmode);
        setShowHidden(tshowhidden);
        setFilters( tfilters );
        setShowFreeSpace( ucb->getState() );
        tut=atoi(tsg->getText());
        setUpdatetime(tut);

        setQuicksearchEnabled( no_qs_cb->getState() );
        setFilteredSearchEnabled( fs_cb->getState() );

        setHighlightBookmarkPrefix( t_dir_settings.getHighlightBookmarkPrefix() );
        setBookmarkFilter( t_dir_settings.getBookmarkFilter() );
        setSpecificBookmarkLabel( t_dir_settings.getSpecificBookmarkLabel() );
    }
  
    delete win;

    return endmode;
}

void NormalMode::down()
{
    int row;
    
    if ( ce == NULL ) return;
    
    finishsearchmode();
    
    if ( ce->getActiveFE() == NULL ) {
        makeRowActive( 0 );
        lv->showActive();
    } else {
        row = lv->getActiveRow();
        row++;
        if ( lv->isValidRow( row ) == true ) {
            makeRowActive( row );
            lv->showActive();
        }
    }
}

void NormalMode::up()
{
    int row;
    
    if ( ce == NULL ) return;
    
    finishsearchmode();
    
    if ( ce->getActiveFE() == NULL ) {
        makeRowActive( 0 );
        lv->showActive();
    } else {
        row = lv->getActiveRow();
        row--;
        if ( lv->isValidRow( row ) == true ) {
            makeRowActive( row );
            lv->showActive();
        }
        
    }
}

void NormalMode::toggleHidden()
{
    setShowHidden( ( _dir_filter_sets.getShowHidden() == true ) ? false : true );
}

void NormalMode::setShowHidden( bool nhf )
{
    finishsearchmode();
  
    _dir_filter_sets.setShowHidden( nhf );
    rebuildView();
    setName();
    if ( lv != NULL ) lv->showActive();
}

void NormalMode::startAction(FileEntry *tfe)
{
    WCFiletype *ft;
    bool fallbackEnterDir;
    if(tfe==NULL) return;
    if(tfe->isCorrupt==true) return; // Ziel existiert sowieso nicht, alles koennen wir nichts
    // machen

    finishsearchmode();
  
    fallbackEnterDir = false;
    ft = tfe->filetype;
    if ( ft == NULL ) {
        if ( tfe->isDir() == true ) {
            ft = wconfig->getdirtype();
        } else {
            ft = wconfig->getnotyettype();
        }
    }

    if ( ft != NULL ) {
        if ( tfe->isDir() == true ) {
            List *l = ft->getDoubleClickActions();
            if ( l != NULL ) {
                if ( l->size() < 1 ) fallbackEnterDir = true;
            }
        }
        if ( fallbackEnterDir == false ) {
            if ( checkForEmptyActionList( ft ) == true ) {
                // empty action list
                ActionMessage amsg( parentlister->getWorker() );
                StartProgOp spop;

                // prepare command list with startprogop
                List *templist = new List();
                templist->addElement( &spop );
                spop.setStart( StartProgOp::STARTPROGOP_START_IN_TERMINAL_AND_WAIT4KEY );
                spop.setRequestFlags( true );
                spop.setGlobal( true );
                spop.setGUIMsg( catalog.getLocale( 646 ) );
                //TODO Vielleicht auch Moeglichkeit haben, um Initialfenster im startprogop einzustellen

                // workaround for handling ..
                if ( strcmp( tfe->name, ".." ) == 0 ) {
                    amsg.mode = amsg.AM_MODE_SPECIAL;
                    amsg.setFE( tfe );
                } else {
                    prepareContext( amsg );
                }

                amsg.filetype = ft;
                amsg.m_action_descr = RefCount<ActionDescr>( new DoubleClickAction::DoubleClickActionDescr() );
                parentlister->getWorker()->interpret( templist, &amsg );

                delete templist;
            } else {
                ActionMessage amsg( parentlister->getWorker() );

                // workaround for handling ..
                if ( strcmp( tfe->name, ".." ) == 0 ) {
                    amsg.mode = amsg.AM_MODE_SPECIAL;
                    amsg.setFE( tfe );
                } else {
                    prepareContext( amsg );
                }

                amsg.filetype = ft;
                amsg.m_action_descr = RefCount<ActionDescr>( new DoubleClickAction::DoubleClickActionDescr() );
                parentlister->getWorker()->interpret( ft->getDoubleClickActions(), &amsg );
            }
        }
    } else {
        if ( tfe->isDir() == true ) {
            fallbackEnterDir = true;
        }
    }
    if ( fallbackEnterDir == true ) {
        // duplicate fullname because enterDir could delete
        // the directory and with it the fullname
        std::string s1( tfe->fullname );

        // fallback for directories
        enterDir( s1.c_str() );
    }
}

void NormalMode::cyclicfunc(cyclicfunc_mode_t mode)
{
    WCFiletype *ft;
    int trow;
    int side;
    const std::vector<WorkerTypes::listcol_t> *dis;
    bool doflush, dontCheckVirtual;
    time_t now;

    if(mode==CYCLICFUNC_MODE_NORMAL) {
        if(ce!=NULL) {
            FileEntry *fe = ce->getCheckFE();

            side = parentlister->getSide();
            dis = wconfig->getVisCols( side );
            dontCheckVirtual = wconfig->getDontCheckVirtual();
  
            ft = NULL;

            doflush = false;
      
            if ( visChanged == false ) {
                if ( lv->getYOffset() != oldlvy ) visChanged = true;
            }

            // first fill the slave buffer
        
            if ( ( visChanged == true ) && ( reclist->isFull_locked() == false ) ) {
                FileEntry *topfe;
                int toppos;
                int useelements = a_min( lv->getMaxDisplayV(), lv->getElements() - lv->getYOffset() );
          
                toppos = lv->getYOffset();

                topfe = getFEForRow( toppos );
                while ( ( topfe != NULL ) && ( useelements > 0 ) && ( reclist->isFull_locked() == false ) ) {
                    if ( ( topfe->reclistQueued == false ) &&
                         ( topfe->filetype == NULL ) &&
                         ( topfe->isDir() == false ) ) {
                        // add topfe because:
                        // 1.it's visible
                        // 2.type is not checked
                        // 3.it's not queued
                        // 4.reclist is not full
                        // 5.it's a file
                        ft_rec_list::ft_rec_list_t *te = new ft_rec_list::ft_rec_list_t;
                        te->name = topfe->fullname;
                        te->fe = topfe;
                        te->dontCheckContent = ce->getDontCheck();
                        if ( ( te->dontCheckContent == false ) &&
                             ( dontCheckVirtual == true ) ) {
                            if ( worker_islocal( te->name ) == 0 ) te->dontCheckContent = true;
                        }

                        reclist->put_locked( te );
                        m_ft_thread.putRequest( te->name, te->dontCheckContent );
                    }
                    useelements--;
            
                    // find next visible entry
                    toppos++;
                    if ( lv->isValidRow( toppos ) ) {
                        topfe = getFEForRow( toppos );
                    } else topfe = NULL;
                }
                if (  useelements == 0 ) {
                    // all visible are already known or we added them
                    visChanged = false;
                    oldlvy = lv->getYOffset();
                }
            }
        
            while ( ( fe != NULL ) && ( reclist->isFull_locked() == false ) ){
                if ( ( fe->reclistQueued == false ) && ( fe->filetype == NULL ) ) {
                    // when fe was visible we already added it to the list
                    // it's not wrong to still add it but it's just double work
                    ft_rec_list::ft_rec_list_t *te = new ft_rec_list::ft_rec_list_t;
                    te->name = fe->fullname;
                    te->fe = fe;
                    te->dontCheckContent = ce->getDontCheck();
                    if ( ( te->dontCheckContent == false ) &&
                         ( dontCheckVirtual == true ) ) {
                        if ( worker_islocal( te->name ) == 0 ) te->dontCheckContent = true;
                    }
  
                    reclist->put_locked( te );
                    m_ft_thread.putRequest( te->name, te->dontCheckContent );
                }
                ce->next_checkfe();
  
                fe = ce->getCheckFE();
            }
      
            // now check for results
            trow = 0;
        
            m_ft_thread.res_list_lock();
            NM_Filetype_Thread::ft_recres_list_t *rte;
            ft_rec_list::ft_rec_list_t *te;
            while ( m_ft_thread.res_list_is_empty_locked() == false ) {
                // alle Resultate entnehmen
                rte = m_ft_thread.res_list_remove_locked();
                if ( rte != NULL ) {
                    // mit oberstem Element von reclist vergleichen
  
                    te = reclist->gettop_locked();
                    if ( te != NULL ) {
                        if ( strcmp( te->name, rte->name.c_str() ) == 0 ) {
                            // correct answer
                            te = reclist->remove_locked();
#ifdef DEBUG
                            //                printf("type for %s:%d\n",te->name,rte->ft_index);
#endif
                            if ( rte->ft_index != NULL ) {
                                ft = findFiletype( rte->ft_index );
                            } else {
                                ft = wconfig->getunknowntype();
                            }
                
                            if ( ft != NULL ) {
                                // change fe and lvc to filetype
                                fe = te->fe;
                                fe->filetype = ft;
                                if ( rte->custom_color.get() != NULL &&
                                     rte->custom_color->getColorSet() != 0 ) {
                                    fe->setCustomColor( *rte->custom_color );
                                }
  
                                while( lv->isValidRow( trow ) == true ) {
                                    if ( lv->getData( trow ) == fe->getID() ) break;
                                    trow++;
                                }
                                if ( lv->isValidRow( trow ) == false ) {
                                    // lvc not found
                                    // because we start always at the last position
                                    // let's search again from 0

#ifdef DEBUG
                                    printf( "filetype lvc search rollover\n" );
#endif
  
                                    trow = 0;
                                    while( lv->isValidRow( trow ) == true ) {
                                        if ( lv->getData( trow ) == fe->getID() ) break;
                                        trow++;
                                    }
                                }
                  
                                if ( lv->isValidRow( trow ) == true ) {
                                    fe->setLVC_DND( lv, trow, dis );
                                    //lv->redraw( trow );
                                    doflush = true;
                                } else {
                                    // lvc not found
                                    // this is not a problem because filters,... could be changed
                  
                                    // reset pos for next element
                                    trow = 0;
                                }
                  
                                if ( fe == ce->getFirstNULLFT() ) {
                                    // firstnullft is no longer the first element with null ft
                                    ce->next_firstnullft();
                                }
                            }
                
                            delete te;
                        } else {
#ifdef DEBUG
                            printf( "wrong answer: %s\n", rte->name.c_str() );
#endif
                        }
                    }
                    delete rte;
                }
            }
        
            m_ft_thread.res_list_unlock();
      
            if ( doflush == true ) {
#ifdef DEBUG
                printf("Flush\n");
#endif
                lv->simpleRedrawUpdatedRows();
            }
            if ( ce->getFirstNULLFT() != NULL ) {
                // still some work to do so update busyFlag
                now = time( NULL );
                if ( ( lastBusyFlagUpdate < now ) || ( busyFlag == 0 ) ) {
                    lastBusyFlagUpdate = now;
                    busyFlag++;
                    if ( busyFlag > 4 ) busyFlag = 1;
                    setName();
                }
            } else {
                // recognition done so clear busyFlag
                if ( busyFlag > 0 ) {
                    busyFlag = 0;
                    setName();
                }
            }
            showFreeSpace(false);

            updateOnBookmarkChange();
        }
    } else {
        // get slave in reinit state

        // put it in reinit mode
        m_ft_thread.switchOrder( NM_Filetype_Thread::THREAD_STOP );

        // update filetype list
        ft_list_update();
    
        // clear lists
        ft_rec_list_clear();
        ft_recres_list_clear();
    
        m_ft_thread.switchOrder( NM_Filetype_Thread::THREAD_RUN );

        setupLVFields();
  
        NMCacheEntry *tce;
        int id=cache->initEnum();
        tce=(NMCacheEntry*)cache->getFirstElement(id);
        while(tce!=NULL) {
            tce->restartcheck();
            tce->checkfordcd();
            tce=(NMCacheEntry*)cache->getNextElement(id);
        }
        cache->closeEnum(id);

        bookmarksChanged();
        updateOnBookmarkChange();
    }
}

const char* NormalMode::getLocaleName()
{
    return getStaticLocaleName();
}

const char* NormalMode::getStaticLocaleName()
{
    return catalog.getLocale(173);
}

int NormalMode::load()
{
    int found_error = 0;
    int tsortmode;
    NM_Filter *tfi;
    std::string str1;
    std::list<NM_Filter> tfilters;

    tsortmode = SORT_NAME;
    for (;;) {
        if ( worker_token == HIDDENFILES_WCP ) {
            readtoken();

            if ( worker_token != '=' ) {
                found_error = 1;
                break;
            }
            readtoken();

            if ( worker_token == SHOW_WCP ) {
                setShowHidden( true );
            } else if ( worker_token == HIDE_WCP ) {
                setShowHidden( false );
            } else {
                found_error = 1;
                break;
            }
            readtoken();

            if ( worker_token != ';' ) {
                found_error = 1;
                break;
            }
            readtoken();
        } else if ( worker_token == SORTBY_WCP ) {
            readtoken();

            if ( worker_token != '=' ) {
                found_error = 1;
                break;
            }
            readtoken();

            if ( worker_token == SIZE_WCP ) {
                tsortmode = SORT_SIZE | ( tsortmode & ~0xff );
            } else if ( worker_token == ACCTIME_WCP ) {
                tsortmode = SORT_ACCTIME | ( tsortmode & ~0xff );
            } else if ( worker_token == MODTIME_WCP ) {
                tsortmode = SORT_MODTIME | ( tsortmode & ~0xff );
            } else if ( worker_token == CHGTIME_WCP ) {
                tsortmode = SORT_CHGTIME | ( tsortmode & ~0xff );
            } else if ( worker_token == TYPE_WCP ) {
                tsortmode = SORT_TYPE | ( tsortmode & ~0xff );
            } else if ( worker_token == OWNER_WCP ) {
                tsortmode = SORT_OWNER | ( tsortmode & ~0xff );
            } else if ( worker_token == INODE_WCP ) {
                tsortmode = SORT_INODE | ( tsortmode & ~0xff );
            } else if ( worker_token == NLINK_WCP ) {
                tsortmode = SORT_NLINK | ( tsortmode & ~0xff );
            } else if ( worker_token == PERMISSION_WCP ) {
                tsortmode = SORT_PERMISSION | ( tsortmode & ~0xff );
            } else if ( worker_token == NAME_WCP ) {
                tsortmode = SORT_NAME | ( tsortmode & ~0xff );
            } else {
                found_error = 1;
                break;
            }
            readtoken();

            if ( worker_token != ';' ) {
                found_error = 1;
                break;
            }
            readtoken();
        } else if ( worker_token == SORTFLAG_WCP ) {
            readtoken();

            if ( worker_token != '=' ) {
                found_error = 1;
                break;
            }
            readtoken();

            if ( worker_token == REVERSE_WCP ) {
                tsortmode |= SORT_REVERSE;
            } else if ( worker_token == DIRLAST_WCP ) {
                tsortmode |= SORT_DIRLAST;
                tsortmode &= ~SORT_DIRMIXED;
            } else if ( worker_token == DIRMIXED_WCP ) {
                tsortmode |= SORT_DIRMIXED;
                tsortmode &= ~SORT_DIRLAST;
            } else {
                found_error = 1;
                break;
            }
            readtoken();

            if ( worker_token != ';' ) {
                found_error = 1;
                break;
            }
            readtoken();
        } else if ( worker_token == SHOWFREESPACE_WCP ) {
            readtoken();

            if ( worker_token != '=' ) {
                found_error = 1;
                break;
            }
            readtoken();

            if ( worker_token == YES_WCP ) {
                showfreespace = true;
            } else if ( worker_token == NO_WCP ) {
                showfreespace = false;
            } else {
                found_error = 1;
                break;
            }
            readtoken();

            if ( worker_token != ';' ) {
                found_error = 1;
                break;
            }
            readtoken();
        } else if ( worker_token == UPDATETIME_WCP ) {
            readtoken();

            if ( worker_token != '=' ) {
                found_error = 1;
                break;
            }
            readtoken();

            if ( worker_token == NUM_WCP ) {
                setUpdatetime( yylval.num );
            } else {
                found_error = 1;
                break;
            }
            readtoken();

            if ( worker_token != ';' ) {
                found_error = 1;
                break;
            }
            readtoken();
        } else if ( worker_token == FILTER_WCP ) {
            readtoken();

            if ( worker_token != LEFTBRACE_WCP ) {
                found_error = 1;
                break;
            }
            readtoken();
      
            tfi = new NM_Filter();
            if ( tfi->load() == 0 ) {
                tfilters.push_back( *tfi );
            } else {
                delete tfi;
                found_error = 1;
                break;
            }
            delete tfi;

            if ( worker_token != RIGHTBRACE_WCP ) {
                found_error = 1;
                break;
            }
            readtoken();
        } else if ( worker_token == SSHALLOW_WCP ) {
            readtoken();

            if ( worker_token != '=' ) {
                found_error = 1;
                break;
            }
            readtoken();

            if ( worker_token == NEVER_WCP ) {
                ssh_allow = SSH_NEVER;
            } else if ( worker_token == ALWAYS_WCP ) {
                ssh_allow = SSH_ALWAYS;
            } else if ( worker_token == ASK_WCP ) {
                ssh_allow = SSH_ASK;
            } else {
                found_error = 1;
                break;
            }
            readtoken();

            if ( worker_token != ';' ) {
                found_error = 1;
                break;
            }
            readtoken();
        } else if ( worker_token == FIELD_WIDTH_WCP ) {
            readtoken();

            if ( worker_token != '=' ) {
                found_error = 1;
                break;
            }
            readtoken();

            if ( worker_token != STRING_WCP ) {
                found_error = 1;
                break;
            }

            str1 = yylval.strptr;

            readtoken();
            if ( worker_token != ',' ) {
                found_error = 1;
                break;
            }
            readtoken();
            if ( worker_token != NUM_WCP ) {
                found_error = 1;
                break;
            }
            _loaded_field_width[str1] = yylval.num;

            readtoken();
            if ( worker_token != ';' ) {
                found_error = 1;
                break;
            }
            readtoken();
        } else if ( worker_token == QUICKSEARCHENABLED_WCP ) {
            readtoken();

            if ( worker_token != '=' ) {
                found_error = 1;
                break;
            }
            readtoken();

            if ( worker_token == YES_WCP ) {
                quicksearch_enabled = true;
            } else if ( worker_token == NO_WCP ) {
                quicksearch_enabled = false;
            } else {
                found_error = 1;
                break;
            }
            readtoken();

            if ( worker_token != ';' ) {
                found_error = 1;
                break;
            }
            readtoken();
        } else if ( worker_token == FILTEREDSEARCHENABLED_WCP ) {
            readtoken();

            if ( worker_token != '=' ) {
                found_error = 1;
                break;
            }
            readtoken();

            if ( worker_token == YES_WCP ) {
                setFilteredSearchEnabled( true );
            } else if ( worker_token == NO_WCP ) {
                setFilteredSearchEnabled( false );
            } else {
                found_error = 1;
                break;
            }
            readtoken();

            if ( worker_token != ';' ) {
                found_error = 1;
                break;
            }
            readtoken();
        } else {
            break;
        }
    }
    setSortmode( tsortmode );
    setFilters( tfilters );
  
    return found_error;
}

//TODO kann das nicht mal weg?
int NormalMode::loadBin(Datei *fh)
{
    int chunksize=fh->getInt();

    int cs=Datei::getUCharSize(),
        is=Datei::getIntSize();
    unsigned char c1;
    int size;
    int tsortmode=0;
    bool tshowHidden=false;
    NM_Filter *tfi;
    std::list<NM_Filter> tfilters;

    if(chunksize>=(4*cs+is)) {
        tshowHidden=(fh->getUChar()==1)?true:false;
        c1=fh->getUChar();
        switch(c1) {
            case 1:
                tsortmode=SORT_SIZE;
                break;
            case 2:
                tsortmode=SORT_ACCTIME;
                break;
            case 3:
                tsortmode=SORT_MODTIME;
                break;
            case 4:
                tsortmode=SORT_CHGTIME;
                break;
            case 5:
                tsortmode=SORT_TYPE;
                break;
            case 6:
                tsortmode=SORT_OWNER;
                break;
            default:
                tsortmode=SORT_NAME;
                break;
        }
        if(fh->getUChar()==1) tsortmode|=SORT_REVERSE;
        c1=fh->getUChar();
        switch(c1) {
            case 1:
                tsortmode|=SORT_DIRLAST;
                break;
            case 2:
                tsortmode=SORT_DIRMIXED;
                break;
            default:
                break;
        }
        size=fh->getInt();
        chunksize-=4*cs+is;
    
        for(int i=0;i<size;i++) {
            tfi=new NM_Filter();
            tfi->loadBin( fh );
            tfilters.push_back( *tfi );
            delete tfi;
        }
        // apply new values
        setShowHidden(tshowHidden);
        setSortmode(tsortmode);
        setFilters( tfilters );
        // this is added from version 2.0.2 so for file from older version
        if(chunksize>=(cs+is)) {
            showfreespace=(fh->getUChar()==1)?true:false;
            chunksize-=cs;
            setUpdatetime(fh->getInt());
            chunksize-=is;
        }
    }
    while(chunksize>0) {
        fh->getUChar();
        chunksize--;
    }
    return 0;
}

bool NormalMode::save(Datei *fh)
{
    char numbuf[A_BYTESFORNUMBER( int )];
    std::map<std::string,int>::iterator it1;
    std::string str1;

    if ( fh == NULL ) return false;
 
    fh->configPutPair( "hiddenfiles" , ( _dir_filter_sets.getShowHidden() == true ) ? "show" : "hide" );
    switch ( sortmode & 0xff ) {
        case SORT_SIZE:
            fh->configPutPair( "sortby", "size" );
            break;
        case SORT_ACCTIME:
            fh->configPutPair( "sortby", "acctime" );
            break;
        case SORT_MODTIME:
            fh->configPutPair( "sortby", "modtime" );
            break;
        case SORT_CHGTIME:
            fh->configPutPair( "sortby", "chgtime" );
            break;
        case SORT_TYPE:
            fh->configPutPair( "sortby", "type" );
            break;
        case SORT_OWNER:
            fh->configPutPair( "sortby", "owner" );
            break;
        case SORT_INODE:
            fh->configPutPair( "sortby", "inode" );
            break;
        case SORT_NLINK:
            fh->configPutPair( "sortby", "nlink" );
            break;
        case SORT_PERMISSION:
            fh->configPutPair( "sortby", "permission" );
            break;
        default:
            fh->configPutPair( "sortby", "name" );
            break;
    }
    if ( ( sortmode & SORT_REVERSE ) != 0 ) fh->configPutPair( "sortflag", "reverse" );
    if ( ( sortmode & SORT_DIRLAST ) != 0 ) fh->configPutPair( "sortflag", "dirlast" );
    else if ( ( sortmode & SORT_DIRMIXED ) != 0 ) fh->configPutPair( "sortflag", "dirmixed" );
  
    fh->configPutPairBool( "showfreespace", showfreespace );
    fh->configPutPairNum( "updatetime", updatetime );

    fh->configPutPairBool( "quicksearchenabled", quicksearch_enabled );
    fh->configPutPairBool( "filteredsearchenabled", _filtered_search_enabled );

    switch ( ssh_allow ) {
        case SSH_NEVER:
            fh->configPutPair( "sshallow", "never" );
            break;
        case SSH_ALWAYS:
            fh->configPutPair( "sshallow", "always" );
            break;
        default:
            fh->configPutPair( "sshallow", "ask" );
            break;
    }

    if ( lv != NULL ) {
        writeFieldWidthsToLoadMap();
    }

    for ( it1 = _loaded_field_width.begin(); it1 != _loaded_field_width.end(); it1++ ) {
        sprintf( numbuf, "%d", (*it1).second );
        str1 = "field_width = ";
        str1 += "\"";
        str1 += (*it1).first;
        str1 += "\"";
        str1 += ",";
        str1 += numbuf;
        str1 += ";";
        fh->configPutInfo( str1.c_str(), false );
    }

    std::list<NM_Filter> filters = _dir_filter_sets.getFilters();
    std::list<NM_Filter>::iterator fil_it;

    for ( fil_it = filters.begin();
          fil_it != filters.end();
          fil_it++ ) {
        fh->configOpenSection( "filter" );
        (*fil_it).save( fh );
        fh->configCloseSection(); //filter
    }
    return false;
}

void NormalMode::top()
{
    if ( ce == NULL ) return;
    
    finishsearchmode();
    
    makeRowActive( 0 );
    lv->showActive();
}

void NormalMode::last()
{
    int row;
    
    if ( ce == NULL ) return;
    
    finishsearchmode();
    
    row = lv->getElements() - 1;
    if ( lv->isValidRow( row ) == true ) {
        makeRowActive( row );
        lv->showActive();
    }
}

void NormalMode::pageup()
{
    int row;
    
    if ( ce == NULL ) return;
    
    finishsearchmode();

    if ( ce->getActiveFE() == NULL ) {
        makeRowActive( 0 );
        lv->showActive();
    } else {
        row = lv->getActiveRow() - lv->getMaxDisplayV() + 1;
        if ( lv->isValidRow( row ) == false )
            row = 0;
        
        makeRowActive( row );
        lv->showActive();
    }
}

void NormalMode::pagedown()
{
    int row;
    
    if ( ce == NULL ) return;
    
    finishsearchmode();

    if ( ce->getActiveFE() == NULL ) {
        makeRowActive( 0 );
        lv->showActive();
    } else {
        row = lv->getActiveRow() + lv->getMaxDisplayV() - 1;
        if ( lv->isValidRow( row ) == false )
            row = lv->getElements() - 1;
        if ( lv->isValidRow( row ) == false )
            row = 0;
        
        makeRowActive( row );
        lv->showActive();
    }
}

void NormalMode::select()
{
    if(ce==NULL) return;
    if ( ce->getActiveFE() == NULL ) return;

    finishsearchmode();
  
    int row=lv->getActiveRow();
    if( row < 0 ) {
        printf("oops, this should never happen!\n");
        return;
    }
    if(lv->getSelect(row)==false) lv->setSelect(row, true); else lv->setSelect(row,false);
    selhandler(lv,row );
    down();
}

void NormalMode::selectall()
{
    if(ce==NULL) return;
    int row;
  
    finishsearchmode();
  
    /* LVCs auswaehlen */
    row = 1;
    while( row < lv->getElements() ) {
        lv->setSelect(row,true);
        row++;
    }
    /* FEs auswaehlen */

    for ( Verzeichnis::verz_it it1 = ce->begin();
          it1 != ce->end();
          it1++ ) {
        FileEntry *fe = *it1;
        if ( fe->use == true && strcmp( fe->name, ".." ) != 0 )
            fe->select = true;
    }
    ce->recalcStats();
    lv->redraw();
    showCacheState();
}

void NormalMode::selectnone()
{
    if(ce==NULL) return;
    int row;
  
    finishsearchmode();
  
    /* LVCs deselecten */
    row = 0;
    while( row < lv->getElements()) {
        lv->setSelect(row,false);
        row++;
    }
    /* FEs deselecten */
    for ( Verzeichnis::verz_it it1 = ce->begin();
          it1 != ce->end();
          it1++ ) {
        FileEntry *fe = *it1;
        if ( fe->use == true )
            fe->select = false;
    }
    ce->recalcStats();
    lv->redraw();
    showCacheState();
}

void NormalMode::invertall()
{
    if(ce==NULL) return;
    int row;
  
    finishsearchmode();
  
    /* LVCs invertieren */
    row = 1;
    while(row<lv->getElements()) {
        lv->setSelect(row,lv->getSelect(row)==true?false:true);
        row++;
    }
    /* FEs invertieren */
    for ( Verzeichnis::verz_it it1 = ce->begin();
          it1 != ce->end();
          it1++ ) {
        FileEntry *fe = *it1;
        if ( fe->use == true && strcmp( fe->name, ".." ) != 0 )
            fe->select = ( fe->select == true ? false : true );
    }
    ce->recalcStats();
    lv->redraw();
    showCacheState();
}

void NormalMode::parent()
{
    char oentry[128];
    if(ce==NULL) return;
    int row;
  
    const char *adir=ce->getDir();
    char *ndir=ParentDir(adir,oentry,127);
    enterDir(ndir);
    /* oentry finden und aktivieren */
    FileEntry *fe = NULL;
    for ( Verzeichnis::verz_it it1 = ce->begin();
          it1 != ce->end();
          it1++ ) {
        fe = *it1;
        if ( fe->use == true )
            if ( strcmp( fe->name, oentry ) == 0 )
                break;
        fe = NULL;
    }
    if ( fe!=NULL) {
        /* Eintrag gefunden */
        ce->setActiveFE( fe );
        row = 0;
        while( row < lv->getElements() ) {
            if ( lv->getData(row) == fe->getID() ) break;
            row++;
        }
        if( row < lv->getElements()) {
            makeListViewRowActive(row);
            lv->showActive();
            aguix->Flush();
        }
    }
    _freesafe(ndir);
}

void NormalMode::enterDirActive()
{
    if(ce==NULL) return;
    if ( ce->getActiveFE() != NULL ) {
        if ( ce->getActiveFE()->isDir() == true ) {
            // copy fullname because enterDir can delete directory
            std::string s1( ce->getActiveFE()->fullname );
            enterDir( s1.c_str() );
        }
    }
}

void NormalMode::filterselect( const char *filter, int mode )
{
    /* Steht in filter[0] ein /, werden Verzeichnisse betrachtet, sonst Dateien */
    if(ce==NULL) return;
    int row;
    const char *realfilter;
    bool tselect=(mode==0)?true:false;
    bool usedir;
  
    finishsearchmode();
  
    if(filter[0]=='/') {
        usedir=true;
        realfilter=&filter[1];
    } else {
        usedir=false;
        realfilter=filter;
    }
  
    for ( Verzeichnis::verz_it it1 = ce->begin();
          it1 != ce->end();
          it1++ ) {
        FileEntry *fe = *it1;
        if ( fe->use == true && strcmp( fe->name, ".." ) != 0 )
            if ( fe->match( realfilter ) == true ) 
                if ( fe->isDir() == usedir )
                    fe->select = tselect;
    }

    row = 0;
    Verzeichnis::verz_it it2 = ce->begin();
    while( row < lv->getElements()) {
        while ( (*it2)->getID() != lv->getData( row ) )
            it2++;
        lv->setSelect( row, (*it2)->select );
        row++;
    }
    lv->redraw();
    ce->recalcStats();
    showCacheState();
}

const char *NormalMode::getCurrentDir()
{
    if ( ce != NULL ) return ce->getDir();
    return NULL;
}

bool
NormalMode::isCorrectViewProg(const char *str)
{
    const char *pos;
    if(strlen(str)<1) return true;
    pos=strstr(str,"%s");
    if(pos!=NULL) return true;
    return false;
}

void NormalMode::removeEntry( FileEntry *fe, int row )
{
    if ( ce == NULL ) return;
    if ( fe == NULL ) return;
    if ( ce->dirOpened() == false ) return;
    if ( lv->isValidRow( row ) == false ||
         row == 0 )
        return;
    
    if ( getFEForRow( row ) != fe )
        return;
    
    // we will remove an entry so clear this list
    // so it won't contain wrong pointers
    ft_rec_list_clear();
    
    // if it's the active one then move the active up by one
    if ( fe == ce->getActiveFE() ) {
        int prev_row = row - 1;
        
        ce->setActiveFE( getFEForRow( prev_row ) );
        makeListViewRowActive( prev_row );
        lv->showActive();
    }
    
    ce->removeEntry( fe );
    
    lv->deleteRow( row );
    //TODO make it optional to use FPS limit
    lv->timeLimitedRedraw();
    
    delete fe;
    
    // update cache info
    showCacheState();
}

void
NormalMode::update(bool reset_dirsizes)
{
    update( reset_dirsizes, false );
}

void
NormalMode::update( bool reset_dirsizes, bool keep_filetypes )
{
    if ( ce == NULL )
        return;
    
    char *dir;
  
    finishsearchmode();
    
    dir = dupstring( ce->getDir() );
    for(;;) {
        int e = enterDir( dir );
        if ( e == 0 ) break;  // read ok, so finish
        if ( strcmp( dir, "/" ) == 0 ) break;  // there is no dir we can display anymore, so finish
        char *tstr = ParentDir( dir, NULL );
        _freesafe( dir );
        dir = tstr;
    }
    _freesafe( dir );

    if ( ce != NULL ) {
        if ( keep_filetypes == true ) {
            ce->reset_checkfe();
        } else {
            ce->restartcheck();
        }
        if ( reset_dirsizes == true )
            ce->reset_dirsizes();
    }
    
    rebuildView();
}

void
NormalMode::makedir()
{
    if(ce==NULL) return;
    FileEntry *tfe;
    int row;
    char *buttonstr,*textstr,*return_str,*destdir;
    int erg;
  
    finishsearchmode();
  
    textstr = dupstring( catalog.getLocale(152) );
    buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                strlen(catalog.getLocale(8))+1);
    sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
            catalog.getLocale(8));
    erg=string_request(catalog.getLocaleCom(27),textstr,"",buttonstr,&return_str);
    _freesafe(buttonstr);
    _freesafe( textstr );
    if((erg==0)&&(strlen(return_str)>0)) {
        destdir=(char*)_allocsafe(strlen(ce->getDir())+1+strlen(return_str)+1);
        strcpy(destdir,ce->getDir());
        if(strlen(destdir)>1) strcat(destdir,"/");
        strcat(destdir,return_str);

        if ( worker_mkdir( destdir, S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH ) != 0 ) {
            textstr=(char*)_allocsafe(strlen(catalog.getLocale(127))+strlen(destdir)+1);
            sprintf(textstr,catalog.getLocale(127),destdir);
            buttonstr = dupstring( catalog.getLocale(11) );
            erg=request(catalog.getLocale(347),textstr,buttonstr);
            _freesafe(textstr);
            _freesafe( buttonstr );
        } else {
            // update the dir
            update(false);
            // find the new dir
            for ( Verzeichnis::verz_it tfe_it1 = ce->begin();
                  tfe_it1 != ce->end();
                  tfe_it1++ ) {
                tfe = *tfe_it1;
                if(strcmp(tfe->name,return_str)==0) {
                    // found
                    ce->setActiveFE( tfe );
                    row = 0;
                    while( row < lv->getElements() ) {
                        if ( lv->getData( row ) == tfe->getID() ) {
                            // lvc found
                            makeListViewRowActive( row );
                            break;
                        }
                        row++;
                    }
                    break;
                }
            }
            lv->showActive();
            lv->redraw();
        }
        _freesafe(destdir);
    }
    _freesafe(return_str);
}

void NormalMode::deselect( FileEntry *fe, int row)
{
    if((lv->isValidRow(row)==false)||(fe==NULL)) return;
    if(ce==NULL) return;
    if(fe->select==false) return;
    if(lv->isValidRow(row)) {
        lv->setSelect(row,false);
        lv->showRow(row);
    }
    fe->select=false;
    ce->recalcStats();
    showCacheState();
}

void NormalMode::renamef(struct NM_renameorder *renorder)
{
    NM_specialsourceInt *ss1;
    FileEntry *fe;
    int erg;
    bool skip,cancel,nameok;
    char *newname,*buttonstr,*textstr,*newfullname,*return_str;
    const std::vector<WorkerTypes::listcol_t> *dis;
    std::list<NM_specialsourceInt*> *renlist;
    std::list<NM_specialsourceInt*>::iterator iti1;
#if 0
    int oldx,oldy;
#endif

    if ( getCurrentDir() == NULL ) {
        // no current dir => no files to rename
        return;
    }

    finishsearchmode();
  
    dis = wconfig->getVisCols( parentlister->getSide() );
  
    renlist = new std::list<NM_specialsourceInt*>;
    switch(renorder->source) {
        case NM_renameorder::NM_SPECIAL:
            buildSpecialSourceIntList( *( renorder->sources ),
                                       *renlist );
            break;
        case NM_renameorder::NM_ONLYACTIVE:
            getSelFiles( renlist, NM_GETFILES_ONLYACTIVE, false );
            break;
        default:  // all selected entries
            getSelFiles( renlist, NM_GETFILES_SELORACT, false );
            break;
    }
  
    cancel=skip=false;

    for ( iti1 = renlist->begin(); iti1 != renlist->end(); iti1++ ) {
        if ( cancel == true ) break;
        ss1 = *iti1;
        // now rename ss1->entry
        if ( ss1->row >= 0 ) {
            lv->setVisMark( ss1->row, true );
            lv->showRow( ss1->row );
        }
        fe = ss1->entry();
        if(fe!=NULL) {
            newname=dupstring(fe->name);
            newfullname=NULL;
            for(nameok=false;nameok==false;) {
                buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                            strlen(catalog.getLocale(225))+1+
                                            strlen(catalog.getLocale(8))+1);
                sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                        catalog.getLocale(225),
                        catalog.getLocale(8));
                textstr=(char*)_allocsafe(strlen(catalog.getLocale(148))+strlen(fe->name)+1);
                sprintf(textstr,catalog.getLocale(148),fe->name);
                erg=string_request(catalog.getLocale(149),textstr,newname,buttonstr,&return_str, Requester::REQUEST_SELECTALL );
                _freesafe(buttonstr);
                _freesafe(textstr);
                _freesafe(newname);
                newname=return_str;
                if(erg==2) {
                    cancel=true;
                    break;
                } else if(erg==1) {
                    break;
                } else {
                    if(strcmp(newname,fe->name)==0) {
                        // same name as orig
                        // so skip this entry
                        break;
                    } else if ( strlen( newname ) < 1 ) {
                        // empty string->repeat
                        continue;
                    }
                    newfullname = (char*)_allocsafe( strlen( getCurrentDir() ) + 1 + strlen( newname ) + 1 );
                    if ( strlen( getCurrentDir() ) > 1 ) sprintf( newfullname, "%s/%s", getCurrentDir(), newname );
                    else sprintf( newfullname, "%s%s", getCurrentDir(), newname );
                    if(Datei::lfileExistsExt(newfullname)!=Datei::D_FE_NOFILE) {
                        textstr=(char*)_allocsafe(strlen(catalog.getLocale(279))+strlen(newname)+1);
                        sprintf(textstr,catalog.getLocale(279),newname);
                        buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                                    strlen(catalog.getLocale(225))+1+
                                                    strlen(catalog.getLocale(8))+1);
                        sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(230),
                                catalog.getLocale(225),
                                catalog.getLocale(8));
                        erg=request(catalog.getLocale(123),textstr,buttonstr);
                        _freesafe(buttonstr);
                        _freesafe(textstr);
                        if(erg==1) break;
                        else if(erg==2) {
                            cancel=true;
                            break;
                        }
                        _freesafe(newfullname);
                        newfullname=NULL;
                    } else {
                        nameok=true;
                    }
                }
            }
            if((nameok==true)&&(newfullname!=NULL)) {
                erg = worker_rename( fe->fullname, newfullname );
                if(erg!=0) {
                    textstr = dupstring( catalog.getLocale(151) );
                    buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                                strlen(catalog.getLocale(8))+1);
                    sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                            catalog.getLocale(8));
                    erg=request(catalog.getLocale(347),textstr,buttonstr);
                    _freesafe(buttonstr);
                    _freesafe( textstr );
                    if(erg==1) cancel=true;
                } else {
                    // deselect and change lvc temp.
                    if(lv->isValidRow(ss1->row)==true) deselect(fe,ss1->row);
                    //TODO: Man koennte eine helper-Funktion einfuehren, um
                    //      name/fullname zu setzen
                    _freesafe(fe->name);
                    _freesafe(fe->fullname);
                    fe->name=newname;
                    newname=NULL;
                    if(fe->name[0]=='.') fe->isHidden=true;
                    else fe->isHidden=false;          
                    fe->fullname=newfullname;
                    newfullname=NULL;
                    if(lv->isValidRow(ss1->row)==true) {
                        //TODO: Eigentlich muesste hier auch das DND neugesetzt werden, aber ich weiss nicht die Position
                        // d.h. ich muesste erst suchen.
                        // allerdings kann das auch so bleiben,da am Ende ohnehin buildLister gemacht wird
                        fe->setLVC( lv, ss1->row, dis );
                        lv->redraw();
                        aguix->Flush();
                    }
                }
            }
            if(newname!=NULL) _freesafe(newname);
            if(newfullname!=NULL) _freesafe(newfullname);
        }
        if ( ss1->row >= 0 ) {
            lv->setVisMark( ss1->row, false );
        }
    }
  
    for ( iti1 = renlist->begin(); iti1 != renlist->end(); iti1++ ) {
        ss1 = *iti1;
        delete ss1;
    }
    delete renlist;
    // now update
  
    update(false);
    lv->showActive();
}

void NormalMode::dirsize(struct NM_dirsizeorder *dsorder)
{
    NM_specialsourceInt *ss1;
    bool enter;
    const std::vector<WorkerTypes::listcol_t> *dis;
    int oldx,oldy;
    std::list<NM_specialsourceInt*> *dslist;
    std::list<NM_specialsourceInt*>::iterator iti1;
  
    finishsearchmode();
  
    oldx=lv->getXOffset();
    oldy=lv->getYOffset();

    if(dsorder==NULL) return;
    if ( getCurrentDir() == NULL ) return;
  
    dis = wconfig->getVisCols( parentlister->getSide() );
  
    dslist = new std::list<NM_specialsourceInt*>;
    switch(dsorder->source) {
        case NM_dirsizeorder::NM_SPECIAL:
            buildSpecialSourceIntList( *( dsorder->sources ),
                                       *dslist );
            break;
        case NM_dirsizeorder::NM_ONLYACTIVE:
            getSelFiles( dslist, NM_GETFILES_ONLYACTIVE, false );
            break;
        default:  // all selected entries
            getSelFiles( dslist, NM_GETFILES_SELORACT, false );
            break;
    }

    // create the NM_CopyOp_Dir for each dir in copylist
  
    parentlister->getWorker()->setWaitCursor();
    aguix->getLastKeyRelease();
    aguix->getLastMouseRelease();

    bool cancel = false;

    for ( iti1 = dslist->begin(); iti1 != dslist->end() && cancel == false; iti1++ ) {
        ss1 = *iti1;
        enter=false;
        if ( ss1->entry()->isDir() == true ) {
            if ( ss1->entry()->isLink == false ) enter = true;
        }
        if(enter==true) {
            if ( getDirSize( ss1->entry(), ss1->entry()->dirsize, cancel ) == 0 ) {
                if(lv->isValidRow(ss1->row)==true) {
                    ss1->entry()->setLVC( lv, ss1->row, dis );
                    lv->redraw();
                    aguix->Flush();
                }
            } else {
                ss1->entry()->dirsize = -1;
            }
        }
    }

    for ( iti1 = dslist->begin(); iti1 != dslist->end(); iti1++ ) {
        ss1 = *iti1;
        delete ss1;
    }
    delete dslist;

    // now update
    ce->forceUpdate();
    rebuildView();
    lv->setYOffset(oldy);
    lv->setXOffset(oldx);
    lv->redraw();

    parentlister->getWorker()->unsetWaitCursor();
}

int
NormalMode::getDirSize( FileEntry*rootfe, loff_t &return_size, bool &cancel )
{
    FileEntry *fe;
    bool enter;
    loff_t size,recsize;
    Verzeichnis *verz=new Verzeichnis();
    time_t now;
  
    if(verz->readDir(rootfe->fullname)!=0) {
        delete verz;
        return -1;
    }
  
    if ( verz->dirOpened() == false ) {
        delete verz;
        return -1;
    }

    now = time( NULL );
    if ( m_last_statebar_update != now ) {
        FileNameShrinker fs;
        AFontWidth fw( aguix, NULL );
        std::string shrinked_text;
      
        shrinked_text = fs.shrink( rootfe->fullname,
                                   parentlister->getWorker()->getStatebarWidth(),
                                   fw );

        parentlister->setStatebarText( shrinked_text.c_str() );
        m_last_statebar_update = now;
    }

    size=0;
    for ( Verzeichnis::verz_it fe_it1 = verz->begin();
          fe_it1 != verz->end() && cancel == false;
          fe_it1++ ) {
        fe = *fe_it1;
        if(strcmp(fe->name,"..")!=0) {
            enter=false;
            if(fe->isDir()==true) {
                if(fe->isLink==false) enter=true;
            }
            if(enter==true) {
                recsize = 0;
                if ( getDirSize( fe, recsize, cancel ) == 0 ) {
                    if(recsize>0) size+=recsize;
                }
            } else {
                // is not dir (mostly a file but can also be links ...)
                size += fe->size();
            }
        }
    }

    //TODO this is a rather ugly hack to handle redraws and key presses
    aguix->doXMsgs( NULL, false );
    if ( aguix->getLastKeyRelease() == XK_Escape ) {
        cancel = true;
    }

    delete verz;
    return_size = size;

    // return a negative value to indicate that the return_size is not valid
    return ( cancel == true ) ? -2 : 0;
}

void NormalMode::simdd()
{
    if ( getCurrentDir() == NULL ) {
        return;
    }
    if ( ce->getActiveFE() != NULL ) startAction( ce->getActiveFE() );
}

void NormalMode::searchentry( bool ignore_case, bool reverse_search, bool infix_search )
{
    setEntrySearchCaseSensitive( ( ignore_case == true ) ? false : true );

    m_filtered_search_infix_search = infix_search;

    if(searchmodeon==false) startsearchmode();
    else {
        shownextentry( NULL, reverse_search );
    }
}

void NormalMode::startsearchmode()
{
    if ( getCurrentDir() == NULL ) {
        return;
    }
    oldstr=dupstring(sg->getText());
    lastsearch=dupstring("");
    sg->setText("");
    sg->setForbidPosChange(true);
    sg->setStrongKeyCapture(false);
    sg->activate();
    searchmodeon=true;

    m_flexible_matching_active = false;

    setEntrySearchString( "" );

    parentlister->getWorker()->setIgnoreKey( XK_Tab, true );
}

void NormalMode::finishsearchmode()
{
    if(searchmodeon==false) return;
    sg->setForbidPosChange(false);
    sg->setStrongKeyCapture(true);
    sg->setText(oldstr);
    sg->deactivate();
    _freesafe(oldstr);
    if(lastsearch!=NULL) _freesafe(lastsearch);
    else debugmsg("oops2\n");
    lastsearch=NULL;
    searchmodeon=false;

    parentlister->getWorker()->setIgnoreKey( XK_Tab, false );
}

bool NormalMode::shownextentry( const char *str, bool reverse_search, bool start_at_same )
{
    FileEntry *fe;
    bool found;
    int row;
    std::string old_string_filter, string_filter;
  
    found = false;

    if ( getCurrentDir() == NULL ) {
        return found;
    }

    if ( str == NULL )
        str = lastsearch;
    if ( str == NULL )
        return found;

    old_string_filter = getEntrySearchString();

    if ( m_filtered_search_infix_search == true ) {
        string_filter = "*";
    } else {
        string_filter = "";
    }

    string_filter += str;
    string_filter += "*";

    if ( string_filter == "**" ) {
        string_filter = "*";
    }

    bool flexible_matching_enabled = getFlexibleMatchingMode();

    // store old value since we change it here
    bool old_flexible_matching_activated = m_flexible_matching_active;
    // flag for forcing setEntrySearchString() which takes more time
    bool force_comlete_set_string = false;
    // flag when the search string must be restored
    bool restore_search_string = false;

    FileEntry *old_active_fe = ce->getActiveFE();

    // check whether we need to test for enabling or disabling flexible matching
    if ( flexible_matching_enabled && lastsearch != NULL ) {
        if ( m_flexible_matching_active == false &&
             strlen( str ) > strlen( lastsearch ) ) {

            _entrysearch_matcher.setMatchString( string_filter );

            Verzeichnis::verz_it pos_it1;

            for ( pos_it1 = ce->begin();
                  pos_it1 != ce->end();
                  pos_it1++ ) {

                fe = *pos_it1;
                if ( fe->use == true ) {
                    if ( _entrysearch_matcher.match( fe->name ) == true ) {
                        break;
                    }
                }
            }

            if ( pos_it1 != ce->end() ) {
                // found an entry so continue using no flexible matching
            } else {
                // no entry found so enable flexible matching
                m_flexible_matching_active = true;
                force_comlete_set_string = true;
            }
        } else if ( m_flexible_matching_active == true && 
                    strlen( str ) < strlen( lastsearch ) ) {
            // test whether it matches without flexible matching
            setEntrySearchString( string_filter, false );

            restore_search_string = true;

            Verzeichnis::verz_it pos_it1;

            for ( pos_it1 = ce->begin();
                  pos_it1 != ce->end();
                  pos_it1++ ) {
                if ( (*pos_it1)->use ) {
                    if ( strcmp( (*pos_it1)->name, ".." ) != 0 ) {
                        break;
                    }
                }
            }

            if ( pos_it1 != ce->end() ) {
                // found an entry so disable flexible matching
                m_flexible_matching_active = false;
            } else {
                // no entry found so continue using flexible matching
                force_comlete_set_string = true;
            }
        }
    }

    if ( m_flexible_matching_active ) {
        string_filter = "*" + StringMatcherFNMatch::convertMatchStringToFlexibleMatch( string_filter ) + "*";
    }

    if ( force_comlete_set_string ) {
        setEntrySearchString( string_filter, false );
        restore_search_string = true;
    } else {
        /* first test whether an entry exists so don't use setEntrySearchString here
         * as it would also modify dirfilter */
        _entrysearch_matcher.setMatchString( string_filter );
    }

    Verzeichnis::verz_it pos_it1 = ce->begin();

    if ( ce->getActiveFE() != NULL ) {
        for ( pos_it1 = ce->begin();
              pos_it1 != ce->end();
              pos_it1++ ) {
            if ( (*pos_it1) == ce->getActiveFE() )
                break;
        }
        if ( start_at_same == false ) {
            if ( reverse_search == false ) {
                if ( pos_it1 != ce->end() )
                    pos_it1++;
                if ( pos_it1 == ce->end() )
                    pos_it1 = ce->begin();
            } else {
                if ( pos_it1 != ce->begin() )
                    pos_it1--;
                else {
                    pos_it1 = ce->end();
                    pos_it1--;
                }
            }
        }
    }
  
    for ( int i = 0; i < 2 && found == false; i++ ) {
        while ( pos_it1 != ce->end() && found == false ) {
            fe = *pos_it1;
            if ( fe->use == true ) {
                if ( _entrysearch_matcher.match( fe->name ) == true ) {
                    // hit
                    ce->setActiveFE( fe );
                    row = 0;
                    while(lv->isValidRow(row)==true) {
                        if ( lv->getData( row ) == fe->getID() ) {
                            makeListViewRowActive( row );
                            break;
                        }
                        row++;
                    }
                    found = true;
                }
            }
            if ( reverse_search == true ) {
                if ( pos_it1 != ce->begin() )
                    pos_it1--;
                else
                    pos_it1 = ce->end();  // set to end to stop while loop
            } else {
                pos_it1++;
            }
        }
        if ( reverse_search == true ) {
            pos_it1 = ce->end();
            pos_it1--;
        } else {
            pos_it1 = ce->begin(); // if not found try from pos 0 again
        }
    }

    if ( found == true ) {
        setEntrySearchString( string_filter );
        //TODO do rebuildView only if filtered search is enabled
        lv->showActive();
        lv->redraw();

        showSearchModeCompletion( string_filter );
    } else {
        if ( old_flexible_matching_activated != m_flexible_matching_active ) {
            m_flexible_matching_active = old_flexible_matching_activated;
        }

        if ( restore_search_string ) {
            ce->setActiveFE( old_active_fe );
            setEntrySearchString( old_string_filter );
            lv->showActive();
            lv->redraw();
        } else {
            _entrysearch_matcher.setMatchString( old_string_filter );
        }
    }
  
    return found;
}

void NormalMode::enterpath()
{
    sg->activate();
}

void NormalMode::scrolllister(int dir)
{
    lv->setXOffset( lv->getXOffset() + dir * lv->getHScrollStep() );
    aguix->Flush();
}

void NormalMode::setActiveDir2Other()
{
    Lister *l1;
    ListerMode *lm1;
    NormalMode *nm1;

    if(ce==NULL) return;
    if ( ce->getActiveFE() != NULL ) {
        if ( ce->getActiveFE()->isDir() == true ) {
            l1=parentlister->getWorker()->getOtherLister(parentlister);
            if(l1!=NULL) {
                l1->switch2Mode(0);
                lm1=l1->getActiveMode();
                if(lm1!=NULL) {
                    if(lm1->isType("NormalMode")==true) {
                        std::string s1( ce->getActiveFE()->fullname );

                        nm1=(NormalMode*)lm1;
                        nm1->enterDir( s1.c_str() );
                    }
                }
            }
        }
    }
}

void NormalMode::createsymlink(struct NM_createsymlinkorder *cslorder)
{
    NM_specialsourceInt *ss1;
    FileEntry *fe;
    int erg;
    bool skip,cancel,nameok;
    char *newname,*buttonstr,*textstr,*newfullname,*return_str;
    char *sldest;
    Lister *olister=NULL;
    NormalMode *nm2=NULL,*updatenm=NULL;
    ListerMode *lm=NULL;
    std::list<NM_specialsourceInt*> *csllist;
    std::list<NM_specialsourceInt*>::iterator iti1;

    if ( getCurrentDir() == NULL ) {
        // no current dir => no files to rename
        return;
    }

    if ( ce->dirOpened() == false )
        return;

    finishsearchmode();
  
    if ( strcmp( getCurrentDir(), cslorder->destdir ) == 0 ) {
        updatenm=this;
    }
    olister=parentlister->getWorker()->getOtherLister(parentlister);
    if(olister==NULL) return;
    lm=olister->getActiveMode();
    if(lm!=NULL) {
        if(lm->isType("NormalMode")==true) {
            nm2=(NormalMode*)lm;
            if ( nm2->getCurrentDir() != NULL ) {
                if ( strcmp( nm2->getCurrentDir(), cslorder->destdir ) == 0 ) {
                    updatenm=nm2;
                }
            }
        }
    }

    csllist = new std::list<NM_specialsourceInt*>;
    switch(cslorder->source) {
        case NM_createsymlinkorder::NM_SPECIAL:
            buildSpecialSourceIntList( *( cslorder->sources ),
                                       *csllist );
            break;
        case NM_createsymlinkorder::NM_ONLYACTIVE:
            getSelFiles( csllist, NM_GETFILES_ONLYACTIVE, false );
            break;
        default:  // all selected entries
            getSelFiles( csllist, NM_GETFILES_SELORACT, false );
            break;
    }
  
    cancel=skip=false;
    for ( iti1 = csllist->begin(); iti1 != csllist->end(); iti1++ ) {
        if ( cancel == true ) break;
        ss1 = *iti1;
        fe = ss1->entry();
        if ( ss1->row >= 0 ) {
            lv->setVisMark( ss1->row, true );
            lv->showRow( ss1->row );
        }
        if(fe!=NULL) {
            newname=dupstring(fe->name);
            newfullname=NULL;
            for(nameok=false;nameok==false;) {
                buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                            strlen(catalog.getLocale(225))+1+
                                            strlen(catalog.getLocale(8))+1);
                sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                        catalog.getLocale(225),
                        catalog.getLocale(8));
                textstr=(char*)_allocsafe(strlen(catalog.getLocale(207))+strlen(fe->name)+1);
                sprintf(textstr,catalog.getLocale(207),fe->name);
                erg=string_request(catalog.getLocaleCom(36),textstr,newname,buttonstr,&return_str);
                _freesafe(buttonstr);
                _freesafe(textstr);
                _freesafe(newname);
                newname=return_str;
                if(erg==2) {
                    cancel=true;
                    break;
                } else if(erg==1) {
                    break;
                } else {
                    newfullname=(char*)_allocsafe(strlen(cslorder->destdir)+1+strlen(newname)+1);
                    if(strlen(cslorder->destdir)>1) sprintf(newfullname,"%s/%s",cslorder->destdir,newname);
                    else sprintf(newfullname,"%s%s",cslorder->destdir,newname);
                    if(Datei::lfileExistsExt(newfullname)!=Datei::D_FE_NOFILE) {
                        textstr=(char*)_allocsafe(strlen(catalog.getLocale(279))+strlen(newname)+1);
                        sprintf(textstr,catalog.getLocale(279),newname);
                        buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                                    strlen(catalog.getLocale(225))+1+
                                                    strlen(catalog.getLocale(8))+1);
                        sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(230),
                                catalog.getLocale(225),
                                catalog.getLocale(8));
                        erg=request(catalog.getLocale(123),textstr,buttonstr);
                        _freesafe(buttonstr);
                        _freesafe(textstr);
                        if(erg==1) break;
                        else if(erg==2) {
                            cancel=true;
                            break;
                        }
                        _freesafe(newfullname);
                        newfullname=NULL;
                    } else {
                        nameok=true;
                    }
                }
            }
            if((nameok==true)&&(newfullname!=NULL)) {
                // now create symlink newfullname which points to fe->fullname or fe->name
                if(cslorder->local==true) {
                    sldest = Datei::getRelativePath( fe->fullname, cslorder->destdir );
                    if ( sldest == NULL ) {
                        //TODO: I currently can't imagine a situation but the call
                        //      can return NULL (shouldn't happen because I give
                        //      correct info) so in this case just use the fullname
                        //      But give the user a requester?
                        sldest = dupstring( fe->fullname );
                    }
                } else sldest = dupstring( fe->fullname );
                erg = worker_symlink(sldest,newfullname);
                _freesafe( sldest );
                if(erg!=0) {
                    textstr=(char*)_allocsafe(strlen(catalog.getLocale(205))+strlen(newfullname)+1);
                    sprintf(textstr,catalog.getLocale(205),newfullname);
                    buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                                strlen(catalog.getLocale(8))+1);
                    sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                            catalog.getLocale(8));
                    erg=request(catalog.getLocale(347),textstr,buttonstr);
                    _freesafe(buttonstr);
                    _freesafe(textstr);
                    if(erg==1) cancel=true;
                } else {
                    // deselect
                    if(lv->isValidRow(ss1->row)==true) deselect(fe,ss1->row);
                }
            }
            if(newname!=NULL) _freesafe(newname);
            if(newfullname!=NULL) _freesafe(newfullname);
        }
        if ( ss1->row >= 0 ) {
            lv->setVisMark( ss1->row, false );
        }
    }
  
    for ( iti1 = csllist->begin(); iti1 != csllist->end(); iti1++ ) {
        ss1 = *iti1;
        delete ss1;
    }
    delete csllist;
    // now update
    if(updatenm!=NULL) updatenm->update(false);
}

void NormalMode::changesymlink(struct NM_changesymlinkorder *chslorder)
{
    NM_specialsourceInt *ss1;
    FileEntry *fe;
    int erg;
    bool cancel;
    char *buttonstr,*textstr,*return_str;
    char buffer[4096];
    int tx;
    std::list<NM_specialsourceInt*> *csllist;
    std::list<NM_specialsourceInt*>::iterator iti1;

    if ( getCurrentDir() == NULL ) {
        // no current dir => no files to rename
        return;
    }

    finishsearchmode();
  
    csllist = new std::list<NM_specialsourceInt*>;
    switch(chslorder->source) {
        case NM_changesymlinkorder::NM_SPECIAL:
            buildSpecialSourceIntList( *( chslorder->sources ),
                                       *csllist );
            break;
        case NM_changesymlinkorder::NM_ONLYACTIVE:
            getSelFiles( csllist, NM_GETFILES_ONLYACTIVE, false );
            break;
        default:  // all selected entries
            getSelFiles( csllist, NM_GETFILES_SELORACT, false );
            break;
    }
  
    cancel=false;

    for ( iti1 = csllist->begin(); iti1 != csllist->end(); iti1++ ) {
        if ( cancel == true ) break;
        ss1 = *iti1;
        fe = ss1->entry();
        if ( ss1->row >= 0 ) {
            lv->setVisMark( ss1->row, true );
            lv->showRow( ss1->row );
        }
        if(fe!=NULL) {
            if(fe->isLink==true) {
                tx = worker_readlink( fe->fullname, buffer, 4095 );
                if(tx>=0) {
                    buffer[tx]=0;
                    buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                                strlen(catalog.getLocale(225))+1+
                                                strlen(catalog.getLocale(8))+1);
                    sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                            catalog.getLocale(225),
                            catalog.getLocale(8));
                    textstr=(char*)_allocsafe(strlen(catalog.getLocale(206))+strlen(fe->name)+1);
                    sprintf(textstr,catalog.getLocale(206),fe->name);
                    erg=string_request(catalog.getLocaleCom(37),textstr,buffer,buttonstr,&return_str);
                    _freesafe(buttonstr);
                    _freesafe(textstr);
                    if(erg==2) {
                        // cancel
                        cancel=true;
                    } else if(erg==0) {
                        // ok
                        if(strcmp(buffer,return_str)!=0) {
                            // only if new value differs
                            if ( worker_unlink( fe->fullname ) == 0 ) {
                                if ( worker_symlink( return_str, fe->fullname ) == 0 ) {
                                    // deselect
                                    if(lv->isValidRow(ss1->row)==true) deselect(fe,ss1->row);
                                } else {
                                    textstr=(char*)_allocsafe(strlen(catalog.getLocale(205))+strlen(fe->fullname)+1);
                                    sprintf(textstr,catalog.getLocale(205),fe->fullname);
                                    buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                                                strlen(catalog.getLocale(8))+1);
                                    sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                                            catalog.getLocale(8));
                                    erg=request(catalog.getLocale(347),textstr,buttonstr);
                                    _freesafe(buttonstr);
                                    _freesafe(textstr);
                                    if(erg==1) cancel=true;
                                }
                            } else {
                                textstr=(char*)_allocsafe(strlen(catalog.getLocale(291))+strlen(fe->fullname)+1);
                                sprintf(textstr,catalog.getLocale(291),fe->fullname);
                                buttonstr = dupstring( catalog.getLocale(11) );
                                request(catalog.getLocale(347),textstr,buttonstr);
                                _freesafe(textstr);
                                _freesafe( buttonstr );
                            }
                        }
                    }
                    _freesafe(return_str);
                }
            }
        }
        if ( ss1->row >= 0 ) {
            lv->setVisMark( ss1->row, false );
        }
    }
  
    for ( iti1 = csllist->begin(); iti1 != csllist->end(); iti1++ ) {
        ss1 = *iti1;
        delete ss1;
    }
    delete csllist;

    update( false );
}

int NormalMode::getSelFiles( std::list<NM_specialsourceExt*> *list, nm_getfiles_t selmode, bool unselect )
{
    int res;
    std::list<NM_specialsourceInt*> *tlist;
    std::list<NM_specialsourceInt*>::iterator iti1;
    NM_specialsourceInt *ss1;
    NM_specialsourceExt *sse1;

    if ( list == NULL ) return -1;

    tlist = new std::list<NM_specialsourceInt*>;
    res = getSelFiles( tlist, selmode, unselect );
    if ( res < 0 ) {
        delete tlist;
        return -1;
    }

    for ( iti1 = tlist->begin(); iti1 != tlist->end(); iti1++ ) {
        ss1 = *iti1;
        sse1 = new NM_specialsourceExt( ss1->entry() );
    
        list->push_back( sse1 );

        delete ss1;
    }
    delete tlist;
    return res;
}

void NormalMode::setSortmode(int nmode)
{
    if ( ISVALID_SORTMODE( nmode ) ) {
        nmode&=0xff|SORT_REVERSE|SORT_DIRLAST|SORT_DIRMIXED;
        sortmode=nmode;

        _dir_sort_sets.setSortMode( sortmode );
        // aktualisieren
        rebuildView();
        if ( ce != NULL )
            if ( ce->getActiveFE() != NULL )
                lv->centerActive();
        setName();
    }
}

void NormalMode::setFilters( const std::list<NM_Filter> &filters )
{
    _dir_filter_sets.setFilters( filters );
    // aktualisieren
    rebuildView();
    if ( ce != NULL )
        if ( ce->getActiveFE() != NULL )
            lv->centerActive();
    setName();
}

void NormalMode::configureFilters( bool *tshowhidden, std::list<NM_Filter> &filters,
                                   DirFilterSettings &dir_settings )
{
    int trow;
    int pos;
    const int cincw = AContainer::ACONT_MINH +
        AContainer::ACONT_MINW +
        AContainer::ACONT_MAXH;
    const int cincwnr = cincw +
        AContainer::ACONT_NORESIZE;
    const int cfix = AContainer::ACONT_MINH +
        AContainer::ACONT_MINW +
        AContainer::ACONT_MAXH +
        AContainer::ACONT_MAXW;
    const int cmin = AContainer::ACONT_MINH +
        AContainer::ACONT_MINW;
    std::list<NM_Filter> usetf;
    std::list<NM_Filter>::iterator fil_it1;

    usetf = filters;

    AWindow *win = new AWindow( aguix, 10, 10, 10, 10, 0, catalog.getLocale( 159 ) );
    win->create();

    AContainer *ac1 = win->setContainer( new AContainer( win, 1, 6 ), true );
    ac1->setMinSpace( 5 );
    ac1->setMaxSpace( 5 );

    ChooseButton *chb=(ChooseButton*)ac1->add( new ChooseButton( aguix, 0, 0,
                                                                 (*tshowhidden==true)?1:0,
                                                                 catalog.getLocale(357),LABEL_RIGHT,1,0), 0, 0, cincwnr );

    FieldListView *filv = (FieldListView*)ac1->add( new FieldListView( aguix,
                                                                       0,
                                                                       0,
                                                                       100,
                                                                       10 * aguix->getCharHeight(),
                                                                       0 ), 0, 1, cmin );
    filv->setHBarState(2);
    filv->setVBarState(2);
    filv->setNrOfFields( 3 );
    filv->setFieldWidth( 1, 1 );

    AContainer *ac1_1 = ac1->add( new AContainer( win, 3, 1 ), 0, 2 );
    ac1_1->setMinSpace( 0 );
    ac1_1->setMaxSpace( 0 );
    ac1_1->setBorderWidth( 0 );

    Button *newb = (Button*)ac1_1->add( new Button( aguix, 0, 0,
                                                    catalog.getLocale( 167 ), 1, 0, 0 ), 0, 0, cincw );
    Button *delb = (Button*)ac1_1->add( new Button( aguix, 0, 0,
                                                    catalog.getLocale( 169 ), 1, 0, 0 ), 1, 0, cincw );
    Button *editb = (Button*)ac1_1->add( new Button( aguix, 0, 0,
                                                     catalog.getLocale( 168 ), 1, 0, 0 ), 2, 0, cincw );

    Button *uallb = (Button*)ac1->add( new Button( aguix, 0, 0,
                                                   catalog.getLocale( 194 ), 1, 0, 0 ), 0, 3, cincw );

    AContainerBB *ac1_3 = (AContainerBB*)ac1->add( new AContainerBB( win, 1, 4 ), 0, 4 );
    ac1_3->setMinSpace( 5 );
    ac1_3->setMaxSpace( -1 );
    ac1_3->setBorderWidth( 5 );

    ac1_3->add( new Text( aguix, 0, 0, catalog.getLocale( 845 ), 1 ), 0, 0, AContainer::CO_INCWNR );

    AContainer *ac1_3_1 = ac1_3->add( new AContainer( win, 2, 1 ), 0, 1 );
    ac1_3_1->setMinSpace( 5 );
    ac1_3_1->setMaxSpace( 5 );
    ac1_3_1->setBorderWidth( 0 );

    ac1_3_1->add( new Text( aguix, 0, 0, catalog.getLocale( 840 ), 1 ), 0, 0, AContainer::CO_FIX );
    CycleButton *bookmark_filter_cycb = (CycleButton*)ac1_3_1->add( new CycleButton( aguix, 0, 0, 100, 1, 0, 0 ), 1, 0, AContainer::CO_INCW );
    bookmark_filter_cycb->addOption( catalog.getLocale( 841 ) );
    bookmark_filter_cycb->addOption( catalog.getLocale( 842 ) );
    bookmark_filter_cycb->addOption( catalog.getLocale( 843 ) );
    switch ( dir_settings.getBookmarkFilter() ) {
        case DirFilterSettings::SHOW_ONLY_BOOKMARKS:
            bookmark_filter_cycb->setOption( 1 );
            break;
        case DirFilterSettings::SHOW_ONLY_LABEL:
            bookmark_filter_cycb->setOption( 2 );
            break;
        case DirFilterSettings::SHOW_ALL:
        default:
            bookmark_filter_cycb->setOption( 0 );
            break;
    }
    bookmark_filter_cycb->resize( bookmark_filter_cycb->getMaxSize(),
                                  bookmark_filter_cycb->getHeight() );
    ac1_3_1->readLimits();

    AContainer *ac1_3_2 = ac1_3->add( new AContainer( win, 3, 2 ), 0, 2 );
    ac1_3_2->setMinSpace( 0 );
    ac1_3_2->setMaxSpace( 0 );
    ac1_3_2->setBorderWidth( 0 );

    ac1_3_2->setMinWidth( 5, 1, 0 );
    ac1_3_2->setMaxWidth( 5, 1, 0 );

    ac1_3_2->add( new Text( aguix, 0, 0, catalog.getLocale( 844 ), 1 ), 0, 1, AContainer::CO_FIX );

    FieldListView *label_lv = (FieldListView*)ac1_3_2->add( new FieldListView( aguix,
                                                                               0, 0,
                                                                               40, 40, 0 ),
                                                            2, 0, AContainer::CO_MIN );
    label_lv->setHBarState( 2 );
    label_lv->setVBarState( 2 );

    std::list<std::string> cats = Worker::getBookmarkDBInstance().getCats();
    const std::map<std::string, WConfig::ColorDef::label_colors_t> labels = wconfig->getColorDefs().getLabelColors();
    std::map<std::string, WConfig::ColorDef::label_colors_t>::const_iterator label_it;
  
    for ( label_it = labels.begin(); label_it != labels.end(); ++label_it ) {
        if ( std::find( cats.begin(), cats.end(),
                        label_it->first ) == cats.end() ) {
            cats.push_back( label_it->first );
        }
    }
    if ( dir_settings.getSpecificBookmarkLabel().length() > 0 &&
         std::find( cats.begin(), cats.end(),
                    dir_settings.getSpecificBookmarkLabel() ) == cats.end() ) {
        cats.push_back( dir_settings.getSpecificBookmarkLabel() );
    }
    for ( std::list<std::string>::iterator it1 = cats.begin();
          it1 != cats.end();
          ++it1 ) {
        int label_row = label_lv->addRow();
        label_lv->setText( label_row, 0, *it1 );
        label_lv->setPreColors( label_row, FieldListView::PRECOLOR_ONLYACTIVE );
    }
    label_lv->resize( label_lv->getWidth(), 6 * aguix->getCharHeight() );
    ac1_3_2->readLimits();

    StringGadget *bookmark_label_sg = (StringGadget*)ac1_3_2->add( new StringGadget( aguix, 0, 0, 100,
                                                                                     dir_settings.getSpecificBookmarkLabel().c_str(), 0 ), 2, 1, AContainer::CO_INCW );

    ChooseButton *highlight_prefix_cb = (ChooseButton*)ac1_3->add( new ChooseButton( aguix, 0, 0,
                                                                                     dir_settings.getHighlightBookmarkPrefix(),
                                                                                     catalog.getLocale( 846 ), LABEL_LEFT,
                                                                                     1, 0 ), 0, 3, AContainer::CO_INCWNR );

    AContainer *ac1_2 = ac1->add( new AContainer( win, 2, 1 ), 0, 5 );
    ac1_2->setMinSpace( 5 );
    ac1_2->setMaxSpace( -1 );
    ac1_2->setBorderWidth( 0 );
    Button *okb =(Button*)ac1_2->add( new Button( aguix,
                                                  0,
                                                  0,
                                                  catalog.getLocale( 11 ),
                                                  1,
                                                  0,
                                                  0 ), 0, 0, cfix );
    Button *cancelb = (Button*)ac1_2->add( new Button( aguix,
                                                       0,
                                                       0,
                                                       catalog.getLocale( 8 ),
                                                       1,
                                                       0,
                                                       0 ), 1, 0, cfix );

    pos = 0;
    for ( fil_it1 = usetf.begin();
          fil_it1 != usetf.end();
          fil_it1++ ) {
        trow = filv->addRow();
        setLVC4Filter( filv, trow, *fil_it1 );
        filv->setPreColors( trow, FieldListView::PRECOLOR_ONLYSELECT );
    }
    filv->redraw();

    win->setDoTabCycling( true );
    win->contMaximize( true );
    win->show();
  
    AGMessage *msg;
    while((msg=aguix->GetMessage(NULL))!=NULL) aguix->ReplyMessage(msg);
    int ende=0;
    while(ende==0) {
        msg=aguix->WaitMessage(win);
        if(msg!=NULL) {
            if(msg->type==AG_CLOSEWINDOW) {
                if(msg->closewindow.window==win->getWindow()) ende=-1;

            } else if ( msg->type == AG_STRINGGADGET_CONTENTCHANGE ) {
                if ( msg->stringgadget.sg == bookmark_label_sg ) {
                    bookmark_filter_cycb->setOption( 2 );
                }
            } else if ( msg->type == AG_FIELDLV_ONESELECT ||
                        msg->type == AG_FIELDLV_MULTISELECT ) {
                if ( msg->fieldlv.lv == label_lv ) {
                    int label_row = label_lv->getActiveRow();
                    if ( label_lv->isValidRow( label_row ) == true ) {
                        bookmark_label_sg->setText( label_lv->getText( label_row, 0 ).c_str() );
                    }
                    bookmark_filter_cycb->setOption( 2 );
                }
            } else if(msg->type==AG_BUTTONCLICKED) {
                if(msg->button.button==okb) ende=1;
                else if(msg->button.button==cancelb) ende=-1;
                else if(msg->button.button==newb) {
                    NM_Filter tfi;
          
                    if ( configureFilter( tfi ) == 0 ) {
                        trow = filv->addRow();
                        setLVC4Filter( filv, trow, tfi );
                        filv->setPreColors( trow, FieldListView::PRECOLOR_ONLYSELECT );
                        filv->setActiveRow( trow );
                        filv->showActive();
                        usetf.push_back( tfi );
                        filv->redraw();
                    }
                } else if(msg->button.button==delb) {
                    pos = 0;
                    while ( filv->isValidRow( pos ) == true ) {
                        if ( filv->getSelect( pos ) == true ) {
                            int poscopy = pos;
                            fil_it1 = usetf.begin();
                            while ( fil_it1 != usetf.end() && poscopy > 0 ) {
                                fil_it1++;
                                poscopy--;
                            }
                
                            if ( fil_it1 != usetf.end() ) {
                                usetf.erase( fil_it1 );
                                filv->deleteRow( pos );
                                filv->redraw();
                                pos--;
                            }
                        }
                        pos++;
                    }
                } else if(msg->button.button==editb) {
                    pos = 0;
                    while ( filv->isValidRow( pos ) == true ) {
                        if ( filv->getSelect( pos ) == true ) {
                            int poscopy = pos;
                            fil_it1 = usetf.begin();
                            while ( fil_it1 != usetf.end() && poscopy > 0 ) {
                                fil_it1++;
                                poscopy--;
                            }

                            if ( fil_it1 != usetf.end() ) {
                                if ( configureFilter( *fil_it1 ) == 0 ) {
                                    setLVC4Filter( filv, pos, *fil_it1 );
                                    filv->redraw();
                                }
                            }
                        }
                        pos++;
                    }
                } else if(msg->button.button==uallb) {
                    pos = 0;
                    for ( fil_it1 = usetf.begin();
                          fil_it1 != usetf.end();
                          fil_it1++ ) {
                        fil_it1->setCheck( NM_Filter::INACTIVE );
                        if ( filv->isValidRow( pos ) == true ) {
                            setLVC4Filter( filv, pos, *fil_it1 );
                            pos++;
                        }
                    }
                    filv->redraw();
                }
            }
            aguix->ReplyMessage(msg);
        }
    }
    if(ende==1) {
        // insert usetf in tfilters
        filters = usetf;
        *tshowhidden = chb->getState();

        dir_settings.setHighlightBookmarkPrefix( highlight_prefix_cb->getState() );
        dir_settings.setSpecificBookmarkLabel( bookmark_label_sg->getText() );

        switch ( bookmark_filter_cycb->getSelectedOption() ) {
            case 1:
                dir_settings.setBookmarkFilter( DirFilterSettings::SHOW_ONLY_BOOKMARKS );
                break;
            case 2:
                dir_settings.setBookmarkFilter( DirFilterSettings::SHOW_ONLY_LABEL );
                break;
            case 0:
            default:
                dir_settings.setBookmarkFilter( DirFilterSettings::SHOW_ALL );
                break;
        }
    }
  
    delete win;
}

void NormalMode::setLVC4Filter( FieldListView *filv, int row, NM_Filter &fi )
{
    if ( filv == NULL ) return;
    const char *p = fi.getPattern();
  
    filv->setText( row, 0, ( p != NULL ) ? p : "" );
    if ( fi.getCheck() == 1 ) filv->setText( row, 2, catalog.getLocale( 170 ) );
    else if ( fi.getCheck() == 2 ) filv->setText( row, 2, catalog.getLocale( 171 ) );
    else filv->setText( row, 2, "" );
}

int NormalMode::configureFilter( NM_Filter &fi )
{
    const char *tstr;
    const int cincw = AContainer::ACONT_MINH +
        AContainer::ACONT_MINW +
        AContainer::ACONT_MAXH;
    const int cfix = AContainer::ACONT_MINH +
        AContainer::ACONT_MINW +
        AContainer::ACONT_MAXH +
        AContainer::ACONT_MAXW;

    AWindow *win = new AWindow( aguix, 10, 10, 10, 10, 0, catalog.getLocale( 172 ) );
    win->create();

    AContainer *ac1 = win->setContainer( new AContainer( win, 1, 3 ), true );
    ac1->setMinSpace( 5 );
    ac1->setMaxSpace( 5 );

    AContainer *ac1_1 = ac1->add( new AContainer( win, 2, 1 ), 0, 0 );
    ac1_1->setMinSpace( 5 );
    ac1_1->setMaxSpace( 5 );
    ac1_1->setBorderWidth( 0 );
    ac1_1->add( new Text( aguix, 0, 0, catalog.getLocale( 93 ), 1 ), 0, 0, cfix );

    tstr = fi.getPattern();
    StringGadget *tsg = (StringGadget*)ac1_1->add( new StringGadget( aguix, 0, 0, 100,
                                                                     ( tstr != NULL ) ? tstr : "", 0 ), 1, 0, cincw );

    CycleButton *cyb = (CycleButton*)ac1->add( new CycleButton( aguix, 0, 0, 100, 1, 0, 0 ), 0, 1, cincw );
    cyb->addOption(catalog.getLocale(358));
    cyb->addOption(catalog.getLocale(170));
    cyb->addOption(catalog.getLocale(171));
    if ( fi.getCheck() == 1 ) cyb->setOption(1);
    else if ( fi.getCheck() == 2 ) cyb->setOption(2);
    else cyb->setOption(0);
    cyb->resize(cyb->getMaxSize(),cyb->getHeight());
    ac1->readLimits();

    AContainer *ac1_2 = ac1->add( new AContainer( win, 2, 1 ), 0, 2 );
    ac1_2->setMinSpace( 5 );
    ac1_2->setMaxSpace( -1 );
    ac1_2->setBorderWidth( 0 );
    Button *okb =(Button*)ac1_2->add( new Button( aguix,
                                                  0,
                                                  0,
                                                  catalog.getLocale( 11 ),
                                                  1,
                                                  0,
                                                  0 ), 0, 0, cfix );
    Button *cancelb = (Button*)ac1_2->add( new Button( aguix,
                                                       0,
                                                       0,
                                                       catalog.getLocale( 8 ),
                                                       1,
                                                       0,
                                                       0 ), 1, 0, cfix );
  
    win->setDoTabCycling( true );
    win->contMaximize( true );
    win->show();
    AGMessage *msg;
    while((msg=aguix->GetMessage(NULL))!=NULL) aguix->ReplyMessage(msg);
    int ende=0;
    while(ende==0) {
        msg=aguix->WaitMessage(win);
        if(msg!=NULL) {
            if(msg->type==AG_CLOSEWINDOW) {
                if(msg->closewindow.window==win->getWindow()) ende=-1;
            } else if(msg->type==AG_BUTTONCLICKED) {
                if(msg->button.button==okb) ende=1;
                else if(msg->button.button==cancelb) ende=-1;
            }
            aguix->ReplyMessage(msg);
        }
    }
    if(ende==1) {
        switch(cyb->getSelectedOption()) {
            case 1:
                fi.setCheck( NM_Filter::INCLUDE );
                break;
            case 2:
                fi.setCheck( NM_Filter::EXCLUDE );
                break;
            default:
                fi.setCheck( NM_Filter::INACTIVE );
                break;
        }
        fi.setPattern( tsg->getText() );
    }
  
    delete win;
    return (ende==1)?0:1;
}

/*
 * NormalMode::rebuildView
 *
 * rebuild listview and update statebar infos
 */
void NormalMode::rebuildView()
{
    if ( lv != NULL ) {
        int oldx = lv->getXOffset();
        int oldy = lv->getYOffset();
        buildListView();
        lv->setYOffset( oldy );
        lv->setXOffset( oldx );
        showCacheState();
        setName();
        updateTabs();
    }
}

void NormalMode::setName()
{
    bool filteractive;
    std::string tstr;
    bool doUpdate;
  
    filteractive = _dir_filter_sets.filterActive();

    if ( filteredSearchActive() == true ) {
        tstr = "(";

        if ( m_flexible_matching_active ) {
            tstr += "~";

            std::string s = getEntrySearchString();

            if ( s != m_cached_values.previous_match_string ) {
                m_cached_values.cleaned_match_string = cleanFlexibleMatchString( s );
            }

            tstr += m_cached_values.cleaned_match_string;
        } else {
            tstr += getEntrySearchString();
        }
        tstr += ") ";
    }
  
    switch ( busyFlag ) {
        case 1:
            tstr += '|';
            break;
        case 2:
            tstr += '/';
            break;
        case 3:
            tstr += '-';
            break;
        case 4:
            tstr += '\\';
            break;
        default:
            tstr += ' ';
            break;
    }

    tstr += ' ';

    switch(sortmode&0xff) {
        case SORT_NAME:
            tstr += 'N';
            break;
        case SORT_SIZE:
            tstr += 'S';
            break;
        case SORT_CHGTIME:
            tstr += 'C';
            break;
        case SORT_MODTIME:
            tstr += 'M';
            break;
        case SORT_ACCTIME:
            tstr += 'A';
            break;
        case SORT_TYPE:
            tstr += 'T';
            break;
        case SORT_OWNER:
            tstr += 'O';
            break;
        case SORT_INODE:
            tstr += 'I';
            break;
        case SORT_NLINK:
            tstr += 'L';
            break;
        case SORT_PERMISSION:
            tstr += 'P';
            break;
        default:
            tstr += '?';
            break;
    }

    if ( ( sortmode & SORT_REVERSE ) == SORT_REVERSE ) tstr += 'R';
    else tstr += ' ';

    if ( _dir_filter_sets.getShowHidden() == false ) tstr += 'H';
    else tstr += ' ';

    /*  if ( filteredSearchActive() == true ) {
        tstr += '+';
        } else */
    if ( filteractive == true ) {
        tstr += '*';
    } else {
        tstr += ' ';
    }

    if ( _dir_filter_sets.getBookmarkFilter() != DirFilterSettings::SHOW_ALL ) {
        tstr += 'B';
    } else {
        tstr += ' ';
    }

    tstr += "  ";
    tstr += catalog.getLocale( 173 );
  
    if ( tstr != namestr ) doUpdate = true;
    else doUpdate = false;

    namestr = tstr;

    if ( doUpdate == true ) updateName();
}

/*
 * update lvb name with current name and freespace str
 */
void NormalMode::updateName()
{
    std::string tstr;
  
    tstr = namestr;
    tstr += freespacestr;
    //TODO: Vielleicht mit aktuellen String vergleichen, um redraw
    //      zu vermeiden?
    parentlister->setName( tstr.c_str() );
}

void NormalMode::reconfig()
{
    int side = parentlister->getSide();
    AGUIXFont *afont = aguix->getFont( wconfig->getFont( 2 + side ).c_str() );
    int sgh = ( ( afont != NULL ) ? afont->getCharHeight() : aguix->getCharHeight() ) +8;
    int hbw;

    writeFieldWidthsToLoadMap();

    lv->setMBG(wconfig->getListerBG());
    lv->setHBarState( ( wconfig->getHBarTop( side ) == true ) ? 1 : 2 );
    lv->setVBarState( ( wconfig->getVBarLeft( side ) == true ) ? 1 : 2 );
    lv->setHBarHeight( wconfig->getHBarHeight( side ) );
    lv->setVBarWidth( wconfig->getVBarWidth( side ) );
    lv->setShowHeader( wconfig->getShowHeader( side ) );
    lv->setFont( wconfig->getFont( 2 + side ).c_str() );
    sg->setFont( wconfig->getFont( 2 + side ).c_str() );
    hb[0]->setFont( wconfig->getFont( 2 + side ).c_str() );
    hb[1]->setFont( wconfig->getFont( 2 + side ).c_str() );
    parentb->setFont( wconfig->getFont( 2 + side ).c_str() );
    lv->setHeaderFG( wconfig->getLVHeader( 0 ) );
    lv->setHeaderBG( wconfig->getLVHeader( 1 ) );
  
    sg->resize( sg->getWidth(), sgh );
    hbw = aguix->getTextWidth( "<", afont ) + 10;
    hb[0]->resize( hbw, sgh );
    hb[1]->resize( hbw, sgh );
    parentb->resize( aguix->getTextWidth( "..", afont ) + 10, sgh );

    m_tab_b->setFont( wconfig->getFont( 2 + side ).c_str() );
    m_tab_new->setFont( wconfig->getFont( 2 + side ).c_str() );
    m_tab_new->resize( aguix->getTextWidth( "N", afont ) + 10, m_tab_new->getHeight() );
    m_tab_close->setFont( wconfig->getFont( 2 + side ).c_str() );
    m_tab_close->resize( aguix->getTextWidth( "X", afont ) + 10, m_tab_new->getHeight() );
    m_tab_cont->readLimits();

    m_cont->readLimits();
    m_cont2->readLimits();

    parentawindow->updateCont();
    lv->centerActive();
}

void NormalMode::showFreeSpace(bool force)
{
    int erg;

    if ( showfreespace == true ) {
        updateSpaceTimeout( updatetime * 1000 );
    } else {
        updateSpaceTimeout( 0 );
    }

    if ( force == true ) erg = updateFreeSpaceStr( 0 );
    else erg = updateFreeSpaceStr( updatetime );
    if ( erg != 0 ) updateName();
}

/*
 * method updates freespacestr
 * returns 0 for no change, 1 for change
 */
int NormalMode::updateFreeSpaceStr( int ti )
{
    char *tstr;
    int len;
    time_t now;
    int erg, rv = 0;

    if(showfreespace==true) {
        now=time(NULL);
        if ( ( now - lastfsupdate >= ti ) || ( lasteagain == true ) ) {
            if ( getCurrentDir() != NULL ) {
                erg = parentlister->getWorker()->PS_readSpace( getCurrentDir() );
                if ( erg == 0 ) {
                    std::string spaceh, freeh;

                    spaceh = parentlister->getWorker()->PS_getSpaceH();
                    freeh = parentlister->getWorker()->PS_getFreeSpaceH();
            
                    len = freeh.length() +
                        spaceh.length() +
                        strlen( catalog.getLocale( 849 ) ) + 1;

                    tstr = (char*)_allocsafe( len );
                    sprintf( tstr, catalog.getLocale( 849 ), freeh.c_str(), spaceh.c_str() );
                    freespacestr = tstr;
                    _freesafe( tstr );
                    lasteagain = false;
                    rv = 1;
                } else if ( erg == EAGAIN ) {
                    // no valid value so lets try again next call
                    lasteagain = true;
                } else {
                    // clear string since data is not available
                    freespacestr = "";
                    rv = 1;
                    lasteagain = false;
                }
                lastfsupdate=now;
            }
        }
    } else {
        if ( ! freespacestr.empty() ) {
            freespacestr = "";
            rv = 1;
        }
    }
    return rv;
}

void NormalMode::setShowFreeSpace(bool v)
{
    showfreespace=v;
    setName();
    showFreeSpace(true);
}

void NormalMode::setUpdatetime(int nv)
{
    updatetime=nv;
    if(updatetime<1) updatetime=1;
    parentlister->getWorker()->PS_setLifetime( (double)updatetime );
}

bool NormalMode::startdnd(DNDMsg *dm)
{
    bool returnvalue=false;
    if(dm->getStart()->element==lv) {
        returnvalue=true;

        int row = dm->getStart()->specialinfo.value;
        if(lv->isValidRow(row)==true) {
            if(ce!=NULL) {
                if ( ce->dirOpened() == true ) {
                    FileEntry *fe = getFEForRow( row );
                    if(fe!=NULL) {
                        // found entry for dnd
                        // now use the type to call the dnd action

                        WCFiletype *ft;
                        ActionMessage amsg( parentlister->getWorker() );
                        amsg.startLister=parentlister;
                        amsg.mode=amsg.AM_MODE_DNDACTION;

                        if(fe->isDir()==false) {
                            ft=fe->filetype;
                            if ( ft == NULL ) ft = wconfig->getnotyettype(); // not yet checked
                        } else {
                            ft = wconfig->getdirtype();
                        }
                        if ( ft == NULL ) ft = wconfig->getvoidtype();
                        if(ft!=NULL) {
                            if ( ft->getDNDActions()->size() < 1 ) ft = wconfig->getvoidtype();
                            if(ft!=NULL) {
                                amsg.dndmsg = dm;
                                amsg.filetype = ft;
                                amsg.m_action_descr = RefCount<ActionDescr>( new DNDAction::DNDActionDescr() );
                                parentlister->getWorker()->interpret(ft->getDNDActions(),&amsg);
                            }
                        }
                    }
                }
            }
        }
    }
    return returnvalue;
}

bool NormalMode::isyours( Widget *elem )
{
    if(elem==lv) return true;
    return false;
}

FileEntry *NormalMode::getFE4DNDSTART( const AGDNDSTART *st )
{
    int row;
    FileEntry *fe=NULL;
    NMRowData *rdp;

    if ( st == NULL ) return NULL;
    if ( st->specialinfo.rowDataP == NULL ) return NULL;
    if( ( lv != NULL ) && ( st->element == lv ) ) {
        // lookup strategy:
        // 1.check if row (value) contains FE
        // 2.if not, find FE
        // 3.check fullname with found FE
        //TODO:search for fullname
        //     this is no bug but would improve it
        //     this would also only help users which
        //     try to trigger this
        row = st->specialinfo.value;
        if(lv->isValidRow(row)==true) {
            if(ce!=NULL) {
                if ( ce->dirOpened() == true ) {
                    fe = getFEForRow( row );
                    //TODO:for more secure cast
                    //if ( typeid( *(st->specialinfo.rowDataP) ) == typeid( NMRowData ) );
                    rdp = (NMRowData*)st->specialinfo.rowDataP;
                    if ( fe->equals( rdp->getFE() ) == false ) {
                        fe = NULL;
                        for ( Verzeichnis::verz_it fe_it1 = ce->begin();
                              fe_it1 != ce->end();
                              fe_it1++ ) {
                            fe = *fe_it1;
                            if ( fe->equals( rdp->getFE() ) == true ) break;
                            fe = NULL;
                        }
                    }
                    if ( fe != NULL ) {
                        // The idea for this test was that the DND msg contains a pointer to an old
                        // FileEntry which could be overwritten by another instance so I check the fullname
                        // As I now store copies of the FE inside the NMRowData and use equals() to find
                        // the corresponding FE this test is obsolete but doesn't hurt
                        if ( strcmp( fe->fullname, rdp->getFullname() ) != 0 ) fe = NULL;
                    }
                }
            }
        }
    }
    return fe;
}

void NormalMode::unsetAllFilters()
{
    _dir_filter_sets.unsetAllFilters();
    // aktualisieren
    rebuildView();
    if ( ce != NULL )
        if ( ce->getActiveFE() != NULL )
            lv->centerActive();
    setName();
}

void NormalMode::setFilter(const char *filter,nm_filter_t mode)
{
    NM_Filter::check_t newmode;

    switch ( mode ) {
        case NM_FILTER_EXCLUDE:
            newmode = NM_Filter::EXCLUDE;
            break;
        case NM_FILTER_UNSET:
            newmode = NM_Filter::INACTIVE;
            break;
        default:
            newmode = NM_Filter::INCLUDE;
            break;
    }
    _dir_filter_sets.setFilter( filter, newmode );
    
    // aktualisieren
    rebuildView();
    if ( ce != NULL )
        if ( ce->getActiveFE() != NULL )
            lv->centerActive();
    setName();
}

bool NormalMode::ft_rec_list::isFull_locked()
{
    if ( elements >= size ) return true;
    return false;
}

bool NormalMode::ft_rec_list::isEmpty_locked()
{
    if ( elements < 1 ) return true;
    return false;
}

int NormalMode::ft_rec_list::put_locked( ft_rec_list_t *elem )
{
    int pos;

    if ( elem == NULL ) return -1;
    if ( isFull_locked() == true ) return -1;

    // don't accept any pointer in next
    elem->next = NULL;
    if ( tail != NULL ) {
        // add behind last
        tail->next = elem;
        tail = elem;
    } else {
        head = tail = elem;
    }

    if ( elem->fe != NULL )
        elem->fe->reclistQueued = true;

    pos = elements++;
    return pos;
}

NormalMode::ft_rec_list::ft_rec_list_t *NormalMode::ft_rec_list::remove_locked()
{
    ft_rec_list_t *te;

    if ( elements == 0 ) return NULL;
  
    te = head;
    head = head->next;
    if ( head == NULL )
        tail = NULL;
    elements--;

    if ( te->fe != NULL )
        te->fe->reclistQueued = false;

    // leave no pointer in out list
    te->next = NULL;  
    return te;
}

NormalMode::ft_rec_list::ft_rec_list()
{
    size = 10;
    head = NULL;
    tail = NULL;
    elements = 0;
}

NormalMode::ft_rec_list::~ft_rec_list()
{
    ft_rec_list_t *te;

    while ( isEmpty_locked() == false ) {
        te = remove_locked();
        delete te;
    }
}

NormalMode::ft_rec_list::ft_rec_list_t *NormalMode::ft_rec_list::gettop_locked()
{
    return head;
}

void NormalMode::ft_rec_list_clear()
{
    ft_rec_list::ft_rec_list_t *te;
    NMCacheEntry *tce;
    int id;

    while ( reclist->isEmpty_locked() == false ) {
        te = reclist->remove_locked();
        delete te;
    }
    // reset checkfe for all caches
    id = cache->initEnum();
    tce = (NMCacheEntry*)cache->getFirstElement( id );
    while ( tce != NULL ) {
        tce->resetCheckFE2FirstNULLFT();
        tce = (NMCacheEntry*)cache->getNextElement( id );
    }
    cache->closeEnum( id );
  
    visChanged = true;

    m_ft_thread.clearRequests();
}

void NormalMode::ft_recres_list_clear()
{
    m_ft_thread.res_list_clear();
}

/*
 * ft_list_clear
 *
 * will delete all filetypes from list
 *
 * only allowed for master thread
 */
void NormalMode::ft_list_clear()
{
    m_ft_thread.clear_filetypes();
}

/*
 * ft_list_update
 *
 * will update the filetype-list to the current config
 *
 * only allowed for master thread
 */
void NormalMode::ft_list_update()
{
    List *ftlist;

    ftlist = wconfig->getFiletypes();
    m_ft_thread.copy_filetypes( ftlist );
}

void NormalMode::setupLVFields()
{
    int side = parentlister->getSide();
    const std::vector<WorkerTypes::listcol_t> *sets;
    int i;
    int used_fields, li;
    AGUIXFont *afont;
    int space_width;
    int cur_field;
    int tw;
  
    sets = wconfig->getVisCols( side );
    if ( sets == NULL ) return;
  
    used_fields = sets->size();

    afont = aguix->getFont( wconfig->getFont( 2 + side ).c_str() );
    space_width = aguix->getTextWidth( " ", afont );

    if ( used_fields == 0 ) {
        // oops, what's this?
        lv->setNrOfFields( 1 );  // 0 fields are not supported, blame the author (me ;-) )
    } else {
        lv->setNrOfFields( used_fields * 2 - 1 );
  
        cur_field = 0;
        for( i = 0; i < (int)sets->size(); i++ ) {
            if ( i > 0 ) {
                lv->setFieldWidthQ( cur_field++, space_width );
            }
            lv->setFieldTextMergedQ( cur_field, true );
            lv->setFieldTextResizeable( cur_field, true );
            li = WorkerTypes::getAvailListColEntry( (*sets)[i] );
            if ( li >= 0 ) {
                lv->setFieldTextQ( cur_field, catalog.getLocale( WorkerTypes::availListCols[li].catalogid ) );
            }
            switch ( (*sets)[i] ) {
                case WorkerTypes::LISTCOL_NAME:
                    lv->setFieldAlignQ( cur_field, FieldListView::ALIGN_LEFT );

                    tw = getAndEraseLoadedFieldWidth( "LISTCOL_NAME" );
                    if ( tw < -1 || tw > 10000 ) tw = -1;

                    lv->setFieldWidthQ( cur_field, tw );
	  
                    _field_map["LISTCOL_NAME"] = cur_field;
                    break;
                case WorkerTypes::LISTCOL_SIZE:
#ifdef LEFTJUSTIFY
                    lv->setFieldAlignQ( cur_field, FieldListView::ALIGN_LEFT );
#else
                    lv->setFieldAlignQ( cur_field, FieldListView::ALIGN_RIGHT );
#endif
                    tw = getAndEraseLoadedFieldWidth( "LISTCOL_SIZE" );
                    if ( tw < -1 || tw > 10000 ) tw = -1;

                    lv->setFieldWidthQ( cur_field, tw );

                    _field_map["LISTCOL_SIZE"] = cur_field;
                    break;
                case WorkerTypes::LISTCOL_SIZEH:
#ifdef LEFTJUSTIFY
                    lv->setFieldAlignQ( cur_field, FieldListView::ALIGN_LEFT );
#else
                    lv->setFieldAlignQ( cur_field, FieldListView::ALIGN_RIGHT );
#endif
                    tw = getAndEraseLoadedFieldWidth( "LISTCOL_SIZEH" );
                    if ( tw < -1 || tw > 10000 ) tw = -1;

                    lv->setFieldWidthQ( cur_field, tw );

                    _field_map["LISTCOL_SIZEH"] = cur_field;
                    break;
                case WorkerTypes::LISTCOL_TYPE:
                    lv->setFieldAlignQ( cur_field, FieldListView::ALIGN_LEFT );

                    tw = getAndEraseLoadedFieldWidth( "LISTCOL_TYPE" );
                    if ( tw < -1 || tw > 10000 ) tw = -1;
                    lv->setFieldWidthQ( cur_field, tw );

                    _field_map["LISTCOL_TYPE"] = cur_field;
                    break;
                case WorkerTypes::LISTCOL_PERM:
                    lv->setFieldAlignQ( cur_field, FieldListView::ALIGN_LEFT );

                    tw = getAndEraseLoadedFieldWidth( "LISTCOL_PERM" );
                    if ( tw < -1 || tw > 10000 ) tw = -1;
                    lv->setFieldWidthQ( cur_field, tw );

                    _field_map["LISTCOL_PERM"] = cur_field;
                    break;
                case WorkerTypes::LISTCOL_OWNER:
                    lv->setFieldAlignQ( cur_field, FieldListView::ALIGN_LEFT );

                    tw = getAndEraseLoadedFieldWidth( "LISTCOL_OWNER" );
                    if ( tw < -1 || tw > 10000 ) tw = -1;
                    lv->setFieldWidthQ( cur_field, tw );

                    _field_map["LISTCOL_OWNER"] = cur_field;
                    break;
                case WorkerTypes::LISTCOL_DEST:
                    lv->setFieldAlignQ( cur_field, FieldListView::ALIGN_LEFT );

                    tw = getAndEraseLoadedFieldWidth( "LISTCOL_DEST" );
                    if ( tw < -1 || tw > 10000 ) tw = -1;
                    lv->setFieldWidthQ( cur_field, tw );

                    _field_map["LISTCOL_DEST"] = cur_field;
                    break;
                case WorkerTypes::LISTCOL_MOD:
                    lv->setFieldAlignQ( cur_field, FieldListView::ALIGN_LEFT );

                    tw = getAndEraseLoadedFieldWidth( "LISTCOL_MOD" );
                    if ( tw < -1 || tw > 10000 ) tw = -1;
                    lv->setFieldWidthQ( cur_field, tw );

                    _field_map["LISTCOL_MOD"] = cur_field;
                    break;
                case WorkerTypes::LISTCOL_ACC:
                    lv->setFieldAlignQ( cur_field, FieldListView::ALIGN_LEFT );

                    tw = getAndEraseLoadedFieldWidth( "LISTCOL_ACC" );
                    if ( tw < -1 || tw > 10000 ) tw = -1;
                    lv->setFieldWidthQ( cur_field, tw );

                    _field_map["LISTCOL_ACC"] = cur_field;
                    break;
                case WorkerTypes::LISTCOL_CHANGE:
                    lv->setFieldAlignQ( cur_field, FieldListView::ALIGN_LEFT );

                    tw = getAndEraseLoadedFieldWidth( "LISTCOL_CHANGE" );
                    if ( tw < -1 || tw > 10000 ) tw = -1;
                    lv->setFieldWidthQ( cur_field, tw );

                    _field_map["LISTCOL_CHANGE"] = cur_field;
                    break;
                case WorkerTypes::LISTCOL_INODE:
                    lv->setFieldAlignQ( cur_field, FieldListView::ALIGN_RIGHT );

                    tw = getAndEraseLoadedFieldWidth( "LISTCOL_INODE" );
                    if ( tw < -1 || tw > 10000 ) tw = -1;
                    lv->setFieldWidthQ( cur_field, tw );

                    _field_map["LISTCOL_INODE"] = cur_field;
                    break;
                case WorkerTypes::LISTCOL_NLINK:
                    lv->setFieldAlignQ( cur_field, FieldListView::ALIGN_RIGHT );

                    tw = getAndEraseLoadedFieldWidth( "LISTCOL_NLINK" );
                    if ( tw < -1 || tw > 10000 ) tw = -1;
                    lv->setFieldWidthQ( cur_field, tw );

                    _field_map["LISTCOL_NLINK"] = cur_field;
                    break;
                case WorkerTypes::LISTCOL_BLOCKS:
                    lv->setFieldAlignQ( cur_field, FieldListView::ALIGN_RIGHT );

                    tw = getAndEraseLoadedFieldWidth( "LISTCOL_BLOCKS" );
                    if ( tw < -1 || tw > 10000 ) tw = -1;
                    lv->setFieldWidthQ( cur_field, tw );

                    _field_map["LISTCOL_BLOCKS"] = cur_field;
                    break;
                default:
                    break;
            }
            cur_field++;
        }
        lv->redraw();
    }
}

void NormalMode::fixSpecialSourceList( const std::list<NM_specialsourceInt*> *l, int deleted_row )
{
    std::list<NM_specialsourceInt*>::const_iterator iti1;
    if ( l == NULL ) return;
    if ( deleted_row < 0 ) return;
  
    NM_specialsourceInt *ss1;
  
    for ( iti1 = l->begin(); iti1 != l->end(); iti1++ ) {
        ss1 = *iti1;
        if ( ss1->row == deleted_row ) {
            ss1->row = -1;
        } else if ( ss1->row > deleted_row ) {
            ss1->row--;
        }
    }
}

void NormalMode::showCacheAbs( int pos )
{
    NMCacheEntry *tce;
    int id;

    finishsearchmode();
  
    if ( ce != NULL ) {
        ce->setPos( lv->getYOffset() );
        ce->setXPos( lv->getXOffset() );
        ce->setActiveIsVisible( lv->isRowVisible( lv->getActiveRow() ) );
    }
  
    id = cache->initEnum();
    tce = (NMCacheEntry*)cache->getElementAt( id, pos );
    if ( tce != NULL ) {
        // use copy of dir just to be sure we don't have a freed pointer
        std::string tstr = tce->getDir();
        enterDir( tstr.c_str() );
    }
    cache->closeEnum( id );
}

/*
 * isColumnVisible
 *
 * checks if given column is currently visible
 */
bool NormalMode::isColumnVisible( const WorkerTypes::listcol_t c )
{
    int side = parentlister->getSide();
    const std::vector<WorkerTypes::listcol_t> *dis = wconfig->getVisCols( side );
    int i;
    bool found = false;
  
    if ( dis == NULL ) return false;
    for ( i = 0; i < (int)dis->size(); i++ ) {
        if ( (*dis)[i] == c ) {
            found = true;
            break;
        }
    }
    return found;
}

void NormalMode::deselect( const FileEntry *fe )
{
    FileEntry *tfe;
    int row;

    if ( ce == NULL ) return;
    if ( ce->dirOpened() == false ) return;

    row = 0;
    tfe = NULL;
  
    for ( Verzeichnis::verz_it tfe_it1 = ce->begin();
          tfe_it1 != ce->end();
          tfe_it1++ ) {
        tfe = *tfe_it1;
        if ( tfe->use == true ) {
            if ( tfe->equals( *fe ) == true ) break;
            row++;
        }
        tfe = NULL;
    }
    if (tfe != NULL ) {
        /* found */
        deselect( tfe, row );
    }
}

/*
 * getSpecialsourceForFE
 *
 * creates a NM_specialsource for the given fe if found!
 * can return NULL
 */
NM_specialsourceExt *NormalMode::getSpecialsourceForFE( const FileEntry *searchfe )
{
    int row;
    NM_specialsourceExt *ss2;
    FileEntry *fe;

    if ( ce == NULL ) return NULL;
    if ( ce->dirOpened() == false ) return NULL;
    if ( searchfe == NULL ) return NULL;

    row = 0;
    ss2 = NULL;

    for ( Verzeichnis::verz_it fe_it1 = ce->begin();
          fe_it1 != ce->end();
          fe_it1++ ) {
        fe = *fe_it1;
        if ( fe->use == true ) {
            if ( fe->equals( searchfe ) == true ) {
                ss2 = new NM_specialsourceExt( fe );
                break;
            }
            row++;
        }
    }

    return ss2;
}

NM_specialsourceExt *NormalMode::getSpecialsourceForFilename( const std::string &filename )
{
    int row;
    NM_specialsourceExt *ss2;
    FileEntry *fe;

    if ( ce == NULL ) return NULL;
    if ( ce->dirOpened() == false ) return NULL;
    if ( filename.empty() ) return NULL;

    row = 0;
    ss2 = NULL;

    for ( Verzeichnis::verz_it fe_it1 = ce->begin();
          fe_it1 != ce->end();
          fe_it1++ ) {
        fe = *fe_it1;
        if ( fe->use == true ) {
            if ( filename == fe->fullname ) {
                ss2 = new NM_specialsourceExt( fe );
                break;
            }
            row++;
        }
    }

    return ss2;
}

void NormalMode::lvbDoubleClicked()
{
    showCacheList();
}

int NormalMode::showCacheList()
{
    AWindow *win;
    int my_w, my_h, endmode, mw, mh;
    AGMessage *msg;
    int row, my_lastrow;
    FieldListView *cachelv;
    StringGadget *cachesg;
    char *tstr1, *tstr2;
    int caches = cache->size();
    NMCacheEntry *tce = NULL;
    int id;
    int parents, cachebegin;
    struct timeval my_lastclick, acttime;
    const char *tstr, *ctstr2;
  
    win = new AWindow( aguix, 10, 10, 10, 10, 0, catalog.getLocale( 523 ) );
    win->create();

    AContainer *ac1 = win->setContainer( new AContainer( win, 1, 3 ), true );
    ac1->setMinSpace( 5 );
    ac1->setMaxSpace( 5 );
    ac1->setBorderWidth( 5 );

    ac1->add( new Text( aguix, 0, 0, catalog.getLocale( 524 ), 1 ),
              0, 0, AContainer::CO_INCWNR );

    AContainer *ac1_2 = ac1->add( new AContainer( win, 1, 2 ), 0, 1 );
    ac1_2->setMinSpace( 0 );
    ac1_2->setMaxSpace( 0 );
    ac1_2->setBorderWidth( 0 );
  
    cachelv = (FieldListView*)ac1_2->add( new FieldListView( aguix, 0, 0, 50, 50 ,0 ),
                                          0, 0, AContainer::CO_MIN );
    cachelv->setHBarState( 2 );
    cachelv->setVBarState( 2 );
  
    cachesg = (StringGadget*)ac1_2->add( new StringGadget( aguix, 0, 0, 50, "", 0 ),
                                         0, 1, AContainer::CO_INCW );
  
    AContainer *ac1_1 = ac1->add( new AContainer( win, 2, 1 ), 0, 2 );
    ac1_1->setMinSpace( 5 );
    ac1_1->setMaxSpace( -1 );
    ac1_1->setBorderWidth( 0 );
    Button *okb =(Button*)ac1_1->add( new Button( aguix,
                                                  0,
                                                  0,
                                                  catalog.getLocale( 11 ),
                                                  1,
                                                  0,
                                                  0 ), 0, 0, AContainer::CO_FIX );
    Button *cancelb = (Button*)ac1_1->add( new Button( aguix,
                                                       0,
                                                       0,
                                                       catalog.getLocale( 8 ),
                                                       1,
                                                       0,
                                                       0 ), 1, 0, AContainer::CO_FIX );

    ctstr2 = getCurrentDir();
    if ( ctstr2 != NULL ) {
        tstr1 = dupstring( ctstr2 );
        for ( ;; ) {
            tstr2 = ParentDir( tstr1, NULL );
            if ( tstr2 == NULL ) break;
      
            // add tstr2 to the lv
            row = cachelv->addRow();
            cachelv->setText( row, 0 , tstr2 );
            cachelv->setPreColors( row, FieldListView::PRECOLOR_ONLYACTIVE );
      
            _freesafe( tstr1 );
            tstr1 = tstr2;
            if ( strcmp( tstr1, "/" ) == 0 ) break;
        }
        _freesafe( tstr1 );
    }
  
    parents = cachelv->getElements();

    std::string delim1( strlen( catalog.getLocale( 523 ) ), '-' );
    row = cachelv->addRow();
    cachelv->setText( row, 0, delim1 );
    cachelv->setPreColors( row, FieldListView::PRECOLOR_NOTSELORACT );
    row = cachelv->addRow();
    cachelv->setText( row, 0, catalog.getLocale( 523 ) );
    cachelv->setPreColors( row, FieldListView::PRECOLOR_NOTSELORACT );
    row = cachelv->addRow();
    cachelv->setText( row, 0, catalog.getLocale( 812 ) );
    cachelv->setPreColors( row, FieldListView::PRECOLOR_NOTSELORACT );
    row = cachelv->addRow();
    cachelv->setText( row, 0, delim1 );
    cachelv->setPreColors( row, FieldListView::PRECOLOR_NOTSELORACT );
  
    cachebegin = row + 1;

    if ( caches > 0 ) {
        id = cache->initEnum();
        tce = (NMCacheEntry*)cache->getFirstElement( id );
        while ( tce != NULL ) {
            row = cachelv->addRow();
            cachelv->setText( row, 0, tce->getDir() );
            cachelv->setPreColors( row, FieldListView::PRECOLOR_ONLYACTIVE );
            tce = (NMCacheEntry*)cache->getNextElement( id );
        }
        cache->closeEnum( id );
    }

    // maximize cachelv and store sizes
    cachelv->maximizeX();
    cachelv->maximizeY();
    my_w = cachelv->getWidth();
    my_h = cachelv->getHeight();

    // now resize container to 80% screen size
    mw = aguix->getRootWindowWidth() * 80 / 100;
    mh = aguix->getRootWindowHeight() * 80 / 100;
    ac1->resize( mw, mh );
    ac1->rearrange();

    // and check whether the cachelv needs less space
    // and set min parameter to match 80% or less
    if ( my_w < cachelv->getWidth() ) {
        ac1_2->setMinWidth( my_w, 0, 0 );
    } else {
        ac1_2->setMinWidth( cachelv->getWidth(), 0, 0 );
    }
    if ( my_h < cachelv->getHeight() ) {
        ac1_2->setMinHeight( my_h, 0, 0 );
    } else {
        ac1_2->setMinHeight( cachelv->getHeight(), 0, 0 );
    }

    win->contMaximize( true );
    win->setDoTabCycling( true );
    cachesg->takeFocus();

    if ( parents > 0 ) {
        cachelv->setActiveRow( 0 );
        cachesg->setText( cachelv->getText( 0, 0 ).c_str() );
    } else if ( caches > 0 ) {
        cachelv->setActiveRow( cachebegin );
        cachesg->setText( cachelv->getText( cachebegin, 0 ).c_str() );
    }

    // catch msgs before mapping, this are mostly resize msgs
    while ( ( msg = aguix->GetMessage( win ) ) != NULL ) aguix->ReplyMessage( msg );

    win->show();

    my_lastrow = -1;
    timerclear( &my_lastclick );
    for( endmode = -1; endmode == -1; ) {
        msg = aguix->WaitMessage( win );
        if ( msg != NULL ) {
            switch ( msg->type ) {
                case AG_CLOSEWINDOW:
                    if ( msg->closewindow.window == win->getWindow() ) endmode = 1;
                    break;
                case AG_BUTTONCLICKED:
                    if ( msg->button.button == okb ) endmode = 0;
                    else if ( msg->button.button == cancelb ) endmode = 1;
                    break;
                case AG_FIELDLV_ONESELECT:
                case AG_FIELDLV_MULTISELECT:
                    if ( msg->fieldlv.lv == cachelv ) {
                        row = cachelv->getActiveRow();
                        if ( cachelv->isValidRow( row ) == true ) {
                            if ( ( row < parents ) ||
                                 ( row >= cachebegin ) ) {
                                // this is one of the parents
                                cachesg->setText( cachelv->getText( row, 0 ).c_str() );
                                if ( my_lastrow == row ) {
                                    gettimeofday( &acttime, NULL );
                                    if ( aguix->isDoubleClick( &acttime, &my_lastclick ) == true ) {
                                        endmode = 0;
                                        timerclear( &my_lastclick );
                                    } else {
                                        my_lastclick.tv_sec = acttime.tv_sec;
                                        my_lastclick.tv_usec = acttime.tv_usec;
                                    }
                                } else {
                                    my_lastrow = row;
                                    gettimeofday( &my_lastclick, NULL );
                                }
                            }
                        }
                    }
                    break;
                case AG_STRINGGADGET_CONTENTCHANGE:
                    if ( msg->stringgadget.sg == cachesg ) {
                        // to avoid doubleclick
                        timerclear( &my_lastclick );
                        //TODO: U.U. merkt user ein Lag, deshalb koennte man ein Flag setzen und
                        // ausserhalb messen, wenn letzte Nachricht her ist und dann suchen
                        // Aber dann muss ich getMessage machen
                        tstr = cachesg->getText();
                        for ( row = 0; row < cachelv->getElements(); row++ ) {
                            if ( strcmp( tstr, cachelv->getText( row, 0 ).c_str() ) == 0 ) {
                                if ( ( ( row >= 0 ) && ( row < parents ) ) ||
                                     ( ( row >= cachebegin ) && ( row < ( cachebegin + caches ) ) ) ) {
                                    cachelv->setActiveRow( row );
                                    cachelv->showActive();
                                }
                                break;
                            }
                        }
                    }
                    break;
                case AG_STRINGGADGET_CANCEL:
                    if ( msg->stringgadget.sg == cachesg ) endmode = 1;
                    break;
                case AG_STRINGGADGET_OK:
                    if ( msg->stringgadget.sg == cachesg ) endmode = 0;
                    break;
                case AG_KEYPRESSED:
                    if ( win->isParent( msg->key.window, false ) == true ) {
                        switch ( msg->key.key ) {
                            case XK_Up:
                                row = cachelv->getActiveRow();
                                if ( cachelv->isValidRow( row ) == true ) {
                                    if ( ( row >= parents ) && ( row <= cachebegin ) ) {
                                        row = parents - 1;
                                    } else {
                                        row--;
                                    }
                                    if ( row >= 0 ) {
                                        cachelv->setActiveRow( row );
                                        cachelv->showActive();
                                        cachesg->setText( cachelv->getText( row, 0 ).c_str() );
                                        my_lastrow = -1;
                                    }
                                }
                                break;
                            case XK_Down:
                                row = cachelv->getActiveRow();
                                if ( cachelv->isValidRow( row ) == true ) {
                                    if ( ( row >= ( parents - 1 ) ) && ( row < cachebegin ) ) {
                                        row = cachebegin;
                                    } else {
                                        row++;
                                    }
                                    if ( cachelv->isValidRow( row ) == true ) {
                                        cachelv->setActiveRow( row );
                                        cachelv->showActive();
                                        cachesg->setText( cachelv->getText( row, 0 ).c_str() );
                                        my_lastrow = -1;
                                    }
                                }
                                break;
                            case XK_Prior:
                                row = cachelv->getActiveRow();
                                if ( cachelv->isValidRow( row ) == true ) {
                                    row -= cachelv->getMaxDisplayV() - 1;
                                    if ( ( row >= parents ) && ( row < cachebegin ) ) row = parents - 1;
                                    if ( row < 0 ) {
                                        if ( parents > 0 ) row = 0;
                                        else row = cachebegin;
                                    }
                                } else {
                                    if ( parents > 0 ) row = 0;
                                    else if ( caches > 0 ) row = cachebegin;
                                }
                                if ( cachelv->isValidRow( row ) == true ) {
                                    cachelv->setActiveRow( row );
                                    cachelv->showActive();
                                    cachesg->setText( cachelv->getText( row, 0 ).c_str() );
                                    my_lastrow = -1;
                                }
                                break;
                            case XK_Next:
                                row = cachelv->getActiveRow();
                                if ( cachelv->isValidRow( row ) == true ) {
                                    row += cachelv->getMaxDisplayV() - 1;
                                    if ( ( row >= parents ) && ( row < cachebegin ) ) row = cachebegin;
                                    if ( row >= ( cachebegin + caches ) ) {
                                        if ( caches > 0 ) row = cachebegin + caches - 1;
                                        else row = parents - 1;
                                    }
                                } else {
                                    if ( parents > 0 ) row = 0;
                                    else if ( caches > 0 ) row = cachebegin;
                                }
                                if ( cachelv->isValidRow( row ) == true ) {
                                    cachelv->setActiveRow( row );
                                    cachelv->showActive();
                                    cachesg->setText( cachelv->getText( row, 0 ).c_str() );
                                    my_lastrow = -1;
                                }
                                break;
                            case XK_Home:
                                row = -1;
                                if ( parents > 0 ) row = 0;
                                else if ( caches > 0 ) row = cachebegin;
                                if ( cachelv->isValidRow( row ) == true ) {
                                    cachelv->setActiveRow( row );
                                    cachelv->showActive();
                                    cachesg->setText( cachelv->getText( row, 0 ).c_str() );
                                    my_lastrow = -1;
                                }
                                break;
                            case XK_End:
                                row = -1;
                                if ( caches > 0 ) row = cachebegin + caches - 1;
                                else if ( parents > 0 ) row = parents - 1;
                                if ( cachelv->isValidRow( row ) == true ) {
                                    cachelv->setActiveRow( row );
                                    cachelv->showActive();
                                    cachesg->setText( cachelv->getText( row, 0 ).c_str() );
                                    my_lastrow = -1;
                                }
                                break;
                            case XK_Return:
                            case XK_KP_Enter:
                                if ( cancelb->getHasFocus() == false ) {
                                    endmode = 0;
                                }
                                break;
                            case XK_Escape:
                                endmode = 1;
                                break;
                        }
                    }
                    break;
            }
            aguix->ReplyMessage( msg );
        }
    }
    if ( endmode == 0 ) {
        // ok->take cachesg and enterdir
        enterDir( cachesg->getText() );
    }
    delete win;
    return endmode;
}

int NormalMode::request( const char *title,
                         const char *text,
                         const char *buttons,
                         Requester::request_flags_t flags )
{
    if ( lv != NULL ) lv->redraw();
    return req->request( title, text, buttons, flags );
}

int NormalMode::string_request( const char *title,
                                const char *lines,
                                const char *default_str,
                                const char *buttons,
                                char **return_str,
                                Requester::request_flags_t flags )
{
    if ( lv != NULL ) lv->redraw();
    return req->string_request( title, lines, default_str, buttons, return_str, flags );
}

int NormalMode::request_choose( const char *title,
				const char *text,
				const char *choose_text,
				bool &choose_var,
				const char *buttons,
				Requester::request_flags_t flags )
{
    if ( lv != NULL ) lv->redraw();
    return req->request_choose( title, text, choose_text, choose_var, buttons, flags );
}

int NormalMode::request( CopyOpWin *cowin,
                         const char *title,
                         const char *text,
                         const char *buttons,
                         Requester::request_flags_t flags )
{
    if ( lv != NULL ) lv->redraw();
    if ( cowin != NULL ) {
        return cowin->request( title, text, buttons, flags );
    } else {
        return req->request( title, text, buttons, flags );
    }
}

int NormalMode::string_request( CopyOpWin *cowin,
                                const char *title,
                                const char *lines,
                                const char *default_str,
                                const char *buttons,
                                char **return_str,
                                Requester::request_flags_t flags )
{
    if ( lv != NULL ) lv->redraw();
    if ( cowin != NULL ) {
        return cowin->string_request( title, lines, default_str, buttons, return_str, flags );
    } else {
        return req->string_request( title, lines, default_str, buttons, return_str, flags );
    }
}

int NormalMode::request( DeleteOpWin *dowin,
                         const char *title,
                         const char *text,
                         const char *buttons,
                         Requester::request_flags_t flags )
{
    if ( lv != NULL ) lv->redraw();
    if ( dowin != NULL ) {
        return dowin->request( title, text, buttons, flags );
    } else {
        return req->request( title, text, buttons, flags );
    }
}

int NormalMode::string_request( DeleteOpWin *dowin,
                                const char *title,
                                const char *lines,
                                const char *default_str,
                                const char *buttons,
                                char **return_str,
                                Requester::request_flags_t flags )
{
    if ( lv != NULL ) lv->redraw();
    if ( dowin != NULL ) {
        return dowin->string_request( title, lines, default_str, buttons, return_str, flags );
    } else {
        return req->string_request( title, lines, default_str, buttons, return_str, flags );
    }
}

void NormalMode::changeSortModeForField( int field )
{
    int side = parentlister->getSide();
    int realfield;
    const std::vector<WorkerTypes::listcol_t> *sets;
    int used_fields;
    int newsortmode = sortmode;

    sets = wconfig->getVisCols( side );
    if ( sets == NULL ) return;
  
    used_fields = sets->size();
  
    realfield = field / 2;
    if ( ( realfield >= 0 ) && ( realfield < used_fields ) ) {
        switch ( (*sets)[realfield] ) {
            case WorkerTypes::LISTCOL_NAME:
                if ( ( newsortmode & 0xff ) == SORT_NAME ) {
                    // toggle reverse
                    newsortmode ^= SORT_REVERSE;
                } else {
                    newsortmode = SORT_NAME | ( newsortmode & ~0xff );
                    newsortmode &= ~SORT_REVERSE;
                }
                break;
            case WorkerTypes::LISTCOL_SIZE:
            case WorkerTypes::LISTCOL_SIZEH:
                if ( ( newsortmode & 0xff ) == SORT_SIZE ) {
                    // toggle reverse
                    newsortmode ^= SORT_REVERSE;
                } else {
                    newsortmode = SORT_SIZE | ( newsortmode & ~0xff );
                    newsortmode &= ~SORT_REVERSE;
                }
                break;
            case WorkerTypes::LISTCOL_TYPE:
                if ( ( newsortmode & 0xff ) == SORT_TYPE ) {
                    // toggle reverse
                    newsortmode ^= SORT_REVERSE;
                } else {
                    newsortmode = SORT_TYPE | ( newsortmode & ~0xff );
                    newsortmode &= ~SORT_REVERSE;
                }
                break;
            case WorkerTypes::LISTCOL_OWNER:
                if ( ( newsortmode & 0xff ) == SORT_OWNER ) {
                    // toggle reverse
                    newsortmode ^= SORT_REVERSE;
                } else {
                    newsortmode = SORT_OWNER | ( newsortmode & ~0xff );
                    newsortmode &= ~SORT_REVERSE;
                }
                break;
            case WorkerTypes::LISTCOL_MOD:
                if ( ( newsortmode & 0xff ) == SORT_MODTIME ) {
                    // toggle reverse
                    newsortmode ^= SORT_REVERSE;
                } else {
                    newsortmode = SORT_MODTIME | ( newsortmode & ~0xff );
                    newsortmode &= ~SORT_REVERSE;
                }
                break;
            case WorkerTypes::LISTCOL_ACC:
                if ( ( newsortmode & 0xff ) == SORT_ACCTIME ) {
                    // toggle reverse
                    newsortmode ^= SORT_REVERSE;
                } else {
                    newsortmode = SORT_ACCTIME | ( newsortmode & ~0xff );
                    newsortmode &= ~SORT_REVERSE;
                }
                break;
            case WorkerTypes::LISTCOL_CHANGE:
                if ( ( newsortmode & 0xff ) == SORT_CHGTIME ) {
                    // toggle reverse
                    newsortmode ^= SORT_REVERSE;
                } else {
                    newsortmode = SORT_CHGTIME | ( newsortmode & ~0xff );
                    newsortmode &= ~SORT_REVERSE;
                }
                break;
            case WorkerTypes::LISTCOL_INODE:
                if ( ( newsortmode & 0xff ) == SORT_INODE ) {
                    // toggle reverse
                    newsortmode ^= SORT_REVERSE;
                } else {
                    newsortmode = SORT_INODE | ( newsortmode & ~0xff );
                    newsortmode &= ~SORT_REVERSE;
                }
                break;
            case WorkerTypes::LISTCOL_NLINK:
                if ( ( newsortmode & 0xff ) == SORT_NLINK ) {
                    // toggle reverse
                    newsortmode ^= SORT_REVERSE;
                } else {
                    newsortmode = SORT_NLINK | ( newsortmode & ~0xff );
                    newsortmode &= ~SORT_REVERSE;
                }
                break;
            case WorkerTypes::LISTCOL_BLOCKS:
                //no sort available
                break;
            case WorkerTypes::LISTCOL_PERM:
                if ( ( newsortmode & 0xff ) == SORT_PERMISSION ) {
                    // toggle reverse
                    newsortmode ^= SORT_REVERSE;
                } else {
                    newsortmode = SORT_PERMISSION | ( newsortmode & ~0xff );
                    newsortmode &= ~SORT_REVERSE;
                }
                break;
            case WorkerTypes::LISTCOL_DEST:
                //no sort available
                break;
            default:
                break;
        }
        if ( newsortmode != sortmode ) {
            setSortmode( newsortmode );
        }
    }
}

WCFiletype *NormalMode::findFiletype( std::vector<unsigned int> *v )
{
    unsigned int in1, p, tp;
    const std::list<WCFiletype*> *subtype;
    std::list<WCFiletype*>::const_iterator it1;
    WCFiletype *ft;
  
    ft = NULL;

    // check tree depth
    p = v->size();
    if ( p > 0 ) {
        // we need at least the root type
        in1 = (*v)[p - 1];  // get position
    
        ft =  (WCFiletype*)wconfig->getFiletypes()->getElementAt( in1 );
        // root type, no problem if it is NULL
    
        // now go for each tree depth
        p = v->size();
        if ( p > 1 ) {
            // we need one more depth at least
      
            p--; // skip root index
      
            do {
                // now for each depth

                p--; // jump to next level
                // p is correct entry in "v"

                if ( ft == NULL ) break;

                // get position in current tree depth
                in1 = (*v)[p];

                // now find subtype
                // because of std::list we need to iterate
                subtype = ft->getSubTypeList();
                if ( subtype == NULL ) ft = NULL;  // oops, index but no subtype
                else {
                    // iterate through subtype list and find "in1" position
                    for ( it1 = subtype->begin(), tp = 0; it1 != subtype->end(); it1++, tp++ ) {
                        if ( tp == in1 ) break;
                    }
	  
                    if ( it1 != subtype->end() ) {
                        // found subtype at index "in1"
                        ft = *it1;
                    } else {
                        ft = NULL;
                    }
                }
                // stop when we processed last depth
            } while ( p > 0 );
        }
    }
    return ft;
}

int NormalMode::getSelFiles( std::list<NM_specialsourceInt*> *list, nm_getfiles_t selmode, bool unselect )
{
    int added = 0;
    int row;
    NM_specialsourceInt *ss2;
    FileEntry *fe;

    if ( ce == NULL ) return -1;
    if ( ce->dirOpened() == false ) return -1;
    if ( list == NULL ) return -1;

    if ( selmode != NM_GETFILES_ONLYACTIVE ) {
        row = 0;
        for ( Verzeichnis::verz_it fe_it1 = ce->begin();
              fe_it1 != ce->end();
              fe_it1++ ) {
            fe = *fe_it1;
            if ( fe->use == true ) {
                if ( ( strcmp( fe->name, ".." ) != 0 ) && ( fe->select == true ) ) {
                    ss2 = new NM_specialsourceInt( fe );
                    ss2->row = row;
                    ss2->cod = NULL;
                    list->push_back( ss2 );
                    added++;
                    if ( unselect == true ) deselect( fe, ss2->row );
                }
                row++;
            }
        }
    }

    // for ONLYACTIVE always take the active
    // otherwise only if not ONLYSELECT and empty list
    if ( ( selmode == NM_GETFILES_ONLYACTIVE ) ||
         ( ( selmode != NM_GETFILES_ONLYSELECT ) && ( list->size() == 0 ) ) ) {
        if ( ce->getActiveFE() != NULL ) {
            row = lv->getActiveRow();
            if ( lv->isValidRow( row ) == true ) {
                if ( lv->getData( row ) == ce->getActiveFE()->getID() ) {
                    if ( strcmp( ce->getActiveFE()->name, ".." ) != 0 ) {
                        ss2 = new NM_specialsourceInt( ce->getActiveFE() );
                        ss2->row = row;
                        ss2->cod = NULL;
                        list->push_back( ss2 );
                        added++;
                        if ( unselect == true ) deselect( ce->getActiveFE(), ss2->row );
                    }
                }
            }
        }
    }
  
    return added;
}

void NormalMode::freeSelFiles( std::list<NM_specialsourceExt*> *splist )
{
    std::list<NM_specialsourceExt*>::iterator it1;
  
    if ( splist == NULL ) return;
    for ( it1 = splist->begin(); it1 != splist->end(); it1++ ) {
        if ( *it1 != NULL ) delete *it1;
    }
}

NormalMode::NM_specialsourceInt::NM_specialsourceInt( FileEntry *tfe )
{
    fe = tfe;
    row = -1;
    cod = NULL;
}

NormalMode::NM_specialsourceInt::~NM_specialsourceInt()
{
}

FileEntry *NormalMode::findFE( const FileEntry *searchfe )
{
    FileEntry *fe;

    if ( ce == NULL ) return NULL;
    if ( ce->dirOpened() == false ) return NULL;

    fe = NULL;
    for ( Verzeichnis::verz_it fe_it1 = ce->begin();
          fe_it1 != ce->end();
          fe_it1++ ) {
        fe = *fe_it1;
        if ( fe->equals( searchfe ) == true ) {
            break;
        }
        fe = NULL;
    }

    return fe;
}

void NormalMode::setSSHAllow( enum ssh_allow_t nv )
{
    ssh_allow = nv;
}

bool NormalMode::checkForEmptyActionList( WCFiletype *ft )
{
    List *l;
    bool res = true;

    while ( ft != NULL ) {
        l = ft->getDoubleClickActions();
        //TODO ParentOp pruefen, da so trotz Aktion leere Liste vorkommen kann
        if ( ( l != NULL ) && ( l->size() > 0 ) ) {
            res = false;
            break;
        }
        ft = ft->getParentType();
    }
    return res;
}

int NormalMode::getAndEraseLoadedFieldWidth( std::string str1 )
{
    int tw = -2;
    if ( _loaded_field_width.count( str1 ) > 0 ) {
        tw = _loaded_field_width[str1];
        _loaded_field_width.erase( str1 );
    }
    return tw;
}

int NormalMode::writeFieldWidthsToLoadMap()
{
    std::map<std::string,int>::iterator it1;
    int tw;
    std::string str1;

    if ( lv != NULL ) {
        for ( it1 = _field_map.begin(); it1 != _field_map.end(); it1++ ) {
            tw = lv->getFieldWidth( (*it1).second );
            _loaded_field_width[(*it1).first] = tw;
        }
    }
    return 0;
}

void NormalMode::setQuicksearchEnabled( bool nv )
{
    quicksearch_enabled = nv;
}

int NormalMode::enterPath( const std::string &fullname )
{
    NWC::FSEntry e( fullname );
    int erg = 0;

    if ( e.entryExists() == true ) {
        if ( e.isDir( true ) == true ) {
            erg = enterDir( e.getFullname().c_str() );
        } else {
            std::string dirname = e.getDirname();
            erg = enterDir( dirname.c_str() );
      
            std::string basename = e.getBasename();
            activateEntry( basename );
        }
    }
    return erg;
}

void NormalMode::makeRowActive( int row )
{
    if ( ce == NULL || lv == NULL )
        return;
  
    if ( row < 1 )
        row = 0;

    if ( lv->isValidRow( row ) == false )
        return;

    ce->setActiveFE( getFEForRow( row ) );
    makeListViewRowActive( row );
}

/*
 * does lv->setActiveRow if lister is active
 */
void NormalMode::makeListViewRowActive( int row )
{
    if ( parentlister->isActive() == true )
        lv->setActiveRow( row );
}

FileEntry *NormalMode::getFEForRow( int row )
{
    if ( ce == NULL ) return NULL;

    if ( ce->getSerialOfIDs() != _lv_ids_serial ) {
        request( catalog.getLocale( 347 ),
                 "Some list view IDs don't match the directory!|That's not supposed to happen and I will abort the program.|Please report this error the author!",
                 "Ok" );
        abort();
    }

    if ( lv->isValidRow( row ) == false )
        return NULL;
    
    int id = lv->getData( row );
    return ce->getEntryByID( id );
}

void NormalMode::setEntrySearchString( const std::string &str, bool update_lv )
{
    _entrysearch_matcher.setMatchString( str );
    updateDirFilter();

    if ( update_lv == true && lv != NULL ) {
        bool active_visible = lv->isRowVisible( lv->getActiveRow() );

        rebuildView();

        if ( lv->isRowVisible( lv->getActiveRow() ) == false &&
             active_visible == true ) {
            lv->centerActive();
        }
    }
}

std::string NormalMode::getEntrySearchString()
{
    return _entrysearch_matcher.getMatchString();
}

void NormalMode::setEntrySearchCaseSensitive( bool nv )
{
    _entrysearch_matcher.setMatchCaseSensitive( nv );
}

void NormalMode::updateDirFilter()
{
    if ( _entrysearch_matcher.getMatchString().empty() == true ||
         _entrysearch_matcher.getMatchString() == "*" ||
         searchmodeon == false ||
         _filtered_search_enabled == false ) {
        _dir_filter_sets.setStringFilter( std::auto_ptr<StringMatcher>() );
        m_filtered_search_active = false;
    } else {
        _dir_filter_sets.setStringFilter( std::auto_ptr<StringMatcher>( new StringMatcherFNMatch( _entrysearch_matcher ) ) );
        m_filtered_search_active = true;
    }
}

bool NormalMode::filteredSearchActive() const
{
    return m_filtered_search_active;
}

void NormalMode::setFilteredSearchEnabled( bool nv )
{
    _filtered_search_enabled = nv;
}

void NormalMode::activateEntry( const std::string &name )
{
    FileEntry *fe;
    int row;
  
    if ( ce == NULL ) return;
    if ( ce->dirOpened() == false ) return;

    for ( Verzeichnis::verz_it fe_it1 = ce->begin();
          fe_it1 != ce->end();
          fe_it1++ ) {
        fe = *fe_it1;
        if ( fe->use == true ) {
            if ( strcmp( fe->name, name.c_str() ) == 0 ) {
                // hit
                ce->setActiveFE( fe );
                row = 0;
                while(lv->isValidRow(row)==true) {
                    if ( lv->getData( row ) == fe->getID() ) {
                        makeListViewRowActive( row );
                        break;
                    }
                    row++;
                }
                lv->showActive();
                lv->redraw();
                break;
            }
        }
    }
  
    return;
}

int NormalMode::buildSpecialSourceIntList( const std::list<NM_specialsourceExt*> &inlist,
                                           std::list<NM_specialsourceInt*> &outlist )
{
    std::list<NM_specialsourceExt*>::const_iterator ite1;
    int count = 0;
    
    for ( ite1 = inlist.begin();
          ite1 != inlist.end();
          ite1++ ) {
        NM_specialsourceExt *sse = *ite1;
        
        // skip entry ..
        if ( strcmp( sse->entry()->name, ".." ) == 0 )
            continue;

        // search for internal FileEntry matching entry from external source
        // remember, external FileEntry is a copy
        FileEntry *fe = findFE( sse->entry() );

        if ( fe != NULL ) {
            // valid entry found
            NM_specialsourceInt *ss2 = new NM_specialsourceInt( fe );
            ss2->cod = NULL;
            outlist.push_back( ss2 );
            count++;
        }
    }

    return count;
}

void NormalMode::deactivateFilteredSearch()
{
    finishsearchmode();
    if ( filteredSearchActive() == true ) {
        setEntrySearchString( "" );
    }
}

void NormalMode::buildContextPopUpMenu( FileEntry *entry )
{
    freePopUpMenus();

    int descr_id = 0;

    if ( entry == NULL ) return;
    if ( entry->isCorrupt == true ) return;
    if ( strcmp( entry->name, ".." ) == 0 ) return;

    std::list<PopUpMenu::PopUpEntry> m1;
    PopUpMenu::PopUpEntry e1;

    m_current_popup_settings.reset();

    e1.type = PopUpMenu::SUBMENU;
    e1.name = catalog.getLocale( 833 );
    std::list<PopUpMenu::PopUpEntry> labelmenu = buildLabelPopUpData( entry );
    e1.submenu = &labelmenu;
    e1.id = descr_id++;

    m_current_popup_settings.label_menu_id = e1.id;

    m1.push_back( e1 );
    e1.type = PopUpMenu::HLINE;
    m1.push_back( e1 );

    e1.name = catalog.getLocale( 768 );
    e1.type = PopUpMenu::HEADER;
    m1.push_back( e1 );
    e1.type = PopUpMenu::HLINE;
    m1.push_back( e1 );
    
    m_current_popup_settings.popup_descr[descr_id] = RefCount<ActionDescr>( new DoubleClickAction::DoubleClickActionDescr() );

    e1.type = PopUpMenu::NORMAL;
    e1.name = catalog.getLocale( 252 );
    e1.id = descr_id++;
    m1.push_back( e1 );
    
    WCFiletype *ft;
    ft = entry->filetype;
    if ( ft == NULL ) {
        if ( entry->isDir() == true ) {
            ft = wconfig->getdirtype();
        } else {
            ft = wconfig->getnotyettype();
        }
    }

    List *s_l = ft->getActionList( true, ShowAction::ShowActionDescr() );
    if ( s_l != NULL && s_l->size() > 0 ) {
        m_current_popup_settings.popup_descr[descr_id] = RefCount<ActionDescr>( new ShowAction::ShowActionDescr() );

        e1.type = PopUpMenu::NORMAL;
        e1.name = catalog.getLocale( 253 );
        e1.id = descr_id++;
        m1.push_back( e1 );
    }

    List *rs_l = ft->getActionList( true, RawShowAction::RawShowActionDescr() );
    if ( rs_l != NULL && rs_l->size() > 0 ) {
        m_current_popup_settings.popup_descr[descr_id] = RefCount<ActionDescr>( new RawShowAction::RawShowActionDescr() );

        e1.type = PopUpMenu::NORMAL;
        e1.name = catalog.getLocale( 254 );
        e1.id = descr_id++;
        m1.push_back( e1 );
    }

    for ( int i = 0; i < 10; i++ ) {
        List *u_l = ft->getActionList( true, UserAction::UserActionDescr( i ) );
        if ( u_l != NULL && u_l->size() > 0 ) {
            m_current_popup_settings.popup_descr[descr_id] = RefCount<ActionDescr>( new UserAction::UserActionDescr( i ) );

            e1.type = PopUpMenu::NORMAL;
            //TODO not nice using index for locale but works
            e1.name = catalog.getLocale( 255 + i );
            e1.id = descr_id++;
            m1.push_back( e1 );
        }
    }

    std::list<std::string> action_names_of_current_entry_list;
    std::set< std::string > action_names_of_current_entry,
        common_action_names_of_all_entries,
        all_action_names_of_all_entries;

    ft->fillNamesOfCustomActions( true, action_names_of_current_entry_list );
    for ( std::list<std::string>::const_iterator it1 = action_names_of_current_entry_list.begin();
          it1 != action_names_of_current_entry_list.end();
          it1++ ) {
        action_names_of_current_entry.insert( *it1 );
        common_action_names_of_all_entries.insert( *it1 );
        all_action_names_of_all_entries.insert( *it1 );
    }

    //TODO build submenu for all selected entries

    std::list<NM_specialsourceInt*> tlist;
    int res = getSelFiles( &tlist, NM_GETFILES_ONLYSELECT, false );
    if ( res > 0 ) {
        for ( std::list<NM_specialsourceInt*>::const_iterator iti1 = tlist.begin();
              iti1 != tlist.end();
              iti1++ ) {
            WCFiletype *tft = (*iti1)->entry()->filetype;

            if ( tft == NULL ) {
                if ( (*iti1)->entry()->isDir() == true ) {
                    tft = wconfig->getdirtype();
                } else {
                    tft = wconfig->getnotyettype();
                }
            }

            if ( tft != NULL ) {
                std::list<std::string> custom_actions;
                std::set< std::string > temp_names, new_names;

                tft->fillNamesOfCustomActions( true, custom_actions );
                for ( std::list<std::string>::const_iterator it1 = custom_actions.begin();
                      it1 != custom_actions.end();
                      it1++ ) {
                    temp_names.insert( *it1 );

                    all_action_names_of_all_entries.insert( *it1 );
                }

                std::set_intersection( common_action_names_of_all_entries.begin(),
                                       common_action_names_of_all_entries.end(),
                                       temp_names.begin(),
                                       temp_names.end(),
                                       std::inserter( new_names, new_names.begin() ) );

                common_action_names_of_all_entries = new_names;
            }

            delete *iti1;
        }

        tlist.clear();
    }

    // first all common actions
    for ( std::set< std::string >::const_iterator it1 = common_action_names_of_all_entries.begin();
          it1 != common_action_names_of_all_entries.end();
          it1++ ) {
        m_current_popup_settings.popup_descr[descr_id] = RefCount<ActionDescr>( new RunCustomAction::RunCustomActionDescr( *it1 ) );

        e1.type = PopUpMenu::NORMAL;
        //e1.name = AGUIXUtils::formatStringToString( catalog.getLocale( 779 ), it1->c_str() );
        e1.name = it1->c_str();
        e1.id = descr_id++;
        m1.push_back( e1 );
    }

    // now all of current entry
    std::list<PopUpMenu::PopUpEntry> current_entry_menu;

    if ( action_names_of_current_entry.size() > common_action_names_of_all_entries.size() ) {
        e1.type = PopUpMenu::SUBMENU;
        e1.name = catalog.getLocale( 970 );
        e1.submenu = &current_entry_menu;
        e1.id = descr_id++;
        m_current_popup_settings.entry_is_action_submenu[e1.id] = true;
        m1.push_back( e1 );

        for ( std::set< std::string >::const_iterator it1 = action_names_of_current_entry.begin();
              it1 != action_names_of_current_entry.end();
              it1++ ) {

            if ( common_action_names_of_all_entries.count( *it1 ) < 1 ) {
                m_current_popup_settings.popup_descr[descr_id] = RefCount<ActionDescr>( new RunCustomAction::RunCustomActionDescr( *it1 ) );

                e1.type = PopUpMenu::NORMAL;
                e1.name = it1->c_str();
                e1.id = descr_id++;
                current_entry_menu.push_back( e1 );
            }
        }
    }

#if 0
    // this is currently disabled, it might be too confusing to start an action defined by some other entry
    // this code is also not complete
    // and now all actions
    std::list<PopUpMenu::PopUpEntry> all_entries_menu;

    if ( all_action_names_of_all_entries.size() > common_action_names_of_all_entries.size() ) {
        int count = 0;
        for ( std::set< std::string >::const_iterator it1 = all_action_names_of_all_entries.begin();
              it1 != all_action_names_of_all_entries.end();
              it1++ ) {
            if ( common_action_names_of_all_entries.count( *it1 ) < 1 &&
                 action_names_of_current_entry.count( *it1 ) < 1 ) {
                count++;
            }
        }

        if ( count > 0 ) {
            e1.type = PopUpMenu::SUBMENU;
#warning TODO TR
            e1.name = "additional actions defined by any of the selected entries"/*catalog.getLocale( 768 )*/;
            e1.submenu = &all_entries_menu;
            e1.id = descr_id++;
            m1.push_back( e1 );

            for ( std::set< std::string >::const_iterator it1 = all_action_names_of_all_entries.begin();
                  it1 != all_action_names_of_all_entries.end();
                  it1++ ) {
                if ( common_action_names_of_all_entries.count( *it1 ) < 1 &&
                     action_names_of_current_entry.count( *it1 ) < 1) {
                    m_current_popup_settings.popup_descr[descr_id] = RefCount<ActionDescr>( new RunCustomAction::RunCustomActionDescr( *it1 ) );

                    e1.type = PopUpMenu::NORMAL;
                    e1.name = it1->c_str();
                    e1.id = descr_id++;
                    all_entries_menu.push_back( e1 );
                }
            }
        }
    }
#endif

    m_current_popup_settings.lv_popup_menu = new PopUpMenu( aguix, m1 );
    m_current_popup_settings.lv_popup_menu->create();
}

void NormalMode::lv_pressed( int row, bool open_under_mouse )
{
    if ( ce == NULL || lv == NULL ) return;
    
    if ( lv->isValidRow( row ) == false ) return;
    makeRowActive( row );
    
    FileEntry *fe = getFEForRow( row );
    if ( fe != NULL ) {
        buildContextPopUpMenu( fe );
        m_current_popup_settings.entry_for_popup = std::auto_ptr<FileEntry>( new FileEntry( *fe ) );
        if ( m_current_popup_settings.lv_popup_menu != NULL ) {
            if ( open_under_mouse == true ) {
                m_current_popup_settings.lv_popup_menu->show();
            } else {
                int tx, ty;
                
                aguix->getWidgetRootPosition( lv, &tx, &ty );
                tx += lv->getWidth() / 2;
                ty += lv->getHeight() / 2;
                m_current_popup_settings.lv_popup_menu->show( tx, ty, PopUpMenu::POPUP_CENTER_IN_POSITION );
                //TODO better query the position of the active row from LV and use
                // this position
            }
        }
    }
}

void NormalMode::startLVPopUpAction( AGMessage *msg )
{
    if ( msg == NULL || ce == NULL || lv == NULL ) return;
    
    FileEntry *fe = getFEForRow( lv->getActiveRow() );
    if ( fe == NULL ) return;

    if ( m_current_popup_settings.entry_for_popup.get() == NULL ) return;
    if ( m_current_popup_settings.entry_for_popup->equals( *fe ) == false ) return;
    // ok, still the same entry
    
    int id = msg->popupmenu.clicked_entry_id;

    if ( m_current_popup_settings.entry_is_action_submenu[ id ] == true) {
        if ( msg->popupmenu.recursive_ids != NULL &&
             ! msg->popupmenu.recursive_ids->empty() ) {
            id = msg->popupmenu.recursive_ids->front();
        }
    }
    
    if ( m_current_popup_settings.popup_descr.count( id ) > 0 ) {
        RefCount<ActionDescr> action_descr = m_current_popup_settings.popup_descr[id];

        if ( action_descr.getVal() != NULL ) {
            if ( dynamic_cast<DoubleClickAction::DoubleClickActionDescr*>( action_descr.getVal() ) != NULL ) {
                startAction( fe );
            } else {

                ActionMessage amsg( parentlister->getWorker() );

                prepareContext( amsg );

                WCFiletype *ft;
                ft = fe->filetype;
                if ( ft == NULL ) {
                    if ( fe->isDir() == true ) {
                        ft = wconfig->getdirtype();
                    } else {
                        ft = wconfig->getnotyettype();
                    }
                }

                amsg.filetype = ft;

                List *action_list = NULL;

                amsg.m_action_descr = action_descr;
                action_list = ft->getActionList( true, *( action_descr ) );

                if ( action_list != NULL ) {
                    parentlister->getWorker()->interpret( action_list, &amsg );
                }
            }
        }
    } else {
        // no action so check other menu options
        if ( msg->popupmenu.recursive_ids != NULL &&
             msg->popupmenu.recursive_ids->empty() == false ) {
            if ( msg->popupmenu.recursive_ids->back() == m_current_popup_settings.label_menu_id ) {
                std::list<int> entry_ids = *msg->popupmenu.recursive_ids;
                entry_ids.pop_back();
                handleLabelPopUp( fe, entry_ids, msg );
            }
        }
    }
}

void NormalMode::openContextMenu()
{
    if ( ce == NULL || lv == NULL ) return;

    lv_pressed( lv->getActiveRow(), false );
}

void NormalMode::bookmarksChanged()
{
    m_bookmarks_has_been_changed = true;
}

void NormalMode::updateOnBookmarkChange()
{
    if ( m_bookmarks_has_been_changed == true ) {
        _dir_filter_sets.bookmarksChanged();
        m_dir_bookmarks_sets.bookmarksChanged();
        
        //TODO rebuildView is enough currently but update would be more
        // complete
        //update( false, true );
        rebuildView();
        
        m_bookmarks_has_been_changed = false;
    }
}


std::list<PopUpMenu::PopUpEntry> NormalMode::buildLabelPopUpData( FileEntry *entry )
{
    int descr_id = 0;
    std::list<PopUpMenu::PopUpEntry> m1;

    if ( entry == NULL ) return m1;
    if ( entry->isCorrupt == true ) return m1;
    if ( strcmp( entry->name, ".." ) == 0 ) return m1;

    std::auto_ptr<BookmarkDBEntry> db_entry = Worker::getBookmarkDBInstance().getEntry( entry->fullname );

    PopUpMenu::PopUpEntry e1;
    e1.name = catalog.getLocale( 834 );
    if ( db_entry.get() != NULL ) {
        e1.type = PopUpMenu::NORMAL;
    } else {
        e1.type = PopUpMenu::GREYED_OUT;
    }
    e1.id = descr_id++;

    m_current_popup_settings.label_id_to_action[e1.id] = label_popup_table_t( label_popup_table_t::REMOVE_LABEL, "" );

    m1.push_back( e1 );

    e1.type = PopUpMenu::HLINE;
    m1.push_back( e1 );
    
    std::list<std::string> cats = Worker::getBookmarkDBInstance().getCats();
    const std::map<std::string, WConfig::ColorDef::label_colors_t> labels = wconfig->getColorDefs().getLabelColors();
    std::map<std::string, WConfig::ColorDef::label_colors_t>::const_iterator label_it;

    for ( label_it = labels.begin(); label_it != labels.end(); ++label_it ) {
        if ( label_it->first == "" ) continue;

        if ( db_entry.get() == NULL ||
             db_entry->getCategory() != label_it->first ) {
            e1.type = PopUpMenu::NORMAL;
        } else {
            e1.type = PopUpMenu::GREYED_OUT;
        }
        e1.name = label_it->first;
        e1.id = descr_id++;

        e1.fg = label_it->second.normal_fg;
        e1.bg = label_it->second.normal_bg;

        m_current_popup_settings.label_id_to_action[e1.id] = label_popup_table_t( label_popup_table_t::ADD_LABEL, label_it->first );

        m1.push_back( e1 );
    }
    e1.fg = e1.bg = -1;

    bool first = true;
    for ( std::list<std::string>::iterator it1 = cats.begin();
          it1 != cats.end();
          ++it1 ) {
        if ( *it1 == "" ) continue;

        if ( labels.find( *it1 ) == labels.end() ) {

            if ( first == true ) {
                e1.type = PopUpMenu::HLINE;
                m1.push_back( e1 );
                first = false;
            }

            if ( db_entry.get() == NULL ||
                 db_entry->getCategory() != *it1 ) {
                e1.type = PopUpMenu::NORMAL;
            } else {
                e1.type = PopUpMenu::GREYED_OUT;
            }
            e1.name = *it1;
            e1.id = descr_id++;

            m_current_popup_settings.label_id_to_action[e1.id] = label_popup_table_t( label_popup_table_t::ADD_LABEL, *it1 );

            m1.push_back( e1 );
        }
    }

    if ( first == true ) {
        e1.type = PopUpMenu::HLINE;
        m1.push_back( e1 );
        first = false;
    }
    
    e1.type = PopUpMenu::EDITABLE;
    e1.name = catalog.getLocale( 910 );
    e1.id = descr_id++;
    
    m_current_popup_settings.label_id_to_action[e1.id] = label_popup_table_t( label_popup_table_t::ADD_CUSTOM_LABEL, "" );
    
    m1.push_back( e1 );

    e1.type = PopUpMenu::HLINE;
    m1.push_back( e1 );
    
    if ( db_entry.get() == NULL ||
         db_entry->getCategory() != "" ) {
        e1.type = PopUpMenu::NORMAL;
    } else {
        e1.type = PopUpMenu::GREYED_OUT;
    }
    e1.name = catalog.getLocale( 847 );
    e1.id = descr_id++;
    
    m_current_popup_settings.label_id_to_action[e1.id] = label_popup_table_t( label_popup_table_t::ADD_LABEL, "" );
    
    m1.push_back( e1 );
    
    return m1;
}

void NormalMode::handleLabelPopUp( AGMessage *msg )
{
    if ( msg == NULL || ce == NULL || lv == NULL ) return;

    if ( msg->popupmenu.menu != m_current_popup_settings.label_popup_menu.get() ) return;
    
    FileEntry *fe = getFEForRow( lv->getActiveRow() );
    if ( fe == NULL ) return;

    if ( m_current_popup_settings.entry_for_popup.get() == NULL ) return;
    if ( m_current_popup_settings.entry_for_popup->equals( *fe ) == false ) return;
    // ok, still the same entry
    
    std::list<int> entry_ids = *msg->popupmenu.recursive_ids;
    handleLabelPopUp( fe, entry_ids, msg );
}

void NormalMode::handleLabelPopUp( FileEntry *fe, const std::list<int> &entry_ids, AGMessage *msg )
{
    if ( entry_ids.empty() == true ) return;
    if ( fe == NULL ) return;
    if ( msg == NULL ) return;
    if ( ! ( msg->type == AG_POPUPMENU_CLICKED ||
             msg->type == AG_POPUPMENU_ENTRYEDITED ) ) return;

    int id = entry_ids.back();
    
    std::map<int, label_popup_table_t>::const_iterator it1 = m_current_popup_settings.label_id_to_action.find( id );
    if ( it1 != m_current_popup_settings.label_id_to_action.end() ) {
        // found entry

        BookmarkDBProxy &bookmarks = Worker::getBookmarkDBInstance();
        //TODO read necessary?
        bookmarks.read();

        std::auto_ptr<BookmarkDBEntry> entry = bookmarks.getEntry( fe->fullname );

        bool use_parent = true;

        if ( fe->isDir() ) use_parent = false;

        if ( it1->second.type == label_popup_table_t::REMOVE_LABEL ) {
            if ( entry.get() != NULL ) {
                bookmarks.delEntry( *entry );
                bookmarks.write();
            }
        } else if ( it1->second.type == label_popup_table_t::ADD_LABEL ) {
            if ( entry.get() != NULL ) {
                BookmarkDBEntry newentry( *entry );
                newentry.setCategory( it1->second.label );
                bookmarks.updateEntry( *entry, newentry );
            } else {
                BookmarkDBEntry newentry( it1->second.label, fe->fullname, "", use_parent );
                bookmarks.addEntry( newentry );
            }
            bookmarks.write();
        } else if ( it1->second.type == label_popup_table_t::ADD_CUSTOM_LABEL ) {
            if ( msg->popupmenu.edit_content != NULL ) {
                std::string custom_label = *msg->popupmenu.edit_content;
                if ( entry.get() != NULL ) {
                    BookmarkDBEntry newentry( *entry );
                    newentry.setCategory( custom_label );
                    bookmarks.updateEntry( *entry, newentry );
                } else {
                    BookmarkDBEntry newentry( custom_label, fe->fullname, "", use_parent );
                    bookmarks.addEntry( newentry );
                }
                bookmarks.write();
            }
        }
        
        updateOnBookmarkChange();
    }
}

void NormalMode::buildLabelPopUpMenu( FileEntry *entry )
{
    freePopUpMenus();

    if ( entry == NULL ) return;
    if ( entry->isCorrupt == true ) return;
    if ( strcmp( entry->name, ".." ) == 0 ) return;

    m_current_popup_settings.reset();

    std::list<PopUpMenu::PopUpEntry> labelmenu = buildLabelPopUpData( entry );

    m_current_popup_settings.label_menu_id = -2;

    m_current_popup_settings.label_popup_menu = std::auto_ptr<PopUpMenu>( new PopUpMenu( aguix, labelmenu ) );
    m_current_popup_settings.label_popup_menu->create();
}

void NormalMode::openLabelPopUp()
{
    if ( ce == NULL || lv == NULL ) return;

    int row = lv->getActiveRow();

    if ( lv->isValidRow( row ) == false ) return;
    
    FileEntry *fe = getFEForRow( row );
    if ( fe != NULL ) {
        buildLabelPopUpMenu( fe );
        m_current_popup_settings.entry_for_popup = std::auto_ptr<FileEntry>( new FileEntry( *fe ) );
        if ( m_current_popup_settings.label_popup_menu.get() != NULL ) {
            int tx, ty;
            
            aguix->getWidgetRootPosition( lv, &tx, &ty );
            tx += lv->getWidth() / 2;
            ty += lv->getHeight() / 2;
            m_current_popup_settings.label_popup_menu->show( tx, ty, PopUpMenu::POPUP_CENTER_IN_POSITION );
            //TODO better query the position of the active row from LV and use
            // this position
        }
    }
}

void NormalMode::freePopUpMenus()
{
    m_current_popup_settings.reset();
}

void NormalMode::setHighlightBookmarkPrefix( bool nv )
{
    _dir_filter_sets.setHighlightBookmarkPrefix( nv );
    rebuildView();
}

bool NormalMode::getHighlightBookmarkPrefix() const
{
    return _dir_filter_sets.getHighlightBookmarkPrefix();
}

void NormalMode::setBookmarkFilter( DirFilterSettings::bookmark_filter_t v )
{
    finishsearchmode();
  
    _dir_filter_sets.setBookmarkFilter( v );
    rebuildView();
    setName();
    if ( lv != NULL ) lv->showActive();
}

DirFilterSettings::bookmark_filter_t NormalMode::getBookmarkFilter() const
{
    return _dir_filter_sets.getBookmarkFilter();
}

void NormalMode::setSpecificBookmarkLabel( const std::string &l )
{
    finishsearchmode();
    
    _dir_filter_sets.setSpecificBookmarkLabel( l );
    rebuildView();
    setName();
    if ( lv != NULL ) lv->showActive();
}

const std::string &NormalMode::getSpecificBookmarkLabel() const
{
    return _dir_filter_sets.getSpecificBookmarkLabel();
}

NormalMode::nm_filter_t NormalMode::checkFilter( const char *filter ) throw( int )
{
    NM_Filter::check_t newmode = NM_Filter::INACTIVE;
    try {
        newmode = _dir_filter_sets.checkFilter( filter );
    } catch ( int i ) {
        throw i;
    }

    switch ( newmode ) {
        case NM_Filter::EXCLUDE:
            return NM_FILTER_EXCLUDE;
            break;
        case NM_Filter::INCLUDE:
            return NM_FILTER_INCLUDE;
            break;
        default:
            break;
    }
    return NM_FILTER_UNSET;
}

NormalMode::PopUpSettings::PopUpSettings() : label_menu_id( -1 ),
                                             lv_popup_menu( NULL ),
                                             m_tab( -1 )
{}

void NormalMode::PopUpSettings::reset()
{
    label_menu_id = -1;
    label_id_to_action.clear();
    label_popup_menu.reset();
    tab_popup_menu.reset();
    popup_descr.clear();
    entry_for_popup.reset();
    if ( lv_popup_menu != NULL ) {
        delete lv_popup_menu;
        lv_popup_menu = NULL;
    }
    m_tab = -1;
}

int NormalMode::queryLVDimensions( int &tx, int &ty,
                                   int &tw, int &th )
{
    if ( lv != NULL ) {
        aguix->getWidgetRootPosition( lv, &tx, &ty );
        tw = lv->getWidth();
        th = lv->getHeight();
        return 0;
    }
    return 1;
}

void NormalMode::updateTabs()
{
    if ( m_tab_b == NULL ) return;

    if ( lv != NULL && ce != NULL ) {
        setTabOption( m_tab_b->getSelectedOption(), ce->getDir() );
    } else {
        setTabOption( m_tab_b->getSelectedOption(), "" );
    }
}

void NormalMode::newTab()
{
    int tab = -1;

    if ( m_tab_b == NULL ) return;

    if ( ce != NULL ) {
        tab = addTabOption( ce->getDir() );
    } else {
        tab = addTabOption( "" );
    }
    if ( tab >= 0 ) {
        m_tab_b->setOption( tab );
    }
    m_tab_b->redraw();
}

void NormalMode::closeCurrentTab()
{
    if ( m_tab_b == NULL ) return;

    if ( m_tab_b->getNrOfOptions() < 2 ) return;

    int tab = m_tab_b->getSelectedOption();

    removeTabOption( tab );

    tab = m_tab_b->getSelectedOption();
    switchToTab( tab );
}

void NormalMode::closeTab( int tab )
{
    if ( m_tab_b == NULL ) return;

    if ( m_tab_b->getNrOfOptions() < 2 ) return;

    if ( tab < 0 || tab >= m_tab_b->getNrOfOptions() ) return;

    int current_tab = m_tab_b->getSelectedOption();

    removeTabOption( tab );

    if ( current_tab > tab ) {
        switchToTab( current_tab - 1 );
    } else {
        switchToTab( current_tab );
    }
}

void NormalMode::switchToTab( int tab )
{
    if ( m_tab_b == NULL ) return;

    const char *dirstr = m_tab_entries.getEntry( tab ).c_str();

    if ( dirstr != NULL ) {
        m_tab_b->setOption( tab );
        if ( strlen( dirstr ) > 0 ) {

            char *dir = dupstring( dirstr );
            for(;;) {
                int e = enterDir( dir );
                if ( e == 0 ) break;  // read ok, so finish
                if ( strcmp( dir, "/" ) == 0 ) break;  // there is no dir we can display anymore, so finish
                char *tstr = ParentDir( dir, NULL );
                _freesafe( dir );
                dir = tstr;
            }
            _freesafe( dir );
        }
    }
}

void NormalMode::switchToTabDirection( int direction )
{
    if ( m_tab_b == NULL ) return;

    int tab = m_tab_b->getSelectedOption() + ( direction % m_tab_b->getNrOfOptions() );
    tab += m_tab_b->getNrOfOptions();
    tab %= m_tab_b->getNrOfOptions();
    switchToTab( tab );
}

NMCacheEntry *NormalMode::findOldestCacheEntryNotInTab()
{
    int tabs = m_tab_b->getNrOfOptions();
    NMCacheEntry *tce = (NMCacheEntry*)cache->getFirstElement();
    while ( tce != NULL ) {
        int tab = -1;
        for ( int i = 0; i < tabs; i++ ) {
            const char *dirstr = m_tab_entries.getEntry( i ).c_str();
            if ( dirstr != NULL &&
                 strcmp( tce->getDir(), dirstr ) == 0 ) {
                tab = i;
            }
        }
        if ( tab == -1 ) break;
        tce = (NMCacheEntry*)cache->getNextElement();
    }

    return tce;
}

void NormalMode::setTabOption( int pos, const std::string &path )
{
    if ( pos < 0 || pos > m_tab_b->getNrOfOptions() ) return;

    if ( pos == m_tab_b->getNrOfOptions() ) {
        addTabOption( path );
    } else {
        if ( path.length() < 2 ) {
            m_tab_b->setOption( pos, path.c_str() );
        } else {
            m_tab_b->setOption( pos, NWC::Path::basename( path ).c_str() );
        }
        m_tab_entries.updateEntry( pos, path );
    }
}

int NormalMode::addTabOption( const std::string &path )
{
    m_tab_entries.insertEntry( path );
    if ( path.length() < 2 ) {
        return m_tab_b->addOption( path.c_str() );
    } else {
        return m_tab_b->addOption( NWC::Path::basename( path ).c_str() );
    }
}

void NormalMode::removeTabOption( int pos )
{
    if ( pos < 0 || pos >= m_tab_b->getNrOfOptions() ) return;

    m_tab_entries.removeEntry( pos );
    m_tab_b->removeOption( pos );
}

void NormalMode::updateSpaceTimeout( loff_t time_ms )
{
    if ( m_current_space_update_ms == time_ms ) return;

    if ( m_current_space_update_ms != 0 ) {
        parentlister->getWorker()->unregisterTimeout( m_current_space_update_ms );
        m_current_space_update_ms = 0;
    }
    
    if ( time_ms > 0 ) {
        parentlister->getWorker()->registerTimeout( time_ms );
        m_current_space_update_ms = time_ms;
    }
}

std::list<PopUpMenu::PopUpEntry> NormalMode::buildTabPopUpData()
{
    int descr_id = 0;
    std::list<PopUpMenu::PopUpEntry> m1;

    PopUpMenu::PopUpEntry e1;
    // e1.name = "Tab menu";
    // e1.type = PopUpMenu::HEADER;
    // e1.id = descr_id++;
    // m1.push_back( e1 );

    // e1.type = PopUpMenu::HLINE;
    // m1.push_back( e1 );
    
    e1.type = PopUpMenu::NORMAL;
    e1.name = catalog.getLocale( 852 );
    e1.id = descr_id++;
    
    m_current_popup_settings.tab_id_to_action[e1.id] = tab_popup_table_t( tab_popup_table_t::NEW_TAB );
    
    m1.push_back( e1 );

    if ( m_tab_entries.getNrOfEntries() < 2 ) {
        e1.type = PopUpMenu::GREYED_OUT;
    } else {
        e1.type = PopUpMenu::NORMAL;
    }
    e1.name = catalog.getLocale( 952 );
    e1.id = descr_id++;
    
    m_current_popup_settings.tab_id_to_action[e1.id] = tab_popup_table_t( tab_popup_table_t::DEL_TAB );
    
    m1.push_back( e1 );

    e1.type = PopUpMenu::NORMAL;
    e1.name = catalog.getLocale( 953 );
    e1.id = descr_id++;
    
    m_current_popup_settings.tab_id_to_action[e1.id] = tab_popup_table_t( tab_popup_table_t::TAB_TO_OTHER_SIDE );
    
    m1.push_back( e1 );

    e1.type = PopUpMenu::NORMAL;
    e1.name = catalog.getLocale( 954 );
    e1.id = descr_id++;
    
    m_current_popup_settings.tab_id_to_action[e1.id] = tab_popup_table_t( tab_popup_table_t::TAB_TO_OTHER_SIDE_AS_NEW );
    
    m1.push_back( e1 );
    
    return m1;
}

void NormalMode::openTabPopUp( int tab )
{
    freePopUpMenus();

    m_current_popup_settings.reset();

    std::list<PopUpMenu::PopUpEntry> tabmenu = buildTabPopUpData();

    m_current_popup_settings.tab_popup_menu = std::auto_ptr<PopUpMenu>( new PopUpMenu( aguix, tabmenu ) );
    m_current_popup_settings.tab_popup_menu->create();
    m_current_popup_settings.tab_popup_menu->show();
    m_current_popup_settings.m_tab = tab;
}

void NormalMode::handleTabPopUp( AGMessage *msg )
{
    if ( msg == NULL || lv == NULL ) return;

    if ( ! ( msg->type == AG_POPUPMENU_CLICKED ||
             msg->type == AG_POPUPMENU_ENTRYEDITED ) ) return;

    if ( msg->popupmenu.menu != m_current_popup_settings.tab_popup_menu.get() ) return;
    
    std::list<int> entry_ids = *msg->popupmenu.recursive_ids;

    if ( entry_ids.empty() == true ) return;

    int id = entry_ids.back();
    
    std::map<int, tab_popup_table_t>::const_iterator it1 = m_current_popup_settings.tab_id_to_action.find( id );

    if ( it1 != m_current_popup_settings.tab_id_to_action.end() ) {
        // found entry

        if ( it1->second.type == tab_popup_table_t::NEW_TAB ) {
            newTab();
        } else if ( it1->second.type == tab_popup_table_t::DEL_TAB ) {
            closeTab( m_current_popup_settings.m_tab );
        } else if ( it1->second.type == tab_popup_table_t::TAB_TO_OTHER_SIDE ) {
            std::string dir = getDirectoryOfTab( m_current_popup_settings.m_tab );
            if ( ! dir.empty() ) {
                Lister *l2;
                ListerMode *lm1;
                l2 = parentlister->getWorker()->getOtherLister( parentlister );
                lm1 = l2->getActiveMode();
                if ( lm1 != NULL ) {
                    std::list< RefCount< ArgClass > > args;

                    args.push_back( new StringArg( dir ) );
                    lm1->runCommand( "enter_dir", args );
                }
            }
        } else if ( it1->second.type == tab_popup_table_t::TAB_TO_OTHER_SIDE_AS_NEW ) {
            std::string dir = getDirectoryOfTab( m_current_popup_settings.m_tab );
            if ( ! dir.empty() ) {
                Lister *l2;
                ListerMode *lm1;
                l2 = parentlister->getWorker()->getOtherLister( parentlister );
                lm1 = l2->getActiveMode();
                if ( lm1 != NULL ) {
                    lm1->runCommand( "new_tab" );
                    std::list< RefCount< ArgClass > > args;

                    args.push_back( new StringArg( dir ) );
                    lm1->runCommand( "enter_dir", args );
                }
            }
        }
    }
}

std::string NormalMode::getDirectoryOfTab( int tab ) const
{
    if ( tab < 0 || tab >= m_tab_entries.getNrOfEntries() ) return "";
    return m_tab_entries.getEntry( tab );
}

void NormalMode::enter_dir( const std::list< RefCount< ArgClass > > &args )
{
    if ( ! args.empty() ) {
        RefCount< ArgClass > a1 = args.front();
        if ( a1.getVal() != NULL ) {
            StringArg *a2 = dynamic_cast< StringArg *>( a1.getVal() );
            if ( a2 != NULL ) {
                enterDir( a2->getValue().c_str() );
            }
        }
    }
}

void NormalMode::showSearchModeCompletion( const std::string &str )
{
    if ( ce == NULL ) return;
    if ( lv == NULL ) return;

    int nr_of_entries = ce->getNrOfFiles( 0 ) + ce->getNrOfDirs( 0 ) + 1;

    if ( nr_of_entries > 1000 ) return;

    std::list< std::string > strings;
    for ( Verzeichnis::verz_it it1 = ce->begin();
          it1 != ce->end();
          it1++ ) {
        FileEntry *fe = *it1;
        if ( fe->use == true && strcmp( fe->name, ".." ) != 0 ) {
            strings.push_back( fe->name );
        }
    }

    std::string string_filter;

    if ( m_filtered_search_infix_search == true ) {
        string_filter = "*";
    } else {
        string_filter = "";
    }

    string_filter += str;

    RefCount< StringMatcherFNMatch > matcher = new StringMatcherFNMatch( _entrysearch_matcher );
    StringCompletion sc( matcher );
    
    std::string completion = sc.findCompletion( string_filter, strings );
    
    std::string newtext = getSearchModeString();

    if ( ! completion.empty() ) {
        newtext += " [";
        newtext += completion;
        newtext += "]";
    }

    sg->setText( newtext.c_str() );

    m_last_completion = completion;
}

std::string NormalMode::getSearchModeString()
{
    if ( searchmodeon ) {
        std::string newtext = sg->getText();
        newtext.resize( sg->getCursor() );
        return newtext;
    } else {
        return "";
    }
}

void NormalMode::resetToLastSearchString()
{
    if ( searchmodeon && lastsearch != NULL ) {
        std::string newtext = lastsearch;

        if ( ! m_last_completion.empty() ) {
            newtext += " [";
            newtext += m_last_completion;
            newtext += "]";
        }

        sg->setText( newtext.c_str() );
        sg->setCursor( strlen( lastsearch ) );
    }
}

void NormalMode::setFlexibleMatchingMode( bool nv )
{
    m_flexible_matching_enabled = nv;
}

bool NormalMode::getFlexibleMatchingMode() const
{
    return m_flexible_matching_enabled;
}

std::string NormalMode::cleanFlexibleMatchString( const std::string &str )
{
    std::string res;

    for ( const char *cstr = str.c_str();
          *cstr != '\0'; ) {
        if ( *cstr != '*' ) {
            size_t l = UTF8::getLenOfCharacter( cstr );

            if ( l < 1 ) break;

            while ( l > 0 ) {
                res += *cstr++;
                l--;
            }
        } else {
            cstr++;
        }
    }

    return res;
}

/**
 * this function will prepare the elements use to run the action
 * it will put all selected entries into the actionmessage
 */

int NormalMode::prepareContext( ActionMessage &amsg )
{
    RefCount< NWC::VirtualDir > vd = new NWC::VirtualDir( "context", false, false, false );
    std::list<NM_specialsourceInt*> tlist;
    std::string active_fullname;

    // first get the active entry
    int res = getSelFiles( &tlist, NM_GETFILES_ONLYACTIVE, false );
    if ( res > 0 ) {
        for ( std::list<NM_specialsourceInt*>::const_iterator iti1 = tlist.begin();
              iti1 != tlist.end();
              iti1++ ) {
            NM_specialsourceInt *ss1 = *iti1;

            vd->add( NWC::FSEntry( ss1->entry()->fullname ) );

            active_fullname = ss1->entry()->fullname;

            delete ss1;
        }
        tlist.clear();
    }

    // now also get the selected entries but skip the active entry

    res = getSelFiles( &tlist, NM_GETFILES_ONLYSELECT, false );
    if ( res > 0 ) {
        for ( std::list<NM_specialsourceInt*>::const_iterator iti1 = tlist.begin();
              iti1 != tlist.end();
              iti1++ ) {
            NM_specialsourceInt *ss1 = *iti1;

            if ( active_fullname != ss1->entry()->fullname ) {
                vd->add( NWC::FSEntry( ss1->entry()->fullname ) );
            }

            delete ss1;
        }

    }

    if ( ! vd->empty() ) {
        amsg.setEntriesToConsider( vd );
        return 1;
    }

    return 0;
}
