/* volumemanagerui.cc
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2009-2010 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "volumemanagerui.hh"
#include <aguix/aguix.h>
#include <aguix/awindow.h>
#include <aguix/text.h>
#include <aguix/fieldlistview.h>
#include <aguix/button.h>
#include <aguix/solidbutton.h>
#include <algorithm>
#include "hw_volume.hh"
#include "hw_volume_manager.hh"
#include <functional>
#include "nwc_path.hh"
#include "worker.h"
#include "pers_string_list.hh"
#include <aguix/textview.h>

VolumeManagerUI::VolumeManagerUI( AGUIX &aguix, HWVolumeManager &hwman ) : m_aguix( aguix ),
                                                                           m_hwman( hwman ),
                                                                           m_status_info( "",
                                                                                          RefCount<AWidth>( new AFontWidth( &aguix,
                                                                                                                            NULL ) ) )
{
    std::string cfgfile = Worker::getWorkerConfigDir();
#ifdef DEBUG
    cfgfile = NWC::Path::join( cfgfile, "volman_ignore_devices2" );
#else
    cfgfile = NWC::Path::join( cfgfile, "volman_ignore_devices" );
#endif
    m_hide_entries = std::auto_ptr< PersistentStringList >( new PersistentStringList( cfgfile ) );

    m_win = std::auto_ptr<AWindow>( new AWindow( &m_aguix,
                                                 0, 0,
                                                 500, 400,
                                                 0, catalog.getLocale( 871 ) ) );
    m_win->create();

    AContainer *cont0 = m_win->setContainer( new AContainer( m_win.get(), 1, 4 ), true );
    cont0->setMaxSpace( 5 );
    
    cont0->add( new Text( &m_aguix, 0, 0, catalog.getLocale( 872 ), 1 ),
                0, 0, AContainer::CO_INCW );
    m_cont2 = cont0->add( new AContainer( m_win.get(), 2, 1 ), 0, 1 );
    m_cont2->setBorderWidth( 0 );
    m_cont2->setMinSpace( 5 );
    m_cont2->setMaxSpace( 5 );
    
    m_lv = dynamic_cast<FieldListView*>( m_cont2->add( new FieldListView( &m_aguix, 0, 0, 
                                                                          200, 200, 0 ),
                                                       0, 0, AContainer::CO_MIN ) );
    m_lv->setNrOfFields( 3 );
    m_lv->setShowHeader( true );
    m_lv->setFieldText( 0, catalog.getLocale( 873 ) );
    m_lv->setFieldText( 1, catalog.getLocale( 874 ) );
    m_lv->setFieldText( 2, catalog.getLocale( 875 ) );
    m_lv->setHBarState( 2 );
    m_lv->setVBarState( 2 );
    m_lv->setAcceptFocus( true );
    m_lv->setDisplayFocus( true );
    m_lv->setGlobalFieldSpace( 5 );

    AContainer *cont4 = m_cont2->add( new AContainer( m_win.get(), 1, 4 ), 1, 0 );
    cont4->setBorderWidth( 0 );
    cont4->setMinSpace( 5 );
    cont4->setMaxSpace( 5 );
    
    m_mountb = dynamic_cast<Button*>( cont4->add( new Button( &m_aguix, 0, 0, catalog.getLocale( 885 ), 1, 0, 0 ),
                                                  0, 0, AContainer::CO_FIX ) );
    m_unmountb = dynamic_cast<Button*>( cont4->add( new Button( &m_aguix, 0, 0, catalog.getLocale( 886 ), 1, 0, 0 ),
                                                     0, 1, AContainer::CO_FIX ) );
    m_hideb = dynamic_cast<Button*>( cont4->add( new Button( &m_aguix,
                                                             0, 0,
                                                             catalog.getLocale( 887 ),
                                                             catalog.getLocale( 888 ),
                                                             1, 1,
                                                             0, 0,
                                                             0 ),
                                                 0, 2, AContainer::CO_FIX ) );
    AContainer *cont7 = cont0->add( new AContainer( m_win.get(), 2, 1 ), 0, 2 );
    cont7->setBorderWidth( 0 );
    cont7->setMinSpace( 5 );
    cont7->setMaxSpace( 5 );
    
    cont7->add( new Text( &m_aguix, 0, 0, catalog.getLocale( 876 ), 1 ),
                0, 0, AContainer::CO_FIX );
    m_status_tv = new TextView( &m_aguix, 0, 0, 100, 50, 0, "", m_status_info );
    cont7->add( m_status_tv, 1, 0, AContainer::CO_INCW );
    m_status_tv->create();
    m_status_tv->show();
    m_status_tv->setLineWrap( true );

    TextView::ColorDef tv_cd = m_status_tv->getColors();
    tv_cd.setBackground( 0 );
    tv_cd.setTextColor( 1 );
    m_status_tv->setColors( tv_cd );

    AContainer *cont10 = cont0->add( new AContainer( m_win.get(), 2, 1 ), 0, 3 );
    cont10->setBorderWidth( 0 );
    cont10->setMinSpace( 5 );
    cont10->setMaxSpace( -1 );
    
    m_okb = dynamic_cast<Button*>( cont10->add( new Button( &m_aguix, 0, 0,
                                                            catalog.getLocale( 877 ),
                                                            1, 0, 0 ),
                                                0, 0, AContainer::CO_FIX ) );
    m_closeb = dynamic_cast<Button*>( cont10->add( new Button( &m_aguix, 0, 0,
                                                               catalog.getLocale( 633 ),
                                                               1, 0, 0 ),
                                                   1, 0, AContainer::CO_FIX ) );
   
    m_win->contMaximize( true );
    m_win->setDoTabCycling( true );
}

VolumeManagerUI::~VolumeManagerUI()
{
}

int VolumeManagerUI::mainLoop()
{
    updateLV();

    maximizeWin();
    m_win->show();

    m_lv->takeFocus();

    AGMessage *msg;
    int endmode = 0;

    {
        std::string start_entry = NWC::Path::join( m_dirname,
                                                   m_basename );
        int best_hit = -1;
        std::string::size_type last_len = 0;
        for ( int row = 0;; row++ ) {
            if ( m_lv->isValidRow( row ) == false ) break;
            std::string d = m_lv->getText( row, 0 );
            if ( ! d.empty() ) {
                if ( start_entry.compare( 0, d.length(), d ) == 0 ) {
                    if ( d.length() > last_len ) {
                        best_hit = row;
                        last_len = d.length();
                    }
                }
            }
        }
        if ( best_hit >= 0 ) {
            m_lv->setActiveRow( best_hit );
            m_lv->showActive();
        } else {
            for ( int row = 0;; row++ ) {
                if ( m_lv->isValidRow( row ) == false ) break;
                std::string d = m_lv->getText( row, 2 );
                if ( ! d.empty() ) {
                    if ( start_entry.compare( 0, d.length(), d ) == 0 ) {
                        if ( d.length() > last_len ) {
                            best_hit = row;
                            last_len = d.length();
                        }
                    }
                }
            }
            if ( best_hit >= 0 ) {
                m_lv->setActiveRow( best_hit );
                m_lv->showActive();
            }
        }
    }

    for ( ; endmode == 0; ) {
        msg = m_aguix.WaitMessage( NULL );
        if ( msg != NULL ) {
            switch ( msg->type ) {
              case AG_CLOSEWINDOW:
                  endmode = -1;
                  break;
              case AG_BUTTONCLICKED:
                  if ( msg->button.button == m_okb ) {
                      endmode = 1;
                  } else if ( msg->button.button == m_closeb ) {
                      endmode = -1;
                  } else if ( msg->button.button == m_mountb ) {
                      mountRow( m_lv->getActiveRow() );
                  } else if ( msg->button.button == m_unmountb ) {
                      unmountRow( m_lv->getActiveRow() );
                  } else if ( msg->button.button == m_hideb ) {
                      if ( msg->button.state == 2 ) {
                          editHideList();
                      } else {
                          hide( m_lv->getActiveRow() );
                      }
                  }
                  break;
              case AG_KEYPRESSED:
                  if ( msg->key.key == XK_Return ) {
                      endmode = 1;
                  } else if ( msg->key.key == XK_Escape ) {
                      endmode = -1;
                  } else if ( msg->key.key == XK_m ) {
                      mountRow( m_lv->getActiveRow() );
                  } else if ( msg->key.key == XK_u ) {
                      unmountRow( m_lv->getActiveRow() );
                  }
                  break;
              case AG_FIELDLV_DOUBLECLICK:
                  if ( msg->fieldlv.lv == m_lv ) {
                      // double click in lv, actual element is unimportant here
                      endmode = 1;
                  }
                  break;
#if 0
                case AG_FIELDLV_HEADERCLICKED:
                    if ( ( msg->fieldlv.lv == m_lv ) &&
                         ( msg->fieldlv.button == Button1 ) ) {
                        // store current active entry
                        BookmarkDBEntry active_entry = getActiveEntry( current_cat );

                        // switch sort mode
                        switch ( msg->fieldlv.row ) {
                            case 0:
                                m_filtered_data->toggleSortMode( BookmarkDBFilter::SORT_BY_NAME );
                                showData( current_cat );
                                break;
                            case 2:
                                m_filtered_data->toggleSortMode( BookmarkDBFilter::SORT_BY_ALIAS );
                                showData( current_cat );
                                break;
                            case 4:
                                m_filtered_data->toggleSortMode( BookmarkDBFilter::SORT_BY_CATEGORY );
                                showData( current_cat );
                                break;
                            default:
                                break;
                        }

                        // find previously stored entry and activate it
                        std::list<BookmarkDBEntry> e = m_filtered_data->getEntries( current_cat );
                        int row = 0;
                        for ( std::list<BookmarkDBEntry>::iterator it1 = e.begin();
                              it1 != e.end();
                              it1++, row++ ) {
                            if ( *it1 == active_entry ) {
                                m_lv->setActiveRow( row );
                                break;
                            }
                        }
                    }
                    break;
#endif
            }
            m_aguix.ReplyMessage( msg );
        }
    }

    if ( endmode == 1 ) {
        int row = m_lv->getActiveRow();
        if ( m_lv->isValidRow( row ) == true ) {
            m_dir_to_enter = m_row_to_volume[row].getMountPoint();
        }
    }

    m_win->hide();
    
    return endmode;
}

std::string VolumeManagerUI::getDirectoryToEnter() const
{
    return m_dir_to_enter;
}

void VolumeManagerUI::maximizeWin()
{
    int old_w = m_lv->getWidth();
    int old_h = m_lv->getHeight();

    m_lv->maximizeX();
    m_lv->maximizeY();

    if ( m_lv->getWidth() <= old_w &&
         m_lv->getHeight() <= old_h ) {
        m_cont2->rearrange();
    } else {
        int my_w = m_lv->getWidth() + 10;
        int my_h = m_lv->getHeight() + 10;

        if ( my_w < 400 ) my_w = 400;
        if ( my_h < 300 ) my_h = 300;

        int mw = m_aguix.getRootWindowWidth() * 80 / 100;
        int mh = m_aguix.getRootWindowHeight() * 80 / 100;
        m_cont2->resize( mw, mh );
        m_cont2->rearrange();

        if ( my_w < m_lv->getWidth() ) {
            m_cont2->setMinWidth( my_w, 0, 0 );
        } else {
            m_cont2->setMinWidth( m_lv->getWidth(), 0, 0 );
        }
        if ( my_h < m_lv->getHeight() ) {
            m_cont2->setMinHeight( my_h, 0, 0 );
        } else {
            m_cont2->setMinHeight( m_lv->getHeight(), 0, 0 );
        }
        m_win->contMaximize( true );
    }
}

static bool volumeInList( const HWVolume &vol,
                          const std::list<HWVolume> &list )
{
    if ( std::find( list.begin(), list.end(), vol ) != list.end() ) return true;
    return false;
}

struct hwvolume_cmpf : public std::binary_function<HWVolume, HWVolume, bool>
{
    bool operator()( const HWVolume &lhs,
                     const HWVolume &rhs ) const
    {
        return lhs.getDevice() < rhs.getDevice();
    }
};

void VolumeManagerUI::updateLV()
{
    std::list<HWVolume> l = m_hwman.getVolumes();
    std::list<HWVolume> nl = m_hwman.getNewVolumes( false );

    int cur_row = m_lv->getActiveRow();
    std::string cur_dev, cur_smp;
    if ( m_lv->isValidRow( cur_row ) ) {
        cur_dev = m_lv->getText( cur_row, 1 );
        cur_smp = m_lv->getText( cur_row, 0 );
    }
    
    l.sort( hwvolume_cmpf() );

    m_lv->setSize( 0 );
    m_row_to_volume.clear();
    
    for ( std::list<HWVolume>::const_iterator it1 = l.begin();
          it1 != l.end();
          it1++ ) {
        if ( ! m_hide_entries->contains( it1->getDevice() ) ) {
            int row = m_lv->addRow();
            m_lv->setText( row, 0, it1->getSupposedMountPoint() );
            m_lv->setText( row, 1, it1->getDevice() );
            m_lv->setText( row, 2, it1->getMountPoint() );
            m_lv->setPreColors( row, FieldListView::PRECOLOR_ONLYACTIVE );
            m_row_to_volume[row] = *it1;

            if ( volumeInList( *it1, nl ) ) {
                m_lv->setVisMark( row, true );
            }

            if ( cur_dev.empty() == false &&
                 it1->getDevice() == cur_dev &&
                 it1->getSupposedMountPoint() == cur_smp ) {
                m_lv->setActiveRow( row );
            }
        }
    }
    m_lv->showActive();
    m_lv->redraw();
}

void VolumeManagerUI::mountRow( int row )
{
    if ( m_lv->isValidRow( row ) ) {
        HWVolume vol = m_row_to_volume[row];

        m_win->setCursor( AGUIX::WAIT_CURSOR );
        m_status_info.setContent( catalog.getLocale( 878 ) );
        m_status_tv->textStorageChanged();
        m_aguix.Flush();

        std::string error;
        
        if ( m_hwman.mount( vol, error ) == 0 ) {
            m_status_info.setContent( catalog.getLocale( 879 ) );
        } else {
            m_status_info.setContent( error );
        }
        m_status_tv->textStorageChanged();

        m_win->unsetCursor();

        updateLV();
        maximizeWin();
    }
}

void VolumeManagerUI::unmountRow( int row )
{
    if ( m_lv->isValidRow( row ) ) {
        HWVolume vol = m_row_to_volume[row];

        m_win->setCursor( AGUIX::WAIT_CURSOR );
        m_status_info.setContent( catalog.getLocale( 880 ) );
        m_status_tv->textStorageChanged();
        m_aguix.Flush();

        std::string error;

        if ( m_hwman.umount( vol, error ) == 0 ) {
            m_status_info.setContent( catalog.getLocale( 881 ) );
        } else {
            m_status_info.setContent( error );
        }
        m_status_tv->textStorageChanged();

        m_win->unsetCursor();

        updateLV();
        maximizeWin();
    }
}

void VolumeManagerUI::setCurrentDirname( const std::string &dirname )
{
    m_dirname = dirname;
}

void VolumeManagerUI::setCurrentBasename( const std::string &basename )
{
    m_basename = basename;
}

void VolumeManagerUI::hide( int row )
{
    if ( m_lv->isValidRow( row ) ) {
        HWVolume vol = m_row_to_volume[row];
        m_hide_entries->pushEntry( vol.getDevice() );
        updateLV();
    }
}

void VolumeManagerUI::editHideList()
{
    AWindow *win = new AWindow( &m_aguix,
                                0, 0,
                                500, 400,
                                0, catalog.getLocale( 882 ) );
    win->create();
    
    AContainer *cont0 = win->setContainer( new AContainer( win, 1, 4 ), true );
    cont0->setMaxSpace( 5 );
    
    cont0->add( new Text( &m_aguix, 0, 0, catalog.getLocale( 883 ), 1 ),
                0, 0, AContainer::CO_INCW );
    
    FieldListView *lv = dynamic_cast<FieldListView*>( cont0->add( new FieldListView( &m_aguix, 0, 0, 
                                                                                     200, 200, 0 ),
                                                                  0, 1, AContainer::CO_MIN ) );
    lv->setHBarState( 2 );
    lv->setVBarState( 2 );
    lv->setAcceptFocus( true );
    lv->setDisplayFocus( true );

    {
        std::list< std::string > list = m_hide_entries->getList();
        for ( std::list< std::string >::const_iterator it1 = list.begin();
              it1 != list.end();
              it1++ ) {
            int row = lv->addRow();
            lv->setText( row, 0, *it1 );
        }
    }

    Button *remove_b = dynamic_cast<Button*>( cont0->add( new Button( &m_aguix, 0, 0, catalog.getLocale( 884 ), 1, 0, 0 ),
                                                          0, 2, AContainer::CO_FIX ) );
    AContainer *cont1 = cont0->add( new AContainer( win, 2, 1 ), 0, 3 );
    cont1->setBorderWidth( 0 );
    cont1->setMinSpace( 5 );
    cont1->setMaxSpace( -1 );
    
    Button *okb = dynamic_cast<Button*>( cont1->add( new Button( &m_aguix, 0, 0,
                                                                 catalog.getLocale( 11 ),
                                                                 1, 0, 0 ),
                                                     0, 0, AContainer::CO_FIX ) );
    Button *cancelb = dynamic_cast<Button*>( cont1->add( new Button( &m_aguix, 0, 0,
                                                                     catalog.getLocale( 8 ),
                                                                     1, 0, 0 ),
                                                         1, 0, AContainer::CO_FIX ) );
    
    win->contMaximize( true );
    win->setDoTabCycling( true );
    win->show();

    AGMessage *msg;
    int endmode = 0;

    for ( ; endmode == 0; ) {
        msg = m_aguix.WaitMessage( NULL );
        if ( msg != NULL ) {
            switch ( msg->type ) {
                case AG_CLOSEWINDOW:
                    endmode = -1;
                    break;
                case AG_BUTTONCLICKED:
                    if ( msg->button.button == okb ) {
                        endmode = 1;
                    } else if ( msg->button.button == cancelb ) {
                        endmode = -1;
                    } else if ( msg->button.button == remove_b ) {
                        for ( int row = 0; row < lv->getElements(); ) {
                            if ( lv->getSelect( row ) == true ) {
                                lv->deleteRow( row );
                            } else {
                                row++;
                            }
                        }
                        lv->redraw();
                    }
                    break;
                default:
                    break;
            }
            m_aguix.ReplyMessage( msg );
        }
    }
    
    if ( endmode == 1 ) {
        std::set< std::string > devices_to_use;

        for ( int row = 0; row < lv->getElements(); row++ ) {
            devices_to_use.insert( lv->getText( row, 0 ) );
        }

        std::list< std::string > list = m_hide_entries->getList();
        for ( std::list< std::string >::const_iterator it1 = list.begin();
              it1 != list.end();
              it1++ ) {
            if ( devices_to_use.count( *it1 ) == 0 ) {
                m_hide_entries->removeEntry( *it1 );
            }
        }
    }

    delete win;

    updateLV();
}
