/* searchop.hh
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2006 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef SEARCHOP_HH
#define SEARCHOP_HH

#include "wdefines.h"
#include "functionproto.h"
#include <string>
#include <list>
#include "searchthread.hh"
#include "nwc_virtualdir.hh"
#include <aguix/fieldlistview.h>

class SearchOp : public FunctionProto
{
public:
  SearchOp();
  ~SearchOp();
  SearchOp( const SearchOp &other );
  SearchOp &operator=( const SearchOp &other );

  SearchOp *duplicate() const;
  bool isName(const char *);
  const char *getName();

  using FunctionProto::run;
  int run( WPUContext*, ActionMessage* );
  const char *getDescription();
  bool save( Datei * );
  int configure();

  void setEditCommand( const std::string &str );
  void setShowPrevResults( bool nv );
  
  static const char *name;
protected:
  void updateState( class Text &state_text );

  typedef enum { BUTTON_CONTINUE, BUTTON_SUSPEND, BUTTON_IDLE } buttonmode_t;
  void prepareButton( class Button &startb, buttonmode_t mode );
  void prepareStartCont( bool force = false );

  bool consumeResult( SearchThread::SearchResult &result, class FieldListView &lv );
  void prepareNewSearch();

  void enterResult( ActionMessage *msg, const std::string &fullname );

  class SearchSettings
  {
  public:
    SearchSettings();
    SearchSettings( const std::string &basedir,
                    const std::string &matchname,
                    bool matchname_re,
                    bool matchname_fullpath,
                    bool matchname_case,
                    const std::string &matchcontent,
                    bool matchcontent_case,
                    bool follow_symlinks,
                    bool search_vfs );
    ~SearchSettings();

    std::string getBaseDir() const;
    std::string getMatchName() const;
    bool getMatchNameRE() const;
    bool getMatchNameFullPath() const;
    bool getMatchNameCase() const;
    std::string getMatchContent() const;
    bool getMatchContentCase() const;
    int getActiveRow() const;
    bool getFollowSymlinks() const;
    bool getSearchVFS() const;

    void setActiveRow( int nv );
  private:
    std::string _basedir;
    std::string _matchname;
    bool _matchname_re;
    bool _matchname_fullpath;
    bool _matchname_case;
    std::string _matchcontent;
    bool _matchcontent_case;
    int _active_row;
    bool _follow_symlinks;
    bool _search_vfs;
  };

  class ResultStore
  {
  public:
    ResultStore( const std::string &searchname,
                 bool follow_symlinks = false,
                 bool search_vfs = false );
    ~ResultStore();
    NWC::VirtualDir *getDir();
    NWC::VirtualDir *getClonedDir() const;
    std::list<SearchThread::SearchResult> *getRes();

    void setSearchSettings( const SearchSettings &sets );
    SearchSettings getSearchSettings() const;
    void setActiveRow( int row );

    void removeEntries( const std::list<std::string> &entry_list );
  private:
    NWC::VirtualDir *_dir;
    std::list<SearchThread::SearchResult> *_res;

    SearchSettings _settings;

    void removeEntry( const std::string &entry );
  };
 
  void clearStoredResults();
  void buildLV();
  void buildStoreLV();
  ResultStore *getResultStore( int pos, const SearchSettings *sets = NULL );

  void stopSearch();
  void gatherResults( int get_number_of_results = -1 );
  void startSearch();
  void newSearch();
  SearchSettings getCurrentSearchSettings();
  void storeSearchSettings();
  void restoreSearchSettings();
  void setCurrentResultStore( int pos, const SearchSettings *sets = NULL );
  void storeActiveRow( int row );

  void cleanupPtrs();

  void removeSelectedResults();
  std::auto_ptr<std::list<std::string> > getSelectedResults();
  SearchThread::SearchResult getResultForRow( int row );

  void viewResult();
  void editResult();
  void editFile( const std::string &filename, int line_number );

  void initColorDefs();

  // Infos to save
  std::string _edit_command;
  bool _show_prev_results;

  // temp variables
  class AContainer *_start_ac;
  class Text *_startdir_text;
  class StringGadget *_startdir_sg;
  class FieldListView *_storelv;
  bool _last_startcont_state;
  FieldListView::ColorDef filecol, dircol, resultcol;
  
  std::string _last_result;
  ResultStore *_rst;
  FieldListView *_resultlv;
  SearchThread *_sth;
  class StringGadget *_matchname_sg;
  class ChooseButton *_matchname_regexp_cb;
  class ChooseButton *_matchname_fullpath_cb;
  class ChooseButton *_matchname_case_cb;
  class StringGadget *_matchcontent_sg;
  class ChooseButton *_matchcontent_case_cb;
  class ChooseButton *_follow_symlinks_cb;
  class ChooseButton *_search_vfs_cb;

  std::string _initial_start_dir;
    
  bool _search_started;  ///< flag indicates whether search was started but it's not the accurate status of the search thread

  // result store shared by all searchop instances
  // public methods need to acquire lock, protected methods consider this structued as locked
  // THIS IS NOT REALLY THREAD-SAFE! multiple SearchOp instances can use
  // this structure but they are not allowed to run in to threads currently
  // (which is not possible at the moment)
  // currently it's secured by a lock so multiple instaces block each other
  // but because of the pointer in the list it's not easily shareable
  static std::list<ResultStore*> _stored_results;
  static int _stored_results_ctr;
  static MutEx _stored_results_lock;
};

#endif
