/*---------------------------------------------------------------------------*\

    This program analyses audio data for DTMF tones.


         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2007 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License version 2.1 as published by the Free Software Foundation.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
         MA  02110-1301  USA

\*---------------------------------------------------------------------------*/

#include "vpbapi.h"
#include "dtmf.h"

#include <cstdlib>
#include <cstring>


static int arg_exists(int argc, char *argv[], const char *arg)
{ //{{{
	for(int i = 0; i < argc; ++i)
		if(strcmp(argv[i],arg) == 0) return i;

	return 0;
} //}}}

static void usage(const char *argv0)
{ //{{{
	printf("usage: %s [--block size] file\n\n", argv0);
	printf("  --block size  Granularity of analysis.  Default (and max) is 32768 (2sec)\n");
} //}}}

int main(int argc, char *argv[])
{ //{{{
	if(argc < 2) {
		usage(argv[0]);
		exit(EXIT_FAILURE);
	}

	size_t          NBUF = 32768;
	int             arg;

	if((arg = arg_exists(argc,argv,"--block")) != 0) {
		// We must ensure NBUF is always a multiple of the sample size.
		NBUF = atoi(argv[arg+1]) & ~1;
		if( NBUF > 32768 || NBUF < 2 ) {
			fprintf(stderr, "Invalid --block size %u bytes ...\n\n", (unsigned)NBUF);
			exit(EXIT_FAILURE);
		}
	}

	int16_t         buf[NBUF];
	char            keys[16];
	int             keydown  = 0;
	unsigned int    start    = 0;
	WFILE          *wav;

	vpb_wave_open_read(&wav, argv[argc - 1]);

	size_t          size     = vpb_wave_get_size(wav);
	void           *dtmf_dec = dtmf_init();

	printf("Reading %u bytes in %u byte blocks ...\n", (unsigned)size, (unsigned)NBUF);
	while( size > 1 ) {
		size_t n = size > NBUF ? NBUF : size;

		n = vpb_wave_read(wav, (char*)buf, n );
		size -= n;

		int digits_ret = dtmf_decoder(dtmf_dec, keys, sizeof(keys),
							buf, n / sizeof(int16_t),
							&keydown);

		printf("%#7x %6.2fs: Detected ", start, start / 16384.0);
		start += n;

		if( digits_ret == 0 ) printf("no digits\n");
		else {
			for( int i = 0; i < digits_ret; ++i ) printf("%c ", keys[i]);
			printf("\n");
		}
	}

	vpb_wave_close_read(wav);
	dtmf_close(dtmf_dec);

	return EXIT_SUCCESS;
} //}}}

