
// created 06.2003 by Stefan Kleine Stegemann
// 
// licensed under GPL

#include "DocumentWindow.h"
#include "DocumentWindowController.h"
#include "ViewPDF.h"

#include <Foundation/NSString.h>
#include <Foundation/NSException.h>
#include <Foundation/NSArray.h>
#include <Foundation/NSEnumerator.h>
#include <Foundation/NSUserDefaults.h>
#include <AppKit/NSButton.h>
#include <AppKit/NSEvent.h>
#include <AppKit/NSImage.h>
#include <AppKit/NSTextField.h>
#include <AppKit/NSFont.h>
#include <AppKit/NSProgressIndicator.h>


/*
 * Non-Public methods of the DocumentView class.
 */
@interface DocumentWindow(Private)
- (void) createContentView;
- (void) createDocumentView;
- (void) createToolbar;
- (void) rearrangeToolbarButtons: (BOOL)displayPageNav;
- (NSButton*) makeToolbarButtonWithTitle: (NSString*)title
                                   image: (NSString*)imageName;
- (void) makeToolbarProgressBar;
- (void) addToToolbar: (id)aView
           atPosition: (int)pos;
@end


/*
 * A window that displays a ViewableDocument and provides
 * some controls to control the view (eg. zoom controls).
 * A DocumentWindow is controlled by an instance of
 * DocumentWindowController. 
 */
@implementation DocumentWindow

/*
 * Designated initializer.
 */
- (id) init
{
   NSRect  contentRect;

   contentRect = NSMakeRect(100, 100, 400, 200);

   if (self = [super initWithContentRect: contentRect
                     styleMask: (NSTitledWindowMask
                                 | NSClosableWindowMask
                                 | NSMiniaturizableWindowMask
                                 | NSResizableWindowMask)
                     backing: NSBackingStoreBuffered
                     defer: NO])
   {
      myController = nil;

      [self setTitle: @"ViewPDF Window"];
      [self createContentView];
      [self setMinSize: NSMakeSize(100, TOOLBAR_HEIGHT + 100)];
      
      toolbarVisible = ![[NSUserDefaults standardUserDefaults] 
                           boolForKey: PREFS_TOOLBAR_VISIBLE];
      [self createToolbar];
      [self setToolbarVisible: !toolbarVisible];

      [self createDocumentView];

      [self setDisplayPageNavigation: NO];
   }

   return self;
}


/*
 * Set the windows controller.
 */
- (void) setController: (DocumentWindowController*)controller
{
   myController = controller;
   // myController needs not to be retained because this
   // is done by the document-based application model

   [[[documentView scrollView] firstPageButton] setAction: @selector(firstPage:)];
   [[[documentView scrollView] firstPageButton] setTarget: myController];
   [[[documentView scrollView] prevPageButton] setAction: @selector(previousPage:)];
   [[[documentView scrollView] prevPageButton] setTarget: myController];
   [[[documentView scrollView] nextPageButton] setAction: @selector(nextPage:)];
   [[[documentView scrollView] nextPageButton] setTarget: myController];   
   [[[documentView scrollView] lastPageButton] setAction: @selector(lastPage:)];
   [[[documentView scrollView] lastPageButton] setTarget: myController];
   [[[documentView scrollView] pageField] setDelegate: myController];

   [tbPrevPage setAction: @selector(previousPage:)];
   [tbPrevPage setTarget: myController];
   [tbNextPage setAction: @selector(nextPage:)];
   [tbNextPage setTarget: myController];
   [tbZoomIn setAction: @selector(zoomIn:)];
   [tbZoomIn setTarget: myController];
   [tbZoomOut setAction: @selector(zoomOut:)];
   [tbZoomOut setTarget: myController];
   [tbZoomToRealSize setAction: @selector(zoomToRealSize:)];
   [tbZoomToRealSize setTarget: myController];   
   [tbSizeToFit setAction: @selector(sizePageToFit:)];
   [tbSizeToFit setTarget: myController];
   [tbSizeToFitWidth setAction: @selector(sizePageToFitWidth:)];
   [tbSizeToFitWidth setTarget: myController];   
}


/*
 * Invoked by the controller when the controller has been
 * initialized with this window.
 */
- (void) windowInitializedWithController
{
   return;
}



- (void) dealloc
{
   NSLog(@"dealloc DocumentWindow, retain count is %d", [self retainCount]);

   [[self contentView] release];
   [documentView release];
   [toolbar release];

   [tbNextPage release];
   [tbPrevPage release];
   [tbZoomIn release];
   [tbZoomOut release];
   [tbZoomToRealSize release];
   [tbSizeToFit release];
   [tbSizeToFitWidth release];
   [tbProgressBar release];

   [tbElements release];

   [super dealloc];
}


/*
 * Catch control-keys.
 */
- (void) keyDown: (NSEvent*)theEvent
{
   NSString* chars = [theEvent characters];
   unichar   firstChar = 0;
   BOOL      shiftKey = [theEvent modifierFlags] & NSShiftKeyMask;
   
   if ([chars length] > 0)
   {
      firstChar = [chars characterAtIndex: 0];
      switch (firstChar)
      {
         case NSLeftArrowFunctionKey:
            if (!shiftKey)
            {
               [myController scrollHorizontal: self direction: SCROLL_LEFT];
            }
            else
            {
               [myController previousPage: self];
            }
            break;

         case NSRightArrowFunctionKey:
            if (!shiftKey)
            {
               [myController scrollHorizontal: self direction: SCROLL_RIGHT];
            }
            else
            {
               [myController nextPage: self];
            }
            break;

         case NSUpArrowFunctionKey:
            [myController scrollVertical: self direction: SCROLL_UP];
            break;

         case NSDownArrowFunctionKey:
            [myController scrollVertical: self direction: SCROLL_DOWN];
            break;

         case NSPageUpFunctionKey:
            [myController scrollPage: self direction: SCROLL_UP];
            break;

         case NSPageDownFunctionKey:
            [myController scrollPage: self direction: SCROLL_DOWN];
            break;
      }
   }
}


- (PDFContentView*) documentView
{
   return documentView;
}


- (NSProgressIndicator*) toolbarProgressBar
{
   return tbProgressBar;
}


/*
 * Sets whether controls to navigate through pages are
 * displayed or not. These controls are disabled by default.
 */
- (void) setDisplayPageNavigation: (BOOL)displayPageNav
{
   [[[self documentView] scrollView] setDisplayPageControls: displayPageNav];
   [self rearrangeToolbarButtons: displayPageNav];
}


/*
 * Set whether the toolbar is displayed or not.
 */
- (void) setToolbarVisible: (BOOL)visible
{
   NSRect contentRect = [[self contentView] frame];
   float  winHeight = contentRect.size.height;
   NSSize docViewSize = [[self documentView] frame].size;

   if (visible == [self toolbarIsVisible])
   {
      return;
   }

   toolbarVisible = visible;

   if (visible)
   {
      NSRect toolbarFrame =
         NSMakeRect(0,
                    (contentRect.size.height - TOOLBAR_HEIGHT + 1),
                    contentRect.size.width,
                    TOOLBAR_HEIGHT);

      [toolbar setFrame: toolbarFrame];
      [[self contentView] addSubview: toolbar];

      winHeight += TOOLBAR_HEIGHT;
      docViewSize.height = winHeight - TOOLBAR_HEIGHT;
   }
   else
   {
      [toolbar removeFromSuperview];
      
      winHeight -= TOOLBAR_HEIGHT;
      docViewSize.height = winHeight;
   }
   
   [self setContentSize: NSMakeSize(contentRect.size.width,
                                    winHeight)];

   [[self documentView] setFrameSize: docViewSize];

   [[self contentView] setNeedsDisplay: YES];
}


/*
 * Returns YES if the toolbar is displayed, otherwise NO.
 */
- (BOOL) toolbarIsVisible
{
   return toolbarVisible;
}

@end



@implementation DocumentWindow(Private)

/*
 * Create the image view and add it to the window.
 */
- (void) createDocumentView
{
   NSRect frame = [[self contentView] frame];

   if ([self toolbarIsVisible])
   {
      frame.size.height -= TOOLBAR_HEIGHT;
   }
   
   documentView = [[PDFContentView alloc] initWithFrame: frame];

   [[self contentView] addSubview: documentView];
}


/*
 * Create the window's toolbar. As long as NSToolbar is not
 * fully available on GNUstep, the toolbar will be a view
 * with some buttons.
 */
- (void) createToolbar
{
   toolbar = [[NSView alloc] initWithFrame: NSMakeRect(0, 0, 10, 10)];
   // correct size is set int setToolbarVisible
   [toolbar setAutoresizingMask: (NSViewMinYMargin | NSViewWidthSizable)];

   tbElements = [[NSMutableArray alloc] initWithCapacity: 0];

   tbPrevPage = [self makeToolbarButtonWithTitle: @"Previous"
                      image: @"tb_prev_page.tiff"];
   
   tbNextPage = [self makeToolbarButtonWithTitle: @"Next"
                      image: @"tb_next_page.tiff"];

   tbZoomIn = [self makeToolbarButtonWithTitle: @"Bigger"
                    image: @"zoom_in.tiff"];
   
   tbZoomOut = [self makeToolbarButtonWithTitle: @"Smaller"
                     image: @"zoom_out.tiff"];

   tbZoomToRealSize = [self makeToolbarButtonWithTitle: @"1:1"
                            image: @"zoom_11.tiff"];

   tbSizeToFit = [self makeToolbarButtonWithTitle: @"Fit Window"
                       image: @"fit_page.tiff"];

   tbSizeToFitWidth = [self makeToolbarButtonWithTitle: @"Fit Width"
                            image: @"fit_width.tiff"];

   [self makeToolbarProgressBar];

   [self rearrangeToolbarButtons: NO];
}


/*
 * Create the windows content view.
 */
- (void) createContentView
{
   NSView* contentView;
   NSRect contentRect;
   NSRect contentViewFrame;

   contentRect = [[self contentView] frame];
   contentViewFrame = NSMakeRect(0, 0, 
                                 contentRect.size.width,
                                 contentRect.size.height);
   contentView = [[NSView alloc] initWithFrame: contentViewFrame];
   [contentView setAutoresizesSubviews: YES];
   [contentView setAutoresizingMask: (NSViewWidthSizable |
                                      NSViewHeightSizable)];
   
   [self setContentView: contentView];
}


/*
 * Create the buttons displayed on the toolbar. Buttons
 * for page navigation are shown depending on the value
 * of displayPageNav.
 */
- (void) rearrangeToolbarButtons: (BOOL)displayPageNav
{
   NSButton*     aButton;
   NSEnumerator* e;
   int           firstNonNavButton;

   e = [tbElements objectEnumerator];
   while (aButton = [e nextObject])
   {
      [aButton removeFromSuperview];
   }

   
   if (displayPageNav)
   {
      [self addToToolbar: tbPrevPage atPosition: 1];
      [self addToToolbar: tbNextPage atPosition: 2];
      firstNonNavButton = 3;
   }
   else
   {
      firstNonNavButton = 1;
   }
   
   [self addToToolbar: tbZoomIn atPosition: firstNonNavButton];
   [self addToToolbar: tbZoomOut atPosition: firstNonNavButton + 1];
   [self addToToolbar: tbZoomToRealSize atPosition: firstNonNavButton + 2];
   [self addToToolbar: tbSizeToFit atPosition: firstNonNavButton + 3];
   [self addToToolbar: tbSizeToFitWidth atPosition: firstNonNavButton + 4];

   //[self addToToolbar: tbProgressBar atPosition: firstNonNavButton + 5];

   [toolbar setNeedsDisplay: YES];
}


/*
 * Create a button that is shown on the toolbar. A toolbar
 * button will have an image and a text that is displayed
 * below the image.
 */
- (NSButton*) makeToolbarButtonWithTitle: (NSString*)title
                                   image: (NSString*)imageName
{
   NSRect frame;
   NSButton* theButton;

   frame = NSMakeRect(1,
                      1,
                      TOOLBAR_HEIGHT - 4,
                      TOOLBAR_HEIGHT - 4);

   theButton = [[NSButton alloc] initWithFrame: frame];
   [theButton setFont: [NSFont systemFontOfSize: 8.0]];
   [theButton setButtonType: NSMomentaryLight];
   [theButton setTitle: title];
   [theButton setImagePosition: NSImageAbove];
   [theButton setImage: [NSImage imageNamed: imageName]];
   [theButton setBordered: YES];

   return theButton;
}


/*
 * Create the progress bar that can be displayed on the toolbar.
 */
- (void) makeToolbarProgressBar
{
   NSRect frame = NSMakeRect(1,
                             (TOOLBAR_HEIGHT / 2) - (TOOLBAR_HEIGHT / 8),
                             TOOLBAR_HEIGHT,
                             TOOLBAR_HEIGHT / 4);

   tbProgressBar = [[NSProgressIndicator alloc] initWithFrame: frame];

   [tbProgressBar setBezeled: YES];
   [tbProgressBar setIndeterminate: NO];
   [tbProgressBar setVertical: NO];
   [tbProgressBar setMinValue: 0];
}


/*
 * Add a button to the toolbar at the specified position.
 */
- (void) addToToolbar: (id)aView
           atPosition: (int)pos
{
   NSAssert(aView != nil, @"toolbar element is nil");

   NSRect frame = [aView frame];
   [aView setFrameOrigin: NSMakePoint(8.0 + (TOOLBAR_HEIGHT * (pos - 1)),
                                      frame.origin.y)];

   [toolbar addSubview: aView];
   [tbElements addObject: aView];
}

@end
