/*
 *
 *   Copyright (C) 2005-2010 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

#include <stdlib.h>
#include <string.h>
#include <UgUri.h>
#include <UgUtils.h>
#include <UgString.h>
#include <UgRegistry.h>
#include <UgCategory.h>
#include <UgData-download.h>
#include <UgPlugin.h>

const	UgDataClass*	UgCategoryClass = NULL;
const	UgDataClass*	UgRelationClass = NULL;

// ----------------------------------------------------------------------------
// UgCategory
void	ug_category_init (UgCategory* category)
{
	category->defaults = ug_dataset_new ();
	category->limit.active   = 3;
	category->limit.finished = 300;
	category->limit.recycled = 300;
}

void	ug_category_finalize (UgCategory* category)
{
	g_free (category->name);
	if (category->defaults)
		ug_dataset_unref (category->defaults);
}

void	ug_category_assign (UgCategory* category, UgCategory* src)
{
	ug_str_set (&category->name, src->name, -1);
	category->limit.active = src->limit.active;
	category->limit.finished = src->limit.finished;
	category->limit.recycled = src->limit.recycled;
	ug_data_assign (category->defaults, src->defaults);
}

gboolean	ug_category_refresh (UgCategory* category, UgDataset* dataset)
{
	UgRelation*		relation;
	UgMessage*		msg_list;
	UgMessage*		msg;
	UgState			plugin_state;
	union {
		UgDataCommon*	common;
		UgDataHttp*		http;
	} data;

	// if not active, return FALSE.
	relation = UG_DATASET_RELATION (dataset);
	if (relation->plugin == NULL)
		return FALSE;
	// dispatch messages
	msg_list = ug_plugin_pop_all (relation->plugin);
	for (msg = msg_list;  msg;  msg = msg->next) {
		// call watch functions
		if (relation->watch.func) {
			if (relation->watch.func (category, dataset, msg, relation->watch.data) == TRUE)
				continue;
		}
		if (category->watch.func) {
			if (category->watch.func (category, dataset, msg, category->watch.data) == TRUE)
				continue;
		}

		// default message handler
		switch (msg->type) {
		case UG_MESSAGE_STATE:
			if (msg->data.v_int != UG_STATE_ACTIVE)
				relation->hints &= ~UG_HINT_ACTIVE;
			break;

		case UG_MESSAGE_PROGRESS:
			ug_plugin_get (relation->plugin, UG_DATA_TYPE_INSTANCE,
						ug_dataset_realloc (dataset, UgProgressClass, 0));
			break;

		case UG_MESSAGE_ERROR:
			relation->hints |= UG_HINT_ERROR;
			break;

		case UG_MESSAGE_INFO:
			switch (msg->code) {
			case UG_MESSAGE_INFO_RETRY:
				UG_DATASET_COMMON (dataset)->retry_count++;
				break;

			case UG_MESSAGE_INFO_COMPLETE:
				relation->hints |= UG_HINT_COMPLETED;
				break;

			case UG_MESSAGE_INFO_FINISH:
				relation->hints |= UG_HINT_FINISHED;
				relation->hints &= ~UG_HINT_ERROR;
				break;

			case UG_MESSAGE_INFO_RESUMABLE:
				relation->hints &= ~UG_HINT_RESUMED_FIELD;
				relation->hints |= UG_HINT_RESUMABLE;
				break;

			case UG_MESSAGE_INFO_UNRESUMABLE:
				relation->hints &= ~UG_HINT_RESUMED_FIELD;
				relation->hints |= UG_HINT_UNRESUMABLE;
				break;

			default:
				break;
			}
			// End of switch (msg->code)
			break;
		// End of UG_MESSAGE_INFO

		case UG_MESSAGE_DATA:
			switch (msg->code) {
			case UG_MESSAGE_DATA_FILE_CHANGED:
				if (msg->data.v_string) {
					data.common = UG_DATASET_COMMON (dataset);
					ug_str_set (&data.common->file, msg->data.v_string, -1);
				}
				break;

			case UG_MESSAGE_DATA_URL_CHANGED:
			// HTTP message
			case UG_MESSAGE_DATA_HTTP_LOCATION:		// redirection
				if (msg->data.v_string) {
					data.common = UG_DATASET_COMMON (dataset);
					ug_str_set (&data.common->url, msg->data.v_string, -1);
					data.http = ug_dataset_realloc (dataset, UgDataHttpClass, 0);
					data.http->redirection_count++;
				}
				break;

			default:
				break;
			}
			// End of switch (msg->code)
			break;
		// End of UG_MESSAGE_DATA

		default:
			break;
		}
		// End of switch (msg->type)
	}
	// free message list
	ug_data_list_free (msg_list);

	// if not active, return FALSE.
	ug_plugin_get_state (relation->plugin, &plugin_state);
	if (plugin_state != UG_STATE_ACTIVE)
		return FALSE;
	return TRUE;
}

gboolean	ug_category_start (UgCategory* category, UgDataset* dataset)
{
	UgRelation*		relation;

	relation = UG_DATASET_RELATION (dataset);
	// create plug-in
	if (relation->plugin == NULL) {
		relation->plugin = ug_plugin_new_by_data (dataset);
		if (relation->plugin == NULL) {
			relation->hints |=  UG_HINT_ERROR;
			return FALSE;
		}
	}
	// change plug-in state
	ug_plugin_set_state (relation->plugin, UG_STATE_ACTIVE);
	// set relation->hints
	relation->hints &= ~UG_HINT_PAUSED;
	relation->hints |=  UG_HINT_ACTIVE;
	// reset
//	UG_DATASET_COMMON (dataset)->retry_count = 0;
//	http = ug_dataset_get (dataset, UgDataHttpClass, 0);
//	http->redirection_count = 0;
	return TRUE;
}

void	ug_category_stop (UgCategory* category, UgDataset* dataset)
{
	UgRelation*		relation;

	relation = UG_DATASET_RELATION (dataset);
	if (relation->plugin) {
		ug_plugin_set_state (relation->plugin, UG_STATE_NULL);
		ug_plugin_unref (relation->plugin);
		relation->plugin = NULL;
	}
	// set relation->hints
	relation->hints &= ~UG_HINT_ACTIVE;
//	relation->hints |=  UG_HINT_PAUSED;
}

// ----------------------------------------------------------------------------
// Category.indices load/save
//
static void ug_int_list_start_element (GMarkupParseContext*	context,
                                       const gchar*			element_name,
                                       const gchar**		attr_names,
                                       const gchar**		attr_values,
                                       GList**				list,
                                       GError**				error)
{
	guint	index;
	int		value;

	for (index=0; attr_names[index]; index++) {
		if (strcmp (attr_names[index], "value") != 0)
			continue;
		value = atoi (attr_values[index]);
		*list = g_list_prepend (*list, GINT_TO_POINTER (value));
	}

	g_markup_parse_context_push (context, &ug_markup_skip_parser, NULL);
}

static GMarkupParser	ug_int_list_parser =
{
	(gpointer) ug_int_list_start_element,
	(gpointer) g_markup_parse_context_pop,
	NULL,
	NULL,
	NULL
};

void	ug_int_list_in_markup (GList** list, GMarkupParseContext* context)
{
	g_markup_parse_context_push (context, &ug_int_list_parser, list);
}

void	ug_int_list_to_markup (GList** list, UgMarkup* markup)
{
	GList*		link;
	guint		value;

	for (link = g_list_last (*list);  link;  link = link->prev) {
		value = GPOINTER_TO_INT (link->data);
		ug_markup_write_element_start (markup, "int value='%d'", value);
		ug_markup_write_element_end   (markup, "int");
	}
}


// ----------------------------------------------------------------------------
// CategoryList load/save
//
static void ug_category_data_start_element (GMarkupParseContext*	context,
                                            const gchar*		element_name,
                                            const gchar**		attr_names,
                                            const gchar**		attr_values,
                                            GList**				list,
                                            GError**			error)
{
	UgCategory*		category;

	if (strcmp (element_name, "category") != 0) {
		g_markup_parse_context_push (context, &ug_markup_skip_parser, NULL);
		return;
	}
	if (UgCategoryClass == NULL)
		UgCategoryClass = ug_data_class_find (UG_CATEGORY_CLASS_NAME);

	// user must register data class of UgCategory.
	category = ug_data_new (UgCategoryClass);
	*list = g_list_prepend (*list, category);
	g_markup_parse_context_push (context, &ug_data_parser, category);
}

static GMarkupParser	ug_category_data_parser =
{
	(gpointer) ug_category_data_start_element,
	(gpointer) g_markup_parse_context_pop,
	NULL,
	NULL,
	NULL
};

static void ug_category_list_start_element (GMarkupParseContext*	context,
                                            const gchar*		element_name,
                                            const gchar**		attr_names,
                                            const gchar**		attr_values,
                                            GList**				list,
                                            GError**			error)
{
	guint	index;

	if (strcmp (element_name, "UgetCategoryList") == 0) {
		for (index=0; attr_names[index]; index++) {
			if (strcmp (attr_names[index], "version") != 0)
				continue;
			if (strcmp (attr_values[index], "1") == 0) {
				g_markup_parse_context_push (context, &ug_category_data_parser, list);
				return;
			}
			// others...
			break;
		}
	}

	g_markup_parse_context_push (context, &ug_markup_skip_parser, NULL);
}

static GMarkupParser	ug_category_list_parser =
{
	(gpointer) ug_category_list_start_element,
	(gpointer) g_markup_parse_context_pop,
	NULL,
	NULL,
	NULL
};

GList*	ug_category_list_load (const gchar* file)
{
	GList*		category_list;

	category_list = NULL;
	ug_markup_parse (file, &ug_category_list_parser, &category_list);
	return category_list;
}

gboolean	ug_category_list_save (GList* list, const gchar* file, UgCategoryGetList get_list)
{
	UgCategory*	category;
	UgMarkup*	markup;
	GList*		link;
	guint		index;

	markup = ug_markup_new ();
	if (ug_markup_write_start (markup, file, TRUE) == FALSE) {
		ug_markup_free (markup);
		return FALSE;
	}

	ug_markup_write_element_start (markup, "UgetCategoryList version='1'");
	for (list = g_list_last (list);  list;  list = list->prev) {
		category = list->data;
		// create UgCategory.indices
		category->indices = get_list (category);
		for (link = category->indices;  link;  link = link->next) {
			index = UG_DATASET_RELATION ((UgDataset*) link->data)->index;
			link->data = GINT_TO_POINTER (index);
		}
		// output
		ug_markup_write_element_start (markup, "category");
		ug_data_to_markup ((UgData*) list->data, markup);
		ug_markup_write_element_end (markup, "category");
		// free UgCategory.indices
		g_list_free (category->indices);
		category->indices = NULL;
	}
	ug_markup_write_element_end (markup, "UgetCategoryList");

	ug_markup_write_end (markup);
	return	TRUE;
}

void	ug_category_list_link (GList* list, GList* download_list, UgCategoryAddFunc add_func)
{
	GPtrArray*	array;
	UgCategory*	category;
	GList*		link;
	guint		index;

	// create array from download_list
	array = g_ptr_array_sized_new (g_list_length (download_list));
	for (link = download_list;  link;  link = link->next)
		array->pdata[array->len++] = link->data;

	// link jobs in category
	for (;  list;  list = list->next) {
		category = list->data;
		// get jobs from array by index
		for (link = category->indices;  link;  link = link->next) {
			index = GPOINTER_TO_INT (link->data);
			if (index < array->len)
				add_func (category, g_ptr_array_index (array, index));
		}
		// free list
		g_list_free (category->indices);
		category->indices = NULL;
	}

	// free array
	g_ptr_array_free (array, TRUE);
}


// ----------------------------------------------------------------------------
// DownloadList load/save
//
static void ug_download_data_start_element (GMarkupParseContext*	context,
                                            const gchar*		element_name,
                                            const gchar**		attr_names,
                                            const gchar**		attr_values,
                                            GList**				list,
                                            GError**			error)
{
	UgDataset*		dataset;

	if (strcmp (element_name, "download") != 0) {
		g_markup_parse_context_push (context, &ug_markup_skip_parser, NULL);
		return;
	}

	dataset = ug_dataset_new ();
	*list = g_list_prepend (*list, dataset);
	g_markup_parse_context_push (context, &ug_data_parser, dataset);
}

static GMarkupParser	ug_download_data_parser =
{
	(gpointer) ug_download_data_start_element,
	(gpointer) g_markup_parse_context_pop,
	NULL,
	NULL,
	NULL
};

static void ug_download_list_start_element (GMarkupParseContext*	context,
                                            const gchar*		element_name,
                                            const gchar**		attr_names,
                                            const gchar**		attr_values,
                                            GList**				list,
                                            GError**			error)
{
	guint	index;

	if (strcmp (element_name, "UgetDownloadList") == 0) {
		for (index=0; attr_names[index]; index++) {
			if (strcmp (attr_names[index], "version") != 0)
				continue;
			if (strcmp (attr_values[index], "1") == 0) {
				g_markup_parse_context_push (context, &ug_download_data_parser, list);
				return;
			}
			// others...
			break;
		}
	}

	g_markup_parse_context_push (context, &ug_markup_skip_parser, NULL);
}

static GMarkupParser	ug_download_list_parser =
{
	(gpointer) ug_download_list_start_element,
	(gpointer) g_markup_parse_context_pop,
	NULL,
	NULL,
	NULL
};

GList*	ug_download_list_load (const gchar* download_file)
{
	UgDataCommon*	common;
	GList*			list;
	GList*			link;

	list = NULL;
	ug_markup_parse (download_file, &ug_download_list_parser, &list);
	// attachment
	for (link = list;  link;  link = link->next) {
		common = UG_DATASET_COMMON ((UgDataset*) link->data);
		if (common == NULL)
			continue;
		ug_attachment_ref (common->attached_stamp);
	}
	// return
	return list;
}

gboolean	ug_download_list_save (GList* list, const gchar* download_file)
{
	UgRelation*	relation;
	UgMarkup*	markup;
	guint		index;

	markup = ug_markup_new ();
	if (ug_markup_write_start (markup, download_file, TRUE) == FALSE) {
		ug_markup_free (markup);
		return FALSE;
	}

	for (index = 0;  list;  list = list->next, index++) {
		relation = UG_DATASET_RELATION ((UgDataset*) list->data);
		if (relation)
			relation->index = index;
		if (list->next == NULL)
			break;
	}

	ug_markup_write_element_start (markup, "UgetDownloadList version='1'");
	for (;  list;  list = list->prev) {
		ug_markup_write_element_start (markup, "download");
		ug_data_to_markup ((UgData*) list->data, markup);
		ug_markup_write_element_end (markup, "download");
	}
	ug_markup_write_element_end (markup, "UgetDownloadList");

	ug_markup_write_end (markup);
	return	TRUE;
}

// ----------------------------------------------------------------------------
// Below utility functions can be used by g_list_foreach()

gboolean	ug_download_attachment_create (UgDataset* dataset)
{
	UgDataHttp*		http;
	UgDataCommon*	common;
	gboolean		result;
	gchar*			file;
	gchar*			dir;
	guint			dir_len;

	result = FALSE;
	// UgDataCommon
	common = UG_DATASET_COMMON (dataset);
	if (common == NULL)
		common = ug_dataset_alloc_front (dataset, UgDataCommonClass);
	else if (common->url == NULL && common->file)
		result = TRUE;
	// UgDataHttp
	http = ug_dataset_get (dataset, UgDataHttpClass, 0);
	if (http && (http->cookie_file || http->post_file))
		result = TRUE;

	// create attachment folder
	if (result == FALSE)
		return FALSE;
	dir = ug_attachment_alloc (&common->attached_stamp);
	if (dir == NULL)
		return FALSE;
	dir_len = strlen (dir);
	g_free (common->attached_folder);
	common->attached_folder = dir;

	// UgDataCommon
	if (common  &&  common->url == NULL  &&  common->file &&
	    dir_len != strspn (dir, common->file))
	{
		file = g_build_filename (dir, "common-file", NULL);
		// copy file and save path
		if (ug_copy_file (common->file, file) == -1)
			g_free (file);
		else {
			g_free (common->file);
			common->file = file;
		}
	}
	// UgDataHttp
	if (http) {
		if (http->cookie_file && dir_len != strspn (dir, http->cookie_file))
		{
			file = g_build_filename (dir, "http-CookieFile", NULL);
			// copy file and save path
			if (ug_copy_file (http->cookie_file, file) == -1)
				g_free (file);
			else {
				g_free (http->cookie_file);
				http->cookie_file = file;
			}
		}
		if (http->post_file && dir_len != strspn (dir, http->post_file))
		{
			file = g_build_filename (dir, "http-PostFile", NULL);
			// copy file and save path
			if (ug_copy_file (http->post_file, file) == -1)
				g_free (file);
			else {
				g_free (http->post_file);
				http->post_file = file;
			}
		}
	}

	return TRUE;
}

gboolean	ug_download_attachment_set (UgDataset* dest_data, UgDataset* src_data)
{
	union {
		UgDataHttp*		http;
		UgDataCommon*	common;
	} src;
	union {
		UgDataHttp*		http;
		UgDataCommon*	common;
	} dest;

	// common
	src.common = UG_DATASET_COMMON (src_data);
	if (src.common == NULL || src.common->attached_stamp == 0)
		return FALSE;
	dest.common = ug_dataset_realloc (dest_data, UgDataCommonClass, 0);
	dest.common->attached_stamp = src.common->attached_stamp;
	ug_str_set (&dest.common->attached_folder, src.common->attached_folder, -1);
	// reference count
	ug_attachment_ref (dest.common->attached_stamp);
	// http
	src.http  = ug_dataset_get (src_data, UgDataHttpClass, 0);
	if (src.http) {
		dest.http = ug_dataset_realloc (dest_data, UgDataHttpClass, 0);
		ug_str_set (&dest.http->post_file,   src.http->post_file,   -1);
		ug_str_set (&dest.http->cookie_file, src.http->cookie_file, -1);
	}
	return TRUE;
}

void	ug_download_temp_delete (UgDataset* dataset)
{
	UgDataCommon*	common;
	gchar*			file;
	gchar*			path;

	common = UG_DATASET_COMMON (dataset);
	if (common && common->file) {
		file = g_strconcat (common->file, ".ug_", NULL);
		if (common->folder == NULL)
			ug_delete_file (file);
		else {
			path = g_build_filename (common->folder, file, NULL);
			ug_delete_file (path);
			g_free (path);
		}
		g_free (file);
	}
}

void	ug_download_data_complete (UgDataset* dataset)
{
	UgUriFull		urifull;
	UgDataCommon*	common;
	const gchar*	string;
	guint			length;

	common = UG_DATASET_COMMON (dataset);
	if (common == NULL)
		return;
	if (ug_uri_full_init (&urifull, common->url) == 0)
		return;
	// file
	if (common->file == NULL) {
		string = ug_uri_full_get_file (&urifull);
		if (string)
			common->file = (gchar*) string;
		else
			common->file = g_strdup ("index.htm");
	}
	// user
	if (common->user == NULL) {
		length = ug_uri_full_user (&urifull, &string);
		if (length)
			common->password = g_strndup (string, length);
	}
	// password
	if (common->password == NULL) {
		length = ug_uri_full_password (&urifull, &string);
		if (length)
			common->password = g_strndup (string, length);
	}
	// Remove user & password from URL
	if (urifull.authority != urifull.host) {
		memmove ((char*) urifull.authority, (char*) urifull.host,
				strlen (urifull.host) + 1);
	}
}


