/*
 *
 *   Copyright (C) 2005-2010 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

#include <stdlib.h>
#include <string.h>
#include <uglib.h>
#include <ug_data_download.h>
#include <ug_category-cmd.h>

#define	UG_DATASET_RELATION_CMD(dataset)		( (UgRelationCmd*) UG_DATASET_RELATION (dataset) )

static void     ug_category_cmd_init     (UgCategoryCmd* ccmd);
static void     ug_category_cmd_finalize (UgCategoryCmd* ccmd);
static gboolean ug_category_cmd_watch    (UgCategoryCmd* ccmd, UgDataset* dataset, UgMessage* message, gpointer user_data);
// UgCategoryCmdIter
static void		ug_category_cmd_iter_add    (UgCategoryCmdIter* iter, UgDataset* dataset);
static void		ug_category_cmd_iter_remove (UgCategoryCmdIter* iter, UgDataset* dataset);
static void		ug_category_cmd_iter_switch (UgCategoryCmdIter* iter, UgDataset* dataset, GQueue* ccmd_queue);

// define UgDataEntry for UgDataClass
static UgDataEntry	category_cmd_data_entry[] =
{
	UG_CATEGORY_DATA_ENTRY,
	{NULL},			// null-terminated
};

static UgDataClass	category_cmd_data_class =
{
	UG_CATEGORY_CLASS_NAME,		// name
	NULL,						// reserve
	sizeof (UgCategoryCmd),		// instance_size
	category_cmd_data_entry,	// entry

	(UgInitFunc)     ug_category_cmd_init,
	(UgFinalizeFunc) ug_category_cmd_finalize,
	(UgAssignFunc)   ug_category_assign,
};
// extern
const	UgDataClass*	UgCategoryCmdClass = &category_cmd_data_class;

static void ug_category_cmd_init (UgCategoryCmd* ccmd)
{
	// UgCategory
	ug_category_init ((UgCategory*) ccmd);
	ccmd->limit.active = 1;
	ccmd->watch.func = (UgWatchFunc) ug_category_cmd_watch;
	// UgCategoryCmd
	ccmd->queued = g_queue_new ();
	ccmd->active = g_queue_new ();
	ccmd->finished = g_queue_new ();
	ccmd->recycled = g_queue_new ();
}

static void ug_category_cmd_queue_free (UgCategoryCmd* ccmd, GQueue* ccmd_queue)
{
	GList*	link;

	for (link = ccmd_queue->head;  link;  link = link->next)
		ug_category_cmd_remove (ccmd, link->data);
	g_queue_free (ccmd_queue);
}

static void ug_category_cmd_finalize (UgCategoryCmd* ccmd)
{
	// UgCategory
	ug_category_finalize ((UgCategory*) ccmd);
	// UgCategoryCmd
	ug_category_cmd_queue_free (ccmd, ccmd->queued);
	ug_category_cmd_queue_free (ccmd, ccmd->active);
	ug_category_cmd_queue_free (ccmd, ccmd->finished);
	ug_category_cmd_queue_free (ccmd, ccmd->recycled);
}

UgCategoryCmd*	ug_category_cmd_new  (UgCategoryCmd* primary)
{
	UgCategoryCmd*	ccmd;

	ccmd = ug_data_new (&category_cmd_data_class);
	ccmd->primary = primary;
	return ccmd;
}

void	ug_category_cmd_free (UgCategoryCmd* ccmd)
{
	ug_data_free (ccmd);
}

void	ug_category_cmd_add (UgCategoryCmd* secondary, UgDataset* dataset)
{
	UgCategoryCmd*		primary;
	UgRelationCmd*		relation;

	// add and set UgRelationCmdClass to dataset
	UG_DATASET_RELATION_CLASS (dataset) = UgRelationCmdClass;
	relation = UG_DATASET_RELATION_CMD (dataset);
	if (relation == NULL)
		relation = ug_dataset_alloc_front (dataset, UgRelationCmdClass);

	if (secondary->primary)
		primary = secondary->primary;
	else {
		primary = secondary;
		secondary = NULL;
	}
	// add job to primary & secondary category
	if (primary && relation->primary.ccmd == NULL) {
		relation->primary.ccmd = primary;
		ug_category_cmd_iter_add (&relation->primary, dataset);
	}
	if (secondary && relation->secondary.ccmd == NULL) {
		relation->secondary.ccmd = secondary;
		ug_category_cmd_iter_add (&relation->secondary, dataset);
	}
}

void	ug_category_cmd_remove (UgCategoryCmd* ccmd, UgDataset* dataset)
{
	UgRelationCmd*		relation;

	// delete and temp file
	ug_download_temp_delete (dataset);

	relation = UG_DATASET_RELATION_CMD (dataset);
	if (relation->primary.ccmd) {
		// remove job from primary category
		ug_category_cmd_iter_remove (&relation->primary, dataset);
	}
	if (relation->secondary.ccmd) {
		// stop active job in secondary category
		ug_category_stop ((UgCategory*) relation->secondary.ccmd, dataset);
		// remove job from secondary category
		ug_category_cmd_iter_remove (&relation->secondary, dataset);
	}
}

void	ug_category_cmd_switch (UgCategoryCmd* secondary, UgDataset* dataset, UgCategoryHints hint)
{
	UgCategoryCmd*	primary;
	UgRelationCmd*	relation;
	GQueue*			primary_queue   = NULL;
	GQueue*			secondary_queue = NULL;

	relation = UG_DATASET_RELATION_CMD (dataset);
	primary   = relation->primary.ccmd;
	secondary = relation->secondary.ccmd;
	// select queue
	if (hint & UG_HINT_RECYCLED) {
		if (primary)
			primary_queue = primary->recycled;
		if (secondary)
			secondary_queue = secondary->recycled;
		relation->hints &= ~(UG_HINT_FINISHED | UG_HINT_ACTIVE);
		relation->hints |=  UG_HINT_RECYCLED;
	}
	else if (hint & UG_HINT_FINISHED) {
		if (primary)
			primary_queue = primary->finished;
		if (secondary)
			secondary_queue = secondary->finished;
		relation->hints &= ~(UG_HINT_RECYCLED | UG_HINT_ACTIVE);
		relation->hints |=  UG_HINT_FINISHED;
	}
	else if (hint & UG_HINT_ACTIVE) {
		if (primary)
			primary_queue = primary->active;
		if (secondary)
			secondary_queue = secondary->active;
		relation->hints &= ~UG_HINT_UNRUNNABLE;
		relation->hints |=  UG_HINT_ACTIVE;
	}
	else {
		if (primary)
			primary_queue = primary->queued;
		if (secondary)
			secondary_queue = secondary->queued;
		relation->hints &= ~(UG_HINT_FINISHED | UG_HINT_RECYCLED | UG_HINT_ACTIVE);
	}

	if (primary)
		ug_category_cmd_iter_switch (&relation->primary, dataset, primary_queue);
	if (secondary)
		ug_category_cmd_iter_switch (&relation->secondary, dataset, secondary_queue);
}

void	ug_category_cmd_clear  (UgCategoryCmd* ccmd, UgCategoryHints hint, guint from_nth)
{
	GQueue*		queue;

	// select queue
	if (hint & UG_HINT_RECYCLED)
		queue = ccmd->recycled;
	else if (hint & UG_HINT_FINISHED)
		queue = ccmd->finished;
	else if (hint & UG_HINT_ACTIVE)
		queue = ccmd->active;
	else
		queue = ccmd->queued;

	while (queue->length > from_nth)
		ug_category_cmd_remove (ccmd, queue->tail->data);
}

void	ug_category_cmd_move_to (UgCategoryCmd* ccmd, UgDataset* dataset, UgCategoryCmd* ccmd_dest)
{
	UgRelationCmd*		relation;

	relation = UG_DATASET_RELATION_CMD (dataset);
	if (relation->primary.ccmd == ccmd_dest || relation->secondary.ccmd == ccmd_dest)
		return;

	if (relation->secondary.ccmd) {
		// stop active job in secondary category
		ug_category_stop ((UgCategory*) relation->secondary.ccmd, dataset);
		// move
		ug_dataset_ref (dataset);
		ug_category_cmd_iter_remove (&relation->secondary, dataset);
		relation->secondary.ccmd = ccmd_dest;
		ug_category_cmd_iter_add (&relation->secondary, dataset);
		ug_dataset_unref (dataset);
	}
}

// return 0: no active job
// return 1: no active job, some job stop by user
// return 2: running
gint	ug_category_cmd_run (UgCategoryCmd* ccmd)
{
	GList*			link;
	GList*			link_next;
	UgDataset*		dataset;

	// refresh
	for (link = ccmd->active->head;  link;  link = link_next) {
		link_next = link->next;
		dataset = link->data;
		if (ug_category_refresh ((UgCategory*) ccmd, dataset) == FALSE)
			ug_category_cmd_switch (ccmd, dataset, UG_HINT_FINISHED);
	}
	// start new job
	while (ccmd->active->length < ccmd->limit.active) {
		dataset = g_queue_peek_head (ccmd->queued);
		if (dataset == NULL)
			break;
		if (ug_category_start ((UgCategory*) ccmd, dataset))
			ug_category_cmd_switch (ccmd, dataset, UG_HINT_ACTIVE);
		else
			ug_category_cmd_switch (ccmd, dataset, UG_HINT_FINISHED);
	}
	// clear exceed job in finished queue
	ug_category_cmd_clear (ccmd, UG_HINT_FINISHED, ccmd->limit.finished);
	// return 0 if category stop
	return (ccmd->active->length) ? 2 : 0;
}

GList*	ug_category_cmd_get_list (UgCategoryCmd* ccmd)
{
	GList*	link;
	GList*	list = NULL;

	for (link = ccmd->recycled->tail;  link;  link = link->prev)
		list = g_list_prepend (list, link->data);

	for (link = ccmd->finished->tail;  link;  link = link->prev)
		list = g_list_prepend (list, link->data);

	for (link = ccmd->queued->tail;  link;  link = link->prev)
		list = g_list_prepend (list, link->data);

	for (link = ccmd->active->tail;  link;  link = link->prev)
		list = g_list_prepend (list, link->data);

	return list;
}

guint	ug_category_cmd_list_run (GList* list)
{
	guint	count;

	for (count = 0;  list;  list = list->next) {
		if (ug_category_cmd_run (list->data) > 0)
			count++;
	}
	return count;
}

static gboolean ug_category_cmd_watch (UgCategoryCmd* ccmd, UgDataset* dataset, UgMessage* message, gpointer user_data)
{
	UgDataCommon*	common;
	gchar*			name;

	common = UG_DATASET_COMMON (dataset);
	if (common->name)
		name = common->name;
	else if (common->url)
		name = common->url;
	else if (common->file)
		name = common->file;
	else
		name = NULL;

	switch (message->type) {
	case UG_MESSAGE_STATE:
		if (message->data.v_int == UG_STATE_ACTIVE)
			g_print ("Start  %s\n", name);
		else
			g_print ("Stop   %s\n", name);
		break;

	case UG_MESSAGE_ERROR:
		g_print ("Error  %s\n", name);
		break;

	case UG_MESSAGE_INFO:
		if (message->code == UG_MESSAGE_INFO_COMPLETE)
			g_print ("Finish %s\n", name);
		break;

	default:
		break;
	}
	return FALSE;
}


// ----------------------------------------------------------------------------
// UgCategoryCmdIter : iterator for UgCategoryCmd
//
static void	ug_category_cmd_iter_add (UgCategoryCmdIter* iter, UgDataset* dataset)
{
	UgRelationCmd*	relation;
	UgCategoryCmd*	ccmd;

	ccmd = iter->ccmd;
	relation = UG_DATASET_RELATION_CMD (dataset);
	ug_dataset_ref (dataset);
	// select queue & add job to it
	if (relation->hints & UG_HINT_RECYCLED) {
		g_queue_push_head (ccmd->recycled, dataset);
		iter->queue = ccmd->recycled;
		iter->link  = ccmd->recycled->head;
	}
	else if (relation->hints & UG_HINT_FINISHED) {
		g_queue_push_head (ccmd->finished, dataset);
		iter->queue = ccmd->finished;
		iter->link  = ccmd->finished->head;
	}
	else {
		g_queue_push_tail (ccmd->queued, dataset);
		iter->queue = ccmd->queued;
		iter->link  = ccmd->queued->tail;
	}
}

static void	ug_category_cmd_iter_remove (UgCategoryCmdIter* iter, UgDataset* dataset)
{
	// remove job from queue
	iter->ccmd = NULL;
	g_queue_delete_link (iter->queue, iter->link);
	ug_dataset_unref (dataset);
}

static void	ug_category_cmd_iter_switch (UgCategoryCmdIter* iter, UgDataset* dataset, GQueue* ccmd_queue)
{
	UgCategoryCmd*	ccmd;

	// Don't switch job to the same queue.
	if (iter->queue == ccmd_queue)
		return;
	// switch job to new queue
	g_queue_delete_link (iter->queue, iter->link);
	iter->queue = ccmd_queue;
	ccmd = iter->ccmd;
	if (ccmd_queue == ccmd->finished || ccmd_queue == ccmd->recycled) {
		g_queue_push_head (ccmd_queue, dataset);
		iter->link  = ccmd_queue->head;
	}
	else {
		g_queue_push_tail (ccmd_queue, dataset);
		iter->link  = ccmd_queue->tail;
	}
}


// ----------------------------------------------------------------------------
// UgRelationCmd
//
static UgDataEntry	relation_cmd_data_entry[] =
{
	UG_RELATION_DATA_ENTRY,
	{NULL},			// null-terminated
};

static UgDataClass	relation_cmd_data_class =
{
	UG_RELATION_CLASS_NAME,		// name
	NULL,						// reserve
	sizeof (UgRelationCmd),		// instance_size
	relation_cmd_data_entry,	// entry

	(UgInitFunc)     NULL,
	(UgFinalizeFunc) NULL,
	(UgAssignFunc)   NULL,
};
// extern
const	UgDataClass*	UgRelationCmdClass = &relation_cmd_data_class;

