/*
 *
 *   Copyright (C) 2005-2010 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

#include <assert.h>
#include <errno.h>
#include <string.h>

#ifdef _WIN32
#include <windows.h>
#endif
#include <gio/gio.h>
#include <ug_utils.h>
#include <ug_stdio.h>

// ----------------------------------------------------------------------------
// string functions

/*  This function works like strcspn ()
 *  If string_len == -1, string is null-terminated.
 *  It return the index of the first character in string that is in charset.
 *  If none of the characters in string is in charset, then the return value is the length of string.
 */
unsigned int ug_str_find_charset (const char* string, int string_len, unsigned int offset, const char* charset)
{
	const char* string_end;
	const char* string_cur;
	const char* ch;

	if (string_len == -1)
		string_len = (int) strlen (string);
	string_end = string + string_len;

	for (string_cur = string + offset;  string_cur < string_end;  string_cur++) {
		for (ch = charset;  *ch;  ch++) {
			if (*string_cur == *ch)
				return (unsigned int) (string_cur - string);
		}
	}

	return string_len;
}

/*
 *  This function try to skip specified character from offset in string.
 *  If found other than specified character, return current offset.
 *  Otherwise return original offset.
 */
unsigned int ug_str_skip_charset (const char* string, int string_len, unsigned int offset, const char* charset)
{
	const char* string_cur;
	const char* string_end;
	const char* ch;

	if (string_len == -1)
		string_len = (int) strlen (string);
	string_end = string + string_len;

	for (string_cur = string + offset;  string_cur < string_end;  string_cur++) {
		for (ch = charset;  ;  ch++) {
			if (*ch == 0)
				goto exit;
			if (*string_cur == *ch)
				break;
		}
	}

exit:
	return (unsigned int) (string_cur - string);
}

/*
 *  This function try to clear specified character from tail of string. (set character to '\0')
 *  return new string_len.
 */
unsigned int ug_str_clear_tail_charset (char* string, int string_len, const char* charset)
{
	const char*	ch;
	char*		string_cur;

	if (string_len == -1)
		string_len = (int) strlen (string);

	for (string_cur = string +string_len -1;  string_cur >= string;  string_cur--) {
		for (ch = charset;  ;  ch++) {
			if (*ch == 0)
				goto exit;
			if (*string_cur == *ch) {
				*string_cur = 0;
				break;
			}
		}
	}

exit:
	return (unsigned int) (string_cur - string + 1);
}

/*
 *  count line length until character '\r' , '\n' ,or end of string from offset in string.
 *  return : length of line exclude line terminator.
 */
unsigned int ug_str_line_len (const char* string, int string_len, unsigned int offset)
{
	unsigned int result;

	result = ug_str_find_charset(string, string_len, offset, "\r\n");
	if (result > offset)
		return result - offset;
	return 0;
}

/*
 *  return : offset of next line in string.
 *  return string_len if no next line.
 */
unsigned int ug_str_line_next (const char* string, int string_len, unsigned int offset)
{
	const char* string_end;
	const char* string_cur;

	if (string_len == -1)
		string_len = (int) strlen (string);
	string_end = string + string_len;

	for (string_cur = string + offset;  string_cur < string_end;  string_cur++) {
		if (*string_cur == '\n')
			return (unsigned int) (string_cur - string) + 1;
	}

	return string_len;
}

/*
 * It free original string that point by string_pointer and copy a new src string to string_pointer.
 * If src_len is -1, src string is null-terminated.
 * If src is NULL, *string_pointer will set NULL.
 *
 *  char* temp_str = strdup ("123");
 *
 *  ug_str_set (&temp_str, "45678", -1);		// temp_str will set "45678"
 */
void	ug_str_set (char** string_pointer, const char* src, int src_len)
{
	g_free (*string_pointer);

	if (src == NULL || *src == 0) {
		*string_pointer = NULL;
		return;
	}

	if (src_len == -1)
		*string_pointer = g_strdup (src);
	else
		*string_pointer = g_strndup (src, src_len);
}

/*
 * convert double to string
 * If value large than 1024, it will append unit string like "KiB", "MiB", "GiB", "TiB", or "PiB"  to string.
 */
gchar*	ug_str_dtoa_unit (gdouble value, gint precision, const gchar* tail)
{
	static const gchar*	unit_array[] = {"", " KiB", " MiB", " GiB", " TiB", " PiB"};
	static const guint	unit_array_len = sizeof (unit_array) / sizeof (gchar*);
	guint	index;

	for (index=0;  index < unit_array_len -1;  index++) {
		if (value < 1024.0)
			break;
		value /= 1024.0;
	}
	if (index == 0)
		precision = 0;

	return g_strdup_printf ("%.*f%s%s",
			precision, value, unit_array[index], (tail) ? tail : "");
}

/*
 * convert time to string (hh:mm:ss)
 */
gchar*	ug_str_from_time (guint seconds, gboolean limit_99_99_99)
{
	guint		hour, minute;
	guint		day, year;

	minute  = seconds / 60;
	hour    = minute / 60;
	minute  = minute % 60;
	seconds = seconds % 60;

	if (hour < 100)
		return g_strdup_printf ("%.2u:%.2u:%.2u", hour, minute, seconds);
	else if (limit_99_99_99)
		return g_strdup ("99:99:99");
	else {
		day    = hour / 24;
		year   = day / 365;
		if (year)
			return g_strdup ("> 1 year");
		else
			return g_strdup_printf ("%u days", day);
	}
}


// ------------------------------------------------------------------
// file & directory functions
//
#ifndef _WIN32
int  ug_rename (const gchar *old_filename, const gchar *new_filename)
{
	if (g_get_filename_charsets (NULL))
		return g_rename (old_filename, new_filename);
	else {
		gchar *cp_old_filename = g_filename_from_utf8 (old_filename, -1, NULL, NULL, NULL);
		gchar *cp_new_filename = g_filename_from_utf8 (new_filename, -1, NULL, NULL, NULL);
		int save_errno;
		int retval;

		if (cp_old_filename == NULL || cp_new_filename == NULL) {
			g_free (cp_old_filename);
			g_free (cp_new_filename);
			errno = EINVAL;
			return -1;
		}

		retval = g_rename (cp_old_filename, cp_new_filename);
		save_errno = errno;

		g_free (cp_old_filename);
		g_free (cp_new_filename);

		errno = save_errno;
		return retval;
	}
}

int  ug_unlink (const gchar *filename)
{
	if (g_get_filename_charsets (NULL))
		return g_unlink (filename);
	else {
		gchar *cp_filename = g_filename_from_utf8 (filename, -1, NULL, NULL, NULL);
		int save_errno;
		int retval;

		if (cp_filename == NULL) {
			errno = EINVAL;
			return -1;
		}

		retval = g_unlink (cp_filename);
		save_errno = errno;

		g_free (cp_filename);

		errno = save_errno;
		return retval;
	}
}

int  ug_create_dir (const gchar *dir_utf8)
{
	if (g_get_filename_charsets (NULL))
		return g_mkdir (dir_utf8, 0755);
	else {
		gchar *cp_filename = g_filename_from_utf8 (dir_utf8, -1, NULL, NULL, NULL);
		int save_errno;
		int retval;

		if (cp_filename == NULL) {
			errno = EINVAL;
			return -1;
		}

		retval = g_mkdir (cp_filename, 0755);
		save_errno = errno;

		g_free (cp_filename);

		errno = save_errno;
		return retval;
	}
}

int  ug_delete_dir (const gchar *dir_utf8)
{
	if (g_get_filename_charsets (NULL))
		return g_rmdir (dir_utf8);
	else {
		gchar *cp_filename = g_filename_from_utf8 (dir_utf8, -1, NULL, NULL, NULL);
		int save_errno;
		int retval;

		if (cp_filename == NULL) {
			errno = EINVAL;
			return -1;
		}

		retval = g_rmdir (cp_filename);
		save_errno = errno;

		g_free (cp_filename);

		errno = save_errno;
		return retval;
	}
}
#endif

#ifdef _WIN32
int  ug_copy_file (const gchar *src_file_utf8, const gchar *new_file_utf8)
{
	gboolean	result;
	gunichar2*	src_file_wcs;
	gunichar2*	new_file_wcs;

	src_file_wcs = g_utf8_to_utf16 (src_file_utf8, -1, NULL, NULL, NULL);
	new_file_wcs = g_utf8_to_utf16 (new_file_utf8, -1, NULL, NULL, NULL);
	result = CopyFileW (src_file_wcs, new_file_wcs, FALSE);
	g_free (src_file_wcs);
	g_free (new_file_wcs);
	if (result == 0)
		return -1;
	return 0;
}
#else
int  ug_copy_file (const gchar *src_file_utf8, const gchar *new_file_utf8)
{
	int		src_fd;
	int		new_fd;
	char*	buf;
	int		buf_len;
	int		result = 0;

	new_fd = ug_fd_open (new_file_utf8,
	                     UG_FD_O_BINARY | UG_FD_O_WRONLY | UG_FD_O_CREAT,
	                     UG_FD_S_IREAD | UG_FD_S_IWRITE | UG_FD_S_IRGRP | UG_FD_S_IROTH);
	if (new_fd == -1)
		return -1;

	src_fd = ug_fd_open (src_file_utf8, UG_FD_O_BINARY | UG_FD_O_RDONLY, UG_FD_S_IREAD);
	if (src_fd == -1) {
		ug_fd_close (new_fd);
		return -1;
	}
	// read & write
	buf = g_malloc (8192);
	for (;;) {
		buf_len = ug_fd_read (src_fd, buf, 8192);
		if (buf_len <=0)
			break;
		if (ug_fd_write (new_fd, buf, buf_len) != buf_len) {
			result = -1;
			break;
		}
	}
	// clear
	g_free (buf);
	ug_fd_close (src_fd);
	ug_fd_close (new_fd);
	return result;
}
#endif	// _WIN32

// This function use complex way to handle directory because some locale encoding doesn't avoid '\\' or '/'.
int  ug_create_dir_all (const gchar* dir, gint len)
{
	const gchar*	dir_end;
	const gchar*	element_end;	// path element
	gchar*			element_os;

	if (len == -1)
		len = strlen (dir);
	dir_end = dir + len;
	element_end = dir;

	for (;;) {
		// skip directory separator "\\\\" or "//"
		for (;  element_end < dir_end;  element_end++) {
			if (*element_end != G_DIR_SEPARATOR)
				break;
		}
		if (element_end == dir_end)
			return 0;
		// get directory name [dir, element_end)
		for (;  element_end < dir_end;  element_end++) {
			if (*element_end == G_DIR_SEPARATOR)
				break;
		}
		// create directory by locale encoding name.
		element_os = g_filename_from_utf8 (dir, element_end - dir, NULL, NULL, NULL);
		if (element_os == NULL)
			break;
		if (g_file_test (element_os, G_FILE_TEST_EXISTS) == FALSE) {
			if (g_mkdir (element_os, 0755) == -1) {
				g_free (element_os);
				break;
			}
		}
		else if (g_file_test (element_os, G_FILE_TEST_IS_DIR) == FALSE) {
			g_free (element_os);
			break;
		}
		g_free (element_os);
	}
	return -1;
}

int  ug_delete_dir_recursive (const gchar *utf8_dir)
{
	GDir*		dir;
	gchar*		name;
	gchar*		path;
	gchar*		locale_dir;
	gboolean	is_dir;

	if (g_get_filename_charsets (NULL)) {
		locale_dir = NULL;
		dir = g_dir_open (utf8_dir, 0, NULL);
	}
	else {
		locale_dir = g_filename_from_utf8 (utf8_dir, -1, NULL, NULL, NULL);
		dir = g_dir_open (locale_dir, 0, NULL);
	}

	if (dir == NULL) {
		g_free (locale_dir);
		return -1;
	}

	for (;;) {
		name = (gchar*) g_dir_read_name (dir);
		if (name == NULL)
			break;
		if (locale_dir == NULL) {
			path = g_build_filename (utf8_dir, name, NULL);
			is_dir = g_file_test (path, G_FILE_TEST_IS_DIR);
		}
		else {
			name = g_build_filename (locale_dir, name, NULL);
			path = g_filename_to_utf8 (name, -1, NULL, NULL, NULL);
			is_dir = g_file_test (name, G_FILE_TEST_IS_DIR);
			g_free (name);
		}
		// delete file or directory
		if (is_dir)
			ug_delete_dir_recursive (path);
		else
			ug_delete_file (path);
		g_free (path);
	}
	g_free (locale_dir);
	g_dir_close (dir);
	ug_delete_dir (utf8_dir);
	return 0;
}


// ------------------------------------------------------------------
// others
//
#ifdef _WIN32
gboolean	ug_launch_default_app (const gchar* folder, const gchar* file)
{
	gchar*		path;
	gunichar2*	path_wcs;

	if (folder == NULL)
		path = g_build_filename (file, NULL);
	else
		path = g_build_filename (folder, file, NULL);
	if (g_file_test (path, G_FILE_TEST_EXISTS) == FALSE) {
		g_free (path);
		return FALSE;
	}

	// UNICODE
	path_wcs = g_utf8_to_utf16 (path, -1, NULL, NULL, NULL);
	g_free (path);
	ShellExecuteW (NULL, L"open", path_wcs, NULL, NULL, SW_SHOW);
	g_free (path_wcs);

	return TRUE;
}

void	ug_shutdown (void)
{
	HANDLE hToken;
	TOKEN_PRIVILEGES tkp;

	if (OpenProcessToken (GetCurrentProcess(), TOKEN_ADJUST_PRIVILEGES|TOKEN_QUERY, &hToken))
	{
		LookupPrivilegeValue (NULL, SE_SHUTDOWN_NAME, &tkp.Privileges[0].Luid);
		tkp.PrivilegeCount = 1;
		tkp.Privileges[0].Attributes = SE_PRIVILEGE_ENABLED;
		AdjustTokenPrivileges (hToken, FALSE, &tkp, 0, (PTOKEN_PRIVILEGES)NULL, 0);
		ExitWindowsEx (EWX_SHUTDOWN | EWX_POWEROFF, 0);
	}

	ExitWindowsEx (EWX_SHUTDOWN | EWX_POWEROFF, 0);
}
#else
gboolean	ug_launch_default_app (const gchar* folder, const gchar* file)
{
	GError* error = NULL;
	GFile*	gfile;
	gchar*	uri;
	gchar*	path;
	gchar*	path_wcs;

	path = g_build_filename (folder, file, NULL);
	path_wcs = g_filename_from_utf8 (path, -1, NULL, NULL, NULL);
	g_free (path);
	if (g_file_test (path_wcs, G_FILE_TEST_EXISTS) == FALSE) {
		g_free (path_wcs);
		return FALSE;
	}

	gfile = g_file_new_for_path (path_wcs);
	g_free (path_wcs);
	uri = g_file_get_uri (gfile);
	g_object_unref (gfile);
	g_app_info_launch_default_for_uri (uri, NULL, &error);
	g_free (uri);

	if (error) {
#ifndef NDEBUG
		g_print ("%s", error->message);
#endif
		g_error_free (error);
	}

	return TRUE;
}

void	ug_shutdown (void)
{
	g_spawn_command_line_async ("poweroff", NULL);
//	g_spawn_command_line_async ("shutdown -h -P now", NULL);
	// change to runlevel 0
}
#endif

