//                              -*- Mode: C++ -*- 
// 
// uC++ Version 5.3.0, Copyright (C) Peter A. Buhr 1994
// 
// uMachContext.cc -- 
// 
// Author           : Peter Buhr
// Created On       : Fri Feb 25 15:46:42 1994
// Last Modified By : Peter A. Buhr
// Last Modified On : Mon Oct 10 10:55:48 2005
// Update Count     : 517
//
// This  library is free  software; you  can redistribute  it and/or  modify it
// under the terms of the GNU Lesser General Public License as published by the
// Free Software  Foundation; either  version 2.1 of  the License, or  (at your
// option) any later version.
// 
// This library is distributed in the  hope that it will be useful, but WITHOUT
// ANY  WARRANTY;  without even  the  implied  warranty  of MERCHANTABILITY  or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
// for more details.
// 
// You should  have received a  copy of the  GNU Lesser General  Public License
// along  with this library.
// 


#define __U_KERNEL__
#include <uC++.h>
#include <uAlign.h>
#include <uProfiler.h>

//#include <uDebug.h>

#include <cstdio>					// fprintf
#include <cerrno>

#if defined( __sparc__ ) || defined( __mips__ ) || defined( __x86_64__ ) || defined( __ia64__ )
extern "C" void uInvokeStub( uMachContext * );
#endif // __sparc__ || __mips__

#if defined( __sparc__ )
#if defined(  __solaris__ )
#include <sys/stack.h>
#else
#include <machine/trap.h>
#include <machine/asm_linkage.h>
#include <machine/psl.h>
#endif // __solaris__
#endif // __sparc__


extern "C" void _pthread_deletespecific( void * );	// see pthread simulation


void uMachContext::uInvokeCoroutine( uBaseCoroutine &This ) { // magically invoke the "main" of the most derived class
    // Called from the kernel when starting a coroutine or task so must switch
    // back to user mode.

    This.setState( uBaseCoroutine::Active );		// set state of next coroutine to active
    THREAD_GETMEM( uSelf )->enableInterrupts();

    if ( uThisTask().uProfileActive && uProfiler::uProfiler_PostallocateMetricMemory ) {
	(*uProfiler::uProfiler_PostallocateMetricMemory)( uProfiler::uProfilerInstance, uThisTask() );
    } // if

    // At this point, execution is on the stack of the new coroutine or task
    // that has just been switched to by the kernel.  Therefore, interrupts can
    // legitimately occur now.

    try {
	This.uCoStarter();				// moved from uCoroutineMain to allow recursion on "main"
	This.main();					// start coroutine's "main" routine
    } catch( uBaseCoroutine::UnHandledException &ex ) {
	ex.multiple = true;				// set multiple exceptions
	_Resume ex _At This.resumer();                    
	This.notHalted = false;				// terminate coroutine
	This.suspend();					// restart last resumer, which should immediately propagate the nonlocal exception
    } catch( ... ) {					// unhandled exception ?
	char msg[uEHMMaxMsg], name[uEHMMaxName + sizeof("...")]; // room for "...\0"
	strcat( strcpy( msg, "an unhandled exception of type " ), uEHM::getCurrentEventName( uEHM::uThrowF, name, uEHMMaxName ) );
	_Resume uBaseCoroutine::UnHandledException( msg ) _At This.resumer();
	This.notHalted = false;				// terminate coroutine
	This.suspend();					// restart last resumer, which should immediately propagate the nonlocal exception
    } // try
    This.uCoFinish();
} // uMachContext::uInvokeCoroutine


void uMachContext::uInvokeTask( uBaseTask &This ) {	// magically invoke the "main" of the most derived class
    // Called from the kernel when starting a coroutine or task so must switch
    // back to user mode.

#if defined(__U_MULTI__)
    THREAD_SETMEM( uActiveTask, ((uBaseTask *)&This) );
#endif

    errno = 0;						// reset errno for each task
    This.currCoroutine->setState( uBaseCoroutine::Active ); // set state of next coroutine to active
    This.setState( uBaseTask::Running );
    THREAD_GETMEM( uSelf )->enableInterrupts();

    if ( uThisTask().uProfileActive && uProfiler::uProfiler_PostallocateMetricMemory ) {
	(*uProfiler::uProfiler_PostallocateMetricMemory)( uProfiler::uProfilerInstance, uThisTask() );
    } // if

    // At this point, execution is on the stack of the new coroutine or task
    // that has just been switched to by the kernel.  Therefore, interrupts can
    // legitimately occur now.

    try {
	This.main();					// start task's "main" routine
	_Disable {					// task has terminated so ignore concurrent exceptions
	    This.notHalted = false;
	    This.setState( uBaseTask::Terminate );

	    // Clean up storage associated with the task for pthread
	    // thread-specific data, e.g., exception handling associates
	    // thread-specific data with any task.

	    if ( This.pthreadData != NULL ) {
		_pthread_deletespecific( This.pthreadData );
	    } // if
	    This.getSerial().uLeave2();
	    uAbort( "(uMachContext::uInvokeTask() : internal error, CONTROL NEVER REACHES HERE!" );
	} // _Disable
    } catch( uEHM::uDualClass &evt ) {
	evt.defaultTerminate();
	cleanup( This );				// preserve current exception
    } catch( uEHM::uThrowClass &evt ) {
	evt.defaultTerminate();
	cleanup( This );				// preserve current exception
    } catch( ... ) {
	cleanup( This );				// preserve current exception
    } // try
    // CONTROL NEVER REACHES HERE!
} // uMachContext::uInvokeTask


void uMachContext::cleanup( uBaseTask &This ) {
    try {
	std::terminate();				// call task terminate routine
    } catch( ... ) {					// control should not return
    } // try

    if ( This.pthreadData != NULL ) {			// see above for explanation
	_pthread_deletespecific( This.pthreadData );
    } // if

    uEHM::uTerminate();					// call abort terminate routine
} // uMachContext::cleanup


void *uMachContext::uRtnAdr( void (*rtn)() ) {
#if defined( __linux__ ) && defined( __ia64__ )
    if ( ! rtn ) return NULL;

    struct TOC {
	void *rtn;
	void *toc;
    };

    return ((TOC *)rtn)->rtn;
#else
    return (void *)rtn;
#endif
} // uMachContext::uRtnAdr


#if defined( __linux__ ) && defined( __ia64__ )
void *uMachContext::uGpAdr( void (*rtn)() ) {
    if ( ! rtn ) return NULL;

    struct TOC {
	void *rtn;
	void *toc;
    };

    return ((TOC *)rtn)->toc;
} // uMachContext::uGpAdr
#endif


unsigned int uMachContext::uMagic() {
    return 0x12345678;
} // uMachContext::uMagic


void uMachContext::uStartHere( void (*uInvoke)( uMachContext & ) ) {

/*******************************************************
  magic number (16 bytes)
  ,-----------------. \ <--- uBase (stack grows down)
  |                 | |
  |    task stack   | } uSize
  |                 | |
  `-----------------' / <--- uLimit
  magic number (16 bytes)
  ,-----------------.
  |                 |
  | __U_CONTEXT_T__ |
  |                 |
0 `-----------------' <--- uStorage
*******************************************************/

#if defined( __i386__ )

    // set base and limits of stack

    uLimit = (char *)uStorage + uCeiling( sizeof(__U_CONTEXT_T__), 16 ) + 16;
    uBase = (char *)uLimit + uSize;

#if ! defined( __U_SWAPCONTEXT__ )
    struct uFakeStack {
	void *uFixedRegisters[3];			// fixed registers ebx, edi, esi
	void *uReturn;
	void *uDummyReturn;
	void *uArgument;
    };

    ((uContext_t *)uStorage)->uSP = (char *)uBase - sizeof( uFakeStack );
    ((uContext_t *)uStorage)->uFP = NULL;		// terminate stack with NULL fp

    ((uFakeStack *)(((uContext_t *)uStorage)->uSP))->uArgument = this; // argument to uInvoke
    ((uFakeStack *)(((uContext_t *)uStorage)->uSP))->uReturn = uRtnAdr( (void (*)())uInvoke );
#else
    // set base and limits of stack

    uLimit = (char *)uStorage + sizeof(ucontext_t) + 16;
    uBase = (char *)uLimit + uSize;

    if ( ::getcontext( (ucontext *)uStorage ) == -1 ) {	// initialize ucontext area
	uAbort( "internal error, getcontext failed" );
    } // if
    ((ucontext *)uStorage)->uc_stack.ss_sp = (char *)uLimit;
    ((ucontext *)uStorage)->uc_stack.ss_size = uSize; 
    ((ucontext *)uStorage)->uc_stack.ss_flags = 0;
    ::makecontext( (ucontext *)uStorage, (void (*)())uInvoke, 2, this );
    // TEMPORARY: stack is incorrectly initialized to allow stack walking
    ((ucontext *)uStorage)->uc_mcontext.gregs[ REG_EBP ] = 0; // terminate stack with NULL fp
#if defined( __U_MULTI__ ) && defined( __U_ONETIMER__ )
    sigaddset( (sigset_t*)&((ucontext *)uStorage)->uc_sigmask, SIGALRM );
#endif // __U_MULTI__ && __U_ONETIMER__
#endif

#elif defined( __x86_64__ )

    // set base and limits of stack

    uLimit = (char *)uStorage + uCeiling( sizeof(__U_CONTEXT_T__), 16 ) + 16;
    uBase = (char *)uLimit + uSize;

#if ! defined( __U_SWAPCONTEXT__ )
    struct uFakeStack {
	void *uFixedRegisters[5];			// fixed registers rbx, r12, r13, r14, r15
	void *uReturn;
    };

    ((uContext_t *)uStorage)->uSP = (char *)uBase - sizeof( uFakeStack );
    ((uContext_t *)uStorage)->uFP = NULL;		// terminate stack with NULL fp

    ((uFakeStack *)(((uContext_t *)uStorage)->uSP))->uReturn = uRtnAdr( (void (*)())uInvokeStub );
    ((uFakeStack *)(((uContext_t *)uStorage)->uSP))->uFixedRegisters[0] = this;
    ((uFakeStack *)(((uContext_t *)uStorage)->uSP))->uFixedRegisters[1] = uRtnAdr( (void (*)())uInvoke );
#else
    // makecontext is difficult to support.  See http://sources.redhat.com/bugzilla/show_bug.cgi?id=404
    #error uC++ : internal error, swapcontext cannot be used on the x86_64 architecture
#endif

#elif defined( __ia64__ )

#if ! defined( __U_SWAPCONTEXT__ )

    // set base and limits of stack

    uLimit = (char *)uStorage + uCeiling( sizeof(__U_CONTEXT_T__), 16 );
    uBase = (char *)uLimit + uSize;

    struct uFakeStack {
	void *scratch1[2];				// abi-mandated scratch area
	struct uPreservedState {
	    void *b5;
	    void *b4;
	    void *b3;
	    void *b2;
	    void *b1;
	    void *b0;
	    void *pr;
	    void *lc;
	    void *pfs;
	    void *fpsr;
	    void *unat;
	    void *spill_unat;
	    void *rnat;
	    void *r7;
	    void *r6;
	    void *r5;
	    void *r4;
	    void *r1;
	} preserved;
	void *scratch2[2];				// abi-mandated scratch area
    };

    ((uContext_t *)uStorage)->uSP = (char *)uBase - sizeof( uFakeStack );
    ((uContext_t *)uStorage)->uBSP = (char*)uLimit + 16;
    sigemptyset( &((uContext_t *)uStorage)->uSigmask );
#ifdef __U_MULTI__
    sigaddset( &((uContext_t *)uStorage)->uSigmask, SIGALRM );
#endif // __U_MULTI__
    memset( ((uContext_t *)uStorage)->uSP, 0, sizeof( uFakeStack ) );
    ((uFakeStack *)(((uContext_t *)uStorage)->uSP))->preserved.b0 = uRtnAdr( (void (*)())uInvokeStub );
    ((uFakeStack *)(((uContext_t *)uStorage)->uSP))->preserved.r1 = uGpAdr( (void (*)())uInvokeStub );
#if defined( __INTEL_COMPILER )
    ((uFakeStack *)(((uContext_t *)uStorage)->uSP))->preserved.fpsr = (void*)__getReg( _IA64_REG_AR_FPSR );
#else
    asm ( "mov.m %0 = ar.fpsr" : "=r" (((uFakeStack *)(((uContext_t *)uStorage)->uSP))->preserved.fpsr) );
#endif // __INTEL_COMPILER
    ((uFakeStack *)(((uContext_t *)uStorage)->uSP))->preserved.r4 = this;
    ((uFakeStack *)(((uContext_t *)uStorage)->uSP))->preserved.b1 = uRtnAdr( (void (*)())uInvoke );
    ((uFakeStack *)(((uContext_t *)uStorage)->uSP))->preserved.b2 = NULL; // null terminate for stack walking
#else
    // set base and limits of stack

    uLimit = (char *)uStorage + sizeof(ucontext_t) + 16;
    uBase = (char *)uLimit + uSize;

    if ( ::getcontext( (ucontext *)uStorage ) == -1 ) {	// initialize ucontext area
	uAbort( "internal error, getcontext failed" );
    } // if
    ((ucontext *)uStorage)->uc_stack.ss_sp = (char *)uLimit;
    ((ucontext *)uStorage)->uc_stack.ss_size = uSize;
    ((ucontext *)uStorage)->uc_stack.ss_flags = 0;
    ::makecontext( (ucontext *)uStorage, (void (*)())uInvoke, 2, this );

    // TEMPORARY: getcontext doesn't get the signal mask properly
    if ( sigprocmask( SIG_BLOCK, NULL, (sigset_t*)&(((ucontext *)uStorage)->uc_sigmask) ) == -1 ) {
        uAbort( "internal error, sigprocmask" );
    } // if
#if defined( __U_MULTI__ ) && defined( __U_ONETIMER__ )
    sigaddset( (sigset_t*)&((ucontext *)uStorage)->uc_sigmask, SIGALRM );
#endif // __U_MULTI__ && __U_ONETIMER__
#endif

#elif defined( __sparc__ )

    // set base and limits of stack

    uLimit = (char *)uStorage + uCeiling( sizeof(__U_CONTEXT_T__), 16 ) + 16;
    uBase = (char *)uLimit + uSize;

#if ! defined( __U_SWAPCONTEXT__ )
    struct uFakeStack {
	void *uLocalRegs[8];
	void *uInRegs[8];
#if defined( _ILP32 )
	void *uStructRetAddress;
#endif
	void *uCalleeRegArgs[6];
    };

    ((uContext_t *)uStorage)->uFP = (char *)(SA( (unsigned long)uBase - STACK_ALIGN + 1 ) - SA( MINFRAME ) - STACK_BIAS );
    ((uContext_t *)uStorage)->uSP = (char *)((uContext_t *)uStorage)->uFP - SA( MINFRAME );
    ((uContext_t *)uStorage)->uPC = (char *)uRtnAdr( (void (*)())uInvokeStub ) - 8;

    ((uFakeStack *)((char *)((uContext_t *)uStorage)->uFP + STACK_BIAS))->uInRegs[0] = this; // argument to uInvoke
    ((uFakeStack *)((char *)((uContext_t *)uStorage)->uFP + STACK_BIAS))->uInRegs[1] = (void *)uInvoke; // uInvoke routine
    ((uFakeStack *)((char *)((uContext_t *)uStorage)->uFP + STACK_BIAS))->uInRegs[6] = NULL; // terminate stack with NULL fp
#else
    if ( ::getcontext( (ucontext *)uStorage ) == -1 ) {	// initialize ucontext area
	uAbort( " : internal error, getcontext failed" );
    } // if
    ((ucontext *)uStorage)->uc_stack.ss_sp = (char *)uBase - 8;	// TEMPORARY: -8 for bug in Solaris (fixed in Solaris 4.10)
    ((ucontext *)uStorage)->uc_stack.ss_size = uSize; 
    ((ucontext *)uStorage)->uc_stack.ss_flags = 0;
    ::makecontext( (ucontext *)uStorage, (void (*)(...))uInvoke, 2, this );
    // TEMPORARY: stack is incorrectly initialized to allow stack walking
    *((int *)uBase - 8) = 0;				// terminate stack with NULL fp
#if defined( __U_MULTI__ ) && defined( __U_ONETIMER__ )
    sigaddset( (sigset_t*)&((ucontext *)uStorage)->uc_sigmask, SIGALRM );
#endif // __U_MULTI__ && __U_ONETIMER__
#endif

#elif defined( __mips__ )

#if ! defined( __U_SWAPCONTEXT__ )
    struct uFakeStack {
	unsigned long long uFixedRegisters[11];		// fixed registers r16-r23,r30,r31,gp
	void *uDummyAlign[1];				// make multiple of 16
    };

    uLimit = (char *)uStorage + uCeiling( sizeof(__U_CONTEXT_T__), 16 ) + 16;
    uBase = (char *)uStorage + uSize;

    ((uContext_t *)uStorage)->uSP = (char *)uBase - sizeof( uFakeStack );

    ((uFakeStack *)((uContext_t *)uStorage)->uSP)->uFixedRegisters[0] = reinterpret_cast<unsigned long long>( this ); // argument to uInvoke (r16)
    ((uFakeStack *)((uContext_t *)uStorage)->uSP)->uFixedRegisters[1] = reinterpret_cast<unsigned long long>( uRtnAdr( (void (*)())uInvoke ) );	// uInvoke routine (r17)
    ((uFakeStack *)((uContext_t *)uStorage)->uSP)->uFixedRegisters[8] = 0; // terminate stack with NULL fp (r30)
    ((uFakeStack *)((uContext_t *)uStorage)->uSP)->uFixedRegisters[9] = reinterpret_cast<unsigned long long>( uRtnAdr( (void (*)())uInvokeStub ) ); // (r31)

#else
    // set base and limits of stack

    uLimit = (char *)uStorage + sizeof(ucontext_t) + 16;
    uBase = (char *)uLimit + uSize;

    if ( ::getcontext( (ucontext *)uStorage ) == -1 ) {	// initialize ucontext area
	uAbort( "internal error, getcontext failed" );
    } // if
    ((ucontext *)uStorage)->uc_stack.ss_sp = (char *)uBase - 8;	// TEMPORARY: -8 for bug in Solaris (fixed in Solaris 4.10)
    ((ucontext *)uStorage)->uc_stack.ss_size = uSize; 
    ((ucontext *)uStorage)->uc_stack.ss_flags = 0;
    ::makecontext( (ucontext *)uStorage, (void (*)())uInvoke, 2, this );
#if defined( __U_MULTI__ ) && defined( __U_ONETIMER__ )
    sigaddset( (sigset_t*)&((ucontext *)uStorage)->uc_sigmask, SIGALRM );
#endif // __U_MULTI__ && __U_ONETIMER__
#endif

#else
    #error uC++ : internal error, unsupported architecture
#endif

    // set magic words on stack

    *(unsigned int *)((char *)uLimit - 16) = uMagic();
    *(unsigned int *)uBase = uMagic();
} // uMachContext::uStartHere


void uMachContext::uCreateContext( unsigned int stacksize ) { // used by all constructors
    uSize = uCeiling( stacksize, 16 );
    uLimit = uBase = NULL;
    uErrno = 0;
    uExtras.uAllExtras = 0;
} // uMachContext::uCreateContext


uMachContext::uMachContext( unsigned int stacksize ) {
    uCreateContext( stacksize );
    // use malloc because "new" handles the overflow
    uStorage = malloc( uCeiling( sizeof(__U_CONTEXT_T__), 16 ) + uSize + 32 );
    if ( uStorage == NULL ) {				// allocate stack from heap
	uAbort( "Attempt to allocate %d bytes of storage for coroutine or task execution-state but insufficient memory available.", stacksize );
    } // if
} // uMachContext::uMachContext


uMachContext::~uMachContext() {
    free( uStorage );
} // uMachContext::~uMachContext


void *uMachContext::stackPointer() const {
    if ( &uThisCoroutine() == this ) {			// accessing myself ?
	void *sp;					// use my current stack value
#if defined( __i386__ )
	asm("movl %%esp,%0" : "=m" (sp) : );
#elif defined( __x86_64__ )
	asm("movq %%rsp,%0" : "=m" (sp) : );
#elif defined( __ia64__ )
#if defined( __INTEL_COMPILER )
	sp = (void*)__getReg( _IA64_REG_SP );
#else
	asm("mov %0 = r12" : "=r" (sp) : );
#endif // __INTEL_COMPILER
#elif defined( __sparc__ )
	asm("mov %%sp,%0" : "=r" (sp) : );
	sp = (void*)((char*)sp + STACK_BIAS);
#elif defined( __mips__ )
#ifdef _ABIN32
	asm("sw $sp,%0" : "=m" (sp) : );
#elif _ABI64
	asm("sd $sp,%0" : "=m" (sp) : );
#else
	#error uC++ : internal error, unsupported architecture
#endif
#else
	#error uC++ : internal error, unsupported architecture
#endif
	return sp;
    } else {						// accessing another coroutine
#if defined( __U_SWAPCONTEXT__ )
	return (void *)(((ucontext_t *)uStorage)->uc_mcontext.
#if defined( __linux__ ) && defined( __i386__ )
	    gregs[REG_ESP]);
#elif defined( __linux__ ) && defined( __x86_64__ )
	    gregs[REG_RSP]);
#elif defined( __linux__ ) && defined( __ia64__ )
	    sc_gr[12]);
#elif defined( __solaris__ ) && defined( __sparc__ )
	    (void*)((char*)gregs[REG_SP] + STACK_BIAS));
#elif defined( __irix__ ) &&  defined( __mips__ )
	    gregs[CTX_SP]);
#else
	    #error uC++ : internal error, unsupported architecture
#endif
#else
#if defined( __sparc__ )
	return (void*)((char*)((uContext_t *)uStorage)->uSP + STACK_BIAS);
#else
	return ((uContext_t *)uStorage)->uSP;
#endif // __sparc__
#endif
    } // if
} // uMachContext::stackPointer


#if defined( __ia64__ )
void *uMachContext::registerStackPointer() const {
#if defined( __U_SWAPCONTEXT__ )
    if ( &uThisCoroutine() == this ) {			// accessing myself ?
#if defined( __INTEL_COMPILER )
	return (void*)__getReg( _IA64_REG_AR_BSP );
#else
	void *bsp;					// use my current stack value
	asm("mov %0 = ar.bsp" : "=r" (bsp) : );
	return bsp;
#endif // __INTEL_COMPILER
    } else {						// accessing another coroutine
	return (void *)(((ucontext_t *)uStorage)->uc_mcontext.sc_ar_bsp);
    } // if
#else
    return ((uContext_t *)uStorage)->uBSP;
#endif
} // uMachContext::registerStackPointer
#endif // __ia64__


unsigned int uMachContext::stackSize() const {
    return uSize;
} // uMachContext::stackSize


ptrdiff_t uMachContext::stackFree() const {
    return (char *)stackPointer() - (char *)uLimit;
} // uMachContext::stackFree


ptrdiff_t uMachContext::stackUsed() const {
    ptrdiff_t amt = (char *)uBase - (char *)stackPointer();
    return amt < 0 ? -amt : amt;			// abs, for stacks growing up
} // uMachContext::stackUsed


void uMachContext::verify() {
    // Ignore the boot task as it uses the UNIX stack.
    if ( uStorage == ((uBaseTask *)uKernelModule::uTaskBoot)->uStorage ) return;

    if ( stackPointer() < uLimit ) {
	uAbort( "Stack overflow detected: stack pointer 0x%p below limit 0x%p.\n"
		"Possible cause is allocation of large stack frame(s) and/or deep call stack.",
		stackPointer(), uLimit );
    } else if ( (char *)stackPointer() + 4 * 1024 < uLimit ) {
	fprintf( stderr, "WARNING within 4K of stack limit: stack pointer 0x%p and limit 0x%p.\n"
		 "Possible cause is allocation of large stack frame(s) and/or deep call stack.\n",
		 stackPointer(), uLimit );
    } else if ( stackPointer() > uBase ) {
	uAbort( "Stack underflow detected.\n"
		"Possible cause is corrupted stack frame via overwriting memory." );
    } else if ( ( *(unsigned int *)uBase != uMagic() ) || ( *(unsigned int *)((char *)uLimit - 16) != uMagic() ) ) {
	uAbort( "Stack corruption detected.\n"
		"Possible cause is corrupted stack frame via overwriting memory." );
#if defined( __ia64__ )
    } else if ( registerStackPointer() >= stackPointer() ) {
        // on ia64 the stack grows from both ends; when the two stack pointers cross, we have overflow
	uAbort( "Stack overflow detected: stack pointer 0x%p at or below register stack pointer 0x%p.\n"
		"Possible cause is allocation of large stack frame(s) and/or deep call stack.",
		stackPointer(), registerStackPointer() );
#endif // __ia64__
    } // if
} // uMachContext::verify


void uMachContext::uExtraSave() {
    // Don't need to test extras bit, it is sufficent to check context list

    uContext *context;
    for ( uSeqIter<uContext> iter(uAdditionalContexts); iter >> context; ) {
	context->save();
    } // for
} // uMachContext::uExtraSave


void uMachContext::uExtraRestore() {
    // Don't need to test extras bit, it is sufficent to check context list

    uContext *context;
    for ( uSeqIter<uContext> iter(uAdditionalContexts); iter >> context; ) {
	context->restore();
    } // for
} // uMachContext::uExtraRestore


// Local Variables: //
// compile-command: "gmake install" //
// End: //
