
/*
 * Licensed Materials - Property of IBM
 *
 * trousers - An open source TCG Software Stack
 *
 * (C) Copyright International Business Machines Corp. 2004
 *
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <inttypes.h>

#include "trousers/tss.h"
#include "spi_internal_types.h"
#include "tcs_internal_types.h"
#include "tcs_tsp.h"
#include "tcsps.h"
#include "tcs_utils.h"
#include "tcs_int_literals.h"
#include "capabilities.h"
#include "tcslog.h"
#include "req_mgr.h"
#include "tcsd_wrap.h"
#include "tcsd.h"

TSS_RESULT
TCSP_SetOwnerInstall_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
			      TSS_BOOL state	/* in  */
    )
{
	UINT64 offset;
	UINT32 paramSize;
	TSS_RESULT result;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering SetOwnerInstall");
	if ((result = ctx_verify_context(hContext)))
		return result;

	offset = 10;
	LoadBlob_BOOL(&offset, state, txBlob, "State");
	LoadBlob_Header(TPM_TAG_RQU_COMMAND, offset, TPM_ORD_SetOwnerInstall,
			txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		return result;

	result = UnloadBlob_Header(txBlob, &paramSize);
	offset = 10;
	LogResult("SetOwnerInstall", result);
	return result;
}

TSS_RESULT
TCSP_TakeOwnership_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
			    UINT16 protocolID,	/* in */
			    UINT32 encOwnerAuthSize,	/* in  */
			    BYTE * encOwnerAuth,	/* in */
			    UINT32 encSrkAuthSize,	/* in */
			    BYTE * encSrkAuth,	/* in */
			    UINT32 srkInfoSize,	/*in */
			    BYTE * srkInfo,	/*in */
			    TPM_AUTH * ownerAuth,	/* in, out */
			    UINT32 * srkKeySize,	/*out */
			    BYTE ** srkKey	/*out */
    )
{
	UINT64 offset;
	UINT32 paramSize;
	TSS_RESULT result;
	TCPA_KEY srkKeyContainer;
	BYTE oldAuthDataUsage;
	UINT64 bugOffset;
	BYTE newSRK[1024];
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	if ((result = ctx_verify_context(hContext)))
		goto done;

	if ((result = auth_mgr_check(hContext, ownerAuth->AuthHandle)))
		goto done;

	/* Check on the Atmel Bug Patch */
	offset = 0;
	UnloadBlob_KEY(&offset, srkInfo, &srkKeyContainer);
	oldAuthDataUsage = srkKeyContainer.authDataUsage;
	LogDebug("auth data usage is %.2X", oldAuthDataUsage);

	offset = 10;
	LoadBlob_UINT16(&offset, protocolID, txBlob, "prot id");
	LoadBlob_UINT32(&offset, encOwnerAuthSize, txBlob,
			"enc owner auth size");
	LoadBlob(&offset, encOwnerAuthSize, txBlob, encOwnerAuth,
			"enc owner auth");
	LoadBlob_UINT32(&offset, encSrkAuthSize, txBlob,
			"srk auth size");
	LoadBlob(&offset, encSrkAuthSize, txBlob, encSrkAuth,
			"srk auth");

	LoadBlob(&offset, srkInfoSize, txBlob, srkInfo, "srk");

	LoadBlob_Auth(&offset, txBlob, ownerAuth);
	LoadBlob_Header(TPM_TAG_RQU_AUTH1_COMMAND, offset,
			TPM_ORD_TakeOwnership, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		goto done;

	result = UnloadBlob_Header(txBlob, &paramSize);

	offset = 10;
	if (result == 0) {
		if ((result = UnloadBlob_KEY(&offset, txBlob, &srkKeyContainer)))
			goto done;

		*srkKeySize = offset - 10;
		*srkKey = calloc(1, *srkKeySize);
		if (*srkKey == NULL) {
			destroy_key_refs(&srkKeyContainer);
			LogError("malloc of %u bytes failed.", *srkKeySize);
			result = TCSERR(TSS_E_OUTOFMEMORY);
			goto done;
		}
		if (srkKeyContainer.authDataUsage != oldAuthDataUsage) {
			LogDebug("AuthDataUsage was changed by TPM.  Atmel Bug. Fixing it in PS");
			srkKeyContainer.authDataUsage = oldAuthDataUsage;
		}
		memcpy(*srkKey, &txBlob[10], *srkKeySize);

		memset(srkKeyContainer.pubKey.key, 0, srkKeyContainer.pubKey.keyLength);
		bugOffset = 0;
		LoadBlob_KEY(&bugOffset, newSRK, &srkKeyContainer);

		UnloadBlob_Auth(&offset, txBlob, ownerAuth);

		/* Once the key file is created, it stays forever. There could be
		 * migratable keys in the hierarchy that are still useful to someone.
		 */
		result = ps_remove_key(&SRK_UUID);
		if (result != TSS_SUCCESS && result != TCSERR(TSS_E_PS_KEY_NOTFOUND)) {
			destroy_key_refs(&srkKeyContainer);
			LogError("Error removing SRK from key file.");
			goto done;
		}

		if ((result = ps_write_key(&SRK_UUID, &NULL_UUID, NULL, 0, newSRK, bugOffset))) {
			destroy_key_refs(&srkKeyContainer);
			LogError("Error writing SRK to disk");
			goto done;
		}
		result = mc_add_entry_srk(SRK_TPM_HANDLE, SRK_TPM_HANDLE, &srkKeyContainer);
		if (result != TSS_SUCCESS) {
			destroy_key_refs(&srkKeyContainer);
			LogError("Error creating SRK mem cache entry");
			*srkKeySize = 0;
			free(*srkKey);
		}
		destroy_key_refs(&srkKeyContainer);
	}
	LogResult("TakeOwnership", result);
done:
	auth_mgr_release_auth(ownerAuth, NULL, hContext);
	return result;
}

TSS_RESULT
TCSP_OIAP_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
		   TCS_AUTHHANDLE *authHandle,	/* out */
		   TCPA_NONCE *nonce0	/* out */
    )
{
	UINT64 offset;
	TSS_RESULT result;
	UINT32 paramSize;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering TCSI_OIAP");

	if ((result = ctx_verify_context(hContext)))
		return result;

	offset = 10;
	LoadBlob_Header(TPM_TAG_RQU_COMMAND, offset, TPM_ORD_OIAP, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		return result;

	offset = 10;
	result = UnloadBlob_Header(txBlob, &paramSize);

	if (!result) {
		UnloadBlob_UINT32(&offset, authHandle, txBlob, "authHandle");
		UnloadBlob(&offset, TCPA_NONCE_SIZE, txBlob,
				nonce0->nonce, "n0");
	}

	LogResult("OIAP", result);
	return result;
}

TSS_RESULT
TCSP_OSAP_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
		   TCPA_ENTITY_TYPE entityType,	/* in */
		   UINT32 entityValue,	/* in */
		   TCPA_NONCE nonceOddOSAP,	/* in */
		   TCS_AUTHHANDLE * authHandle,	/* out */
		   TCPA_NONCE * nonceEven,	/* out */
		   TCPA_NONCE * nonceEvenOSAP	/* out */
    )
{
	UINT64 offset;
	TSS_RESULT result;
	UINT32 paramSize;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering OSAP");
	if ((result = ctx_verify_context(hContext)))
		return result;

	offset = 10;
	LoadBlob_UINT16(&offset, entityType, txBlob, "entity type");
	LoadBlob_UINT32(&offset, entityValue, txBlob, "entity value");
	LoadBlob(&offset, TCPA_NONCE_SIZE, txBlob, nonceOddOSAP.nonce, "nonce osap");
	LoadBlob_Header(TPM_TAG_RQU_COMMAND, offset, TPM_ORD_OSAP, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		return result;

	offset = 10;

	result = UnloadBlob_Header(txBlob, &paramSize);
	if (!result) {
		UnloadBlob_UINT32(&offset, authHandle, txBlob, "auth handle");
		UnloadBlob(&offset, TCPA_NONCE_SIZE, txBlob, nonceEven->nonce, "n0");
		UnloadBlob(&offset, TCPA_NONCE_SIZE, txBlob, nonceEvenOSAP->nonce, "n0 osap");
	}
	LogResult("OSAP", result);

	return result;
}

TSS_RESULT
TCSP_ChangeAuth_Internal(TCS_CONTEXT_HANDLE contextHandle,	/* in */
			 TCS_KEY_HANDLE parentHandle,	/* in */
			 TCPA_PROTOCOL_ID protocolID,	/* in */
			 TCPA_ENCAUTH newAuth,	/* in */
			 TCPA_ENTITY_TYPE entityType,	/* in */
			 UINT32 encDataSize,	/* in */
			 BYTE *encData,	/* in */
			 TPM_AUTH *ownerAuth,	/* in, out */
			 TPM_AUTH *entityAuth,	/* in, out */
			 UINT32 *outDataSize,	/* out */
			 BYTE **outData	/* out */
    )
{
	UINT64 offset;
	UINT32 paramSize;
	TSS_RESULT result;
	TCPA_KEY_HANDLE keySlot;
	TCS_KEY_HANDLE tcsKeyHandleToEvict;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering Changeauth");
	if ((result = ctx_verify_context(contextHandle)))
		goto done;

	if ((result = auth_mgr_check(contextHandle, ownerAuth->AuthHandle)))
		goto done;
	if ((result = auth_mgr_check(contextHandle, entityAuth->AuthHandle)))
		goto done;

	if ((result = ensureKeyIsLoaded(contextHandle, parentHandle, &keySlot)))
		goto done;

	offset = 10;
	LoadBlob_UINT32(&offset, keySlot, txBlob, "handle");
	LoadBlob_UINT16(&offset, protocolID, txBlob, "prot ID");
	LoadBlob(&offset, TCPA_ENCAUTH_SIZE, txBlob,
			newAuth.authdata, "encauth");
	LoadBlob_UINT16(&offset, entityType, txBlob, "entity type");
	LoadBlob_UINT32(&offset, encDataSize, txBlob, "enc data size");
	LoadBlob(&offset, encDataSize, txBlob, encData, "enc data");
	LoadBlob_Auth(&offset, txBlob, ownerAuth);
	LoadBlob_Auth(&offset, txBlob, entityAuth);

	LoadBlob_Header(TPM_TAG_RQU_AUTH2_COMMAND, offset,
			TPM_ORD_ChangeAuth, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		goto done;

	offset = 10;
	result = UnloadBlob_Header(txBlob, &paramSize);

	if (!result) {
		UnloadBlob_UINT32(&offset, outDataSize, txBlob,
				  "out data size");
		*outData = calloc(1, *outDataSize);
		if (*outData == NULL) {
			LogError("malloc of %d bytes failed.", *outDataSize);
			result = TCSERR(TSS_E_OUTOFMEMORY);
		} else {
			UnloadBlob(&offset, *outDataSize, txBlob, *outData, "outdata");
		}
		UnloadBlob_Auth(&offset, txBlob, ownerAuth);
		UnloadBlob_Auth(&offset, txBlob, entityAuth);

		/* if the malloc above failed, terminate the 2 new auth handles and exit */
		if (result)
			goto done;

		/****************************************
		 *	Check if ET is a key.  If it is, we need to 
		 *		1 - Evict the key if loaded
		 *		2 - update the knowledge entries where applicable.
		 *		3 - Update the PS if applicable.
		 *		4 - Reload the key ( optional ) - Not doing it here
		 ****************************************/
		if (entityType == TCPA_ET_KEYHANDLE ||
		    entityType == TCPA_ET_KEY) {
			LogDebug("entity type is a key.  Check if storage/knowledge must be updated");
			/*---	Compare the EncData against the TCS tables */
#if 0
			/*---	Check PS */
			LogDebug("Checking PS");
			uuidKeyToEvict = mc_get_uuid_by_encdata(encData);
			if (uuidKeyToEvict != NULL) {
				LogDebug("UUID is not NULL, replace storage");
				replaceEncData_PS(*uuidKeyToEvict, encData, *outData);
			}
#endif
			tcsKeyHandleToEvict = mc_get_handle_by_encdata(encData);
			LogDebug("tcsKeyHandle being evicted is %.8X", tcsKeyHandleToEvict);
			/*---	If it was found in knowledge, replace it */
			if (tcsKeyHandleToEvict != 0) {
				key_mgr_evict(contextHandle, tcsKeyHandleToEvict);
				mc_update_encdata(encData, *outData);
			}

		}
	}
	LogResult("ChangeAuth", result);
done:
	auth_mgr_release_auth(ownerAuth, entityAuth, contextHandle);
	return result;
}

TSS_RESULT
TCSP_ChangeAuthOwner_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
			      TCPA_PROTOCOL_ID protocolID,	/* in */
			      TCPA_ENCAUTH newAuth,	/* in */
			      TCPA_ENTITY_TYPE entityType,	/* in */
			      TPM_AUTH * ownerAuth	/* in, out */
    )
{
	UINT64 offset;
	UINT32 paramSize;
	TSS_RESULT result;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering ChangeAuthOwner");

	if ((result = ctx_verify_context(hContext)))
		goto done;

	if ((result = auth_mgr_check(hContext, ownerAuth->AuthHandle)))
		goto done;

	offset = 10;
	LoadBlob_UINT16(&offset, protocolID, txBlob, "prot id");
	LoadBlob(&offset, 20, txBlob, newAuth.authdata, "enc auth");
	LoadBlob_UINT16(&offset, entityType, txBlob, "entity type");
	LoadBlob_Auth(&offset, txBlob, ownerAuth);
	LoadBlob_Header(TPM_TAG_RQU_AUTH1_COMMAND, offset, TPM_ORD_ChangeAuthOwner, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		goto done;

	offset = 10;
	result = UnloadBlob_Header(txBlob, &paramSize);
	UnloadBlob_Auth(&offset, txBlob, ownerAuth);

	LogResult("ChangeAuthOwner", result);
done:
	auth_mgr_release_auth(ownerAuth, NULL, hContext);
	return result;
}

TSS_RESULT
TCSP_ChangeAuthAsymStart_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
				  TCS_KEY_HANDLE idHandle,	/* in */
				  TCPA_NONCE antiReplay,	/* in */
				  UINT32 KeySizeIn,	/* in */
				  BYTE * KeyDataIn,	/* in */
				  TPM_AUTH * pAuth,	/* in, out */
				  UINT32 * KeySizeOut,	/* out */
				  BYTE ** KeyDataOut,	/* out */
				  UINT32 * CertifyInfoSize,	/* out */
				  BYTE ** CertifyInfo,	/* out */
				  UINT32 * sigSize,	/* out */
				  BYTE ** sig,	/* out */
				  TCS_KEY_HANDLE * ephHandle	/* out */
    )
{
#if 0
#warning Locking trouble in evictFirstKey
	UINT16 offset;
	UINT32 paramSize;
	TSS_RESULT result;
	UINT32 keySlot;
	TCPA_CERTIFY_INFO certifyInfo;
	TCPA_KEY tempKey;
	UINT32 tempSize;
	TCPA_KEY_PARMS keyParmsContainer;
	TSS_BOOL canLoad;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering ChangeAuthAsymStart");
	if ((result = ctx_verify_context(hContext)))
		goto done;

	if (pAuth != NULL) {
		LogDebug("Auth Command");
		if ((result = auth_mgr_check(hContext, pAuth->AuthHandle)))
			goto done;
	} else {
		LogDebug("No Auth");
	}

	if ((result = ensureKeyIsLoaded(hContext, idHandle, &keySlot)))
		goto done;

	LogDebug("Checking for room to load the eph key");
	offset = 0;
	if ((result = UnloadBlob_KEY_PARMS(&offset, KeyDataIn, &keyParmsContainer)))
		goto done;

	/* if we can't load the key, evict keys until we can */
	if ((result = canILoadThisKey(&keyParmsContainer, &canLoad)))
		goto done;

	while (canLoad == FALSE) {
		/* Evict a key that isn't the parent */
		if ((result = evictFirstKey(idHandle)))
			goto done;

		if ((result = canILoadThisKey(&keyParmsContainer, &canLoad)))
			goto done;
	}

	offset = 10;
	LoadBlob_UINT32(&offset, keySlot, txBlob, "idhandle");
	LoadBlob(&offset, TCPA_NONCE_SIZE, txBlob, antiReplay.nonce, "nonce");
/*	LoadBlob_KEY_PARMS( &offset, txBlob, &tempKeyParms ); */
/*	LoadBlob_UINT32( &offset, KeySizeIn, txBlob, "temp key size" ); */
	LoadBlob(&offset, KeySizeIn, txBlob, KeyDataIn, "Temp Key");

	if (pAuth != NULL) {
		LoadBlob_Auth(&offset, txBlob, pAuth);
		LoadBlob_Header(TPM_TAG_RQU_AUTH1_COMMAND, offset,
				TPM_ORD_ChangeAuthAsymStart, txBlob);
	} else {
		LoadBlob_Header(TPM_TAG_RQU_COMMAND, offset,
				TPM_ORD_ChangeAuthAsymStart, txBlob);
	}

	if ((result = req_mgr_submit_req(txBlob)))
		goto done;

	offset = 10;
	result = UnloadBlob_Header(txBlob, &paramSize);
	if (result == 0) {
		UnloadBlob_CERTIFY_INFO(&offset, txBlob, &certifyInfo);
		*CertifyInfoSize = offset - 10;
		*CertifyInfo = malloc(*CertifyInfoSize);
		if (*CertifyInfo == NULL) {
			LogError("malloc of %u bytes failed.", *CertifyInfoSize);
			result = TCSERR(TSS_E_OUTOFMEMORY);
			goto done;
		}
		memcpy(*CertifyInfo, &txBlob[offset - *CertifyInfoSize],
		       *CertifyInfoSize);
		UnloadBlob_UINT32(&offset, sigSize, txBlob, "sig size");
		*sig = malloc(*sigSize);
		if (*sig == NULL) {
			LogError("malloc of %u bytes failed.", *sigSize);
			result = TCSERR(TSS_E_OUTOFMEMORY);
			goto done;
		}
		UnloadBlob(&offset, *sigSize, txBlob, *sig, "sig");
		UnloadBlob_UINT32(&offset, ephHandle, txBlob, "eph handle");
		tempSize = offset;
		UnloadBlob_KEY(&offset, txBlob, &tempKey);
		*KeySizeOut = offset - tempSize;
		*KeyDataOut = malloc(*KeySizeOut);
		if (*KeyDataOut == NULL) {
			LogError("malloc of %u bytes failed.", *KeySizeOut);
			result = TCSERR(TSS_E_OUTOFMEMORY);
			goto done;
		}
		memcpy(*KeyDataOut, &txBlob[offset - *KeySizeOut], *KeySizeOut);
		if (pAuth != NULL)
			UnloadBlob_Auth(&offset, txBlob, pAuth);
	}

	LogResult("ChangeAuthAsymStart", result);
done:
	auth_mgr_release_auth(pAuth, NULL, hContext);
	return result;
#else
	return TSPERR(TSS_E_NOTIMPL);
#endif
}

TSS_RESULT
TCSP_ChangeAuthAsymFinish_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
				   TCS_KEY_HANDLE parentHandle,	/* in */
				   TCS_KEY_HANDLE ephHandle,	/* in */
				   TCPA_ENTITY_TYPE entityType,	/* in */
				   TCPA_HMAC newAuthLink,	/* in */
				   UINT32 newAuthSize,	/* in */
				   BYTE * encNewAuth,	/* in */
				   UINT32 encDataSizeIn,	/* in */
				   BYTE * encDataIn,	/* in */
				   TPM_AUTH * ownerAuth,	/* in, out */
				   UINT32 * encDataSizeOut,	/* out */
				   BYTE ** encDataOut,	/* out */
				   TCPA_SALT_NONCE * saltNonce,	/* out */
				   TCPA_DIGEST * changeProof	/* out */
    )
{
	UINT64 offset;
	UINT32 paramSize;
	TSS_RESULT result;
	UINT32 keySlot;
#if 0
	TCPA_CERTIFY_INFO certifyInfo;
	TCPA_KEY tempKey;
	UINT32 tempSize;
	TSS_UUID *uuidKeyToEvict;
#endif
	TCS_KEY_HANDLE tcsKeyHandleToEvict;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering ChangeAuthAsymFinish");
	if ((result = ctx_verify_context(hContext)))
		goto done;

	if (ownerAuth != NULL) {
		LogDebug("Auth used");
		if ((result = auth_mgr_check(hContext, ownerAuth->AuthHandle)))
			goto done;
	} else {
		LogDebug("No Auth");
	}
	if ((result = ensureKeyIsLoaded(hContext, parentHandle, &keySlot)))
		goto done;

	offset = 10;
	LoadBlob_UINT32(&offset, keySlot, txBlob, "idhandle");
	LoadBlob_UINT32(&offset, ephHandle, txBlob, "ephHandle");
	LoadBlob_UINT16(&offset, entityType, txBlob, "entity Type");
	LoadBlob(&offset, 20, txBlob, newAuthLink.digest, "newAuthLink");
	LoadBlob_UINT32(&offset, newAuthSize, txBlob, "newAuthSize");
	LoadBlob(&offset, newAuthSize, txBlob, encNewAuth, "encNewauth");
	LoadBlob_UINT32(&offset, encDataSizeIn, txBlob, "encDatasize ");
	LoadBlob(&offset, encDataSizeIn, txBlob, encDataIn, "encDataIn");

	if (ownerAuth != NULL) {
		LoadBlob_Auth(&offset, txBlob, ownerAuth);
		LoadBlob_Header(TPM_TAG_RQU_AUTH1_COMMAND, offset,
				TPM_ORD_ChangeAuthAsymFinish, txBlob);
	} else {
		LoadBlob_Header(TPM_TAG_RQU_COMMAND, offset,
				TPM_ORD_ChangeAuthAsymFinish, txBlob);
	}

	if ((result = req_mgr_submit_req(txBlob)))
		goto done;

	offset = 10;
	result = UnloadBlob_Header(txBlob, &paramSize);
	if (!result) {
		UnloadBlob_UINT32(&offset, encDataSizeOut, txBlob,
				  "outDataSize");
		*encDataOut = calloc(1, *encDataSizeOut);
		if (*encDataOut == NULL) {
			LogError("malloc of %u bytes failed.", *encDataSizeOut);
			result = TCSERR(TSS_E_OUTOFMEMORY);
			goto done;
		}
		UnloadBlob(&offset, *encDataSizeOut, txBlob,
			   *encDataOut, "outData");
		UnloadBlob(&offset, 20, txBlob, saltNonce->nonce, "salt Nonce");
		UnloadBlob(&offset, 20, txBlob, changeProof->digest,
			   "changeProof");
		if (ownerAuth != NULL)
			UnloadBlob_Auth(&offset, txBlob, ownerAuth);

		/****************************************
		 *	Check if ET is a key.  If it is, we need to 
		 *		1 - Evict the key if loaded
		 *		2 - update the knowledge entries where applicable.
		 *		3 - Update the PS if applicable.
		 *		4 - Reload the key ( optional ) - Not doing it here
		 ****************************************/
		if (entityType == TCPA_ET_KEYHANDLE ||
		    entityType == TCPA_ET_KEY) {
			/*---	Compare the EncData against the TCS tables */
			tcsKeyHandleToEvict = mc_get_handle_by_encdata(encDataIn);
#if 0
			/*---	Check PS */
			uuidKeyToEvict = mc_get_uuid_by_encdata(encDataIn);
			if (uuidKeyToEvict != NULL) {
				replaceEncData_PS(*uuidKeyToEvict,
						  encDataIn, *encDataOut);
			}
#endif
			/*---	If it was found in knowledge, replace it */
			if (tcsKeyHandleToEvict != 0) {
				key_mgr_evict(hContext, tcsKeyHandleToEvict);
				mc_update_encdata(encDataIn, *encDataOut);
			}
		}
	}

	LogResult("ChangeAuthAsymFinish", result);
done:
	auth_mgr_release_auth(ownerAuth, NULL, hContext);
	return result;
}

TSS_RESULT
internal_TerminateHandle(TCS_AUTHHANDLE handle)
{
	UINT64 offset;
	UINT32 paramSize;
	TSS_RESULT result;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	offset = 10;
	LoadBlob_UINT32(&offset, handle, txBlob, "handle");
	LoadBlob_Header(TPM_TAG_RQU_COMMAND, offset, TPM_ORD_Terminate_Handle, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		return result;

	offset = 10;
	return UnloadBlob_Header(txBlob, &paramSize);
}

TSS_RESULT
TCSP_TerminateHandle_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
			      TCS_AUTHHANDLE handle	/* in */
    )
{
	TSS_RESULT result;

	LogDebug("Entering TCSI_TerminateHandle");
	if ((result = ctx_verify_context(hContext)))
		return result;

	if ((result = auth_mgr_check(hContext, handle)))
		return result;

	result = auth_mgr_release_auth_handle(handle, hContext, 1);

	LogResult("Terminate Handle", result);
	return result;
}

TSS_RESULT
TCSP_ActivateTPMIdentity_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
				  TCS_KEY_HANDLE idKey,	/* in */
				  UINT32 blobSize,	/* in */
				  BYTE * blob,	/* in */
				  TPM_AUTH * idKeyAuth,	/* in, out */
				  TPM_AUTH * ownerAuth,	/* in, out */
				  UINT32 * SymmetricKeySize,	/* out */
				  BYTE ** SymmetricKey	/* out */
    )
{
	UINT64 offset;
	UINT64 authOffset;
	TCPA_SYMMETRIC_KEY symKey;
	TSS_RESULT result;
	UINT32 paramSize;
	UINT32 keySlot;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("TCSP_ActivateTPMIdentity");

	if ((result = ctx_verify_context(hContext)))
		goto done;

	if (idKeyAuth != NULL) {
		if ((result = auth_mgr_check(hContext, idKeyAuth->AuthHandle)))
			goto done;
	}
	if ((result = auth_mgr_check(hContext, ownerAuth->AuthHandle)))
		goto done;

	if ((result = ensureKeyIsLoaded(hContext, idKey, &keySlot)))
		goto done;

	offset = 10;
	LoadBlob_UINT32(&offset, keySlot, txBlob, "id key handle");
	LoadBlob_UINT32(&offset, blobSize, txBlob, "blob size");
	LoadBlob(&offset, blobSize, txBlob, blob, "in blob");
	if (idKeyAuth != NULL) {
		LoadBlob_Auth(&offset, txBlob, idKeyAuth);
		LoadBlob_Auth(&offset, txBlob, ownerAuth);
		LoadBlob_Header(TPM_TAG_RQU_AUTH2_COMMAND,
				offset,
				TPM_ORD_ActivateTPMIdentity, txBlob);
	} else {
		LoadBlob_Auth(&offset, txBlob, ownerAuth);
		LoadBlob_Header(TPM_TAG_RQU_AUTH1_COMMAND,
				offset,
				TPM_ORD_ActivateTPMIdentity, txBlob);
	}

	if ((result = req_mgr_submit_req(txBlob)))
		goto done;

	result = UnloadBlob_Header(txBlob, &paramSize);

	if (!result) {
		/* We don't know what kind of key the symmetric key is, or how big it is.
		 * So, call UnloadBlob_SYMMETRIC_KEY to parse through the returned data
		 * and create the expanded TCPA_SYMMETRIC_KEY structure.  Then, serialize
		 * that data to pass back to the TSP. */
		offset = 10;
		if ((result = UnloadBlob_SYMMETRIC_KEY(&offset, txBlob, &symKey)))
			goto done;

		/* After parsing through the symmetric key, offset will point to the auth
		 * structure(s) */
		authOffset = offset;

		*SymmetricKey = calloc(1, offset - 10);
		if (*SymmetricKey == NULL) {
			free(symKey.data);
			LogError("malloc of %" PRIu64 " bytes failed.", offset - 10);
			result = TCSERR(TSS_E_OUTOFMEMORY);
			goto done;
		}
		*SymmetricKeySize = offset - 10;
		offset = 0;
		LoadBlob_SYMMETRIC_KEY(&offset, *SymmetricKey, &symKey);
		free(symKey.data);

		if (idKeyAuth != NULL) {
			UnloadBlob_Auth(&authOffset, txBlob, idKeyAuth);
		}
		UnloadBlob_Auth(&authOffset, txBlob, ownerAuth);
	}

done:
	auth_mgr_release_auth(idKeyAuth, ownerAuth, hContext);
	return result;
}

TSS_RESULT
TCSP_Extend_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
		     TCPA_PCRINDEX pcrNum,	/* in */
		     TCPA_DIGEST inDigest,	/* in */
		     TCPA_PCRVALUE * outDigest	/* out */
    )
{
	UINT64 offset;
	TSS_RESULT result;
	UINT32 paramSize;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering Extend");
	if ((result = ctx_verify_context(hContext)))
		return result;

	/* PCRs are numbered 0 - (NUM_PCRS - 1), thus the >= */
	if (pcrNum >= tpm_metrics.num_pcrs)
		return TCSERR(TSS_E_BAD_PARAMETER);

	if (tcsd_options.kernel_pcrs & (1 << pcrNum)) {
		LogInfo("PCR %d is configured to be kernel controlled. Extend request denied.",
				pcrNum);
		return TCSERR(TSS_E_FAIL);
	}

	if (tcsd_options.firmware_pcrs & (1 << pcrNum)) {
		LogInfo("PCR %d is configured to be firmware controlled. Extend request denied.",
				pcrNum);
		return TCSERR(TSS_E_FAIL);
	}

	offset = 10;

	LoadBlob_UINT32(&offset, pcrNum, txBlob, "pcrNum");
	LoadBlob(&offset, TCPA_DIGEST_SIZE, txBlob, inDigest.digest,
		 "in digest");

	LoadBlob_Header(TPM_TAG_RQU_COMMAND, offset, TPM_ORD_Extend, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		return result;

	offset = 10;
	result = UnloadBlob_Header(txBlob, &paramSize);
	if (!result) {
		UnloadBlob(&offset, TCPA_DIGEST_SIZE, txBlob,
			   outDigest->digest, "digest");
	}
	LogResult("Extend", result);
	return result;
}

TSS_RESULT
TCSP_PcrRead_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
		      TCPA_PCRINDEX pcrNum,	/* in */
		      TCPA_PCRVALUE * outDigest	/* out */
    )
{
	UINT64 offset;
	TSS_RESULT result;
	UINT32 paramSize;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering PCRRead");

	if ((result = ctx_verify_context(hContext)))
		return result;

	/* PCRs are numbered 0 - (NUM_PCRS - 1), thus the >= */
	if (pcrNum >= tpm_metrics.num_pcrs)
		return TCSERR(TSS_E_BAD_PARAMETER);

	offset = 10;
	LoadBlob_UINT32(&offset, pcrNum, txBlob, "pcrnum");

	LoadBlob_Header(TPM_TAG_RQU_COMMAND, offset, TPM_ORD_PcrRead, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		return result;

	offset = 10;
	result = UnloadBlob_Header(txBlob, &paramSize);
	if (!result) {
		UnloadBlob(&offset, TCPA_DIGEST_SIZE, txBlob,
			   outDigest->digest, "digest");
	}
	LogResult("PCR Read", result);
	return result;
}

TSS_RESULT
TCSP_Quote_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
		    TCS_KEY_HANDLE keyHandle,	/* in */
		    TCPA_NONCE antiReplay,	/* in */
		    UINT32 pcrDataSizeIn,	/* in */
		    BYTE * pcrDataIn,	/* in */
		    TPM_AUTH * privAuth,	/* in, out */
		    UINT32 * pcrDataSizeOut,	/* out */
		    BYTE ** pcrDataOut,	/* out */
		    UINT32 * sigSize,	/* out */
		    BYTE ** sig	/* out */
    )
{

	UINT64 offset = 0;
	UINT32 paramSize;
	TSS_RESULT result;
	UINT32 keySlot;
	TCPA_PCR_COMPOSITE pcrComp;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering quote");

	if ((result = ctx_verify_context(hContext)))
		goto done;

	if (privAuth != NULL) {
		LogDebug("Auth Used");
		if ((result = auth_mgr_check(hContext, privAuth->AuthHandle)))
			goto done;
	} else {
		LogDebug("No Auth");
	}
	if ((result = ensureKeyIsLoaded(hContext, keyHandle, &keySlot)))
		goto done;

	offset = 10;
	LoadBlob_UINT32(&offset, keySlot, txBlob, "key handle");
	LoadBlob(&offset, TCPA_NONCE_SIZE, txBlob, antiReplay.nonce, "anti nonce");
	LoadBlob(&offset, pcrDataSizeIn, txBlob, pcrDataIn, "Pcr Data");
	if (privAuth != NULL) {
		LoadBlob_Auth(&offset, txBlob, privAuth);
		LoadBlob_Header(TPM_TAG_RQU_AUTH1_COMMAND,
				offset, TPM_ORD_Quote, txBlob);
	} else {
		LoadBlob_Header(TPM_TAG_RQU_COMMAND, offset,
				TPM_ORD_Quote, txBlob);
	}
	if ((result = req_mgr_submit_req(txBlob)))
		goto done;

	offset = 10;
	result = UnloadBlob_Header(txBlob, &paramSize);

	if (!result) {
		if ((result = UnloadBlob_PCR_COMPOSITE(&offset, txBlob, &pcrComp)))
			goto done;
		free(pcrComp.select.pcrSelect);
		free(pcrComp.pcrValue);

		*pcrDataSizeOut = offset - 10;
		*pcrDataOut = calloc(1, *pcrDataSizeOut);
		if (*pcrDataOut == NULL) {
			LogError("malloc of %u bytes failed.", *pcrDataSizeOut);
			result = TCSERR(TSS_E_OUTOFMEMORY);
			goto done;
		}
		memcpy(*pcrDataOut, &txBlob[10], *pcrDataSizeOut);
		UnloadBlob_UINT32(&offset, sigSize, txBlob, "sigsize");
		*sig = calloc(1, *sigSize);
		if (*sig == NULL) {
			free(*pcrDataOut);
			LogError("malloc of %u bytes failed.", *sigSize);
			result = TCSERR(TSS_E_OUTOFMEMORY);
			goto done;
		}
		UnloadBlob(&offset, *sigSize, txBlob, *sig, "sig");
		if (privAuth != NULL)
			UnloadBlob_Auth(&offset, txBlob, privAuth);
	}
	LogResult("Quote", result);
done:
	auth_mgr_release_auth(privAuth, NULL, hContext);
	return result;
}

TSS_RESULT
TCSP_DirWriteAuth_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
			   TCPA_DIRINDEX dirIndex,	/* in */
			   TCPA_DIRVALUE newContents,	/* in */
			   TPM_AUTH * ownerAuth	/* in, out */
    )
{
	UINT64 offset;
	UINT32 paramSize;
	TSS_RESULT result;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering dirwriteauth");
	if ((result = ctx_verify_context(hContext)))
		goto done;

	if ((result = auth_mgr_check(hContext, ownerAuth->AuthHandle)))
		goto done;

	if (dirIndex > tpm_metrics.num_dirs) {
		result = TCSERR(TSS_E_BAD_PARAMETER);
		goto done;
	}

	offset = 10;
	LoadBlob_UINT32(&offset, dirIndex, txBlob, "dir index");
	LoadBlob(&offset, TCPA_DIRVALUE_SIZE, txBlob,
			newContents.digest, "new contents");
	LoadBlob_Auth(&offset, txBlob, ownerAuth);
	LoadBlob_Header(TPM_TAG_RQU_AUTH1_COMMAND, offset,
			TPM_ORD_DirWriteAuth, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		goto done;

	offset = 10;
	result = UnloadBlob_Header(txBlob, &paramSize);

	if (!result) {
		UnloadBlob_Auth(&offset, txBlob, ownerAuth);
	}
	LogResult("DirWriteAuth", result);
done:
	auth_mgr_release_auth(ownerAuth, NULL, hContext);
	return result;
}

TSS_RESULT
TCSP_DirRead_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
		      TCPA_DIRINDEX dirIndex,	/* in */
		      TCPA_DIRVALUE * dirValue	/* out */
    )
{
	UINT64 offset;
	UINT32 paramSize;
	TSS_RESULT result;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering DirRead");
	if ((result = ctx_verify_context(hContext)))
		return result;

	if (dirValue == NULL)
		return TCSERR(TSS_E_BAD_PARAMETER);

	if (dirIndex > tpm_metrics.num_dirs)
		return TCSERR(TSS_E_BAD_PARAMETER);

	offset = 10;
	LoadBlob_UINT32(&offset, dirIndex, txBlob, "dir index");
	LoadBlob_Header(TPM_TAG_RQU_COMMAND, offset, TPM_ORD_DirRead, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		return result;

	offset = 10;
	result = UnloadBlob_Header(txBlob, &paramSize);
	if (!result) {
		UnloadBlob(&offset, TCPA_DIRVALUE_SIZE, txBlob,
			   dirValue->digest, "digest");
	}
	LogResult("DirRead", result);
	return result;
}

TSS_RESULT
TCSP_Seal_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
		   TCS_KEY_HANDLE keyHandle,	/* in */
		   TCPA_ENCAUTH encAuth,	/* in */
		   UINT32 pcrInfoSize,	/* in */
		   BYTE * PcrInfo,	/* in */
		   UINT32 inDataSize,	/* in */
		   BYTE * inData,	/* in */
		   TPM_AUTH * pubAuth,	/* in, out */
		   UINT32 * SealedDataSize,	/* out */
		   BYTE ** SealedData	/* out */
    )
{
	UINT64 offset;
	TSS_RESULT result;
	UINT32 paramSize;
	TCPA_KEY_HANDLE keySlot;
	TCPA_STORED_DATA storedData;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering Seal");
	if (!pubAuth)
		return TCSERR(TSS_E_BAD_PARAMETER);

	if ((result = ctx_verify_context(hContext)))
		goto done;

	if ((result = auth_mgr_check(hContext, pubAuth->AuthHandle)))
		goto done;

	if ((result = ensureKeyIsLoaded(hContext, keyHandle, &keySlot)))
		goto done;

	if (keySlot == 0) {
		result = TCSERR(TSS_E_FAIL);
		goto done;
	}

	offset = 10;
	LoadBlob_UINT32(&offset, keySlot, txBlob, "handle");
	LoadBlob(&offset, TCPA_ENCAUTH_SIZE, txBlob,
			encAuth.authdata, "encauth");
	LoadBlob_UINT32(&offset, pcrInfoSize, txBlob, "pcr info size");
	LoadBlob(&offset, pcrInfoSize, txBlob, PcrInfo, "pcr info");
	LoadBlob_UINT32(&offset, inDataSize, txBlob, "in data size");
	LoadBlob(&offset, inDataSize, txBlob, inData, "in data");

	LoadBlob_Auth(&offset, txBlob, pubAuth);
	LoadBlob_Header(TPM_TAG_RQU_AUTH1_COMMAND, offset, TPM_ORD_Seal, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		goto done;

	offset = 10;
	result = UnloadBlob_Header(txBlob, &paramSize);

	if (!result) {
		if ((result = UnloadBlob_STORED_DATA(&offset, txBlob, &storedData)))
			goto done;
		free(storedData.sealInfo);
		free(storedData.encData);

		*SealedDataSize = offset - 10;
		*SealedData = calloc(1, *SealedDataSize);
		if (*SealedData == NULL) {
			LogError("malloc of %u bytes failed.", *SealedDataSize);
			result = TCSERR(TSS_E_OUTOFMEMORY);
			goto done;
		}
		memcpy(*SealedData, &txBlob[10], *SealedDataSize);
		UnloadBlob_Auth(&offset, txBlob, pubAuth);
	}
	LogResult("Seal", result);
done:
	auth_mgr_release_auth(pubAuth, NULL, hContext);
	return result;
}

TSS_RESULT
TCSP_Unseal_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
		     TCS_KEY_HANDLE parentHandle,	/* in */
		     UINT32 SealedDataSize,	/* in */
		     BYTE * SealedData,	/* in */
		     TPM_AUTH * parentAuth,	/* in, out */
		     TPM_AUTH * dataAuth,	/* in, out */
		     UINT32 * DataSize,	/* out */
		     BYTE ** Data	/* out */
    )
{
	UINT64 offset;
	UINT32 paramSize;
	TSS_RESULT result;
	TCPA_KEY_HANDLE keySlot;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering Unseal");

	if (dataAuth == NULL)
		return TCSERR(TSS_E_BAD_PARAMETER);

	if ((result = ctx_verify_context(hContext)))
		goto done;

	if (parentAuth != NULL) {
		LogDebug("Auth used");
		if ((result = auth_mgr_check(hContext, parentAuth->AuthHandle)))
			goto done;
	} else {
		LogDebug("No Auth");
	}

	if ((result = auth_mgr_check(hContext, dataAuth->AuthHandle)))
		goto done;

	if ((result = ensureKeyIsLoaded(hContext, parentHandle, &keySlot)))
		goto done;

	if (keySlot == 0) {
		result = TCSERR(TSS_E_FAIL);
		goto done;
	}

	offset = 10;
	LoadBlob_UINT32(&offset, keySlot, txBlob, "handle");
	LoadBlob(&offset, SealedDataSize, txBlob, SealedData, "data");
	if (parentAuth != NULL) {
		LoadBlob_Auth(&offset, txBlob, parentAuth);
		LoadBlob_Auth(&offset, txBlob, dataAuth);
		LoadBlob_Header(TPM_TAG_RQU_AUTH2_COMMAND,
				offset, TPM_ORD_Unseal, txBlob);
	} else {
		LoadBlob_Auth(&offset, txBlob, dataAuth);
		LoadBlob_Header(TPM_TAG_RQU_AUTH1_COMMAND,
				offset, TPM_ORD_Unseal, txBlob);
	}
	if ((result = req_mgr_submit_req(txBlob)))
		goto done;

	offset = 10;
	result = UnloadBlob_Header(txBlob, &paramSize);

	if (!result) {
		UnloadBlob_UINT32(&offset, DataSize, txBlob,
				  "sealed data size");
		*Data = calloc(1, *DataSize);
		if (*Data == NULL) {
			LogError("malloc of %u bytes failed.", *DataSize);
			result = TCSERR(TSS_E_OUTOFMEMORY);
			goto done;
		}
		UnloadBlob(&offset, *DataSize, txBlob, *Data, "sealed data");
		if (parentAuth != NULL)
			UnloadBlob_Auth(&offset, txBlob, parentAuth);
		UnloadBlob_Auth(&offset, txBlob, dataAuth);
	}
	LogResult("Unseal", result);
done:
	auth_mgr_release_auth(parentAuth, dataAuth, hContext);
	return result;
}

TSS_RESULT
TCSP_UnBind_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
		     TCS_KEY_HANDLE keyHandle,	/* in */
		     UINT32 inDataSize,	/* in */
		     BYTE * inData,	/* in */
		     TPM_AUTH * privAuth,	/* in, out */
		     UINT32 * outDataSize,	/* out */
		     BYTE ** outData	/* out */
    )
{
	UINT32 paramSize;
	TSS_RESULT result;
	UINT64 offset;
	TCPA_KEY_HANDLE keySlot;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering TCSI_UnBind");
	if ((result = ctx_verify_context(hContext)))
		goto done;

	if (privAuth != NULL) {
		LogDebug("Auth Used");
		if ((result = auth_mgr_check(hContext, privAuth->AuthHandle)))
			goto done;
	} else {
		LogDebug("No Auth");
	}

	LogDebugFn("calling ensureKeyIsLoaded for TCS handle 0x%x",
		   keyHandle);
	if ((result = ensureKeyIsLoaded(hContext, keyHandle, &keySlot)))
		goto done;

	offset = 10;
	LoadBlob_UINT32(&offset, keySlot, txBlob, "key handle");
	LoadBlob_UINT32(&offset, inDataSize, txBlob, "data size");
	LoadBlob(&offset, inDataSize, txBlob, inData, "in data");
	if (privAuth != NULL) {
		LoadBlob_Auth(&offset, txBlob, privAuth);
		LoadBlob_Header(TPM_TAG_RQU_AUTH1_COMMAND,
				offset, TPM_ORD_UnBind, txBlob);
	} else
		LoadBlob_Header(TPM_TAG_RQU_COMMAND, offset,
				TPM_ORD_UnBind, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		goto done;

	offset = 10;
	result = UnloadBlob_Header(txBlob, &paramSize);

	if (!result) {
		UnloadBlob_UINT32(&offset, outDataSize, txBlob, "out data size");
		*outData = calloc(1, *outDataSize);
		if (*outData == NULL) {
			LogError("malloc of %d bytes failed.", *outDataSize);
			result = TCSERR(TSS_E_OUTOFMEMORY);
		} else {
			UnloadBlob(&offset, (*outDataSize), txBlob, *outData, "out data");
		}
		if (privAuth != NULL)
			UnloadBlob_Auth(&offset, txBlob, privAuth);
	}

done:
	auth_mgr_release_auth(privAuth, NULL, hContext);
	return result;
}

TSS_RESULT
TCSP_CreateMigrationBlob_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
				  TCS_KEY_HANDLE parentHandle,	/* in */
				  TSS_MIGRATE_SCHEME migrationType,	/* in */
				  UINT32 MigrationKeyAuthSize,	/* in */
				  BYTE * MigrationKeyAuth,	/* in */
				  UINT32 encDataSize,	/* in */
				  BYTE * encData,	/* in */
				  TPM_AUTH * parentAuth,	/* in, out */
				  TPM_AUTH * entityAuth,	/* in, out */
				  UINT32 * randomSize,	/* out */
				  BYTE ** random,	/* out */
				  UINT32 * outDataSize,	/* out */
				  BYTE ** outData	/* out */
    )
{
	UINT64 offset;
	UINT32 paramSize;
	TSS_RESULT result;
	TCPA_KEY_HANDLE keyHandle;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering TPM_CreateMigrationBlob");

	if ((result = ctx_verify_context(hContext)))
		goto done;

	if (parentAuth != NULL) {
		if ((result = auth_mgr_check(hContext, parentAuth->AuthHandle)))
			goto done;
	}

	if ((result = auth_mgr_check(hContext, entityAuth->AuthHandle)))
		goto done;

	if ((result = ensureKeyIsLoaded(hContext, parentHandle, &keyHandle)))
		goto done;

	switch (migrationType) {
		case TSS_MS_MIGRATE:
			migrationType = TCPA_MS_MIGRATE;
			break;
		case TSS_MS_REWRAP:
			migrationType = TCPA_MS_REWRAP;
			break;
		case TSS_MS_MAINT:
			migrationType = TCPA_MS_MAINT;
			break;
		default:
			/* Let the TPM return an error */
			break;
	}

	offset = 10;
	LoadBlob_UINT32(&offset, keyHandle, txBlob, "parent handle");
	LoadBlob_UINT16(&offset, migrationType, txBlob, "mig type");
	LoadBlob(&offset, MigrationKeyAuthSize, txBlob,
			MigrationKeyAuth, "mig key auth");
	LoadBlob_UINT32(&offset, encDataSize, txBlob, "enc size");
	LoadBlob(&offset, encDataSize, txBlob, encData, "enc data");
	if (parentAuth) {
		LoadBlob_Auth(&offset, txBlob, parentAuth);
		LoadBlob_Auth(&offset, txBlob, entityAuth);
		LoadBlob_Header(TPM_TAG_RQU_AUTH2_COMMAND, offset,
				TPM_ORD_CreateMigrationBlob, txBlob);
	} else {
		LoadBlob_Auth(&offset, txBlob, entityAuth);
		LoadBlob_Header(TPM_TAG_RQU_AUTH1_COMMAND, offset,
				TPM_ORD_CreateMigrationBlob, txBlob);
	}

	if ((result = req_mgr_submit_req(txBlob)))
		goto done;

	offset = 10;
	result = UnloadBlob_Header(txBlob, &paramSize);

	if (result == TSS_SUCCESS) {
		UnloadBlob_UINT32(&offset, randomSize, txBlob, "random size");
		*random = calloc(1, *randomSize);
		if (*random == NULL) {
			LogError("malloc of %u bytes failed.", *randomSize);
			result = TCSERR(TSS_E_OUTOFMEMORY);
			goto done;
		}

		UnloadBlob(&offset, *randomSize, txBlob, *random, "random");
		UnloadBlob_UINT32(&offset, outDataSize, txBlob, "out data size");
		*outData = calloc(1, *outDataSize);
		if (*outData == NULL) {
			free(*random);
			LogError("malloc of %u bytes failed.", *outDataSize);
			result = TCSERR(TSS_E_OUTOFMEMORY);
			goto done;
		} else {
			UnloadBlob(&offset, *outDataSize, txBlob, *outData, "out data");
		}
		if (parentAuth != NULL)
			UnloadBlob_Auth(&offset, txBlob, parentAuth);
		UnloadBlob_Auth(&offset, txBlob, entityAuth);
	}
	LogResult("TPM_CreateMigrationBlob", result);

done:
	auth_mgr_release_auth(entityAuth, parentAuth, hContext);
	return result;
}

TSS_RESULT
TCSP_ConvertMigrationBlob_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
				   TCS_KEY_HANDLE parentHandle,	/* in */
				   UINT32 inDataSize,	/* in */
				   BYTE * inData,	/* in */
				   UINT32 randomSize,	/* in */
				   BYTE * random,	/* in */
				   TPM_AUTH * parentAuth,	/* in, out */
				   UINT32 * outDataSize,	/* out */
				   BYTE ** outData	/* out */
    )
{
	TSS_RESULT result;
	UINT32 paramSize;
	UINT64 offset;
	TCPA_KEY_HANDLE keySlot;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("ConvertMigBlob");
	if ((result = ctx_verify_context(hContext)))
		goto done;

	if (parentAuth != NULL) {
		LogDebug("Auth Used");
		if ((result = auth_mgr_check(hContext, parentAuth->AuthHandle)))
			goto done;
	} else {
		LogDebug("No Auth");
	}
	if ((result = ensureKeyIsLoaded(hContext, parentHandle, &keySlot)))
		goto done;

	offset = 10;
	LoadBlob_UINT32(&offset, keySlot, txBlob, "parent handle");
	LoadBlob_UINT32(&offset, inDataSize, txBlob, "in data size");
	LoadBlob(&offset, inDataSize, txBlob, inData, "in data");
	LoadBlob_UINT32(&offset, randomSize, txBlob, "random size");
	LoadBlob(&offset, randomSize, txBlob, random, "random");
	if (parentAuth != NULL) {
		LoadBlob_Auth(&offset, txBlob, parentAuth);
		LoadBlob_Header(TPM_TAG_RQU_AUTH1_COMMAND,
				offset,
				TPM_ORD_ConvertMigrationBlob, txBlob);
	} else {
		LoadBlob_Header(TPM_TAG_RQU_COMMAND, offset,
				TPM_ORD_ConvertMigrationBlob, txBlob);
	}

	if ((result = req_mgr_submit_req(txBlob)))
		goto done;

	offset = 10;
	result = UnloadBlob_Header(txBlob, &paramSize);

	if (!result) {
		UnloadBlob_UINT32(&offset, outDataSize, txBlob,
				  "out data size");
		*outData = calloc(1, *outDataSize);
		if (*outData == NULL) {
			LogError("malloc of %d bytes failed.", *outDataSize);
			result = TCSERR(TSS_E_OUTOFMEMORY);
		} else {
			UnloadBlob(&offset, *outDataSize, txBlob, *outData, "out data");
		}
		if (parentAuth != NULL) {
			UnloadBlob_Auth(&offset, txBlob, parentAuth);
		}
	}
	LogResult("***Leaving ConvertMigrationBlob with result ", result);
done:
	auth_mgr_release_auth(parentAuth, NULL, hContext);
	return result;
}

TSS_RESULT
TCSP_AuthorizeMigrationKey_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
				    TSS_MIGRATE_SCHEME migrateScheme,	/* in */
				    UINT32 MigrationKeySize,	/* in */
				    BYTE * MigrationKey,	/* in */
				    TPM_AUTH * ownerAuth,	/* in, out */
				    UINT32 * MigrationKeyAuthSize,	/* out */
				    BYTE ** MigrationKeyAuth	/* out */
    )
{

	TSS_RESULT result;
	UINT32 paramSize;
	UINT64 offset;
	TCPA_MIGRATIONKEYAUTH container;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("TCSP_AuthorizeMigrationKey");
	if ((result = ctx_verify_context(hContext)))
		goto done;

	if ((result = auth_mgr_check(hContext, ownerAuth->AuthHandle)))
		goto done;

	switch (migrateScheme) {
		case TSS_MS_MIGRATE:
			migrateScheme = TCPA_MS_MIGRATE;
			break;
		case TSS_MS_REWRAP:
			migrateScheme = TCPA_MS_REWRAP;
			break;
		case TSS_MS_MAINT:
			migrateScheme = TCPA_MS_MAINT;
			break;
		default:
			/* Let the TPM return an error */
			break;
	}

	offset = 10;
	LoadBlob_UINT16(&offset, migrateScheme, txBlob, "migation scheme");
	LoadBlob(&offset, MigrationKeySize, txBlob, MigrationKey, "pubKey");
	LoadBlob_Auth(&offset, txBlob, ownerAuth);
	LoadBlob_Header(TPM_TAG_RQU_AUTH1_COMMAND, offset,
			TPM_ORD_AuthorizeMigrationKey, txBlob);
	if ((result = req_mgr_submit_req(txBlob)))
		goto done;

	offset = 10;
	result = UnloadBlob_Header(txBlob, &paramSize);

	if (!result) {
		if ((result = UnloadBlob_MIGRATIONKEYAUTH(&offset, txBlob, &container)))
			goto done;
		free(container.migrationKey.pubKey.key);
		free(container.migrationKey.algorithmParms.parms);

		*MigrationKeyAuthSize = offset - 10;
		*MigrationKeyAuth = calloc(1, *MigrationKeyAuthSize);
		if (*MigrationKeyAuth == NULL) {
			LogError("malloc of %d bytes failed.", *MigrationKeyAuthSize);
			result = TCSERR(TSS_E_OUTOFMEMORY);
		} else {
			memcpy(*MigrationKeyAuth, &txBlob[10], *MigrationKeyAuthSize);
		}

		UnloadBlob_Auth(&offset, txBlob, ownerAuth);
	}
	LogDebugFn("TPM_AuthorizeMigrationKey result: 0x%x", result);
done:
	auth_mgr_release_auth(ownerAuth, NULL, hContext);
	return result;

}

TSS_RESULT
TCSP_CertifyKey_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
			 TCS_KEY_HANDLE certHandle,	/* in */
			 TCS_KEY_HANDLE keyHandle,	/* in */
			 TCPA_NONCE antiReplay,	/* in */
			 TPM_AUTH * certAuth,	/* in, out */
			 TPM_AUTH * keyAuth,	/* in, out */
			 UINT32 * CertifyInfoSize,	/* out */
			 BYTE ** CertifyInfo,	/* out */
			 UINT32 * outDataSize,	/* out */
			 BYTE ** outData	/* out */
    )
{
	UINT64 offset;
	UINT32 paramSize;
	TSS_RESULT result;
	TCPA_KEY_HANDLE certKeySlot, keySlot;
	TCPA_CERTIFY_INFO certifyContainer;
	UINT16 tag;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering Certify Key");
	offset = 10;
	if ((result = ctx_verify_context(hContext)))
		goto done;

	if (certAuth != NULL) {
		LogDebug("Auth Used for Cert signing key");
		if ((result = auth_mgr_check(hContext, certAuth->AuthHandle)))
			goto done;
	} else {
		LogDebug("No Auth used for Cert signing key");
	}

	if (keyAuth != NULL) {
		LogDebug("Auth Used for Key being signed");
		if ((result = auth_mgr_check(hContext, keyAuth->AuthHandle)))
			goto done;
	} else {
		LogDebug("No Auth used for Key being signed");
	}

	if ((result = ensureKeyIsLoaded(hContext, certHandle, &certKeySlot)))
		goto done;

	if ((result = ensureKeyIsLoaded(hContext, keyHandle, &keySlot)))
		goto done;

	LoadBlob_UINT32(&offset, certKeySlot, txBlob, "cert handle");
	LoadBlob_UINT32(&offset, keySlot, txBlob, "key handle");
	LoadBlob(&offset, TCPA_NONCE_SIZE, txBlob,
			antiReplay.nonce, "anti replay");

	tag = TPM_TAG_RQU_COMMAND;
	if (certAuth != NULL) {
		tag++;
		LoadBlob_Auth(&offset, txBlob, certAuth);
	}
	if (keyAuth != NULL) {
		tag++;
		LoadBlob_Auth(&offset, txBlob, keyAuth);
	}
	LoadBlob_Header(tag, offset, TPM_ORD_CertifyKey, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		goto done;

	offset = 10;

	result = UnloadBlob_Header(txBlob, &paramSize);

	if (!result) {
		if ((result = UnloadBlob_CERTIFY_INFO(&offset, txBlob, &certifyContainer)))
			goto done;
		free(certifyContainer.algorithmParms.parms);
		free(certifyContainer.PCRInfo);

		*CertifyInfoSize = offset - 10;
		*CertifyInfo = calloc(1, *CertifyInfoSize);
		if (*CertifyInfo == NULL) {
			LogError("malloc of %u bytes failed.", *CertifyInfoSize);
			result = TCSERR(TSS_E_OUTOFMEMORY);
			goto done;
		} else {
			memcpy(*CertifyInfo, &txBlob[10], *CertifyInfoSize);
		}

		UnloadBlob_UINT32(&offset, outDataSize, txBlob, "out data size");
		*outData = calloc(1, *outDataSize);
		if (*outData == NULL) {
			free(*CertifyInfo);
			LogError("malloc of %u bytes failed.", *outDataSize);
			result = TCSERR(TSS_E_OUTOFMEMORY);
			goto done;
		} else {
			UnloadBlob(&offset, *outDataSize, txBlob, *outData, "out data");
		}

		if (certAuth != NULL) {
			UnloadBlob_Auth(&offset, txBlob, certAuth);
		}
		if (keyAuth != NULL) {
			UnloadBlob_Auth(&offset, txBlob, keyAuth);
		}
	}
	LogResult("Certify Key", result);
done:
	auth_mgr_release_auth(certAuth, keyAuth, hContext);
	return result;
}

TSS_RESULT
TCSP_Sign_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
		   TCS_KEY_HANDLE keyHandle,	/* in */
		   UINT32 areaToSignSize,	/* in */
		   BYTE * areaToSign,	/* in */
		   TPM_AUTH * privAuth,	/* in, out */
		   UINT32 * sigSize,	/* out */
		   BYTE ** sig	/* out */
    )
{
	UINT64 offset;
	UINT32 paramSize;
	TSS_RESULT result;
	TCPA_KEY_HANDLE keySlot;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering Sign");
	if ((result = ctx_verify_context(hContext)))
		return result;

	if (privAuth != NULL) {
		LogDebug("Auth Used");
		if ((result = auth_mgr_check(hContext, privAuth->AuthHandle)))
			goto done;
	} else {
		LogDebug("No Auth");
	}

	if ((result = ensureKeyIsLoaded(hContext, keyHandle, &keySlot)))
		goto done;

	offset = 10;

	LoadBlob_UINT32(&offset, keySlot, txBlob, "key handle");
	LoadBlob_UINT32(&offset, areaToSignSize, txBlob, "size");
	LoadBlob(&offset, areaToSignSize, txBlob, areaToSign,
			"area to sign");
	if (privAuth != NULL) {
		LoadBlob_Auth(&offset, txBlob, privAuth);
		LoadBlob_Header(TPM_TAG_RQU_AUTH1_COMMAND,
				offset, TPM_ORD_Sign, txBlob);
	} else {
		LoadBlob_Header(TPM_TAG_RQU_COMMAND, offset,
				TPM_ORD_Sign, txBlob);
	}

	if ((result = req_mgr_submit_req(txBlob)))
		goto done;

	offset = 10;
	result = UnloadBlob_Header(txBlob, &paramSize);

	if (!result) {
		UnloadBlob_UINT32(&offset, sigSize, txBlob, "sig size");
		*sig = calloc(1, *sigSize);
		if (*sig == NULL) {
			LogError("malloc of %d bytes failed.", *sigSize);
			result = TCSERR(TSS_E_OUTOFMEMORY);
			goto done;
		}
		UnloadBlob(&offset, *sigSize, txBlob, *sig, "sig");
		if (privAuth != NULL)
			UnloadBlob_Auth(&offset, txBlob, privAuth);
	}
	LogResult("sign", result);
done:
	auth_mgr_release_auth(privAuth, NULL, hContext);
	return result;
}

/*
 * Get a random number generated by the TPM.  Most (all?) TPMs return a maximum number of random
 * bytes that's less than the max allowed to be returned at the TSP level, which is 4K bytes.
 * According to the TPM compliance work posted here: http://www.prosec.rub.de/tpmcompliance.html,
 * some TPMs return as little as 132 bytes per query, which would require about 30 loops to get 4K.
 * We'll be extremely conservative here and loop 50 times, since it won't affect performance on
 * TPMs that return more bytes.
 */
TSS_RESULT
TCSP_GetRandom_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
			UINT32 * bytesRequested,	/* in, out */
			BYTE ** randomBytes	/* out */
    )
{
	UINT64 offset;
	TSS_RESULT result;
	UINT32 paramSize, totalReturned = 0, bytesReturned, retries = 50;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE], *rnd_tmp = NULL;

	LogDebugFn("%u bytes", *bytesRequested);

	if ((result = ctx_verify_context(hContext)))
		return result;

	do {
		offset = 10;
		LoadBlob_UINT32(&offset, *bytesRequested - totalReturned,
				txBlob, "requested");
		LoadBlob_Header(TPM_TAG_RQU_COMMAND, offset, TPM_ORD_GetRandom,
				txBlob);

		if ((result = req_mgr_submit_req(txBlob)))
			return result;

		offset = 10;
		result = UnloadBlob_Header(txBlob, &paramSize);
		if (!result) {
			UnloadBlob_UINT32(&offset, &bytesReturned, txBlob,
					  "random bytes size");

			LogDebugFn("received %u bytes from the TPM",
				   bytesReturned);

			rnd_tmp = realloc(rnd_tmp, totalReturned + bytesReturned);
			if (rnd_tmp == NULL) {
				LogError("malloc of %d bytes failed.",
					 bytesReturned);
				return TCSERR(TSS_E_OUTOFMEMORY);
			}
			UnloadBlob(&offset, bytesReturned, txBlob,
				   &rnd_tmp[totalReturned], "random bytes");
			totalReturned += bytesReturned;
		} else {
			free(rnd_tmp);
			return result;
		}
	} while (totalReturned < *bytesRequested && retries--);

	if (totalReturned != *bytesRequested) {
		LogDebugFn("Only %u random bytes recieved from TPM.", totalReturned);
		free(rnd_tmp);
		result = TCSERR(TSS_E_FAIL);
	} else
		*randomBytes = rnd_tmp;

	return result;
}

TSS_RESULT
TCSP_StirRandom_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
			 UINT32 inDataSize,	/* in */
			 BYTE * inData	/* in */
    )
{
	UINT64 offset;
	UINT32 paramSize;
	TSS_RESULT result;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering stir random");

	if ((result = ctx_verify_context(hContext)))
		return result;

	if (inDataSize > 255) {
		LogDebugFn("inData is too large! (%u bytes)", inDataSize);
		return TCSERR(TSS_E_BAD_PARAMETER);
	}

	offset = 10;
	LoadBlob_UINT32(&offset, inDataSize, txBlob, "in data size");
	LoadBlob(&offset, inDataSize, txBlob, inData, "in data");
	LoadBlob_Header(TPM_TAG_RQU_COMMAND, offset, TPM_ORD_StirRandom,
			txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		return result;

	offset = 10;
	result = UnloadBlob_Header(txBlob, &paramSize);
	LogResult("Stir random", result);
	return result;
}

TSS_RESULT
internal_TCSGetCap(TCS_CONTEXT_HANDLE hContext,
		   TCPA_CAPABILITY_AREA capArea,
		   UINT32 subCapSize, BYTE * subCap,
		   UINT32 * respSize, BYTE ** resp)
{
	UINT32 tcsSubCapContainer;
	UINT64 offset;
	TSS_RESULT result;
	TCPA_VERSION tcsVersion = INTERNAL_CAP_TCS_VERSION;

	if ((result = ctx_verify_context(hContext)))
		return result;

	LogDebug("Checking Software Cap of TCS");
	switch (capArea) {
	case TSS_TCSCAP_ALG:
		LogDebug("TSS_TCSCAP_ALG");
		tcsSubCapContainer = Decode_UINT32(subCap);
		*respSize = 1;
		*resp = malloc(1);
		if (*resp == NULL) {
			LogError("malloc of %d bytes failed.", 1);
			return TCSERR(TSS_E_OUTOFMEMORY);
		}
		switch (tcsSubCapContainer) {
		case TSS_ALG_RSA:
			(*resp)[0] = INTERNAL_CAP_TCS_ALG_RSA;
			break;
		case TSS_ALG_DES:
			(*resp)[0] = INTERNAL_CAP_TCS_ALG_DES;
			break;
		case TSS_ALG_3DES:
			(*resp)[0] = INTERNAL_CAP_TCS_ALG_3DES;
			break;
		case TSS_ALG_SHA:
			(*resp)[0] = INTERNAL_CAP_TCS_ALG_SHA;
			break;
		case TSS_ALG_AES:
			(*resp)[0] = INTERNAL_CAP_TCS_ALG_AES;
			break;
		case TSS_ALG_HMAC:
			(*resp)[0] = INTERNAL_CAP_TCS_ALG_HMAC;
			break;
		default:
			free(*resp);
			*resp = NULL;
			return TCSERR(TSS_E_FAIL);
		}
		break;
	case TSS_TCSCAP_VERSION:
		LogDebug("TSS_TCSCAP_VERSION");
		*resp = calloc(1, 4);
		if (*resp == NULL) {
			LogError("malloc of %d bytes failed.", 4);
			return TCSERR(TSS_E_OUTOFMEMORY);
		}
		offset = 0;
		LoadBlob_VERSION(&offset, *resp, &tcsVersion);
		*respSize = offset;
		break;
	case TSS_TCSCAP_PERSSTORAGE:
		LogDebug("TSS_TCSCAP_PERSSTORAGE");
		*respSize = 1;
		*resp = malloc(1);
		if (*resp == NULL) {
			LogError("malloc of %d byte failed.", 1);
			return TCSERR(TSS_E_OUTOFMEMORY);
		}
		(*resp)[0] = INTERNAL_CAP_TCS_PERSSTORAGE;
		break;

	case TSS_TCSCAP_CACHING:
		LogDebug("TSS_TCSCAP_CACHING");
		tcsSubCapContainer = Decode_UINT32(subCap);
		if (tcsSubCapContainer == TSS_TCSCAP_PROP_KEYCACHE) {
			*respSize = 1;
			*resp = malloc(1);
			if (*resp == NULL) {
				LogError("malloc of %d byte failed.", 1);
				return TCSERR(TSS_E_OUTOFMEMORY);
			}
			(*resp)[0] = INTERNAL_CAP_TCS_CACHING_KEYCACHE;
		} else if (tcsSubCapContainer == TSS_TCSCAP_PROP_AUTHCACHE) {
			*respSize = 1;
			*resp = malloc(1);
			if (*resp == NULL) {
			LogError("malloc of %d byte failed.", 1);
				return TCSERR(TSS_E_OUTOFMEMORY);
			}
			(*resp)[0] = INTERNAL_CAP_TCS_CACHING_AUTHCACHE;
		} else {
			LogDebugFn("Bad subcap");
			return TCSERR(TSS_E_FAIL);
		}
		break;
	case TSS_TCSCAP_MANUFACTURER:
		tcsSubCapContainer = Decode_UINT32(subCap);
		if (tcsSubCapContainer == TSS_TCSCAP_PROP_MANUFACTURER_ID) {
			*respSize = sizeof(UINT32);
			*resp = malloc(sizeof(UINT32));
			if (*resp == NULL) {
				LogError("malloc of %zd byte failed.", sizeof(UINT32));
				return TCSERR(TSS_E_OUTOFMEMORY);
			}
			*(UINT32 *)(*resp) = INTERNAL_CAP_TCS_MANUFACTURER_ID;
		} else if (tcsSubCapContainer == TSS_TCSCAP_PROP_MANUFACTURER_STR) {
			BYTE str[] = INTERNAL_CAP_TCS_MANUFACTURER_STR;

			*respSize = INTERNAL_CAP_TCS_MANUFACTURER_STR_LEN;
			*resp = malloc(INTERNAL_CAP_TCS_MANUFACTURER_STR_LEN);
			if (*resp == NULL) {
				LogError("malloc of %d byte failed.", 1);
				return TCSERR(TSS_E_OUTOFMEMORY);
			}
			memcpy(*resp, str, INTERNAL_CAP_TCS_MANUFACTURER_STR_LEN);
		} else {
			LogDebugFn("Bad subcap");
			return TCSERR(TSS_E_FAIL);
		}
		break;
	default:
		return TCSERR(TSS_E_FAIL);
	}

	return TSS_SUCCESS;
}

TSS_RESULT
TCS_GetCapability_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
			   TCPA_CAPABILITY_AREA capArea,	/* in */
			   UINT32 subCapSize,	/* in */
			   BYTE * subCap,	/* in */
			   UINT32 * respSize,	/* out */
			   BYTE ** resp	/* out */
    )
{
	TSS_RESULT result;

	if ((result = ctx_verify_context(hContext)))
		return result;

	return internal_TCSGetCap(hContext, capArea, subCapSize, subCap,
				  respSize, resp);
}

TSS_RESULT
TCSP_GetCapability_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
			    TCPA_CAPABILITY_AREA capArea,	/* in */
			    UINT32 subCapSize,	/* in */
			    BYTE * subCap,	/* in */
			    UINT32 * respSize,	/* out */
			    BYTE ** resp	/* out */
    )
{
	UINT64 offset;
	UINT32 paramSize;
	TSS_RESULT result;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	if ((result = ctx_verify_context(hContext)))
		return result;

	LogDebug("Entering Get Cap");
	offset = 10;
	LoadBlob_UINT32(&offset, capArea, txBlob, "capArea");
	LoadBlob_UINT32(&offset, subCapSize, txBlob, "sub cap size");
	LoadBlob(&offset, subCapSize, txBlob, subCap, "sub cap");
	LoadBlob_Header(TPM_TAG_RQU_COMMAND, offset,
			TPM_ORD_GetCapability, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		return result;

	offset = 10;
	result = UnloadBlob_Header(txBlob, &paramSize);
	if (!result) {
		UnloadBlob_UINT32(&offset, respSize, txBlob, "resp size");
		*resp = malloc(*respSize);
		if (*resp == NULL) {
			LogError("malloc of %d bytes failed.", *respSize);
			return TCSERR(TSS_E_OUTOFMEMORY);
		}
		UnloadBlob(&offset, *respSize, txBlob, *resp, "resp");
	}
	LogResult("Get Cap", result);
	return result;
}

TSS_RESULT
TCSP_GetCapabilityOwner_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
				 TPM_AUTH * pOwnerAuth,	/* out */
				 TCPA_VERSION * pVersion,	/* out */
				 UINT32 * pNonVolatileFlags,	/* out */
				 UINT32 * pVolatileFlags	/* out */
    )
{
	UINT64 offset;
	TSS_RESULT result;
	UINT32 paramSize;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering Getcap owner");

	if ((result = ctx_verify_context(hContext)))
		goto done;

	if ((result = auth_mgr_check(hContext, pOwnerAuth->AuthHandle)))
		goto done;

	offset = 10;
	LoadBlob_Auth(&offset, txBlob, pOwnerAuth);
	LoadBlob_Header(TPM_TAG_RQU_AUTH1_COMMAND, offset,
			TPM_ORD_GetCapabilityOwner, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		goto done;

	offset = 10;
	result = UnloadBlob_Header(txBlob, &paramSize);

	if (!result) {
		UnloadBlob_VERSION(&offset, txBlob, pVersion);
		UnloadBlob_UINT32(&offset, pNonVolatileFlags, txBlob,
				  "nonvolflags");
		UnloadBlob_UINT32(&offset, pVolatileFlags, txBlob, "vol flags");
		UnloadBlob_Auth(&offset, txBlob, pOwnerAuth);
	}

	LogResult("GetCapowner", result);
done:
	auth_mgr_release_auth(pOwnerAuth, NULL, hContext);
	return result;
}

TSS_RESULT
TCSP_CreateEndorsementKeyPair_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
				       TCPA_NONCE antiReplay,	/* in */
				       UINT32 endorsementKeyInfoSize,	/* in */
				       BYTE * endorsementKeyInfo,	/* in */
				       UINT32 * endorsementKeySize,	/* out */
				       BYTE ** endorsementKey,	/* out */
				       TCPA_DIGEST * checksum	/* out */
    )
{
	UINT64 offset;
	UINT32 paramSize;
	TSS_RESULT result;
	TCPA_PUBKEY pubKey;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	if ((result = ctx_verify_context(hContext)))
		return result;

	offset = 10;
	LoadBlob(&offset, TCPA_NONCE_SIZE, txBlob, antiReplay.nonce,
		 "anit replay");
	LoadBlob(&offset, endorsementKeyInfoSize, txBlob,
		 endorsementKeyInfo, "ek stuff");
	LoadBlob_Header(TPM_TAG_RQU_COMMAND, offset,
			TPM_ORD_CreateEndorsementKeyPair, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		return result;

	offset = 10;
	result = UnloadBlob_Header(txBlob, &paramSize);
	if (!result) {
		if ((result = UnloadBlob_PUBKEY(&offset, txBlob, &pubKey)))
			goto done;
		free(pubKey.pubKey.key);
		free(pubKey.algorithmParms.parms);

		*endorsementKeySize = offset - 10;
		*endorsementKey = malloc(*endorsementKeySize);
		if (*endorsementKey == NULL) {
			LogError("malloc of %u bytes failed.",
							 *endorsementKeySize);
			return TCSERR(TSS_E_OUTOFMEMORY);
		}
		memcpy(*endorsementKey, &txBlob[10], *endorsementKeySize);

		UnloadBlob(&offset, TCPA_DIGEST_SIZE, txBlob,
			   checksum->digest, "digest");
	}
done:
	LogDebug("Leaving CreateEKPair with result: 0x%x", result);
	return result;
}

TSS_RESULT
TCSP_ReadPubek_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
			TCPA_NONCE antiReplay,	/* in */
			UINT32 * pubEndorsementKeySize,	/* out */
			BYTE ** pubEndorsementKey,	/* out */
			TCPA_DIGEST * checksum	/* out */
    )
{
	UINT64 offset;
	UINT32 paramSize;
	TSS_RESULT result;
	TCPA_PUBKEY pubkey;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebugFn("Enter");

	if ((result = ctx_verify_context(hContext)))
		return result;

	offset = 10;
	LoadBlob(&offset, 20, txBlob, antiReplay.nonce, "anti replay");
	LoadBlob_Header(TPM_TAG_RQU_COMMAND, offset, TPM_ORD_ReadPubek, txBlob);
	if ((result = req_mgr_submit_req(txBlob)))
		return result;

	offset = 10;
	result = UnloadBlob_Header(txBlob, &paramSize);

	if (!result) {
		if ((result = UnloadBlob_PUBKEY(&offset, txBlob, &pubkey)))
			goto done;
		free(pubkey.pubKey.key);
		free(pubkey.algorithmParms.parms);

		*pubEndorsementKeySize = (UINT32) (offset - 10);
		*pubEndorsementKey = malloc(*pubEndorsementKeySize);
		if (*pubEndorsementKey == NULL) {
			LogError("malloc of %u bytes failed.", *pubEndorsementKeySize);
			return TCSERR(TSS_E_OUTOFMEMORY);
		}
		memcpy(*pubEndorsementKey, &txBlob[10], *pubEndorsementKeySize);
		UnloadBlob(&offset, TCPA_DIGEST_SIZE, txBlob,
			   checksum->digest, "digest");
	}
done:
	LogDebugFn("result: 0x%x", result);
	return result;
}

TSS_RESULT
TCSP_DisablePubekRead_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
			       TPM_AUTH * ownerAuth	/* in, out */
    )
{
	UINT64 offset;
	UINT32 paramSize;
	TSS_RESULT result;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("DisablePubekRead");

	if ((result = ctx_verify_context(hContext)))
		goto done;

	if ((result = auth_mgr_check(hContext, ownerAuth->AuthHandle)))
		goto done;

	offset = 10;
	LoadBlob_Auth(&offset, txBlob, ownerAuth);
	LoadBlob_Header(TPM_TAG_RQU_AUTH1_COMMAND, offset,
			TPM_ORD_DisablePubekRead, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		goto done;

	offset = 10;
	result = UnloadBlob_Header(txBlob, &paramSize);

	if (!result) {
		UnloadBlob_Auth(&offset, txBlob, ownerAuth);
	}
done:
	auth_mgr_release_auth(ownerAuth, NULL, hContext);
	return result;
}

TSS_RESULT
TCSP_OwnerReadPubek_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
			     TPM_AUTH * ownerAuth,	/* in, out */
			     UINT32 * pubEndorsementKeySize,	/* out */
			     BYTE ** pubEndorsementKey	/* out */
    )
{
	UINT32 paramSize;
	TSS_RESULT result;
	UINT64 offset;
	TCPA_PUBKEY container;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering OwnerReadPubek");

	if ((result = ctx_verify_context(hContext)))
		goto done;

	if ((result = auth_mgr_check(hContext, ownerAuth->AuthHandle)))
		goto done;

	offset = 10;
	LoadBlob_Auth(&offset, txBlob, ownerAuth);

	LoadBlob_Header(TPM_TAG_RQU_AUTH1_COMMAND, offset,
			TPM_ORD_OwnerReadPubek, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		goto done;

	result = UnloadBlob_Header(txBlob, &paramSize);

	if (!result) {
		offset = 10;
		/* Call UnloadBlob to parse the data and set its size in &offset */
		if ((result = UnloadBlob_PUBKEY(&offset, txBlob, &container)))
			goto done;

		free(container.pubKey.key);
		free(container.algorithmParms.parms);

		*pubEndorsementKeySize = offset - 10;
		*pubEndorsementKey = malloc(*pubEndorsementKeySize);
		if (*pubEndorsementKey == NULL) {
			LogError("malloc of %u bytes failed.", *pubEndorsementKeySize);
			result = TCSERR(TSS_E_OUTOFMEMORY);
			goto done;
		}
		memcpy(*pubEndorsementKey, &txBlob[10], *pubEndorsementKeySize);
		UnloadBlob_Auth(&offset, txBlob, ownerAuth);
	}
	LogResult("Owner Read Pubek", result);
done:
	auth_mgr_release_auth(ownerAuth, NULL, hContext);
	return result;
}

TSS_RESULT
TCSP_SelfTestFull_Internal(TCS_CONTEXT_HANDLE hContext	/* in */
    )
{
	UINT32 paramSize;
	TSS_RESULT result;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering Self Test Full");
	if ((result = ctx_verify_context(hContext)))
		return result;

	LoadBlob_Header(TPM_TAG_RQU_COMMAND, 0x0A, TPM_ORD_SelfTestFull,
			txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		return result;

	result = UnloadBlob_Header(txBlob, &paramSize);
	LogResult("Self Test Full", result);
	return result;
}

TSS_RESULT
TCSP_CertifySelfTest_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
			      TCS_KEY_HANDLE keyHandle,	/* in */
			      TCPA_NONCE antiReplay,	/* in */
			      TPM_AUTH * privAuth,	/* in, out */
			      UINT32 * sigSize,	/* out */
			      BYTE ** sig	/* out */
    )
{
	UINT64 offset;
	UINT32 paramSize;
	TSS_RESULT result;
	TCPA_KEY_HANDLE keySlot;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering Certify Self Test");

	if ((result = ctx_verify_context(hContext)))
		goto done;

	if (privAuth != NULL) {
		LogDebug("Auth Used");
		if ((result = auth_mgr_check(hContext, privAuth->AuthHandle)))
			goto done;
	} else {
		LogDebug("No Auth");
	}

	if ((result = ensureKeyIsLoaded(hContext, keyHandle, &keySlot)))
		goto done;

	offset = 10;
	LoadBlob_UINT32(&offset, keySlot, txBlob, "key handle");
	LoadBlob(&offset, TCPA_NONCE_SIZE, txBlob,
			antiReplay.nonce, "nonoce");
	if (privAuth != NULL) {
		LoadBlob_Auth(&offset, txBlob, privAuth);
		LoadBlob_Header(TPM_TAG_RQU_AUTH1_COMMAND,
				offset, TPM_ORD_CertifySelfTest,
				txBlob);
	} else {
		LoadBlob_Header(TPM_TAG_RQU_COMMAND, offset,
				TPM_ORD_CertifySelfTest, txBlob);
	}

	if ((result = req_mgr_submit_req(txBlob)))
		goto done;

	offset = 10;
	result = UnloadBlob_Header(txBlob, &paramSize);

	if (!result) {
		UnloadBlob_UINT32(&offset, sigSize, txBlob, "sig size");
		*sig = malloc(*sigSize);
		if (*sig == NULL) {
			LogError("malloc of %d bytes failed.", *sigSize);
			result = TCSERR(TSS_E_OUTOFMEMORY);
			goto done;
		}
		UnloadBlob(&offset, *sigSize, txBlob, *sig, "sig");
		if (privAuth != NULL)
			UnloadBlob_Auth(&offset, txBlob, privAuth);
	}
	LogResult("Certify Self Test", result);
done:
	auth_mgr_release_auth(privAuth, NULL, hContext);
	return result;
}

TSS_RESULT
TCSP_GetTestResult_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
			    UINT32 * outDataSize,	/* out */
			    BYTE ** outData	/* out */
    )
{
	TSS_RESULT result;
	UINT32 paramSize;
	UINT64 offset;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering Get Test Result");
	if ((result = ctx_verify_context(hContext)))
		return result;

	LoadBlob_Header(TPM_TAG_RQU_COMMAND, 0x0A, TPM_ORD_GetTestResult, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		return result;

	offset = 10;

	result = UnloadBlob_Header(txBlob, &paramSize);
	if (!result) {
		UnloadBlob_UINT32(&offset, outDataSize, txBlob, "data size");
		*outData = malloc(*outDataSize);
		if (*outData == NULL) {
			LogError("malloc of %d bytes failed.", *outDataSize);
			return TCSERR(TSS_E_OUTOFMEMORY);
		}
		UnloadBlob(&offset, *outDataSize, txBlob, *outData, "outdata");
		LogBlob(*outDataSize, *outData);
	}
	LogResult("Get Test Result", result);
	return result;
}

TSS_RESULT
TCSP_OwnerSetDisable_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
			      TSS_BOOL disableState,	/* in */
			      TPM_AUTH * ownerAuth	/* in, out */
    )
{
	UINT64 offset;
	UINT32 paramSize;
	TSS_RESULT result;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	offset = 10;

	if ((result = ctx_verify_context(hContext)))
		goto done;

	if ((result = auth_mgr_check(hContext, ownerAuth->AuthHandle)))
		goto done;

	LoadBlob_BOOL(&offset, disableState, txBlob, "State");
	LoadBlob_Auth(&offset, txBlob, ownerAuth);
	LoadBlob_Header(TPM_TAG_RQU_AUTH1_COMMAND, offset,
			TPM_ORD_OwnerSetDisable, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		goto done;

	offset = 10;
	result = UnloadBlob_Header(txBlob, &paramSize);

	if (!result) {
		UnloadBlob_Auth(&offset, txBlob, ownerAuth);
	}
done:
	auth_mgr_release_auth(ownerAuth, NULL, hContext);
	return result;
}

TSS_RESULT
TCSP_OwnerClear_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
			 TPM_AUTH * ownerAuth	/* in, out */
    )
{
	UINT64 offset;
	UINT32 paramSize;
	TSS_RESULT result;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering OwnerClear");

	if ((result = ctx_verify_context(hContext)))
		goto done;

	if ((result = auth_mgr_check(hContext, ownerAuth->AuthHandle)))
		goto done;

	offset = 10;
	LoadBlob_Auth(&offset, txBlob, ownerAuth);
	LoadBlob_Header(TPM_TAG_RQU_AUTH1_COMMAND, offset,
			TPM_ORD_OwnerClear, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		goto done;

	offset = 10;
	result = UnloadBlob_Header(txBlob, &paramSize);

	if (!result) {
		UnloadBlob_Auth(&offset, txBlob, ownerAuth);
	}
	LogResult("Ownerclear", result);
done:
	auth_mgr_release_auth(ownerAuth, NULL, hContext);
	return result;
}

TSS_RESULT
TCSP_DisableOwnerClear_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
				TPM_AUTH * ownerAuth	/* in, out */
    )
{
	UINT64 offset;
	UINT32 paramSize;
	TSS_RESULT result;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering DisableownerClear");

	if ((result = ctx_verify_context(hContext)))
		goto done;

	if ((result = auth_mgr_check(hContext, ownerAuth->AuthHandle)))
		goto done;

	offset = 10;
	LoadBlob_Auth(&offset, txBlob, ownerAuth);
	LoadBlob_Header(TPM_TAG_RQU_AUTH1_COMMAND, offset,
			TPM_ORD_DisableOwnerClear, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		goto done;

	offset = 10;
	result = UnloadBlob_Header(txBlob, &paramSize);

	if (!result) {
		UnloadBlob_Auth(&offset, txBlob, ownerAuth);
	}
	LogResult("DisableOwnerClear", result);
done:
	auth_mgr_release_auth(ownerAuth, NULL, hContext);
	return result;
}

TSS_RESULT
TCSP_ForceClear_Internal(TCS_CONTEXT_HANDLE hContext	/* in */
    )
{
	UINT32 paramSize;
	TSS_RESULT result;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering Force Clear");
	if ((result = ctx_verify_context(hContext)))
		return result;

	LoadBlob_Header(TPM_TAG_RQU_COMMAND, 0x0A, TPM_ORD_ForceClear, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		return result;

	result = UnloadBlob_Header(txBlob, &paramSize);
	LogResult("Force Clear", result);
	return result;
}

TSS_RESULT
TCSP_DisableForceClear_Internal(TCS_CONTEXT_HANDLE hContext	/* in */
    )
{
	UINT32 paramSize;
	TSS_RESULT result;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering Disable Force Clear");
	if ((result = ctx_verify_context(hContext)))
		return result;

	LoadBlob_Header(TPM_TAG_RQU_COMMAND, 0x0A,
			TPM_ORD_DisableForceClear, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		return result;

	result = UnloadBlob_Header(txBlob, &paramSize);
	LogResult("Disable Force Clear", result);
	return result;

}

TSS_RESULT
TCSP_PhysicalPresence_Internal(TCS_CONTEXT_HANDLE hContext, /* in */
			TCPA_PHYSICAL_PRESENCE fPhysicalPresence /* in */
    )
{
	UINT64 offset;
	UINT32 paramSize;
	TSS_RESULT result = TCSERR(TSS_E_NOTIMPL);
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];
	char runlevel;

	runlevel = platform_get_runlevel();

	if (runlevel != 's' && runlevel != 'S' && runlevel != '1') {
		LogInfo("Physical Presence command denied: Must be in single"
				" user mode.");
		return TCSERR(TSS_E_NOTIMPL);
	}

	if ((result = ctx_verify_context(hContext)))
		return result;

	offset = 10;
	LoadBlob_UINT16(&offset, fPhysicalPresence, txBlob, NULL);
	LoadBlob_Header(TPM_TAG_RQU_COMMAND, offset,
			TPM_ORD_PhysicalPresence, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		return result;

	return UnloadBlob_Header(txBlob, &paramSize);
}

TSS_RESULT
TCSP_PhysicalDisable_Internal(TCS_CONTEXT_HANDLE hContext	/* in */
    )
{
	UINT32 paramSize;
	TSS_RESULT result;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering Physical Disable");
	if ((result = ctx_verify_context(hContext)))
		return result;

	/* XXX ooh, magic */
	LoadBlob_Header(TPM_TAG_RQU_COMMAND, 0x0A,
			TPM_ORD_PhysicalDisable, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		return result;

	result = UnloadBlob_Header(txBlob, &paramSize);
	LogResult("Physical Disable", result);

	return result;
}

TSS_RESULT
TCSP_PhysicalEnable_Internal(TCS_CONTEXT_HANDLE hContext	/* in */
    )
{
	TSS_RESULT result;
	UINT32 paramSize;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering Physical Enable");
	if ((result = ctx_verify_context(hContext)))
		return result;

	LoadBlob_Header(TPM_TAG_RQU_COMMAND, 0x0A,
			TPM_ORD_PhysicalEnable, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		return result;

	result = UnloadBlob_Header(txBlob, &paramSize);
	LogResult("Physical Enable", result);

	return result;
}

TSS_RESULT
TCSP_PhysicalSetDeactivated_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
				     TSS_BOOL state	/* in */
    )
{
	UINT64 offset;
	UINT32 paramSize;
	TSS_RESULT result;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering Physical Set Decativated");
	if ((result = ctx_verify_context(hContext)))
		return result;

	offset = 10;
	LoadBlob_BOOL(&offset, state, txBlob, "State");
	LoadBlob_Header(TPM_TAG_RQU_COMMAND, offset,
			TPM_ORD_PhysicalSetDeactivated, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		return result;

	result = UnloadBlob_Header(txBlob, &paramSize);
	LogResult("PhysicalSetDeactivated", result);
	return result;
}

TSS_RESULT
TCSP_SetTempDeactivated_Internal(TCS_CONTEXT_HANDLE hContext	/* in */
    )
{
	UINT32 paramSize;
	TSS_RESULT result;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering Set Temp Deactivated");
	if ((result = ctx_verify_context(hContext)))
		return result;

	LoadBlob_Header(TPM_TAG_RQU_COMMAND, 0x0A,
			TPM_ORD_SetTempDeactivated, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		return result;

	result = UnloadBlob_Header(txBlob, &paramSize);
	LogResult("SetTempDeactivated", result);

	return result;
}

TSS_RESULT
TCSP_FieldUpgrade_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
			   UINT32 dataInSize,	/* in */
			   BYTE * dataIn,	/* in */
			   UINT32 * dataOutSize,	/* out */
			   BYTE ** dataOut,	/* out */
			   TPM_AUTH * ownerAuth	/* in, out */
    )
{
	TSS_RESULT result;
	UINT32 paramSize;
	UINT64 offset;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Field Upgrade");

	if ((result = ctx_verify_context(hContext)))
		goto done;

	if ((result = auth_mgr_check(hContext, ownerAuth->AuthHandle)))
		goto done;

	offset = 10;
	if (dataInSize != 0) {
		LoadBlob_UINT32(&offset, dataInSize, txBlob,
				"data size");
		LoadBlob(&offset, dataInSize, txBlob, dataIn, "data");
	}
	LoadBlob_Auth(&offset, txBlob, ownerAuth);

	LoadBlob_Header(TPM_TAG_RQU_AUTH1_COMMAND, offset,
			TPM_ORD_FieldUpgrade, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		goto done;

	result = UnloadBlob_Header(txBlob, &paramSize);

	if (!result) {
		offset = 10;
		if (dataInSize != 0) {
			UnloadBlob_UINT32(&offset, dataOutSize, txBlob, "size");
			*dataOut = malloc(*dataOutSize);
			if (*dataOut == NULL) {
				LogError("malloc of %u bytes failed.",
								 *dataOutSize);
				result = TCSERR(TSS_E_OUTOFMEMORY);
				goto done;
			}
			UnloadBlob(&offset, *dataOutSize, txBlob,
				   *dataOut, "data");
		}
		UnloadBlob_Auth(&offset, txBlob, ownerAuth);
	}
	LogResult("Field Upgrade", result);
done:
	auth_mgr_release_auth(ownerAuth, NULL, hContext);
	return result;
}

TSS_RESULT
TCSP_SetRedirection_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
			     TCS_KEY_HANDLE keyHandle,	/* in */
			     UINT32 c1,	/* in */
			     UINT32 c2,	/* in */
			     TPM_AUTH * privAuth	/* in, out */
    )
{
	TSS_RESULT result;
	UINT32 paramSize;
	UINT64 offset;
	TCPA_KEY_HANDLE keySlot;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Set Redirection");

	if ((result = ctx_verify_context(hContext)))
		goto done;

	if (privAuth != NULL) {
		if ((result = auth_mgr_check(hContext, privAuth->AuthHandle)))
			goto done;
	}

	if ((result = ensureKeyIsLoaded(hContext, keyHandle, &keySlot))) {
		result = TCSERR(TSS_E_FAIL);
		goto done;
	}

	offset = 10;
	LoadBlob_UINT32(&offset, keySlot, txBlob, "key slot");
	LoadBlob_UINT32(&offset, c1, txBlob, "c1");
	LoadBlob_UINT32(&offset, c2, txBlob, "c2");
	if (privAuth != NULL) {
		LoadBlob_Auth(&offset, txBlob, privAuth);
		LoadBlob_Header(TPM_TAG_RQU_AUTH1_COMMAND,
				offset, TPM_ORD_SetRedirection, txBlob);
	} else {
		LoadBlob_Header(TPM_TAG_RQU_COMMAND, offset,
				TPM_ORD_SetRedirection, txBlob);
	}
	if ((result = req_mgr_submit_req(txBlob)))
		goto done;

	result = UnloadBlob_Header(txBlob, &paramSize);

	if (!result) {
		offset = 10;
		if (privAuth != NULL)
			UnloadBlob_Auth(&offset, txBlob, privAuth);
	}
	LogResult("Set Redirection", result);
done:
	auth_mgr_release_auth(privAuth, NULL, hContext);
	return result;
}

TSS_RESULT
TCSP_CreateMaintenanceArchive_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
				       TSS_BOOL generateRandom,	/* in */
				       TPM_AUTH * ownerAuth,	/* in, out */
				       UINT32 * randomSize,	/* out */
				       BYTE ** random,	/* out */
				       UINT32 * archiveSize,	/* out */
				       BYTE ** archive	/* out */
    )
{
	TSS_RESULT result;
	UINT32 paramSize;
	UINT64 offset;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Create Main Archive");

	if ((result = ctx_verify_context(hContext)))
		goto done;

	if ((result = auth_mgr_check(hContext, ownerAuth->AuthHandle)))
		goto done;

	offset = 10;
	LoadBlob_BOOL(&offset, generateRandom, txBlob, "gen rand");
	LoadBlob_Auth(&offset, txBlob, ownerAuth);

	LoadBlob_Header(TPM_TAG_RQU_AUTH1_COMMAND, offset,
			TPM_ORD_CreateMaintenanceArchive, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		goto done;

	result = UnloadBlob_Header(txBlob, &paramSize);

	if (!result) {
		offset = 10;
		UnloadBlob_UINT32(&offset, randomSize, txBlob, "random size");
		*random = malloc(*randomSize);
		if (*random == NULL) {
			LogError("malloc of %d bytes failed.", *randomSize);
			result = TCSERR(TSS_E_OUTOFMEMORY);
			goto done;
		} else {
			UnloadBlob(&offset, *randomSize, txBlob, *random, "random");
		}

		UnloadBlob_UINT32(&offset, archiveSize, txBlob, "archive size");
		*archive = malloc(*archiveSize);
		if (*archive == NULL) {
			free(*random);
			LogError("malloc of %d bytes failed.", *archiveSize);
			result = TCSERR(TSS_E_OUTOFMEMORY);
			goto done;
		} else {
			UnloadBlob(&offset, *archiveSize, txBlob, *archive, "archive");
		}

		UnloadBlob_Auth(&offset, txBlob, ownerAuth);
	}
	LogResult("Create Main Archive", result);
done:
	auth_mgr_release_auth(ownerAuth, NULL, hContext);
	return result;
}

TSS_RESULT
TCSP_LoadMaintenanceArchive_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
				     UINT32 dataInSize,	/* in */
				     BYTE * dataIn,	/* in */
				     TPM_AUTH * ownerAuth,	/* in, out */
				     UINT32 * dataOutSize,	/* out */
				     BYTE ** dataOut	/* out */
    )
{
	TSS_RESULT result;
	UINT32 paramSize;
	UINT64 offset;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Load Maint Archive");

	if ((result = ctx_verify_context(hContext)))
		goto done;

	if ((result = auth_mgr_check(hContext, ownerAuth->AuthHandle)))
		goto done;

	offset = 10;
	if (dataInSize != 0) {
		LoadBlob_UINT32(&offset, dataInSize, txBlob,
				"vendor data size");
		LoadBlob(&offset, dataInSize, txBlob, dataIn,
				"vendor data");
	}
	LoadBlob_Auth(&offset, txBlob, ownerAuth);

	LoadBlob_Header(TPM_TAG_RQU_AUTH1_COMMAND, offset,
			TPM_ORD_LoadMaintenanceArchive, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		goto done;

	result = UnloadBlob_Header(txBlob, &paramSize);

	if (!result) {
		offset = 10;
		if (dataInSize != 0) {
			UnloadBlob_UINT32(&offset, dataOutSize, txBlob, "vendor data size");
			*dataOut = calloc(1, *dataOutSize);
			if (*dataOut == NULL) {
				LogError("malloc of %u bytes failed.",
								 *dataOutSize);
				result = TCSERR(TSS_E_OUTOFMEMORY);
				goto done;
			}
			UnloadBlob(&offset, *dataOutSize, txBlob,
				   *dataOut, "vendor data");
		}
		UnloadBlob_Auth(&offset, txBlob, ownerAuth);
	}
	LogResult("Load Maint Archive", result);
done:
	auth_mgr_release_auth(ownerAuth, NULL, hContext);
	return result;
}

TSS_RESULT
TCSP_KillMaintenanceFeature_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
				     TPM_AUTH * ownerAuth	/* in, out */
    )
{
	TSS_RESULT result;
	UINT32 paramSize;
	UINT64 offset;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	if ((result = ctx_verify_context(hContext)))
		goto done;

	if ((result = auth_mgr_check(hContext, ownerAuth->AuthHandle)))
		goto done;

	offset = 10;
	LoadBlob_Auth(&offset, txBlob, ownerAuth);

	LoadBlob_Header(TPM_TAG_RQU_AUTH1_COMMAND, offset,
			TPM_ORD_KillMaintenanceFeature, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		goto done;

	result = UnloadBlob_Header(txBlob, &paramSize);

	if (!result) {
		offset = 10;
		UnloadBlob_Auth(&offset, txBlob, ownerAuth);
	}
done:
	auth_mgr_release_auth(ownerAuth, NULL, hContext);
	return result;
}

TSS_RESULT
TCSP_LoadManuMaintPub_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
			       TCPA_NONCE antiReplay,	/* in */
			       UINT32 PubKeySize,	/* in */
			       BYTE * PubKey,	/* in */
			       TCPA_DIGEST * checksum	/* out */
    )
{
	TSS_RESULT result;
	UINT32 paramSize;
	UINT64 offset;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering Load Manu Maint Pub");

	offset = 10;
	LoadBlob(&offset, 20, txBlob, antiReplay.nonce, "checksum");
	LoadBlob(&offset, PubKeySize, txBlob, PubKey, "pubkey");
	LoadBlob_Header(TPM_TAG_RQU_COMMAND, offset,
			TPM_ORD_LoadManuMaintPub, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		return result;

	result = UnloadBlob_Header(txBlob, &paramSize);
	if (!result) {
		offset = 10;
		UnloadBlob(&offset, 20, txBlob, checksum->digest, "checksum");
	}
	LogResult("Load Manu Maint Pub", result);
	return result;
}

TSS_RESULT
TCSP_ReadManuMaintPub_Internal(TCS_CONTEXT_HANDLE hContext,	/* in */
			       TCPA_NONCE antiReplay,	/* in */
			       TCPA_DIGEST * checksum	/* out */
    )
{
	TSS_RESULT result;
	UINT32 paramSize;
	UINT64 offset;
	BYTE txBlob[TSS_TPM_TXBLOB_SIZE];

	LogDebug("Entering Read Manu Maint Pub");

	offset = 10;
	LoadBlob(&offset, 20, txBlob, antiReplay.nonce, "checksum");
	LoadBlob_Header(TPM_TAG_RQU_COMMAND, offset,
			TPM_ORD_ReadManuMaintPub, txBlob);

	if ((result = req_mgr_submit_req(txBlob)))
		return result;

	result = UnloadBlob_Header(txBlob, &paramSize);
	if (!result) {
		offset = 10;
		UnloadBlob(&offset, 20, txBlob, checksum->digest, "checksum");
	}
	LogResult("Read Manu Maint Pub", result);
	return result;
}
