/*
 *  Copyright (C) 2004 Steve Harris
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  $Id: main.c,v 1.4 2004/03/03 08:41:53 swh Exp $
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <jack/jack.h>
#include <getopt.h>
#include <gtk/gtk.h>

#ifdef HAVE_LADCCA
#include <ladcca/ladcca.h>

cca_client_t *cca_client;
#endif

#include "threads.h"
#include "interface.h"
#include "meters.h"
#include "support.h"
#include "main.h"

#define DEFAULT_BUF_LENGTH 10 /* in seconds */

#define DEBUG(lvl, txt...) \
    if (verbosity >= lvl) fprintf(stderr, PACKAGE ": " txt)

const static int verbosity = 0;

gboolean idle_cb(gpointer data);
void cleanup();

GtkWidget *main_window;

int num_ports = DEFAULT_NUM_PORTS;
unsigned int buf_length = DEFAULT_BUF_LENGTH; /* in seconds */

char *client_name = DEFAULT_CLIENT_NAME;
char *prefix = DEFAULT_PREFIX;

jack_port_t *ports[MAX_PORTS];
jack_client_t *client;

GdkPixbuf *img_on, *img_off, *img_busy;
GdkPixbuf *icon_on, *icon_off;

int main(int argc, char *argv[])
{
    unsigned int i;
    int opt;
    char port_name[32];
    pthread_t dt;
#ifdef HAVE_LADCCA
    cca_args_t *cca_args = cca_extract_args(&argc, &argv);
#endif

    while ((opt = getopt(argc, argv, "c:t:n:p:h")) != -1) {
	switch (opt) {
	case 'c':
	    num_ports = atoi(optarg);
	    DEBUG(1, "ports: %d\n", num_ports);
	    break;
	case 't':
	    buf_length = atoi(optarg);
	    DEBUG(1, "buffer: %ds\n", buf_length);
	    break;
	case 'n':
	    client_name = optarg;
	    DEBUG(1, "client name: %s\n", client_name);
	    break;
	case 'p':
	    prefix = optarg;
	    DEBUG(1, "prefix: %s\n", prefix);
	    break;
	case 'h':
	    /* Force help to be shown */
	    num_ports = 0;
	    break;
	default:
	    num_ports = 0;
	    break;
	}
    }

    if (num_ports < 1 || num_ports > MAX_PORTS) {
	fprintf(stderr, "Usage %s: [-c channels] [-n jack-name] "
			"[-t buffer-length] [-p file prefix]\n",
			argv[0]);
	fprintf(stderr, "\tchannels must be in the range 1-8, default %d\n",
			DEFAULT_NUM_PORTS);
	fprintf(stderr, "\tjack-name, default \"%s\"\n", DEFAULT_CLIENT_NAME);
	fprintf(stderr, "\tfile-prefix, default \"%s\"\n", DEFAULT_PREFIX);
	fprintf(stderr, "\tbuffer-length, default %d secs\n", DEFAULT_BUF_LENGTH);
	exit(1);
    }

    /* Register with jack */
    if ((client = jack_client_new(client_name)) == 0) {
	DEBUG(0, "jack server not running?\n");
	exit(1);
    }
    DEBUG(1, "registering as %s\n", client_name);

    process_init(buf_length);

#ifdef HAVE_LADCCA
    cca_client = cca_init (cca_args, "TimeMachine",
                     0, /* would be CCA_Config_Data_Set etc. */
                     CCA_PROTOCOL (2,0));
    if (!cca_client) {
	DEBUG(1, "could not initialise LADCCA\n");
    }
#endif

    jack_set_process_callback(client, process, 0);

    if (jack_activate(client)) {
	DEBUG(0, "cannot activate JACK client");
	exit(1);
    }
#ifdef HAVE_LADCCA
    cca_jack_client_name(cca_client, client_name);
#endif

    /* Create the jack ports */
    for (i = 0; i < num_ports; i++) {
	snprintf(port_name, 31, "in_%d", i + 1);
	ports[i] = jack_port_register(client, port_name,
				      JACK_DEFAULT_AUDIO_TYPE,
				      JackPortIsInput, 0);
    }

    /* Start the disk thread */
    pthread_create(&dt, NULL, (void *)&writer_thread, NULL);

    gtk_init(&argc, &argv);

    add_pixmap_directory(PACKAGE_DATA_DIR "/" PACKAGE "/pixmaps");
    add_pixmap_directory("pixmaps");

    img_on = create_pixbuf("on.png");
    img_off = create_pixbuf("off.png");
    img_busy = create_pixbuf("busy.png");
    icon_on = create_pixbuf("on-icon.png");
    icon_off = create_pixbuf("off-icon.png");

    main_window = create_window(client_name);
    gtk_window_set_icon(GTK_WINDOW(main_window), icon_off);
    gtk_widget_show(main_window);

    bind_meters();
    g_timeout_add(100, meter_tick, NULL);
    gtk_idle_add(idle_cb, cca_client);

    gtk_main();

    cleanup();

    /* We can't ever get here, but it keeps gcc quiet */
    return 0;
}

void cleanup()
{
    /* Leave the jack graph */
    jack_client_close(client);

    recording_quit();

    while(!recording_done) {
	usleep(1000);
    }

    DEBUG(0, "exiting\n");
    fflush(stderr);

    /* And were done */
    exit(0);
}

gboolean idle_cb(gpointer data)
{
    cca_client_t *cca_client = (cca_client_t *)data;
    cca_event_t *event;
    cca_config_t *config;

    while ((event = cca_get_event(cca_client))) {
	if (cca_event_get_type(event) == CCA_Save_Data_Set) {
#if 0

Doesnt look like we need to do any of this stuff as LADCCA restores the
command line

	    config = cca_config_new();
	    cca_config_set_key(config, "ports");
	    cca_config_set_value(config, &num_ports, sizeof(num_ports));
	    cca_send_config(cca_client, config);
	    config = cca_config_new();
	    cca_config_set_key(config, "buffer");
	    cca_config_set_value(config, &buf_length, sizeof(buf_length));
	    cca_send_config(cca_client, config);
#endif
	} else if (cca_event_get_type(event) == CCA_Quit) {
	    cleanup();
	} else {
	    DEBUG(0, "unhandled LADCCA event: type %d, '%s''\n",
		   cca_event_get_type(event),
		   cca_event_get_string(event));
	}
    }

    while ((config = cca_get_config(cca_client))) {
	DEBUG(0, "got unexpected LADCCA config: %s\n",
	       cca_config_get_key(config));
    }

    usleep(10000);

    return TRUE;
}

/* vi:set ts=8 sts=4 sw=4: */
