/** @file main.c
 * Thy core *//* -*- mode: c; c-file-style: "gnu" -*-
 * main.c -- Thy core
 * Copyright (C) 2002, 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy.
 *
 * Thy is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 dated June, 1991.
 *
 * Thy is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <time.h>
#include <unistd.h>

#include "compat/compat.h"
#include "bh-libs/list.h"

#include "auth.h"
#include "config.h"
#include "fabs.h"
#include "daemon.h"
#include "nqueue.h"
#include "options.h"
#include "queue.h"
#include "session.h"
#include "stats.h"
#include "thy.h"
#include "tls.h"
#include "types.h"
#include "worker.h"

/** Status flag for signalling a pending signal. */
static volatile sig_atomic_t signal_in_progress = 0;
/** Status flag for signalling a pending SIGCHLD.
 */
static volatile sig_atomic_t sigchild = 0;
/** Maximum number of connections.
 * This variable holds the maximum number of file descriptors Thy can
 * keep open at any given time.
 */
long int _THY_MAXCONN;
static pid_t main_pid; /**< PID of the main process. */
static char *pidfile; /**< Name of the file to log Thy's PID into. */
#if THY_OPTION_TLS
typedef void (*thy_sighandler_t)(int); /**< Dummy typedef for systems
					  that don't have it. */
static thy_sighandler_t sighup_old = NULL; /**< Old SIGHUP handler. */
#endif
size_t thy_active_cgis = 0; /**< Number of active CGIs. */

/** Print server statistics.
 * This handler prints server statistics upon receipt of an USR1
 * signal.
 */
static void
thy_stats (int sig)
{
  const config_t *config = config_get ();
  time_t uptime = thy_stats_uptime ();
  int d, h, m, s;
  long reqs = thy_stats_requests ();
  size_t transfers = thy_stats_transfers ();

  d = uptime / (24 * 60 * 60);
  h = (uptime - d * 24 * 60 * 60) / (60 * 60);
  m = (uptime - d * 24 * 60 * 60 - h * 60 * 60) / 60;
  s = (uptime - d * 24 * 60 * 60 - h * 60 * 60 - m * 60);

  bhc_log ("Server uptime: %d days, %02d:%02d:%02d", d, h, m, s);
  bhc_log ("Accepted %ld connections, served " SIZET_FORMAT " bytes",
	   reqs, transfers);
  bhc_log ("Active connections: %ld", thy_nq_act ());

  if (sig > 0)
    {
      signal (sig, thy_stats);
      alarm (config->options.stats);
    }
}

/** @internal A generic signal handler.
 */
static void
sighandler (int sig)
{
  if (signal_in_progress)
    raise (sig);
  signal_in_progress = 1;

  if (sig != SIGQUIT)
    {
      bhc_log ("Got signal %d, shutting down", sig);
    }
  thy_stats (-1);
  closelog ();
  if (main_pid == getpid ())
    {
      thy_worker_quit ();
      auth_done ();
      if (pidfile)
	unlink (pidfile);
    }
  signal (sig, SIG_DFL);
  if (sig == SIGSEGV)
    raise (sig);
  else
    exit (0);
}

/** @internal SIGCHLD handler.
 * Turns on the signalling flag.
 */
static void
sigchld_handler (int sig)
{
  sigchild = 1;
  signal (sig, sigchld_handler);
}

/** @internal Child reaper.
 * Waits for a child to exit, then iterate through the list of pending
 * sessions and clean up all the unnecessary children.
 */
static void
cleanup_children (void)
{
  pid_t child;
  int status;
  long int i;
  session_t *session;

  while ((child = waitpid (-1, &status, WNOHANG)) > (pid_t) 0)
    {
      for (i = 0; i < _THY_MAXCONN; i++)
	{
	  session = queue_get (i);
	  if (!session)
	    continue;
	  if (session->cgi.child == child)
	    {
	      session->cgi.running = 0;
	      thy_active_cgis--;
	      if (status)
		queue_del (i);
	      break;
	    }
	}
    }
  sigchild = 0;
}

#if THY_OPTION_TLS
/** @internal SIGHUP handler.
 * Reinitialises TLS params.
 */
static void
sighup_handler (int sig)
{
  if (sighup_old)
    sighup_old (sig);

  thy_tls_reinit (1);
  signal (sig, sighup_handler);
}
#endif

/** @internal Shutdown thy.
 * Upon exit, this one kills all the children, and does some
 * cleanup. Called via atexit().
 */
static void
_thy_shutdown (void)
{
  long int i;
  session_t *session;

  if (getpid() != main_pid)
    return;

  for (i = 0; i < _THY_MAXCONN; i++)
    {
      session = queue_get (i);
      if (!session)
	continue;
      if (session->cgi.child != -1)
	kill (session->cgi.child, SIGTERM);
    }

  thy_worker_quit ();
  auth_done ();
}

/** @internal Check if we need to start the Authoriser.
 * Iterates through the config maps, and checks if anyone has -o auth
 * enabled.
 *
 * @returns One if th Authenticator needs to be started, zero
 * otherwise.
 */
static int
_thy_auth_start (void)
{
  const config_t *config = config_get ();
  thy_config_map_t *mconfig;
  size_t i;

  if (config->defconf->options.auth == THY_BOOL_TRUE)
    return 1;

  for (i = 0; i < bhl_list_size (config->maps); i++)
    {
      bhl_list_get (config->maps, i, (void **)&mconfig);
      if (mconfig->config->options.auth == THY_BOOL_TRUE)
	{
	  free (mconfig);
	  return 1;
	}
      free (mconfig);
    }
  return 0;
}

/** The main program body.
 */
int
main (int argc, char **argv, char **envp)
{
  config_t *config;
  int pending;
  int pidfilefd = -1;
  thy_listener_t *listeners;

  thy_stats_start ();

  _THY_MAXCONN = sysconf (_SC_OPEN_MAX);
  if (_THY_MAXCONN < 0)
    _THY_MAXCONN = 1024;

  pidfile = NULL;

  openlog ("thy", LOG_PID, LOG_DAEMON);

  /* Read configuration */
  bhc_setproctitle_init (argc, argv, envp);
  thy_servername_init ();
  config_parse (argc, argv);
  config = config_get ();
  bhc_setproctitle ("thy");

  if (config->pidfile)
    {
      if (!access (config->pidfile, F_OK))
	{
	  fprintf (stderr, "Pidfile (%s) already exists. Exiting.\n",
		   config->pidfile);
	  bhc_error ("Pidfile (%s) already exists. Exiting.",
		     config->pidfile);
	  exit (1);
	}
      if ((pidfilefd = open (config->pidfile,
			     O_WRONLY | O_CREAT | O_TRUNC, 0644)) == -1)
	{
	  fprintf (stderr, "Cannot create pidfile (%s): %s.\n",
		   config->pidfile, strerror (errno));
	  bhc_error ("Cannot create pidfile (%s): %s.", config->pidfile,
		     strerror (errno));
	  exit (1);
	}
      fchown (pidfilefd, config->uid, (gid_t)-1);
      if (fcntl (pidfilefd, F_SETFD, FD_CLOEXEC) == -1)
	{
	  bhc_error ("fcntl: %s", strerror (errno));
	  exit (1);
	}
    }

#if THY_OPTION_FORK
  /* Fork off ... */
  if (config->options.daemon == THY_BOOL_TRUE)
    {
      if (daemon (0, 0))
	{
	  bhc_error ("daemon: %s", strerror (errno));
	  exit (1);
	}
    }
#endif

  bhc_log ("Starting %s...", thy_servername (NULL));

  signal (SIGPIPE, SIG_IGN);

  if (_thy_auth_start())
    {
      if (auth_init () != 0)
	{
	  bhc_error ("%s", "Could not start the authenticator."
		     "Authentication disabled.");
	  config->defconf->options.auth = THY_BOOL_FALSE;
	}
    }
  if (thy_worker_init () != 0)
    {
      bhc_error ("%s", "Could not start the Worker.");
      config->options.worker = THY_BOOL_FALSE;
    }

  main_pid = getpid ();
  if (config->pidfile && pidfilefd >= 0)
    {
      char *mypid = NULL;

      asprintf (&mypid, "%d\n", main_pid);
      write (pidfilefd, mypid, strlen (mypid));
      close (pidfilefd);
      free (mypid);
    }
  pidfile = config->pidfile;

#ifndef THY_OPTION_DEBUG
  chdir (config->defconf->webroot);
#endif

  signal (SIGTERM, sighandler);
  signal (SIGSEGV, sighandler);
  signal (SIGQUIT, sighandler);

#if THY_OPTION_TLS
  thy_tls_enabled = THY_BOOL_UNSET;
  thy_tls_init ();
  sighup_old = signal (SIGHUP, sighup_handler);
#else
  thy_tls_enabled = THY_BOOL_FALSE;
  signal (SIGHUP, SIG_IGN);
#endif

#ifndef THY_OPTION_DEBUG
  if (config->options.chroot == THY_BOOL_TRUE)
    {
      chdir (config->defconf->webroot);
      chroot (".");
      chdir ("/");
      config->defconf->webroot = "/";
    }
#endif

  if ((listeners = daemon_init ()) == NULL)
    {
      bhc_error ("%s", "No listening sockets could be initialised. "
		 "Exiting.");
      bhc_exit (1);
    }

  fabs_init ();
  queue_init ();
  if (thy_nq_init () != 0)
    {
      bhc_error ("%s", "Error while initialising the network queues. "
		 "Exiting.");
      bhc_exit (1);
    }

  signal (SIGCHLD, sigchld_handler);
  signal (SIGALRM, thy_stats);
  atexit (_thy_shutdown);
  alarm (config->options.stats);

  bhc_debug ("Max FDs: %ld", _THY_MAXCONN);

  /* Main loop */
  for (;;)
    {
      if ((pending = daemon_select (listeners, config->stimeout)) < 0)
	bhc_exit (1);
      if (sigchild)
	{
	  cleanup_children ();
	  continue;
	}

      if (!pending)
	daemon_accept (listeners);

      daemon_handle_io ();
    }
}
