/*
 * SCSI target daemon core functions
 *
 * Copyright (C) 2005-2007 FUJITA Tomonori <tomof@acm.org>
 * Copyright (C) 2005-2007 Mike Christie <michaelc@cs.wisc.edu>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, version 2 of the
 * License.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */
#include <errno.h>
#include <fcntl.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/socket.h>

#include "list.h"
#include "util.h"
#include "tgtd.h"
#include "driver.h"
#include "target.h"
#include "scsi.h"
#include "tgtadm.h"

static LIST_HEAD(device_type_list);

int device_type_register(struct device_type_template *t)
{
	list_add_tail(&t->device_type_siblings, &device_type_list);
	return 0;
}

static struct device_type_template *device_type_lookup(int type)
{
	struct device_type_template *t;

	list_for_each_entry(t, &device_type_list, device_type_siblings) {
		if (t->type == type)
			return t;
	}
	return NULL;
}

static LIST_HEAD(target_list);

static struct target *target_lookup(int tid)
{
	struct target *target;
	list_for_each_entry(target, &target_list, target_siblings)
		if (target->tid == tid)
			return target;
	return NULL;
}

static struct it_nexus *it_nexus_lookup(int tid, uint64_t itn_id)
{
	struct target *target;
	struct it_nexus *itn;

	target = target_lookup(tid);
	if (!target)
		return NULL;

	list_for_each_entry(itn, &target->it_nexus_list, nexus_siblings) {
		if (itn->itn_id == itn_id)
			return itn;
	}
	return NULL;
}

int it_nexus_create(int tid, uint64_t itn_id, int host_no, char *info)
{
	int i;
	struct target *target;
	struct it_nexus *itn;

	dprintf("%d %" PRIu64 "%d\n", tid, itn_id, host_no);
	/* for reserve/release code */
	if (!itn_id)
		return -EINVAL;

	itn = it_nexus_lookup(tid, itn_id);
	if (itn)
		return -EEXIST;

	target = target_lookup(tid);

	itn = zalloc(sizeof(*itn));
	if (!itn)
		return -ENOMEM;

	itn->itn_id = itn_id;
	itn->host_no = host_no;
	itn->nexus_target = target;
	itn->info = info;

	for (i = 0; i < ARRAY_SIZE(itn->cmd_hash_list); i++)
		INIT_LIST_HEAD(&itn->cmd_hash_list[i]);

	list_add_tail(&itn->nexus_siblings, &target->it_nexus_list);

	return 0;
}

int it_nexus_destroy(int tid, uint64_t itn_id)
{
	int i;
	struct it_nexus *itn;

	dprintf("%d %" PRIu64 "\n", tid, itn_id);

	itn = it_nexus_lookup(tid, itn_id);
	if (!itn)
		return -ENOENT;

	for (i = 0; i < ARRAY_SIZE(itn->cmd_hash_list); i++)
		if (!list_empty(&itn->cmd_hash_list[i]))
			return -EBUSY;

	list_del(&itn->nexus_siblings);
	free(itn);
	return 0;
}

static struct scsi_lu *device_lookup(struct target *target, uint64_t lun)
{
	struct scsi_lu *lu;

	list_for_each_entry(lu, &target->device_list, device_siblings)
		if (lu->lun == lun)
			return lu;
	return NULL;
}

static struct scsi_cmd *cmd_lookup(int tid, uint64_t itn_id, uint64_t tag)
{
	struct scsi_cmd *cmd;
	struct it_nexus *itn;

	itn = it_nexus_lookup(tid, itn_id);
	if (!itn)
		return NULL;

	list_for_each_entry(cmd, &itn->cmd_hash_list[hashfn(tag)], c_hlist) {
		if (cmd->tag == tag)
			return cmd;
	}
	return NULL;
}

static void cmd_hlist_insert(struct it_nexus *itn, struct scsi_cmd *cmd)
{
	struct list_head *list = &itn->cmd_hash_list[hashfn(cmd->tag)];
	list_add(&cmd->c_hlist, list);
}

static void cmd_hlist_remove(struct scsi_cmd *cmd)
{
	list_del(&cmd->c_hlist);
}

static void tgt_cmd_queue_init(struct tgt_cmd_queue *q)
{
	q->active_cmd = 0;
	q->state = 0;
	INIT_LIST_HEAD(&q->queue);
}

static int tgt_device_path_update(struct target *target,
				  struct scsi_lu *lu, char *path)
{
	int err, dev_fd;
	uint64_t size;

	path = strdup(path);
	if (!path)
		return TGTADM_NOMEM;

	err = lu->bst->bs_open(lu, path, &dev_fd, &size);
	if (err) {
		free(path);
		return TGTADM_INVALID_REQUEST;
	}

	lu->fd = dev_fd;
	lu->addr = 0;
	lu->size = size;
	lu->path = path;
	lu->attrs.online = 1;

	return 0;
}

static struct scsi_lu *
__device_lookup(int tid, uint64_t lun, struct target **t)
{
	struct target *target;
	struct scsi_lu *lu;

	target = target_lookup(tid);
	if (!target)
		return NULL;

	lu = device_lookup(target, lun);
	if (!lu)
		return NULL;

	*t = target;
	return lu;
}

int tgt_device_create(int tid, int dev_type, uint64_t lun, char *args, int backing)
{
	char *p = NULL;
	int ret = 0;
	struct target *target;
	struct scsi_lu *lu, *pos;
	struct device_type_template *t;

	dprintf("%d %" PRIu64 "\n", tid, lun);

	target = target_lookup(tid);
	if (!target)
		return TGTADM_NO_TARGET;

	lu = device_lookup(target, lun);
	if (lu) {
		eprintf("device %" PRIu64 " already exists\n", lun);
		return TGTADM_LUN_EXIST;
	}

	if (dev_type == TYPE_SPT)
		lu = zalloc(sizeof(*lu) + sg_bst.bs_datasize);
	else
		lu = zalloc(sizeof(*lu) + target->bst->bs_datasize);

	if (!lu)
		return TGTADM_NOMEM;

	t = device_type_lookup(dev_type);
	if (t) {
		lu->dev_type_template = *t;
		lu->bst = target->bst;
	} else {
		eprintf("Unknown device type %d\n", dev_type);
		ret = TGTADM_INVALID_REQUEST;
		goto free_lu;
	}

	lu->tgt = target;

	lu->lun = lun;
	lu->lu_state = SCSI_LU_RUNNING;
	tgt_cmd_queue_init(&lu->cmd_queue);

 	if (lu->dev_type_template.lu_init) {
		ret = lu->dev_type_template.lu_init(lu);
		if (ret)
			goto free_lu;
	}

	if (backing) {
		if (!*args)
			return TGTADM_INVALID_REQUEST;

		p = strchr(args, '=');
		if (!p)
			return TGTADM_INVALID_REQUEST;
		p++;

		ret = tgt_device_path_update(target, lu, p);
		if (ret)
			goto free_lu;
	}

	if (tgt_drivers[target->lid]->lu_create)
		tgt_drivers[target->lid]->lu_create(lu);

	list_for_each_entry(pos, &target->device_list, device_siblings) {
		if (lu->lun < pos->lun)
			break;
	}
	list_add_tail(&lu->device_siblings, &pos->device_siblings);

	dprintf("Add a logical unit %" PRIu64 " to the target %d\n", lun, tid);
	return ret;
free_lu:
	free(lu);
	return ret;
}

int tgt_device_destroy(int tid, uint64_t lun, int force)
{
	struct target *target;
	struct scsi_lu *lu;

	dprintf("%u %" PRIu64 "\n", tid, lun);

	/* lun0 is special */
	if (!lun && !force)
		return TGTADM_INVALID_REQUEST;

	lu = __device_lookup(tid, lun, &target);
	if (!lu) {
		eprintf("device %" PRIu64 " not found\n", lun);
		return TGTADM_NO_LUN;
	}

	if (!list_empty(&lu->cmd_queue.queue) || lu->cmd_queue.active_cmd)
		return TGTADM_LUN_ACTIVE;

	if (lu->dev_type_template.lu_exit)
		lu->dev_type_template.lu_exit(lu);

	if (lu->path) {
		free(lu->path);
		lu->bst->bs_close(lu);
	}

	list_del(&lu->device_siblings);
	free(lu);

	return 0;
}

struct lu_phy_attr *lu_attr_lookup(int tid, uint64_t lun)
{
	struct target *target;
	struct scsi_lu *lu;

	lu = __device_lookup(tid, lun, &target);
	if (!lu)
		return NULL;
	return &lu->attrs;
}

/**
 * dtd_load_unload  --  Load / unload media
 * @tid:	Target ID
 * @lun:	LUN
 * @load:	True if load, not true - unload
 * @file:	filename of 'media' top open
 *
 * load/unload media from the DATA TRANSFER DEVICE.
 */
int dtd_load_unload(int tid, uint64_t lun, int load, char *file)
{
	struct target *target;
	struct scsi_lu *lu;
	int err = TGTADM_SUCCESS;

	lu = __device_lookup(tid, lun, &target);
	if (!lu)
		return TGTADM_NO_LUN;

	if (!lu->attrs.removable)
		return TGTADM_INVALID_REQUEST;

	if (lu->path) {
		close(lu->fd);
		free(lu->path);
		lu->path = NULL;
	}

	lu->size = 0;
	lu->fd = 0;
	lu->attrs.online = 0;

	if (load) {
		lu->path = strdup(file);
		if (!lu->path)
			return TGTADM_NOMEM;
		lu->fd = backed_file_open(file, O_RDWR|O_LARGEFILE, &lu->size);
		if (lu->fd < 0)
			return TGTADM_UNSUPPORTED_OPERATION;
		lu->attrs.online = 1;
	}
	return err;
}

int device_reserve(struct scsi_cmd *cmd)
{
	struct scsi_lu *lu;

	lu = device_lookup(cmd->c_target, cmd->dev->lun);
	if (!lu) {
		eprintf("invalid target and lun %d %" PRIu64 "\n",
			cmd->c_target->tid, cmd->dev->lun);
		return 0;
	}

	if (lu->reserve_id && lu->reserve_id != cmd->cmd_itn_id) {
		dprintf("already reserved %" PRIu64 " %" PRIu64 "\n",
			lu->reserve_id, cmd->cmd_itn_id);
		return -EBUSY;
	}

	lu->reserve_id = cmd->cmd_itn_id;
	return 0;
}

int device_release(int tid, uint64_t itn_id, uint64_t lun, int force)
{
	struct target *target;
	struct scsi_lu *lu;

	lu = __device_lookup(tid, lun, &target);
	if (!lu) {
		eprintf("invalid target and lun %d %" PRIu64 "\n", tid, lun);
		return 0;
	}

	if (force || lu->reserve_id == itn_id) {
		lu->reserve_id = 0;
		return 0;
	}

	return -EBUSY;
}

int device_reserved(struct scsi_cmd *cmd)
{
	struct scsi_lu *lu;

	lu = device_lookup(cmd->c_target, cmd->dev->lun);
	if (!lu || !lu->reserve_id || lu->reserve_id == cmd->cmd_itn_id)
		return 0;
	return -EBUSY;
}

int tgt_device_update(int tid, uint64_t dev_id, char *params)
{
	int err = TGTADM_INVALID_REQUEST;
	struct target *target;
	struct scsi_lu *lu;

	target = target_lookup(tid);
	if (!target)
		return TGTADM_NO_TARGET;

	lu = device_lookup(target, dev_id);
	if (!lu) {
		eprintf("device %" PRIu64 " not found\n", dev_id);
		return TGTADM_NO_LUN;
	}

	if (lu->dev_type_template.lu_config)
		err = lu->dev_type_template.lu_config(lu, params);

	return err;
}

static int cmd_enabled(struct tgt_cmd_queue *q, struct scsi_cmd *cmd)
{
	int enabled = 0;

	if (cmd->attribute != MSG_SIMPLE_TAG)
		dprintf("non simple attribute %" PRIx64 " %x %" PRIu64 " %d\n",
			cmd->tag, cmd->attribute, cmd->dev ? cmd->dev->lun : UINT64_MAX,
			q->active_cmd);

	switch (cmd->attribute) {
	case MSG_SIMPLE_TAG:
		if (!queue_blocked(q))
			enabled = 1;
		break;
	case MSG_ORDERED_TAG:
		if (!queue_blocked(q) && !queue_active(q))
			enabled = 1;
		break;
	case MSG_HEAD_TAG:
		enabled = 1;
		break;
	default:
		eprintf("unknown command attribute %x\n", cmd->attribute);
		cmd->attribute = MSG_ORDERED_TAG;
		if (!queue_blocked(q) && !queue_active(q))
			enabled = 1;
	}

	return enabled;
}

static void cmd_post_perform(struct tgt_cmd_queue *q, struct scsi_cmd *cmd)
{
	q->active_cmd++;
	switch (cmd->attribute) {
	case MSG_ORDERED_TAG:
	case MSG_HEAD_TAG:
		set_queue_blocked(q);
		break;
	}
}

int target_cmd_queue(int tid, struct scsi_cmd *cmd)
{
	struct target *target;
	struct tgt_cmd_queue *q;
	struct it_nexus *itn;
	int result, enabled = 0;
	uint64_t dev_id, itn_id = cmd->cmd_itn_id;

	itn = it_nexus_lookup(tid, itn_id);
	if (!itn) {
		eprintf("invalid nexus %d %" PRIx64 "\n", tid, itn_id);
		return -ENOENT;
	}

	cmd->c_target = target = itn->nexus_target;

	dev_id = scsi_get_devid(target->lid, cmd->lun);
	cmd->dev_id = dev_id;
	dprintf("%p %x %" PRIx64 "\n", cmd, cmd->scb[0], dev_id);
	cmd->dev = device_lookup(target, dev_id);
	/* use LUN0 */
	if (!cmd->dev)
		cmd->dev = list_entry(target->device_list.next, struct scsi_lu,
				      device_siblings);

	/* service delivery or target failure */
	if (target->target_state != SCSI_TARGET_RUNNING ||
	    (cmd->dev->lu_state != SCSI_LU_RUNNING))
		return -EBUSY;

	cmd_hlist_insert(itn, cmd);

	q = &cmd->dev->cmd_queue;

	enabled = cmd_enabled(q, cmd);
	dprintf("%p %x %" PRIx64 " %d\n", cmd, cmd->scb[0], dev_id, enabled);

	if (enabled) {
		result = scsi_cmd_perform(itn->host_no, cmd);

		cmd_post_perform(q, cmd);

		dprintf("%" PRIx64 " %x %" PRIx64 " %" PRIu64 " %u %d %d\n",
			cmd->tag, cmd->scb[0], cmd->uaddr, cmd->offset,
			cmd->len, result, cmd->async);

		set_cmd_processed(cmd);
		if (!cmd->async)
			tgt_drivers[target->lid]->cmd_end_notify(itn_id, result, cmd);
	} else {
		set_cmd_queued(cmd);
		dprintf("blocked %" PRIx64 " %x %" PRIu64 " %d\n",
			cmd->tag, cmd->scb[0], cmd->dev->lun, q->active_cmd);

		list_add_tail(&cmd->qlist, &q->queue);
	}

	return 0;
}

void target_cmd_io_done(struct scsi_cmd *cmd, int result)
{
	tgt_drivers[cmd->c_target->lid]->cmd_end_notify(cmd->cmd_itn_id,
							result, cmd);
	return;
}

static void post_cmd_done(struct tgt_cmd_queue *q)
{
	struct scsi_cmd *cmd, *tmp;
	int enabled, result;

	list_for_each_entry_safe(cmd, tmp, &q->queue, qlist) {
		enabled = cmd_enabled(q, cmd);
		if (enabled) {
			struct it_nexus *nexus;

			nexus = it_nexus_lookup(cmd->c_target->tid, cmd->cmd_itn_id);
			if (!nexus)
				eprintf("BUG: %" PRIu64 "\n", cmd->cmd_itn_id);

			list_del(&cmd->qlist);
			dprintf("perform %" PRIx64 " %x\n", cmd->tag, cmd->attribute);
			result = scsi_cmd_perform(nexus->host_no, cmd);
			cmd_post_perform(q, cmd);
			set_cmd_processed(cmd);
			if (!cmd->async) {
				tgt_drivers[cmd->c_target->lid]->cmd_end_notify(
					cmd->cmd_itn_id, result, cmd);
			}
		} else
			break;
	}
}

static void __cmd_done(struct target *target, struct scsi_cmd *cmd)
{
	struct tgt_cmd_queue *q;
	int err;

	cmd_hlist_remove(cmd);

	err = target->bst->bs_cmd_done(cmd);

	dprintf("%d %" PRIx64 " %u %d\n", cmd->mmapped, cmd->uaddr, cmd->len, err);

	q = &cmd->dev->cmd_queue;
	q->active_cmd--;
	switch (cmd->attribute) {
	case MSG_ORDERED_TAG:
	case MSG_HEAD_TAG:
		clear_queue_blocked(q);
		break;
	}

	post_cmd_done(q);
}

struct scsi_cmd *target_cmd_lookup(int tid, uint64_t itn_id, uint64_t tag)
{
	struct scsi_cmd *cmd;

	cmd = cmd_lookup(tid, itn_id, tag);
	if (!cmd)
		eprintf("Cannot find cmd %d %" PRIx64 " %" PRIx64 "\n",
			tid, itn_id, tag);

	return cmd;
}

void target_cmd_done(struct scsi_cmd *cmd)
{
	struct mgmt_req *mreq;

	mreq = cmd->mreq;
	if (mreq && !--mreq->busy) {
		mreq->result = mreq->function == ABORT_TASK ? -EEXIST : 0;
		mreq->itn_id = cmd->cmd_itn_id;
		tgt_drivers[cmd->c_target->lid]->mgmt_end_notify(mreq);
		free(mreq);
	}

	__cmd_done(cmd->c_target, cmd);
}

static int abort_cmd(struct target* target, struct mgmt_req *mreq,
		     struct scsi_cmd *cmd)
{
	int err = 0;

	eprintf("found %" PRIx64 " %lx\n", cmd->tag, cmd->state);

	if (cmd_processed(cmd)) {
		/*
		 * We've already sent this command to kernel space.
		 * We'll send the tsk mgmt response when we get the
		 * completion of this command.
		 */
		cmd->mreq = mreq;
		err = -EBUSY;
	} else {
		__cmd_done(target, cmd);
		tgt_drivers[cmd->c_target->lid]->cmd_end_notify(cmd->cmd_itn_id,
								TASK_ABORTED, cmd);
	}
	return err;
}

static int abort_task_set(struct mgmt_req *mreq, struct target* target,
			  uint64_t itn_id, uint64_t tag, uint8_t *lun, int all)
{
	struct scsi_cmd *cmd, *tmp;
	struct it_nexus *itn;
	int i, err, count = 0;

	eprintf("found %" PRIx64 " %d\n", tag, all);

	list_for_each_entry(itn, &target->it_nexus_list, nexus_siblings) {
		for (i = 0; i < ARRAY_SIZE(itn->cmd_hash_list); i++) {
			struct list_head *list = &itn->cmd_hash_list[i];
			list_for_each_entry_safe(cmd, tmp, list, c_hlist) {
				if ((all && itn->itn_id == itn_id) ||
				    (cmd->tag == tag && itn->itn_id == itn_id) ||
				    (lun && !memcmp(cmd->lun, lun, sizeof(cmd->lun)))) {
					err = abort_cmd(target, mreq, cmd);
					if (err)
						mreq->busy++;
					count++;
				}
			}
		}
	}
	return count;
}

void target_mgmt_request(int tid, uint64_t itn_id, uint64_t req_id,
			 int function, uint8_t *lun, uint64_t tag, int host_no)
{
	struct target *target;
	struct mgmt_req *mreq;
	int err = 0, count, send = 1;

	target = target_lookup(tid);
	if (!target) {
		eprintf("invalid tid %d\n", tid);
		return;
	}

	mreq = zalloc(sizeof(*mreq));
	if (!mreq)
		return;
	mreq->mid = req_id;
	mreq->itn_id = itn_id;
	mreq->function = function;
	mreq->host_no = host_no;

	switch (function) {
	case ABORT_TASK:
		count = abort_task_set(mreq, target, itn_id, tag, NULL, 0);
		if (mreq->busy)
			send = 0;
		if (!count)
			err = -EEXIST;
		break;
	case ABORT_TASK_SET:
		count = abort_task_set(mreq, target, itn_id, 0, NULL, 1);
		if (mreq->busy)
			send = 0;
		break;
	case CLEAR_ACA:
	case CLEAR_TASK_SET:
		eprintf("Not supported yet %x\n", function);
		err = -EINVAL;
		break;
	case LOGICAL_UNIT_RESET:
		device_release(target->tid, itn_id,
			       scsi_get_devid(target->lid, lun), 1);
		count = abort_task_set(mreq, target, itn_id, 0, lun, 0);
		if (mreq->busy)
			send = 0;
		break;
	default:
		err = -EINVAL;
		eprintf("Unknown task management %x\n", function);
	}

	if (send) {
		mreq->result = err;
		tgt_drivers[target->lid]->mgmt_end_notify(mreq);
		free(mreq);
	}
}

struct account_entry {
	int aid;
	char *user;
	char *password;
	struct list_head account_siblings;
};

static LIST_HEAD(account_list);

static struct account_entry *__account_lookup_id(int aid)
{
	struct account_entry *ac;

	list_for_each_entry(ac, &account_list, account_siblings)
		if (ac->aid == aid)
			return ac;
	return NULL;
}

static struct account_entry *__account_lookup_user(char *user)
{
	struct account_entry *ac;

	list_for_each_entry(ac, &account_list, account_siblings)
		if (!strcmp(ac->user, user))
			return ac;
	return NULL;
}

int account_lookup(int tid, int type, char *user, int ulen, char *password, int plen)
{
	int i;
	struct target *target;
	struct account_entry *ac;

	target = target_lookup(tid);
	if (!target)
		return -ENOENT;

	if (type == ACCOUNT_TYPE_INCOMING) {
		for (i = 0; target->account.nr_inaccount; i++) {
			ac = __account_lookup_id(target->account.in_aids[i]);
			if (ac) {
				if (!strcmp(ac->user, user))
					goto found;
			}
		}
	} else {
		ac = __account_lookup_id(target->account.out_aid);
		if (ac) {
			strncpy(user, ac->user, ulen);
			goto found;
		}
	}

	return -ENOENT;
found:
	strncpy(password, ac->password, plen);
	return 0;
}

int account_add(char *user, char *password)
{
	int aid;
	struct account_entry *ac;

	ac = __account_lookup_user(user);
	if (ac)
		return TGTADM_USER_EXIST;

	for (aid = 1; __account_lookup_id(aid) && aid < INT_MAX; aid++)
		;
	if (aid == INT_MAX)
		return TGTADM_TOO_MANY_USER;

	ac = zalloc(sizeof(*ac));
	if (!ac)
		return TGTADM_NOMEM;

	ac->aid = aid;
	ac->user = strdup(user);
	if (!ac->user)
		goto free_account;

	ac->password = strdup(password);
	if (!ac->password)
		goto free_username;

	list_add(&ac->account_siblings, &account_list);
	return 0;
free_username:
	free(ac->user);
free_account:
	free(ac);
	return TGTADM_NOMEM;
}

static int __inaccount_bind(struct target *target, int aid)
{
	int i;

	/* first, check whether we already have this account. */
	for (i = 0; i < target->account.max_inaccount; i++)
		if (target->account.in_aids[i] == aid)
			return TGTADM_USER_EXIST;

	if (target->account.nr_inaccount < target->account.max_inaccount) {
		for (i = 0; i < target->account.max_inaccount; i++)
			if (!target->account.in_aids[i])
				break;
		if (i == target->account.max_inaccount) {
			eprintf("bug %d\n", target->account.max_inaccount);
			return TGTADM_UNKNOWN_ERR;
		}

		target->account.in_aids[i] = aid;
		target->account.nr_inaccount++;
	} else {
		int new_max = target->account.max_inaccount << 1;
		int *buf;

		buf = zalloc(new_max * sizeof(int));
		if (!buf)
			return TGTADM_NOMEM;

		memcpy(buf, target->account.in_aids,
		       target->account.max_inaccount * sizeof(int));
		free(target->account.in_aids);
		target->account.in_aids = buf;
		target->account.in_aids[target->account.max_inaccount] = aid;
		target->account.max_inaccount = new_max;
	}

	return 0;
}

int account_ctl(int tid, int type, char *user, int bind)
{
	struct target *target;
	struct account_entry *ac;
	int i, err = 0;

	target = target_lookup(tid);
	if (!target)
		return TGTADM_NO_TARGET;

	ac = __account_lookup_user(user);
	if (!ac)
		return TGTADM_NO_USER;

	if (bind) {
		if (type == ACCOUNT_TYPE_INCOMING)
			err = __inaccount_bind(target, ac->aid);
		else {
			if (target->account.out_aid)
				err = TGTADM_OUTACCOUNT_EXIST;
			else
				target->account.out_aid = ac->aid;
		}
	} else
		if (type == ACCOUNT_TYPE_INCOMING) {
			for (i = 0; i < target->account.max_inaccount; i++)
				if (target->account.in_aids[i] == ac->aid) {
					target->account.in_aids[i] = 0;
					target->account.nr_inaccount--;
					break;
				}

			if (i == target->account.max_inaccount)
				err = TGTADM_NO_USER;
		} else
			if (target->account.out_aid)
				target->account.out_aid = 0;
			else
				err = TGTADM_NO_USER;

	return err;
}

void account_del(char *user)
{
	struct account_entry *ac;
	struct target *target;

	ac = __account_lookup_user(user);
	if (!ac)
		return;

	list_for_each_entry(target, &target_list, target_siblings) {
		account_ctl(target->tid, ACCOUNT_TYPE_INCOMING, ac->user, 0);
		account_ctl(target->tid, ACCOUNT_TYPE_OUTGOING, ac->user, 0);
	}

	list_del(&ac->account_siblings);
	free(ac->user);
	free(ac->password);
	free(ac);
}

int account_available(int tid, int dir)
{
	struct target *target;

	target = target_lookup(tid);
	if (!target)
		return 0;

	if (dir == ACCOUNT_TYPE_INCOMING)
		return target->account.nr_inaccount;
	else
		return target->account.out_aid;
}

int acl_add(int tid, char *address)
{
	char *str;
	struct target *target;
	struct acl_entry *acl, *tmp;

	target = target_lookup(tid);
	if (!target)
		return TGTADM_NO_TARGET;

	list_for_each_entry_safe(acl, tmp, &target->acl_list, aclent_list)
		if (!strcmp(address, acl->address))
			return TGTADM_ACL_EXIST;

	acl = zalloc(sizeof(*acl));
	if (!acl)
		return TGTADM_NOMEM;

	str = strdup(address);
	if (!str) {
		free(acl);
		return TGTADM_NOMEM;
	}

	acl->address = str;
	list_add_tail(&acl->aclent_list, &target->acl_list);

	return 0;
}

void acl_del(int tid, char *address)
{
	struct target *target;
	struct acl_entry *acl, *tmp;

	target = target_lookup(tid);
	if (!target)
		return;

	list_for_each_entry_safe(acl, tmp, &target->acl_list, aclent_list) {
		if (!strcmp(address, acl->address)) {
			list_del(&acl->aclent_list);
			free(acl->address);
			free(acl);
			break;
		}
	}
}

char *acl_get(int tid, int idx)
{
	int i = 0;
	struct target *target;
	struct acl_entry *acl;

	target = target_lookup(tid);
	if (!target)
		return NULL;

	list_for_each_entry(acl, &target->acl_list, aclent_list) {
		if (idx == i++)
			return acl->address;
	}

	return NULL;
}

/*
 * if we have lots of host, use something like radix tree for
 * efficiency.
 */
static LIST_HEAD(bound_host_list);

struct bound_host {
	int host_no;
	struct target *target;
	struct list_head bhost_siblings;
};

int tgt_bind_host_to_target(int tid, int host_no)
{
	struct target *target;
	struct bound_host *bhost;

	target = target_lookup(tid);
	if (!target) {
		eprintf("can't find a target %d\n", tid);
		return -ENOENT;
	}

	list_for_each_entry(bhost, &bound_host_list, bhost_siblings) {
		if (bhost->host_no == host_no) {
			eprintf("already bound %d\n", host_no);
			return -EINVAL;
		}
	}

	bhost = malloc(sizeof(*bhost));
	if (!bhost)
		return -ENOMEM;

	bhost->host_no = host_no;
	bhost->target = target;

	list_add(&bhost->bhost_siblings, &bound_host_list);

	dprintf("bound the scsi host %d to the target %d\n", host_no, host_no);

	return 0;
}

int tgt_unbind_host_to_target(int tid, int host_no)
{
	struct bound_host *bhost;

	list_for_each_entry(bhost, &bound_host_list, bhost_siblings) {
		if (bhost->host_no == host_no) {
			if (!list_empty(&bhost->target->it_nexus_list)) {
				eprintf("the target has IT_nexus\n");
				return -EBUSY;
			}
			list_del(&bhost->bhost_siblings);
			free(bhost);
			return 0;
		}
	}
	return -ENOENT;
}

int tgt_bound_target_lookup(int host_no)
{
	struct bound_host *bhost;

	list_for_each_entry(bhost, &bound_host_list, bhost_siblings) {
		if (bhost->host_no == host_no)
			return bhost->target->tid;
	}

	return -ENOENT;
}

enum scsi_target_state tgt_get_target_state(int tid)
{
	struct target *target;

	target = target_lookup(tid);
	if (!target)
		return -ENOENT;
	return target->target_state;
}

static struct {
	enum scsi_target_state value;
	char *name;
} target_state[] = {
	{SCSI_TARGET_OFFLINE, "offline"},
	{SCSI_TARGET_RUNNING, "running"},
};

static char *target_state_name(enum scsi_target_state state)
{
	int i;
	char *name = NULL;

	for (i = 0; i < ARRAY_SIZE(target_state); i++) {
		if (target_state[i].value == state) {
			name = target_state[i].name;
			break;
		}
	}
	return name;
}

int tgt_set_target_state(int tid, char *str)
{
	int i, err = TGTADM_INVALID_REQUEST;
	struct target *target;

	target = target_lookup(tid);
	if (!target)
		return TGTADM_NO_TARGET;

	for (i = 0; i < ARRAY_SIZE(target_state); i++) {
		if (!strcmp(target_state[i].name, str)) {
			target->target_state = target_state[i].value;
			err = 0;
			break;
		}
	}

	return err;
}

static char *print_disksize(uint64_t size)
{
	static char buf[64];
	char *format[] = {"", "K", "M", "G", "T"};
	int i;

	memset(buf, 0, sizeof(buf));
	for (i = 1; size >= (1ULL << (i * 10)) && i < ARRAY_SIZE(format); i++)
		;
	i--;
	sprintf(buf, "%" PRIu64 "%s", size >> (i * 10), format[i]);
	return buf;
}

static struct {
	int value;
	char *name;
} disk_type_names[] = {
	{TYPE_DISK, "disk"},
	{TYPE_TAPE, "tape"},
	{TYPE_PRINTER, "printer"},
	{TYPE_PROCESSOR, "processor"},
	{TYPE_WORM, "worm"},
	{TYPE_ROM, "cd/dvd"},
	{TYPE_SCANNER, "scanner"},
	{TYPE_MOD, "optical"},
	{TYPE_MEDIUM_CHANGER, "changer"},
	{TYPE_COMM, "communication"},
	{TYPE_RAID, "controller"},
	{TYPE_ENCLOSURE, "enclosure"},
	{TYPE_RBC, "rbc"},
	{TYPE_OSD, "osd"},
	{TYPE_NO_LUN, "No LUN"}
};

static char *print_type(int type)
{
	int i;
	char *name = NULL;

	for (i = 0; i < ARRAY_SIZE(disk_type_names); i++) {
		if (disk_type_names[i].value == type) {
			name = disk_type_names[i].name;
			break;
		}
	}
	return name;
}


int tgt_target_show_all(char *buf, int rest)
{
	int total = 0, max = rest;
	struct target *target;
	struct scsi_lu *lu;
	struct acl_entry *acl;
	struct it_nexus *nexus;

	list_for_each_entry(target, &target_list, target_siblings) {
		shprintf(total, buf, rest,
			 "Target %d: %s\n"
			 _TAB1 "System information:\n"
			 _TAB2 "Driver: %s\n"
			 _TAB2 "Status: %s\n",
			 target->tid,
			 target->name,
			 tgt_drivers[target->lid]->name,
			 target_state_name(target->target_state));

		shprintf(total, buf, rest, _TAB1 "I_T nexus information:\n");

		list_for_each_entry(nexus, &target->it_nexus_list, nexus_siblings) {
			shprintf(total, buf, rest, _TAB2 "I_T nexus: %" PRIx64 "\n",
				 nexus->itn_id);
			if (nexus->info)
				shprintf(total, buf, rest, "%s", nexus->info);
		}

		shprintf(total, buf, rest, _TAB1 "LUN information:\n");
		list_for_each_entry(lu, &target->device_list, device_siblings)
			shprintf(total, buf, rest,
				 _TAB2 "LUN: %" PRIu64 "\n"
  				 _TAB3 "Type: %s\n"
				 _TAB3 "SCSI ID: %s\n"
				 _TAB3 "SCSI SN: %s\n"
				 _TAB3 "Size: %s\n"
				 _TAB3 "Online: %s\n"
				 _TAB3 "Poweron/Reset: %s\n"
				 _TAB3 "Removable media: %s\n"
				 _TAB3 "Backing store: %s\n",
				 lu->lun,
  				 print_type(lu->attrs.device_type),
				 lu->attrs.scsi_id,
				 lu->attrs.scsi_sn,
				 print_disksize(lu->size),
				 lu->attrs.online ? "Yes" : "No",
				 lu->attrs.reset ? "Yes" : "No",
				 lu->attrs.removable ? "Yes" : "No",
				 lu->path ? : "No backing store");

		if (!strcmp(tgt_drivers[target->lid]->name, "iscsi")) {
			int i, aid;

			shprintf(total, buf, rest, _TAB1
				 "Account information:\n");
			for (i = 0; i < target->account.nr_inaccount; i++) {
				aid = target->account.in_aids[i];
				shprintf(total, buf, rest, _TAB2 "%s\n",
					 __account_lookup_id(aid)->user);
			}
			if (target->account.out_aid) {
				aid = target->account.out_aid;
				shprintf(total, buf, rest,
					 _TAB2 "%s (outgoing)\n",
					 __account_lookup_id(aid)->user);
			}
		}

		shprintf(total, buf, rest, _TAB1 "ACL information:\n");
		list_for_each_entry(acl, &target->acl_list, aclent_list)
			shprintf(total, buf, rest, _TAB2 "%s\n", acl->address);
	}
	return total;
overflow:
	return max;
}

char *tgt_targetname(int tid)
{
	struct target *target;

	target = target_lookup(tid);
	if (!target)
		return NULL;

	return target->name;
}

#define DEFAULT_NR_ACCOUNT 16

int tgt_target_create(int lld, int tid, char *args)
{
	struct target *target, *pos;
	char *p, *q, *targetname = NULL;

	p = args;
	while ((q = strsep(&p, ","))) {
		char *str;

		str = strchr(q, '=');
		if (str) {
			*str++ = '\0';

			if (!strcmp("targetname", q))
				targetname = str;
			else
				eprintf("Unknow option %s\n", q);
		}
	};

	if (!targetname)
		return TGTADM_INVALID_REQUEST;

	target = target_lookup(tid);
	if (target) {
		eprintf("Target id %d already exists\n", tid);
		return TGTADM_TARGET_EXIST;
	}

	target = zalloc(sizeof(*target));
	if (!target)
		return TGTADM_NOMEM;

	target->name = strdup(targetname);
	if (!target->name) {
		free(target);
		return TGTADM_NOMEM;
	}

	target->account.in_aids = zalloc(DEFAULT_NR_ACCOUNT * sizeof(int));
	if (!target->account.in_aids) {
		free(target->name);
		free(target);
		return TGTADM_NOMEM;
	}
	target->account.max_inaccount = DEFAULT_NR_ACCOUNT;

	target->tid = tid;

	INIT_LIST_HEAD(&target->device_list);

	target->bst = tgt_drivers[lld]->default_bst;

	target->target_state = SCSI_TARGET_RUNNING;
	target->lid = lld;

	list_for_each_entry(pos, &target_list, target_siblings)
		if (target->tid < pos->tid)
			break;

	list_add_tail(&target->target_siblings, &pos->target_siblings);

	INIT_LIST_HEAD(&target->acl_list);
	INIT_LIST_HEAD(&target->it_nexus_list);

	tgt_device_create(tid, TYPE_RAID, 0, NULL, 0);

	if (tgt_drivers[lld]->target_create)
		tgt_drivers[lld]->target_create(target);

	dprintf("Succeed to create a new target %d\n", tid);

	return 0;
}

int tgt_target_destroy(int lld_no, int tid)
{
	int ret;
	struct target *target;
	struct acl_entry *acl, *tmp;
	struct scsi_lu *lu;

	target = target_lookup(tid);
	if (!target)
		return TGTADM_NO_TARGET;

	if (!list_empty(&target->it_nexus_list)) {
		eprintf("target %d still has it nexus\n", tid);
		return TGTADM_TARGET_ACTIVE;
	}

	while (!list_empty(&target->device_list)) {
		/* we remove lun0 last */
		lu = list_entry(target->device_list.prev, struct scsi_lu,
				device_siblings);
		ret = tgt_device_destroy(tid, lu->lun, 1);
		if (ret)
			return ret;
	}

	if (tgt_drivers[lld_no]->target_destroy)
		tgt_drivers[lld_no]->target_destroy(tid);

	list_del(&target->target_siblings);

	list_for_each_entry_safe(acl, tmp, &target->acl_list, aclent_list) {
		list_del(&acl->aclent_list);
		free(acl->address);
		free(acl);
	}

	free(target->account.in_aids);
	free(target->name);
	free(target);

	return 0;
}

int account_show(char *buf, int rest)
{
	int total = 0, max = rest;
	struct account_entry *ac;

	if (!list_empty(&account_list))
		shprintf(total, buf, rest, "Account list:\n");

	list_for_each_entry(ac, &account_list, account_siblings)
		shprintf(total, buf, rest, _TAB1 "%s\n", ac->user);

	return total;
overflow:
	return max;
}
