/*

    File: file_mrw.c

    Copyright (C) 1998-2005,2007 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License along
    with this program; if not, write the Free Software Foundation, Inc., 51
    Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include <stdio.h>
#include "types.h"
#include "common.h"
#include "photorec.h"

static const char* header_check_mrw(const unsigned char *buffer, const unsigned int buffer_size, const unsigned int safe_header_only,  t_file_recovery *file_recovery);
static int data_check_mrw(const unsigned char *buffer, const unsigned int buffer_size, t_file_recovery *file_recovery);
static void file_check_mrw(t_file_recovery *file_recovery);


const t_file_hint file_hint_mrw= {
  .extension="mrw",
  .description="Minolta Raw picture",
  .min_header_distance=0,
  .min_filesize=0,
  .max_filesize=PHOTOREC_MAX_FILE_SIZE,
  .recover=1,
  .header_check=&header_check_mrw,
  .data_check=&data_check_mrw,
  .file_check=&file_check_mrw
};

struct hdr {
  uint32_t fourcc;
  uint32_t size;
  char data[0];
}  __attribute__ ((__packed__));

struct prd {
  char ver[8];
  struct {
    uint16_t y;
    uint16_t x;
  } ccd;
  struct {
    uint16_t y;
    uint16_t x;
  } img;
  uint8_t datasize;  // bpp, 12 or 16
  uint8_t pixelsize; // bits used, always 12
  uint8_t storagemethod; // 0x52 means not packed
  uint8_t unknown1;
  uint16_t unknown2;
  uint16_t pattern; // 0x0001 RGGB, or 0x0004 GBRG
}  __attribute__ ((__packed__));

static uint64_t mrw_file_size=0;


/* Minolta */
static const char* header_check_mrw(const unsigned char *buffer, const unsigned int buffer_size, const unsigned int safe_header_only,  t_file_recovery *file_recovery)
{
  const unsigned char mrw_header[4]= { 0x00,'M','R','M'};			/* Minolta Raw */
  const unsigned char prd_header[4]= { 0x00,'P','R','D'};
  if(memcmp(buffer,mrw_header,sizeof(mrw_header))==0)
  {
    const struct hdr *mrmhdr = (const struct hdr*)buffer;
    if(memcmp(mrmhdr->data,prd_header,sizeof(prd_header))==0)
    {
      const struct hdr *prdhdr = (const struct hdr*)&mrmhdr->data;
      /* Picture Raw Dimensions */
      const struct prd *prd = (const struct prd*)&prdhdr->data;
      mrw_file_size= be32(mrmhdr->size)+ 8 +
	((uint64_t)be16(prd->ccd.x) * be16(prd->ccd.y) * prd->datasize + 8 - 1) / 8;
      /*
      ecrit_rapport("size=%lu x=%lu y=%lu datasize=%lu\n", be32(mrmhdr->size),
	  be16(prd->ccd.x), be16(prd->ccd.y), prd->datasize);
      ecrit_rapport("mrw_file_size %lu\n", (long unsigned)mrw_file_size);
      */
      return file_hint_mrw.extension;
    }
  }
  return NULL;
}

static int data_check_mrw(const unsigned char *buffer, const unsigned int buffer_size, t_file_recovery *file_recovery)
{
  if(file_recovery->file_size>=mrw_file_size)
  {
    file_recovery->file_size=mrw_file_size;
    return 2;
  }
  return 1;
}

static void file_check_mrw(t_file_recovery *file_recovery)
{
  if(file_recovery->file_size<mrw_file_size)
    file_recovery->file_size=0;
  else
    file_recovery->file_size=mrw_file_size;
}
