/*

    File: file_mp3.c

    Copyright (C) 1998-2006 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License along
    with this program; if not, write the Free Software Foundation, Inc., 59
    Temple Place - Suite 330, Boston MA 02111-1307, USA.

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include "types.h"
#include "common.h"
#include "photorec.h"
#include "intrf.h"	/* dump_log */

static const char* header_check_mp3(const unsigned char *buffer, const unsigned int buffer_size, const unsigned int safe_header_only,  t_file_recovery *file_recovery);
static int data_check_mp3(const unsigned char *buffer, const unsigned int buffer_size, t_file_recovery *file_recovery);


const t_file_hint file_hint_mp3= {
  .extension="mp3",
  .description="MP3 audio (MPEG ADTS, layer III, v1)",
  .min_header_distance=0,
  .min_filesize=287,
  .max_filesize=PHOTOREC_MAX_FILE_SIZE,
  .recover=1,
  .header_check=&header_check_mp3,
  .data_check=&data_check_mp3,
  .file_check=NULL
};

#define MPEG_V25	0
#define MPEG_V2		0x2
#define MPEG_V1		0x3
#define MPEG_L3	0x01
#define MPEG_L2	0x02
#define MPEG_L1	0x03

static unsigned int frame_offset=0;
  const unsigned int sample_rate_table[4][4]={
    {11025, 12000,  8000, 0},	/* MPEG_V25 */
    {    0,     0,     0, 0},
    {22050, 24000, 16000, 0},	/* MPEG_V2 */
    {44100, 48000, 32000, 0}	/* MPEG_V1 */
  };
  const unsigned int bit_rate_table[4][4][16]=
  {
    /* MPEG_V25 */
    /* MPEG_V_INVALID */
    {
      /* MPEG_L_INVALID */
      { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      /* MPEG_L3 */
      { 0, 8, 16, 24, 32, 40, 48, 56, 64, 80, 96, 112, 128, 144, 160, 0},
      /* MPEG_L2 */
      { 0, 8, 16, 24, 32, 40, 48, 56, 64, 80, 96, 112, 128, 144, 160, 0},
      /* MPEG_L1 */
      { 0, 32, 48, 56, 64, 80, 96, 112, 128, 144, 160, 176, 192, 224, 256, 0}
    },
    {
      /* MPEG_L_INVALID */
      { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      /* MPEG_L3 */
      { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      /* MPEG_L2 */
      { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      /* MPEG_L1 */
      { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
    },
    /* MPEG_V2 */
    {
      /* MPEG_L_INVALID */
      { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      /* MPEG_L3 */
      { 0, 8, 16, 24, 32, 40, 48, 56, 64, 80, 96, 112, 128, 144, 160, 0},
      /* MPEG_L2 */
      { 0, 8, 16, 24, 32, 40, 48, 56, 64, 80, 96, 112, 128, 144, 160, 0},
      /* MPEG_L1 */
      { 0, 32, 48, 56, 64, 80, 96, 112, 128, 144, 160, 176, 192, 224, 256, 0}
    },
    /* MPEG_V1 */
    {
      /* MPEG_L_INVALID */
      { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      /* MPEG_L3 */
      { 0, 32, 40, 48, 56, 64, 80, 96, 112, 128, 160, 192, 224, 256, 320, 0},
      /* MPEG_L2 */
      { 0, 32, 48, 56, 64, 80, 96, 112, 128, 160, 192, 224, 256, 320, 384, 0}, 
      /* MPEG_L1 */
      { 0, 32, 64, 96, 128, 160, 192, 224, 256, 288, 320, 352, 294, 416, 448, 0}
    },
  };

static const char* header_check_mp3(const unsigned char *buffer, const unsigned int buffer_size, const unsigned int safe_header_only,  t_file_recovery *file_recovery)
{
  /*
     	A Frame sync  				11 (length in bits)
    	B MPEG audio version (MPEG-1, 2, etc.) 	2
    	C MPEG layer (Layer I, II, III, etc.) 	2
    	D Protection (if on, then checksum follows header) 	1
	AAAA AAAA AAAB BCCD
	1111 1111 1111 1010

	http://www.dv.co.yu/mpgscript/mpeghdr.htm
  */
  unsigned int mpeg_version	=(buffer[1]>>3)&0x03;
  unsigned int mpeg_layer	=(buffer[1]>>1)&0x03;
  unsigned int bit_rate_key	=(buffer[2]>>4)&0x0F;
  unsigned int sampling_rate_key=(buffer[2]>>2)&0x03;
  unsigned int padding		=(buffer[2]>>1)&0x01;
  unsigned int bit_rate= bit_rate_table[mpeg_version][mpeg_layer][bit_rate_key]*1000;
  unsigned int sample_rate=sample_rate_table[mpeg_version][sampling_rate_key];
  unsigned int frameLengthInBytes=0;
  if(file_recovery!=NULL && file_recovery->file_stat!=NULL && file_recovery->file_stat->file_hint==&file_hint_mp3)
    return NULL;
  /*
  if(frame_offset!=0)
    return NULL;
    */
  if(buffer[0]==0xFF && ((buffer[1]&0xFE)==0xFA))
  {
    if(sample_rate>0 && bit_rate>0)
      frameLengthInBytes = 144 * bit_rate / sample_rate + padding;
      /*
    ecrit_rapport("bit_rate=%u\n",bit_rate);
    ecrit_rapport("sample_rate=%u\n",sample_rate);
    ecrit_rapport("frameLengthInBytes=%u\n",frameLengthInBytes);
    */
    if(frameLengthInBytes==0)
      return NULL;
    if(frameLengthInBytes<buffer_size)
    {
      if(buffer[frameLengthInBytes]==0xFF && ((buffer[frameLengthInBytes+1]&0xFE)==0xFA))
      {
	frame_offset=frameLengthInBytes;
	/*
	ecrit_rapport("header_check_mp3 mp3 found\n");
	*/
	return file_hint_mp3.extension;
      }
    }
  }
  return NULL;
}

static int data_check_mp3(const unsigned char *buffer, const unsigned int buffer_size, t_file_recovery *file_recovery)
{
  unsigned int i=frame_offset;
  if(file_recovery->file_size==0)
    i+=buffer_size/2;
  else
    i-=buffer_size/2;
  /*
  ecrit_rapport("data_check_mp3 0x%x 0x%x %lu\n",i,buffer_size,(long unsigned)file_recovery->file_size);
  dump_log(buffer,buffer_size);
  */
  while(i+4<buffer_size)
  {
    if(buffer[i+0]==0xFF && ((buffer[i+1]&0xE0)==0xE0))
    {
      unsigned int mpeg_version	=(buffer[i+1]>>3)&0x03;
      unsigned int mpeg_layer	=(buffer[i+1]>>1)&0x03;
      unsigned int bit_rate_key	=(buffer[i+2]>>4)&0x0F;
      unsigned int sampling_rate_key=(buffer[i+2]>>2)&0x03;
      unsigned int padding		=(buffer[i+2]>>1)&0x01;
      unsigned int bit_rate= bit_rate_table[mpeg_version][mpeg_layer][bit_rate_key]*1000;
      unsigned int sample_rate=sample_rate_table[mpeg_version][sampling_rate_key];
      unsigned int frameLengthInBytes=0;
      /*
      ecrit_rapport("frame_offset=%u\n",i);
      ecrit_rapport("bit_rate=%u\n",bit_rate);
      ecrit_rapport("sample_rate=%u\n",sample_rate);
      ecrit_rapport("frameLengthInBytes=%u\n",frameLengthInBytes);
      */
      if(sample_rate>0 && bit_rate>0)
	frameLengthInBytes = 144 * bit_rate / sample_rate + padding;
      if(frameLengthInBytes==0)
      {
	/*
	ecrit_rapport("data_check_mp3 END 0x%x 0x%x %lu\n",i,buffer_size,(long unsigned)file_recovery->file_size);
	*/
	if(i>(buffer_size/2))
	  i-=buffer_size/2;
	file_recovery->file_size+=i;
	frame_offset=0;
	return 2;
      }
      i+=frameLengthInBytes;
    }
    else
    {
      if(i>(buffer_size/2))
	i-=buffer_size/2;
      file_recovery->file_size+=i;
      frame_offset=0;
      return 2;
    }
  }
  frame_offset=i;
  /*
  ecrit_rapport("data_check_mp3 frame_offset=0x%x\n\n",frame_offset);
  */
  return 1;
}

