/*

    File: lvm.c

    Copyright (C) 2003-2005 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
 
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include "types.h"
#include "common.h"
#include "lvm.h"
#include "intrf.h"
#include "fnctdsk.h"
static int set_LVM_info(t_partition *partition, const pv_disk_t *pv);
static int set_LVM2_info(t_partition*partition, const unsigned char *buf);

int check_LVM(t_param_disk *disk_car,t_partition *partition,const int debug)
{
  unsigned char buffer[8*DEFAULT_SECTOR_SIZE];
  if(disk_car->read(disk_car,sizeof(buffer), &buffer, partition->part_offset)!=0)
  { return 1; }
  if(test_LVM(disk_car,(pv_disk_t *)&buffer,partition,debug,0)!=0)
    return 1;
  set_LVM_info(partition,(pv_disk_t *)&buffer);
  return 0;
}

int recover_LVM(t_param_disk *disk_car, const pv_disk_t *pv,t_partition *partition,const int debug, const int dump_ind)
{
  if(test_LVM(disk_car,pv,partition,debug,dump_ind)!=0)
    return 1;
  set_LVM_info(partition,pv);
  partition->part_type_i386=P_LVM;
  partition->part_type_sun=PSUN_LVM;
  partition->part_size=(uint64_t)pv->pv_size*disk_car->sector_size;
  if(debug>0)
  {
    ecrit_rapport("part_size %lu\n",(long unsigned)(partition->part_size/disk_car->sector_size));
  }
  return 0;
}

int test_LVM(t_param_disk *disk_car, const pv_disk_t *pv,t_partition *partition,const int debug, const int dump_ind)
{
  if ((strncmp((const char *)pv->id,LVM_ID,sizeof(pv->id)) == 0) && ((pv->version == 1) || (pv->version == 2)))
  {
    uint32_t size;
    if(debug>0 || dump_ind!=0)
    {
      ecrit_rapport("\nLVM magic value at %u/%u/%u\n", offset2cylinder(disk_car,partition->part_offset),offset2head(disk_car,partition->part_offset),offset2sector(disk_car,partition->part_offset));
    }
    if(dump_ind!=0)
    {
      /* There is a little offset ... */
      dump(stdscr,pv,DEFAULT_SECTOR_SIZE);
    }
    if (pv->pv_size > LVM_MAX_SIZE)
      return (1);
    if ((pv->pv_status != 0) && (pv->pv_status != PV_ACTIVE))
      return (1);
    if ((pv->pv_allocatable != 0) && (pv->pv_allocatable != PV_ALLOCATABLE))
      return (1);
    if (pv->lv_cur > MAX_LV)
      return (1);
    if (strlen((const char *)pv->vg_name) > NAME_LEN / 2)
      return (1);
    size = pv->pe_size / LVM_MIN_PE_SIZE * LVM_MIN_PE_SIZE;
    if ((pv->pe_size != size) ||
	(pv->pe_size < LVM_MIN_PE_SIZE) ||
	(pv->pe_size > LVM_MAX_PE_SIZE))
      return (1);

    if (pv->pe_total > ( pv->pe_on_disk.size / sizeof ( disk_pe_t)))
      return (1);
    if (pv->pe_allocated > pv->pe_total)
      return (1);
    partition->upart_type=UP_LVM;
    return 0;
  }
  return 1;
}

static int set_LVM_info(t_partition *partition, const pv_disk_t *pv)
{
  partition->name[0]='\0';
  partition->info[0]='\0';
  snprintf(partition->info,sizeof(partition->info),"LVM");
  return 0;
}

int check_LVM2(t_param_disk *disk_car,t_partition *partition,const int debug)
{
  unsigned char buffer[DEFAULT_SECTOR_SIZE];
  if(disk_car->read(disk_car,sizeof(buffer), &buffer, partition->part_offset+0x200)!=0)
  { return 1; }
  if(test_LVM2(disk_car,(const struct lvm2_label_header *)&buffer,partition,debug,0)!=0)
    return 1;
  set_LVM2_info(partition,buffer);
  return 0;
}

int recover_LVM2(t_param_disk *disk_car, const unsigned char *buf,t_partition *partition,const int debug, const int dump_ind)
{
  const struct lvm2_label_header *lh=(const struct lvm2_label_header *)buf;
  if(test_LVM2(disk_car,lh,partition,debug,dump_ind)!=0)
    return 1;
  set_LVM2_info(partition,buf);
  partition->part_type_i386=P_LVM;
  partition->part_type_sun=PSUN_LVM;
  {
    const struct lvm2_pv_header *pvhdr;
    pvhdr=(const struct lvm2_pv_header *) (buf + le32(lh->offset_xl));
    partition->part_size=le64(pvhdr->device_size_xl);
  }
  if(debug>0)
  {
    ecrit_rapport("part_size %lu\n",(long unsigned)(partition->part_size/disk_car->sector_size));
  }
  return 0;
}

int test_LVM2(t_param_disk *disk_car, const struct lvm2_label_header *lh,t_partition *partition,const int debug, const int dump_ind)
{
  if (memcmp((const char *)lh->type,LVM2_LABEL,sizeof(lh->type)) == 0)
  {
    if(debug>0 || dump_ind!=0)
    {
      ecrit_rapport("\nLVM2 magic value at %u/%u/%u\n", offset2cylinder(disk_car,partition->part_offset),offset2head(disk_car,partition->part_offset),offset2sector(disk_car,partition->part_offset));
    }
    if(le32(lh->offset_xl)>400)
      return 1;
    if(dump_ind!=0)
    {
      /* There is a little offset ... */
      dump(stdscr,lh,DEFAULT_SECTOR_SIZE);
    }
    partition->upart_type=UP_LVM2;
    return 0;
  }
  return 1;
}

static int set_LVM2_info(t_partition*partition, const unsigned char *buf)
{
  partition->name[0]='\0';
  partition->info[0]='\0';
  snprintf(partition->info,sizeof(partition->info),"LVM2");
  return 0;
}
