/*

    File: intrface.c

    Copyright (C) 1998-2005 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
 
#include <stdarg.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include <ctype.h>
#ifdef HAVE_TIME_H
#include <time.h>
#endif
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#include "types.h"
#include "common.h"
#include "lang.h"
#include "intrf.h"
#include "intrface.h"
#include "godmode.h"
#include "fnctdsk.h"
#include "testdisk.h"
#include "adv.h"
#include "analyse.h"
#include "chgtype.h"
#include "edit.h"
#include "savehdr.h"
#include "dir.h"
#include "ext2_dir.h"
#include "fat.h"
#include "fat_dir.h"
#include "ntfs_dir.h"
#include "rfs_dir.h"

extern const t_arch_fnct arch_i386;
extern const t_arch_fnct arch_sun;
extern const t_arch_fnct arch_none;
extern const t_arch_fnct arch_mac;

static void interface_options(int *paranoid,int *dump_ind,int *fast_mode,int *align, int *allow_partial_last_cylinder,int *ask_part_order,int *halt_on_errors, unsigned int *expert, const t_arch_fnct **arch, char**current_cmd);
static t_list_part *interface_load(t_param_disk *disk_car,t_list_part *list_part, const int debug);
static t_list_part *backup_merge(t_list_part *list_part,t_list_part *backup_part, const int debug);
static int write_MBR_code(t_param_disk *disk_car);

void interface_list(t_param_disk *disk_car, const int debug,const int test_recovery, const int saveheader, const int backup)
{
  t_list_part *list_part;
  ecrit_rapport("\nAnalyse ");
  ecrit_rapport("%s\n",disk_car->description(disk_car));
  printf("%s\n",disk_car->description(disk_car));
  printf(msg_PART_HEADER_LONG);
  list_part=disk_car->arch->read_part(disk_car,debug,saveheader);
  aff_buffer(BUFFER_WRITE,"Q");
  if(backup>0)
  {
    partition_save(disk_car,list_part,debug);
  }
  if(test_recovery>0)
  {
    t_list_part *element;
    ecrit_rapport("rebuild_FAT_BS()\n");
    for(element=list_part;element!=NULL;element=element->next)
    {
      if(is_fat(element->part))
      {
	aff_part_rapport(disk_car,element->part);
	rebuild_FAT_BS(disk_car,element->part,debug,0,0,0,NULL);	/* dump_ind */
      }
      /* TODO ntfs */
    }
  }
  delete_list_part(list_part);
}

static int write_MBR_code(t_param_disk *disk_car)
{
  if(disk_car->arch->write_MBR_code==NULL)
  {
    display_message("Function to write a new MBR code not implemented for this partition type.\n");
    return 1;
  }
  wdoprintf(stdscr,msg_WRITE_MBR_CODE);
  if(ask_YN(stdscr)!=0 && ask_confirmation("Write a new copy of MBR code, confirm ? (Y/N)")!=0)
  {
    if(disk_car->arch->write_MBR_code(disk_car))
    {
      display_message("Write error: Can't write new MBR code.\n");
      return 2;
    }
    else
      display_message("You have to reboot for the change to take effect.\n");
  }
  return 0;
}

int write_clean_table(t_param_disk *disk_car)
{
  if(disk_car->arch->erase_list_part==NULL)
  {
    display_message("Clear partition table not implemented for this partition type.\n");
    return 1;
  }
  wdoprintf(stdscr,msg_WRITE_CLEAN_TABLE);
  if(ask_YN(stdscr)!=0 && ask_confirmation("Clear partition table, confirm ? (Y/N)")!=0)
  {
    if(disk_car->arch->erase_list_part(disk_car))
    {
      display_message("Write error: Can't clear partition table.\n");
      return 2;
    }
    else
      display_message("You have to reboot for the change to take effect.\n");
  }
  return 0;
}

int do_curses_testdisk(int debug,int dump_ind, const t_list_disk *list_disk, const int saveheader, const t_arch_fnct **arch, const char *cmd_device, char *cmd_run)
{
  int align=2;
  int allow_partial_last_cylinder=0;
  int ask_part_order=0;
  int command;
  int done=0;
  int fast_mode=0;
  int halt_on_errors=1;
  int menu=0;
  int paranoid=0;
  int real_key;
  unsigned int expert=0;
  const t_list_disk *element_disk;
  const t_list_disk *current_disk=NULL;
  static struct MenuItem menuMain[]=
  {
	{'A',"Analyse","Analyse current partition structure and search for lost partitions"},
	{'T',"Advanced","Advanced boot recovery"},
	{'G',"Geometry", "Change disk geometry (expert only)" },
	{'O',"Options","Modify options"},
	{'D',"Delete","Delete all data in the partition table"},
	{'C',"MBR Code","Write the Classic MBR code to first sector"},
	{'Q',"Quit","Quit program"},
	{'E',"Editor","Basic disk editor"},
	{0,NULL,NULL}
  };
  char *current_cmd=cmd_run;
  if(cmd_device!=NULL)
  {
    for(element_disk=list_disk;element_disk!=NULL;element_disk=element_disk->next)
    {
      if(strcmp(element_disk->disk->device,cmd_device)==0)
	current_disk=element_disk;
    }
  }
  else
    current_disk=list_disk;
  if(current_disk==NULL)
    return 0;
  if(start_ncurses("TestDisk"))
    return 1;
  /* ncurses interface */
  while(done==0)
  {
    aff_copy(stdscr);
    wmove(stdscr,5,0);
    for(element_disk=list_disk;element_disk!=NULL;element_disk=element_disk->next)
    {
      if(element_disk!=current_disk)
	wdoprintf(stdscr,"%s\n",element_disk->disk->description(element_disk->disk));
      else
      {
	wattrset(stdscr,A_STANDOUT);
	wdoprintf(stdscr,"%s\n",element_disk->disk->description(element_disk->disk));
	wattroff(stdscr,A_STANDOUT);
      }
    }
    current_disk->disk->halt_on_errors=halt_on_errors;
    if(current_cmd!=NULL)
    {
      while(current_cmd[0]==',')
	current_cmd++;
      if(strncmp(current_cmd,"analyze",7)==0 || strncmp(current_cmd,"analyse",7)==0)
      {
	current_cmd+=7;
	command='A';
      }
      else if(strncmp(current_cmd,"geometry",8)==0)
      {
	current_cmd+=8;
	command='G';
      }
      else if(strncmp(current_cmd,"advanced",8)==0)
      {
	current_cmd+=8;
	command='T';
      }
      else if(strncmp(current_cmd,"options",7)==0)
      {
	current_cmd+=7;
	command='O';
      }
      else
      {
	if(current_cmd[0]!='\0')
	  ecrit_rapport("error >%s<\n",current_cmd);
	command='Q';
      }
    }
    else
      command = wmenuSelect_ext(stdscr,INTER_MAIN_Y, INTER_MAIN_X, menuMain, 8,
	    "AGDCOTQ", MENU_HORIZ | MENU_BUTTON | MENU_ACCEPT_OTHERS, &menu,&real_key);
    /* e for editor will be added when the editor will be better */
    switch(command)
    {
      case 'a':
      case 'A':
	interface_analyse(current_disk->disk,debug,saveheader,&current_cmd);
	interface_recovery(current_disk->disk,paranoid,debug, dump_ind, fast_mode,align,ask_part_order,expert,&current_cmd);
	break;
      case 'd':
      case 'D':
	wmove(stdscr,INTER_MAIN_Y,INTER_MAIN_X);
	write_clean_table(current_disk->disk);
	break;
      case 'c':
      case 'C':
	wmove(stdscr,INTER_MAIN_Y,INTER_MAIN_X);
	write_MBR_code(current_disk->disk);
	break;
      case 'g':
      case 'G':
	change_geometry(current_disk->disk,&current_cmd);
	break;
      case 'o':
      case 'O':
	{
	  const t_arch_fnct *old_arch=*arch;
	  int old_allow_partial_last_cylinder=allow_partial_last_cylinder;
	  interface_options(&paranoid,&dump_ind,&fast_mode,&align,&allow_partial_last_cylinder,&ask_part_order,&halt_on_errors,&expert,arch,&current_cmd);
	  for(element_disk=list_disk;element_disk!=NULL;element_disk=element_disk->next)
	  {
	    element_disk->disk->arch=*arch;
	  }
	  if((old_allow_partial_last_cylinder!=allow_partial_last_cylinder) ||
	      (old_arch!=*arch))
	    hd_parse_bis(list_disk,allow_partial_last_cylinder,*arch,debug);
	}
	break;
      case 't':
      case 'T':
	interface_adv(current_disk->disk, debug, dump_ind, expert,&current_cmd);
	break;
      case 'q':
      case 'Q':
	done = 1;
	break;
      case 'e':
      case 'E':
	interface_editor(current_disk->disk);
	break;
      case KEY_UP:
	if(current_disk->prev!=NULL)
	  current_disk=current_disk->prev;
	break;
      case KEY_DOWN:
	if(current_disk->next!=NULL)
	  current_disk=current_disk->next;
	break;
#ifdef DEBUG
      default:
	wmove(stdscr,10,0);
	wdoprintf(stdscr,"key %03X %d", command, command);
	wrefresh(stdscr);
	break;
#endif
    }
  }
  end_ncurses();
  ecrit_rapport("\n");
  return 0;
}

void interface_analyse(t_param_disk *disk_car, const int debug, const int saveheader, char**current_cmd)
{
  t_list_part *list_part;
  int command;
  /* log */
  struct MenuItem menuAnalyse[]=
  {
    { 'P', "Previous",""},
    { 'N', "Next","" },
    { 'Q',"Proceed","Try to locate partition"},
    { 'T', "Save","Save partition structure"},
    { 0, NULL, NULL }
  };
  ecrit_rapport("\nAnalyse ");
  ecrit_rapport("%s\n",disk_car->description(disk_car));
  aff_buffer(BUFFER_RESET,"Q");
  /* ncurses interface */
  aff_copy(stdscr);
  if(disk_car->arch->msg_part_type!=NULL)
    mvwaddstr(stdscr,21,0,disk_car->arch->msg_part_type);
  wmove(stdscr,4,0);
  wdoprintf(stdscr,"%s\n",disk_car->description(disk_car));
  mvwaddstr(stdscr,5,0,"Check current partition structure");
  wmove(stdscr,6,0);
  wrefresh(stdscr);
  wdoprintf(stdscr,msg_PART_HEADER_LONG);
  list_part=disk_car->arch->read_part(disk_car,debug,saveheader);
  ecrit_rapport("Current partitions:\n");
  if(*current_cmd!=NULL)
  {
    command='Q';
    while(*current_cmd[0]==',')
      (*current_cmd)++;
    if(strncmp(*current_cmd,"backup",6)==0)
    {
      (*current_cmd)+=6;
      if(list_part!=NULL)
	command='T';
    }
    screen_buffer_to_log();
  }
  else
    command=screen_buffer_display(stdscr,(list_part!=NULL?"QT":"Q"),menuAnalyse);
  if(command=='T')
  {
    ecrit_rapport("Backup partition structure\n");
    partition_save(disk_car,list_part,debug);
  }
  delete_list_part(list_part);
}

int interface_write(t_param_disk *disk_car,t_list_part *list_part,const int can_search_deeper, const int can_ask_minmax_ext, char **current_cmd, int *menu)
{
  t_list_part *parts;
  struct MenuItem menuWrite[]=
  {
    { 'P', "Previous",""},
    { 'N', "Next","" },
    { 'Q', "Quit","Return to main menu"},
    { 'S', "Search!","Search deeper, try to find more partitions"},
    { 'W', "Write","Write partition structure to disk"},
    { 'E', "Extd Part","Maximize/Minimize extended partition"},
    { 0, NULL, NULL }
  };
  int command;
  ecrit_rapport("\ninterface_write()\n");
  aff_buffer(BUFFER_RESET,"Q");
  aff_copy(stdscr);
  wmove(stdscr,4,0);
  wdoprintf(stdscr,"%s",disk_car->description(disk_car));
  wmove(stdscr,5,0);
  mvwaddstr(stdscr,6,0,msg_PART_HEADER_LONG);
  for(parts=list_part;parts!=NULL;parts=parts->next)
    if(parts->part->status!=STATUS_LOG)
      aff_part_buffer(AFF_PART_ORDER,disk_car,parts->part);
  for(parts=list_part;parts!=NULL;parts=parts->next)
    if(parts->part->status==STATUS_LOG)
      aff_part_buffer(AFF_PART_ORDER,disk_car,parts->part);
  if(list_part==NULL)
  {
    aff_buffer(BUFFER_ADD," \nNo partition found or selected for recovery");
    if(*current_cmd!=NULL)
    {
      while(*current_cmd[0]==',')
	(*current_cmd)++;
      if(strncmp(*current_cmd,"search",6)==0)
      {
	(*current_cmd)+=6;
	command='S';
      }
      else
	command='Q';
      screen_buffer_to_log();
    }
    else
      command=screen_buffer_display_ext(stdscr,(can_search_deeper?"S":""),menuWrite,menu);
  }
  else
  {
    if(*current_cmd!=NULL)
    {
      while(*current_cmd[0]==',')
	(*current_cmd)++;
      command='Q';
      if(strncmp(*current_cmd,"search",6)==0)
      {
	(*current_cmd)+=6;
	if(can_search_deeper)
	  command='S';
      }
      else if(strncmp(*current_cmd,"write",5)==0)
      {
	(*current_cmd)+=5;
	if(disk_car->arch->write_part!=NULL)
	  command='W';
      }
      screen_buffer_to_log();
    }
    else
    {
      char options[10];
      options[0]=0;
      if(can_search_deeper)
	strcat(options,"S");
      if(disk_car->arch->write_part!=NULL)
	strcat(options,"W");
      if(can_ask_minmax_ext)
	strcat(options,"E");
      command=screen_buffer_display_ext(stdscr,options,menuWrite,menu);
    }
  }
  return command;
}

static void interface_options(int *paranoid,int *dump_ind,int *fast_mode,int *align, int *allow_partial_last_cylinder,int *ask_part_order,int *halt_on_errors, unsigned int *expert, const t_arch_fnct **arch, char**current_cmd)
{
  /* paranoid: search in partition
   * dump_ind dump sector Y/N
   * fast_mode Partitions are aligned on cylinder boundary Y/N
   * */

  if(*current_cmd!=NULL)
  {
    int keep_asking=1;
    do
    {
      while(*current_cmd[0]==',')
	(*current_cmd)++;
      if(strncmp(*current_cmd,"partition_i386",14)==0)
      {
	(*current_cmd)+=14;
	*arch=&arch_i386;
      }
      else if(strncmp(*current_cmd,"partition_none",14)==0)
      {
	(*current_cmd)+=14;
	*arch=&arch_none;
      }
      else if(strncmp(*current_cmd,"partition_sun",13)==0)
      {
	(*current_cmd)+=13;
	*arch=&arch_sun;
      }
      else if(strncmp(*current_cmd,"partition_mac",13)==0)
      {
	(*current_cmd)+=13;
	*arch=&arch_mac;
      }
      else
	keep_asking=0;
    } while(keep_asking>0);
  }
  else
  {
    int done = FALSE;
    int menu = 9;
    /* ncurses interface */
    while (done==FALSE)
    {
      int car;
      int real_key;
      struct MenuItem menuOptions[]=
      {
	{ 'P', NULL, "Search all partitions" },
	{ 'D',NULL,"Dump essential sectors" },
	{ 'C',NULL,"Partitions are aligned on cylinder/head boundary" },
	{ 'F',NULL,"Don't check every hard disk head" },
	{ 'A',NULL,""},
	{ 'O',NULL,""},
	{ 'H',NULL,"Halt on Disk I/O errors" },
	{ 'E',NULL,"Expert mode"},
	{ 'T',NULL,"Partition table type"},
	{ 'Q',"[Ok]","Return to main menu"},
	{ 0, NULL, NULL }
      };
      menuOptions[0].name=*paranoid?"Paranoid : Yes":"Paranoid : No";
      menuOptions[1].name=*dump_ind?"Dump : Yes":"Dump : No";
      switch(*align)
      {
	case 0:
	  menuOptions[2].name="Cylinder boundary : No";
	  break;
	case 1:
	  menuOptions[2].name="Cylinder boundary : Head boundary only";
	  break;
	case 2:
	  menuOptions[2].name="Cylinder boundary : Yes";
	  break;
      }
      switch(*fast_mode)
      {
	case 0:
	  menuOptions[3].name="Mode : Fast";
	  break;
	case 1:
	  menuOptions[3].name="Mode : Medium";
	  break;
	default:
	  menuOptions[3].name="Mode : Slow";
	  break;
      }
      menuOptions[4].name=*allow_partial_last_cylinder?"Allow partial last cylinder : Yes":"Allow partial last cylinder : No";
      menuOptions[5].name=*ask_part_order?"Ask partition order : Yes":"Ask partition order : No";
      menuOptions[6].name=*halt_on_errors?"Halt on errors : Yes":"Halt on errors : No";
      menuOptions[7].name=*expert?"Expert mode : Yes":"Expert mode : No";
      menuOptions[8].name=(*arch)->part_name;
      aff_copy(stdscr);

      car=wmenuSelect_ext(stdscr,INTER_OPTION_Y, INTER_OPTION_X, menuOptions, 0, "DCAOHMETQ", MENU_VERT, &menu,&real_key);
      switch(car)
      {
	case 'p':
	case 'P':
	  *paranoid=!*paranoid;
	  break;
	case 'd':
	case 'D':
	  *dump_ind=!*dump_ind;
	  break;
	case 'c':
	case 'C':
	  if(*align<2)
	    (*align)++;
	  else
	    *align=0;
	  break;
	case 'f':
	case 'F':
	  if(*fast_mode<2)
	    (*fast_mode)++;
	  else
	    *fast_mode=0;
	  break;
	case 'a':
	case 'A':
	  *allow_partial_last_cylinder=!*allow_partial_last_cylinder;
	  break;
	case 'o':
	case 'O':
	  *ask_part_order=!*ask_part_order;
	  break;
	case 'h':
	case 'H':
	  *halt_on_errors=!*halt_on_errors;
	  break;
	case 'e':
	case 'E':
	  *expert=!*expert;
	  break;
	case 't':
	case 'T':
	  if(real_key==KEY_LEFT)
	  {
	    if(*arch==&arch_i386)
	      *arch=&arch_none;
	    else if(*arch==&arch_sun)
	      *arch=&arch_i386;
	    else if(*arch==&arch_mac)
	      *arch=&arch_sun;
	    else
	      *arch=&arch_mac;
	  }
	  else
	  {
	    if(*arch==&arch_i386)
	      *arch=&arch_sun;
	    else if(*arch==&arch_sun)
	      *arch=&arch_mac;
	    else if(*arch==&arch_mac)
	      *arch=&arch_none;
	    else
	      *arch=&arch_i386;
	  }
	  break;
	case key_ESC:
	case 'q':
	case 'Q':
	  done = TRUE;
	  break;
      }
    }
  }
  /* write new options to log file */
  ecrit_rapport("New options :\n Paranoid : %s\n Dump : %s\n ",
      *paranoid?"Yes":"No",
      *dump_ind?"Yes":"No");
  switch(*align)
  {
    case 0:
      ecrit_rapport("Cylinder boundary : No");
      break;
    case 1:
      ecrit_rapport("Cylinder boundary : Head boundary only");
      break;
    case 2:
      ecrit_rapport("Cylinder boundary : Yes");
      break;
  }
  ecrit_rapport("\n ");
  switch(*fast_mode)
  {
    case 0:
      ecrit_rapport("Mode : Fast");
      break;
    case 1:
      ecrit_rapport("Mode : Medium");
      break;
    default:
      ecrit_rapport("Mode : Slow");
      break;
  }
  ecrit_rapport("\n Allow partial last cylinder : %s\n Ask partition order : %s\n Halt on errors : %s\n Expert mode : %s\n %s\n",
      *allow_partial_last_cylinder?"Yes":"No",
      *ask_part_order?"Yes":"No",
      *halt_on_errors?"Yes":"No",
      *expert?"Yes":"No",
      (*arch)->part_name);
}

t_list_part *ask_structure(t_param_disk *disk_car,t_list_part *list_part, const int debug, char **current_cmd)
{
  int quit=0;
  int offset=0;
  int pos_num=0;
  t_list_part *pos=list_part;
  int rewrite=1;
  do
  {
    int i;
    int command;
    t_list_part *parts;
    int structure_status;
    if(rewrite)
    {
      aff_copy(stdscr);
      wmove(stdscr,4,0);
      wdoprintf(stdscr,"%s",disk_car->description(disk_car));
      mvwaddstr(stdscr,5,0,msg_PART_HEADER);
      rewrite=0;
    }
    structure_status=disk_car->arch->test_structure(list_part);
    for(i=0,parts=list_part;(parts!=NULL) && (i<offset);parts=parts->next,i++);
    for(i=offset;(parts!=NULL) &&((i-offset)<INTER_STRUCTURE);i++,parts=parts->next)
    {
      wmove(stdscr,6+i-offset,0);
      wclrtoeol(stdscr);	/* before addstr for BSD compatibility */
      if(parts==pos)
      {
	wattrset(stdscr,A_STANDOUT);
      }
      /* TODO status letter in green unless D(eleted) or structure_status!=0 */
      if(structure_status==0 && parts->part->status!=STATUS_DELETED && has_colors())
	wbkgdset(stdscr,' ' | COLOR_PAIR(2));
      aff_part(stdscr,AFF_PART_NONL,disk_car,parts->part);
      if(structure_status==0 && parts->part->status!=STATUS_DELETED && has_colors())
	wbkgdset(stdscr,' ' | COLOR_PAIR(0));
      if(parts==pos)
      {
	wattroff(stdscr,A_STANDOUT);
	wmove(stdscr,24,0);
	wclrtoeol(stdscr);	/* before addstr for BSD compatibility */
	if(parts->part->info[0]!='\0')
	{
	  wdoprintf(stdscr,"%s, ",parts->part->info);
	}
	wdoprintf(stdscr,"%lu MB",(long unsigned)(parts->part->part_size/(1024*1024)));
      }
    }
    if(structure_status==0)
      mvwaddstr(stdscr,20,0,msg_STRUCT_OK);
    else
    {
      if(has_colors())
	wbkgdset(stdscr,' ' | COLOR_PAIR(1));
      mvwaddstr(stdscr,20,0,msg_STRUCT_BAD);
      if(has_colors())
	bkgdset(' ' | COLOR_PAIR(0));
    }
    if(list_part!=NULL && disk_car->arch->msg_part_type!=NULL)
    {
      mvwaddstr(stdscr,20,16,"Use arrow keys to change partition characteristics:");
      mvwaddstr(stdscr,21,0,disk_car->arch->msg_part_type);
    }
    wmove(stdscr,22,0);
    wclrtoeol(stdscr);	/* before addstr for BSD compatibility */
    waddstr(stdscr,"Keys A: add partition, L: load backup, ");
    if(list_part==NULL)
    {
      waddstr(stdscr,"ENTER: to continue");
    }
    else
    {
      if(pos->part->arch==NULL || pos->part->arch==disk_car->arch)
      {
	waddstr(stdscr,"T: change type, ");
      }
      switch(pos->part->upart_type)
      {
	case UP_EXT2:
	case UP_EXT3:
	case UP_RFS:
	case UP_RFS2:
	case UP_RFS3:
	case UP_FAT12:
	case UP_FAT16:
	case UP_FAT32:
	case UP_NTFS:
	  waddstr(stdscr,"P: list files, ");
	  break;
	default:
	  break;
      }
      mvwaddstr(stdscr,23,5, "ENTER: to continue");
    }
    wrefresh(stdscr);
    if(*current_cmd!=NULL)
    {
      while(*current_cmd[0]==',')
	(*current_cmd)++;
      if(strncmp(*current_cmd,"list",4)==0)
      {
	(*current_cmd)+=4;
	command='p';
      }
      else
	command='q';
    }
    else
      command=wgetch(stdscr);
    switch(command)
    {
      case 'q':
      case '\r':
      case '\n':
      case KEY_ENTER:
#ifdef PADENTER
      case PADENTER:
#endif
      case 'M':
	quit=1;
	break;
      case KEY_UP:
	if(list_part!=NULL)
	{
	  only_one_bootable(list_part,pos);
	  if(pos->prev!=NULL)
	  {
	    pos=pos->prev;
	    pos_num--;
	  }
	  if(pos_num<offset)
	    offset--;
	}
	break;
      case KEY_DOWN:
	if(list_part!=NULL)
	{
	  only_one_bootable(list_part,pos);
	  if(pos->next!=NULL)
	  {
	    pos=pos->next;
	    pos_num++;
	  }
	  if(pos_num>=offset+INTER_STRUCTURE)
	    offset++;
	}
	break;
      case KEY_PPAGE:
	if(list_part!=NULL)
	{
	  only_one_bootable(list_part,pos);
	  for(i=0;(i<INTER_STRUCTURE) && (pos->prev!=NULL);i++)
	  {
	    pos=pos->prev;
	    pos_num--;
	    if(pos_num<offset)
	      offset--;
	  }
	}
	break;
      case KEY_NPAGE:
	if(list_part!=NULL)
	{
	  only_one_bootable(list_part,pos);
	  for(i=0;(i<INTER_STRUCTURE) && (pos->next!=NULL);i++)
	  {
	    pos=pos->next;
	    pos_num++;
	    if(pos_num>=offset+INTER_STRUCTURE)
	      offset++;
	  }
	}
	break;
      case KEY_RIGHT:
      case '+':
      case ' ':
	if(list_part!=NULL)
	{
	  if(pos->part->arch==NULL || pos->part->arch==disk_car->arch)
	    disk_car->arch->set_next_status(disk_car,pos->part);
	}
	break;
      case KEY_LEFT:
      case '-':
	if(list_part!=NULL)
	{
	  if(pos->part->arch==NULL || pos->part->arch==disk_car->arch)
	    disk_car->arch->set_prev_status(disk_car,pos->part);
	}
	break;
      case 'a':
      case 'A':
	{
	  list_part=disk_car->arch->add_partition(disk_car,list_part, debug);
	  rewrite=1;
	  offset=0;
	  pos_num=0;
	  pos=list_part;
	}
	break;
      case 't':
      case 'T':
	if(list_part!=NULL)
	{
	  if(pos->part->arch==NULL || pos->part->arch==disk_car->arch)
	  {
	    rewrite=1;
	    change_part_type(disk_car,pos->part);
	  }
	}
	break;
      case 'p':
      case 'P':
	if(list_part!=NULL)
	{
	  WINDOW *window=newwin(0,0,0,0);	/* full screen */
	  aff_copy(window);
	  ecrit_rapport("\n");
	  switch(pos->part->upart_type)
	  {
	    case UP_FAT12:
	    case UP_FAT16:
	    case UP_FAT32:
	      dir_partition_fat(window,disk_car,pos->part,debug,current_cmd);
	      break;
	    case UP_EXT2:
	    case UP_EXT3:
	      dir_partition_ext2(window,disk_car,pos->part,debug,current_cmd);
	      break;
	    case UP_RFS:
	    case UP_RFS2:
	    case UP_RFS3:
	      dir_partition_reiser(window,disk_car,pos->part,debug,current_cmd);
	      break;
	    case UP_NTFS:
	      dir_partition_ntfs(window,disk_car,pos->part,debug,current_cmd);
	      break;
	    default:
	      break;
	  }
	  delwin(window);
	  (void) clearok(stdscr, TRUE);
#ifdef HAVE_TOUCHWIN
	  touchwin(stdscr);
#endif
	}
	break;
      case 'l':
      case 'L':
        list_part=interface_load(disk_car,list_part,debug);
	rewrite=1;
	offset=0;
	pos_num=0;
	pos=list_part;
        break;
      default:
/*	ecrit_rapport("ask_structure command=%x\n",command); */
	break;
    }
  } while(quit==0);
  return list_part;
}

static t_list_part *backup_merge(t_list_part *list_part,t_list_part *backup_part, const int debug)
{
  t_list_part *partition;
  for(partition=backup_part;partition!=NULL;partition=partition->next)
  {
    t_partition *new_partition=partition_new();
    dup_t_partition(new_partition,partition->part);
    list_part=insert_new_partition(list_part, new_partition);
  }
  return list_part;
}

static t_list_part *interface_load(t_param_disk *disk_car,t_list_part *list_part, const int debug)
{
  t_backup_disk *list_backup=partition_load(disk_car,debug);
  {
    t_backup_disk *backup;
    ecrit_rapport("interface_load\n");
    for(backup=list_backup;backup!=NULL;backup=backup->next)
    {
      t_list_part *element;
      ecrit_rapport("%s %s",backup->description,ctime(&backup->my_time));
      for(element=backup->list_part;element!=NULL;element=element->next)
	aff_part_rapport(disk_car,element->part);
    }
  }
  {
    int quit=0;
    int offset=0;
    int pos_num=0;
    t_backup_disk *pos=list_backup;
    int rewrite=1;
    int menu=3;   /* default : quit */
    struct MenuItem menuLoadBackup[]=
    {
      { 'P', "Previous",""},
      { 'N', "Next","" },
      { 'L',"Load","Load partition structure from backup and try to location partition"},
      { 'Q',"Cancel","Don't use backup and try to locate partition"},
      { 0, NULL, NULL }
    };
    char options[20];
    do
    {
      int i;
      t_backup_disk *backup;
      if(rewrite)
      {
	aff_copy(stdscr);
	mvwaddstr(stdscr,4,0,disk_car->description(disk_car));
	if(list_backup!=NULL)
	{
	  mvwaddstr(stdscr,5,0,"Choose the backup you want to restore:");
	  mvwaddstr(stdscr,20,0,"PS: Don't worry you will have to confirm the partition restoration.");
	}
	else
	{
	  mvwaddstr(stdscr,5,0,"No backup found!");
	}
	rewrite=0;
      }
      if(list_backup!=NULL)
      {
	for(i=0,backup=list_backup;(backup!=NULL) && (i<offset);backup=backup->next,i++);
	for(i=offset;(backup!=NULL) &&((i-offset)<INTER_STRUCTURE);i++,backup=backup->next)
	{
	  wmove(stdscr,8+i-offset,0);
	  wclrtoeol(stdscr);	/* before addstr for BSD compatibility */
	  if(backup==pos)
	  {
	    wattrset(stdscr,A_STANDOUT);
	    wdoprintf(stdscr,"%s %s",backup->description,ctime(&backup->my_time));
	    wattroff(stdscr,A_STANDOUT);
	  } else
	  {
	    wdoprintf(stdscr,"%s %s",backup->description,ctime(&backup->my_time));
	  }
	}
	if(i<=INTER_STRUCTURE && backup==NULL)
	{
	  strncpy(options,"LQ",sizeof(options));
	  menu=0;
	}
	else
	{
	  strncpy(options,"PNLQ",sizeof(options));
	  menu=2;
	}
      }
      else
      {
	menu=0;
	strncpy(options,"Q",sizeof(options));
      }
      switch(wmenuSelect(stdscr,INTER_DUMP_Y,INTER_DUMP_X, menuLoadBackup, 8, options, MENU_HORIZ| MENU_BUTTON | MENU_ACCEPT_OTHERS, menu))
      {
	case 'q':
	case 'Q':
	  quit=1;
	  break;
	case 'l':
	case 'L':
	  quit=2;
	  break;
	case KEY_UP:
	  if(list_backup!=NULL)
	  {
	    if(pos->prev!=NULL)
	    {
	      pos=pos->prev;
	      pos_num--;
	    }
	    if(pos_num<offset)
	      offset--;
	  }
	  break;
	case KEY_DOWN:
	  if(list_backup!=NULL)
	  {
	    if(pos->next!=NULL)
	    {
	      pos=pos->next;
	      pos_num++;
	    }
	    if(pos_num>=offset+INTER_STRUCTURE)
	      offset++;
	  }
	  break;
	case KEY_PPAGE:
	  if(list_backup!=NULL)
	  {
	    for(i=0;(i<INTER_STRUCTURE) && (pos->prev!=NULL);i++)
	    {
	      pos=pos->prev;
	      pos_num--;
	      if(pos_num<offset)
		offset--;
	    }
	  }
	  break;
	case KEY_NPAGE:
	  if(list_backup!=NULL)
	  {
	    for(i=0;(i<INTER_STRUCTURE) && (pos->next!=NULL);i++)
	    {
	      pos=pos->next;
	      pos_num++;
	      if(pos_num>=offset+INTER_STRUCTURE)
		offset++;
	    }
	  }
	  break;
	default:
	  /*	ecrit_rapport("ask_structure car=%x\n",car); */
	  break;
      }
    } while(quit==0);
    if(pos!=NULL && quit==2)
    {
      t_list_part *partition;
      for(partition=pos->list_part;partition!=NULL;partition=partition->next)
      {
	/* Check partition and load partition name */
	disk_car->arch->check_part(disk_car,debug,partition->part,0);
      }
      list_part=backup_merge(list_part,pos->list_part,debug);
    }
  }
  { /* Cleanup */
    t_backup_disk *backup;
    backup=list_backup;
    while(backup!=NULL)
    {
      t_backup_disk *next=backup->next;
      delete_list_part(backup->list_part);
      FREE(backup);
      backup=next;
    }
  }
  return list_part;
}

int interface_superblock(t_param_disk *disk_car,t_list_part *list_part, char**current_cmd)
{
  t_list_part *parts;
  t_partition *old_part=NULL;
  struct MenuItem menuSuperblock[]=
  {
    { 'P', "Previous",""},
    { 'N', "Next","" },
    { 'Q',"Quit","Return to Advanced menu"},
    { 0, NULL, NULL }
  };
  aff_buffer(BUFFER_RESET,"Q");
  aff_copy(stdscr);
  wmove(stdscr,4,0);
  wdoprintf(stdscr,"%s",disk_car->description(disk_car));
  wmove(stdscr,5,0);
  mvwaddstr(stdscr,6,0,msg_PART_HEADER_LONG);
  for(parts=list_part;parts!=NULL;parts=parts->next)
  {
    if(old_part==NULL || (old_part->part_offset!=parts->part->part_offset) ||
      old_part->part_size!=parts->part->part_size ||
      old_part->part_type_i386!=parts->part->part_type_i386 ||
      old_part->part_type_sun!=parts->part->part_type_sun ||
      old_part->part_type_mac!=parts->part->part_type_mac ||
      old_part->upart_type!=parts->part->upart_type)
    {
      aff_part_buffer(AFF_PART_SHORT,disk_car,parts->part);
      old_part=parts->part;
    }
    aff_buffer(BUFFER_ADD,"superblock %u, blocksize=%u\n",parts->part->boot_sector,parts->part->blocksize);
  }
  if(*current_cmd!=NULL)
  {
    screen_buffer_to_log();
    return 0;
  }
  return screen_buffer_display(stdscr,"",menuSuperblock);
}
