/*

    File: intrf.c

    Copyright (C) 1998-2005 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
 
#include <stdarg.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include <ctype.h>
#ifdef HAVE_TIME_H
#include <time.h>
#endif
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#include "types.h"
#include "common.h"
#include "lang.h"
#include "intrf.h"
#include "fnctdsk.h"

#define GS_DEFAULT -1
#define GS_key_ESCAPE -2

int get_string(char *str, int len, char *def)
{
    int c;
    int i = 0;
    int x, y;
    int use_def = FALSE;
    curs_set(1);
    getyx(stdscr, y, x);
    wclrtoeol(stdscr);
    str[0] = 0;

    if (def != NULL) {
	mvwaddstr(stdscr,y, x, def);
	wmove(stdscr,y, x);
	use_def = TRUE;
    }

    wrefresh(stdscr);
    while ((c = wgetch(stdscr)) != '\n' && c != key_CR
#ifdef PADENTER
      && c!= PADENTER
#endif
      ) {
	switch (c) {
	  /* escape is generated by enter from keypad */
	  /*
	case key_ESC:
	    wmove(stdscr,y, x);
	    wclrtoeol(stdscr);
	    curs_set(0);
	    wrefresh(stdscr);
	    return GS_key_ESCAPE;
	    */
	case KEY_DC:
	case KEY_BACKSPACE:
	    if (i > 0) {
		str[--i] = 0;
		mvaddch(y, x+i, ' ');
		wmove(stdscr,y, x+i);
	    } else if (use_def) {
		wclrtoeol(stdscr);
		use_def = FALSE;
	    }
	    break;
	default:
	    if (i < len && isprint(c)) {
		mvaddch(y, x+i, c);
		if (use_def) {
		    wclrtoeol(stdscr);
		    use_def = FALSE;
		}
		str[i++] = c;
		str[i] = 0;
	    }
	}
	wrefresh(stdscr);
    }
    curs_set(0);
    wrefresh(stdscr);
    if (use_def)
	return GS_DEFAULT;
    else
	return i;
}

int wgetch_nodelay(WINDOW *window)
{
  int res;
  nodelay(window,TRUE);
  res=wgetch(window);
  nodelay(window,FALSE);
  return res;
}

/*
 * Actual function which prints the button bar and highlights the active button
 * Should not be called directly. Call function menuSelect instead.
 */

int wmenuUpdate(WINDOW *window, int y, int x, const struct MenuItem *menuItems, const unsigned int itemLength, const char *available, const int menuType, unsigned int current)
{
    unsigned int i, lmargin = x, ymargin = y;
    const char *mcd;

    /* Print available buttons */
    for( i = 0; menuItems[i].key!=0; i++ )
    {
      char buff[80];
      unsigned int lenName;
      const char *mi;
      wmove(window, y, x ); wclrtoeol(window);

	/* Search next available button */
	while( menuItems[i].key!=0 && strchr(available, menuItems[i].key)==NULL )
	{
	    i++;
	}
	if( menuItems[i].key==0 ) break; /* No more menu items */

	/* If selected item is not available and we have bypassed it,
	   make current item selected */
	if( current < i && menuItems[current].key < 0 ) current = i;

	/* If current item is selected, highlight it */
	if( current == i )
	{
	  /*attron( A_REVERSE )*/
	  wattrset(window,A_STANDOUT);
	}

	/* Print item */
	mi = menuItems[i].name;
	lenName = strlen( mi );
	if(lenName>=sizeof(buff))
	{
	  ecrit_rapport("\nBUG: %s\n",mi);
	}
	if(lenName > itemLength)
	{
	  if( menuType & MENU_BUTTON )
		snprintf(buff, sizeof(buff),"[%s]",mi);
	  else
		snprintf(buff, sizeof(buff),"%s",mi);
	}
	else
	{
	  if( menuType & MENU_BUTTON )
		snprintf( buff, sizeof(buff),"[%*s%-*s]", (itemLength - lenName) / 2, "",
			(itemLength - lenName + 1) / 2 + lenName, mi );
	  else
		snprintf( buff, sizeof(buff),"%*s%-*s", (itemLength - lenName) / 2, "",
			(itemLength - lenName + 1) / 2 + lenName, mi );
	}
	mvwaddstr(window, y, x, buff );

	/* Lowlight after selected item */
	if( current == i )
	{
	  /*attroff( A_REVERSE )*/
	  wattroff(window,A_STANDOUT);
	}
	/* Calculate position for the next item */
	if( menuType & MENU_VERT )
	{
	    y += 1;
	    if( y >= WARNING_START )
	    {
		y = ymargin;
		x += itemLength + MENU_SPACING;
		if( menuType & MENU_BUTTON ) x += 2;
	    }
	}
	else
	{
	    x += itemLength + MENU_SPACING;
	    if( menuType & MENU_BUTTON ) x += 2;
	    if( x > COLUMNS - lmargin - 12 )
	    {
		x = lmargin;
		y ++ ;
	    }
	}
    }
    /* Print the description of selected item */
    mcd = menuItems[current].desc;
    mvwaddstr(window, WARNING_START + 1, (COLUMNS - strlen( mcd )) / 2, mcd );
    return y;
}

/* This function takes a list of menu items, lets the user choose one *
 * and returns the value keyboard shortcut of the selected menu item  */

int wmenuSelect(WINDOW *window, int y, int x, const struct MenuItem *menuItems, const unsigned int itemLength, const char *available, int menuType, unsigned int menuDefault)
{
  unsigned int current=menuDefault;
  return wmenuSelect_ext(window, y, x, menuItems, itemLength, available, menuType, &current,NULL);
}

int wmenuSelect_ext(WINDOW *window, int y, int x, const struct MenuItem *menuItems, const unsigned int itemLength, const char *available, int menuType, unsigned int *current, int *real_key)
{
    int i, ylast = y, key = 0;
    /*
    if( ( menuType & ( MENU_HORIZ | MENU_VERT ) )==0 )    
    {
	wdoprintf(window,"Menu without direction. Defaulting horizontal.");
	menuType |= MENU_HORIZ;
    }
    */
    /* Warning: current may be out of bound, not checked */
    /* Make sure that the current is one of the available items */
    while(strchr(available, menuItems[*current].key)==NULL)
    {
	(*current)++ ;
	if( menuItems[*current].key==0 )
	{
	  *current = 0;
	}
    }
    /* Repeat until allowable choice has been made */
    while( key==0 )
    {
	/* Display the menu */
	ylast = wmenuUpdate( window, y, x, menuItems, itemLength, available,
			    menuType, *current );
	wrefresh(window);
	/* Don't put wgetch after the following wclrtoeol */
	key = wgetch(window);
	if(real_key!=NULL)
	  *real_key=key;

	/* Clear out all prompts and such */
	for( i = y; i < ylast; i ++ )
	{
	    wmove(window, i, x );
	    wclrtoeol(window);
	}
	wmove(window, WARNING_START + 1, 0 );
	wclrtoeol(window);
	/* Cursor keys */
	switch(key)
	{
	  case KEY_UP:
	    if( (menuType & MENU_VERT)!=0 )
	    {
	      do {
		if( (*current)-- == 0 )
		{
		  while( menuItems[(*current)+1].key ) (*current) ++ ;
		}
	      } while( strchr( available, menuItems[*current].key )==NULL );
	      key = 0;
	    }
	    break;
	  case KEY_DOWN:
	    if( (menuType & MENU_VERT)!=0 )
	    {
	      do {
		(*current) ++ ;
		if( menuItems[*current].key==0 ) *current = 0 ;
	      } while( strchr( available, menuItems[*current].key )==NULL );
	      key = 0;
	    }
	    break;
	  case KEY_RIGHT:
	    if( (menuType & MENU_HORIZ)!=0 )
	    {
	      do {
		(*current) ++ ;
		if( menuItems[*current].key==0 ) 
		{
		  *current = 0 ;
		}
	      } while( strchr( available, menuItems[*current].key )==NULL );
	      key = 0;
	    }
	    break;
	  case KEY_LEFT:
	    if( (menuType & MENU_HORIZ) !=0)
	    {
	      do {
		if( (*current)-- == 0 )
		{
		  while( menuItems[(*current) + 1].key ) (*current) ++ ;
		}
	      } while( strchr( available, menuItems[*current].key )==NULL );
	      key = 0;
	    }
	    break;
	}
	/* Enter equals to the keyboard shortcut of current menu item */
	if((key==13) || (key==10) || (key==KEY_ENTER) ||(((menuType & MENU_VERT) != 0) && (key==KEY_RIGHT || key==KEY_LEFT)))
	    key = menuItems[*current].key;
#ifdef PADENTER
	if(key==PADENTER)
	  key = menuItems[*current].key;
#endif

	/* Should all keys to be accepted? */
	if( key && (menuType & MENU_ACCEPT_OTHERS)!=0 ) break;
	/* Is pressed key among acceptable ones */
	if( key!=0 && (strchr(available, toupper(key))!=NULL || strchr(available, key)!=NULL))
	    break;
	/* The key has not been accepted so far -> let's reject it */
#ifdef DEBUG
	if( key )
	{
		wmove(window,5,0);
		wdoprintf(window,"key %03X",key);
	    putchar( BELL );
	}
#endif
	key = 0;
    }
    /* Clear out prompts and such */
    for( i = y; i <= ylast; i ++ )
    {
	wmove(window, i, x );
	wclrtoeol(window);
    }
    wmove(window, WARNING_START + 1, 0 );
    wclrtoeol(window);
    return key;
}

/* Function menuSelect takes way too many parameters  *
 * Luckily, most of time we can do with this function */

int wmenuSimple(WINDOW *window,const struct MenuItem *menuItems, int menuDefault)
{
    unsigned int i, j, itemLength = 0;
    char available[MENU_MAX_ITEMS];

    for(i = 0; menuItems[i].key; i++)
    {
	j = strlen(menuItems[i].name);
	if( j > itemLength ) itemLength = j;
	available[i] = menuItems[i].key;
    }
    available[i] = 0;
    return wmenuSelect(window,18, 0, menuItems, itemLength, available, MENU_HORIZ | MENU_BUTTON, menuDefault);
}

/* End of command menu support code */

unsigned long int ask_number(const unsigned long int val_cur, const unsigned long int val_min, const unsigned long int val_max, const char * _format, ...)
{
  char res[200];
  char res2[200];
  char response[LINE_LENGTH];
  char def[LINE_LENGTH];
  unsigned long int tmp_val;
  va_list ap;
  va_start(ap,_format);
  vsnprintf(res,sizeof(res),_format,ap);
  snprintf(res2,sizeof(res2),"(%lu-%lu) :",val_min,val_max);
  va_end(ap);

  mvwaddstr(stdscr,COMMAND_LINE_Y, COMMAND_LINE_X, res);
  waddstr(stdscr,res2);
  sprintf(def, "%lu", val_cur);
  if (get_string(response, LINE_LENGTH, def) > 0)
  {
    tmp_val = atol(response);
    if (tmp_val >= val_min && tmp_val <= val_max)
      return tmp_val;
  }
  return val_cur;
}

void dump_log(const void *nom_dump,unsigned int lng)
{
  unsigned int i,j;
  unsigned int nbr_line;
  unsigned char car;
  nbr_line=(lng+0x10-1)/0x10;
  /* write dump to log file*/
  for (i=0; (i<nbr_line); i++)
  {
    ecrit_rapport("%04X ",i*0x10);
    for(j=0; j< 0x10;j++)
    {
      if(i*0x10+j<lng)
      {
	car=*((const unsigned char *)nom_dump+i*0x10+j);
	ecrit_rapport("%02x", car);
      }
      else
	ecrit_rapport("  ");
      if(j%4==(4-1))
	ecrit_rapport(" ");
    }
    ecrit_rapport("  ");
    for(j=0; j< 0x10;j++)
    {
      if(i*0x10+j<lng)
      {
	car=*((const unsigned char*)nom_dump+i*0x10+j);
	if ((car<32)||(car >= 127))
	  ecrit_rapport(".");
	else
	  ecrit_rapport("%c",  car);
      }
      else
	ecrit_rapport("  ");
    }
    ecrit_rapport("\n");
  }
}

void dump(WINDOW *window, const void *nom_dump,unsigned int lng)
{
  unsigned int i,j;
  unsigned int nbr_line;
  unsigned char car;
  unsigned int pos=0;
  int done=0;
  int menu=2;   /* default : quit */
  const char *options="PNQ";
  struct MenuItem menuDump[]=
  {
    { 'P', "Previous",""},
    { 'N', "Next","" },
    { 'Q',"Quit","Quit dump section"},
    { 0, NULL, NULL }
  };
  dump_log(nom_dump,lng);
  nbr_line=(lng+0x10-1)/0x10;
  if(nbr_line<=DUMP_MAX_LINES)
  {
    options="Q";
  }
  /* ncurses interface */
  mvwaddstr(window,DUMP_Y,DUMP_X,msg_DUMP_HEXA);
  /* On pourrait utiliser wscrl */
  do
  {
    for (i=pos; (i<nbr_line)&&((i-pos)<DUMP_MAX_LINES); i++)
    {
      wmove(window,DUMP_Y+i-pos,DUMP_X);
      wclrtoeol(window);
      wdoprintf(window,"%04X ",i*0x10);
      for(j=0; j< 0x10;j++)
      {
	if(i*0x10+j<lng)
	{
	  car=*((const unsigned char*)nom_dump+i*0x10+j);
	  wdoprintf(window,"%02x", car);
	}
	else
	  wdoprintf(window,"  ");
	if(j%4==(4-1))
	  wdoprintf(window," ");
      }
      wdoprintf(window,"  ");
      for(j=0; j< 0x10;j++)
      {
	if(i*0x10+j<lng)
	{
	  car=*((const unsigned char*)nom_dump+i*0x10+j);
	  if ((car<32)||(car >= 127))
	    wdoprintf(window,".");
	  else
	    wdoprintf(window,"%c",  car);
	}
	else
	  wdoprintf(window," ");
      }
    }
    switch (wmenuSelect(window,INTER_DUMP_Y, INTER_DUMP_X, menuDump, 8, options, MENU_HORIZ | MENU_BUTTON | MENU_ACCEPT_OTHERS, menu))
    {
      case 'p':
      case 'P':
      case KEY_UP:
	if(strchr(options,'N')!=NULL)
	{
	  menu=0;
	  if(pos>0)
	    pos--;
	}
	break;
      case 'n':
      case 'N':
      case KEY_DOWN:
	if(strchr(options,'N')!=NULL)
	{
	  menu=1;
	  if(pos<nbr_line-DUMP_MAX_LINES)
	    pos++;
	}
	break;
      case KEY_PPAGE:
	if(strchr(options,'N')!=NULL)
	{
	  menu=0;
	  if(pos>DUMP_MAX_LINES-1)
	    pos-=DUMP_MAX_LINES-1;
	  else
	    pos=0;
	}
	break;
      case KEY_NPAGE:
	if(strchr(options,'N')!=NULL)
	{
	  menu=1;
	  if(pos<nbr_line-DUMP_MAX_LINES-(DUMP_MAX_LINES-1))
	    pos+=DUMP_MAX_LINES-1;
	  else
	    pos=nbr_line-DUMP_MAX_LINES;
	}
	break;
      case key_ESC:
      case 'q':
      case 'Q':
	done = TRUE;
	break;
    }
  } while(done==FALSE);
}

void dump2_log(const void *dump_1, const void *dump_2, const unsigned int lng)
{
  unsigned int i,j;
  unsigned int nbr_line;
  nbr_line=(lng+0x08-1)/0x08;
  /* write dump to log file*/
  for (i=0; (i<nbr_line); i++)
  {
    ecrit_rapport("%04X ",i*0x08);
    for(j=0; j<0x08;j++)
    {
      if(i*0x08+j<lng)
      {
	unsigned char car=*((const unsigned char *)dump_1+i*0x08+j);
	ecrit_rapport("%02x", car);
      }
      else
	ecrit_rapport("  ");
      if(j%4==(4-1))
	ecrit_rapport(" ");
    }
    ecrit_rapport("  ");
    for(j=0; j<0x08;j++)
    {
      if(i*0x08+j<lng)
      {
	unsigned char car=*((const unsigned char*)dump_1+i*0x08+j);
	if ((car<32)||(car >= 127))
	  ecrit_rapport(".");
	else
	  ecrit_rapport("%c",  car);
      }
      else
	ecrit_rapport(" ");
    }
    ecrit_rapport("  ");
    for(j=0; j<0x08;j++)
    {
      if(i*0x08+j<lng)
      {
	unsigned char car=*((const unsigned char *)dump_2+i*0x08+j);
	ecrit_rapport("%02x", car);
      }
      else
	ecrit_rapport("  ");
      if(j%4==(4-1))
	ecrit_rapport(" ");
    }
    ecrit_rapport("  ");
    for(j=0; j<0x08;j++)
    {
      if(i*0x08+j<lng)
      {
	unsigned char car=*((const unsigned char*)dump_2+i*0x08+j);
	if ((car<32)||(car >= 127))
	  ecrit_rapport(".");
	else
	  ecrit_rapport("%c",  car);
      }
      else
	ecrit_rapport(" ");
    }
    ecrit_rapport("\n");
  }
}

void dump2(WINDOW *window, const void *dump_1, const void *dump_2, const unsigned int lng)
{
  unsigned int i,j;
  unsigned int nbr_line;
  unsigned int pos=0;
  int done=0;
  int menu=2;   /* default : quit */
  const char *options="PNQ";
  struct MenuItem menuDump[]=
  {
    { 'P', "Previous",""},
    { 'N', "Next","" },
    { 'Q',"Quit","Quit dump section"},
    { 0, NULL, NULL }
  };
  dump2_log(dump_1, dump_2, lng);
  /* ncurses interface */
  nbr_line=(lng+0x08-1)/0x08;
  if(nbr_line<=DUMP_MAX_LINES)
  {
    options="Q";
  }
  do
  {
    for (i=pos; (i<nbr_line)&&((i-pos)<DUMP_MAX_LINES); i++)
    {
      wmove(window,DUMP_Y+i-pos,DUMP_X);
      wclrtoeol(window);
      wdoprintf(window,"%04X ",i*0x08);
      for(j=0; j<0x08;j++)
      {
	if(i*0x08+j<lng)
	{
	  unsigned char car1=*((const unsigned char*)dump_1+i*0x08+j);
	  unsigned char car2=*((const unsigned char*)dump_2+i*0x08+j);
	  if(car1!=car2)
	    wattrset(window,A_STANDOUT);
	  wdoprintf(window,"%02x", car1);
	  if(car1!=car2)
	    wattroff(window,A_STANDOUT);
	}
	else
	  wdoprintf(window," ");
	if(j%4==(4-1))
	  wdoprintf(window," ");
      }
      wdoprintf(window,"  ");
      for(j=0; j<0x08;j++)
      {
	if(i*0x08+j<lng)
	{
	  unsigned char car1=*((const unsigned char*)dump_1+i*0x08+j);
	  unsigned char car2=*((const unsigned char*)dump_2+i*0x08+j);
	  if(car1!=car2)
	    wattrset(window,A_STANDOUT);
	  if ((car1<32)||(car1 >= 127))
	    wdoprintf(window,".");
	  else
	    wdoprintf(window,"%c",  car1);
	  if(car1!=car2)
	    wattroff(window,A_STANDOUT);
	}
	else
	  wdoprintf(window," ");
      }
      wdoprintf(window,"  ");
      for(j=0; j<0x08;j++)
      {
	if(i*0x08+j<lng)
	{
	  unsigned char car1=*((const unsigned char*)dump_1+i*0x08+j);
	  unsigned char car2=*((const unsigned char*)dump_2+i*0x08+j);
	  if(car1!=car2)
	    wattrset(window,A_STANDOUT);
	  wdoprintf(window,"%02x", car2);
	  if(car1!=car2)
	    wattroff(window,A_STANDOUT);
	  if(j%4==(4-1))
	    wdoprintf(window," ");
	}
	else
	  wdoprintf(window," ");
      }
      wdoprintf(window,"  ");
      for(j=0; j<0x08;j++)
      {
	if(i*0x08+j<lng)
	{
	  unsigned char car1=*((const unsigned char*)dump_1+i*0x08+j);
	  unsigned char car2=*((const unsigned char*)dump_2+i*0x08+j);
	  if(car1!=car2)
	    wattrset(window,A_STANDOUT);
	  if ((car2<32)||(car2 >= 127))
	    wdoprintf(window,".");
	  else
	    wdoprintf(window,"%c",  car2);
	  if(car1!=car2)
	    wattroff(window,A_STANDOUT);
	}
	else
	  wdoprintf(window," ");
      }
    }
    switch (wmenuSelect(window,INTER_DUMP_Y, INTER_DUMP_X, menuDump, 8, options, MENU_HORIZ | MENU_BUTTON | MENU_ACCEPT_OTHERS, menu))
    {
      case 'p':
      case 'P':
      case KEY_UP:
	if(strchr(options,'N')!=NULL)
	{
	  menu=0;
	  if(pos>0)
	    pos--;
	}
	break;
      case 'n':
      case 'N':
      case KEY_DOWN:
	if(strchr(options,'N')!=NULL)
	{
	  menu=1;
	  if(pos<nbr_line-DUMP_MAX_LINES)
	    pos++;
	}
	break;
      case KEY_PPAGE:
	if(strchr(options,'N')!=NULL)
	{
	  menu=0;
	  if(pos>DUMP_MAX_LINES-1)
	    pos-=DUMP_MAX_LINES-1;
	  else
	    pos=0;
	}
	break;
      case KEY_NPAGE:
	if(strchr(options,'N')!=NULL)
	{
	  menu=1;
	  if(pos<nbr_line-DUMP_MAX_LINES-(DUMP_MAX_LINES-1))
	    pos+=DUMP_MAX_LINES-1;
	  else
	    pos=nbr_line-DUMP_MAX_LINES;
	}
	break;
      case key_ESC:
      case 'q':
      case 'Q':
	done = TRUE;
	break;
    }
  } while(done==FALSE);
}

static char intr_buffer_screen[MAX_LINES][LINE_LENGTH+1];
static int intr_nbr_line=0;

int aff_buffer(const buffer_cmd_t cmd, const char *_format, ...)
{
  switch(cmd)
  {
	case BUFFER_RESET:
	  {
		int i;
		intr_nbr_line=0;
		for(i=0;i<MAX_LINES;i++)
		  memset(intr_buffer_screen[i],0,LINE_LENGTH+1);
	  }
	  break;
	case BUFFER_ADD:
	  {
		char tmp_line[BUFFER_LINE_LENGTH+1];
		char *pos_in_tmp_line=tmp_line-1;
		va_list ap;
		va_start(ap,_format);
		memset(tmp_line,'\0',sizeof(tmp_line));
		vsnprintf(tmp_line,BUFFER_LINE_LENGTH,_format,ap);
		va_end(ap);
		while(pos_in_tmp_line!=NULL && (intr_nbr_line<MAX_LINES))
		{
		  char *ret_ligne= strchr(++pos_in_tmp_line,'\n');
		  if(ret_ligne!=NULL)
		  {
			strncat(intr_buffer_screen[intr_nbr_line],pos_in_tmp_line,
				ret_ligne-pos_in_tmp_line<LINE_LENGTH-strlen(intr_buffer_screen[intr_nbr_line])?ret_ligne-pos_in_tmp_line:LINE_LENGTH-strlen(intr_buffer_screen[intr_nbr_line]));
			if(strlen(intr_buffer_screen[intr_nbr_line])>0)
			  intr_nbr_line++;
		  }
		  else
		  {
			strncat(intr_buffer_screen[intr_nbr_line],pos_in_tmp_line,LINE_LENGTH-strlen(intr_buffer_screen[intr_nbr_line]));
		  }
		  pos_in_tmp_line=ret_ligne;
		}
/*	ecrit_rapport("aff_intr_buffer_screen %d =>%s<=\n",intr_nbr_line,tmp_line); */
		if(intr_nbr_line>=MAX_LINES)
		{
		  ecrit_rapport("aff_intr_buffer_screen too much lines =>%s<=\n",tmp_line);
		}
	  }
	  break;
	case BUFFER_SHOW:
	  {
	    int i;
	    int pos=intr_nbr_line-DUMP_MAX_LINES<0?0:intr_nbr_line-DUMP_MAX_LINES;
	    if(intr_buffer_screen[intr_nbr_line][0]!='\0')
	      intr_nbr_line++;
	    /* curses interface */
	    for (i=pos; (i<intr_nbr_line)&&((i-pos)<DUMP_MAX_LINES); i++)
	    {
	      wmove(stdscr,DUMP_Y+1+i-pos,DUMP_X);
	      wclrtoeol(stdscr);
	      wdoprintf(stdscr,"%s",intr_buffer_screen[i]);
	    }
	    wrefresh(stdscr);
	  }
	  break;
	case BUFFER_WRITE:
	  {
	    int i;
	    if(intr_buffer_screen[intr_nbr_line][0]!='\0')
	      intr_nbr_line++;
	    /* to log file and stdout */
	    for(i=0;i<intr_nbr_line;i++)
	    {
	      printf("%s\n",intr_buffer_screen[i]);
	      ecrit_rapport("%s\n",intr_buffer_screen[i]);
	    }
	  }
	  break;
  }
  return 0;
}

void screen_buffer_to_log()
{
  int i;
  if(intr_buffer_screen[intr_nbr_line][0]!='\0')
    intr_nbr_line++;
  /* to log file */
  for(i=0;i<intr_nbr_line;i++)
    ecrit_rapport("%s\n",intr_buffer_screen[i]);
}

int screen_buffer_display(WINDOW *window, const char *options_org, const struct MenuItem *menuItems)
{
  int menu=-1;
  return screen_buffer_display_ext(window,options_org,menuItems,&menu);
}

int screen_buffer_display_ext(WINDOW *window, const char *options_org, const struct MenuItem *menuItems, int *menu)
{
  int i;
  int pos=0;
  int done=0;
  char options[20];
  struct MenuItem menuDefault[]=
  {
    { 'P', "Previous",""},
    { 'N', "Next","" },
    { 'Q', "Quit","Quit this section"},
    { 0, NULL, NULL }
  };
  if(intr_nbr_line<=DUMP_MAX_LINES)
  {
    strncpy(options,"Q",sizeof(options));
    if(*menu<0)
      *menu=0;       /* default = quit */
  }
  else
  {
    strncpy(options,"PNQ",sizeof(options));
    if(*menu<0)
      *menu=1;       /* default = next */
  }
  strncat(options,options_org,sizeof(options)-strlen(options));
  if(intr_buffer_screen[intr_nbr_line][0]!='\0')
    intr_nbr_line++;
  /* to log file */
  for(i=0;i<intr_nbr_line;i++)
    ecrit_rapport("%s\n",intr_buffer_screen[i]);
  /* curses interface */
  do
  {
    int key;
    for (i=pos; (i<intr_nbr_line)&&((i-pos)<DUMP_MAX_LINES); i++)
    {
      wmove(window,DUMP_Y+i-pos,DUMP_X);
      wclrtoeol(window);
      wdoprintf(window,"%s",intr_buffer_screen[i]);
    }
    key=wmenuSelect_ext(window,INTER_DUMP_Y, INTER_DUMP_X, (menuItems!=NULL?menuItems:menuDefault), 8, options, MENU_HORIZ | MENU_BUTTON | MENU_ACCEPT_OTHERS, menu,NULL);
    switch (key)
    {
      case key_ESC:
      case 'q':
      case 'Q':
	done = TRUE;
	break;
      case 'p':
      case 'P':
      case KEY_UP:
	if(pos>0)
	  pos--;
	break;
      case 'n':
      case 'N':
      case KEY_DOWN:
	if(pos<intr_nbr_line-DUMP_MAX_LINES)
	  pos++;
	break;
      case KEY_PPAGE:
	if(pos>DUMP_MAX_LINES-1)
	  pos-=DUMP_MAX_LINES-1;
	else
	  pos=0;
	break;
      case KEY_NPAGE:
	if(pos<intr_nbr_line-DUMP_MAX_LINES-(DUMP_MAX_LINES)-1)
	  pos+=DUMP_MAX_LINES-1;
	else
	  pos=intr_nbr_line-DUMP_MAX_LINES>0?intr_nbr_line-DUMP_MAX_LINES:0;
	break;
      default:
	if(strchr(options,toupper(key))!=NULL)
	  return toupper(key);
	break;
    }
  } while(done!=TRUE);
  return 0;
}

void aff_CHS_rapport(const t_CHS * CHS)
{
  ecrit_rapport("%5u %3u %2u ", CHS->cylinder, CHS->head, CHS->sector);
}

void aff_CHS(const t_CHS * CHS)
{
  wdoprintf(stdscr,"%5u %3u %2u ", CHS->cylinder, CHS->head, CHS->sector);
}

void aff_CHS_buffer(const t_CHS * CHS)
{
  aff_buffer(BUFFER_ADD,"%5u %3u %2u ", CHS->cylinder, CHS->head, CHS->sector);
}

const char *aff_part_aux(const aff_part_type_t newline, const t_param_disk *disk_car, const t_partition *partition)
{
  char status=' ';
  static char msg[200];
  unsigned int pos=0;
  const t_arch_fnct *arch=(partition->arch!=NULL?partition->arch:disk_car->arch);
  msg[sizeof(msg)-1]=0;
  switch(newline)
  {
    case AFF_PART_ORDER:
      if((partition->status!=STATUS_EXT_IN_EXT) && (partition->order!=NO_ORDER))
	pos+=snprintf(&msg[pos],sizeof(msg)-pos-1,"%2d ", partition->order);
      else
	pos+=snprintf(&msg[pos],sizeof(msg)-pos-1,"   ");
      break;
    case AFF_PART_NONL:
    case AFF_PART_SHORT:
      break;
  }
  if(newline!=AFF_PART_SHORT)
  {
    switch(partition->status)
    {
      case STATUS_PRIM:           status='P'; break;
      case STATUS_PRIM_BOOT:      status='*'; break;
      case STATUS_EXT:            status='E'; break;
      case STATUS_EXT_IN_EXT:     status='X'; break;
      case STATUS_LOG:            status='L'; break;
      case STATUS_DELETED:        status='D'; break;
      default:			  status=' '; break;
    }
  }
  pos+=snprintf(&msg[pos],sizeof(msg)-pos-1,"%c", status);
  if(arch->get_partition_name(partition)!=NULL)
    pos+=snprintf(&msg[pos],sizeof(msg)-pos-1, " %-20s ",
	arch->get_partition_name(partition));
  else
    pos+=snprintf(&msg[pos],sizeof(msg)-pos-1, " Sys=%02X               ", arch->get_part_type(partition));
  if(disk_car->CHS.head==0 && disk_car->CHS.sector==1)
  {
    pos+=snprintf(&msg[pos],sizeof(msg)-pos-1, " %10lu ", (long unsigned)(partition->part_offset/disk_car->sector_size));
    pos+=snprintf(&msg[pos],sizeof(msg)-pos-1, "%10lu ", (long unsigned)((partition->part_offset+partition->part_size-1)/disk_car->sector_size));
  }
  else
  {
    pos+=snprintf(&msg[pos],sizeof(msg)-pos-1,"%5u %3u %2u ",
	offset2cylinder(disk_car,partition->part_offset),
	offset2head(    disk_car,partition->part_offset),
	offset2sector(  disk_car,partition->part_offset));
    pos+=snprintf(&msg[pos],sizeof(msg)-pos-1, "%5u %3u %2u ",
	offset2cylinder(disk_car,partition->part_offset+partition->part_size-1),
	offset2head(    disk_car,partition->part_offset+partition->part_size-1),
	offset2sector(  disk_car,partition->part_offset+partition->part_size-1));
  }
  pos+=snprintf(&msg[pos],sizeof(msg)-pos-1,"%10lu", (long unsigned)(partition->part_size/disk_car->sector_size));
  if(partition->name[0]!='\0')
    pos+=snprintf(&msg[pos],sizeof(msg)-pos-1, " [%s]",partition->name);
  return msg;
}

void aff_part(WINDOW *window,const aff_part_type_t newline,const t_param_disk *disk_car,const t_partition *partition)
{
  const char *msg;
  msg=aff_part_aux(newline, disk_car, partition);
  wdoprintf(window,"%s",msg);
}

void aff_part_rapport(const t_param_disk *disk_car,const t_partition *partition)
{
  const char *msg;
  msg=aff_part_aux(AFF_PART_ORDER, disk_car, partition);
  ecrit_rapport("%s",msg);
  if(partition->info[0]!='\0')
    ecrit_rapport("\n     %s, %lu MB",partition->info,(long unsigned)(partition->part_size/(1024*1024)));
  ecrit_rapport("\n");
}

void aff_part_buffer(const aff_part_type_t newline,const t_param_disk *disk_car,const t_partition *partition)
{
  const char *msg;
  msg=aff_part_aux(newline, disk_car, partition);
  aff_buffer(BUFFER_ADD,"%s\n", msg);
}

void aff_LBA2CHS(const t_param_disk *disk_car, const unsigned long int pos_LBA)
{
  unsigned long int tmp;
  unsigned long int cylinder, head, sector;
  tmp=disk_car->CHS.sector;
  sector=(pos_LBA%tmp)+1;
  tmp=pos_LBA/tmp;
  cylinder=tmp/(disk_car->CHS.head+1);
  head=tmp%(disk_car->CHS.head+1);
  wdoprintf(stdscr,"%lu/%lu/%lu", cylinder,head,sector);
}

void aff_LBA2CHS_rapport(const t_param_disk *disk_car, const unsigned long int pos_LBA)
{
  unsigned long int tmp;
  unsigned long int cylinder, head, sector;
  tmp=disk_car->CHS.sector;
  sector=(pos_LBA%tmp)+1;
  tmp=pos_LBA/tmp;
  cylinder=tmp/(disk_car->CHS.head+1);
  head=tmp%(disk_car->CHS.head+1);
  ecrit_rapport("%lu/%lu/%lu", cylinder,head,sector);
}

int ask_YN(WINDOW *window)
{
  char res;
  curs_set(1);
  wrefresh(window);
  do
  {
    res=toupper(wgetch(window));
  } while((res!=c_NO)&&(res!=c_YES));
  curs_set(0);
  wdoprintf(window,"%c\n",res);
  return (res==c_YES);
}

int ask_confirmation(const char*_format, ...)
{
  char buffer[800];
  va_list ap;
  int res;
  WINDOW *window=newwin(0,0,0,0);	/* full screen */
  aff_copy(window);
  va_start(ap,_format);
  vsnprintf(buffer,sizeof(buffer),_format,ap);
  va_end(ap);
  wmove(window,4,0);
  waddstr(window,buffer);
  res=ask_YN(window);
  delwin(window);
  (void) clearok(stdscr, TRUE);
#ifdef HAVE_TOUCHWIN
  touchwin(stdscr);
#endif
  return res;
}

int wdoprintf(WINDOW *window, const char *_format, ...)
{
  char res[800];
  va_list ap;
  va_start(ap,_format);
  vsnprintf(res,sizeof(res),_format,ap);
  va_end(ap);
  res[sizeof(res)-1]='\0';
#ifdef __MINGW32__
  {
    int len=strlen(res);
    if(res[len-1]=='\n' && len<sizeof(res)-1)
    {
      res[len]='\r';
      res[len+1]=0;
    }
  }
#endif
  return waddstr(window,res);
}

/*
int display_message_ext(const char *_format, ...)
{
  char res[800];
  va_list ap;
  va_start(ap,_format);
  vsnprintf(res,sizeof(res),_format,ap);
  va_end(ap);
  res[sizeof(res)-1]='\0';
#ifdef __MINGW32__
  {
    int len=strlen(res);
    if(res[len-1]=='\n' && len<sizeof(res)-1)
    {
      res[len]='\r';
      res[len+1]=0;
    }
  }
#endif
  return display_message(res);
}
*/

int display_message(const char*msg)
{
  int pipo=0;
  static struct MenuItem menuGeometry[]=
  {
    { 'O', "Ok", "" },
    { 0, NULL, NULL }
  };
  WINDOW *window=newwin(0,0,0,0);	/* full screen */
  aff_copy(window);
  mvwaddstr(window,5,0,msg);
  wmenuSimple(window,menuGeometry, pipo);
  delwin(window);
  (void) clearok(stdscr, TRUE);
#ifdef HAVE_TOUCHWIN
  touchwin(stdscr);
#endif
  ecrit_rapport("%s",msg);
  return 0;
}

void not_implemented(const char *msg)
{
  WINDOW *window=newwin(0,0,0,0);	/* full screen */
  aff_copy(window);
  wmove(window,7,0);
  wdoprintf(window,"Function %s not implemented",msg);
  ecrit_rapport("Function %s not implemented\n",msg);
  wmove(window,22,0);
  wattrset(window,A_STANDOUT);
  wdoprintf(window,"[  Abort ]");
  wattroff(window,A_STANDOUT);
  wrefresh(window);
  wgetch(window);
  delwin(window);
  (void) clearok(stdscr, TRUE);
#ifdef HAVE_TOUCHWIN
  touchwin(stdscr);
#endif
}

int start_ncurses(const char *prog_name)
{
/* use_env(TRUE); */
#ifdef HAVE_SETENV
#ifdef DJGPP
#elif defined(TARGET_BSD)
    setenv("TERM","cons25",0);
#elif defined(TARGET_LINUX)
    setenv("TERM","linux",0);
#elif defined(__CYGWIN__)
    setenv("TERM","cygwin",0);
#endif
#endif
#if defined(DJGPP) || defined(__MINGW32__)
  if(initscr()==NULL)
  {
    printf("initscr() has failed. Exiting\n");
  }
#else
  if(newterm(NULL,stdout,stdin)==NULL)
  {
#ifdef HAVE_SETENV
    setenv("TERMINFO",".",1);
    if(newterm(NULL,stdout,stdin)==NULL)
#endif
    {
      ecrit_rapport("Terminfo file is missing.\n");
      printf("Terminfo file is missing.\nCheck if you have not forgot to extract the subdirectories.\n");
      return 1;
    }
  }
#endif
  if(LINES<25)
  {
    ecrit_rapport("Terminal has only %u lines\n",LINES);
    endwin();
    printf("%s need 25 lines to work.\nPlease enlarge the terminal and restart %s.\n",prog_name,prog_name);
    return 1;
  }
  noecho();
#ifndef DJGPP
  nonl(); /*don't use for Dos version but enter will work with it... dilema */
#endif
  /*  intrflush(stdscr, FALSE); */
  cbreak();
  /* Should solve a problem with users who redefined the colors */
  if(has_colors())
  {
    start_color();
    init_pair(1, COLOR_RED, COLOR_BLACK);
    init_pair(2, COLOR_GREEN, COLOR_BLACK);
  }
  curs_set(0);
  return 0;
}

int end_ncurses()
{
  wclear(stdscr);
  wrefresh(stdscr);
  nl();
  endwin();
  return 0;
}

int check_enter_or_s(WINDOW *window)
{
  switch(wgetch_nodelay(window))
  {
    case KEY_ENTER:
#ifdef PADENTER
    case PADENTER:
#endif
    case '\n':
    case '\r':
    case 's':
    case 'S':
      return 1;
      break;
  }
  return 0;
}
