/*

    File: godmode.c

    Copyright (C) 1998-2005 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
 
#include <ctype.h>      /* tolower */
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include "types.h"
#include "common.h"
#include "fnctdsk.h"
#include "analyse.h"
#include "lang.h"
#include "godmode.h"
#include "testdisk.h"
#include "fat.h"
#include "ext2.h"
#include "intrf.h"
#include "intrface.h"
#include "md.h"
#include "adv.h"
#define RO 1
#define RW 0
#define MAX_SEARCH_LOCATION 1024
extern const t_arch_fnct arch_mac;
extern const t_arch_fnct arch_none;
static void align_structure(const t_param_disk *disk_car, t_list_part *list_part,const int align);
static void ask_mbr_order(t_param_disk *disk_car,t_list_part *list_part);
static t_list_part *reduce_structure(t_list_part *list_part);
static t_list_part *add_ext_part_i386(t_param_disk *disk_car, t_list_part *list_part, const int max_ext, const int align,const int debug);
static int use_backup(t_param_disk *disk_car, const t_list_part *list_part, const int debug,const int dump_ind, const unsigned int expert, char**current_cmd);
static int interface_part_bad(t_param_disk *disk_car,t_list_part *list_part_bad);
static void warning_geometry(t_param_disk *disk_car, const unsigned int recommanded_heads_per_cylinder, char **current_cmd);
static t_list_part *search_part_raid(t_param_disk *disk_car, t_list_part *list_part, t_list_part *list_part_bad, const int debug, const int dump_ind, const int interface);

static void align_structure(const t_param_disk *disk_car, t_list_part *list_part, const int align)
{
  t_list_part *element;
  if(align==0)
    return ;
  for(element=list_part;element!=NULL;element=element->next)
  {
    t_CHS end;
    offset2CHS(disk_car,element->part->part_offset+element->part->part_size-1,&end);
    if(align>0)
    {
      end.sector=disk_car->CHS.sector;
    }
    if(align>1)
    {
      end.head=disk_car->CHS.head;
    }
    element->part->part_size=(uint64_t)CHS2offset(disk_car, &end)-element->part->part_offset+disk_car->sector_size;
  }
}


void only_one_bootable( t_list_part *list_part, t_list_part *part_boot)
{
  t_list_part *element;
  if(part_boot->part->status==STATUS_PRIM_BOOT)
    for(element=list_part;element!=NULL;element=element->next)
    {
      if((element!=part_boot)&&(element->part->status==STATUS_PRIM_BOOT))
	element->part->status=STATUS_PRIM;
    }
}

static int interface_part_bad(t_param_disk *disk_car,t_list_part *list_part)
{
  int quit=0;
  int offset=0;
  int pos_num=0;
  t_list_part *pos=list_part;
  if(list_part==NULL)
    return 1;
  aff_copy(stdscr);
  wmove(stdscr,4,0);
  wdoprintf(stdscr,"%s",disk_car->description(disk_car));
  wmove(stdscr,6,0);
  wdoprintf(stdscr,"Check the harddisk size: HD jumpers settings, BIOS detection, OS limitation...");
  wmove(stdscr,7,0);
  if(list_part->next==NULL)
  {
    wdoprintf(stdscr,"The following partition can't be recovered. The harddisk seems too small!");
  } else {
    wdoprintf(stdscr,"The following partitions can't be recovered. The harddisk seems too small!");
  }
  mvwaddstr(stdscr,8,0,msg_PART_HEADER);
  wmove(stdscr,22,0);
  wattrset(stdscr,A_STANDOUT);
  wdoprintf(stdscr,"[  Quit  ]");
  wattroff(stdscr,A_STANDOUT);
  do
  {
    int i;
    int car;
    t_list_part *parts;
    for(i=0,parts=list_part;(parts!=NULL) && (i<offset);parts=parts->next,i++);
    for(i=offset;(parts!=NULL) &&((i-offset)<INTER_BAD_PART);i++,parts=parts->next)
    {
      wmove(stdscr,9+i-offset,0);
      wclrtoeol(stdscr);	/* before addstr for BSD compatibility */
      if(parts==pos)
      {
	wattrset(stdscr,A_STANDOUT);
	aff_part(stdscr,AFF_PART_NONL,disk_car,parts->part);
	wattroff(stdscr,A_STANDOUT);
	wmove(stdscr,24,0);
	wclrtoeol(stdscr);	/* before addstr for BSD compatibility */
	if(parts->part->info[0]!='\0')
	{
	  wdoprintf(stdscr,"%s, ",parts->part->info);
	}
	wdoprintf(stdscr,"%lu MB",(long unsigned)(parts->part->part_size/(1024*1024)));
      } else
      {
	aff_part(stdscr,AFF_PART_NONL,disk_car,parts->part);
      }
    }
    wrefresh(stdscr);
    car=wgetch(stdscr);
    switch(car)
    {
      case 'q':
      case '\r':
      case '\n':
      case KEY_ENTER:
#ifdef PADENTER
      case PADENTER:
#endif
      case 'M':
	quit=1;
	break;
      case KEY_UP:
	if(list_part!=NULL)
	{
	  only_one_bootable(list_part,pos);
	  if(pos->prev!=NULL)
	  {
	    pos=pos->prev;
	    pos_num--;
	  }
	  if(pos_num<offset)
	    offset--;
	}
	break;
      case KEY_DOWN:
	if(list_part!=NULL)
	{
	  only_one_bootable(list_part,pos);
	  if(pos->next!=NULL)
	  {
	    pos=pos->next;
	    pos_num++;
	  }
	  if(pos_num>=offset+INTER_BAD_PART)
	    offset++;
	}
	break;
      case KEY_PPAGE:
	if(list_part!=NULL)
	{
	  only_one_bootable(list_part,pos);
	  for(i=0;(i<INTER_BAD_PART) && (pos->prev!=NULL);i++)
	  {
	    pos=pos->prev;
	    pos_num--;
	    if(pos_num<offset)
	      offset--;
	  }
	}
	break;
      case KEY_NPAGE:
	if(list_part!=NULL)
	{
	  only_one_bootable(list_part,pos);
	  for(i=0;(i<INTER_BAD_PART) && (pos->next!=NULL);i++)
	  {
	    pos=pos->next;
	    pos_num++;
	    if(pos_num>=offset+INTER_BAD_PART)
	      offset++;
	  }
	}
	break;
      default:
	break;
    }
  } while(quit==0);
  return 0;
}

static void warning_geometry(t_param_disk *disk_car, const unsigned int recommanded_heads_per_cylinder, char**current_cmd)
{
  int quit=0;
  aff_copy(stdscr);
  wmove(stdscr,4,0);
  wdoprintf(stdscr,"%s",disk_car->description(disk_car));
  wmove(stdscr,6,0);
  ecrit_rapport("Warning: the current number of heads per cylinder is %u but the correct value may be %u.\n",disk_car->CHS.head+1,recommanded_heads_per_cylinder);
  wdoprintf(stdscr,"Warning: the current number of heads per cylinder is %u",disk_car->CHS.head+1);
  wmove(stdscr,7,0);
  wdoprintf(stdscr,"but the correct value may be %u.",recommanded_heads_per_cylinder);
  wmove(stdscr,8,0);
  wdoprintf(stdscr,"Use the Geometry menu to change this value.");
  wmove(stdscr,22,0);
  wattrset(stdscr,A_STANDOUT);
  wdoprintf(stdscr,"[  Quit  ]");
  wattroff(stdscr,A_STANDOUT);
  wrefresh(stdscr);
  if(*current_cmd!=NULL)
    return;
  do
  {
    int car;
    car=wgetch(stdscr);
    switch(car)
    {
      case 'q':
      case '\r':
      case '\n':
      case KEY_ENTER:
#ifdef PADENTER
      case PADENTER:
#endif
      case 'M':
	quit=1;
	break;
      default:
	break;
    }
  } while(quit==0);
}

static t_list_part *search_part_raid(t_param_disk *disk_car, t_list_part *list_part, t_list_part *list_part_bad, const int debug, const int dump_ind, const int interface)
{
  t_list_part *element;
  unsigned int test_current=0;
  unsigned int test_total=0;
  int ind_stop=0;
  for(element=list_part_bad;element!=NULL;element=element->next)
  {
    if(element->part->upart_type!=UP_RAID)
      test_total+=6;
  }
  if(interface)
  {
    aff_copy(stdscr);
    wmove(stdscr,4,0);
    wdoprintf(stdscr,"%s",disk_car->description(disk_car));
    wmove(stdscr,22,0);
    wattrset(stdscr,A_STANDOUT);
    waddstr(stdscr,"  Stop  ");
    wattroff(stdscr,A_STANDOUT);
  }
  /* Little trick to detect Linux software Raid quickly */
  for(element=list_part_bad;element!=NULL && ind_stop==0;element=element->next)
  {
    if(element->part->upart_type!=UP_RAID)
    {
      unsigned char buffer[8*DEFAULT_SECTOR_SIZE];
      struct mdp_superblock_s *sb=(struct mdp_superblock_s*)&buffer;
      unsigned int disk_factor;
      unsigned int help_factor;
      if(debug>1)
      {
	ecrit_rapport("search_part_raid:\n");
	aff_part_rapport(disk_car,element->part);
      }
      for(disk_factor=6;disk_factor>=1 && ind_stop==0;disk_factor--)
      { /* disk_factor=1, detect Raid 0/1 */
        /* disk_factor>1, detect Raid 5 */
	if(interface)
	{
	  wmove(stdscr,6,0);
	  wdoprintf(stdscr,"Searching for software Raid partition: %02d%%",test_current*100/test_total);
	  wrefresh(stdscr);
	  ind_stop|=check_enter_or_s(stdscr);
	}
	test_current++;
	for(help_factor=0;help_factor<=MD_MAX_CHUNK_SIZE/MD_RESERVED_BYTES+3 && ind_stop==0;help_factor++)
	{
	  uint64_t offset=(uint64_t)MD_NEW_SIZE_SECTORS((element->part->part_size/disk_factor+help_factor*MD_RESERVED_BYTES-1)/MD_RESERVED_BYTES*MD_RESERVED_BYTES/512)*512;
	  if(debug>1)
	  {
	    ecrit_rapport("search_part_raid offset=%llu\n",(long long unsigned)offset);
	  }
	  if(disk_car->read(disk_car,sizeof(buffer), &buffer, element->part->part_offset+offset)==0)
	  {
	    t_partition *new_partition=partition_new();
	    dup_t_partition(new_partition,element->part);
	    if(test_MD(disk_car,sb,new_partition,debug,dump_ind)==0)
	    {
	      set_MD_info(disk_car,sb,new_partition,debug,dump_ind);
	      new_partition->part_type_i386=P_RAID;
	      new_partition->part_type_sun=PSUN_RAID;
	      new_partition->part_type_mac=PMAC_LINUX;
	      new_partition->part_size=offset+MD_RESERVED_BYTES;
	      aff_part_rapport(disk_car,new_partition);
	      list_part=insert_new_partition(list_part,new_partition);
	    }
	    else
	      FREE(new_partition);
	  }
	}
      }
    }
  }
  return list_part;
}

t_list_part *search_part(t_param_disk *disk_car, const int paranoid, const int debug, const int dump_ind, const int fast_mode, const int interface)
{
  int ind_stop=0;
  uint64_t try_offset[MAX_SEARCH_LOCATION];
  unsigned int try_offset_nbr=0;
  unsigned int old_permil=0;
  t_CHS start;
  t_list_part *list_part=NULL;
  t_list_part *list_part_bad=NULL;
  t_partition *partition=partition_new();
  if(interface!=0)
  {
    wmove(stdscr,22,0);
    wattrset(stdscr,A_STANDOUT);
    waddstr(stdscr,"  Stop  ");
    wattroff(stdscr,A_STANDOUT);
  }
  aff_buffer(BUFFER_RESET,"Q");
  ecrit_rapport("\nsearch_part()\n");
  ecrit_rapport("%s\n",disk_car->description(disk_car));
  start.head=(disk_car->arch==&arch_none?0:1);
  for(start.cylinder=0;(start.cylinder<=disk_car->CHS.cylinder)&&(ind_stop==0);start.cylinder++,start.head=0)
  {
    if(interface!=0)
    {
      unsigned int permil=start.cylinder*1000/(disk_car->CHS.cylinder>0?disk_car->CHS.cylinder:1);
      if(permil>old_permil || disk_car->arch!=&arch_mac)
      {
	wmove(stdscr,ANALYSE_Y,ANALYSE_X);
	wclrtoeol(stdscr);
	wdoprintf(stdscr,"Analyse cylinder %5u/%u: %02u%%",start.cylinder,disk_car->CHS.cylinder,start.cylinder*100/(disk_car->CHS.cylinder>0?disk_car->CHS.cylinder:1));
	wrefresh(stdscr);
	old_permil=permil;
      }
    }
    for(;(start.head<=disk_car->CHS.head)&&(ind_stop==0);start.head++)
    {
      unsigned int sector_inc;
      int part_offset;
      if(interface!=0)
      {
	ind_stop|=check_enter_or_s(stdscr);
      }
      for(start.sector=1,sector_inc=1,part_offset=0;(start.sector<=disk_car->CHS.sector);start.sector+=sector_inc)
      {
	int res=0;
	unsigned int i;
	int search_now=0;
	partition_reset(partition);	/* must not reset lba */
	partition->part_offset=CHS2offset(disk_car,&start);
	for(i=0;i<try_offset_nbr;i++)
	{
	  if(try_offset[i]==partition->part_offset)
	    search_now=1;
	}
	if((start.sector==1) && ((start.head<=2)||(fast_mode==2)||(search_now>0)))
	{
	  part_offset++;
	  switch(part_offset)
	  {
	    case 1:
	      sector_inc=0;
	      res=search_type_64(disk_car,partition,debug,dump_ind);
	      break;
	    case 2:
	      res=search_type_2(disk_car,partition,debug,dump_ind);
	      break;
	    case 3:
	      res=search_type_1(disk_car,partition,debug,dump_ind);
	      break;
	    case 4:
	      /* Try to catch disklabel before BSD FFS partition */
	      res=search_type_128(disk_car,partition,debug,dump_ind);
	      break;
	    case 5:
	      /* Try to catch disklabel before BSD FFS partition */
	      res=search_type_16(disk_car,partition,debug,dump_ind);
	      break;
	    case 6:
	      res=search_type_0(disk_car,partition,debug,dump_ind);
	      sector_inc=1;
	      break;
	    default:
	      ecrit_rapport("\nBUG: unusual value of part_offset\n");
	      break;
	  }
	}
	else if(start.sector==7 && fast_mode>0 && ((start.head<=2)||(fast_mode==2)||(search_now>0)))
	{
	  res=search_FAT_backup(disk_car,partition,debug,dump_ind);
	}
	else if(start.sector==disk_car->CHS.sector && (fast_mode>0))
	{
	  if((start.head==disk_car->CHS.head) || (fast_mode>1) || (search_now>0))
	  {
	    res=search_NTFS_backup(disk_car,partition,debug,dump_ind);
	  }
	}
	else if(start.sector==disk_car->CHS.sector-1 && (fast_mode>0))
	{
	  if((start.head==disk_car->CHS.head) || (fast_mode>1) || (search_now>0))
	  {
	    res=search_HFS_backup(disk_car,partition,debug,dump_ind);
	  }
	}
	if(res<0)
	{
	  if(interface!=0)
	  {
	    wdoprintf(stdscr,msg_READ_ERROR_AT, start.cylinder,start.head,start.sector,(unsigned long)(partition->part_offset/disk_car->sector_size));
	  }
	}
	if(res<=0 && paranoid!=0)
	{
	  int s_log_block_size;
	  uint64_t ref_offset=partition->part_offset;
	  /* try backup superblock */
	  /* It must be in paranoid mode because it can hide otherwise other partition type */
	  /* Block size: 1024, 2048 or 4096 bytes (8192 bytes on Alpha systems) */
	  /* From e2fsprogs-1.34/lib/ext2fs/initialize.c: set_field(s_first_data_block, super->s_log_block_size ? 0 : 1); */
	  /* Assumes that TestDisk is not running under Alpha and s_blocks_per_group=8 * block size */
	  for(s_log_block_size=0;(s_log_block_size<=2)&&(res<1);s_log_block_size++)
	  {
	    /* sparse superblock feature: The groups chosen are 0, 1 and powers of 3, 5 and 7. */
	    /* Checking group 3 */
	    const uint64_t hd_offset=3*(EXT2_MIN_BLOCK_SIZE<<s_log_block_size)*8*(EXT2_MIN_BLOCK_SIZE<<s_log_block_size)+(s_log_block_size==0?2*DEFAULT_SECTOR_SIZE:0);
	    if(ref_offset>=hd_offset &&
	     offset2sector(disk_car,ref_offset-hd_offset)==1 &&  
	     (offset2head(disk_car,ref_offset-hd_offset)<=2 || fast_mode==2 || (search_now>0 && fast_mode>0)))
	    {
	      unsigned char buffer[2*DEFAULT_SECTOR_SIZE];
	      if(disk_car->read(disk_car,sizeof(buffer), &buffer, ref_offset)==0)
	      {
		const struct ext2_super_block *sb=(const struct ext2_super_block*)&buffer;
		if(le16(sb->s_block_group_nr)>0)
		{
		  if(recover_EXT2(disk_car,sb,partition,debug,dump_ind)==0)
		    res=1;
		}
	      }
	      else
	      {
		res=-1;
	      }
	    }
	  }
	}
	if(res<=0 && fast_mode>1)
	{
	  /* Because this superblock is at the end of the partition, you should be
	   * in slow mode to find it... */
	  unsigned char buffer[8*DEFAULT_SECTOR_SIZE];
	  if(disk_car->read(disk_car,sizeof(buffer), &buffer, partition->part_offset)!=0)
	  {
	    res=-1;
	  }
	  else
	  {
	    if(recover_MD(disk_car,(const struct mdp_superblock_s*)&buffer,partition,debug,dump_ind)==0)
	      res=1;
	    else
	      res=0;
	  }
	}
	if(res>0)
	{
	  partition->status=STATUS_DELETED;
	  aff_part_rapport(disk_car,partition);
	  aff_part_buffer(AFF_PART_ORDER,disk_car,partition);
	  if(interface)
	  {
	    aff_buffer(BUFFER_SHOW,"Q");
	  }
	  if(disk_car->arch->is_part_known(partition)!=0 && partition->part_size>1 && (partition->part_offset>0 || disk_car->arch==&arch_none))
	  {
	    uint64_t pos_fin;
	    pos_fin=partition->part_offset+partition->part_size-1;
	    if(pos_fin<=disk_car->disk_size)
	    {
	      {
		t_partition *new_partition=partition_new();
		dup_t_partition(new_partition,partition);
		list_part=insert_new_partition(list_part,new_partition);
		/* Never use new_partition after insert_new_partition, it may have been freed */
	      }
	      {
		uint64_t next_part_offset=partition->part_offset+partition->part_size-1+1;
		uint64_t head_size=disk_car->CHS.sector*disk_car->sector_size;
		if(try_offset_nbr<MAX_SEARCH_LOCATION-1)
		{
		  try_offset[try_offset_nbr++]=next_part_offset;
		}
		if(try_offset_nbr<MAX_SEARCH_LOCATION-1)
		{
		  try_offset[try_offset_nbr++]=next_part_offset+head_size;
		}
		if(next_part_offset%head_size!=0)
		{
		  if(try_offset_nbr<MAX_SEARCH_LOCATION-1)
		  {
		    try_offset[try_offset_nbr++]=(next_part_offset+head_size-1)/head_size*head_size;
		  }
		  if(try_offset_nbr<MAX_SEARCH_LOCATION-1)
		  {
		    try_offset[try_offset_nbr++]=(next_part_offset+head_size-1)/head_size*head_size+head_size;
		  }
		}
	      }
	      if((paranoid==0) && (partition->part_offset+partition->part_size-disk_car->sector_size> CHS2offset(disk_car,&start)))
	      {
		offset2CHS(disk_car,partition->part_offset+partition->part_size-1,&start);
		part_offset=0;
		sector_inc=1;
	      }
	    }
	    else
	    {
	      {
		t_partition *new_partition=partition_new();
		dup_t_partition(new_partition,partition);
		list_part_bad=insert_new_partition(list_part_bad,new_partition);
		/* Never use new_partition after insert_new_partition, it may have been freed */
	      }
	      ecrit_rapport("This partition ends after the disk limits. ");
	      if(debug>0)
	      {
		ecrit_rapport("(start=%lu, size=%lu, end=%lu, disk end=%lu)\n",
		    (unsigned long)(partition->part_offset/disk_car->sector_size),
		    (unsigned long)(partition->part_size/disk_car->sector_size),
		    (unsigned long)(pos_fin/disk_car->sector_size),
		    (unsigned long)(disk_car->disk_size/disk_car->sector_size));
	      }
	    }
	  }
	  else
	  {
	    if(debug>0)
	    {
	      ecrit_rapport("Partition not added.\n");
	    }
	  }
	}
      }
    }
  }
  FREE(partition);
  list_part=search_part_raid(disk_car,list_part,list_part_bad,debug,dump_ind,interface);
  if(interface!=0 && list_part_bad!=NULL)
  {
    interface_part_bad(disk_car,list_part_bad);
  }
  delete_list_part(list_part_bad);
  return list_part;
}

static void ask_mbr_order(t_param_disk *disk_car,t_list_part *list_part)
{
  t_partition *table[4];
  t_partition *table2[4];
  int nbr_prim=0;
  int i,pos=0;
  int res;
  int car;
  int quit=0;
  t_list_part *element;
  /* Initialisation */
  aff_copy(stdscr);
  wmove(stdscr,4,0);
  wdoprintf(stdscr,"%s",disk_car->description(disk_car));
  mvwaddstr(stdscr,5,0,msg_MBR_ORDER);
  mvwaddstr(stdscr,6,0,msg_PART_HEADER_LONG);
  for(element=list_part;element!=NULL;element=element->next)
  {
    if((element->part->order>0) && (element->part->order<5))
      table[nbr_prim++]=element->part;
  }
  /* */
  ecrit_rapport("\nSelection des partitions primaires\n");
  for(i=0;i<nbr_prim;i++)
      aff_part_rapport(disk_car,table[i]);
  /* */
  do
  {
    unsigned int order;
    /* sort table into table2 */
    int part=0;
    res=0;
    for(order=1;order<=4;order++)
    {
      int nbr=0;
      for(i=0;i<nbr_prim;i++)
	if(table[i]->order==order)
	{
	  table2[part++]=table[i];
	  nbr++;
	}
      res|=(nbr>1);
    }
    if(part!=nbr_prim)
    {
      ecrit_rapport("\nBUG part %d, nbr_prim %d\n", part, nbr_prim);
    }
    for(i=0;i<nbr_prim;i++)
    {
      wmove(stdscr,5+2+i,0);
      wclrtoeol(stdscr);
      if(i==pos)
	standout();
      aff_part(stdscr,AFF_PART_ORDER,disk_car,table2[i]);
      if(i==pos)
	standend();
    }
    wmove(stdscr,20,0);
    if(res)
      wdoprintf(stdscr,msg_MBR_ORDER_BAD);
    else
      wdoprintf(stdscr,msg_MBR_ORDER_GOOD);
    wrefresh(stdscr);
    car=wgetch(stdscr);
    quit=0;
    switch(car)
    {
      case KEY_UP:
	if(--pos<0)
	  pos=nbr_prim-1;
	break;
      case KEY_DOWN:
	if(++pos>=nbr_prim)
	  pos=0;
	break;
      case KEY_PPAGE:
	pos=0;
	break;
      case KEY_NPAGE:
	pos=nbr_prim-1;
	break;
      case '1':
      case '2':
      case '3':
      case '4':
	table2[pos]->order=car-'0';
	break;
      case KEY_RIGHT:
      case ' ':
      case '+':
	if(++table2[pos]->order>4)
	  table2[pos]->order=1;
	break;
      case KEY_LEFT:
      case '-':
	if(--table2[pos]->order<1)
	  table2[pos]->order=4;
	break;
      case 'q':
      case '\r':
      case '\n':
      case KEY_ENTER:
#ifdef PADENTER
      case PADENTER:
#endif
      case 'M':
	quit=1;
	break;
    }
    wrefresh(stdscr);
  } while(res!=0 || quit==0);
}


static t_list_part *reduce_structure(t_list_part *list_part)
{ 
  t_list_part *element=list_part;
  t_list_part *prev=NULL;
  while(element)
  {
    t_list_part *next=element->next;
    if(element->part->status==STATUS_DELETED)
    {
      if(prev==NULL)
	list_part=next;
      else
	prev->next=next;
      if(next!=NULL)
	next->prev=prev;
      FREE(element->part);
      FREE(element);
    }
    else
      prev=element;
    element=next;
  }
  return list_part;
}

static t_list_part *add_ext_part_i386(t_param_disk *disk_car, t_list_part *list_part, const int max_ext, const int align, const int debug)
{
  /* list_part need to be sorted! */
  /* All extended partitions of an P_EXTENDX are P_EXTENDED */
  t_list_part *element;
  t_list_part *deb=NULL;
  t_list_part *fin=NULL;
  int nbr_entries=0;
  t_CHS start,end;
  t_partition *new_partition;
  unsigned int order=0;
  for(element=list_part;element!=NULL;)
  {
    if(element->part->status==STATUS_EXT)
    {
      /* remove already existing extended partition */
      t_list_part *next=element->next;
      if(element->prev!=NULL)
	element->prev->next=element->next;
      if(element->next!=NULL)
	element->next->prev=element->prev;
      order=element->part->order;
      if(element==list_part)
	list_part=next;
      FREE(element->part);
      FREE(element);
      element=next;
    }
    else
    {
      if(element->part->status==STATUS_LOG)
      {
	if(deb==NULL)
	{
	  deb=element;
	  nbr_entries++;
	}
	fin=element;
      }
      else
	nbr_entries++;
      element=element->next;
    }
  }
  if(deb==NULL)
    return list_part;
  if((nbr_entries==4)||(max_ext!=0))
  {
    if(debug>1)
    {
      ecrit_rapport("add_ext_part_i386: max\n");
    }
    if(deb->prev==NULL)
    {
      offset2CHS(disk_car,deb->part->part_offset-1,&start);
      if((align>0) && (start.cylinder>0||start.head>1))
      {
	start.cylinder=0;
	start.head=1;
	start.sector=1;
      }
    }
    else
    {
      start.cylinder=offset2cylinder(disk_car,deb->prev->part->part_offset+deb->prev->part->part_size-1)+1;
      start.head=0;
      start.sector=1;
      if(CHS2offset(disk_car,&start)>=deb->part->part_offset)
      {
	offset2CHS(disk_car,deb->prev->part->part_offset+deb->prev->part->part_size-1+1,&start);
	start.sector=1;
	start.head++;
	if(start.head>=disk_car->CHS.head)
	{
	  start.cylinder++;
	  start.head=0;
	}
	if(CHS2offset(disk_car,&start)>=deb->part->part_offset)
	{
	  offset2CHS(disk_car,deb->prev->part->part_offset+deb->prev->part->part_size-1+1,&start);
	}
      }
    }
    if(fin->next==NULL)
    {
      end.cylinder=disk_car->CHS.cylinder;
      end.head=disk_car->CHS.head;
      end.sector=disk_car->CHS.sector;
    }
    else
    {
      end.cylinder=offset2cylinder(disk_car,fin->next->part->part_offset)-1; /* 8 october 2002 */
      end.head=disk_car->CHS.head;
      end.sector=disk_car->CHS.sector;
      if(CHS2offset(disk_car,&end)<=fin->part->part_offset+fin->part->part_size-1)
      {
	offset2CHS(disk_car,fin->next->part->part_offset-1,&end);
	end.sector=disk_car->CHS.sector;
	if(end.head>0)
	  end.head--;
	else
	{
	  end.cylinder--;
	  end.head=disk_car->CHS.head;
	}
	if(CHS2offset(disk_car,&end)<=fin->part->part_offset+fin->part->part_size-1)
	{
	  offset2CHS(disk_car,fin->next->part->part_offset-1,&end);
	}
      }
    }
  }
  else
  {
    if(debug>1)
    {
      ecrit_rapport("add_ext_part_i386: min\n");
    }
    offset2CHS(disk_car,deb->part->part_offset-1,&start);
    start.sector=1;
    if(deb->prev && CHS2offset(disk_car,&start)<=deb->prev->part->part_offset+deb->prev->part->part_size-1)
    {
      offset2CHS(disk_car,deb->part->part_offset-1,&start);
      start.sector=1;
      if(CHS2offset(disk_car,&start)<=deb->prev->part->part_offset+deb->prev->part->part_size-1)
      {
	offset2CHS(disk_car,deb->part->part_offset-1,&start);
      }
    }
    offset2CHS(disk_car,fin->part->part_offset+fin->part->part_size-1,&end);
    end.head=disk_car->CHS.head;
    end.sector=disk_car->CHS.sector;
    if(fin->next && CHS2offset(disk_car,&end)>=fin->next->part->part_offset)
    {
      offset2CHS(disk_car,fin->part->part_offset+fin->part->part_size-1,&end);
      end.sector=disk_car->CHS.sector;
    }
    if(fin->next && CHS2offset(disk_car,&end)>=fin->next->part->part_offset)
    {
      offset2CHS(disk_car,fin->part->part_offset+fin->part->part_size-1,&end);
    }
  }
  new_partition=partition_new();
  new_partition->order=order;
  new_partition->part_type_i386=(end.cylinder>1023?P_EXTENDX:P_EXTENDED);
  new_partition->status=STATUS_EXT;
  new_partition->part_offset=CHS2offset(disk_car,&start);
  new_partition->part_size=(uint64_t)CHS2offset(disk_car,&end)-new_partition->part_offset+disk_car->sector_size;
  return insert_new_partition(list_part, new_partition);
}

static int use_backup(t_param_disk *disk_car, const t_list_part *list_part, const int debug,const int dump_ind, const unsigned int expert, char**current_cmd)
{
  const t_list_part *element;
  if(debug>1)
  {
    ecrit_rapport("use_backup\n");
  }
  for(element=list_part;element!=NULL;element=element->next)
  {
    if(element->part->boot_sector!=0)
    {
      switch(element->part->upart_type)
      {
	case UP_FAT32:
	  fat32_boot_sector(disk_car, element->part, debug, dump_ind, expert,current_cmd);
	  break;
	case UP_NTFS:
	  ntfs_boot_sector(disk_car, element->part, debug, dump_ind,current_cmd);
	  break;
	default:
	  ecrit_rapport("Need to fix\n");
	  aff_part_rapport(disk_car,element->part);
	  break;
      }
    }
  }
  return 0;
}

int interface_recovery(t_param_disk *disk_car,int paranoid,const int debug,const int dump_ind, int fast_mode, const int align, const int ask_part_order, const unsigned int expert, char **current_cmd)
{
  int res_interface_write;
  int menu=-1;
  do
  {
    t_list_part *list_part;
    t_list_part *element;
    int can_search_deeper=1;
    aff_copy(stdscr);
    wmove(stdscr,4,0);
    wdoprintf(stdscr,"%s",disk_car->description(disk_car));
    wmove(stdscr,5,0);
    res_interface_write=0;
    if(paranoid==1 && (fast_mode==2 || (fast_mode==1 &&expert==0)))
      can_search_deeper=0;
    list_part=search_part(disk_car,paranoid,debug,dump_ind,fast_mode,1);
    if(list_part!=NULL)
    {
      unsigned int head_max;
      head_max=get_geometry_from_list_part(disk_car, list_part, debug);
      if(disk_car->CHS.head!=head_max)
      {
	warning_geometry(disk_car,head_max+1,current_cmd);
      }
    }
    align_structure(disk_car,list_part,align);
    list_part=search_part_raid(disk_car,list_part,list_part,debug,dump_ind,1);
    align_structure(disk_car,list_part,align);
#ifdef DEBUG
    check_list_part(list_part);
#endif
    disk_car->arch->init_structure(disk_car,list_part,debug);
#ifdef DEBUG
    check_list_part(list_part);
#endif
    if(debug>0)
    {
      /* Ajoute dans le fichier de log les partitions trouvees */
      ecrit_rapport("\nResults\n");
      for(element=list_part;element!=NULL;element=element->next)
	aff_part_rapport(disk_car,element->part);
    }
    wdoprintf(stdscr,msg_GOD_MODE);
    list_part=ask_structure(disk_car,list_part,debug,current_cmd);
#ifdef DEBUG
    check_list_part(list_part);
#endif
    if(disk_car->arch->test_structure(list_part)==0)
    {
      int do_again=0;
      int max_ext=0;
      int can_ask_minmax_ext=0;
#ifdef DEBUG
      check_list_part(list_part);
#endif
      list_part=reduce_structure(list_part);
#ifdef DEBUG
      check_list_part(list_part);
#endif
      /* Sort list_part */
      list_part=sort_list_part(list_part);
#ifdef DEBUG
      check_list_part(list_part);
#endif
      /* Creer la partition etendue globale, cherche  aligner la partition */
      /* if(disk_car->arch==&arch_i386) */
      {
	t_list_part *parts;
	uint64_t partext_offset=0;
	uint64_t partext_size=0;
	list_part=add_ext_part_i386(disk_car,list_part,!max_ext,2,debug);
	for(parts=list_part;parts!=NULL;parts=parts->next)
	  if(parts->part->status==STATUS_EXT)
	  {
	    partext_offset=parts->part->part_offset;
	    partext_size=parts->part->part_size;
	  }
#ifdef DEBUG
	check_list_part(list_part);
#endif
	if(partext_offset>0)
	{
	  list_part=add_ext_part_i386(disk_car,list_part,max_ext,2,debug);
	  for(parts=list_part;parts!=NULL;parts=parts->next)
	    if(parts->part->status==STATUS_EXT)
	    {
	      if(partext_offset!=parts->part->part_offset || partext_size!=parts->part->part_size)
		can_ask_minmax_ext=1;
	    }
	}
#ifdef DEBUG
	check_list_part(list_part);
#endif
      }
      list_part=disk_car->arch->init_part_order(disk_car,list_part);
      if(ask_part_order!=0)
      {
	/* Demande l'ordre des entrees dans le MBR */
	ask_mbr_order(disk_car,list_part);
	/* Demande l'ordre des partitions etendues */
      }
      do
      {
	do_again=0;
	res_interface_write=interface_write(disk_car,list_part,can_search_deeper,can_ask_minmax_ext,current_cmd,&menu);
	switch(res_interface_write)
	{
	  case 'W':
	    if(disk_car->arch->write_part!=NULL)
	    {
	      if(ask_confirmation("Write partition table, confirm ? (Y/N)"))
	      {
		ecrit_rapport("write!\n");
		if(disk_car->arch->write_part(disk_car,list_part,RW,debug,align))
		{
		  display_message(msg_PART_WR_ERR);
		}
		else
		{
		  use_backup(disk_car,list_part,debug,dump_ind,expert,current_cmd);
		  display_message("You will have to reboot for the change to take effect.\n");
		}
	      }
	      else
		ecrit_rapport("Don't write, no confirmation\n");
	    }
	    break;
	  case 0:
	    if(disk_car->arch->write_part!=NULL)
	    {
	      ecrit_rapport("simulate write!\n");
	      disk_car->arch->write_part(disk_car,list_part,RO,debug,align);
	    }
	    break;
	  case 'S':
	    paranoid=1;
	    if(fast_mode<2)
	      fast_mode++;
	    break;
	  case 'E':
	    max_ext=!max_ext;
	    list_part=add_ext_part_i386(disk_car,list_part,max_ext,2,debug);
	    do_again=1;
	    break;
	}
      }
      while(do_again==1);
    }
    else
    {
      display_message("Invalid partition structure");
    }
    delete_list_part(list_part);
  } while(res_interface_write=='S');
  return 0;
}
