/*

    File: jfs.c

    Copyright (C) 2004 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
 
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include "types.h"
#include "common.h"
#include "jfs_superblock.h"
#include "jfs.h"
#include "intrf.h"
#include "fnctdsk.h"

static int test_JFS(t_param_disk *disk_car, const struct jfs_superblock *sb,t_partition *partition,const int debug, const int dump_ind);

int check_JFS(t_param_disk *disk_car,t_partition *partition,const int debug)
{
  unsigned char buffer[8*DEFAULT_SECTOR_SIZE];
  if(disk_car->read(disk_car,sizeof(buffer), &buffer, partition->part_offset+64*512)!=0)
  { return 1; }
  if(test_JFS(disk_car,(struct jfs_superblock*)&buffer,partition,debug,0)!=0)
    return 1;
  set_JFS_info(disk_car,(struct jfs_superblock*)&buffer,partition,debug,0);
  return 0;
}

int set_JFS_info(t_param_disk *disk_car, const struct jfs_superblock *sb,t_partition *partition,const int debug, const int dump_ind)
{
  partition->info[0]='\0';
  partition->name[0]='\0';
  if(sb->s_version==1)
  {
    set_part_name(partition,sb->s_fpack,11);
  }
  snprintf(partition->info,sizeof(partition->info),"JFS %u",(unsigned int)sb->s_version);
  return 0;
}

/*
Primary superblock is at 0x8000
*/
int recover_JFS(t_param_disk *disk_car, const struct jfs_superblock *sb,t_partition *partition,const int debug, const int dump_ind)
{
  if(test_JFS(disk_car,sb,partition,debug,dump_ind)!=0)
    return 1;
  set_JFS_info(disk_car,sb,partition,debug,dump_ind);
  partition->part_type=(unsigned char)P_LINUX;
  partition->part_size=(uint64_t)sb->s_pbsize*le64(sb->s_size)+
    sb->s_bsize*(le24(sb->s_fsckpxd.len)+le24(sb->s_logpxd.len));
  partition->boot_sector=0;
  partition->blocksize=le32(sb->s_bsize);
  if(debug!=0)
  {
    ecrit_rapport("\n");
    ecrit_rapport("recover_JFS: s_blocksize=%lu\n",partition->blocksize);
    ecrit_rapport("recover_JFS: s_size %lu\n",(long unsigned int)le64(sb->s_size));
    ecrit_rapport("recover_JFS: s_fsckpxd.len:%d\n", (int)le24(sb->s_fsckpxd.len));
    ecrit_rapport("recover_JFS: s_logpxd.len:%d\n", (int)le24(sb->s_logpxd.len));
    ecrit_rapport("recover_JFS: part_size %lu\n",(long unsigned)(partition->part_size/disk_car->sector_size));
  }
  return 0;
}

static int test_JFS(t_param_disk *disk_car, const struct jfs_superblock *sb,t_partition *partition,const int debug, const int dump_ind)
{
  if(memcmp(sb->s_magic,"JFS1",4)!=0)
    return 1;
  if(dump_ind!=0)
  {
    ecrit_rapport("\nJFS magic value at %u/%u/%u\n", offset2cylinder(disk_car,partition->part_offset),offset2head(disk_car,partition->part_offset),offset2sector(disk_car,partition->part_offset));
    /* There is a little offset ... */
    dump(stdscr,sb,DEFAULT_SECTOR_SIZE);
  }
  /*
  if( sb->s_agsize >= (1 << L2BPERDMAP) ) {
    return 2;
  }
  if(partition->part_size!=0 && (partition->part_size<le64(sb->s_size)))
    return 8;
    */
  partition->upart_type=UP_JFS;
  return 0;
}
