/*

    File: rfs_dir.c

    Copyright (C) 1998-2004 Christophe GRENIER <grenier@cgsecurity.org>
    Some code from Yury Umanets <torque@ukrpost.net>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
 
#ifdef HAVE_LIBREISERFS
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <time.h>
#include "dal/dal.h"
#ifdef HAVE_DAL_FILE_DAL_H
#include "dal/file_dal.h"
#endif
#ifdef HAVE_DAL_FILE_H
#include "dal/file.h"
#endif
#include "reiserfs/reiserfs.h"
#include "types.h"
#include "common.h"
#include "intrface.h"
#include "rfs.h"
#include "dir.h"

struct list_dir_struct {
	t_file_data *dir_list;
	t_file_data *current_file;
	reiserfs_fs_t *current_fs;
	int flags;
};

#ifdef HAVE_STRUCT_DAL_OPS_DEV
dev_t dal_dev(dal_t *dal) {
    
    if (!dal)
	return 0;
    return (dev_t)dal->dev;
}
#endif

#ifdef HAVE_STRUCT_DAL_OPS_DEV
size_t dal_block_size(dal_t *dal) {
    if (!dal) return 0;
#ifdef HAVE_DAL_T_BLOCK_SIZE
    return dal->block_size;
#else
    return dal->blocksize;
#endif
}
#else
unsigned dal_get_blocksize(dal_t *dal) {
  if (!dal)
    return 0;
#ifdef HAVE_DAL_T_BLOCK_SIZE
    return dal->block_size;
#else
    return dal->blocksize;
#endif
}
#endif

/*
size_t dal_blocksize(dal_t *dal) {

	if (!dal) return 0;
#ifdef HAVE_DAL_T_BLOCK_SIZE
    return dal->block_size;
#else
    return dal->blocksize;
#endif

}
*/

char *dal_error(dal_t *dal) {
#ifdef HAVE_DAL_T_ERROR
    return dal->error;
#else
    return "";
#endif
}

static int file_read(dal_t *dal, void *buff, blk_t block, blk_t count) {
  dword off, blocklen;
  t_my_data *my_data=(t_my_data*)dal->data;
/* ecrit_rapport("reiser file_read(dal=%p,buff=%p,block=%ld, count=%ld)\n",dal,buff,block,count); */
  if (!dal || !buff)
    return 0;
#ifdef HAVE_DAL_T_BLOCK_SIZE
  off = (dword)block * ((dword)dal->block_size/SECTOR_SIZE);
  blocklen = (dword)count * ((dword)dal->block_size/SECTOR_SIZE);
#else
  off = (dword)block * ((dword)dal->blocksize/SECTOR_SIZE);
  blocklen = (dword)count * ((dword)dal->blocksize/SECTOR_SIZE);
#endif
/* ecrit_rapport("blocklen=%ld\n",blocklen); */
  if(my_data->disk_car->read(my_data->disk_car,blocklen,buff,my_data->partition->lba+off))
    return 0;
  return 1;
}

static int file_write(dal_t *dal, void *buff, blk_t block, blk_t count)
{
  dword off, blocklen;
/* ecrit_rapport("reiser file_write\n"); */
  if (!dal || !buff)
    return 0;
#ifdef HAVE_DAL_T_BLOCK_SIZE
  off = (dword)block * ((dword)dal->block_size/SECTOR_SIZE);
  blocklen = (dword)count * ((dword)dal->block_size/SECTOR_SIZE);
#else
  off = (dword)block * ((dword)dal->blocksize/SECTOR_SIZE);
  blocklen = (dword)count * ((dword)dal->blocksize/SECTOR_SIZE);
#endif
/* if(my_data->disk_car->write(my_data->disk_car,blocklen,buff,my_data->partition->lba+off)) */
    return 0;
/* return 1; */
}

static int file_sync(dal_t *dal)
{
  if (!dal) return 0;
/* ecrit_rapport("reiser file_sync\n"); */
  return 1;
}

static int file_flags(dal_t *dal)
{
/* ecrit_rapport("reiser file_flags\n"); */
  if (!dal) return 0;
  return dal->flags;
}

static int file_equals(dal_t *dal1, dal_t *dal2)
{
/* ecrit_rapport("reiser file_equals\n"); */
  if (!dal1 || !dal2)
    return 0;
  return !strcmp((char *)dal1->data, (char *)dal2->data);
}

#ifdef HAVE_STRUCT_DAL_OPS_DEV
static int file_stat(dal_t *dal,struct stat *st)
{
/* ecrit_rapport("reiser file_stat\n"); */
  if (!dal)
    return 0;
/* return (unsigned int)st.st_dev; */
  return 1;
}
#else
static unsigned int file_stat(dal_t *dal)
{
/* ecrit_rapport("reiser file_stat\n"); */
  if (!dal)
    return 0;
/* return (unsigned int)st.st_dev; */
  return 1;
}
#endif

#ifdef HAVE_STRUCT_DAL_OPS_DEV
static blk_t   file_len(dal_t *dal) {
#else
static count_t file_len(dal_t *dal) {
#endif
  dword max_off = 0;
  t_my_data *my_data=(t_my_data*)dal->data;
/* ecrit_rapport("reiser file_len\n"); */
  if (!dal) return 0;
  max_off=my_data->partition->part_size*SECTOR_SIZE;
#ifdef HAVE_DAL_T_BLOCK_SIZE
  return max_off / dal->block_size;
#else
  return max_off / dal->blocksize;
#endif
}

static struct dal_ops ops = {
	read: file_read, 
	write: file_write, 
	sync: file_sync, 
	flags: file_flags, 
	equals: file_equals, 
	stat: file_stat, 
	len: file_len,
#ifdef HAVE_STRUCT_DAL_OPS_DEV
	dev: dal_dev
#endif
};

void dal_close(dal_t *dal) {
/* ecrit_rapport("reiser dal_close\n"); */
	
	if (!dal) return;
	
	dal->ops = NULL;
	dal->data = NULL;
	FREE(dal);
}

void file_close(dal_t *dal) {
/* ecrit_rapport("reiser file_close\n"); */
	if (!dal) return;
	dal_close(dal);
}

static int power_of_two(unsigned long value) {
	return (value & -value) == value;
}

#ifdef HAVE_STRUCT_DAL_OPS_DEV
dal_t *dal_open(struct dal_ops *myops, const void *dev, size_t blocksize, int flags, void *data)
#else
dal_t *dal_open(struct dal_ops *myops, unsigned blocksize, int flags, void *data) 
#endif
{
	dal_t *dal;
/* ecrit_rapport("reiser dal_open\n"); */
	
	if (!myops) return NULL;
	
	if (!power_of_two(blocksize)) {
		fprintf(stderr, "Block size isn't power of two.\n");
		return NULL;
	}	
	
	if (!(dal = (dal_t *)MALLOC(sizeof(*dal))))
		return NULL;

	memset(dal, 0, sizeof(*dal));
	
	dal->ops = myops;
#ifdef HAVE_DAL_T_BLOCK_SIZE
	dal->block_size = blocksize;
#else
	dal->blocksize = blocksize;
#endif
#ifdef HAVE_DAL_T_ENTITY
	dal->entity= NULL;
#endif
#ifdef HAVE_DAL_T_NAME
	strncpy(dal->name, "/dev/reiserfs",sizeof(dal->name));
#endif
#ifdef HAVE_DAL_T_ERROR
	dal->error[0]=0;
#endif
	dal->data = data;
	dal->flags = flags;
	return dal;
}

#ifdef HAVE_STRUCT_DAL_OPS_DEV
int dal_set_block_size(dal_t *dal, unsigned blocksize) {

	if (!dal) return 0;
	
	if (!power_of_two(blocksize))
		return 0;
#ifdef HAVE_DAL_T_BLOCK_SIZE
	dal->block_size = blocksize;
#else
	dal->blocksize = blocksize;
#endif
	return 1;
}
#else
int dal_set_blocksize(dal_t *dal, unsigned blocksize) {

	if (!dal) return 0;
	
	if (!power_of_two(blocksize))
		return 0;
#ifdef HAVE_DAL_T_BLOCK_SIZE
	dal->block_size = blocksize;
#else
	dal->blocksize = blocksize;
#endif

	return 1;
}
#endif


int dal_read(dal_t *dal, void *buff, blk_t block, blk_t count) {

/* ecrit_rapport("reiser dal_read\n"); */
	if (!dal) return 0;

	if (dal->ops->read)
		return dal->ops->read(dal, buff, block, count);
	
	return 0;
}

int dal_write(dal_t *dal, void *buff, blk_t block, blk_t count) {

/* ecrit_rapport("reiser dal_write\n"); */
	if (!dal) return 0;
	
	if (dal->ops->write)
		return dal->ops->write(dal, buff, block, count);
		
	return 0;
}
	
int dal_sync(dal_t *dal) {

	if (!dal) return 0;

	if (dal->ops->sync)
		return dal->ops->sync(dal);
	
	return 0;	
}

int dal_flags(dal_t *dal) {

	if (!dal) return 0;

	if (dal->ops->flags)
		return dal->ops->flags(dal);
	
	return 0;
}

int dal_equals(dal_t *dal1, dal_t *dal2) {
/* ecrit_rapport("reiserfs dal_equals\n"); */
	
	if (!dal1 || !dal2) return 0;

	if (dal1->ops->equals)
		return dal1->ops->equals(dal1, dal2);
	
	return 0;	
}

#if defined(HAVE_STRUCT_DAL_OPS_DEV)
int dal_stat(dal_t *dal, struct stat *mystat)
{
/* ecrit_rapport("reiserfs dal_stat\n"); */
	if (!dal) return 0;
	if (dal->ops->stat)
		return dal->ops->stat(dal,mystat);
	return 0;
}
#else
unsigned int dal_stat(dal_t *dal) {
/* ecrit_rapport("reiserfs dal_stat\n"); */
	if (!dal) return 0;
	if (dal->ops->stat)
		return dal->ops->stat(dal);
	return 0;
}
#endif

blk_t dal_len(dal_t *dal) {
/* ecrit_rapport("reiserfs dal_len\n"); */
	
	if (!dal) 
		return 0;

	if (dal->ops->len)
		return dal->ops->len(dal);

	return 0;
}

t_file_data *reiser_dir(t_param_disk *disk_car, const t_diskext *partition, t_dir_data *dir_data, const unsigned long int cluster);
t_file_data *reiser_dir(t_param_disk *disk_car, const t_diskext *partition, t_dir_data *dir_data, const unsigned long int cluster)
{
  struct list_dir_struct *ls=(struct list_dir_struct*)dir_data->private_dir_data;
  reiserfs_dir_t *dir;
  reiserfs_dir_entry_t entry;
  ls->dir_list=NULL;
  ls->current_file=NULL;
  if (!(dir = reiserfs_dir_open(ls->current_fs, dir_data->current_directory))) {
    aff_buffer(BUFFER_ADD,"Couldn't open dir\n");
    ecrit_rapport("Couldn't open dir\n");
    return NULL;
  }
  while (reiserfs_dir_read(dir, &entry))
  {
    unsigned int thislen;
    char name[MAX_NAME_LEN(DEFAULT_BLOCK_SIZE)];
    reiserfs_object_t *entity;
    strncpy(name,dir_data->current_directory,sizeof(name));
    strcat(name,"/");
    strcat(name,entry.de_name);
    if((entity=reiserfs_object_create(ls->current_fs,name,1)))
    {
      t_file_data *new_file=MALLOC(sizeof(*new_file));
      thislen=(MAX_NAME_LEN(DEFAULT_BLOCK_SIZE)<DIR_NAME_LEN?MAX_NAME_LEN(DEFAULT_BLOCK_SIZE):DIR_NAME_LEN);
      memcpy(new_file->name,entry.de_name,thislen);
      new_file->name[thislen]='\0';

      new_file->prev=ls->current_file;
      new_file->next=NULL;
      new_file->filestat.st_size=entity->stat.st_size;
      memcpy(&new_file->filestat,&entity->stat,sizeof(new_file->filestat));
      reiserfs_object_free(entity);
      if(ls->current_file)
	ls->current_file->next=new_file;
      else
	ls->dir_list=new_file;
      ls->current_file=new_file;
    }
  }
  reiserfs_dir_close(dir);
  return ls->dir_list;
}


int dir_partition_reiser(WINDOW *window,t_param_disk *disk_car,t_diskext *partition, const int debug)
{
  dal_t *dal;
  reiserfs_fs_t *fs;
  reiserfs_path_node_t *leaf;
  t_my_data my_data;
  int32_t offset = 0;
  int error = 0;

  aff_buffer(BUFFER_RESET,"Q");
  wmove(window,4,0);
  aff_part(window,AFF_PART_NL,disk_car,partition);
  ecrit_rapport("\n");
  aff_part_rapport(disk_car,partition);
  my_data.partition=partition;
  my_data.disk_car=disk_car;
#ifdef HAVE_STRUCT_DAL_OPS_DEV
  dal=dal_open(&ops, "",DEFAULT_BLOCK_SIZE, O_RDONLY, &my_data);
#else
  dal=dal_open(&ops, DEFAULT_BLOCK_SIZE, O_RDONLY, &my_data);
#endif
  if (!dal)
  {
    ecrit_rapport("Couldn't open device\n");
    aff_buffer(BUFFER_ADD,"Couldn't open device\n");
    goto dir_partition_reiser_error;
  }
/* ecrit_rapport("file_open ok\n"); */

#ifdef HAVE_REISERFS_FS_OPEN_FAST
  if (!(fs = reiserfs_fs_open_fast(dal, dal))) {
#else
  if (!(fs = reiserfs_fs_open(dal, dal))) {
#endif
    ecrit_rapport("Couldn't open reiser filesystem %s\n",dal_error(dal));
    aff_buffer(BUFFER_ADD,"Couldn't open reiser filesystem\n");
    goto error_free_dal;
  }
/* ecrit_rapport("reiserfs_fs_open_fast ok\n"); */
  {
    struct list_dir_struct ls;
    t_dir_data dir_data;
    ls.dir_list=NULL;
    ls.current_file=NULL;
    ls.current_fs=fs;
    ls.flags = 0; /*DIRENT_FLAG_INCLUDE_EMPTY; */
    dir_data.window=window;
    dir_data.debug=debug;
    dir_data.private_dir_data=&ls;
    dir_data.get_dir=reiser_dir;
    strncpy(dir_data.current_directory,"/",sizeof(dir_data.current_directory));
    dir_partition(disk_car,partition,&dir_data,2);
  }
  reiserfs_fs_close(fs);
  file_close(dal);
  return 0;
error_free_dal:
  file_close(dal);
dir_partition_reiser_error:
  aff_buffer(BUFFER_DISPLAY,"Q",window);
  return 0xff;    
}

#else
#include "types.h"
#include "common.h"
#include "intrface.h"

int dir_partition_reiser(WINDOW *window,t_param_disk *disk_car,t_diskext *partition)
{
  aff_buffer(BUFFER_RESET,"Q");
  wmove(stdscr,5,0);
  aff_part(window,AFF_PART_NONL,disk_car,partition);
  aff_part_rapport(disk_car,partition);
  aff_buffer(BUFFER_ADD,"Recompile with progsreiserfs library");
  aff_buffer(BUFFER_DISPLAY,"Q",window);
  return 0;
}
#endif
