/* TABLIX, PGA highschool timetable generator                              */
/* Copyright (C) 2002 Tomaz Solc                                           */

/* This program is free software; you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation; either version 2 of the License, or       */
/* (at your option) any later version.                                     */

/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */

/* You should have received a copy of the GNU General Public License       */
/* along with this program; if not, write to the Free Software             */
/* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

/* $Id: export_ttf.c,v 1.4 2004/09/26 18:59:18 avian Exp $ */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "error.h"
#include "data.h"
#include "xmlsup.h"
#include "gettext.h"

#include "scheme.h"
#include "scheme-private.h"

#define BUFSIZE	256
#define START_MARKER "BEGIN TTF BLOCK"
#define END_MARKER "END TTF BLOCK"

const char scheme_init_string[] =
"(define (test-ttf . lst)"
"	(if (test-ttf-loop #t lst 1)"
"    		(quit 2)"
"    		(quit 1)"
"    	)"
")"

"(define (test-ttf-loop x lst i)"
"	(if (null? lst)"
"       		x"
"		( begin"
"			(if (not (car lst))"
"				( begin"
"					(display \"test-ttf: test number \")"
"					(display i)"
"					(display \" failed\")"
"					(newline)"
"				)"
"			)"
" 	         	(test-ttf-loop (and x (car lst)) (cdr lst) (+ i 1))"
"		)"
"     	)"
")";

chromo *cur_chromo;

int get_tuple(scheme *sc, pointer *args)
{
	int tuple;

	if(*args==sc->NIL) {
		fatal("Invalid tuple number");
	}

	if(!is_number(pair_car(*args))) {
		fatal("Invalid tuple number");
	}
	
	tuple=ivalue(pair_car(*args));

	if(tuple<0||tuple>tuplenum) {
		fatal("Invalid tuple number");
	}

	(*args)=pair_cdr(*args);

	return(tuple);
}

pointer sc_testvalue(scheme *sc, pointer args, int value)
{
	int min;
	int max;

	if(args==sc->NIL) {
		return sc->NIL;
	}

	if(!is_number(pair_car(args))) {
		return sc->NIL;
	}

	min=ivalue(pair_car(args));

	if(is_number(pair_car(pair_cdr(args)))) {
		max=ivalue(pair_car(pair_cdr(args)));

		if(value>=min&&value<=max) {
			return(sc->T);
		} else {
			return(sc->F);
		}
	} else {
		if(value==min) {
			return(sc->T);
		} else {
			return(sc->F);
		}
	}
}

pointer sc_period(scheme *sc, pointer args) 
{
	int period;
	int tid;

	tid=get_tuple(sc, &args);

	period=(cur_chromo->inf[tid].time)%PERIODS;
	return(sc_testvalue(sc, args, period));
}

pointer sc_day(scheme *sc, pointer args) 
{
	int day;
	int tid;

	tid=get_tuple(sc, &args);

	day=(cur_chromo->inf[tid].time)/PERIODS;
	return(sc_testvalue(sc, args, day));
}

pointer sc_time(scheme *sc, pointer args) 
{
	int time;
	int tid;

	tid=get_tuple(sc, &args);

	time=cur_chromo->inf[tid].time;
	return(sc_testvalue(sc, args, time));
}

pointer sc_room(scheme *sc, pointer args) 
{
	int room;
	int tid;

	tid=get_tuple(sc, &args);

	room=cur_chromo->inf[tid].room;
	return(sc_testvalue(sc, args, room));
}

pointer sc_getperiod(scheme *sc, pointer args)
{
	int tid;
	int period;

	if(args==sc->NIL) {
		return sc->NIL;
	}

	if(!is_number(pair_car(args))) {
		return sc->NIL;
	}

	tid=ivalue(pair_car(args));

	if(tid<0||tid>tuplenum) {
		fatal("Invalid tuple number");
	}

	period=(cur_chromo->inf[tid].time)%PERIODS;

	return(mk_integer(sc,period));
}

pointer sc_getday(scheme *sc, pointer args)
{
	int tid;
	int day;

	if(args==sc->NIL) {
		return sc->NIL;
	}

	if(!is_number(pair_car(args))) {
		return sc->NIL;
	}

	tid=ivalue(pair_car(args));

	if(tid<0||tid>tuplenum) {
		fatal("Invalid tuple number");
	}

	day=(cur_chromo->inf[tid].time)/PERIODS;

	return(mk_integer(sc,day));
}

pointer sc_gettime(scheme *sc, pointer args)
{
	int tid;
	int time;

	if(args==sc->NIL) {
		return sc->NIL;
	}

	if(!is_number(pair_car(args))) {
		return sc->NIL;
	}

	tid=ivalue(pair_car(args));

	if(tid<0||tid>tuplenum) {
		fatal("Invalid tuple number");
	}

	time=cur_chromo->inf[tid].time;

	return(mk_integer(sc,time));
}

pointer sc_getroom(scheme *sc, pointer args)
{
	int tid;
	int room;

	if(args==sc->NIL) {
		return sc->NIL;
	}

	if(!is_number(pair_car(args))) {
		return sc->NIL;
	}

	tid=ivalue(pair_car(args));

	if(tid<0||tid>tuplenum) {
		fatal("Invalid tuple number");
	}

	room=cur_chromo->inf[tid].room;

	return(mk_integer(sc,room));
}

void output_function(chromo *t, int *cpnt, int *tpnt, char *options, FILE* out, outputext *opnt)
{
	FILE *conffile;
	char line[BUFSIZE];
	int passed=0;

	scheme *sc;

	info("TinyScheme, Copyright (c) 2000, Dimitrios Souflis. All rights reserved.");

	sc=scheme_init_new();
	if(!sc) {
		fatal("Scheme interpreter failed to initialize");
	}

	scheme_set_output_port_file(sc, stdout);

 	scheme_define(sc,sc->global_env,mk_symbol(sc,"period"),mk_foreign_func(sc, sc_period)); 
 	scheme_define(sc,sc->global_env,mk_symbol(sc,"day"),mk_foreign_func(sc, sc_day)); 
 	scheme_define(sc,sc->global_env,mk_symbol(sc,"time"),mk_foreign_func(sc, sc_time)); 
 	scheme_define(sc,sc->global_env,mk_symbol(sc,"room"),mk_foreign_func(sc, sc_room)); 
 	scheme_define(sc,sc->global_env,mk_symbol(sc,"get-period"),mk_foreign_func(sc, sc_getperiod)); 
 	scheme_define(sc,sc->global_env,mk_symbol(sc,"get-day"),mk_foreign_func(sc, sc_getday)); 
 	scheme_define(sc,sc->global_env,mk_symbol(sc,"get-time"),mk_foreign_func(sc, sc_gettime)); 
 	scheme_define(sc,sc->global_env,mk_symbol(sc,"get-room"),mk_foreign_func(sc, sc_getroom)); 

	scheme_load_string(sc,scheme_init_string);

	if(strlen(options)==0) {
		fatal("No config file specified");
	}
	conffile=fopen(options,"r");
	if(conffile==NULL) {
		fatal("Can't open config file");
	}

	while(fgets(line, BUFSIZE, conffile)!=NULL) {
		if(strstr(line,START_MARKER)!=NULL) break;
	}

	if(strstr(line,START_MARKER)==NULL) {
		fatal("'" START_MARKER "' expected");
	}

	cur_chromo=t;

	scheme_load_file(sc,conffile);

	if(sc->retcode==2) {
		passed=1;
	} else if(sc->retcode==1) {
		passed=0;
	} else {
		fatal("Scheme interpreter error");
	}

	fclose(conffile);

	scheme_deinit(sc);

	if(passed) {
		info("All tests passed");
	} else {
		error("Some tests failed");
	}
}
