/* Copyright (C) 2004 Nikos Chantziaras.
 *
 * This file is part of the QTads program.  This program is free
 * software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

/* Qt-specific Tads OS functions (used only by Tads 3).
 *
 * This file is used by both QTads as well as HTML QTads.  Functions
 * that have to be excluded in the HTML-version are inside an #ifndef
 * block:
 *
 *     #ifndef HTMLQT
 *     // Function not needed by HTML QTads.
 *     #endif
 *
 * This file should only contain Tads OS specific functions.  That
 * doesn't mean that you can't use C++ code inside the functions; you
 * can use any C++ feature you want, as long as the function headers
 * are compatible with the prototypes in "osifc.h".  The only exception
 * are static functions (they aren't exported).
 *
 * Again: No functions are allowed in here that aren't listed in
 * "osifc.h", except helper functions.
 */

#include "config.h"

#include <qapplication.h>
#include <qfileinfo.h>
#include <qdir.h>
#include <qtextcodec.h>

#include "os.h"
#include "qtadswarning.h"
#include "qtadsio.h"
#include "qtadsgamewindow.h"
#include "qtadssettings.h"


/* --------------------------------------------------------------------
 * Special file and directory locations.
 */

/* Get the full filename (including directory path) to the executable
 * file.
 *
 * The implementation provided here can handle links.  (Both Unix-links
 * as well as Windows-shortcuts, since Qt supports then both; I don't
 * know how a link looks like on a Mac, but Qt should support that
 * too.)
 *
 * TODO: Search through the PATH env. variable.  Find out how this is
 * supposed to work on the Mac.
 */
int
os_get_exe_filename( char* buf, size_t buflen, const char* argv0 )
{
	QFileInfo file(argv0);
	file.convertToAbs();
	if (!file.exists() or !file.isReadable()) {
		return false;
	}

	// If the file is some form of link, find out where it points to.
	if (file.isSymLink()) {
		while (not file.readLink().isEmpty()) {
			file.setFile(file.readLink());
		}
		file.convertToAbs();
		if (not file.exists() or not file.isReadable()) {
			return false;
		}
	}

	if (file.filePath().length() + 1 > buflen) {
		// The result would not fit in the buffer.
		return false;
	}

	strcpy(buf, file.filePath().ascii());
	return true;
}


/* Get a special directory path.
 *
 * TODO: Implement it.
 */
void
os_get_special_path( char* /*buf*/, size_t /*buflen*/, const char* /*argv0*/, int /*id*/ )
{
	/*
	switch (id) {
	  case OS_GSP_T3_RES:
		strcpy(buf, "/usr/local/share/tads/tads3/re");
		break;
	  case OS_GSP_T3_INC:
		strcpy(buf, "/usr/local/share/tads/tads3/in");
		break;
	  case OS_GSP_T3_LIB:
		strcpy(buf, "/usr/local/share/tads/tads3/li");
		break;
	  case OS_GSP_T3_USER_LIBS:
		strcpy(buf, "./");
		break;
	  default:
		qtadsWarning("Unknown id in os_get_special_path()!");
		buf[0] = '\0';
	}
	*/
}


/* --------------------------------------------------------------------
 */

/* Generate the name of the character set mapping table for Unicode
 * characters to and from the given local character set.
 */
void
os_get_charmap( char* mapname, int charmap_id )
{
#ifndef HTMLQT
	Q_ASSERT(QTadsIO::t3Mode());
#endif

	switch(charmap_id) {
	  case OS_CHARMAP_DISPLAY:
		// Always use UTF-8 for the display, regardless of the
		// local charset; Qt uses Unicode for the display on
		// every system.
		strcpy(mapname, "utf8");
		break;
	  case OS_CHARMAP_FILENAME:
	  case OS_CHARMAP_FILECONTENTS:
		strcpy(mapname, QTextCodec::codecForLocale()->mimeName());
		break;
	  default:
		qWarning("os_get_charmap() got an unknown charmap id");
	  	strcpy(mapname, QTextCodec::codecForLocale()->mimeName());
		break;
	}
}


/* --------------------------------------------------------------------
 * Display routines.
 */

/* Update the display - process any pending drawing immediately.
 */
#ifndef HTMLQT
void
os_update_display()
{
	Q_ASSERT(QTadsIO::t3Mode());

	qApp->processEvents();
}
#endif


/* --------------------------------------------------------------------
 * User Input Routines.
 */

/* Read a string of input with an optional timeout.
 *
 * TODO: Implement it.
 */
#ifndef HTMLQT
int
os_gets_timeout( unsigned char* /*buf*/, size_t /*bufl*/, unsigned long /*timeout_in_milliseconds*/,
		 int /*use_timeout*/ )
{
	Q_ASSERT(QTadsIO::t3Mode());

	return OS_EVT_NOTIMEOUT;
}
#endif


/* Cancel an interrupted editing session.
 *
 * TODO: Implement it.
 */
#ifndef HTMLQT
void
os_gets_cancel( int /*reset*/ )
{
	Q_ASSERT(QTadsIO::t3Mode());
}
#endif


/* --------------------------------------------------------------------
 * Colors.
 */

/* Set the text foreground and background colors.
 *
 * TODO: Only foreground color is supported, as I don't know how to
 *       change the background color of individual characters in a
 *       QTextEdit.
 */
#ifndef HTMLQT
void
os_set_text_color( os_color_t fg, os_color_t /*bg*/ )
{
	Q_ASSERT(QTadsIO::t3Mode());

	if (os_color_is_param(fg)) {
		switch (fg) {
		  case OS_COLOR_P_TEXT:
		  case OS_COLOR_P_INPUT:
		  	// We'll simply switch to the default color by
			// using the "invalid" color (QColor's default
			// ctor creates an invalid color).
			QTadsIO::color(QColor());
			break;
		}
	} else {
		QTadsIO::color(fg);
	}
}
#endif


/* Set the screen background color.
 */
#ifndef HTMLQT
void
os_set_screen_color( os_color_t /*color*/ )
{
	Q_ASSERT(QTadsIO::t3Mode());

	qDebug("os_set_screen_color()");
}
#endif


/* --------------------------------------------------------------------
 * Filename manipulation routines.
 */

/* Extract the path from a filename.
 */
void
os_get_path_name( char* pathbuf, size_t pathbuflen, const char* fname )
{
#ifndef HTMLQT
	Q_ASSERT(QTadsIO::t3Mode());
#endif

	strncpy(pathbuf, QFileInfo(fname).dirPath().ascii(), pathbuflen);
	pathbuf[pathbuflen - 1] = '\0';
}


/* Convert a relative URL into a relative filename path.
 */
void
os_cvt_url_dir( char* result_buf, size_t result_buf_size, const char* src_url, int end_sep )
{
#ifndef HTMLQT
	Q_ASSERT(QTadsIO::t3Mode());
#endif

	QString res(src_url);

	if (end_sep == TRUE and not res.endsWith("/")) {
		res += '/';
	}

	strncpy(result_buf, res.ascii(), result_buf_size);
	result_buf[result_buf_size - 1] = '\0';
}


// This one is only needed by the HTML sources.
#ifdef HTMLQT
/* Determine whether a filename specifies an absolute or relative path.
 */
int
os_is_file_absolute( const char* fname )
{
	return not QFileInfo(fname).isRelative();
}
#endif


/* --------------------------------------------------------------------
 * Banner interface.
 *
 * TODO: Implement it as soon as possible!!!
 */

#ifndef HTMLQT
/* Create a banner window.
 */
void*
os_banner_create( void* /*parent*/, int /*where*/, void* /*other*/, int /*wintype*/, int /*align*/, int /*siz*/,
		  int /*siz_units*/, unsigned long /*style*/ )
{
	Q_ASSERT(QTadsIO::t3Mode());

	return 0;
}


/* Delete a banner.
 */
void
os_banner_delete( void* /*banner_handle*/ )
{
}


/* "Orphan" a banner.
 */
void
os_banner_orphan( void* /*banner_handle*/ )
{
}


/* Get information on the banner.
 */
int
os_banner_getinfo( void* /*banner_handle*/, os_banner_info_t* /*info*/ )
{
	return false;
}


/* Get the character width of the banner.
 */
int
os_banner_get_charwidth( void* /*banner_handle*/ )
{
	return 0;
}


/* Get the character height of the banner.
 */
int
os_banner_get_charheight( void* /*banner_handle*/ )
{
	return 0;
}


/* Display output on a banner.
 */
void
os_banner_disp( void* /*banner_handle*/, const char* /*txt*/, size_t /*len*/ )
{
}


/* Turn HTML mode on in the banner window.
 */
void
os_banner_start_html( void* /*banner_handle*/ )
{
}


/* Turn HTML mode off in the banner window.
 */
void
os_banner_end_html( void* /*banner_handle*/ )
{
}


/* Set the text color in a banner, for subsequent text displays.
 */
void
os_banner_set_color( void* /*banner_handle*/, os_color_t /*fg*/, os_color_t /*bg*/ )
{
}


/* Set the screen color in the banner.
 */
void
os_banner_set_screen_color( void* /*banner_handle*/, os_color_t /*color*/ )
{
}


/* Set the text attributes in a banner, for subsequent text displays.
 */
void
os_banner_set_attr( void* /*banner_handle*/, int /*attr*/ )
{
}


/* Flush output on a banner.
 */
void
os_banner_flush( void* /*banner_handle*/ )
{
}


/* Clear the contents of a banner.
 */
void
os_banner_clear( void* /*banner_handle*/ )
{
}


/* Set the banner's size.
 */
void
os_banner_set_size( void* /*banner_handle*/, int /*siz*/, int /*siz_units*/, int /*is_advisory*/ )
{
}


/* Set the banner to the size of its current contents.
 */
void
os_banner_size_to_contents( void* /*banner_handle*/ )
{
}


/* Set the output coordinates in a text grid window.
 */
void
os_banner_goto( void* /*banner_handle*/, int /*row*/, int /*col*/ )
{
}
#endif // HTMLQT
