/*
 * Functions dealing with the allow-list.
 *
 * Copyright 2005 Andrew Wood, distributed under the Artistic License.
 */

#include "config.h"
#include "spami.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>


/*
 * Return a malloc()ed pointer to a null-terminated string which is the
 * database key for the given allow-list email address.
 */
static char *spam_allowlist__token(char *address)
{
	char *ptr;

	if (address == NULL)
		return NULL;

	ptr = spam_checksum(address, strlen(address));

	if (ptr != NULL)
		ptr[0] = '?';

	return ptr;
}


/*
 * The same as spam_allowlist__token, but converts a copy of the address to
 * lower case first.
 */
static char *spam_allowlist__lctoken(char *address)
{
	char *addrcopy;
	char *key;
	int i;

	addrcopy = strdup(address);
	if (addrcopy == NULL)
		return NULL;

	for (i = 0; addrcopy[i] != 0; i++) {
		if (addrcopy[i] >= 'A' && addrcopy[i] <= 'Z')
			addrcopy[i] += 32;
	}

	key = spam_allowlist__token(addrcopy);

	free(addrcopy);

	return key;
}


/*
 * Look up the given email address in the allow-list, returning 1 if it is
 * present, 0 if not. First the address is checked in lowercase, then as-is.
 */
int spam_allowlist_match(spam_t spam, char *address)
{
	char *key;
	long a, b, c;

	if (spam == NULL)
		return 0;
	if (address == NULL)
		return 0;

	key = spam_allowlist__lctoken(address);
	if (key != NULL) {
		a = 0;
		b = 0;
		c = 0;
		spam_fetch(spam, key, strlen(key), &a, &b, &c);
		free(key);
		if (a > 0)
			return 1;
	}

	key = spam_allowlist__token(address);
	if (key != NULL) {
		a = 0;
		b = 0;
		c = 0;
		spam_fetch(spam, key, strlen(key), &a, &b, &c);
		free(key);
		if (a > 0)
			return 1;
	}

	return 0;
}


/*
 * Add the given email address to the allow-list. It is converted to lower
 * case before being stored.
 */
void spam_allowlist_add(opts_t opts, char *address)
{
	char *key;

	if (opts == NULL)
		return;
	if (address == NULL)
		return;

	key = spam_allowlist__lctoken(address);
	if (key == NULL)
		return;

	spam_store(opts, key, strlen(key), 1, 1, 0);

	free(key);
}


/*
 * Remove the given email address from the allow-list. Both as-is and
 * lower-case versions of the address will be removed.
 */
void spam_allowlist_remove(opts_t opts, char *address)
{
	char *key;
	qdb_datum qkey;

	if (opts == NULL)
		return;
	if (opts->dbw == NULL)
		return;
	if (address == NULL)
		return;

	key = spam_allowlist__token(address);
	if (key != NULL) {
		qkey.data = key;
		qkey.size = strlen(key);
		qdb_delete(opts->dbw, qkey);
		free(key);
	}

	key = spam_allowlist__lctoken(address);
	if (key != NULL) {
		qkey.data = key;
		qkey.size = strlen(key);
		qdb_delete(opts->dbw, qkey);
		free(key);
	}
}


/*
 * Manage the allow-list manually (-e).
 */
int spam_allowlist_manage(opts_t opts)
{
	int inlist;
	spam_t spam;

	if (opts->action == ACTION_MARK_SPAM) {
		spam_allowlist_remove(opts, opts->emailonly);
		return 0;
	} else if (opts->action == ACTION_MARK_NONSPAM) {
		spam_allowlist_add(opts, opts->emailonly);
		return 0;
	}

	spam = calloc(1, sizeof(*spam));
	if (spam == NULL) {
		fprintf(stderr, "%s: %s: %s\n", opts->program_name,
			_("calloc failed"), strerror(errno));
		return 1;
	}

	spam->db1 = opts->dbr1;
	spam->db2 = opts->dbr2;
	spam->db3 = opts->dbr3;
	spam->dbw = opts->dbw;

	spam->db1weight = 1;
	spam->db2weight = 1;
	spam->db3weight = 1;

	inlist = spam_allowlist_match(spam, opts->emailonly);

	free(spam);

	if (inlist) {
		if (opts->no_filter)
			return 0;
		printf("YES\n");
	} else {
		if (opts->no_filter)
			return 1;
		printf("NO\n");
	}

	return 0;
}

/* EOF */
