#!/usr/bin/env python
# -*- coding: ISO-8859-1 -*-

# Copyright (C) 2002, 2003 Jrg Lehmann <joerg@luga.de>
#
# This file is part of PyTone (http://www.luga.de/pytone/)
#
# PyTone is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2
# as published by the Free Software Foundation.
#
# PyTone is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with PyX; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import curses, os, os.path, sys

##############################################################################
# gettext initialization.
##############################################################################

# We have to do this very early, before importing our modules. We
# assume that the locales lie in same dir as this module. This may not
# be the case, if the .mo files are installed at their proper
# location.

try:
    import gettext
    locallocaledir = os.path.join(os.path.dirname(sys.argv[0]), "../locale")
    gettext.install("PyTone", locallocaledir)
except:
    # Disable localization if there is any problem with the above.
    # This works around a problem with Python 2.1
    import __builtin__
    __builtin__.__dict__['_'] = lambda s: s

##############################################################################
# process commandline options and read config file
##############################################################################

import config
# process the command line first, because a different location for the
# config file may be given there
config.processcommandline()
config.processconfig()

import mainscreen
import helper
import hub, events
import log
import services.songdb
import services.player

# Uncomment the next line, if you want to experience a little bit with
# the number of bytecode instructions after which a context switch
# occurs.

# sys.setcheckinterval(250)


##############################################################################
# create .pytone dir in user home
##############################################################################
try:
    os.mkdir(os.path.expanduser("~/.pytone"))
    log.info(_("created PyTone directory ~/.pytone"))
except OSError, e:
    if e.errno!=17:
        raise

##############################################################################
# start various services
##############################################################################


# song databases
songdbids = []
songdbid = services.songdb.initsongdb("main", config.database.main)
if songdbid:
    songdbids.append(songdbid)
songdbid = services.songdb.initsongdb("secondary", config.database.secondary)
if songdbid:
    songdbids.append(songdbid)

# network service
if config.network.enableserver:
    import network
    network.tcpserver("localhost", config.network.port).start()
if config.network.socketfile:
    import network
    network.unixserver(os.path.expanduser(config.network.socketfile)).start()

# Now that everything is setup, we can initialize the players. This has
# to be done after all other services have been setup because
# the players immediately start requesting a new song
playerids = [services.player.initplayer("main", config.player.main),
             services.player.initplayer("secondary", config.player.secondary)]

##############################################################################
# basic curses library setup...
##############################################################################

def cursessetup():
    # Initialize curses library
    stdscr = curses.initscr()

    # Turn off echoing of keys
    curses.noecho()

    # In keypad mode, escape sequences for special keys
    # (like the cursor keys) will be interpreted and
    # a special value like curses.KEY_LEFT will be returned
    stdscr.keypad(1)

    # allow 8-bit characters to be input
    curses.meta(1)

    # enter raw mode, thus disabling interrupt, quit, suspend and flow-control keys
    curses.raw()

    # wait at maximum for 1/10th of seconds for keys pressed by user
    curses.halfdelay(1)

    if config.general.colorsupport == "auto":
        # Try to enable color support
        try:
            curses.start_color()
        except:
            log.warning("terminal does not support colors: disabling color support")

        # now check whether color support really has been enabled
        if curses.has_colors():
            config.configcolor._colorenabled = 1
    elif config.general.colorsupport == "on":
        curses.start_color()
        config.configcolor._colorenabled = 1

    # Check for transparency support of terminal
    # use_default_colors(), which will be integrated in python 2.4.
    # Before that happens we try to use our own cursext c-extension
    try:
        curses.use_default_colors()
        config.configcolor._colors["default"] = -1
    except:
        try:
            import cursext
            if cursext.useDefaultColors():
                config.configcolor._colors["default"] = -1
            else:
                log.warning("terminal does not support transparency")
        except:
            log.warning("transparency support disabled because cursext module is not present")

    # try disabling cursor
    try:
        curses.curs_set(0)
    except:
        log.warning("terminal does not support disabling of cursor")

    # enable all mouse events
    curses.mousemask(curses.ALL_MOUSE_EVENTS)

    return stdscr

##############################################################################
# ... and cleanup
##############################################################################

def cursescleanup():
    # restore terminal settings
    try:
        stdscr.keypad(0)
        curses.echo()
        curses.nocbreak()
        curses.endwin()
    except: 
        pass

##############################################################################
# setup main screen (safety wrapped)
##############################################################################

try:
    stdscr = cursessetup()
    # set m to None as marker in case that something goes wrong in the
    # mainscreen.mainscreen constructor
    m = None
    m = mainscreen.mainscreen(stdscr, songdbids, playerids)
    m.run()
except:
    cursescleanup()
    
    # shutdown all other threads
    hub.hub.notify(events.quit(), 100)
    
    helper.print_exc_plus()
    raise
else:
    cursescleanup()

    # shutdown all other threads
    hub.hub.notify(events.quit(), 100)
