# -*- coding: ISO-8859-1 -*-

# Copyright (C) 2002, 2003 Jrg Lehmann <joerg@luga.de>
#
# This file is part of PyTone (http://www.luga.de/pytone/)
#
# PyTone is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2
# as published by the Free Software Foundation.
#
# PyTone is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with PyX; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import config
import log
import window
import events, hub
import statusbar

from helper import formattime

class playerwin(window.window):
    def __init__(self, screen, layout, channel, playerid):
        self.song = None
        self.time = 0
        self.paused = 0
        self.stopped = 1
        self.playerid = playerid
        self.keybindings = config.keybindings.general
        self.songformat = config.playerwindow.songformat
        self.playerinfofile = config.general.playerinfofile
        h, w, y, x, border = layout
        window.window.__init__(self, screen, h, w, y, x,
                               config.colors.playerwindow,
                               _("Playback Info"), border)

        # we don't want to have the focus
        self.panel.bottom()

        channel.subscribe(events.playbackinfochanged, self.playbackinfochanged)
        channel.subscribe(events.keypressed, self.keypressed)
        self.update()

    def resize(self, layout):
        h, w, y, x, self.border = layout
        window.window.resize(self, h, w, y, x)

    def updatestatusbar(self):
        if self.song and not self.paused and self.keybindings["playerpause"]:
            sbar = statusbar.generatedescription("general", "playerpause")
        else:
            sbar = statusbar.generatedescription("general", "playerstart")
        hub.hub.notify(events.updatestatusbar(1, sbar))
        
    def update(self):
        window.window.update(self)
        self.updatestatusbar()
        self.addstr(1, self.ix, " "*self.iw)
        if self.song:
            self.move(1, self.ix)
            s1 = _("Time:")
            s2 = " %s/%s " % (formattime(self.time), formattime(self.song.length))
            self.addstr(s1, self.colors.description)
            self.addstr(s2, self.colors.content)

            if not self.paused:
                barlen = self.iw-len(s1)-len(s2)
                try:
                    percentplayed = int(barlen*self.time/self.song.length)
                except ZeroDivisionError:
                    percentplayed = 0
                self.addstr("#"*(percentplayed), self.colors.progressbarhigh)
                self.addstr("#"*(barlen-percentplayed), self.colors.progressbar)
            else:
                self.addstr(_("paused"), self.colors.description)

    # event handler

    def playbackinfochanged(self, event):
        if event.playbackinfo.playerid == self.playerid:
            self.song = event.playbackinfo.song
            self.paused = event.playbackinfo.ispaused()
            self.stopped = event.playbackinfo.isstopped()
            if self.song:
                self.settitle("%s%s" % (event.playbackinfo.iscrossfading() and "-> " or "",
                                             self.song.format(self.songformat)))
            else:
                self.settitle(_("Playback Info"))
            self.time = event.playbackinfo.time
            self.update()

            # update player info file, if configured
            if self.playerinfofile:
                try:
                    f = open(config.general.playerinfofile, "w")
                    if self.song:
                        f.write("%s - %s (%s/%s)\n"  % ( self.song.artist,
                                                         self.song.title,
                                                         formattime(self.time),
                                                         formattime(self.song.length)))
                    f.close()
                except IOError, e:
                    log.error(_("error '%s' occured during write to playerinfofile") % e)
                    self.playerinfofile = None
                    

    def keypressed(self, event):
        key = event.key
        if key in self.keybindings["playerstart"] and self.paused:
            hub.hub.notify(events.playerstart(self.playerid))
        elif key in self.keybindings["playerpause"] and not self.paused and not self.stopped:
            hub.hub.notify(events.playerpause(self.playerid))
        elif key in self.keybindings["playerstart"]:
            hub.hub.notify(events.playerstart(self.playerid))
        elif key in self.keybindings["playernextsong"]:
            hub.hub.notify(events.playerforward(self.playerid))   
        elif key in self.keybindings["playerstop"]:
            hub.hub.notify(events.playerstop(self.playerid))
        elif ord("1")<=key-1024<=ord("5") and self.song:
            self.song.rate(key-1024-ord("1")+1)
        else:
            return
        raise hub.TerminateEventProcessing

