# -*- coding: ISO-8859-1 -*-

# Copyright (C) 2002, 2003 Jrg Lehmann <joerg@luga.de>
# 
# Ogg Vorbis decoder interface by Byron Ellacott <bje@apnic.net>.
#
# This file is part of PyTone (http://www.luga.de/pytone/)
#
# PyTone is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2
# as published by the Free Software Foundation.
#
# PyTone is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with PyX; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import mad
try:
    import ogg.vorbis
except:
    pass
import pcm

# two simple adapter classes which give a unified interface to the ogg and mp3
# decoding classes

class decodedmp3file:
    def __init__(self, path):
        self.file = mad.MadFile(path)

    def samplerate(self):
        return self.file.samplerate()

    def ttime(self):
        return self.file.total_time()/1000

    def read(self):
        return self.file.read()

        
class decodedoggfile:
    def __init__(self, path):
        self.file = ogg.vorbis.VorbisFile(path)

    def samplerate(self):
        return self.file.info().rate

    def ttime(self):
        return self.file.time_total(0)

    def read(self):
        buff, bytes, bit = self.file.read()
        return buffer(buff, 0, bytes)

#
# main class
#

class decodedsong:

    """ song decoder and rate converter

    This class is for decoding of a song and the conversion of the
    resulting pcm stream to a defined sample rate. Besides the
    constructor, there is only one method, namely read, which
    returns a pcm frame of or less than a given arbitrary size.

    """
    
    def __init__(self, song, outrate):
        self.song = song
        self.outrate = outrate


        if self.song.type=="mp3":
            self.decodedfile = decodedmp3file(song.path)
        elif self.song.type=="ogg":
            try:
                self.decodedfile = decodedoggfile(song.path)
            except NameError:
                raise RuntimeError("Ogg/Vorbis support not present")
        else:
            raise RuntimeError("Song type '%s' unknown" % song.type)

        # Use the total time given by the decoder library and not the one
        # stored in the database. The former one turns out to be more precise
        # for some VBR songs.
        self.ttime = self.decodedfile.ttime()
        self.samplerate = self.decodedfile.samplerate()

        newbuff = self.decodedfile.read()

        if newbuff:
            self.buff, self.last_l, self.last_r = \
                       pcm.rate_convert(newbuff,
                                        self.samplerate,
                                        None,
                                        None,
                                        self.outrate,
                                        None,
                                        None)
        else:
            self.buff = []
            self.last_l = None
            self.last_r = None
        self.buffpos = 0
        self.ptime = 0
        
    def read(self, size):
        bytesleft = len(self.buff) - self.buffpos

        # fill buffer, if necessary 
        while bytesleft<size:
            newbuff = self.decodedfile.read()
            if newbuff:
                self.buff, self.last_l, self.last_r = \
                           pcm.rate_convert(newbuff,
                                            self.samplerate,
                                            self.buff,
                                            self.buffpos,
                                            self.outrate,
                                            self.last_l,
                                            self.last_r)

                # the new self.buff contains only new data
                self.buffpos = 0
                bytesleft = len(self.buff)
            else:
                size = bytesleft
                break

        oldpos = self.buffpos
        self.buffpos += size
        if self.samplerate:
            self.ptime += 1.0/4*size/self.samplerate
        
        return self.buff[oldpos:self.buffpos]
