#
# Copyright (c) 2002, 2004, 2005 Art Haas
#
# This file is part of PythonCAD.
# 
# PythonCAD is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# PythonCAD is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with PythonCAD; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
# a class for storing selected objects
#

from PythonCAD.Generic import image
from PythonCAD.Generic import layer

class Selection(object):
    """A class used for storing selected objects.

A Selection object has the following methods:

storeObject(): Store a reference to some object.
getObjects(): Return all the objects held within the Selection object.
reset(): Empty the Selection object.

A Selection can have any number of objects added to it, and the
contents of a Selection can be retrieved any number of times.
Once the Selection has had its contents retrieved, however,
the addition of a new object will purge the current contents
of the selection.
    """
    def __init__(self):
        """Initialize a Selection object.

There are no arguments needed for this method.        
        """
        self.__image = None
        self.__layers = {}
        self.__retrieved = False

    def hasObjects(self):
        """Test if there are objects stored in the Selection.

hasObjects()        
        """
        _val = False
        if len(self.__layers):
            for _lid in self.__layers:
                if len(self.__layers[_lid]) > 0:
                    _val = True
                    break
        return _val
    
    def storeObject(self, img, lyr, obj):
        """Store a reference to an object.

storeObject(img, lyr, obj)

The arguments are:

img: An Image object.
lyr: A Layer object.
obj: An object in the drawing (Point, Segment, etc.)
        """
        if not isinstance(img, image.Image):
            raise TypeError, "Invalid image object: " + `img`
        if not isinstance(lyr, layer.Layer):
            raise TypeError, "Invalid layer object: " + `lyr`
        if lyr.getParent() is not img:
            raise ValueError, "Layer not found in Image"
        if (self.__retrieved or
            (self.__image is not None and img is not self.__image)):
            self.reset()
        self.__image = img
        _objlist = self.__layers.setdefault(id(lyr), [])
        _seen = False
        for _i in range(len(_objlist)):
            if obj is _objlist[_i]:
                _seen = True
                break
        if not _seen:
            _objlist.append(obj)
        
    def getObjects(self):
        """Return all the currently selected objects.

getObjects()

This method returns a list.
        """
        _allobjs = []
        for _lid in self.__layers:
            _allobjs.extend(self.__layers[_lid][:])
        self.__retrieved = True
        return _allobjs
        
    def reset(self):
        """Reset the Selection object to empty.

reset()        
        """
        self.__image = None
        self.__layers.clear()
        self.__retrieved = False
