// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.
#include "xgl.h"

#include <stdio.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include <algorithm>
#include <sstream>

namespace visual {


template<typename T>
inline T
clamp( T x, T a, T b)
{
	if (b<a) return std::max( b, std::min( x, a));
	else return std::max( a, std::min( x, b));
}

/**************** xglFont implementation *******************/


xglFont::xglFont(struct xglContext& _cx, 
		 const char *name, 
		 double size) 
	: cx(_cx), refcount(1)
{
	font = gdk_font_load(name);
	if (!font) {
		font = gdk_font_load("-misc-fixed-medium-r-*-*-*-120-*-*-*-*-*-*");
	}
	if (!font) {
		return;
	}
	gdk_font_ref(font);

	cx.makeCurrent();
	listBase = glGenLists(256);
	gdk_gl_use_gdk_font(font, 0, 255, listBase);
	cx.makeNotCurrent();
}

xglFont::~xglFont()
{
}

void
xglFont::draw(const char *c)
{
	if (font) {
		glListBase(listBase);
		glCallLists(strlen(c), GL_UNSIGNED_BYTE, c);
	}
}

double
xglFont::getWidth(const char *c)
{
	int width, ascent, descent, lbearing, rbearing;
	gdk_string_extents(font, c, &lbearing, &rbearing, &width, &ascent, &descent);
	return double(width*2) / cx.width();
}

double
xglFont::ascent()
{
	if (font)
		return double(font->ascent) / cx.height();
	else
		return 0;
}

double
xglFont::descent()
{
	if (font)
		return double(font->descent) / cx.height();
	else
		return 0;
}

void
xglFont::release()
{
	refcount--;
	if (!refcount) {
		gdk_font_unref(font);
		cx.add_pending_glDeleteList( listBase, 256);
		delete(this);
	}
}

glFont*
xglContext::getFont(const char* description, double size)
{
	return new xglFont(*this, description, size);
}


/*************** xglContext implementation *************/

void
xglContext::add_pending_glDeleteList(int base, int howmany)
{
	Cache::write_lock L(list_lock);
	pending_glDeleteLists.push_back( std::make_pair(base, howmany));	
}

xglContext::xglContext() 
 : window(NULL),
   area(NULL),
   wwidth(0), wheight(0),
   buttonState(0),
   buttonsChanged(0),
   Kshift(0), Kalt(0), Kctrl(0),
   mouseLocked(false)
{
}

xglContext::~xglContext()
{
	cleanup();
}

// GNOME versions before 2 don't allow you to create full
// screen windows directly, but we can do it through GDK
// when the underlying X Window is realized.

void 
xglContext::realize_cb( GtkWidget* widget, gpointer data)
{
	GdkWindow * win = GTK_WIDGET(widget)->window;
	
	gdk_window_set_override_redirect( win, TRUE);
	gdk_window_move( win, 0, 0);
	gdk_window_resize( win, gdk_screen_width(), gdk_screen_height());
}


gint
xglContext::configure_cb(GtkWidget *widget,
		GdkEventConfigure *event,
		gpointer data)
{
	xglContext *c = (xglContext *)data;
	c->wwidth = event->width;
	c->wheight = event->height;

	return TRUE;
}

gint
xglContext::motion_notify_cb(GtkWidget *widget,
		GdkEventMotion *event,
		gpointer data)
{
	xglContext *c = (xglContext *)data;
	c->mousePos.set_x( event->x);
	c->mousePos.set_y( event->y);

	c->Kshift = (event->state & GDK_SHIFT_MASK) || (event->state & GDK_LOCK_MASK);
	c->Kctrl  = event->state & GDK_CONTROL_MASK;
	c->Kalt   = event->state & GDK_MOD1_MASK;

	return TRUE;
}

gint
xglContext::delete_cb(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	xglContext *c = (xglContext *)data;
	c->cleanup();
	return TRUE;
}

gint
xglContext::button_press_cb(GtkWidget *widget, GdkEventButton *event,
                                 gpointer data) 
{
	xglContext *c = (xglContext *)data;

	// event.button may be 1-5. 1=left, 2=middle, 3=right (normally)
	if (event->button == 1) {
		c->buttonState |= 1;
		c->buttonsChanged |= 1;
	}
	else if (event->button == 2) {
		c->buttonState |= (1 << 2);
		c->buttonsChanged |= (1 << 2);
	}
	else if (event->button == 3) {
		c->buttonState |= (1 << 1);
		c->buttonsChanged |= (1 << 1);
	}
	
	c->Kshift = (event->state & GDK_SHIFT_MASK) || (event->state & GDK_LOCK_MASK);
	c->Kctrl  = event->state & GDK_CONTROL_MASK;
	c->Kalt   = event->state & GDK_MOD1_MASK;

	return TRUE;
}

gint
xglContext::button_release_cb(GtkWidget *widget, GdkEventButton *event,
                                 gpointer data) 
{
	xglContext *c = (xglContext *)data;

	// event.button may be 1-5. 1=left, 2=middle, 3=right (normally)
	// preserve all of buttonstate, but remove the flag for the button being 
	// released.
	if (event->button == 1) {
		c->buttonState &= ~1;
		c->buttonsChanged |= 1;
	}
	else if (event->button == 2) {
		c->buttonState &= ~(1 << 2);
		c->buttonsChanged |= (1 << 2);
	}
	else if (event->button == 3) {
		c->buttonState &= ~(1 << 1);
		c->buttonsChanged |= (1 << 1);
	}
	
	c->Kshift = (event->state & GDK_SHIFT_MASK) || (event->state & GDK_LOCK_MASK);
	c->Kctrl  = event->state & GDK_CONTROL_MASK;
	c->Kalt   = event->state & GDK_MOD1_MASK;

	return TRUE;
}

gint
xglContext::key_press_cb(GtkWidget *window, GdkEventKey *key, gpointer d)
{
	// self is the 'this' pointer.
	xglContext* self = (xglContext*)d;
	
	// Note that this algorithm will proably fail if the user is using anything 
	// other than a US keyboard.
	std::string ctrl_str;
	// First trap for shift, ctrl, and alt.
	if ((key->state & GDK_SHIFT_MASK) || (key->state & GDK_LOCK_MASK)) {
		ctrl_str += "shift+";
	}
	else if (key->state & GDK_CONTROL_MASK) {
		ctrl_str += "ctrl+";
	}
	else if (key->state & GDK_MOD1_MASK) {
		ctrl_str += "alt+";
	}
  
  
	// Specials, try to match those in wgl.cpp
	guint k = key->keyval;
	std::string key_str;
	switch (k) {
		case GDK_F1:
		case GDK_F2:
		case GDK_F3:
		case GDK_F4:
		case GDK_F5:
		case GDK_F6:
		case GDK_F7:
		case GDK_F8:
		case GDK_F9:
		case GDK_F10:
		case GDK_F11:
		case GDK_F12: {
			// Use braces to destroy s.
			std::ostringstream s;
			s << key_str << 'f' << k-GDK_F1 + 1;
			key_str = s.str();
		}   break;
		case GDK_Page_Up:
			key_str += "page up";
			break;
		case GDK_Page_Down:
			key_str += "page down";
			break;
		case GDK_End:
			key_str += "end";
			break;
		case GDK_Home:
			key_str += "home";
			break;
		case GDK_Left:
			key_str += "left";
			break;
		case GDK_Up:
			key_str += "up";
			break;
		case GDK_Right:
			key_str += "right";
			break;
		case GDK_Down:
			key_str += "down";
			break;	
		case GDK_Print:
			key_str += "print screen";
			break;
		case GDK_Insert:
			key_str += "insert";
			break;
		case GDK_Delete:
			key_str += "delete";
			break;
		case GDK_Num_Lock:
			key_str += "numlock";
			break;
		case GDK_Scroll_Lock:
			key_str += "scrlock";
			break;
		case GDK_BackSpace:
			key_str += "backspace";
			break;
		case GDK_Tab:
			key_str += "\t";
			break;
		case GDK_Return:
			key_str += "\n";
			break;
		case GDK_Escape:
			// Allow the user to delete a fullscreen window this way
			self->cleanup();
			return FALSE;
	}
  
	if (!key_str.empty()) {
		// A special key.
		ctrl_str += key_str;
		self->keys.push( ctrl_str);
	}
	else if ( isprint(k) && (ctrl_str == "ctrl+" || ctrl_str == "alt+")) {
		// A control character
		ctrl_str += static_cast<char>( k);
		self->keys.push(ctrl_str);
	}
	else if ( strlen(key->string) && isprint( key->string[0])) {
		// Anything else.
		self->keys.push( std::string( key->string));
	}
	return TRUE;
}

gint
xglContext::key_release_cb(GtkWidget *window, GdkEventKey *key, gpointer d)
{

	return TRUE;
}


int
xglContext::getShiftKey()
{
	return Kshift;
}

int
xglContext::getAltKey()
{
	return Kalt;
}

int
xglContext::getCtrlKey()
{
	return Kctrl;
}


bool
xglContext::changeWindow(const char* title, int x, int y, int width, int height, int flags)
{
	return false;
}

bool
xglContext::initWindow(const char* title, int x, int y, int width, int height, int flags)
{
	cleanup();

	window = gtk_window_new( GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title( GTK_WINDOW(window), title);
	gtk_container_set_border_width( GTK_CONTAINER(window), 0);

	if (flags & glContext::FULLSCREEN) {
		width  = gdk_screen_width();
		height = gdk_screen_height();
		gtk_signal_connect( GTK_OBJECT(window), "realize",
                     GTK_SIGNAL_FUNC(&xglContext::realize_cb),
                     (gpointer)this);
	}  

	int attrlist[] = {
		GDK_GL_RGBA,
		GDK_GL_DOUBLEBUFFER,
		GDK_GL_DEPTH_SIZE, 12,
		GDK_GL_NONE, // May set this value to something useful farther down.
		GDK_GL_NONE
	};
	
	if (flags & glContext::QB_STEREO) {
		// Try to set stereo
		attrlist[4] = GDK_GL_STEREO;
	}
	area = GTK_WIDGET( gtk_gl_area_new(attrlist));
	if (!area && flags & glContext::QB_STEREO) {
		// You failed, try to use simple doublebuffering.
		attrlist[4] = GDK_GL_NONE;
		area = GTK_WIDGET( gtk_gl_area_new(attrlist));
	}
	if (!area) {
		error_message = "Unable to create OpenGL display widget";
		cleanup();
		return false;
	}


	gtk_container_add(GTK_CONTAINER (window), area);

	gtk_widget_set_events( GTK_WIDGET (area),
	                       GDK_EXPOSURE_MASK 
	                       | GDK_BUTTON_PRESS_MASK
	                       | GDK_BUTTON_RELEASE_MASK
	                       | GDK_POINTER_MOTION_MASK);

	gtk_signal_connect( GTK_OBJECT(window), "configure_event",
	                    GTK_SIGNAL_FUNC(&xglContext::configure_cb),
	                    (gpointer)this);
	gtk_signal_connect( GTK_OBJECT(area), "motion_notify_event",
	                    GTK_SIGNAL_FUNC(&xglContext::motion_notify_cb),
	                    (gpointer)this);
	gtk_signal_connect( GTK_OBJECT(window), "delete_event",
	                    GTK_SIGNAL_FUNC(&xglContext::delete_cb), (gpointer)this);
	gtk_signal_connect( GTK_OBJECT(area), "button_press_event",
	                    GTK_SIGNAL_FUNC(&xglContext::button_press_cb),
	                    (gpointer)this);
	gtk_signal_connect( GTK_OBJECT(area), "button_release_event",
	                    GTK_SIGNAL_FUNC(&xglContext::button_release_cb),
	                    (gpointer)this);
	gtk_signal_connect( GTK_OBJECT(window), "key_press_event",
	                    GTK_SIGNAL_FUNC(&xglContext::key_press_cb),
	                    (gpointer)this);
	gtk_signal_connect( GTK_OBJECT(window), "key_release_event",
	                    GTK_SIGNAL_FUNC(&xglContext::key_release_cb),
	                    (gpointer)this);

	gtk_gl_area_size( GTK_GL_AREA(area), width, height);
	wwidth = width;
	wheight = height;

	gtk_widget_show_all( window);
	
	#if GTK_MAJOR_VERSION >= 2
	// Perform a window move request, only available since Gtk+ 2.0
	if (x >= 0 || y >= 0) {
		// Accept the defaults allocated by the window manager when none
		// was requested by the user.
		int init_x = 0, init_y = 0;
		gtk_window_get_position( GTK_WINDOW(window), &init_x, &init_y);
		if (x < 0)
			x = init_x;
		if (y < 0)
			y = init_y;
		
		// Position trumps dimentions  TODO: Find out why this doesn't work.
		// it could just be the WM, but the size set functions are not being
		// honored.
		int screen_width = gdk_screen_width();
		int screen_height = gdk_screen_height();
		int init_height = 0;
		int init_width = 0;
		gtk_window_get_size( GTK_WINDOW(window), &init_width, &init_height);
		if (x + init_width > screen_width) {
			gtk_gl_area_size( GTK_GL_AREA(area), screen_width - x, init_height);
		}
		if (y + init_height > screen_height) {
			gtk_gl_area_size( GTK_GL_AREA(area), init_width, screen_height - y);
		}
		
		// Move the window.
		gtk_window_move( GTK_WINDOW(window), x, y);
	}
	#endif
	
	return true;
}

bool
xglContext::isOpen()
{
	return window != NULL;
}

void
xglContext::cleanup()
{
	if (area) {
		gtk_widget_destroy( area);
		area = NULL;
	}
	if (window) {
		gtk_widget_destroy( window);
		window = NULL;
	}
	wwidth = 0;
	wheight = 0;
}

void
xglContext::lockMouse()
{
}

void
xglContext::unlockMouse()
{
}

vector
xglContext::getMousePos()
{
	if (!window)
		return vector(0,0,0);

	vector tmp = mousePos;
	tmp.x /= wwidth;
	tmp.y /= wheight;

	return tmp;
}

vector
xglContext::getMouseDelta()
{
	// GL units (% of window)
	vector tmp = mousePos - oldMousePos;
	oldMousePos = mousePos;

	return tmp;
}

int
xglContext::getMouseButtons()
{
	return buttonState;
}

int
xglContext::getMouseButtonsChanged()
{
	int c = buttonsChanged;
	buttonsChanged = 0;
	return c; 
}

std::string
xglContext::getKeys()
{
	if (!keys.empty()) {
		std::string s = keys.front();
		keys.pop();
		return s;
	} 
	return std::string("");
}

void
xglContext::makeCurrent()
{
	gtk_gl_area_make_current( GTK_GL_AREA (area));
	Cache::write_lock L( list_lock);
	for (std::vector<std::pair<int, int> >::iterator i = pending_glDeleteLists.begin();
		i != pending_glDeleteLists.end(); ++i) {
		glDeleteLists(i->first, i->second);	
	}
	pending_glDeleteLists.clear();
}

void
xglContext::makeNotCurrent()
{
}

void
xglContext::swapBuffers()
{
	gtk_gl_area_swapbuffers (GTK_GL_AREA (area));
}

vector
xglContext::origin()
{
  /* FIXME */
	return vector(0,0,0);
}

vector
xglContext::corner()
{
	return origin() + vector(wwidth,wheight);
}

int
xglContext::width()
{
	return wwidth;
}

int
xglContext::height()
{
	return wheight;
}

} // !namespace visual
