// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.
#include "wgl.h"

#include <process.h>

#ifdef _MSC_VER
// MSVC lacks max and min in <algorithm>
#define clamp(x, a, b) (__max(a, __min(x,b) ))

#else // But MinGW has them.
#include <algorithm>

namespace visual {

template<typename T>
inline T
clamp( T x, T a, T b)
{ 
	if (b<a) return std::max( b, std::min( x, a));
	else return std::max( a, std::min( x, b)); 
}

} // !namespace visual
#endif // !_MSC_VER

namespace visual
{

/************* wglFont implementation **************/

wglFont::wglFont(wglContext& _cx, HFONT _font)
	: cx(_cx), hfont(_font), refcount(1)
{
	if (hfont) {
		cx.makeCurrent();
		
		SelectObject(cx.hdc, hfont);
		
		TEXTMETRIC tm;
		GetTextMetrics(cx.hdc, &tm);
		lu_ascent = tm.tmAscent;
		lu_descent = tm.tmDescent;
		
		listBase = glGenLists(256);
		wglUseFontBitmaps(cx.hdc, 0, 255, listBase);
		
		cx.makeNotCurrent();
	}
}

wglFont::~wglFont()
{
}

void wglFont::draw(const char* string) 
{
	if (hfont) {
		glListBase( listBase );
		glCallLists( strlen(string), GL_UNSIGNED_BYTE, string );
	}
}

double wglFont::getWidth(const char *string) 
{
	SelectObject(cx.hdc, hfont);
	SIZE size;
	GetTextExtentPoint32(cx.hdc, string, strlen(string), &size);
	return double(size.cx * 2) / cx.width();
	//return 0.1;
}

double wglFont::ascent() 
{
	return double(lu_ascent*2) / cx.height();
}

double wglFont::descent() 
{
	return double(lu_descent*2) / cx.height();
}

void wglFont::release() 
{
	if (!--refcount) {
		if (hfont) {
			DeleteObject(hfont);
			cx.add_pending_glDeleteList( listBase, 256);
		}
		delete(this);
	}
}

glFont* wglContext::getFont(const char* description, double size) 
{
	if (!description || !*description) description="";
	
	std::pair<std::string, double> key(description, size);
	wglFont* f = fontCache[key];
	if (f) {
		f->addref();
		return f;
	}
	
	if (size) {
		double points = 72.0 * size / GetDeviceCaps(hdc, LOGPIXELSY);
		HFONT wf = CreateFont(-size,0,
			0,0,
			0,
			0,0,0,
			DEFAULT_CHARSET,
			OUT_DEFAULT_PRECIS,
			CLIP_DEFAULT_PRECIS,
			PROOF_QUALITY, 
			VARIABLE_PITCH | FF_SWISS, 
			description ? description : 0);
		f = new wglFont(*this, wf);
	} 
	else {
		f = new wglFont(*this, (HFONT)GetStockObject( SYSTEM_FONT ) );
	}
	
	f->addref();  // xxx font will never be destroyed
	fontCache[key] = f;
	return f;
}

/*************** wglWindowClass interface **************/

void
wglContext::add_pending_glDeleteList(int base, int howmany)
{
	Cache::write_lock L(list_lock);
	pending_glDeleteLists.push_back( std::make_pair(base, howmany));	
}


struct wglWindowClass 
{
	wglWindowClass();
	
	void setContext( wglContext& cx );
	void clearContext( HWND hWnd );
	
	const char* name() { return wc.lpszClassName; }
	
	static wglWindowClass& instance() 
	{
		static wglWindowClass* inst = new wglWindowClass;
		return *inst;
	}
	
private:
	
	WNDCLASS wc;
	static std::map< HWND, wglContext* > contexts;
	
	static LRESULT WINAPI WindowProc( HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam );
};

/************* wglWindowClass implementation **************/

std::map< HWND, wglContext* > wglWindowClass::contexts;

void wglWindowClass::setContext( wglContext& cx ) 
{
	contexts[cx.hWnd] = &cx;
}

void wglWindowClass::clearContext( HWND hWnd ) 
{
	contexts[hWnd] = 0;
}

LRESULT WINAPI wglWindowClass::WindowProc( HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam ) 
{
	wglContext* cx = contexts[hWnd];
	if (cx) 
		return cx->winMessage( hWnd, msg, wParam, lParam );
	
	return DefWindowProc(hWnd, msg, wParam, lParam);
}

wglWindowClass::wglWindowClass() 
{
	memset(&wc,0,sizeof(wc));
	
	wc.lpszClassName = "wglWindowC";
	wc.lpfnWndProc = WindowProc;
	wc.style = CS_OWNDC | CS_VREDRAW | CS_HREDRAW;
	wc.hInstance = GetModuleHandle(0);
	wc.hIcon = LoadIcon( NULL, IDI_APPLICATION );
	wc.hCursor = LoadCursor( NULL, IDC_ARROW );
	wc.hbrBackground = 0;
	wc.lpszMenuName = 0;
	wc.cbClsExtra = 0;
	wc.cbWndExtra = 0;
	
	RegisterClass(&wc);
}

/*************** wglContext implementation *************/

wglContext::wglContext() 
	: cls( wglWindowClass::instance() ),
	hWnd( 0 ),
	hdc( 0 ),
	hglrc( 0 ),
	lasthdc( 0 ),
	lastglrc( 0 ),
	mouseLocked( 0 ),
	buttonState( 0 ),
	buttonChanged( 0 ),
	mouseDelta(0,0,0),
	mousePos(0,0,0),
	current( 0 ),
	Kshift( 0 ),
	Kalt( 0 ),
	Kctrl( 0 )
{
}

wglContext::~wglContext() 
{
	cleanup();
}

void wglContext::error( std::string where) 
{
	unsigned errorcode = GetLastError();
	
	
	char s[1024];
	FormatMessage( FORMAT_MESSAGE_FROM_SYSTEM | 80,  /* flags | max_width */
		0,           /* source */
		errorcode,   /* messageId */
		0,           /* language */
		s,           /* buffer */
		sizeof(s)-1, /* buffer_size */
		0            /* arguments */
		);
	
	error_message = where + ": " + s;
}

bool wglContext::changeWindow(const char* title, int x, int y, int width, int height, int flags) 
{
	if (!hWnd) return false;
	
	vector org = origin();
	
	if ((x>=0 || y>=0 || width>=0 || height>=0) &&
		!MoveWindow( hWnd, x<0 ? org.x : x, y<0 ? org.y : y, width, height, TRUE))
		return false;
	
	if (title && !SetWindowText(hWnd, title))
		return false;
	
	return true;
}

bool wglContext::initWindow(const char* title, int x, int y, int width, int height, int flags) 
{
	cleanup();
	
	RECT screen;
	if (!SystemParametersInfo(SPI_GETWORKAREA, 0, &screen, 0))
		return false;
	
	if (flags & glContext::FULLSCREEN) {
		x = screen.left;
		y = screen.top;
		width  = screen.right - x;
		height = screen.bottom - y;
	} 
	else if (x<0 && y<0 || x>screen.right || y>screen.bottom) {
		x = CW_USEDEFAULT;
		y = CW_USEDEFAULT;
	} 
	else if (x<screen.left) {
		x = screen.left;
	} 
	else if (y<screen.top) {
		y = screen.top;
	}
	
	if (x + width > screen.right) 
		width = screen.right - x;
	if (y + height > screen.bottom) 
		height = screen.bottom - y;
	
	int style;
	if (flags & glContext::FULLSCREEN)
		style = WS_OVERLAPPED | WS_POPUP | WS_MAXIMIZE | WS_VISIBLE;
	else
		style = WS_OVERLAPPEDWINDOW | WS_CLIPCHILDREN | WS_CLIPSIBLINGS;
	
	hWnd = CreateWindow( cls.name(), 
		title, style,
		x, y, width,height,
		NULL, NULL, GetModuleHandle(0), NULL );
	
	if (!hWnd) { error("CreateWindow"); cleanup(); return false; }
	
	cls.setContext( *this );
	
	ShowWindow(hWnd, SW_SHOW);
	
	hdc = GetDC(hWnd);
	
	if (!hdc) { error("GetDC");cleanup(); return false; }
	
	DWORD pfdFormat = PFD_DOUBLEBUFFER;
	if (flags & glContext::QB_STEREO)
		pfdFormat |= PFD_STEREO;
		
	PIXELFORMATDESCRIPTOR pfd = { 
		sizeof(PIXELFORMATDESCRIPTOR),   // size of this pfd 
			1,                               // version number 
			PFD_DRAW_TO_WINDOW |             // output to screen (not an image) 
			PFD_SUPPORT_OPENGL |             // support OpenGL 
			pfdFormat,                       // double buffered or quad buffered
			PFD_TYPE_RGBA,                   // RGBA type 
			24,                              // 24-bit color depth 
			0, 0, 0, 0, 0, 0,                // color bits ignored 
			0,                               // no alpha buffer 
			0,                               // shift bit ignored 
			0,                               // no accumulation buffer 
			0, 0, 0, 0,                      // accum bits ignored 
			32,                              // 32-bit z-buffer 
			0,                               // no stencil buffer 
			0,                               // no auxiliary buffer 
			PFD_MAIN_PLANE,                  // main layer 
			0,                               // reserved 
			0, 0, 0                          // layer masks ignored 
	};
	
	int pixelFormat = ChoosePixelFormat( hdc, &pfd );
	
	if (!pixelFormat) { error("ChoosePixelFormat"); cleanup(); return false; }
	
	if (!DescribePixelFormat( hdc, pixelFormat, sizeof(pfd), &pfd )) {
		error("DescribePixelFormat"); cleanup(); return false; 
	}
	
	if (!SetPixelFormat( hdc, pixelFormat, &pfd )) {
		error("SetPixelFormat"); cleanup(); return false; 
	}
	
	hglrc = wglCreateContext(hdc);
	
	if (!hglrc) { error("wglCreateContext"); cleanup(); return false; }
	
	// Init mouse cursor:
	
	POINT p; 
	GetCursorPos(&p);
	mousePos.x = p.x;
	mousePos.y = p.y;
	
	return true;
}

bool wglContext::isOpen() 
{
	return hWnd != 0;
}

void wglContext::cleanup() 
{
	if (hglrc) {
		wglDeleteContext(hglrc);
		hglrc = 0;
	}
	if (hdc) {
		ReleaseDC(hWnd, hdc);
		hdc = 0;
	}
	if (hWnd) {
		DestroyWindow(hWnd);
		hWnd = 0;
	}
	mustPaint = false; 
}

void wglContext::showMouse() 
{
	ShowCursor(true);
	ClipCursor(0);
}

void wglContext::hideMouse() 
{
	if (!hWnd) return;
	
	POINT p; p.x = 0; p.y = 0;
	ScreenToClient(hWnd,&p);
	RECT r;
	GetClientRect(hWnd,&r);
	r.left -= p.x; r.right -= p.x;
	r.top -= p.y; r.bottom -= p.y;
	ClipCursor(&r);
	
	ShowCursor(false);
}

void wglContext::lockMouse() 
{
	if (mouseLocked || !hWnd) return;
	
	hideMouse();
	mouseLocked = true;
	
	POINT p;
	GetCursorPos(&p);
	mousePos.x = p.x;
	mousePos.y = p.y;
}

void wglContext::unlockMouse() 
{
	if (!mouseLocked) return;
	
	showMouse();
	mouseLocked = false;
}

vector wglContext::getMousePos() 
{
	if (!hWnd) return vector(0,0,0);
	
	RECT r;
	GetClientRect(hWnd, &r);
	POINT p; p.x = mousePos.x; p.y = mousePos.y;
	ScreenToClient(hWnd, &p);
	
	return vector( 
		clamp( (double(p.x) / double(r.right)), 0.0, 1.0 ),
		clamp( (double(p.y) / double(r.bottom)), 0.0, 1.0 ),
		mousePos.z );
}

vector wglContext::getMouseDelta() 
{ // mouse movement in pixels
	if (!hWnd) return vector(0,0,0);
	
	RECT r;
	GetClientRect(hWnd, &r);
	
	vector d( mouseDelta.x,
		mouseDelta.y,
		mouseDelta.z );
	mouseDelta = vector(0,0,0);
	
	return d;
}
int wglContext::getMouseButtons() 
{
	return buttonState;
}

int wglContext::getMouseButtonsChanged() 
{
	unsigned ch = buttonChanged;
	buttonChanged = 0;
	return ch;
}

int wglContext::getShiftKey() 
{
	return Kshift;
}

int wglContext::getAltKey() 
{
	return Kalt;
}

int wglContext::getCtrlKey() 
{
	return Kctrl;
}

LRESULT wglContext::winMessage( HWND, UINT msg, WPARAM wParam, LPARAM lParam) 
{
	switch( msg ) {
    case WM_LBUTTONDOWN:
    case WM_LBUTTONUP:
    case WM_RBUTTONDOWN:
    case WM_RBUTTONUP:
    case WM_MBUTTONDOWN:
    case WM_MBUTTONUP:
    case WM_MOUSEMOVE: { /* Mouse inputs */
		int newState =  ( (wParam&MK_LBUTTON)!=0) +
			(((wParam&MK_RBUTTON)!=0)<<1) +
			(((wParam&MK_MBUTTON)!=0)<<2);    
		
		POINT cursor;
		GetCursorPos(&cursor);
		
		vector delta( cursor.x - mousePos.x,
			cursor.y - mousePos.y,
			0 );
		
		buttonChanged |= (buttonState ^ newState);
		buttonState = newState;
		mouseDelta = mouseDelta + delta;
		Kshift = (GetKeyState(VK_SHIFT) < 0) ||
			(GetKeyState(VK_CAPITAL) & 1);
		Kalt = GetKeyState(VK_MENU) < 0;
		Kctrl = GetKeyState(VK_CONTROL) < 0;
		
		if (mouseLocked) {
			if (mousePos.x != cursor.x ||
				mousePos.y != cursor.y)
				SetCursorPos( mousePos.x, 
				mousePos.y );
		} else {
			mousePos.x = cursor.x;
			mousePos.y = cursor.y;
		}
		
		break;
	} 
    case WM_SYSKEYUP:
    case WM_KEYUP: {
		Kshift = (GetKeyState(VK_SHIFT) < 0) ||
			(GetKeyState(VK_CAPITAL) & 1);
		Kalt = GetKeyState(VK_MENU) < 0;
		Kctrl = GetKeyState(VK_CONTROL) < 0;
		break;
				   }  /* WM_KEYUP */
		
		
    case WM_SYSKEYDOWN:
    case WM_KEYDOWN: {
		char *kNameP;
		char kStr[60],fStr[4];
		
		Kshift = (GetKeyState(VK_SHIFT) < 0) ||
			(GetKeyState(VK_CAPITAL) & 1);
		Kalt = GetKeyState(VK_MENU) < 0;
		Kctrl = GetKeyState(VK_CONTROL) < 0;
		kStr[0] = 0;
		kNameP = NULL;
		
		switch (wParam) {
			
		case VK_F1:
		case VK_F2:
		case VK_F3:
		case VK_F4:
		case VK_F5:
		case VK_F6:
		case VK_F7:
		case VK_F8:
		case VK_F9:
		case VK_F10:
		case VK_F11:
		case VK_F12:
			sprintf(fStr,"f%d",wParam-VK_F1+1);
			kNameP = fStr;
			break;
			
		case VK_PRIOR:
			kNameP = "page up";
			break;
			
		case VK_NEXT:
			kNameP = "page down";
			break;
			
		case VK_END:
			kNameP = "end";
			break;
			
		case VK_HOME:
			kNameP = "home";
			break;
			
		case VK_LEFT:
			kNameP = "left";
			break;
			
		case VK_UP:
			kNameP = "up";
			break;
			
		case VK_RIGHT:
			kNameP = "right";
			break;
			
		case VK_DOWN:
			kNameP = "down";
			break;
			
		case VK_SNAPSHOT:
			kNameP = "print screen";
			break;
			
		case VK_INSERT:
			kNameP = "insert";
			break;
			
		case VK_DELETE:
			kNameP = "delete";
			break;
			
		case VK_NUMLOCK:
			kNameP = "numlock";
			break;
			
		case VK_SCROLL:
			kNameP = "scrlock";
			break;
			
		} /* wParam */
		
		if (kNameP) {
			if (Kctrl) strcat(kStr,"ctrl+");
			if (Kalt) strcat(kStr,"alt+");
			if (Kshift) strcat(kStr,"shift+");
			strcat(kStr,kNameP);
			keys.push_back( std::string(kStr));
		}
		break;
					 } /* WM_KEYDOWN */
		
    case WM_CHAR: {
		int fShift,fAlt,fCtrl;
		char *kNameP;
		char kStr[60],wStr[2];
		
		if ((wParam >= 32) && (wParam <= 126)) { char kk[2];
        kk[0] = wParam;
        kk[1] = 0;
        keys.push_back( std::string(kk)); 
        break;
		}
		
		fShift = (GetKeyState(VK_SHIFT) < 0) ||
			(GetKeyState(VK_CAPITAL) & 1);
		fAlt = GetKeyState(VK_MENU) < 0;
		fCtrl = GetKeyState(VK_CONTROL) < 0;
		kStr[0] = 0;
		kNameP = NULL;
		
		if (!fCtrl && wParam == VK_RETURN)
			kNameP = "\n";
		else if (!fCtrl && wParam == VK_ESCAPE)
			kNameP = "escape";
		else if (!fCtrl && wParam == VK_BACK)
			kNameP = "backspace";
		else if (!fCtrl && wParam == VK_TAB)
			kNameP = "\t";
		else if ((wParam > 0) && (wParam <= 26)) {
			wStr[0] = wParam-1+'a';
			wStr[1] = 0;
			kNameP = wStr;
		} else if (wParam == 27)
			kNameP = "[";
		else if (wParam == 28)
			kNameP = "\\";
		else if (wParam == 29)
			kNameP = "]";
		else if (wParam == 30)
			kNameP = "^";
		else if (wParam == 31)
			kNameP = "_";
		
		if (kNameP) {
			if (fCtrl) strcat(kStr,"ctrl+");
			if (fAlt) strcat(kStr,"alt+");
			if (fShift) strcat(kStr,"shift+");
			strcat(kStr,kNameP);
			keys.push_back( std::string(kStr));
		}
		break;
				  } /* WM_CHAR */
		
    case WM_PAINT:
		// force display refresh
		mustPaint = true;
		makeCurrent();
		glClear(GL_COLOR_BUFFER_BIT);
		swapBuffers();
		makeNotCurrent();
		
		break;
		
    case WM_DESTROY:
		cls.clearContext( hWnd );
		hWnd = 0;
		return 0;
  }
  
  return DefWindowProc( hWnd, msg, wParam, lParam );
}

void wglContext::makeCurrent() 
{
	if (!current) {
		lasthdc = wglGetCurrentDC();
		lastglrc = wglGetCurrentContext();
		wglMakeCurrent( hdc, hglrc );
		Cache::write_lock L( list_lock);
		for (std::vector<std::pair<int, int> >::iterator i = pending_glDeleteLists.begin();
			i != pending_glDeleteLists.end(); ++i) {
			glDeleteLists(i->first, i->second);	
		}
		pending_glDeleteLists.clear();
	}
	current++;
}

void wglContext::makeNotCurrent() 
{
	current--;
	if (!current) {
		wglMakeCurrent( lasthdc, lastglrc );
		lasthdc = 0; lastglrc = 0;
	}
}

void wglContext::swapBuffers() 
{
	SwapBuffers(hdc);
}

vector wglContext::origin() 
{
	RECT r;
	GetWindowRect(hWnd, &r);
	return vector(r.left,r.top);
}

vector wglContext::corner() 
{
	RECT r;
	GetWindowRect(hWnd, &r);
	return vector(r.right,r.bottom); 
}

int wglContext::width() 
{
	RECT r;
	
	if (!hWnd) return 0;
	GetClientRect(hWnd, &r);
	return r.right;
}

int 
wglContext::height() 
{
	RECT r;
	
	if (!hWnd) return 0;
	GetClientRect(hWnd, &r);
	return r.bottom;
}

std::string 
wglContext::getKeys() 
{
	if (keys.size()) {
		std::string s = keys[0];
		keys.erase(keys.begin());
		return s;
	} 
	return std::string("");
}

} // !namespace visual
