# -*- coding: utf-8 -*-
import numpy as np
import pytest
import sys
from typing import Dict, Any, Union, Type

from pyqtgraph import getCupy, getConfigOption, setConfigOption
from pyqtgraph.functions import makeARGB as real_makeARGB
try:
    import cupy
except ImportError:
    cupy = None

IN_2D_INT8 = np.array([[173, 48, 122, 41], [210, 192, 0, 5], [104, 56, 102, 115], [78, 19, 255, 6]], dtype=np.uint8)
IN_RGB_INT8 = np.array(
    [
        [[16, 69, 62], [66, 132, 135], [220, 80, 36], [53, 34, 68]],
        [[140, 23, 113], [0, 63, 206], [96, 255, 100], [226, 182, 155]],
        [[28, 237, 223], [215, 232, 209], [17, 16, 50], [96, 187, 93]],
        [[220, 193, 232], [134, 168, 150], [55, 64, 221], [96, 108, 227]],
    ],
    dtype=np.uint8,
)
IN_RGBA_INT8 = np.array(
    [
        [[151, 252, 73, 107], [28, 221, 35, 0], [87, 122, 126, 114], [47, 59, 24, 200]],
        [[189, 246, 242, 255], [123, 255, 29, 14], [201, 208, 133, 32], [118, 203, 141, 245]],
        [[133, 131, 248, 81], [4, 84, 99, 40], [40, 167, 119, 150], [13, 158, 108, 21]],
        [[156, 221, 166, 250], [77, 188, 13, 166], [0, 1, 185, 25], [83, 35, 103, 120]],
    ],
    dtype=np.uint8,
)
IN_2D_INT16 = np.array(
    [
        [13364, 55041, 40746, 40937],
        [57612, 34247, 34132, 0],
        [10109, 56950, 41856, 21479],
        [14881, 65535, 48079, 11372],
    ],
    dtype=np.uint16,
)
IN_RGB_INT16 = np.array(
    [
        [[55626, 45263, 0], [19468, 39208, 36391], [33255, 8664, 56991], [37588, 31212, 38295]],
        [[58933, 16402, 36905], [9928, 23928, 12418], [16461, 47738, 18189], [17004, 39307, 59941]],
        [[43717, 49573, 9843], [35967, 3891, 39618], [53542, 58151, 29112], [53667, 4092, 35267]],
        [[15957, 21648, 45238], [65535, 47107, 52049], [6342, 34547, 19902], [43386, 37301, 35095]],
    ],
    dtype=np.uint16,
)
IN_RGBA_INT16 = np.array(
    [
        [
            [13060, 40847, 29621, 46719],
            [0, 36509, 33525, 56649],
            [48328, 23093, 47186, 26801],
            [57336, 12247, 30996, 11691],
        ],
        [
            [4863, 41121, 32045, 25250],
            [27779, 65098, 59921, 47771],
            [8906, 18280, 5066, 48587],
            [65535, 25758, 27250, 17284],
        ],
        [
            [52005, 65535, 40746, 65535],
            [33, 57630, 27750, 42371],
            [50176, 35079, 19220, 63662],
            [17702, 5506, 36216, 48303],
        ],
        [
            [61592, 27692, 37436, 7249],
            [54653, 39986, 58441, 12819],
            [20887, 56588, 32440, 85],
            [13457, 14661, 58972, 48779],
        ],
    ],
    dtype=np.uint16,
)
IN_2D_FLOAT = np.array(
    [
        [np.inf, 0.53662884, np.nan, 0.8853132],
        [0.8496698, 0.88006145, 1.0, 0.06621328],
        [0.99158293, 0.8476984, 0.16672458, 0.9887786],
        [0.07076367, 0.66354364, 0.8781082, 0.988832],
    ],
    dtype=np.float32,
)
IN_RGB_FLOAT = np.array(
    [
        [
            [0.23317624, 0.39086635, 0.12795302],
            [0.40659714, 0.9079258, 0.28431135],
            [0.91651599, 0.46082205, 0.16928465],
            [0.29368765, 0.97987488, 0.72257988],
        ],
        [
            [np.nan, 0.72908475, 0.54018012],
            [0.91277435, 0.2842577, 0.73481915],
            [0.33844504, 0.22060913, 0.9802894],
            [0.13995676, 0.34752838, 0.39652277],
        ],
        [
            [0.85315026, 0.19330797, 0.0],
            [0.48584232, 0.04943356, 0.59906953],
            [np.inf, 0.32614581, 0.1483722],
            [0.37340863, 0.35432855, 0.08973532],
        ],
        [
            [0.69666134, 0.52481322, 0.49057305],
            [0.93366339, 0.1428689, 0.6845513],
            [0.27681383, 0.69472673, 0.06750892],
            [0.26349886, 0.25841691, 0.86171104],
        ],
    ]
)
IN_RGBA_FLOAT = np.array(
    [
        [
            [0.97383172, 0.62680971, 0.02285016, np.nan],
            [0.85295433, 0.93014834, 0.59482999, np.inf],
            [0.4017482, 0.79809183, 0.22407464, 0.17327807],
            [0.95953263, 0.69535086, 0.28846483, 0.76970823],
        ],
        [
            [0.11487603, 0.7447609, 0.06767498, 0.98054729],
            [0.66071068, 0.73931366, 0.33155686, 0.81827122],
            [0.78035892, 0.52920802, 0.5671388, 0.31783899],
            [0.81709002, 0.82204682, 0.82584029, 0.49434749],
        ],
        [
            [0.03142089, 0.8322058, 0.31646922, 0.94636969],
            [0.62381573, 0.60052138, 0.50244611, 0.92886007],
            [np.nan, np.nan, 0.02940048, 0.52529675],
            [0.9786162, 0.54928697, 0.2409731, 0.34705397],
        ],
        [
            [0.68922233, np.inf, 0.85027734, 0.35388624],
            [0.16489042, 0.29860162, 0.09349833, 0.67714667],
            [0.25351483, 0.25596098, 0.80461891, 0.99952403],
            [0.0, 1.0, 0.58084746, 0.46211944],
        ],
    ]
)
INPUTS: Dict[Any, np.ndarray] = {
    (np.uint8, "2D"): IN_2D_INT8,
    (np.uint8, "RGB"): IN_RGB_INT8,
    (np.uint8, "RGBA"): IN_RGBA_INT8,
    (np.uint16, "2D"): IN_2D_INT16,
    (np.uint16, "RGB"): IN_RGB_INT16,
    (np.uint16, "RGBA"): IN_RGBA_INT16,
    (np.float32, "2D"): IN_2D_FLOAT,
    (np.float32, "RGB"): IN_RGB_FLOAT,
    (np.float32, "RGBA"): IN_RGBA_FLOAT,
}
LUT8 = np.zeros((255,), dtype=np.uint8)
LUT8[::2] = 255
LUT16 = np.zeros((65535,), dtype=np.uint8)
LUT16[::3] = 255
LUTS: Dict[Any, np.ndarray] = {
    np.uint8: LUT8,
    np.uint16: LUT16,
}
LEVELS = {
    "SIMPLE": (0, 1),
    "RGB": ((0, 255), (1, 250), (100, 160)),
    "RGBA": ((255, 11111), (100, 10000), (0, 255), (127, 255)),
}

EXPECTED_OUTPUTS: Dict[tuple, Union[Type[Exception], np.ndarray]] = {
    (np.uint8, "2D", None, None, None, True): np.array(
        [
            [[173, 173, 173, 255], [48, 48, 48, 255], [122, 122, 122, 255], [41, 41, 41, 255]],
            [[210, 210, 210, 255], [192, 192, 192, 255], [0, 0, 0, 255], [5, 5, 5, 255]],
            [[104, 104, 104, 255], [56, 56, 56, 255], [102, 102, 102, 255], [115, 115, 115, 255]],
            [[78, 78, 78, 255], [19, 19, 19, 255], [255, 255, 255, 255], [6, 6, 6, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", None, None, None, False): np.array(
        [
            [[173, 173, 173, 255], [48, 48, 48, 255], [122, 122, 122, 255], [41, 41, 41, 255]],
            [[210, 210, 210, 255], [192, 192, 192, 255], [0, 0, 0, 255], [5, 5, 5, 255]],
            [[104, 104, 104, 255], [56, 56, 56, 255], [102, 102, 102, 255], [115, 115, 115, 255]],
            [[78, 78, 78, 255], [19, 19, 19, 255], [255, 255, 255, 255], [6, 6, 6, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", None, None, 232, True): np.array(
        [
            [[157, 157, 157, 255], [43, 43, 43, 255], [110, 110, 110, 255], [37, 37, 37, 255]],
            [[191, 191, 191, 255], [174, 174, 174, 255], [0, 0, 0, 255], [4, 4, 4, 255]],
            [[94, 94, 94, 255], [50, 50, 50, 255], [92, 92, 92, 255], [104, 104, 104, 255]],
            [[70, 70, 70, 255], [17, 17, 17, 255], [232, 232, 232, 255], [5, 5, 5, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", None, None, 232, False): np.array(
        [
            [[157, 157, 157, 255], [43, 43, 43, 255], [110, 110, 110, 255], [37, 37, 37, 255]],
            [[191, 191, 191, 255], [174, 174, 174, 255], [0, 0, 0, 255], [4, 4, 4, 255]],
            [[94, 94, 94, 255], [50, 50, 50, 255], [92, 92, 92, 255], [104, 104, 104, 255]],
            [[70, 70, 70, 255], [17, 17, 17, 255], [232, 232, 232, 255], [5, 5, 5, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", None, None, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", None, None, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", None, np.uint8, None, True): np.array(
        [
            [[0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", None, np.uint8, None, False): np.array(
        [
            [[0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", None, np.uint8, 232, True): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", None, np.uint8, 232, False): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", None, np.uint8, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", None, np.uint8, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", None, np.uint16, None, True): np.array(
        [
            [[0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", None, np.uint16, None, False): np.array(
        [
            [[0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", None, np.uint16, 232, True): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", None, np.uint16, 232, False): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", None, np.uint16, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", None, np.uint16, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "SIMPLE", None, None, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "SIMPLE", None, None, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "SIMPLE", None, 232, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "SIMPLE", None, 232, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "SIMPLE", None, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "SIMPLE", None, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "SIMPLE", np.uint8, None, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "SIMPLE", np.uint8, None, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "SIMPLE", np.uint8, 232, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "SIMPLE", np.uint8, 232, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "SIMPLE", np.uint8, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "SIMPLE", np.uint8, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "SIMPLE", np.uint16, None, True): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "SIMPLE", np.uint16, None, False): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "SIMPLE", np.uint16, 232, True): np.array(
        [
            [[0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "SIMPLE", np.uint16, 232, False): np.array(
        [
            [[0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "SIMPLE", np.uint16, 13333, True): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "SIMPLE", np.uint16, 13333, False): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "RGB", None, None, True): Exception,
    (np.uint8, "2D", "RGB", None, None, False): Exception,
    (np.uint8, "2D", "RGB", None, 232, True): Exception,
    (np.uint8, "2D", "RGB", None, 232, False): Exception,
    (np.uint8, "2D", "RGB", None, 13333, True): Exception,
    (np.uint8, "2D", "RGB", None, 13333, False): Exception,
    (np.uint8, "2D", "RGB", np.uint8, None, True): Exception,
    (np.uint8, "2D", "RGB", np.uint8, None, False): Exception,
    (np.uint8, "2D", "RGB", np.uint8, 232, True): Exception,
    (np.uint8, "2D", "RGB", np.uint8, 232, False): Exception,
    (np.uint8, "2D", "RGB", np.uint8, 13333, True): Exception,
    (np.uint8, "2D", "RGB", np.uint8, 13333, False): Exception,
    (np.uint8, "2D", "RGB", np.uint16, None, True): Exception,
    (np.uint8, "2D", "RGB", np.uint16, None, False): Exception,
    (np.uint8, "2D", "RGB", np.uint16, 232, True): Exception,
    (np.uint8, "2D", "RGB", np.uint16, 232, False): Exception,
    (np.uint8, "2D", "RGB", np.uint16, 13333, True): Exception,
    (np.uint8, "2D", "RGB", np.uint16, 13333, False): Exception,
    (np.uint8, "2D", "RGBA", None, None, True): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [122, 122, 122, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [2, 2, 2, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [102, 102, 102, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "RGBA", None, None, False): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [122, 122, 122, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [2, 2, 2, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [102, 102, 102, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "RGBA", None, 232, True): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [110, 110, 110, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [2, 2, 2, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [92, 92, 92, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [232, 232, 232, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "RGBA", None, 232, False): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [110, 110, 110, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [2, 2, 2, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [92, 92, 92, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [232, 232, 232, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "RGBA", None, 13333, True): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [123, 123, 123, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "RGBA", None, 13333, False): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [123, 123, 123, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "RGBA", np.uint8, None, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "RGBA", np.uint8, None, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "RGBA", np.uint8, 232, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "RGBA", np.uint8, 232, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "RGBA", np.uint8, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "RGBA", np.uint8, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "RGBA", np.uint16, None, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "RGBA", np.uint16, None, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "RGBA", np.uint16, 232, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "RGBA", np.uint16, 232, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "RGBA", np.uint16, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "2D", "RGBA", np.uint16, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", None, None, None, True): np.array(
        [
            [[16, 69, 62, 255], [66, 132, 135, 255], [220, 80, 36, 255], [53, 34, 68, 255]],
            [[140, 23, 113, 255], [0, 63, 206, 255], [96, 255, 100, 255], [226, 182, 155, 255]],
            [[28, 237, 223, 255], [215, 232, 209, 255], [17, 16, 50, 255], [96, 187, 93, 255]],
            [[220, 193, 232, 255], [134, 168, 150, 255], [55, 64, 221, 255], [96, 108, 227, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", None, None, None, False): np.array(
        [
            [[62, 69, 16, 255], [135, 132, 66, 255], [36, 80, 220, 255], [68, 34, 53, 255]],
            [[113, 23, 140, 255], [206, 63, 0, 255], [100, 255, 96, 255], [155, 182, 226, 255]],
            [[223, 237, 28, 255], [209, 232, 215, 255], [50, 16, 17, 255], [93, 187, 96, 255]],
            [[232, 193, 220, 255], [150, 168, 134, 255], [221, 64, 55, 255], [227, 108, 96, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", None, None, 232, True): np.array(
        [
            [[14, 62, 56, 255], [60, 120, 122, 255], [200, 72, 32, 255], [48, 30, 61, 255]],
            [[127, 20, 102, 255], [0, 57, 187, 255], [87, 232, 90, 255], [205, 165, 141, 255]],
            [[25, 215, 202, 255], [195, 211, 190, 255], [15, 14, 45, 255], [87, 170, 84, 255]],
            [[200, 175, 211, 255], [121, 152, 136, 255], [50, 58, 201, 255], [87, 98, 206, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", None, None, 232, False): np.array(
        [
            [[56, 62, 14, 255], [122, 120, 60, 255], [32, 72, 200, 255], [61, 30, 48, 255]],
            [[102, 20, 127, 255], [187, 57, 0, 255], [90, 232, 87, 255], [141, 165, 205, 255]],
            [[202, 215, 25, 255], [190, 211, 195, 255], [45, 14, 15, 255], [84, 170, 87, 255]],
            [[211, 175, 200, 255], [136, 152, 121, 255], [201, 58, 50, 255], [206, 98, 87, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", None, None, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [0, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", None, None, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", None, np.uint8, None, True): np.array(
        [
            [[255, 0, 255, 255], [255, 255, 0, 255], [255, 255, 255, 255], [0, 255, 255, 255]],
            [[255, 0, 0, 255], [255, 0, 255, 255], [255, 255, 255, 255], [255, 255, 0, 255]],
            [[255, 0, 0, 255], [0, 255, 0, 255], [0, 255, 255, 255], [255, 0, 0, 255]],
            [[255, 0, 255, 255], [255, 255, 255, 255], [0, 255, 0, 255], [255, 255, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", None, np.uint8, None, False): np.array(
        [
            [[255, 0, 255, 255], [0, 255, 255, 255], [255, 255, 255, 255], [255, 255, 0, 255]],
            [[0, 0, 255, 255], [255, 0, 255, 255], [255, 255, 255, 255], [0, 255, 255, 255]],
            [[0, 0, 255, 255], [0, 255, 0, 255], [255, 255, 0, 255], [0, 0, 255, 255]],
            [[255, 0, 255, 255], [255, 255, 255, 255], [0, 255, 0, 255], [0, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", None, np.uint8, 232, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 0, 255]],
            [[0, 255, 255, 255], [255, 0, 0, 255], [0, 255, 255, 255], [0, 0, 0, 255]],
            [[0, 0, 255, 255], [0, 0, 255, 255], [0, 255, 0, 255], [0, 255, 255, 255]],
            [[255, 0, 0, 255], [0, 255, 255, 255], [255, 255, 0, 255], [0, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", None, np.uint8, 232, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 255, 255, 255]],
            [[255, 255, 0, 255], [0, 0, 255, 255], [255, 255, 0, 255], [0, 0, 0, 255]],
            [[255, 0, 0, 255], [255, 0, 0, 255], [0, 255, 0, 255], [255, 255, 0, 255]],
            [[0, 0, 255, 255], [255, 255, 0, 255], [0, 255, 255, 255], [255, 255, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", None, np.uint8, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", None, np.uint8, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", None, np.uint16, None, True): np.array(
        [
            [[0, 255, 0, 255], [255, 255, 255, 255], [0, 0, 255, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [255, 255, 0, 255], [255, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 255, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [255, 0, 255, 255]],
            [[0, 0, 0, 255], [0, 255, 255, 255], [0, 0, 0, 255], [255, 255, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", None, np.uint16, None, False): np.array(
        [
            [[0, 255, 0, 255], [255, 255, 255, 255], [255, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 255, 255, 255], [0, 0, 255, 255], [0, 0, 0, 255]],
            [[0, 255, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [255, 0, 255, 255]],
            [[0, 0, 0, 255], [255, 255, 0, 255], [0, 0, 0, 255], [0, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", None, np.uint16, 232, True): np.array(
        [
            [[0, 0, 0, 255], [255, 255, 0, 255], [0, 255, 0, 255], [255, 255, 0, 255]],
            [[0, 0, 255, 255], [255, 255, 0, 255], [255, 0, 255, 255], [0, 255, 255, 255]],
            [[0, 0, 0, 255], [255, 0, 0, 255], [255, 0, 255, 255], [255, 0, 255, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 255, 255], [255, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", None, np.uint16, 232, False): np.array(
        [
            [[0, 0, 0, 255], [0, 255, 255, 255], [0, 255, 0, 255], [0, 255, 255, 255]],
            [[255, 0, 0, 255], [0, 255, 255, 255], [255, 0, 255, 255], [255, 255, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 255, 255], [255, 0, 255, 255], [255, 0, 255, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 0, 0, 255], [0, 0, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", None, np.uint16, 13333, True): np.array(
        [
            [[0, 0, 0, 255], [255, 0, 0, 255], [255, 255, 0, 255], [0, 0, 255, 255]],
            [[255, 0, 0, 255], [255, 255, 255, 255], [255, 0, 0, 255], [0, 255, 0, 255]],
            [[255, 0, 0, 255], [255, 0, 0, 255], [255, 0, 0, 255], [255, 255, 0, 255]],
            [[255, 0, 0, 255], [0, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", None, np.uint16, 13333, False): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 255, 255], [0, 255, 255, 255], [255, 0, 0, 255]],
            [[0, 0, 255, 255], [255, 255, 255, 255], [0, 0, 255, 255], [0, 255, 0, 255]],
            [[0, 0, 255, 255], [0, 0, 255, 255], [0, 0, 255, 255], [0, 255, 255, 255]],
            [[0, 0, 255, 255], [255, 255, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "SIMPLE", None, None, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [0, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "SIMPLE", None, None, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "SIMPLE", None, 232, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [0, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "SIMPLE", None, 232, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "SIMPLE", None, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [0, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "SIMPLE", None, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "SIMPLE", np.uint8, None, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "SIMPLE", np.uint8, None, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "SIMPLE", np.uint8, 232, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "SIMPLE", np.uint8, 232, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "SIMPLE", np.uint8, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "SIMPLE", np.uint8, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "SIMPLE", np.uint16, None, True): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [255, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "SIMPLE", np.uint16, None, False): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "SIMPLE", np.uint16, 232, True): np.array(
        [
            [[0, 255, 0, 255], [255, 255, 255, 255], [0, 0, 255, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [255, 255, 0, 255], [255, 255, 0, 255], [0, 0, 0, 255]],
            [[0, 255, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [255, 0, 255, 255]],
            [[0, 0, 0, 255], [0, 255, 255, 255], [0, 0, 0, 255], [255, 255, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "SIMPLE", np.uint16, 232, False): np.array(
        [
            [[0, 255, 0, 255], [255, 255, 255, 255], [255, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 255, 255, 255], [0, 255, 255, 255], [0, 0, 0, 255]],
            [[0, 255, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [255, 0, 255, 255]],
            [[0, 0, 0, 255], [255, 255, 0, 255], [0, 0, 0, 255], [0, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "SIMPLE", np.uint16, 13333, True): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [255, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "SIMPLE", np.uint16, 13333, False): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "RGB", None, None, True): np.array(
        [
            [[16, 69, 0, 255], [66, 134, 148, 255], [220, 80, 0, 255], [53, 33, 0, 255]],
            [[140, 22, 55, 255], [0, 63, 255, 255], [96, 255, 0, 255], [226, 185, 233, 255]],
            [[28, 241, 255, 255], [215, 236, 255, 255], [17, 15, 0, 255], [96, 190, 0, 255]],
            [[220, 196, 255, 255], [134, 171, 212, 255], [55, 64, 255, 255], [96, 109, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "RGB", None, None, False): np.array(
        [
            [[0, 69, 16, 255], [148, 134, 66, 255], [0, 80, 220, 255], [0, 33, 53, 255]],
            [[55, 22, 140, 255], [255, 63, 0, 255], [0, 255, 96, 255], [233, 185, 226, 255]],
            [[255, 241, 28, 255], [255, 236, 215, 255], [0, 15, 17, 255], [0, 190, 96, 255]],
            [[255, 196, 220, 255], [212, 171, 134, 255], [255, 64, 55, 255], [255, 109, 96, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "RGB", None, 232, True): np.array(
        [
            [[14, 63, 0, 255], [60, 122, 135, 255], [200, 73, 0, 255], [48, 30, 0, 255]],
            [[127, 20, 50, 255], [0, 57, 255, 255], [87, 236, 0, 255], [205, 168, 212, 255]],
            [[25, 219, 255, 255], [195, 215, 255, 255], [15, 13, 0, 255], [87, 173, 0, 255]],
            [[200, 178, 255, 255], [121, 155, 193, 255], [50, 58, 255, 255], [87, 99, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "RGB", None, 232, False): np.array(
        [
            [[0, 63, 14, 255], [135, 122, 60, 255], [0, 73, 200, 255], [0, 30, 48, 255]],
            [[50, 20, 127, 255], [255, 57, 0, 255], [0, 236, 87, 255], [212, 168, 205, 255]],
            [[255, 219, 25, 255], [255, 215, 195, 255], [0, 13, 15, 255], [0, 173, 87, 255]],
            [[255, 178, 200, 255], [193, 155, 121, 255], [255, 58, 50, 255], [255, 99, 87, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "RGB", None, 13333, True): np.array(
        [
            [[255, 255, 0, 255], [255, 255, 255, 255], [255, 255, 0, 255], [255, 255, 0, 255]],
            [[255, 255, 255, 255], [0, 255, 255, 255], [255, 255, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 0, 255], [255, 255, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "RGB", None, 13333, False): np.array(
        [
            [[0, 255, 255, 255], [255, 255, 255, 255], [0, 255, 255, 255], [0, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 0, 255], [0, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 255, 255, 255], [0, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "RGB", np.uint8, None, True): np.array(
        [
            [[255, 0, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 255, 255]],
            [[255, 255, 0, 255], [255, 0, 255, 255], [255, 255, 255, 255], [255, 0, 0, 255]],
            [[255, 0, 255, 255], [0, 255, 255, 255], [0, 0, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 0, 255, 255], [0, 255, 255, 255], [255, 0, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "RGB", np.uint8, None, False): np.array(
        [
            [[255, 0, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 0, 0, 255]],
            [[0, 255, 255, 255], [255, 0, 255, 255], [255, 255, 255, 255], [0, 0, 255, 255]],
            [[255, 0, 255, 255], [255, 255, 0, 255], [255, 0, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 0, 255, 255], [255, 255, 0, 255], [255, 0, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "RGB", np.uint8, 232, True): np.array(
        [
            [[255, 0, 255, 255], [255, 255, 0, 255], [255, 0, 255, 255], [255, 255, 255, 255]],
            [[0, 255, 255, 255], [255, 0, 255, 255], [0, 255, 255, 255], [0, 255, 255, 255]],
            [[0, 0, 255, 255], [0, 0, 255, 255], [0, 0, 255, 255], [0, 0, 255, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "RGB", np.uint8, 232, False): np.array(
        [
            [[255, 0, 255, 255], [0, 255, 255, 255], [255, 0, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 0, 255], [255, 0, 255, 255], [255, 255, 0, 255], [255, 255, 0, 255]],
            [[255, 0, 0, 255], [255, 0, 0, 255], [255, 0, 0, 255], [255, 0, 0, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255], [255, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "RGB", np.uint8, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "RGB", np.uint8, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "RGB", np.uint16, None, True): np.array(
        [
            [[0, 0, 255, 255], [255, 0, 0, 255], [0, 0, 255, 255], [0, 255, 255, 255]],
            [[0, 255, 255, 255], [255, 255, 0, 255], [255, 0, 255, 255], [0, 255, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 255, 255], [255, 0, 255, 255]],
            [[0, 0, 0, 255], [0, 255, 255, 255], [0, 255, 0, 255], [255, 255, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "RGB", np.uint16, None, False): np.array(
        [
            [[255, 0, 0, 255], [0, 0, 255, 255], [255, 0, 0, 255], [255, 255, 0, 255]],
            [[255, 255, 0, 255], [0, 255, 255, 255], [255, 0, 255, 255], [0, 255, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 0, 0, 255], [255, 0, 255, 255]],
            [[0, 0, 0, 255], [255, 255, 0, 255], [0, 255, 0, 255], [0, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "RGB", np.uint16, 232, True): np.array(
        [
            [[0, 255, 255, 255], [255, 0, 255, 255], [0, 0, 255, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [255, 255, 0, 255], [255, 0, 255, 255], [0, 255, 0, 255]],
            [[0, 255, 0, 255], [255, 0, 0, 255], [255, 0, 255, 255], [255, 0, 255, 255]],
            [[0, 0, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "RGB", np.uint16, 232, False): np.array(
        [
            [[255, 255, 0, 255], [255, 0, 255, 255], [255, 0, 0, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [0, 255, 255, 255], [255, 0, 255, 255], [0, 255, 0, 255]],
            [[0, 255, 0, 255], [0, 0, 255, 255], [255, 0, 255, 255], [255, 0, 255, 255]],
            [[255, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "RGB", np.uint16, 13333, True): np.array(
        [
            [[0, 0, 255, 255], [255, 255, 0, 255], [255, 255, 255, 255], [0, 255, 255, 255]],
            [[255, 0, 0, 255], [255, 0, 0, 255], [255, 0, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 0, 255], [255, 255, 0, 255], [255, 0, 255, 255], [255, 0, 255, 255]],
            [[255, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [255, 0, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "RGB", np.uint16, 13333, False): np.array(
        [
            [[255, 0, 0, 255], [0, 255, 255, 255], [255, 255, 255, 255], [255, 255, 0, 255]],
            [[0, 0, 255, 255], [0, 0, 255, 255], [255, 0, 255, 255], [0, 0, 0, 255]],
            [[0, 255, 255, 255], [0, 255, 255, 255], [255, 0, 255, 255], [255, 0, 255, 255]],
            [[0, 0, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255], [255, 0, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGB", "RGBA", None, None, True): Exception,
    (np.uint8, "RGB", "RGBA", None, None, False): Exception,
    (np.uint8, "RGB", "RGBA", None, 232, True): Exception,
    (np.uint8, "RGB", "RGBA", None, 232, False): Exception,
    (np.uint8, "RGB", "RGBA", None, 13333, True): Exception,
    (np.uint8, "RGB", "RGBA", None, 13333, False): Exception,
    (np.uint8, "RGB", "RGBA", np.uint8, None, True): Exception,
    (np.uint8, "RGB", "RGBA", np.uint8, None, False): Exception,
    (np.uint8, "RGB", "RGBA", np.uint8, 232, True): Exception,
    (np.uint8, "RGB", "RGBA", np.uint8, 232, False): Exception,
    (np.uint8, "RGB", "RGBA", np.uint8, 13333, True): Exception,
    (np.uint8, "RGB", "RGBA", np.uint8, 13333, False): Exception,
    (np.uint8, "RGB", "RGBA", np.uint16, None, True): Exception,
    (np.uint8, "RGB", "RGBA", np.uint16, None, False): Exception,
    (np.uint8, "RGB", "RGBA", np.uint16, 232, True): Exception,
    (np.uint8, "RGB", "RGBA", np.uint16, 232, False): Exception,
    (np.uint8, "RGB", "RGBA", np.uint16, 13333, True): Exception,
    (np.uint8, "RGB", "RGBA", np.uint16, 13333, False): Exception,
    (np.uint8, "RGBA", None, None, None, True): np.array(
        [
            [[151, 252, 73, 107], [28, 221, 35, 0], [87, 122, 126, 114], [47, 59, 24, 200]],
            [[189, 246, 242, 255], [123, 255, 29, 14], [201, 208, 133, 32], [118, 203, 141, 245]],
            [[133, 131, 248, 81], [4, 84, 99, 40], [40, 167, 119, 150], [13, 158, 108, 21]],
            [[156, 221, 166, 250], [77, 188, 13, 166], [0, 1, 185, 25], [83, 35, 103, 120]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", None, None, None, False): np.array(
        [
            [[73, 252, 151, 107], [35, 221, 28, 0], [126, 122, 87, 114], [24, 59, 47, 200]],
            [[242, 246, 189, 255], [29, 255, 123, 14], [133, 208, 201, 32], [141, 203, 118, 245]],
            [[248, 131, 133, 81], [99, 84, 4, 40], [119, 167, 40, 150], [108, 158, 13, 21]],
            [[166, 221, 156, 250], [13, 188, 77, 166], [185, 1, 0, 25], [103, 35, 83, 120]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", None, None, 232, True): np.array(
        [
            [[137, 229, 66, 97], [25, 201, 31, 0], [79, 110, 114, 103], [42, 53, 21, 181]],
            [[171, 223, 220, 232], [111, 232, 26, 12], [182, 189, 121, 29], [107, 184, 128, 222]],
            [[121, 119, 225, 73], [3, 76, 90, 36], [36, 151, 108, 136], [11, 143, 98, 19]],
            [[141, 201, 151, 227], [70, 171, 11, 151], [0, 0, 168, 22], [75, 31, 93, 109]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", None, None, 232, False): np.array(
        [
            [[66, 229, 137, 97], [31, 201, 25, 0], [114, 110, 79, 103], [21, 53, 42, 181]],
            [[220, 223, 171, 232], [26, 232, 111, 12], [121, 189, 182, 29], [128, 184, 107, 222]],
            [[225, 119, 121, 73], [90, 76, 3, 36], [108, 151, 36, 136], [98, 143, 11, 19]],
            [[151, 201, 141, 227], [11, 171, 70, 151], [168, 0, 0, 22], [93, 31, 75, 109]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", None, None, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [209, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 52, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", None, None, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 209, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 52, 0, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", None, np.uint8, None, True): np.array(
        [
            [[0, 255, 0, 0], [255, 0, 0, 255], [0, 255, 255, 255], [0, 0, 255, 255]],
            [[0, 255, 255, 255], [0, 255, 0, 255], [0, 255, 0, 255], [255, 0, 0, 0]],
            [[0, 0, 255, 0], [255, 255, 0, 255], [255, 0, 0, 255], [0, 255, 255, 0]],
            [[255, 0, 255, 255], [0, 255, 0, 255], [255, 0, 0, 0], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", None, np.uint8, None, False): np.array(
        [
            [[0, 255, 0, 0], [0, 0, 255, 255], [255, 255, 0, 255], [255, 0, 0, 255]],
            [[255, 255, 0, 255], [0, 255, 0, 255], [0, 255, 0, 255], [0, 0, 255, 0]],
            [[255, 0, 0, 0], [0, 255, 255, 255], [0, 0, 255, 255], [255, 255, 0, 0]],
            [[255, 0, 255, 255], [0, 255, 0, 255], [0, 0, 255, 0], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", None, np.uint8, 232, True): np.array(
        [
            [[0, 0, 255, 0], [0, 0, 0, 255], [0, 255, 255, 0], [255, 0, 0, 0]],
            [[0, 0, 255, 255], [0, 255, 255, 255], [255, 0, 0, 0], [0, 255, 255, 255]],
            [[0, 0, 0, 0], [0, 255, 255, 255], [255, 0, 255, 255], [0, 0, 255, 0]],
            [[0, 0, 0, 0], [255, 0, 0, 0], [255, 255, 255, 255], [0, 0, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", None, np.uint8, 232, False): np.array(
        [
            [[255, 0, 0, 0], [0, 0, 0, 255], [255, 255, 0, 0], [0, 0, 255, 0]],
            [[255, 0, 0, 255], [255, 255, 0, 255], [0, 0, 255, 0], [255, 255, 0, 255]],
            [[0, 0, 0, 0], [255, 255, 0, 255], [255, 0, 255, 255], [255, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 255, 0], [255, 255, 255, 255], [0, 0, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", None, np.uint8, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [0, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", None, np.uint8, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", None, np.uint16, None, True): np.array(
        [
            [[0, 255, 0, 0], [0, 0, 0, 255], [255, 0, 255, 255], [0, 0, 255, 0]],
            [[255, 255, 0, 0], [255, 0, 0, 0], [255, 0, 0, 0], [0, 0, 255, 0]],
            [[0, 0, 0, 255], [0, 255, 255, 0], [0, 0, 0, 255], [0, 0, 255, 255]],
            [[255, 0, 0, 0], [0, 0, 0, 0], [255, 0, 0, 0], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", None, np.uint16, None, False): np.array(
        [
            [[0, 255, 0, 0], [0, 0, 0, 255], [255, 0, 255, 255], [255, 0, 0, 0]],
            [[0, 255, 255, 0], [0, 0, 255, 0], [0, 0, 255, 0], [255, 0, 0, 0]],
            [[0, 0, 0, 255], [255, 255, 0, 0], [0, 0, 0, 255], [255, 0, 0, 255]],
            [[0, 0, 255, 0], [0, 0, 0, 0], [0, 0, 255, 0], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", None, np.uint16, 232, True): np.array(
        [
            [[0, 0, 255, 0], [0, 255, 0, 255], [0, 0, 255, 0], [255, 0, 255, 0]],
            [[255, 0, 0, 0], [255, 0, 0, 255], [0, 255, 0, 0], [0, 0, 0, 255]],
            [[0, 0, 255, 0], [255, 0, 255, 255], [255, 0, 255, 0], [0, 0, 0, 0]],
            [[255, 255, 0, 0], [0, 255, 0, 0], [255, 255, 255, 0], [255, 0, 255, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", None, np.uint16, 232, False): np.array(
        [
            [[255, 0, 0, 0], [0, 255, 0, 255], [255, 0, 0, 0], [255, 0, 255, 0]],
            [[0, 0, 255, 0], [0, 0, 255, 255], [0, 255, 0, 0], [0, 0, 0, 255]],
            [[255, 0, 0, 0], [255, 0, 255, 255], [255, 0, 255, 0], [0, 0, 0, 0]],
            [[0, 255, 255, 0], [0, 255, 0, 0], [255, 255, 255, 0], [255, 0, 255, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", None, np.uint16, 13333, True): np.array(
        [
            [[0, 255, 255, 0], [255, 0, 255, 255], [255, 255, 255, 0], [255, 255, 255, 0]],
            [[255, 0, 0, 0], [0, 0, 0, 255], [255, 255, 255, 0], [0, 255, 0, 255]],
            [[255, 255, 255, 0], [0, 255, 0, 255], [255, 0, 255, 255], [0, 0, 255, 255]],
            [[0, 0, 255, 255], [255, 0, 0, 255], [255, 0, 255, 0], [0, 255, 255, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", None, np.uint16, 13333, False): np.array(
        [
            [[255, 255, 0, 0], [255, 0, 255, 255], [255, 255, 255, 0], [255, 255, 255, 0]],
            [[0, 0, 255, 0], [0, 0, 0, 255], [255, 255, 255, 0], [0, 255, 0, 255]],
            [[255, 255, 255, 0], [0, 255, 0, 255], [255, 0, 255, 255], [255, 0, 0, 255]],
            [[255, 0, 0, 255], [0, 0, 255, 255], [255, 0, 255, 0], [255, 255, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "SIMPLE", None, None, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "SIMPLE", None, None, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 0, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "SIMPLE", None, 232, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 232, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "SIMPLE", None, 232, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 232, 0, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "SIMPLE", None, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "SIMPLE", None, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 0, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "SIMPLE", np.uint8, None, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "SIMPLE", np.uint8, None, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "SIMPLE", np.uint8, 232, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "SIMPLE", np.uint8, 232, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "SIMPLE", np.uint8, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "SIMPLE", np.uint8, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "SIMPLE", np.uint16, None, True): np.array(
        [
            [[0, 0, 0, 0], [0, 0, 0, 255], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [255, 0, 0, 0], [0, 0, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "SIMPLE", np.uint16, None, False): np.array(
        [
            [[0, 0, 0, 0], [0, 0, 0, 255], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 255, 0], [0, 0, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "SIMPLE", np.uint16, 232, True): np.array(
        [
            [[0, 255, 0, 0], [0, 0, 0, 255], [255, 0, 255, 255], [0, 0, 255, 0]],
            [[255, 255, 0, 255], [255, 255, 0, 0], [255, 0, 0, 0], [0, 0, 255, 0]],
            [[0, 0, 0, 255], [0, 255, 255, 0], [0, 0, 0, 255], [0, 0, 255, 255]],
            [[255, 0, 0, 0], [0, 0, 0, 0], [255, 0, 0, 0], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "SIMPLE", np.uint16, 232, False): np.array(
        [
            [[0, 255, 0, 0], [0, 0, 0, 255], [255, 0, 255, 255], [255, 0, 0, 0]],
            [[0, 255, 255, 255], [0, 255, 255, 0], [0, 0, 255, 0], [255, 0, 0, 0]],
            [[0, 0, 0, 255], [255, 255, 0, 0], [0, 0, 0, 255], [255, 0, 0, 255]],
            [[0, 0, 255, 0], [0, 0, 0, 0], [0, 0, 255, 0], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "SIMPLE", np.uint16, 13333, True): np.array(
        [
            [[0, 0, 0, 0], [0, 0, 0, 255], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [255, 0, 0, 0], [0, 0, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "SIMPLE", np.uint16, 13333, False): np.array(
        [
            [[0, 0, 0, 0], [0, 0, 0, 255], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 255, 0], [0, 0, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "RGB", None, None, True): Exception,
    (np.uint8, "RGBA", "RGB", None, None, False): Exception,
    (np.uint8, "RGBA", "RGB", None, 232, True): Exception,
    (np.uint8, "RGBA", "RGB", None, 232, False): Exception,
    (np.uint8, "RGBA", "RGB", None, 13333, True): Exception,
    (np.uint8, "RGBA", "RGB", None, 13333, False): Exception,
    (np.uint8, "RGBA", "RGB", np.uint8, None, True): Exception,
    (np.uint8, "RGBA", "RGB", np.uint8, None, False): Exception,
    (np.uint8, "RGBA", "RGB", np.uint8, 232, True): Exception,
    (np.uint8, "RGBA", "RGB", np.uint8, 232, False): Exception,
    (np.uint8, "RGBA", "RGB", np.uint8, 13333, True): Exception,
    (np.uint8, "RGBA", "RGB", np.uint8, 13333, False): Exception,
    (np.uint8, "RGBA", "RGB", np.uint16, None, True): Exception,
    (np.uint8, "RGBA", "RGB", np.uint16, None, False): Exception,
    (np.uint8, "RGBA", "RGB", np.uint16, 232, True): Exception,
    (np.uint8, "RGBA", "RGB", np.uint16, 232, False): Exception,
    (np.uint8, "RGBA", "RGB", np.uint16, 13333, True): Exception,
    (np.uint8, "RGBA", "RGB", np.uint16, 13333, False): Exception,
    (np.uint8, "RGBA", "RGBA", None, None, True): np.array(
        [
            [[0, 3, 73, 0], [0, 3, 35, 0], [0, 0, 126, 0], [0, 0, 24, 145]],
            [[0, 3, 242, 255], [0, 3, 29, 0], [0, 2, 133, 0], [0, 2, 141, 235]],
            [[0, 0, 248, 0], [0, 0, 99, 0], [0, 1, 119, 45], [0, 1, 108, 0]],
            [[0, 3, 166, 245], [0, 2, 13, 77], [0, 0, 185, 0], [0, 0, 103, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "RGBA", None, None, False): np.array(
        [
            [[73, 3, 0, 0], [35, 3, 0, 0], [126, 0, 0, 0], [24, 0, 0, 145]],
            [[242, 3, 0, 255], [29, 3, 0, 0], [133, 2, 0, 0], [141, 2, 0, 235]],
            [[248, 0, 0, 0], [99, 0, 0, 0], [119, 1, 0, 45], [108, 1, 0, 0]],
            [[166, 3, 0, 245], [13, 2, 0, 77], [185, 0, 0, 0], [103, 0, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "RGBA", None, 232, True): np.array(
        [
            [[0, 3, 66, 0], [0, 2, 31, 0], [0, 0, 114, 0], [0, 0, 21, 132]],
            [[0, 3, 220, 232], [0, 3, 26, 0], [0, 2, 121, 0], [0, 2, 128, 213]],
            [[0, 0, 225, 0], [0, 0, 90, 0], [0, 1, 108, 41], [0, 1, 98, 0]],
            [[0, 2, 151, 222], [0, 2, 11, 70], [0, 0, 168, 0], [0, 0, 93, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "RGBA", None, 232, False): np.array(
        [
            [[66, 3, 0, 0], [31, 2, 0, 0], [114, 0, 0, 0], [21, 0, 0, 132]],
            [[220, 3, 0, 232], [26, 3, 0, 0], [121, 2, 0, 0], [128, 2, 0, 213]],
            [[225, 0, 0, 0], [90, 0, 0, 0], [108, 1, 0, 41], [98, 1, 0, 0]],
            [[151, 2, 0, 222], [11, 2, 0, 70], [168, 0, 0, 0], [93, 0, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "RGBA", None, 13333, True): np.array(
        [
            [[0, 204, 255, 0], [0, 162, 255, 0], [0, 29, 255, 0], [0, 0, 255, 255]],
            [[0, 196, 255, 255], [0, 208, 255, 0], [0, 145, 255, 0], [0, 138, 255, 255]],
            [[0, 41, 255, 0], [0, 0, 255, 0], [0, 90, 255, 255], [0, 78, 255, 0]],
            [[0, 162, 255, 255], [0, 118, 255, 255], [0, 0, 255, 0], [0, 0, 255, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "RGBA", None, 13333, False): np.array(
        [
            [[255, 204, 0, 0], [255, 162, 0, 0], [255, 29, 0, 0], [255, 0, 0, 255]],
            [[255, 196, 0, 255], [255, 208, 0, 0], [255, 145, 0, 0], [255, 138, 0, 255]],
            [[255, 41, 0, 0], [255, 0, 0, 0], [255, 90, 0, 255], [255, 78, 0, 0]],
            [[255, 162, 0, 255], [255, 118, 0, 255], [255, 0, 0, 0], [255, 0, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "RGBA", np.uint8, None, True): np.array(
        [
            [[255, 0, 0, 255], [255, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 0]],
            [[255, 0, 255, 255], [255, 0, 0, 255], [255, 255, 0, 255], [255, 255, 0, 0]],
            [[255, 255, 255, 255], [255, 255, 0, 255], [255, 0, 0, 0], [255, 0, 255, 255]],
            [[255, 0, 255, 0], [255, 255, 0, 0], [255, 255, 0, 255], [255, 255, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "RGBA", np.uint8, None, False): np.array(
        [
            [[0, 0, 255, 255], [0, 0, 255, 255], [255, 255, 255, 255], [255, 255, 255, 0]],
            [[255, 0, 255, 255], [0, 0, 255, 255], [0, 255, 255, 255], [0, 255, 255, 0]],
            [[255, 255, 255, 255], [0, 255, 255, 255], [0, 0, 255, 0], [255, 0, 255, 255]],
            [[255, 0, 255, 0], [0, 255, 255, 0], [0, 255, 255, 255], [0, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "RGBA", np.uint8, 232, True): np.array(
        [
            [[255, 0, 255, 255], [255, 255, 0, 255], [255, 255, 255, 255], [255, 255, 0, 255]],
            [[255, 0, 255, 255], [255, 0, 255, 255], [255, 255, 0, 255], [255, 255, 255, 0]],
            [[255, 255, 0, 255], [255, 255, 255, 255], [255, 0, 255, 0], [255, 0, 255, 255]],
            [[255, 255, 0, 255], [255, 255, 0, 255], [255, 255, 255, 255], [255, 255, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "RGBA", np.uint8, 232, False): np.array(
        [
            [[255, 0, 255, 255], [0, 255, 255, 255], [255, 255, 255, 255], [0, 255, 255, 255]],
            [[255, 0, 255, 255], [255, 0, 255, 255], [0, 255, 255, 255], [255, 255, 255, 0]],
            [[0, 255, 255, 255], [255, 255, 255, 255], [255, 0, 255, 0], [255, 0, 255, 255]],
            [[0, 255, 255, 255], [0, 255, 255, 255], [255, 255, 255, 255], [0, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "RGBA", np.uint8, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 0, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 0, 255, 255], [255, 255, 255, 255]],
            [[255, 0, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "RGBA", np.uint8, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 0, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 0, 255, 255], [255, 255, 255, 255]],
            [[255, 0, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "RGBA", np.uint16, None, True): np.array(
        [
            [[255, 0, 0, 255], [255, 0, 0, 255], [255, 0, 255, 255], [255, 255, 255, 0]],
            [[255, 255, 0, 0], [255, 255, 0, 255], [255, 255, 0, 255], [255, 255, 255, 0]],
            [[255, 0, 0, 255], [255, 255, 255, 255], [255, 0, 0, 255], [255, 0, 255, 255]],
            [[255, 0, 0, 0], [255, 255, 0, 0], [255, 255, 0, 255], [255, 255, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "RGBA", np.uint16, None, False): np.array(
        [
            [[0, 0, 255, 255], [0, 0, 255, 255], [255, 0, 255, 255], [255, 255, 255, 0]],
            [[0, 255, 255, 0], [0, 255, 255, 255], [0, 255, 255, 255], [255, 255, 255, 0]],
            [[0, 0, 255, 255], [255, 255, 255, 255], [0, 0, 255, 255], [255, 0, 255, 255]],
            [[0, 0, 255, 0], [0, 255, 255, 0], [0, 255, 255, 255], [0, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "RGBA", np.uint16, 232, True): np.array(
        [
            [[255, 255, 255, 255], [255, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 0, 0], [255, 255, 0, 255], [255, 0, 0, 255], [255, 0, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 0, 255, 0], [255, 0, 0, 255]],
            [[255, 0, 0, 255], [255, 0, 0, 0], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "RGBA", np.uint16, 232, False): np.array(
        [
            [[255, 255, 255, 255], [0, 0, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[0, 255, 255, 0], [0, 255, 255, 255], [0, 0, 255, 255], [0, 0, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 0, 255, 0], [0, 0, 255, 255]],
            [[0, 0, 255, 255], [0, 0, 255, 0], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "RGBA", np.uint16, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 0, 255, 255], [255, 255, 255, 0]],
            [[255, 0, 0, 0], [255, 0, 0, 255], [255, 0, 255, 255], [255, 255, 0, 255]],
            [[255, 0, 255, 255], [255, 255, 0, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
            [[255, 255, 255, 0], [255, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint8, "RGBA", "RGBA", np.uint16, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 0, 255, 255], [255, 255, 255, 0]],
            [[0, 0, 255, 0], [0, 0, 255, 255], [255, 0, 255, 255], [0, 255, 255, 255]],
            [[255, 0, 255, 255], [0, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
            [[255, 255, 255, 0], [0, 0, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", None, None, None, True): np.array(
        [
            [[52, 52, 52, 255], [214, 214, 214, 255], [158, 158, 158, 255], [159, 159, 159, 255]],
            [[224, 224, 224, 255], [133, 133, 133, 255], [132, 132, 132, 255], [0, 0, 0, 255]],
            [[39, 39, 39, 255], [221, 221, 221, 255], [162, 162, 162, 255], [83, 83, 83, 255]],
            [[57, 57, 57, 255], [255, 255, 255, 255], [187, 187, 187, 255], [44, 44, 44, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", None, None, None, False): np.array(
        [
            [[52, 52, 52, 255], [214, 214, 214, 255], [158, 158, 158, 255], [159, 159, 159, 255]],
            [[224, 224, 224, 255], [133, 133, 133, 255], [132, 132, 132, 255], [0, 0, 0, 255]],
            [[39, 39, 39, 255], [221, 221, 221, 255], [162, 162, 162, 255], [83, 83, 83, 255]],
            [[57, 57, 57, 255], [255, 255, 255, 255], [187, 187, 187, 255], [44, 44, 44, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", None, None, 232, True): np.array(
        [
            [[47, 47, 47, 255], [194, 194, 194, 255], [144, 144, 144, 255], [144, 144, 144, 255]],
            [[203, 203, 203, 255], [121, 121, 121, 255], [120, 120, 120, 255], [0, 0, 0, 255]],
            [[35, 35, 35, 255], [201, 201, 201, 255], [148, 148, 148, 255], [76, 76, 76, 255]],
            [[52, 52, 52, 255], [232, 232, 232, 255], [170, 170, 170, 255], [40, 40, 40, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", None, None, 232, False): np.array(
        [
            [[47, 47, 47, 255], [194, 194, 194, 255], [144, 144, 144, 255], [144, 144, 144, 255]],
            [[203, 203, 203, 255], [121, 121, 121, 255], [120, 120, 120, 255], [0, 0, 0, 255]],
            [[35, 35, 35, 255], [201, 201, 201, 255], [148, 148, 148, 255], [76, 76, 76, 255]],
            [[52, 52, 52, 255], [232, 232, 232, 255], [170, 170, 170, 255], [40, 40, 40, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", None, None, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", None, None, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", None, np.uint8, None, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", None, np.uint8, None, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", None, np.uint8, 232, True): np.array(
        [
            [[0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", None, np.uint8, 232, False): np.array(
        [
            [[0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", None, np.uint8, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", None, np.uint8, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", None, np.uint16, None, True): np.array(
        [
            [[0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", None, np.uint16, None, False): np.array(
        [
            [[0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", None, np.uint16, 232, True): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", None, np.uint16, 232, False): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", None, np.uint16, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", None, np.uint16, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "SIMPLE", None, None, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "SIMPLE", None, None, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "SIMPLE", None, 232, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "SIMPLE", None, 232, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "SIMPLE", None, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "SIMPLE", None, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "SIMPLE", np.uint8, None, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "SIMPLE", np.uint8, None, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "SIMPLE", np.uint8, 232, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "SIMPLE", np.uint8, 232, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "SIMPLE", np.uint8, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "SIMPLE", np.uint8, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "SIMPLE", np.uint16, None, True): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "SIMPLE", np.uint16, None, False): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "SIMPLE", np.uint16, 232, True): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "SIMPLE", np.uint16, 232, False): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "SIMPLE", np.uint16, 13333, True): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "SIMPLE", np.uint16, 13333, False): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "RGB", None, None, True): Exception,
    (np.uint16, "2D", "RGB", None, None, False): Exception,
    (np.uint16, "2D", "RGB", None, 232, True): Exception,
    (np.uint16, "2D", "RGB", None, 232, False): Exception,
    (np.uint16, "2D", "RGB", None, 13333, True): Exception,
    (np.uint16, "2D", "RGB", None, 13333, False): Exception,
    (np.uint16, "2D", "RGB", np.uint8, None, True): Exception,
    (np.uint16, "2D", "RGB", np.uint8, None, False): Exception,
    (np.uint16, "2D", "RGB", np.uint8, 232, True): Exception,
    (np.uint16, "2D", "RGB", np.uint8, 232, False): Exception,
    (np.uint16, "2D", "RGB", np.uint8, 13333, True): Exception,
    (np.uint16, "2D", "RGB", np.uint8, 13333, False): Exception,
    (np.uint16, "2D", "RGB", np.uint16, None, True): Exception,
    (np.uint16, "2D", "RGB", np.uint16, None, False): Exception,
    (np.uint16, "2D", "RGB", np.uint16, 232, True): Exception,
    (np.uint16, "2D", "RGB", np.uint16, 232, False): Exception,
    (np.uint16, "2D", "RGB", np.uint16, 13333, True): Exception,
    (np.uint16, "2D", "RGB", np.uint16, 13333, False): Exception,
    (np.uint16, "2D", "RGBA", None, None, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[231, 231, 231, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "RGBA", None, None, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[231, 231, 231, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "RGBA", None, 232, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[210, 210, 210, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "RGBA", None, 232, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[210, 210, 210, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "RGBA", None, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "RGBA", None, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "RGBA", np.uint8, None, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "RGBA", np.uint8, None, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "RGBA", np.uint8, 232, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "RGBA", np.uint8, 232, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "RGBA", np.uint8, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "RGBA", np.uint8, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "RGBA", np.uint16, None, True): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "RGBA", np.uint16, None, False): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "RGBA", np.uint16, 232, True): np.array(
        [
            [[0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "RGBA", np.uint16, 232, False): np.array(
        [
            [[0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "RGBA", np.uint16, 13333, True): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "2D", "RGBA", np.uint16, 13333, False): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", None, None, None, True): np.array(
        [
            [[216, 176, 0, 255], [75, 152, 141, 255], [129, 33, 221, 255], [146, 121, 149, 255]],
            [[229, 63, 143, 255], [38, 93, 48, 255], [64, 185, 70, 255], [66, 152, 233, 255]],
            [[170, 192, 38, 255], [139, 15, 154, 255], [208, 226, 113, 255], [208, 15, 137, 255]],
            [[62, 84, 176, 255], [255, 183, 202, 255], [24, 134, 77, 255], [168, 145, 136, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", None, None, None, False): np.array(
        [
            [[0, 176, 216, 255], [141, 152, 75, 255], [221, 33, 129, 255], [149, 121, 146, 255]],
            [[143, 63, 229, 255], [48, 93, 38, 255], [70, 185, 64, 255], [233, 152, 66, 255]],
            [[38, 192, 170, 255], [154, 15, 139, 255], [113, 226, 208, 255], [137, 15, 208, 255]],
            [[176, 84, 62, 255], [202, 183, 255, 255], [77, 134, 24, 255], [136, 145, 168, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", None, None, 232, True): np.array(
        [
            [[196, 160, 0, 255], [68, 138, 128, 255], [117, 30, 201, 255], [133, 110, 135, 255]],
            [[208, 58, 130, 255], [35, 84, 43, 255], [58, 168, 64, 255], [60, 139, 212, 255]],
            [[154, 175, 34, 255], [127, 13, 140, 255], [189, 205, 103, 255], [189, 14, 124, 255]],
            [[56, 76, 160, 255], [232, 166, 184, 255], [22, 122, 70, 255], [153, 132, 124, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", None, None, 232, False): np.array(
        [
            [[0, 160, 196, 255], [128, 138, 68, 255], [201, 30, 117, 255], [135, 110, 133, 255]],
            [[130, 58, 208, 255], [43, 84, 35, 255], [64, 168, 58, 255], [212, 139, 60, 255]],
            [[34, 175, 154, 255], [140, 13, 127, 255], [103, 205, 189, 255], [124, 14, 189, 255]],
            [[160, 76, 56, 255], [184, 166, 232, 255], [70, 122, 22, 255], [124, 132, 153, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", None, None, 13333, True): np.array(
        [
            [[255, 255, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", None, None, 13333, False): np.array(
        [
            [[0, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", None, np.uint8, None, True): np.array(
        [
            [[255, 255, 255, 255], [0, 255, 0, 255], [0, 0, 0, 255], [255, 0, 0, 255]],
            [[0, 0, 0, 255], [255, 0, 255, 255], [255, 0, 255, 255], [255, 255, 0, 255]],
            [[255, 255, 255, 255], [0, 0, 255, 255], [255, 255, 0, 255], [255, 0, 0, 255]],
            [[255, 255, 255, 255], [255, 0, 255, 255], [255, 255, 0, 255], [255, 0, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", None, np.uint8, None, False): np.array(
        [
            [[255, 255, 255, 255], [0, 255, 0, 255], [0, 0, 0, 255], [0, 0, 255, 255]],
            [[0, 0, 0, 255], [255, 0, 255, 255], [255, 0, 255, 255], [0, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 0, 0, 255], [0, 255, 255, 255], [0, 0, 255, 255]],
            [[255, 255, 255, 255], [255, 0, 255, 255], [0, 255, 255, 255], [255, 0, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", None, np.uint8, 232, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 255, 0, 255], [0, 255, 0, 255]],
            [[255, 255, 255, 255], [0, 255, 0, 255], [255, 255, 255, 255], [255, 0, 255, 255]],
            [[255, 0, 255, 255], [0, 0, 255, 255], [0, 0, 0, 255], [0, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", None, np.uint8, 232, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 255, 0, 255], [0, 255, 0, 255]],
            [[255, 255, 255, 255], [0, 255, 0, 255], [255, 255, 255, 255], [255, 0, 255, 255]],
            [[255, 0, 255, 255], [255, 0, 0, 255], [0, 0, 0, 255], [255, 255, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", None, np.uint8, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", None, np.uint8, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", None, np.uint16, None, True): np.array(
        [
            [[255, 0, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 255, 255, 255]],
            [[0, 0, 0, 255], [0, 255, 0, 255], [255, 0, 255, 255], [255, 0, 0, 255]],
            [[0, 0, 255, 255], [255, 255, 255, 255], [0, 0, 255, 255], [255, 255, 0, 255]],
            [[255, 255, 0, 255], [0, 0, 0, 255], [255, 0, 255, 255], [255, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", None, np.uint16, None, False): np.array(
        [
            [[255, 0, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 0, 255]],
            [[0, 0, 0, 255], [0, 255, 0, 255], [255, 0, 255, 255], [0, 0, 255, 255]],
            [[255, 0, 0, 255], [255, 255, 255, 255], [255, 0, 0, 255], [0, 255, 255, 255]],
            [[0, 255, 255, 255], [0, 0, 0, 255], [255, 0, 255, 255], [0, 0, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", None, np.uint16, 232, True): np.array(
        [
            [[0, 0, 255, 255], [0, 255, 0, 255], [255, 255, 255, 255], [0, 0, 255, 255]],
            [[0, 0, 0, 255], [0, 255, 0, 255], [0, 255, 0, 255], [255, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 0, 0, 255], [255, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", None, np.uint16, 232, False): np.array(
        [
            [[255, 0, 0, 255], [0, 255, 0, 255], [255, 255, 255, 255], [255, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 255, 0, 255], [0, 255, 0, 255], [0, 0, 255, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 255, 255], [0, 0, 255, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", None, np.uint16, 13333, True): np.array(
        [
            [[0, 0, 255, 255], [255, 0, 0, 255], [255, 0, 0, 255], [255, 0, 255, 255]],
            [[0, 255, 0, 255], [255, 0, 255, 255], [255, 0, 0, 255], [255, 0, 0, 255]],
            [[0, 0, 0, 255], [255, 0, 0, 255], [255, 0, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 0, 255], [0, 0, 0, 255], [255, 0, 0, 255], [255, 0, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", None, np.uint16, 13333, False): np.array(
        [
            [[255, 0, 0, 255], [0, 0, 255, 255], [0, 0, 255, 255], [255, 0, 255, 255]],
            [[0, 255, 0, 255], [255, 0, 255, 255], [0, 0, 255, 255], [0, 0, 255, 255]],
            [[0, 0, 0, 255], [0, 0, 255, 255], [255, 0, 255, 255], [0, 0, 0, 255]],
            [[0, 255, 255, 255], [0, 0, 0, 255], [0, 0, 255, 255], [255, 0, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "SIMPLE", None, None, True): np.array(
        [
            [[255, 255, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "SIMPLE", None, None, False): np.array(
        [
            [[0, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "SIMPLE", None, 232, True): np.array(
        [
            [[255, 255, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "SIMPLE", None, 232, False): np.array(
        [
            [[0, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "SIMPLE", None, 13333, True): np.array(
        [
            [[255, 255, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "SIMPLE", None, 13333, False): np.array(
        [
            [[0, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "SIMPLE", np.uint8, None, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "SIMPLE", np.uint8, None, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "SIMPLE", np.uint8, 232, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "SIMPLE", np.uint8, 232, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "SIMPLE", np.uint8, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "SIMPLE", np.uint8, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "SIMPLE", np.uint16, None, True): np.array(
        [
            [[0, 0, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "SIMPLE", np.uint16, None, False): np.array(
        [
            [[255, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "SIMPLE", np.uint16, 232, True): np.array(
        [
            [[0, 0, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "SIMPLE", np.uint16, 232, False): np.array(
        [
            [[255, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "SIMPLE", np.uint16, 13333, True): np.array(
        [
            [[0, 0, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "SIMPLE", np.uint16, 13333, False): np.array(
        [
            [[255, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "RGB", None, None, True): np.array(
        [
            [[255, 255, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "RGB", None, None, False): np.array(
        [
            [[0, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "RGB", None, 232, True): np.array(
        [
            [[255, 255, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "RGB", None, 232, False): np.array(
        [
            [[0, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "RGB", None, 13333, True): np.array(
        [
            [[255, 255, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "RGB", None, 13333, False): np.array(
        [
            [[0, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "RGB", np.uint8, None, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "RGB", np.uint8, None, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "RGB", np.uint8, 232, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "RGB", np.uint8, 232, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "RGB", np.uint8, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "RGB", np.uint8, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "RGB", np.uint16, None, True): np.array(
        [
            [[0, 0, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "RGB", np.uint16, None, False): np.array(
        [
            [[255, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "RGB", np.uint16, 232, True): np.array(
        [
            [[0, 255, 255, 255], [255, 0, 0, 255], [255, 0, 0, 255], [255, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 255, 0, 255], [255, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 255, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[255, 255, 0, 255], [0, 0, 0, 255], [255, 255, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "RGB", np.uint16, 232, False): np.array(
        [
            [[255, 255, 0, 255], [0, 0, 255, 255], [0, 0, 255, 255], [0, 0, 255, 255]],
            [[0, 0, 0, 255], [0, 255, 0, 255], [0, 0, 255, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 255, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 255, 255, 255], [0, 0, 0, 255], [0, 255, 255, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "RGB", np.uint16, 13333, True): np.array(
        [
            [[0, 0, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "RGB", np.uint16, 13333, False): np.array(
        [
            [[255, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGB", "RGBA", None, None, True): Exception,
    (np.uint16, "RGB", "RGBA", None, None, False): Exception,
    (np.uint16, "RGB", "RGBA", None, 232, True): Exception,
    (np.uint16, "RGB", "RGBA", None, 232, False): Exception,
    (np.uint16, "RGB", "RGBA", None, 13333, True): Exception,
    (np.uint16, "RGB", "RGBA", None, 13333, False): Exception,
    (np.uint16, "RGB", "RGBA", np.uint8, None, True): Exception,
    (np.uint16, "RGB", "RGBA", np.uint8, None, False): Exception,
    (np.uint16, "RGB", "RGBA", np.uint8, 232, True): Exception,
    (np.uint16, "RGB", "RGBA", np.uint8, 232, False): Exception,
    (np.uint16, "RGB", "RGBA", np.uint8, 13333, True): Exception,
    (np.uint16, "RGB", "RGBA", np.uint8, 13333, False): Exception,
    (np.uint16, "RGB", "RGBA", np.uint16, None, True): Exception,
    (np.uint16, "RGB", "RGBA", np.uint16, None, False): Exception,
    (np.uint16, "RGB", "RGBA", np.uint16, 232, True): Exception,
    (np.uint16, "RGB", "RGBA", np.uint16, 232, False): Exception,
    (np.uint16, "RGB", "RGBA", np.uint16, 13333, True): Exception,
    (np.uint16, "RGB", "RGBA", np.uint16, 13333, False): Exception,
    (np.uint16, "RGBA", None, None, None, True): np.array(
        [
            [[50, 158, 115, 181], [0, 142, 130, 220], [188, 89, 183, 104], [223, 47, 120, 45]],
            [[18, 160, 124, 98], [108, 253, 233, 185], [34, 71, 19, 189], [255, 100, 106, 67]],
            [[202, 255, 158, 255], [0, 224, 107, 164], [195, 136, 74, 247], [68, 21, 140, 187]],
            [[239, 107, 145, 28], [212, 155, 227, 49], [81, 220, 126, 0], [52, 57, 229, 189]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", None, None, None, False): np.array(
        [
            [[115, 158, 50, 181], [130, 142, 0, 220], [183, 89, 188, 104], [120, 47, 223, 45]],
            [[124, 160, 18, 98], [233, 253, 108, 185], [19, 71, 34, 189], [106, 100, 255, 67]],
            [[158, 255, 202, 255], [107, 224, 0, 164], [74, 136, 195, 247], [140, 21, 68, 187]],
            [[145, 107, 239, 28], [227, 155, 212, 49], [126, 220, 81, 0], [229, 57, 52, 189]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", None, None, 232, True): np.array(
        [
            [[46, 144, 104, 165], [0, 129, 118, 200], [171, 81, 167, 94], [202, 43, 109, 41]],
            [[17, 145, 113, 89], [98, 230, 212, 169], [31, 64, 17, 172], [232, 91, 96, 61]],
            [[184, 232, 144, 232], [0, 204, 98, 149], [177, 124, 68, 225], [62, 19, 128, 170]],
            [[218, 98, 132, 25], [193, 141, 206, 45], [73, 200, 114, 0], [47, 51, 208, 172]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", None, None, 232, False): np.array(
        [
            [[104, 144, 46, 165], [118, 129, 0, 200], [167, 81, 171, 94], [109, 43, 202, 41]],
            [[113, 145, 17, 89], [212, 230, 98, 169], [17, 64, 31, 172], [96, 91, 232, 61]],
            [[144, 232, 184, 232], [98, 204, 0, 149], [68, 124, 177, 225], [128, 19, 62, 170]],
            [[132, 98, 218, 25], [206, 141, 193, 45], [114, 200, 73, 0], [208, 51, 47, 172]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", None, None, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [0, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [6, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 17], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", None, None, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 6, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 17], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", None, np.uint8, None, True): np.array(
        [
            [[255, 255, 0, 0], [255, 255, 255, 255], [255, 0, 0, 255], [0, 0, 255, 0]],
            [[255, 255, 255, 255], [255, 0, 0, 0], [255, 0, 0, 0], [255, 255, 255, 0]],
            [[255, 255, 255, 255], [255, 255, 0, 255], [0, 255, 255, 0], [255, 0, 255, 0]],
            [[0, 0, 0, 255], [255, 0, 0, 0], [0, 255, 255, 255], [255, 0, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", None, np.uint8, None, False): np.array(
        [
            [[0, 255, 255, 0], [255, 255, 255, 255], [0, 0, 255, 255], [255, 0, 0, 0]],
            [[255, 255, 255, 255], [0, 0, 255, 0], [0, 0, 255, 0], [255, 255, 255, 0]],
            [[255, 255, 255, 255], [0, 255, 255, 255], [255, 255, 0, 0], [255, 0, 255, 0]],
            [[0, 0, 0, 255], [0, 0, 255, 0], [255, 255, 0, 255], [0, 0, 255, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", None, np.uint8, 232, True): np.array(
        [
            [[255, 255, 255, 0], [255, 0, 255, 255], [0, 0, 0, 255], [255, 0, 0, 0]],
            [[0, 0, 0, 0], [255, 255, 255, 0], [0, 255, 0, 255], [255, 0, 255, 0]],
            [[255, 255, 255, 255], [255, 255, 255, 0], [0, 255, 255, 0], [255, 0, 255, 255]],
            [[255, 255, 255, 0], [0, 0, 255, 0], [0, 255, 255, 255], [0, 0, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", None, np.uint8, 232, False): np.array(
        [
            [[255, 255, 255, 0], [255, 0, 255, 255], [0, 0, 0, 255], [0, 0, 255, 0]],
            [[0, 0, 0, 0], [255, 255, 255, 0], [0, 255, 0, 255], [255, 0, 255, 0]],
            [[255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 0, 0], [255, 0, 255, 255]],
            [[255, 255, 255, 0], [255, 0, 0, 0], [255, 255, 0, 255], [255, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", None, np.uint8, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", None, np.uint8, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", None, np.uint16, None, True): np.array(
        [
            [[0, 0, 0, 255], [255, 0, 255, 255], [0, 0, 0, 0], [255, 0, 255, 255]],
            [[255, 255, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 255, 0, 0]],
            [[255, 0, 255, 0], [255, 255, 255, 0], [0, 255, 0, 0], [0, 0, 255, 255]],
            [[0, 0, 0, 0], [0, 0, 0, 255], [0, 0, 0, 0], [0, 255, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", None, np.uint16, None, False): np.array(
        [
            [[0, 0, 0, 255], [255, 0, 255, 255], [0, 0, 0, 0], [255, 0, 255, 255]],
            [[0, 255, 255, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 255, 0, 0]],
            [[255, 0, 255, 0], [255, 255, 255, 0], [0, 255, 0, 0], [255, 0, 0, 255]],
            [[0, 0, 0, 0], [0, 0, 0, 255], [0, 0, 0, 0], [0, 255, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", None, np.uint16, 232, True): np.array(
        [
            [[0, 255, 0, 255], [255, 255, 0, 0], [255, 255, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 255, 0]],
            [[0, 0, 255, 0], [255, 255, 0, 0], [255, 0, 0, 255], [0, 0, 0, 0]],
            [[0, 0, 255, 0], [0, 255, 0, 255], [0, 0, 255, 255], [0, 255, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", None, np.uint16, 232, False): np.array(
        [
            [[0, 255, 0, 255], [0, 255, 255, 0], [0, 255, 255, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [255, 0, 0, 0]],
            [[255, 0, 0, 0], [0, 255, 255, 0], [0, 0, 255, 255], [0, 0, 0, 0]],
            [[255, 0, 0, 0], [0, 255, 0, 255], [255, 0, 0, 255], [0, 255, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", None, np.uint16, 13333, True): np.array(
        [
            [[0, 255, 0, 255], [255, 0, 0, 0], [0, 255, 0, 0], [255, 0, 255, 0]],
            [[0, 0, 255, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 255]],
            [[0, 0, 255, 0], [255, 255, 0, 0], [0, 0, 0, 255], [0, 0, 255, 0]],
            [[0, 0, 0, 0], [0, 0, 255, 0], [0, 0, 0, 0], [0, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", None, np.uint16, 13333, False): np.array(
        [
            [[0, 255, 0, 255], [0, 0, 255, 0], [0, 255, 0, 0], [255, 0, 255, 0]],
            [[255, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 255]],
            [[255, 0, 0, 0], [0, 255, 255, 0], [0, 0, 0, 255], [255, 0, 0, 0]],
            [[0, 0, 0, 0], [255, 0, 0, 0], [0, 0, 0, 0], [255, 255, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "SIMPLE", None, None, True): np.array(
        [
            [[255, 255, 255, 255], [0, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "SIMPLE", None, None, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "SIMPLE", None, 232, True): np.array(
        [
            [[255, 255, 255, 255], [0, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "SIMPLE", None, 232, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "SIMPLE", None, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [0, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "SIMPLE", None, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "SIMPLE", np.uint8, None, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "SIMPLE", np.uint8, None, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "SIMPLE", np.uint8, 232, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "SIMPLE", np.uint8, 232, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "SIMPLE", np.uint8, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "SIMPLE", np.uint8, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "SIMPLE", np.uint16, None, True): np.array(
        [
            [[0, 0, 0, 0], [255, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "SIMPLE", np.uint16, None, False): np.array(
        [
            [[0, 0, 0, 0], [0, 0, 255, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "SIMPLE", np.uint16, 232, True): np.array(
        [
            [[0, 0, 0, 0], [255, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [255, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "SIMPLE", np.uint16, 232, False): np.array(
        [
            [[0, 0, 0, 0], [0, 0, 255, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 255, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "SIMPLE", np.uint16, 13333, True): np.array(
        [
            [[0, 0, 0, 0], [255, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "SIMPLE", np.uint16, 13333, False): np.array(
        [
            [[0, 0, 0, 0], [0, 0, 255, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "RGB", None, None, True): Exception,
    (np.uint16, "RGBA", "RGB", None, None, False): Exception,
    (np.uint16, "RGBA", "RGB", None, 232, True): Exception,
    (np.uint16, "RGBA", "RGB", None, 232, False): Exception,
    (np.uint16, "RGBA", "RGB", None, 13333, True): Exception,
    (np.uint16, "RGBA", "RGB", None, 13333, False): Exception,
    (np.uint16, "RGBA", "RGB", np.uint8, None, True): Exception,
    (np.uint16, "RGBA", "RGB", np.uint8, None, False): Exception,
    (np.uint16, "RGBA", "RGB", np.uint8, 232, True): Exception,
    (np.uint16, "RGBA", "RGB", np.uint8, 232, False): Exception,
    (np.uint16, "RGBA", "RGB", np.uint8, 13333, True): Exception,
    (np.uint16, "RGBA", "RGB", np.uint8, 13333, False): Exception,
    (np.uint16, "RGBA", "RGB", np.uint16, None, True): Exception,
    (np.uint16, "RGBA", "RGB", np.uint16, None, False): Exception,
    (np.uint16, "RGBA", "RGB", np.uint16, 232, True): Exception,
    (np.uint16, "RGBA", "RGB", np.uint16, 232, False): Exception,
    (np.uint16, "RGBA", "RGB", np.uint16, 13333, True): Exception,
    (np.uint16, "RGBA", "RGB", np.uint16, 13333, False): Exception,
    (np.uint16, "RGBA", "RGBA", None, None, True): np.array(
        [
            [[255, 255, 255, 255], [0, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[108, 255, 255, 255], [255, 255, 255, 255], [203, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [0, 255, 255, 255], [255, 255, 255, 255], [255, 139, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "RGBA", None, None, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 108, 255], [255, 255, 255, 255], [255, 255, 203, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 0, 255], [255, 255, 255, 255], [255, 139, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "RGBA", None, 232, True): np.array(
        [
            [[255, 255, 255, 255], [0, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[98, 255, 255, 255], [255, 255, 255, 255], [184, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [0, 255, 255, 255], [255, 255, 255, 255], [255, 126, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "RGBA", None, 232, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 98, 255], [255, 255, 255, 255], [255, 255, 184, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 0, 255], [255, 255, 255, 255], [255, 126, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "RGBA", None, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [0, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [0, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "RGBA", None, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "RGBA", np.uint8, None, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 0, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "RGBA", np.uint8, None, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 0, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "RGBA", np.uint8, 232, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "RGBA", np.uint8, 232, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "RGBA", np.uint8, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "RGBA", np.uint8, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "RGBA", np.uint16, None, True): np.array(
        [
            [[0, 0, 0, 0], [255, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [255, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 255], [0, 0, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "RGBA", np.uint16, None, False): np.array(
        [
            [[0, 0, 0, 0], [0, 0, 255, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 255, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 255], [0, 0, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "RGBA", np.uint16, 232, True): np.array(
        [
            [[255, 255, 255, 0], [255, 0, 255, 0], [0, 0, 255, 0], [0, 0, 255, 0]],
            [[0, 0, 255, 0], [255, 0, 255, 0], [0, 255, 0, 0], [255, 0, 255, 0]],
            [[0, 255, 0, 0], [255, 0, 0, 0], [0, 255, 0, 0], [255, 255, 255, 0]],
            [[0, 0, 255, 0], [0, 0, 255, 255], [0, 255, 255, 255], [255, 0, 255, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "RGBA", np.uint16, 232, False): np.array(
        [
            [[255, 255, 255, 0], [255, 0, 255, 0], [255, 0, 0, 0], [255, 0, 0, 0]],
            [[255, 0, 0, 0], [255, 0, 255, 0], [0, 255, 0, 0], [255, 0, 255, 0]],
            [[0, 255, 0, 0], [0, 0, 255, 0], [0, 255, 0, 0], [255, 255, 255, 0]],
            [[255, 0, 0, 0], [255, 0, 0, 255], [255, 255, 0, 255], [255, 0, 255, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "RGBA", np.uint16, 13333, True): np.array(
        [
            [[255, 255, 0, 0], [255, 0, 0, 0], [0, 255, 0, 0], [0, 255, 0, 0]],
            [[0, 255, 0, 0], [255, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [255, 0, 0, 0], [255, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 255], [0, 0, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.uint16, "RGBA", "RGBA", np.uint16, 13333, False): np.array(
        [
            [[0, 255, 255, 0], [0, 0, 255, 0], [0, 255, 0, 0], [0, 255, 0, 0]],
            [[0, 255, 0, 0], [0, 0, 255, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 255, 0], [0, 0, 255, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 255], [0, 0, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "SIMPLE", None, None, True): np.array(
        [
            [[255, 255, 255, 255], [136, 136, 136, 255], [0, 0, 0, 0], [225, 225, 225, 255]],
            [[216, 216, 216, 255], [224, 224, 224, 255], [255, 255, 255, 255], [16, 16, 16, 255]],
            [[252, 252, 252, 255], [216, 216, 216, 255], [42, 42, 42, 255], [252, 252, 252, 255]],
            [[18, 18, 18, 255], [169, 169, 169, 255], [223, 223, 223, 255], [252, 252, 252, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "SIMPLE", None, None, False): np.array(
        [
            [[255, 255, 255, 255], [136, 136, 136, 255], [0, 0, 0, 0], [225, 225, 225, 255]],
            [[216, 216, 216, 255], [224, 224, 224, 255], [255, 255, 255, 255], [16, 16, 16, 255]],
            [[252, 252, 252, 255], [216, 216, 216, 255], [42, 42, 42, 255], [252, 252, 252, 255]],
            [[18, 18, 18, 255], [169, 169, 169, 255], [223, 223, 223, 255], [252, 252, 252, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "SIMPLE", None, 232, True): np.array(
        [
            [[255, 255, 255, 255], [124, 124, 124, 255], [0, 0, 0, 0], [205, 205, 205, 255]],
            [[197, 197, 197, 255], [204, 204, 204, 255], [232, 232, 232, 255], [15, 15, 15, 255]],
            [[230, 230, 230, 255], [196, 196, 196, 255], [38, 38, 38, 255], [229, 229, 229, 255]],
            [[16, 16, 16, 255], [153, 153, 153, 255], [203, 203, 203, 255], [229, 229, 229, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "SIMPLE", None, 232, False): np.array(
        [
            [[255, 255, 255, 255], [124, 124, 124, 255], [0, 0, 0, 0], [205, 205, 205, 255]],
            [[197, 197, 197, 255], [204, 204, 204, 255], [232, 232, 232, 255], [15, 15, 15, 255]],
            [[230, 230, 230, 255], [196, 196, 196, 255], [38, 38, 38, 255], [229, 229, 229, 255]],
            [[16, 16, 16, 255], [153, 153, 153, 255], [203, 203, 203, 255], [229, 229, 229, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "SIMPLE", None, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 0], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "SIMPLE", None, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 0], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "SIMPLE", np.uint8, None, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 0], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "SIMPLE", np.uint8, None, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 0], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "SIMPLE", np.uint8, 232, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 0], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "SIMPLE", np.uint8, 232, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 0], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "SIMPLE", np.uint8, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "SIMPLE", np.uint8, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "SIMPLE", np.uint16, None, True): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 0], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "SIMPLE", np.uint16, None, False): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 0], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "SIMPLE", np.uint16, 232, True): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 0], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "SIMPLE", np.uint16, 232, False): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 0], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "SIMPLE", np.uint16, 13333, True): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 0], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "SIMPLE", np.uint16, 13333, False): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 255, 255, 0], [0, 0, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "RGB", None, None, True): Exception,
    (np.float32, "2D", "RGB", None, None, False): Exception,
    (np.float32, "2D", "RGB", None, 232, True): Exception,
    (np.float32, "2D", "RGB", None, 232, False): Exception,
    (np.float32, "2D", "RGB", None, 13333, True): Exception,
    (np.float32, "2D", "RGB", None, 13333, False): Exception,
    (np.float32, "2D", "RGB", np.uint8, None, True): Exception,
    (np.float32, "2D", "RGB", np.uint8, None, False): Exception,
    (np.float32, "2D", "RGB", np.uint8, 232, True): Exception,
    (np.float32, "2D", "RGB", np.uint8, 232, False): Exception,
    (np.float32, "2D", "RGB", np.uint8, 13333, True): Exception,
    (np.float32, "2D", "RGB", np.uint8, 13333, False): Exception,
    (np.float32, "2D", "RGB", np.uint16, None, True): Exception,
    (np.float32, "2D", "RGB", np.uint16, None, False): Exception,
    (np.float32, "2D", "RGB", np.uint16, 232, True): Exception,
    (np.float32, "2D", "RGB", np.uint16, 232, False): Exception,
    (np.float32, "2D", "RGB", np.uint16, 13333, True): Exception,
    (np.float32, "2D", "RGB", np.uint16, 13333, False): Exception,
    (np.float32, "2D", "RGBA", None, None, True): np.array(
        [
            [[255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 0], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [1, 1, 1, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "RGBA", None, None, False): np.array(
        [
            [[255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 0], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [1, 1, 1, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "RGBA", None, 232, True): np.array(
        [
            [[255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 0], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "RGBA", None, 232, False): np.array(
        [
            [[255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 0], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "RGBA", None, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 0], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [52, 52, 52, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [8, 8, 8, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [45, 45, 45, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "RGBA", None, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [0, 0, 0, 255], [0, 0, 0, 0], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [52, 52, 52, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [8, 8, 8, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [45, 45, 45, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "RGBA", np.uint8, None, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "RGBA", np.uint8, None, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "RGBA", np.uint8, 232, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "RGBA", np.uint8, 232, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "RGBA", np.uint8, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "RGBA", np.uint8, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "RGBA", np.uint16, None, True): np.array(
        [
            [[0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "RGBA", np.uint16, None, False): np.array(
        [
            [[0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "RGBA", np.uint16, 232, True): np.array(
        [
            [[0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "RGBA", np.uint16, 232, False): np.array(
        [
            [[0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "RGBA", np.uint16, 13333, True): np.array(
        [
            [[0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "2D", "RGBA", np.uint16, 13333, False): np.array(
        [
            [[0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "SIMPLE", None, None, True): np.array(
        [
            [[59, 99, 32, 255], [103, 231, 72, 255], [233, 117, 43, 255], [74, 249, 184, 255]],
            [[0, 185, 137, 0], [232, 72, 187, 255], [86, 56, 249, 255], [35, 88, 101, 255]],
            [[217, 49, 0, 255], [123, 12, 152, 255], [255, 83, 37, 255], [95, 90, 22, 255]],
            [[177, 133, 125, 255], [238, 36, 174, 255], [70, 177, 17, 255], [67, 65, 219, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "SIMPLE", None, None, False): np.array(
        [
            [[32, 99, 59, 255], [72, 231, 103, 255], [43, 117, 233, 255], [184, 249, 74, 255]],
            [[137, 185, 0, 0], [187, 72, 232, 255], [249, 56, 86, 255], [101, 88, 35, 255]],
            [[0, 49, 217, 255], [152, 12, 123, 255], [37, 83, 255, 255], [22, 90, 95, 255]],
            [[125, 133, 177, 255], [174, 36, 238, 255], [17, 177, 70, 255], [219, 65, 67, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "SIMPLE", None, 232, True): np.array(
        [
            [[54, 90, 29, 255], [94, 210, 65, 255], [212, 106, 39, 255], [68, 227, 167, 255]],
            [[0, 169, 125, 0], [211, 65, 170, 255], [78, 51, 227, 255], [32, 80, 91, 255]],
            [[197, 44, 0, 255], [112, 11, 138, 255], [255, 75, 34, 255], [86, 82, 20, 255]],
            [[161, 121, 113, 255], [216, 33, 158, 255], [64, 161, 15, 255], [61, 59, 199, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "SIMPLE", None, 232, False): np.array(
        [
            [[29, 90, 54, 255], [65, 210, 94, 255], [39, 106, 212, 255], [167, 227, 68, 255]],
            [[125, 169, 0, 0], [170, 65, 211, 255], [227, 51, 78, 255], [91, 80, 32, 255]],
            [[0, 44, 197, 255], [138, 11, 112, 255], [34, 75, 255, 255], [20, 82, 86, 255]],
            [[113, 121, 161, 255], [158, 33, 216, 255], [15, 161, 64, 255], [199, 59, 61, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "SIMPLE", None, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[0, 255, 255, 0], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "SIMPLE", None, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 0, 0], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[0, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "SIMPLE", np.uint8, None, True): np.array(
        [
            [[0, 0, 255, 255], [0, 0, 255, 255], [0, 0, 0, 255], [255, 0, 255, 255]],
            [[255, 0, 0, 0], [255, 255, 0, 255], [255, 255, 0, 255], [0, 255, 0, 255]],
            [[0, 0, 255, 255], [0, 255, 255, 255], [255, 0, 0, 255], [0, 255, 255, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [255, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "SIMPLE", np.uint8, None, False): np.array(
        [
            [[255, 0, 0, 255], [255, 0, 0, 255], [0, 0, 0, 255], [255, 0, 255, 255]],
            [[0, 0, 255, 0], [0, 255, 255, 255], [0, 255, 255, 255], [0, 255, 0, 255]],
            [[255, 0, 0, 255], [255, 255, 0, 255], [0, 0, 255, 255], [255, 255, 0, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [0, 0, 255, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "SIMPLE", np.uint8, 232, True): np.array(
        [
            [[255, 255, 0, 255], [255, 255, 0, 255], [255, 255, 0, 255], [255, 0, 0, 255]],
            [[255, 0, 0, 0], [0, 0, 255, 255], [255, 0, 0, 255], [255, 255, 0, 255]],
            [[0, 255, 255, 255], [255, 0, 255, 255], [255, 0, 255, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [255, 0, 255, 255], [255, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "SIMPLE", np.uint8, 232, False): np.array(
        [
            [[0, 255, 255, 255], [0, 255, 255, 255], [0, 255, 255, 255], [0, 0, 255, 255]],
            [[0, 0, 255, 0], [255, 0, 0, 255], [0, 0, 255, 255], [0, 255, 255, 255]],
            [[255, 255, 0, 255], [255, 0, 255, 255], [255, 0, 255, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [255, 0, 255, 255], [0, 0, 255, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "SIMPLE", np.uint8, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 0], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "SIMPLE", np.uint8, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 0], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "SIMPLE", np.uint16, None, True): np.array(
        [
            [[0, 0, 255, 255], [255, 0, 0, 255], [255, 0, 255, 255], [0, 0, 0, 255]],
            [[255, 0, 255, 0], [0, 0, 255, 255], [255, 255, 0, 255], [0, 0, 255, 255]],
            [[255, 0, 255, 255], [255, 0, 0, 255], [0, 0, 255, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [0, 0, 255, 255], [0, 255, 0, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "SIMPLE", np.uint16, None, False): np.array(
        [
            [[255, 0, 0, 255], [0, 0, 255, 255], [255, 0, 255, 255], [0, 0, 0, 255]],
            [[255, 0, 255, 0], [255, 0, 0, 255], [0, 255, 255, 255], [255, 0, 0, 255]],
            [[255, 0, 255, 255], [0, 0, 255, 255], [255, 0, 0, 255], [255, 255, 255, 255]],
            [[0, 0, 0, 255], [255, 0, 0, 255], [0, 255, 0, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "SIMPLE", np.uint16, 232, True): np.array(
        [
            [[255, 255, 0, 255], [0, 255, 0, 255], [0, 0, 255, 255], [0, 0, 0, 255]],
            [[255, 0, 0, 0], [0, 0, 0, 255], [255, 255, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 255, 255], [0, 0, 255, 255], [0, 255, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [255, 255, 0, 255], [0, 0, 255, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "SIMPLE", np.uint16, 232, False): np.array(
        [
            [[0, 255, 255, 255], [0, 255, 0, 255], [255, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 255, 0], [0, 0, 0, 255], [0, 255, 255, 255], [0, 0, 0, 255]],
            [[255, 0, 0, 255], [255, 0, 0, 255], [0, 255, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 255, 255, 255], [255, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "SIMPLE", np.uint16, 13333, True): np.array(
        [
            [[255, 255, 0, 255], [255, 255, 0, 255], [255, 255, 0, 255], [255, 0, 0, 255]],
            [[255, 255, 0, 0], [0, 0, 0, 255], [255, 0, 0, 255], [255, 0, 255, 255]],
            [[0, 255, 255, 255], [255, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[255, 0, 255, 255], [0, 0, 0, 255], [255, 0, 255, 255], [255, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "SIMPLE", np.uint16, 13333, False): np.array(
        [
            [[0, 255, 255, 255], [0, 255, 255, 255], [0, 255, 255, 255], [0, 0, 255, 255]],
            [[0, 255, 255, 0], [0, 0, 0, 255], [0, 0, 255, 255], [255, 0, 255, 255]],
            [[255, 255, 0, 255], [0, 0, 255, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[255, 0, 255, 255], [0, 0, 0, 255], [255, 0, 255, 255], [0, 0, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "RGB", None, None, True): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 0], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "RGB", None, None, False): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 0], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 255, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "RGB", None, 232, True): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 0], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [255, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "RGB", None, 232, False): np.array(
        [
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 0], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 255, 255], [0, 0, 0, 255]],
            [[0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255], [0, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "RGB", None, 13333, True): np.array(
        [
            [[12, 0, 0, 255], [21, 0, 0, 255], [47, 0, 0, 255], [15, 0, 0, 255]],
            [[0, 0, 0, 0], [47, 0, 0, 255], [17, 0, 0, 255], [7, 0, 0, 255]],
            [[44, 0, 0, 255], [25, 0, 0, 255], [255, 0, 0, 255], [19, 0, 0, 255]],
            [[36, 0, 0, 255], [48, 0, 0, 255], [14, 0, 0, 255], [13, 0, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "RGB", None, 13333, False): np.array(
        [
            [[0, 0, 12, 255], [0, 0, 21, 255], [0, 0, 47, 255], [0, 0, 15, 255]],
            [[0, 0, 0, 0], [0, 0, 47, 255], [0, 0, 17, 255], [0, 0, 7, 255]],
            [[0, 0, 44, 255], [0, 0, 25, 255], [0, 0, 255, 255], [0, 0, 19, 255]],
            [[0, 0, 36, 255], [0, 0, 48, 255], [0, 0, 14, 255], [0, 0, 13, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "RGB", np.uint8, None, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 0], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "RGB", np.uint8, None, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 0], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "RGB", np.uint8, 232, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 0], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "RGB", np.uint8, 232, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 0], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "RGB", np.uint8, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [0, 255, 255, 255], [0, 255, 255, 255], [0, 255, 255, 255]],
            [[255, 255, 255, 0], [0, 255, 255, 255], [0, 255, 255, 255], [0, 255, 255, 255]],
            [[255, 255, 255, 255], [0, 255, 255, 255], [255, 255, 255, 255], [0, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "RGB", np.uint8, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 0, 255], [255, 255, 0, 255], [255, 255, 0, 255]],
            [[255, 255, 255, 0], [255, 255, 0, 255], [255, 255, 0, 255], [255, 255, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 0, 255], [255, 255, 255, 255], [255, 255, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "RGB", np.uint16, None, True): np.array(
        [
            [[0, 255, 255, 255], [0, 255, 255, 255], [0, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 0], [255, 255, 255, 255], [0, 255, 255, 255], [0, 255, 255, 255]],
            [[255, 255, 255, 255], [0, 255, 255, 255], [0, 255, 255, 255], [0, 255, 255, 255]],
            [[0, 255, 255, 255], [0, 255, 255, 255], [0, 255, 255, 255], [0, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "RGB", np.uint16, None, False): np.array(
        [
            [[255, 255, 0, 255], [255, 255, 0, 255], [255, 255, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 0], [255, 255, 255, 255], [255, 255, 0, 255], [255, 255, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 0, 255], [255, 255, 0, 255], [255, 255, 0, 255]],
            [[255, 255, 0, 255], [255, 255, 0, 255], [255, 255, 0, 255], [255, 255, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "RGB", np.uint16, 232, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 0], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "RGB", np.uint16, 232, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 0], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "RGB", np.uint16, 13333, True): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 0], [0, 255, 255, 255], [0, 255, 255, 255], [0, 255, 255, 255]],
            [[0, 255, 255, 255], [0, 255, 255, 255], [0, 255, 255, 255], [0, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 255, 255, 255], [0, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "RGB", np.uint16, 13333, False): np.array(
        [
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 0], [255, 255, 0, 255], [255, 255, 0, 255], [255, 255, 0, 255]],
            [[255, 255, 0, 255], [255, 255, 0, 255], [255, 255, 0, 255], [255, 255, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 0, 255], [255, 255, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGB", "RGBA", None, None, True): Exception,
    (np.float32, "RGB", "RGBA", None, None, False): Exception,
    (np.float32, "RGB", "RGBA", None, 232, True): Exception,
    (np.float32, "RGB", "RGBA", None, 232, False): Exception,
    (np.float32, "RGB", "RGBA", None, 13333, True): Exception,
    (np.float32, "RGB", "RGBA", None, 13333, False): Exception,
    (np.float32, "RGB", "RGBA", np.uint8, None, True): Exception,
    (np.float32, "RGB", "RGBA", np.uint8, None, False): Exception,
    (np.float32, "RGB", "RGBA", np.uint8, 232, True): Exception,
    (np.float32, "RGB", "RGBA", np.uint8, 232, False): Exception,
    (np.float32, "RGB", "RGBA", np.uint8, 13333, True): Exception,
    (np.float32, "RGB", "RGBA", np.uint8, 13333, False): Exception,
    (np.float32, "RGB", "RGBA", np.uint16, None, True): Exception,
    (np.float32, "RGB", "RGBA", np.uint16, None, False): Exception,
    (np.float32, "RGB", "RGBA", np.uint16, 232, True): Exception,
    (np.float32, "RGB", "RGBA", np.uint16, 232, False): Exception,
    (np.float32, "RGB", "RGBA", np.uint16, 13333, True): Exception,
    (np.float32, "RGB", "RGBA", np.uint16, 13333, False): Exception,
    (np.float32, "RGBA", "SIMPLE", None, None, True): np.array(
        [
            [[248, 159, 5, 0], [217, 237, 151, 255], [102, 203, 57, 44], [244, 177, 73, 196]],
            [[29, 189, 17, 250], [168, 188, 84, 208], [198, 134, 144, 81], [208, 209, 210, 126]],
            [[8, 212, 80, 241], [159, 153, 128, 236], [0, 0, 7, 0], [249, 140, 61, 88]],
            [[175, 255, 216, 90], [42, 76, 23, 172], [64, 65, 205, 254], [0, 255, 148, 117]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "SIMPLE", None, None, False): np.array(
        [
            [[5, 159, 248, 0], [151, 237, 217, 255], [57, 203, 102, 44], [73, 177, 244, 196]],
            [[17, 189, 29, 250], [84, 188, 168, 208], [144, 134, 198, 81], [210, 209, 208, 126]],
            [[80, 212, 8, 241], [128, 153, 159, 236], [7, 0, 0, 0], [61, 140, 249, 88]],
            [[216, 255, 175, 90], [23, 76, 42, 172], [205, 65, 64, 254], [148, 255, 0, 117]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "SIMPLE", None, 232, True): np.array(
        [
            [[225, 145, 5, 0], [197, 215, 138, 255], [93, 185, 51, 40], [222, 161, 66, 178]],
            [[26, 172, 15, 227], [153, 171, 76, 189], [181, 122, 131, 73], [189, 190, 191, 114]],
            [[7, 193, 73, 219], [144, 139, 116, 215], [0, 0, 6, 0], [227, 127, 55, 80]],
            [[159, 255, 197, 82], [38, 69, 21, 157], [58, 59, 186, 231], [0, 232, 134, 107]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "SIMPLE", None, 232, False): np.array(
        [
            [[5, 145, 225, 0], [138, 215, 197, 255], [51, 185, 93, 40], [66, 161, 222, 178]],
            [[15, 172, 26, 227], [76, 171, 153, 189], [131, 122, 181, 73], [191, 190, 189, 114]],
            [[73, 193, 7, 219], [116, 139, 144, 215], [6, 0, 0, 0], [55, 127, 227, 80]],
            [[197, 255, 159, 82], [21, 69, 38, 157], [186, 59, 58, 231], [134, 232, 0, 107]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "SIMPLE", None, 13333, True): np.array(
        [
            [[255, 255, 255, 0], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 0, 255, 0], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [0, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "SIMPLE", None, 13333, False): np.array(
        [
            [[255, 255, 255, 0], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 0, 0, 0], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "SIMPLE", np.uint8, None, True): np.array(
        [
            [[255, 0, 0, 0], [0, 0, 0, 255], [255, 0, 0, 255], [255, 0, 0, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 0, 255, 255]],
            [[255, 255, 255, 0], [0, 0, 255, 255], [255, 255, 0, 0], [0, 255, 0, 255]],
            [[0, 255, 255, 255], [255, 255, 0, 255], [255, 0, 0, 255], [255, 255, 255, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "SIMPLE", np.uint8, None, False): np.array(
        [
            [[0, 0, 255, 0], [0, 0, 0, 255], [0, 0, 255, 255], [0, 0, 255, 255]],
            [[0, 0, 0, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 0, 255, 255]],
            [[255, 255, 255, 0], [255, 0, 0, 255], [0, 255, 255, 0], [0, 255, 0, 255]],
            [[255, 255, 0, 255], [0, 255, 255, 255], [0, 0, 255, 255], [255, 255, 255, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "SIMPLE", np.uint8, 232, True): np.array(
        [
            [[0, 0, 0, 0], [0, 0, 255, 255], [0, 0, 0, 255], [255, 0, 255, 255]],
            [[255, 255, 0, 0], [0, 0, 255, 0], [0, 255, 0, 0], [0, 255, 0, 255]],
            [[0, 0, 0, 0], [255, 0, 255, 0], [255, 255, 255, 0], [0, 0, 0, 255]],
            [[0, 255, 0, 255], [255, 0, 0, 0], [255, 0, 255, 0], [255, 255, 255, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "SIMPLE", np.uint8, 232, False): np.array(
        [
            [[0, 0, 0, 0], [255, 0, 0, 255], [0, 0, 0, 255], [255, 0, 255, 255]],
            [[0, 255, 255, 0], [255, 0, 0, 0], [0, 255, 0, 0], [0, 255, 0, 255]],
            [[0, 0, 0, 0], [255, 0, 255, 0], [255, 255, 255, 0], [0, 0, 0, 255]],
            [[0, 255, 0, 255], [0, 0, 255, 0], [255, 0, 255, 0], [255, 255, 255, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "SIMPLE", np.uint8, 13333, True): np.array(
        [
            [[255, 255, 255, 0], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "SIMPLE", np.uint8, 13333, False): np.array(
        [
            [[255, 255, 255, 0], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "SIMPLE", np.uint16, None, True): np.array(
        [
            [[0, 0, 255, 0], [0, 255, 255, 0], [255, 255, 0, 255], [0, 0, 0, 255]],
            [[0, 255, 0, 255], [255, 255, 0, 255], [0, 0, 255, 255], [255, 0, 0, 255]],
            [[0, 0, 255, 0], [255, 0, 0, 0], [255, 255, 255, 0], [0, 255, 255, 0]],
            [[255, 0, 255, 0], [255, 0, 0, 255], [255, 0, 0, 0], [255, 0, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "SIMPLE", np.uint16, None, False): np.array(
        [
            [[255, 0, 0, 0], [255, 255, 0, 0], [0, 255, 255, 255], [0, 0, 0, 255]],
            [[0, 255, 0, 255], [0, 255, 255, 255], [255, 0, 0, 255], [0, 0, 255, 255]],
            [[255, 0, 0, 0], [0, 0, 255, 0], [255, 255, 255, 0], [255, 255, 0, 0]],
            [[255, 0, 255, 0], [0, 0, 255, 255], [0, 0, 255, 0], [0, 0, 255, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "SIMPLE", np.uint16, 232, True): np.array(
        [
            [[255, 0, 0, 0], [0, 0, 255, 0], [255, 0, 255, 0], [255, 0, 255, 0]],
            [[0, 0, 255, 0], [255, 255, 0, 255], [0, 0, 0, 0], [255, 0, 0, 255]],
            [[0, 0, 0, 255], [255, 0, 0, 0], [255, 255, 255, 0], [0, 0, 0, 0]],
            [[255, 0, 0, 0], [0, 255, 255, 0], [0, 0, 255, 255], [255, 0, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "SIMPLE", np.uint16, 232, False): np.array(
        [
            [[0, 0, 255, 0], [255, 0, 0, 0], [255, 0, 255, 0], [255, 0, 255, 0]],
            [[255, 0, 0, 0], [0, 255, 255, 255], [0, 0, 0, 0], [0, 0, 255, 255]],
            [[0, 0, 0, 255], [0, 0, 255, 0], [255, 255, 255, 0], [0, 0, 0, 0]],
            [[0, 0, 255, 0], [255, 255, 0, 0], [255, 0, 0, 255], [0, 0, 255, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "SIMPLE", np.uint16, 13333, True): np.array(
        [
            [[255, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 255], [0, 0, 255, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [255, 0, 0, 0], [0, 0, 255, 255]],
            [[0, 0, 0, 0], [0, 0, 255, 255], [255, 255, 0, 0], [255, 255, 0, 0]],
            [[255, 0, 0, 0], [0, 255, 0, 0], [0, 0, 0, 255], [255, 0, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "SIMPLE", np.uint16, 13333, False): np.array(
        [
            [[0, 0, 255, 0], [0, 0, 0, 0], [0, 0, 0, 255], [255, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 255, 0], [255, 0, 0, 255]],
            [[0, 0, 0, 0], [255, 0, 0, 255], [0, 255, 255, 0], [0, 255, 255, 0]],
            [[0, 0, 255, 0], [0, 255, 0, 0], [0, 0, 0, 255], [0, 0, 255, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "RGB", None, None, True): Exception,
    (np.float32, "RGBA", "RGB", None, None, False): Exception,
    (np.float32, "RGBA", "RGB", None, 232, True): Exception,
    (np.float32, "RGBA", "RGB", None, 232, False): Exception,
    (np.float32, "RGBA", "RGB", None, 13333, True): Exception,
    (np.float32, "RGBA", "RGB", None, 13333, False): Exception,
    (np.float32, "RGBA", "RGB", np.uint8, None, True): Exception,
    (np.float32, "RGBA", "RGB", np.uint8, None, False): Exception,
    (np.float32, "RGBA", "RGB", np.uint8, 232, True): Exception,
    (np.float32, "RGBA", "RGB", np.uint8, 232, False): Exception,
    (np.float32, "RGBA", "RGB", np.uint8, 13333, True): Exception,
    (np.float32, "RGBA", "RGB", np.uint8, 13333, False): Exception,
    (np.float32, "RGBA", "RGB", np.uint16, None, True): Exception,
    (np.float32, "RGBA", "RGB", np.uint16, None, False): Exception,
    (np.float32, "RGBA", "RGB", np.uint16, 232, True): Exception,
    (np.float32, "RGBA", "RGB", np.uint16, 232, False): Exception,
    (np.float32, "RGBA", "RGB", np.uint16, 13333, True): Exception,
    (np.float32, "RGBA", "RGB", np.uint16, 13333, False): Exception,
    (np.float32, "RGBA", "RGBA", None, None, True): np.array(
        [
            [[0, 0, 0, 0], [0, 0, 0, 255], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 255, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "RGBA", None, None, False): np.array(
        [
            [[0, 0, 0, 0], [0, 0, 0, 255], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 255, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "RGBA", None, 232, True): np.array(
        [
            [[0, 0, 0, 0], [0, 0, 0, 255], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 255, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "RGBA", None, 232, False): np.array(
        [
            [[0, 0, 0, 0], [0, 0, 0, 255], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
            [[0, 255, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0], [0, 0, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "RGBA", None, 13333, True): np.array(
        [
            [[0, 0, 1, 0], [0, 0, 31, 255], [0, 0, 11, 0], [0, 0, 15, 0]],
            [[0, 0, 3, 0], [0, 0, 17, 0], [0, 0, 29, 0], [0, 0, 43, 0]],
            [[0, 0, 16, 0], [0, 0, 26, 0], [0, 0, 1, 0], [0, 0, 12, 0]],
            [[0, 255, 44, 0], [0, 0, 4, 0], [0, 0, 42, 0], [0, 0, 30, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "RGBA", None, 13333, False): np.array(
        [
            [[1, 0, 0, 0], [31, 0, 0, 255], [11, 0, 0, 0], [15, 0, 0, 0]],
            [[3, 0, 0, 0], [17, 0, 0, 0], [29, 0, 0, 0], [43, 0, 0, 0]],
            [[16, 0, 0, 0], [26, 0, 0, 0], [1, 0, 0, 0], [12, 0, 0, 0]],
            [[44, 255, 0, 0], [4, 0, 0, 0], [42, 0, 0, 0], [30, 0, 0, 0]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "RGBA", np.uint8, None, True): np.array(
        [
            [[255, 255, 255, 0], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "RGBA", np.uint8, None, False): np.array(
        [
            [[255, 255, 255, 0], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "RGBA", np.uint8, 232, True): np.array(
        [
            [[255, 255, 255, 0], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "RGBA", np.uint8, 232, False): np.array(
        [
            [[255, 255, 255, 0], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "RGBA", np.uint8, 13333, True): np.array(
        [
            [[255, 255, 0, 0], [255, 255, 0, 255], [255, 255, 0, 255], [255, 255, 0, 255]],
            [[255, 255, 0, 255], [255, 255, 0, 255], [255, 255, 0, 255], [255, 255, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 0, 0], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "RGBA", np.uint8, 13333, False): np.array(
        [
            [[0, 255, 255, 0], [0, 255, 255, 255], [0, 255, 255, 255], [0, 255, 255, 255]],
            [[0, 255, 255, 255], [0, 255, 255, 255], [0, 255, 255, 255], [0, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 255, 255, 0], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "RGBA", np.uint16, None, True): np.array(
        [
            [[255, 255, 0, 0], [255, 255, 0, 0], [255, 255, 255, 255], [255, 255, 0, 255]],
            [[255, 255, 0, 255], [255, 255, 0, 255], [255, 255, 0, 255], [255, 255, 0, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 0, 0], [255, 255, 0, 255]],
            [[255, 0, 0, 255], [255, 255, 255, 255], [255, 255, 0, 255], [255, 255, 0, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "RGBA", np.uint16, None, False): np.array(
        [
            [[0, 255, 255, 0], [0, 255, 255, 0], [255, 255, 255, 255], [0, 255, 255, 255]],
            [[0, 255, 255, 255], [0, 255, 255, 255], [0, 255, 255, 255], [0, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [0, 255, 255, 0], [0, 255, 255, 255]],
            [[0, 0, 255, 255], [255, 255, 255, 255], [0, 255, 255, 255], [0, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "RGBA", np.uint16, 232, True): np.array(
        [
            [[255, 255, 255, 0], [255, 255, 255, 0], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
            [[255, 0, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "RGBA", np.uint16, 232, False): np.array(
        [
            [[255, 255, 255, 0], [255, 255, 255, 0], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 0], [255, 255, 255, 255]],
            [[255, 0, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "RGBA", np.uint16, 13333, True): np.array(
        [
            [[255, 255, 0, 0], [255, 255, 0, 0], [255, 255, 0, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [255, 255, 0, 255], [255, 255, 0, 255], [255, 255, 0, 255]],
            [[255, 255, 0, 255], [255, 255, 0, 255], [255, 255, 0, 0], [255, 255, 255, 255]],
            [[255, 0, 0, 255], [255, 255, 0, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
    (np.float32, "RGBA", "RGBA", np.uint16, 13333, False): np.array(
        [
            [[0, 255, 255, 0], [0, 255, 255, 0], [0, 255, 255, 255], [255, 255, 255, 255]],
            [[255, 255, 255, 255], [0, 255, 255, 255], [0, 255, 255, 255], [0, 255, 255, 255]],
            [[0, 255, 255, 255], [0, 255, 255, 255], [0, 255, 255, 0], [255, 255, 255, 255]],
            [[0, 0, 255, 255], [0, 255, 255, 255], [255, 255, 255, 255], [255, 255, 255, 255]],
        ],
        dtype=np.uint8,
    ),
}


def _error_description(output, test_case):
    return (
        f"'{test_case['name']}' output does not match expectations\n"
        + f"\tExpected:\n{test_case['expected_output']!r}\n\n"
        + f"\tReceived:\n{output!r}\n"
    )


def _do_something_for_every_combo(func):
    for dtype in [np.uint8, np.uint16, np.float32]:
        for in_fmt in ["2D", "RGB", "RGBA"]:
            data = INPUTS[(dtype, in_fmt)]
            for levels_name in [None, "SIMPLE", "RGB", "RGBA"]:
                levels = LEVELS.get(levels_name, None)
                if dtype == np.float32 and levels_name is None:
                    continue
                for lut_type in [None, np.uint8, np.uint16]:
                    lut = LUTS.get(lut_type, None)
                    for scale in [None, 232, 13333]:
                        for use_rgba in [True, False]:
                            key = (dtype, in_fmt, levels_name, lut_type, scale, use_rgba)
                            func(data, key, levels, lut, scale, use_rgba)


def _makeARGB(*args, **kwds):
    img, alpha = real_makeARGB(*args, **kwds)
    if kwds.get('useRGBA'):  # endian independent
        out = img
    elif sys.byteorder == 'little':  # little-endian ARGB32 to B,G,R,A
        out = img
    else:  # big-endian ARGB32 to B,G,R,A
        out = img[..., [3, 2, 1, 0]]
    return out, alpha


def save_reference():
    """
    This saves the output (or exception type) of running makeARGB for every combo of arguments. The
    output isn't fit for immediate inclusion in this file as EXPECTED_OUTPUTS, and needs some replace-all
    work.
    """
    with open("_unformatted_expected_outputs_", "w") as tmp_file:

        def write_expectation_to_file(data, key, levels, lut, scale, use_rgba):
            try:
                output, alpha = _makeARGB(data, lut=lut, levels=levels, scale=scale, useRGBA=use_rgba)
            except Exception as e:
                tmp_file.write(f"{key!r}: {type(e)}\n")
            else:
                tmp_file.write(f"{key!r}: {output!r},\n")

        _do_something_for_every_combo(write_expectation_to_file)


def test_makeARGB_against_generated_references():
    def assert_correct(data, key, levels, lut, scale, use_rgba):
        expectation = EXPECTED_OUTPUTS[key]
        if isinstance(expectation, type) and issubclass(expectation, Exception):
            try:
                _makeARGB(data, lut=lut, levels=levels, scale=scale, useRGBA=use_rgba)
            except Exception as e:
                assert expectation == type(e)
            else:
                assert False, f"makeARGB({key!r}) was supposed to raise {expectation} but didn't raise anything."
        else:
            output, alpha = _makeARGB(data, lut=lut, levels=levels, scale=scale, useRGBA=use_rgba)
            assert (
                output == expectation
            ).all(), f"Incorrect _makeARGB({key!r}) output! Expected:\n{expectation!r}\n  Got:\n{output!r}"

    _do_something_for_every_combo(assert_correct)


@pytest.mark.skipif(cupy is None, reason="CuPy unavailable to test")
def test_cupy_makeARGB_against_generated_references():
    prev_setting = getConfigOption("useCupy")
    try:
        setConfigOption("useCupy", True)

        cupy = getCupy()

        def assert_cupy_correct(data, key, levels, lut, scale, use_rgba):
            data = cupy.asarray(data)
            if lut is not None:
                lut = cupy.asarray(lut)
            expectation = EXPECTED_OUTPUTS[key]
            if isinstance(expectation, type) and issubclass(expectation, Exception):
                try:
                    _makeARGB(data, lut=lut, levels=levels, scale=scale, useRGBA=use_rgba)
                except Exception as e:
                    assert expectation == type(e)
                else:
                    assert False, f"makeARGB({key!r}) was supposed to raise {expectation} but didn't raise anything."
            else:
                expectation = cupy.asarray(expectation)
                output, alpha = _makeARGB(data, lut=lut, levels=levels, scale=scale, useRGBA=use_rgba)
                assert (
                    output == expectation
                ).all(), f"Incorrect _makeARGB({key!r}) output! Expected:\n{expectation!r}\n  Got:\n{output!r}"

        _do_something_for_every_combo(assert_cupy_correct)
    finally:
        setConfigOption("useCupy", prev_setting)


@pytest.mark.filterwarnings("ignore:invalid value encountered")
def test_numba_makeARGB_against_generated_references():
    oldcfg_numba = getConfigOption("useNumba")
    if not oldcfg_numba:
        try:
            import numba
        except ImportError:
            pytest.skip("Numba unavailable to test")

    # useCupy needs to be set to False because it takes
    # precedence over useNumba in rescaleData
    oldcfg_cupy = getConfigOption("useCupy")
    setConfigOption("useCupy", False)
    setConfigOption("useNumba", not oldcfg_numba)
    test_makeARGB_against_generated_references()
    setConfigOption("useNumba", oldcfg_numba)
    setConfigOption("useCupy", oldcfg_cupy)


def test_makeARGB_with_human_readable_code():
    # Many parameters to test here:
    #  * data dtype (ubyte, uint16, float, others)
    #  * data ndim (2 or 3)
    #  * levels (None, 1D, or 2D)
    #  * lut dtype
    #  * lut size
    #  * lut ndim (1 or 2)
    #  * useRGBA argument
    # Need to check that all input values map to the correct output values, especially
    # at and beyond the edges of the level range.

    def checkArrays(a, b):
        # because py.test output is difficult to read for arrays
        if not np.all(a == b):
            comp = []
            for i in range(a.shape[0]):
                if a.shape[1] > 1:
                    comp.append('[')
                for j in range(a.shape[1]):
                    m = a[i, j] == b[i, j]
                    comp.append('%d,%d  %s %s  %s%s' %
                                (i, j, str(a[i, j]).ljust(15), str(b[i, j]).ljust(15),
                                 m, ' ********' if not np.all(m) else ''))
                if a.shape[1] > 1:
                    comp.append(']')
            raise Exception("arrays do not match:\n%s" % '\n'.join(comp))

    def checkImage(img, check, alpha, alphaCheck):
        assert img.dtype == np.ubyte
        assert alpha is alphaCheck
        if alpha is False:
            checkArrays(img[..., 3], 255)

        if np.isscalar(check) or check.ndim == 3:
            checkArrays(img[..., :3], check)
        elif check.ndim == 2:
            checkArrays(img[..., :3], check[..., np.newaxis])
        elif check.ndim == 1:
            checkArrays(img[..., :3], check[..., np.newaxis, np.newaxis])
        else:
            raise Exception('invalid check array ndim')

    # uint8 data tests

    im1 = np.arange(256).astype('ubyte').reshape(256, 1)
    im2, alpha = _makeARGB(im1, levels=(0, 255))
    checkImage(im2, im1, alpha, False)

    im3, alpha = _makeARGB(im1, levels=(0.0, 255.0))
    checkImage(im3, im1, alpha, False)

    im4, alpha = _makeARGB(im1, levels=(255, 0))
    checkImage(im4, 255 - im1, alpha, False)

    im5, alpha = _makeARGB(np.concatenate([im1] * 3, axis=1), levels=[(0, 255), (0.0, 255.0), (255, 0)])
    checkImage(im5, np.concatenate([im1, im1, 255 - im1], axis=1), alpha, False)

    im2, alpha = _makeARGB(im1, levels=(128, 383))
    checkImage(im2[:128], 0, alpha, False)
    checkImage(im2[128:], im1[:128], alpha, False)

    # uint8 data + uint8 LUT
    lut = np.arange(256)[::-1].astype(np.uint8)
    im2, alpha = _makeARGB(im1, lut=lut)
    checkImage(im2, lut, alpha, False)

    # lut larger than maxint
    lut = np.arange(511).astype(np.uint8)
    im2, alpha = _makeARGB(im1, lut=lut)
    checkImage(im2, lut[::2], alpha, False)

    # lut smaller than maxint
    lut = np.arange(128).astype(np.uint8)
    im2, alpha = _makeARGB(im1, lut=lut)
    checkImage(im2, np.linspace(0, 127.5, 256, dtype='ubyte'), alpha, False)

    # lut + levels
    lut = np.arange(256)[::-1].astype(np.uint8)
    im2, alpha = _makeARGB(im1, lut=lut, levels=[-128, 384])
    checkImage(im2, np.linspace(191.5, 64.5, 256, dtype='ubyte'), alpha, False)

    im2, alpha = _makeARGB(im1, lut=lut, levels=[64, 192])
    checkImage(im2, np.clip(np.linspace(384.5, -127.5, 256), 0, 255).astype('ubyte'), alpha, False)

    # uint8 data + uint16 LUT
    lut = np.arange(4096)[::-1].astype(np.uint16) // 16
    im2, alpha = _makeARGB(im1, lut=lut)
    checkImage(im2, np.arange(256)[::-1].astype('ubyte'), alpha, False)

    # uint8 data + float LUT
    lut = np.linspace(10., 137., 256)
    im2, alpha = _makeARGB(im1, lut=lut)
    checkImage(im2, lut.astype('ubyte'), alpha, False)

    # uint8 data + 2D LUT
    lut = np.zeros((256, 3), dtype='ubyte')
    lut[:, 0] = np.arange(256)
    lut[:, 1] = np.arange(256)[::-1]
    lut[:, 2] = 7
    im2, alpha = _makeARGB(im1, lut=lut)
    checkImage(im2, lut[:, None, ::-1], alpha, False)

    # check useRGBA
    im2, alpha = _makeARGB(im1, lut=lut, useRGBA=True)
    checkImage(im2, lut[:, None, :], alpha, False)

    # uint16 data tests
    im1 = np.arange(0, 2 ** 16, 256).astype('uint16')[:, None]
    im2, alpha = _makeARGB(im1, levels=(512, 2 ** 16))
    checkImage(im2, np.clip(np.linspace(-2, 253, 256), 0, 255).astype('ubyte'), alpha, False)

    lut = (np.arange(512, 2 ** 16)[::-1] // 256).astype('ubyte')
    im2, alpha = _makeARGB(im1, lut=lut, levels=(512, 2 ** 16 - 256))
    checkImage(im2, np.clip(np.linspace(257, 2, 256), 0, 255).astype('ubyte'), alpha, False)

    lut = np.zeros(2 ** 16, dtype='ubyte')
    lut[1000:1256] = np.arange(256)
    lut[1256:] = 255
    im1 = np.arange(1000, 1256).astype('uint16')[:, None]
    im2, alpha = _makeARGB(im1, lut=lut)
    checkImage(im2, np.arange(256).astype('ubyte'), alpha, False)

    # float data tests
    im1 = np.linspace(1.0, 17.0, 256)[:, None]
    im2, alpha = _makeARGB(im1, levels=(5.0, 13.0))
    checkImage(im2, np.clip(np.linspace(-128, 383, 256), 0, 255).astype('ubyte'), alpha, False)

    lut = (np.arange(1280)[::-1] // 10).astype('ubyte')
    im2, alpha = _makeARGB(im1, lut=lut, levels=(1, 17))
    checkImage(im2, np.linspace(127.5, 0, 256).astype('ubyte'), alpha, False)

    # nans in image

    # 2d input image, one pixel is nan
    im1 = np.ones((10, 12))
    im1[3, 5] = np.nan
    im2, alpha = _makeARGB(im1, levels=(0, 1))
    assert alpha
    assert im2[3, 5, 3] == 0  # nan pixel is transparent
    assert im2[0, 0, 3] == 255  # doesn't affect other pixels

    # 3d RGB input image, any color channel of a pixel is nan
    im1 = np.ones((10, 12, 3))
    im1[3, 5, 1] = np.nan
    im2, alpha = _makeARGB(im1, levels=(0, 1))
    assert alpha
    assert im2[3, 5, 3] == 0  # nan pixel is transparent
    assert im2[0, 0, 3] == 255  # doesn't affect other pixels

    # 3d RGBA input image, any color channel of a pixel is nan
    im1 = np.ones((10, 12, 4))
    im1[3, 5, 1] = np.nan
    im2, alpha = _makeARGB(im1, levels=(0, 1), useRGBA=True)
    assert alpha
    assert im2[3, 5, 3] == 0  # nan pixel is transparent

    # test sanity checks
    class AssertExc(object):
        def __init__(self, exc=Exception):
            self.exc = exc

        def __enter__(self):
            return self

        def __exit__(self, *args):
            assert args[0] is self.exc, "Should have raised %s (got %s)" % (self.exc, args[0])
            return True

    with AssertExc(TypeError):  # invalid image shape
        _makeARGB(np.zeros((2,), dtype='float'))
    with AssertExc(TypeError):  # invalid image shape
        _makeARGB(np.zeros((2, 2, 7), dtype='float'))
    with AssertExc():  # float images require levels arg
        _makeARGB(np.zeros((2, 2), dtype='float'))
    with AssertExc():  # bad levels arg
        _makeARGB(np.zeros((2, 2), dtype='float'), levels=[1])
    with AssertExc():  # bad levels arg
        _makeARGB(np.zeros((2, 2), dtype='float'), levels=[1, 2, 3])
    with AssertExc():  # can't mix 3-channel levels and LUT
        _makeARGB(np.zeros((2, 2)), lut=np.zeros((10, 3), dtype='ubyte'), levels=[(0, 1)] * 3)
    with AssertExc():  # multichannel levels must have same number of channels as image
        _makeARGB(np.zeros((2, 2, 3), dtype='float'), levels=[(1, 2)] * 4)
    with AssertExc():  # 3d levels not allowed
        _makeARGB(np.zeros((2, 2, 3), dtype='float'), levels=np.zeros([3, 2, 2]))
